/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: poll.c,v 1.46 2003/11/06 10:55:25 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/poll.h>
#include <zorp/stream.h>
#include <zorp/log.h>
#include <zorp/source.h>
#include <zorp/error.h>

#include <glib.h>

#include <sys/types.h>
#ifndef G_OS_WIN32
#  include <sys/poll.h>
#endif
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <assert.h>

/**
 * ZPoll:
 *
 *
 * ZRealPoll is an implementation of the ZPoll interface defined in
 * poll.h. It's a callback based poll loop, which can be used by proxy
 * modules. It holds a collection of ZStream objects and calls their
 * callbacks when a requested I/O event occurs. ZPoll objects are
 * reference counted and are automatically freed when the number of
 * references to a single instance reaches zero.
 *
 **/
typedef struct _ZRealPoll
{
  guint ref_count;
  GMainContext *context;
  GPollFD *pollfd;
  guint pollfd_num;
  gboolean quit;
  GStaticMutex lock;
  GSource *wakeup;
  GHashTable *streams;
} ZRealPoll;

/**
 * ZPollSource:
 *
 * This structure is a special source used to wake up a ZPoll loop from
 * different threads.
 **/
typedef struct _ZPollSource
{
  GSource super;
  gboolean wakeup;
} ZPollSource;

/**
 * z_poll_source_prepare:
 * @s: ZPollSource instance
 * @timeout: poll timeout
 *
 * This is the prepare function of ZPollSource, it basically checks whether
 * the loop was woken up by checking self->wakeup, and returns TRUE in that
 * case. The timeout value returned is always -1, e.g. infinite timeout.
 **/    
static gboolean
z_poll_source_prepare(GSource *s, gint *timeout)
{
  ZPollSource *self = (ZPollSource *)s;
    
  z_enter();
  
  if (self->wakeup)
    {
      return TRUE;
    }
  
  *timeout = -1;
  z_leave();
  return FALSE;
}

/**
 * z_poll_source_check:
 * @s: ZPollSource instance (not used)
 *
 * This is the check function of ZPollSource. As we poll nothing we always
 * return FALSE.
 *
 * FIXME: I think this function should check the value of self->wakeup
 **/
static gboolean
z_poll_source_check(GSource *s G_GNUC_UNUSED)
{
  z_enter();
      
  z_leave();
  return FALSE;
}

/**
 * z_poll_source_dispatch:
 * @s: ZPollSource instance
 * @callback: the callback for @s (not used)
 * @user_data: the data to be passed to @callback (not used)
 *
 * This function simply sets self->wakeup to FALSE to allow the next poll loop
 * to run.
 **/
static gboolean
z_poll_source_dispatch(GSource *s,
                       GSourceFunc  callback G_GNUC_UNUSED,
                       gpointer  user_data G_GNUC_UNUSED)
{
  ZPollSource *self = (ZPollSource *) s;
  
  z_enter();

  self->wakeup = FALSE;

  z_leave();
  return TRUE;
}

GSourceFuncs z_poll_source_funcs = 
{
  z_poll_source_prepare,
  z_poll_source_check,
  z_poll_source_dispatch,
  NULL,
  NULL,
  NULL
};

/**
 * z_poll_stream_unref:
 * @s: ZStream instance to free
 *
 * This function is used as the destroy notify callback of the hashtable in
 * ZPoll. It detaches the source of the stream by calling
 * z_stream_detach_source and also drops the reference to the stream.
 **/
static void 
z_poll_stream_unref(gpointer s)
{
  ZStream *stream = (ZStream *)s;
  
  z_enter();
  z_stream_detach_source(stream);
  z_stream_unref(stream);
  z_leave();
}

/** z_poll_new:
 *
 * This function creates a new ZPoll instance.
 *
 * Returns: a pointer to the new instance
 **/
ZPoll *
z_poll_new(void)
{
  ZRealPoll *self = g_new0(ZRealPoll, 1);
  
  z_enter();
  g_return_val_if_fail( self != NULL, NULL);
  
  self->ref_count = 1;
  self->quit = FALSE;
  self->pollfd_num = 4;
  self->pollfd = g_new(GPollFD, self->pollfd_num);
  self->streams = g_hash_table_new_full(g_direct_hash,
                                        g_direct_equal,
                                        z_poll_stream_unref,
                                        NULL);
  
  self->context = g_main_context_default();
  if (g_main_context_acquire(self->context))
    {
      g_main_context_ref(self->context);
    }
  else
    {
      self->context = g_main_context_new();
      assert(g_main_context_acquire(self->context));
    }
  
  self->wakeup = g_source_new(&z_poll_source_funcs,
                              sizeof(ZPollSource));
  
  g_source_attach(self->wakeup, self->context);
  
  z_leave();
  return (ZPoll *) self;
}

/** 
 * z_poll_destroy:
 * @s: ZPoll instance
 *
 * Used internally to free up an instance when the reference count
 * reaches 0.
 **/
static void
z_poll_destroy(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *)s;

  z_enter();
  if(self->wakeup)
    {
      g_source_destroy(self->wakeup);
      g_source_unref(self->wakeup);
      self->wakeup = NULL;
    }
  g_hash_table_destroy(self->streams);
  g_main_context_release(self->context);
  g_main_context_unref(self->context);
  g_free(self->pollfd);
  
  g_free(self);
  z_leave();
}

/**
 * z_poll_ref:
 * @s: ZPoll instance
 *
 * Increment the reference count of the given ZPoll instance.
 **/
void 
z_poll_ref(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  self->ref_count++;
  z_leave();
}


/**
 * z_poll_unref:
 * @s: ZPoll instance
 *
 * Decrement the reference count of the given ZPoll instance, and free it
 * using z_poll_destroy() if it reaches 0.
 **/
void
z_poll_unref(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;

  z_enter();
  if (self)
    {
      g_assert(self->ref_count > 0);
      self->ref_count--;
      if (self->ref_count == 0)
        z_poll_destroy(s);
    }
  z_leave();
}

/**
 * z_poll_add_stream:
 * @s: ZPoll instance
 * @stream: stream instance
 *
 * Register a ZStream to be monitored by a ZPoll instance.
 **/
void
z_poll_add_stream(ZPoll *s, ZStream *stream)
{
  ZRealPoll *self = (ZRealPoll *) s;

  z_enter();
  
  z_stream_ref(stream);
  g_hash_table_insert(self->streams, stream, NULL);
  z_stream_attach_source(stream, self->context);
  
  z_leave();
}

/**
 * z_poll_remove_stream:
 * @s: ZPoll instance
 * @stream: stream to be removed
 *
 * Remove a ZStream from a ZPoll instance.
 **/
void
z_poll_remove_stream(ZPoll *s, ZStream *stream)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  
  g_hash_table_remove(self->streams, stream);

  z_leave();
}

/**
 * z_poll_iter_timeout:
 * @s: ZPoll instance
 * @timeout: timeout value in milliseconds
 *
 * Run an iteration of the poll loop. Monitor filedescriptors of
 * registered Streams, and call appropriate callbacks.
 *
 * Returns: TRUE if the iteration should be called again.
 **/
guint
z_poll_iter_timeout(ZPoll *s, gint timeout)
{
  ZRealPoll *self = (ZRealPoll *) s;
  gint max_priority = G_PRIORITY_LOW;
  gint polltimeout;
  gint fdnum = 0;
  GPollFunc pollfunc;
  gint rc;

  z_enter();
  z_errno_set(0);
  if (self->quit)
    {
      z_leave();
      return 0;
    }
  
  g_main_context_prepare (self->context, &max_priority);

  fdnum = g_main_context_query(self->context,
                               max_priority,
                               &polltimeout,
                               self->pollfd,
                               self->pollfd_num);

  while (fdnum > (gint)self->pollfd_num)
    {
      /*LOG
        This message reports that the polling fd structure is growing.
       */
      z_log(NULL, CORE_DEBUG, 7, "Polling fd structure growing; old_num='%d'", self->pollfd_num);
      self->pollfd_num *= 2;
      self->pollfd = g_renew(GPollFD, self->pollfd, self->pollfd_num);
      
      fdnum = g_main_context_query(self->context,
                                   max_priority,
                                   &polltimeout,
                                   self->pollfd,
                                   self->pollfd_num);
    }

  if (polltimeout <= -1)
    polltimeout = timeout;
  else if (timeout > -1)
    polltimeout = MIN(polltimeout, timeout);

  pollfunc = g_main_context_get_poll_func(self->context);

  z_trace(NULL, "Entering poll;");
  rc = pollfunc(self->pollfd, fdnum, polltimeout);
  z_trace(NULL, "Returning from poll;");

  g_main_context_check(self->context, max_priority, self->pollfd, fdnum);
  g_main_context_dispatch(self->context);

  if (rc == -1 && !z_errno_is(EINTR))
    {
      z_leave();
      return 0;
    }

  if (rc == 0 && polltimeout == timeout)
    {
      z_errno_set(ETIMEDOUT);
      z_leave();
      return 0;
    }

  z_leave();
  return 1;
}

/**
 * z_poll_wakeup:
 * @s: ZPoll instance
 *
 * Wake up a running poll loop using its wakeup pipe.
 **/
void
z_poll_wakeup(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  ZPollSource *src;
  
  z_enter();
  
  src = (ZPollSource *)self->wakeup;
  src->wakeup = TRUE;
  g_main_context_wakeup(self->context);
  z_leave();
}

/**
 * z_poll_is_running:
 * @s: ZPoll instance
 *
 * Checks whether z_poll_quit was called earlier on this ZPoll object.
 *
 * Returns: TRUE if the poll is still running
 **/
gboolean
z_poll_is_running(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  z_leave();
  return !self->quit;
}

/**
 * z_poll_quit:
 * @s: ZPoll instance
 * 
 * Indicate that this poll loop is to be ended. Can be called from a
 * a thread different from the one running poll.
 **/
void
z_poll_quit(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;  
  
  z_enter();
  
  self->quit = TRUE;
  z_poll_wakeup(s);
  
  z_leave();
}

/**
 * z_poll_get_context:
 * @s: ZPoll instance
 *
 * Return the underlying GMainContext.
 **/
GMainContext *
z_poll_get_context(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  z_leave();
  return self->context;
}
