/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module produces an annotated list of all packages grouped by
section and category.
*****/
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "magpie.h"

extern int mkdir (const char *, mode_t);

/*+
Comparison function for sorting by section, category, priority, 
name, version
+*/
static int cmp_s_c_p (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->name);
	assert (qq->name);

	r = pp->section - qq->section;
	if (r)
		return r;
	
	r = pp->category - qq->category;
	if (r)
		return r;

	r = pp->priority - qq->priority;
	if (r)
		return r;

	r = strcoll (pp->name, qq->name);
	if (r)
		return r;

	return -strcoll (pp->version, qq->version);
}


/*+
Print the intermediate index files.
+*/
static int sections_index (FILE *fp0, int type)
{
	int sidx, pidx, cidx;
	char pathname[256];
	FILE *fp;

	if (type != MAGPIE_ALL_PACKAGES)
		return 0;

	mp_title (fp0, 3, "Grouped by Section");

	/*
	 *	Write the index files.
	 */
	mp_list_open (fp0);
	for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
		if (matrixcnt_s[sidx] == 0)
			continue;

		mp_item_open (fp0);
		mp_url (fp0, "%1$s/index.html.gz", "%1$s", sections[sidx]);
		fprintf (fp0, " (%d)", matrixcnt_s[sidx]);
		mp_item_close (fp0);

		sprintf (pathname, "%s/index.html", sections[sidx]);
		fp = fopen (pathname, "w");
		mp_doc_open (fp, "Section '%s'", sections[sidx]);
		mp_abstract (fp, "\
This page groups packages by section, category, and then priority.\n");

		mp_list_open (fp);
		for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
			if (matrixcnt_sc[sidx][cidx] == 0)
				continue;

			mp_item_open (fp);
			mp_url (fp, "%1$s.html.gz", "%1$s</a>", categories[cidx]);
			fprintf (fp, " (%d)", matrixcnt_sc[sidx][cidx]);
			mp_item_close (fp);

			mp_list_open (fp);
			for (pidx = 0; pidx < CNT_PRIORITIES; pidx++) {
				if (matrixcnt_scp[sidx][cidx][pidx] == 0)
					continue;
				mp_item_open (fp);
				mp_url (fp, "%1$s.html.gz#%2$s", "%2$s",
					categories[cidx], priorities[pidx]);

				fprintf (fp, " (%d)", matrixcnt_scp[sidx][cidx][pidx]);
				mp_item_close (fp);
			}
			mp_list_close (fp);
		}
		mp_list_close (fp);

		mp_doc_close (fp);
		fclose (fp);
		gzip (pathname);
	}
	mp_list_close (fp0);
	return 0;
}


/*+
Print the packages ordered by section/category/priority.
This code is straightforward, but at the cost of 100+ open files.
+*/
static int sections_details (void)
{
	int sidx, pidx, cidx, i;
	struct package_info *p;
	char pathname[256], *s;
	char short_category[30];
	FILE *fp[CNT_SECTIONS][CNT_CATEGORIES];
	int prio[CNT_SECTIONS][CNT_CATEGORIES];
	char errmsg[256+20];

	memset (fp, 0, sizeof fp);
	memset (prio, 0, sizeof prio);

	qsort (cache, cachecnt, sizeof cache[0], cmp_s_c_p);

	/*
	 *	Create the detail files ...
	 */
	for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
		if (matrixcnt_s[sidx] == 0)
			continue;

		mkdir (sections[sidx], 0755);

		for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
			if (matrixcnt_sc[sidx][cidx] == 0)
				continue;

			strcpy (short_category, categories[cidx]);
			s = strchr (short_category, '/');
			if (s)
				*s = '\0';
			sprintf (pathname, "%s/%s.html", sections[sidx], short_category);

			prio[sidx][cidx] = -1;
			fp[sidx][cidx] = fopen (pathname, "w");
			if (fp[sidx][cidx] == NULL) {
				sprintf (errmsg, "fopen(%s)", pathname);
				perror (errmsg);
			}
			else {
				mp_doc_open (fp[sidx][cidx], "Section '%s', Category '%s'",
					sections[sidx], categories[cidx]);
			}
		}
	}

	/*
	 *	... write the content ...
	 */
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		sidx = p->section;
		cidx = p->category;
		pidx = p->priority;

		assert (fp[sidx][cidx]);

		if (prio[sidx][cidx] != p->priority) {
			mp_break (fp[sidx][cidx]);
			mp_name (fp[sidx][cidx], priorities[p->priority]);
			mp_title (fp[sidx][cidx], 3, "Priority '%1$s' (%2$d)",
				priorities[p->priority], matrixcnt_scp[sidx][cidx][pidx]);
			prio[sidx][cidx] = p->priority;
		}

		mp_package (fp[sidx][cidx], p,
			matrixcnt_sc[sidx][cidx] > 500 ? 2 : 1, 0);
	}

	/*
	 *	... and close them.
	 */
	for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
		if (matrixcnt_s[sidx] == 0)
			continue;
		for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
			if (matrixcnt_sc[sidx][cidx] == 0)
				continue;
			mp_doc_close (fp[sidx][cidx]);
			fclose (fp[sidx][cidx]);

			strcpy (short_category, categories[cidx]);
			s = strchr (short_category, '/');
			if (s)
				*s = '\0';
			sprintf (pathname, "%s/%s.html", sections[sidx], short_category);
			gzip (pathname);
		}
	}

	return 0;
}


struct magpie_module mod_sections = { 
	version           : MAGPIE_VERSION,
	description       : "group packages by section",
	init              : sections_details,
	annotated_index   : sections_index
};
