#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "csd.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


/*
 *      Color selection dialog structure:
 */
typedef struct {

        gbool initialized;
        gbool map_state;

        GtkWidget *csd;         /* Color selection dialog widget. */

        csd_color_struct current_color;

        void *client_data;
        void (*color_changed_cb)(void *, csd_color_struct *);

} csd_data_struct;


static void CSDOKCB(GtkWidget *widget, gpointer data);
static void CSDCancelCB(GtkWidget *widget, gpointer data);
static gint CSDCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void CSDColorChangeCB(GtkWidget *widget, gpointer data);

gint CSDInit(void);
void CSDSetTransientFor(GtkWidget *w);
gbool CSDIsQuery(void);
void CSDBreakQuery(void);
gbool CSDGetResponse(
        const gchar *title,
        const gchar *ok_label, const gchar *cancel_label,
        csd_color_struct *start_color,
        csd_color_struct **color_rtn,
        gpointer client_data,
        void (*color_changed_cb)(gpointer, csd_color_struct *)
);
void CSDMap(void);
void CSDUnmap(void);
void CSDShutdown(void);


static gint block_loop_level;
static csd_data_struct csd_data;
static gbool csd_got_user_response;
static csd_color_struct response_color;


/*
 *	Ok button callback.
 */
static void CSDOKCB(GtkWidget *widget, gpointer data)
{
	csd_data_struct *csdd = (csd_data_struct *)data;
        if(csdd == NULL)
            return;

        if(!csdd->initialized)
            return;

	/* Set user response code to TRUE, indicating OK. */
	csd_got_user_response = TRUE;

	/* Copy response color. */
	memcpy(
	    &response_color, &csdd->current_color,
	    sizeof(csd_color_struct)
	);

	/* Unmap. */
	CSDUnmap();

	/* Break out of blocking loop. */
	gtk_main_quit();
	block_loop_level--;
}

/*
 *	Cancel callback.
 */
static void CSDCancelCB(GtkWidget *widget, gpointer data)
{
        csd_data_struct *csdd = (csd_data_struct *)data;
        if(csdd == NULL)
            return;

        if(!csdd->initialized)
            return;

        /* Set user response code to FALSE, indicating Cancel. */
        csd_got_user_response = FALSE;

	/* Unmap. */
        CSDUnmap();

        /* Break out of blocking loop. */
        gtk_main_quit();
	block_loop_level--;
}           

/*
 *      Close callback.
 */
static gint CSDCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	CSDCancelCB(widget, data);
	return(TRUE);
}

/*
 *	Color changed callback.
 */
static void CSDColorChangeCB(GtkWidget *widget, gpointer data)
{
	gdouble vd[3];
	GtkColorSelectionDialog *csd;
	csd_data_struct *csdd = (csd_data_struct *)data;
	if(csdd == NULL)
	    return;

	if(!csdd->initialized)
	    return;

	csd = (GtkColorSelectionDialog *)csdd->csd;
	if(csd == NULL)
	    return;

	/* Get current color. */
	gtk_color_selection_get_color(
	    GTK_COLOR_SELECTION(csd->colorsel), vd
	);

	/* Copy current color values to csdd structure's
	 * csd_color_struct.
	 */
	csdd->current_color.r = vd[0];
	csdd->current_color.g = vd[1];
	csdd->current_color.b = vd[2];
	csdd->current_color.a = 1.0;

	/* Call color changed callback. */
	if(csdd->color_changed_cb != NULL)
	{
	    csdd->color_changed_cb(
		(void *)csdd->client_data,
		&csdd->current_color
	    );
	}
}


/*
 *	Initializes color selection dialog..
 */
gint CSDInit(void)
{
	GtkWidget *w;
	GtkColorSelectionDialog *csd;
	csd_data_struct *csdd = &csd_data;


	/* Reset local globals. */
        block_loop_level = 0;
        csd_got_user_response = FALSE;
	memset(&response_color, 0x00, sizeof(csd_color_struct));
	memset(csdd, 0x00, sizeof(csd_data_struct));

	/* Reset values. */
        csdd->initialized = TRUE;
        csdd->map_state = FALSE;

	/* Create the color selection dialog. */
	w = gtk_color_selection_dialog_new("Select Color");
	csdd->csd = w;
	csd = GTK_COLOR_SELECTION_DIALOG(w);

	/* Connect "delete_event" on main csd window. */
	gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(CSDCloseCB), (gpointer)csdd
        );

	/* Get GtkColorSelection widget. */
	w = csd->colorsel;

	/* Connect "color_changed" signal to the GtkColorSelection
	 * widget.
	 */
	gtk_signal_connect(
	    GTK_OBJECT(w), "color_changed",
	    GTK_SIGNAL_FUNC(CSDColorChangeCB), (gpointer)csdd
	);

	/* Ok button. */
	gtk_signal_connect(
	    GTK_OBJECT(csd->ok_button), "clicked",
            GTK_SIGNAL_FUNC(CSDOKCB), (gpointer)csdd
        );
	gtk_widget_show(csd->ok_button);

	/* Hide reset button. */
	if(csd->reset_button != NULL)
            gtk_widget_hide(csd->reset_button);

	/* Cancel button. */
        gtk_signal_connect(
            GTK_OBJECT(csd->cancel_button), "clicked",
            GTK_SIGNAL_FUNC(CSDCancelCB), (gpointer)csdd
        );
        gtk_widget_show(csd->cancel_button);

        /* Hide help button. */
	if(csd->help_button != NULL)
            gtk_widget_hide(csd->help_button);

	return(0);
}

/*
 *      Sets dialog to be a transient for the given toplevel window
 *      widget w. If w is NULL then no transient for will be unset.
 */
void CSDSetTransientFor(GtkWidget *w)
{
        csd_data_struct *csdd = &csd_data;

        if(!csdd->initialized)
            return;

        if(csdd->csd != NULL)
        {
            if(w != NULL)
            {
                /* Given widget if not NULL, must be a window. */
                if(!GTK_IS_WINDOW(GTK_OBJECT(w)))
                    return;

                gtk_window_set_modal(
                    GTK_WINDOW(csdd->csd), TRUE
                );
                gtk_window_set_transient_for(
                    GTK_WINDOW(csdd->csd), GTK_WINDOW(w)
                );
            }
            else
            {
                gtk_window_set_modal(
                    GTK_WINDOW(csdd->csd), FALSE
                );
                gtk_window_set_transient_for(
                    GTK_WINDOW(csdd->csd), NULL
                );
            }
        }
}


/*
 *      Returns TRUE if currently blocking for query.
 */
gbool CSDIsQuery(void)
{
        if(block_loop_level > 0)
            return(TRUE);
        else
            return(FALSE);
}

/*
 *	Ends query if any and returns a not available response.
 */
void CSDBreakQuery(void)
{
	csd_got_user_response = FALSE;

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;
}

/*
 *	Maps the color selection dialog and sets up the inital values.
 *
 *	Returns TRUE if a color was sselected or FALSE if user canceled.
 *
 *	For most values that are set NULL, the value is left unchanged.
 *	All given values are coppied.
 *
 *	All returned pointer values should be considered statically
 *	allocated, do not deallocate them.
 */
gbool CSDGetResponse(
        const gchar *title,
        const gchar *ok_label, const gchar *cancel_label,
        csd_color_struct *start_color,
        csd_color_struct **color_rtn,
        gpointer client_data,
        void (*color_changed_cb)(gpointer, csd_color_struct *)
)
{
	GtkWidget *w;
	GtkColorSelectionDialog *csd;
	csd_data_struct *csdd = &csd_data;


        /* Do not handle response if already waiting for a response,
         * return with a not available response code.
         */
        if(block_loop_level > 0)
	{
            if(color_rtn != NULL)
                (*color_rtn) = NULL;

            return(FALSE);
	}

	/* Reset global responses values. */
	csd_got_user_response = FALSE;
	memset(&response_color, 0x00, sizeof(csd_color_struct));

	/* Reset returns. */
	if(color_rtn != NULL)
	    (*color_rtn) = NULL;


	/* Color selection dialog must be initialized. */
	if(!csdd->initialized)
	    return(csd_got_user_response);

	/* Get pointer to color selection dialog widget. */
	w = csdd->csd;
	if(w == NULL)
            return(csd_got_user_response);
	csd = GTK_COLOR_SELECTION_DIALOG(w);

	/* Update title if specified. */
	if(title != NULL)
	    gtk_window_set_title(GTK_WINDOW(w), title);

	/* Update button labels. */
	if((ok_label != NULL) && (csd->ok_button != NULL))
	{
	    GtkButton *button = (GtkButton *)csd->ok_button;
	    GtkWidget *w2;

	    w2 = GTK_BIN(button)->child;
	    if((w2 != NULL) ? GTK_IS_LABEL(w2) : 0)
		gtk_label_set_text(GTK_LABEL(w2), ok_label);
	}
        if((cancel_label != NULL) && (csd->cancel_button != NULL))
        {
            GtkButton *button = (GtkButton *)csd->cancel_button;
            GtkWidget *w2;

            w2 = GTK_BIN(button)->child;
            if((w2 != NULL) ? GTK_IS_LABEL(w2) : 0)
                gtk_label_set_text(GTK_LABEL(w2), cancel_label);
        }


	/* Update initial start color if specified. */
	if(start_color != NULL)
	{
	    gdouble vd[3];
	    GtkColorSelection *colorsel;

	    colorsel = GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(w)->colorsel
	    );

	    vd[0] = start_color->r;
	    vd[1] = start_color->g;
	    vd[2] = start_color->b;
/*		  = start_color->a; */

	    gtk_color_selection_set_color(colorsel, vd);

	    csdd->current_color.r = start_color->r;
            csdd->current_color.g = start_color->g;
            csdd->current_color.b = start_color->b;
            csdd->current_color.a = start_color->a;
	}

	/* Set up callbacks. */
	csdd->client_data = client_data;
	csdd->color_changed_cb = color_changed_cb;

        /* Map color selection dialog as needed. */
        CSDMap();

        /* Block GUI untill response. */
	block_loop_level++;
        gtk_main();

        /* Unmap color selection dialog just in case it was not unmapped
	 * from any of the callbacks.
	 */
        CSDUnmap();

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;


	/* Begin setting returns. */

	/* Color return. */
	if(color_rtn != NULL)
	    (*color_rtn) = &response_color;

	return(csd_got_user_response);
}


/*
 *	Maps the color selection dialog as needed.
 */
void CSDMap(void)
{
        csd_data_struct *csdd = &csd_data;


        if(csdd == NULL)
            return;

        if(!csdd->initialized)
            return;

        if(!csdd->map_state)
        {
            GtkWidget *w = csdd->csd;

            if(w != NULL)
                gtk_widget_show(w);

            csdd->map_state = TRUE;
        }
}

/*
 *	Unmaps the color selection dialog as needed.
 */
void CSDUnmap(void)
{
        csd_data_struct *csdd = &csd_data;


	if(csdd == NULL)
	    return;

	if(!csdd->initialized)
	    return;

	if(csdd->map_state)
	{
	    GtkWidget *w = csdd->csd;

	    if(w != NULL)
		gtk_widget_hide(w);

	    csdd->map_state = FALSE;
	}
}

/*
 *	Deallocates color selection dialog resources.
 */
void CSDShutdown(void)
{
	GtkWidget **w;
        csd_data_struct *csdd = &csd_data;


	/* Reset local globals. */
        csd_got_user_response = FALSE; 
        memset(&response_color, 0x00, sizeof(csd_color_struct));

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;


	/* Is color selection dialog initialized? */
	if(csdd->initialized)
	{
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Begin destroying file browser widgets. */
	    w = &csdd->csd;
	    DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET
	}

	/* Clear color selection dialog structure. */
	memset(csdd, 0x00, sizeof(csd_data_struct));
}
