#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/string.h"

#include "guiutils.h"
#include "cdialog.h"

#include "viewer.h"
#include "viewerfio.h"

#include "manedit.h"
#include "maneditop.h"
#include "config.h"

static int ViewerLoadProgressCB(long cur, long max, void *data);
int ViewerLoadFile(
	viewer_struct *v, const char *filename, const char *name
);


/*
 *	Viewer load progress callback.
 */
static int ViewerLoadProgressCB(long cur, long max, void *data)
{
        gfloat percent;
        viewer_struct *v = (viewer_struct *)data;
        if(v == NULL)
            return(0);

        if(max > 0)
            percent = (gfloat)cur / (gfloat)max;
        else
            percent = 1.0;

        ViewerSetStatusProgress(v, percent);

        return(0);
}


/*
 *	Takes the given manual page format file specified by filename
 *	and converts it to groff output as a tempory file and loads
 *	that tempory file. The tempory file is then removed.
 *
 *	Returns non-zero on error.
 */
int ViewerLoadFile(
	viewer_struct *v, const char *filename, const char *name
)
{
	int status;
	FILE *fp;
	char *strptr;
	char *buf;
	int buf_len;

	char *prog_tmp_dir;
	char *stdout_path;
	char *stderr_path;
	const char *converter_cmd;
	GtkAdjustment *vadj;

	medit_core_struct *core_ptr;
	struct stat stat_buf;


	if((v == NULL) || (filename == NULL))
	    return(-1);

	if(!v->initialized)
	    return(-1);

	/* Already processing? */
	if(v->processing)
	    return(-1);

	/* Mark as processing. */
	v->processing = TRUE;

	/* If given filename is an absolute path, then check if it
	 * exists. This is to allow the converter program to search for
	 * relative paths.
	 */
	if(ISPATHABSOLUTE(filename))
	{
	    if(stat(filename, &stat_buf))
	    {
		v->processing = FALSE;
		return(-2);
	    }
	}

        core_ptr = (medit_core_struct *)v->core_ptr;
        if(core_ptr == NULL)
	{
	    v->processing = FALSE;
            return(-1);
	}

	/* Get Manual Page to output converter command. */
	converter_cmd = (const char *)PrefParmGetValueP(
	    core_ptr->pref,
	    MEDIT_PREF_PARM_CONVERTERS_MANPAGE_TO_OUTPUT
	);
	if((converter_cmd == NULL) ? 1 : ((*converter_cmd) == '\0'))
	{
            v->processing = FALSE;

            CDialogGetResponse(
"Command not defined!",
"Manual Page to output converter command is not\n\
defined. Cannot continue with loading of Manual\n\
Page.",
"The command to convert Manual Pages to readable\n\
output has not been defined, therefore the loading\n\
of the Manual Page cannot be done. You can define\n\
this command under edit->preferences->converters.",
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );

            return(-1);
        }

        /* Create tempory files directory for this program as needed
         * (return string needs to be deallocated).
         */
	prog_tmp_dir = MEditCreateTempDir(core_ptr);
        if(prog_tmp_dir == NULL)
        {
            CDialogGetResponse(
"Non-critical internal error!",
"EditorDoPreview(): MEditCreateTempDir() did not return a path.",
"Could not obtain program tempory file directory path,\n\
and thus unable to generate tempory files for preview.\n\
Cannot continue with preview.",
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            v->processing = FALSE;
            return(-1);
        }

        /* Get standard output file path to output conversion to,
	 * note that stdout_path needs to be deallocated later.
	 */
        stdout_path = tempnam(prog_tmp_dir, "pv");
	stderr_path = tempnam(prog_tmp_dir, "pv");

	/* Deallocate prog_tmp_dir, it is no longer needed. */
	free(prog_tmp_dir);
	prog_tmp_dir = NULL;

	/* No output file path available? */
	if(stdout_path == NULL)
        {
	    free(stdout_path);
	    free(stderr_path);

            v->processing = FALSE;
	    return(-1);
	}

	/* Begin converting. */

	ViewerSetBusy(v);
        ViewerSetStatusProgress(v, 0.0);
	ViewerSetStatusMessage(v, "Converting...");

        /* Convert file specified by filename to groff output as a tempory
         * file specified by tmp_path.
         */

/* Define values in enviroment variables? */

	status = MEditExecuteFmtBlock(
	    core_ptr,
	    converter_cmd,	/* Command. */
	    filename,		/* Filename. */
	    NULL,		/* Options. */
	    stdout_path,	/* Stdout path. */
	    stderr_path		/* Stderr path. */
	);

	ViewerSetStatusMessage(v, "Convert done");

	/* If stderr_path exists, notify of error. */
	MEditDialogFromFile(core_ptr, stderr_path);


	/* Begin loading tempory output file. */

	/* Error occured during execution or stdout_path does not exist? */
	if(status || stat(stdout_path, &stat_buf))
	{
	    unlink(stdout_path);
	    free(stdout_path);
	    unlink(stderr_path);
	    free(stderr_path);
            ViewerSetStatusProgress(v, 0.0);
            ViewerSetStatusMessage(v, "Load done");
	    ViewerSetReady(v);

            v->processing = FALSE;

	    return(-2);
	}
	else
	{
	    buf_len = stat_buf.st_size;
	}

	ViewerSetStatusMessage(v, "Loading...");

	if(buf_len < 0)
	    buf_len = 0;

	/* Allocate memory to load file contents. */
	buf = (char *)malloc((buf_len + 1) * sizeof(char));
	if(buf == NULL)
	{
	    unlink(stdout_path);
	    free(stdout_path);
	    unlink(stderr_path);
	    free(stderr_path);
            ViewerSetStatusProgress(v, 0.0);
	    ViewerSetStatusMessage(v, "Load done");
	    ViewerSetReady(v);

            v->processing = FALSE;

	    return(-1);
	}

	/* Open file. */
	fp = FOpen(stdout_path, "rb");
	if(fp == NULL)
	{
            unlink(stdout_path);
            free(stdout_path);
	    unlink(stderr_path);
	    free(stderr_path);
            ViewerSetStatusProgress(v, 0.0);
            ViewerSetStatusMessage(v, "Load done");
            ViewerSetReady(v);

	    free(buf);

            v->processing = FALSE;

	    return(-2);
	}

	/* Begin reading file data into buffer. */
	status = (int)fread(
	    (void *)buf, (size_t)sizeof(char), (size_t)buf_len,
	    fp
	);
	if(status < buf_len)
	{
	    if(status > 0)
	    {
		buf[status] = '\0';
		buf_len = status;
	    }
	}
	else
	{
	    buf[buf_len] = '\0';    /* Yes its allocated. */
	}

	/* Close file. */
	FClose(fp);
	fp = NULL;

	/* Remove tempory output file and deallocate output path. */
	unlink(stdout_path);
	free(stdout_path);
	stdout_path = NULL;

	unlink(stderr_path);
	free(stderr_path);
	stderr_path = NULL;

        ViewerSetStatusProgress(v, 0.5);


	/* Load buffer to viewer. */
	ViewerTextInsert(
	    v, buf, buf_len, (void *)v, ViewerLoadProgressCB
	);

        ViewerSetStatusProgress(v, 1.0);

	/* Deallocate buffer. */
	free(buf);
	buf = NULL;
	buf_len = 0;


	/* Check last manual page path and see if they match new one. */
	if((v->cur_manpage_name != NULL) && (name != NULL))
	{
	    /* Name same? */
	    if((v->cur_manpage_name == name) ?
	        1 : !strcmp(v->cur_manpage_name, name)
	    )
	    {
		/* Names match, so scroll to last view_text position. */
		vadj = ((v->view_text == NULL) ?
		    NULL : GTK_TEXT(v->view_text)->vadj
		);
		if(vadj != NULL)
		{
		    if(v->last_scroll_vpos < vadj->lower)
			v->last_scroll_vpos = vadj->lower;
		    if(v->last_scroll_vpos > (vadj->upper + vadj->page_size))
			v->last_scroll_vpos = vadj->upper + vadj->page_size;
		    gtk_adjustment_set_value(
			vadj, v->last_scroll_vpos
		    );
		}
	    }
	}



	/* Record current opened manual page file on viewer. */
	if((filename != v->cur_manpage_path) && (filename != NULL))
	{
	    free(v->cur_manpage_path);
	    v->cur_manpage_path = strdup(filename);
	}

	/* Record name. */
	if((name != v->cur_manpage_name) && (name != NULL))
        {
            free(v->cur_manpage_name);
            v->cur_manpage_name = strdup(name);
        }


	/* Update last opened path on viewer. */
        free(v->last_open_path);
        v->last_open_path = strdup(filename);
        if(v->last_open_path != NULL)
        {
            strptr = strrchr(v->last_open_path, DIR_DELIMINATOR);
            if(strptr != NULL)
                (*strptr) = '\0';
        }


        ViewerSetStatusProgress(v, 0.0);
        ViewerSetStatusMessage(v, "Load done");

        v->processing = FALSE;

	ViewerUpdateMenus(v);
	ViewerSetReady(v);

	/* Switch to viewer page. */
	if(v->main_notebook != NULL)
	{
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(v->main_notebook), ViewerPageNumView
	    );
	}

	/* Update text on current manpage combo. */
	if(v->manpage_combo != NULL)
	{
	    GtkCombo *combo = (GtkCombo *)v->manpage_combo;
	    const char *tmp_str;

	    tmp_str = (const char *)strrchr((const char *)filename, ' ');
	    if(tmp_str != NULL)
	    {
		tmp_str += 1;
	        GUIComboAddItem(combo, tmp_str);
	        gtk_entry_set_text(GTK_ENTRY(combo->entry), tmp_str);
	    }
	    else
	    {
                GUIComboAddItem(combo, filename);
                gtk_entry_set_text(GTK_ENTRY(combo->entry), filename);
	    }
	    gtk_editable_select_region(GTK_EDITABLE(combo->entry), 0, -1);

	    /* Also if the manpage_combo now contains a full path, select
	     * the `exact' section on the section combo.
	     */
	    if(v->section_combo != NULL)
	    {
		tmp_str = gtk_entry_get_text(GTK_ENTRY(combo->entry));
		if((*tmp_str) == DIR_DELIMINATOR)
		{
		    gtk_entry_set_text(
			GTK_ENTRY(GTK_COMBO(v->section_combo)->entry),
			MEDIT_SECT_NAME_EXACT
		    );
		}
	    }
	}

	return(0);
}
