/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include "dialog_client.h"
#include "win_util.h"

Client*
dialog_client_new(Wm *w, Window win, Client *trans)
{
   client *c = main_client_new(w, win); 
   c->type = dialog;
   c->configure    = &dialog_client_configure;
   c->button_press = &dialog_client_button_press;
   c->redraw       = &dialog_client_redraw;
   c->show         = &dialog_client_show;
   //c->hide       = &dialog_client_hide;
   c->destroy      = &dialog_client_destroy;
   c->get_coverage = &dialog_client_get_coverage;
   c->trans = trans;
   return c;

}

void
dialog_client_get_coverage(Client *c, int *x, int *y, int *w, int *h)
{
   /* TODO - remember window borders - if any */
   *x = c->x; *y = c->y - main_client_title_height(c);
   *w = c->width; *h = c->height + main_client_title_height(c);
}

void
dialog_client_hide(Client *c)
{
   XLowerWindow(c->wm->dpy, c->frame);
}

void
dialog_client_show(Client *c)
{
   XMapWindow(c->wm->dpy, c->window); 
   XMapRaised(c->wm->dpy, c->frame);

   if (client_want_focus(c))
   {
      XSetInputFocus(c->wm->dpy, c->window,
		     RevertToPointerRoot, CurrentTime);
      c->wm->focused_client = c;
   }
   c->mapped = True;
}

void
dialog_client_configure(Client *c)
{
   int max_w = c->wm->dpy_width-32;
   int max_h = c->wm->dpy_height-32;

   dbg("dialog wants %i x %i", c->width, c->height);
   
   if (c->width > max_w) c->width   = max_w;
   if (c->height > max_h) c->height = max_h;

   c->x = (c->wm->dpy_width  - c->width)/2;
   c->y = (c->wm->dpy_height - c->height)/2;

}

void
dialog_client_redraw(Client *c, Bool use_cache)
{
   Theme *t = c->wm->theme;
   int cx,cy;

   if (c->backing == (Pixmap)NULL)
      client_init_backing(c, c->width, main_client_title_height(c));
   
   theme_render_area(c->wm, c->backing, 0, 0,
		     c->wm->dpy_width, main_client_title_height(c) );

    if (c->name) {
       int space_avail = c->width
	  - c->wm->theme->buttons[BUTTON_CLOSE].width
	 - BUTTON_SPACING
	 - (t->padding*2)
	  -8; 			    

#ifdef USE_XFT
      XftDrawString8(c->xftdraw, &(t->xft_fg), t->xftfont,
		     t->padding, t->padding + t->xftfont->ascent,
		     (unsigned char *) c->name,
		     wu_trim_title(c->wm, c->name, space_avail) );
#else
      
        XDrawString(c->wm->dpy, c->backing, t->text_gc,
            t->padding, t->padding + t->font->ascent,
            c->name, wu_trim_title(c->wm, c->name, space_avail));
#endif

    }

    cx = c->width - t->buttons[BUTTON_CLOSE].width - BUTTON_SPACING;
    cy = (main_client_title_height(c) - t->buttons[BUTTON_CLOSE].height)/2;
    theme_paint_pxm(c->wm, c->backing, cx, cy, &t->buttons[BUTTON_CLOSE]);

    XCopyArea(c->wm->dpy, c->backing, c->frame,
	      c->wm->theme->fg_gc, 0, 0, c->width,
	      c->height, 0, 0 );
    
}

void
dialog_client_button_press(Client *c, XButtonEvent *e)
{
   int cx,cy;
   XEvent ev;
   XGCValues gc_vals;
   Theme *t = c->wm->theme;

   cx = c->width
      - c->wm->theme->buttons[BUTTON_CLOSE].width
      - BUTTON_SPACING;
   cy = (main_client_title_height(c) - t->buttons[BUTTON_CLOSE].height)/2;

   if (e->y <= main_client_title_height(c))
   {
      if (e->x >= cx)
      {
	 if (XGrabPointer(c->wm->dpy, c->wm->root, False,
			  (ButtonPressMask|ButtonReleaseMask),
			  GrabModeAsync,
			  GrabModeAsync, None, c->wm->curs, CurrentTime)
	     != GrabSuccess)
	    return;
	 
	 /* paint 'inverted' button */
	 gc_vals.function = GXnand;
	 XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
	 theme_paint_pxm(c->wm, c->frame, cx, cy,
			 &t->buttons[BUTTON_CLOSE]);
	 
	 XSync(c->wm->dpy, False);	    
	 
	 for (;;) 				      /* get release */
	 {
	    XMaskEvent(c->wm->dpy,
		       (ButtonPressMask|ButtonReleaseMask), &ev);
	    switch (ev.type)
	    {
	       case ButtonRelease:
		  dbg("got release");
		  gc_vals.function = GXcopy;
		  XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
		  if (ev.xbutton.x >= cx
		      && ev.xbutton.y > (c->y - main_client_title_height(c))
		      && ev.xbutton.y < c->y)
		     client_deliver_delete(c);
		  else
		     c->redraw(c, True);
		  XUngrabPointer(c->wm->dpy, CurrentTime);
		  return;
	    }
	 }
      } else {
	 wu_drag(c);
      }
   }
}

void dialog_client_destroy(Client *c)
{
   base_client_destroy(c);
}
