/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "dockbar_client.h"

Client*
dockbar_client_new(wm *w, Window win)
{
   client *c = base_client_new(w, win); 

   c->type         = toolbar;
   c->configure    = &dockbar_client_configure;
   c->button_press = &dockbar_client_button_press;
   c->redraw       = &dockbar_client_redraw;
   c->show         = &dockbar_client_show;
   c->hide         = &dockbar_client_hide;
   c->get_coverage = &dockbar_client_get_coverage;
   c->move_resize  = &dockbar_client_move_resize;
   
   c->frame = c->window;
   return c;
}

void
dockbar_client_get_coverage(Client *c, int *x, int *y, int *w, int *h)
{
   int b = c->wm->theme->win_border_width;
   /* TODO - remember window borders - if any */
   *x = c->x; *y = c->y - b;
   *w = c->width + b; *h = c->height + b;
}

void
dockbar_client_configure(Client *c)
{
   int b = c->wm->theme->win_border_width;
   c->y = c->wm->dpy_height - c->wm->config->dock_maxheight - (2*b);
   c->x = 0;

   c->width  = c->wm->dpy_width - (2*b);
   c->height = c->wm->config->dock_maxheight;

   client_set_state(c, NormalState);
}

void
dockbar_client_move_resize(Client *c)
{
   base_client_move_resize(c);
   XResizeWindow(c->wm->dpy, c->frame, c->width, c->height);
   XMoveWindow(c->wm->dpy, c->frame, c->x, c->y);
}

void
dockbar_client_button_press(Client *c, XButtonEvent *e)
{
   int cx,cy;
   XEvent ev;
   XGCValues gc_vals;
   Theme *t = c->wm->theme;
   
   if (client_get_state(c) == NormalState
       && (e->x <= toolbar_win_offset(c)))
   {
      cx = cy = t->padding;

      if (XGrabPointer(c->wm->dpy, c->wm->root, False,
		       (ButtonPressMask|ButtonReleaseMask),
		       GrabModeAsync,
		       GrabModeAsync, None, c->wm->curs, CurrentTime)
	  != GrabSuccess)
	 return;

      /* paint 'inverted' button */
      gc_vals.function = GXnand;
      XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
      theme_paint_pxm(c->wm, c->frame, cx, cy,
		      &t->buttons[BUTTON_TOOLBAR]);
      XSync(c->wm->dpy, False);	    
      for (;;) 				      /* get release */
      {
	 XMaskEvent(c->wm->dpy,
		    (ButtonPressMask|ButtonReleaseMask), &ev);
	 switch (ev.type)
	 {
	    case ButtonRelease:
	       gc_vals.function = GXcopy;
	       XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
	       if (ev.xbutton.x < toolbar_win_offset(c)
		   && ev.xbutton.y > c->y)
		  dockbar_client_hide(c);
	       else
		  c->redraw(c, True);
	       XUngrabPointer(c->wm->dpy, CurrentTime);
	       return;
	 }
      }
      return;
   }
   else if (client_get_state(c) == IconicState)
   {
      dockbar_client_show(c);
   }
}

void
dockbar_client_show(Client *c) /*TODO: show and hide share common static func*/
{
   client *p;

   if (client_get_state(c) == NormalState) return;
   dbg("unshading y is %i ", c->y);

   c->height = c->wm->config->dock_maxheight;

   XGrabServer(c->wm->dpy);
   
   for (p=c->next; p != c; p=p->next) /* move docked apps back into place */
      if (p->type == docked)
	 p->show(p);

   wm_restack(c->wm, c, - (c->wm->config->dock_maxheight - c->wm->config->dock_minheight));
   c->y = c->wm->dpy_height - c->wm->config->dock_maxheight - (2*c->wm->theme->win_border_width);
   c->move_resize(c);
   client_set_state(c, NormalState);
   c->redraw(c, False);
   XUngrabServer(c->wm->dpy);
}

void
dockbar_client_hide(Client *c)
{
   client *p;

   if (client_get_state(c) == IconicState) return;
   XGrabServer(c->wm->dpy);
   
   for (p=c->next; p != c; p=p->next)	 /* hide all the dockapps */
      if (p->type == docked)
	 p->hide(p);
   
   c->y = c->wm->dpy_height-c->wm->config->dock_minheight-(c->wm->theme->win_border_width*2);
   c->height = c->wm->config->dock_minheight;
   c->move_resize(c);
   
   client_set_state(c, IconicState);
   wm_restack(c->wm, c, c->wm->config->dock_maxheight - c->wm->config->dock_minheight);
   c->redraw(c, False);
   XUngrabServer(c->wm->dpy);
}

void
dockbar_client_redraw(Client *c, Bool use_cache)
{
   int cx,cy, size;
   Theme *t = c->wm->theme;

   if (client_get_state(c) == NormalState)
   {
      size = c->wm->config->dock_maxheight;
   } else {
      size = c->wm->config->dock_minheight;
   }
	 
   cx = cy = t->padding;
   
   if (use_cache && c->backing != (Pixmap)NULL)
   {
      
      XCopyArea(c->wm->dpy, c->backing, c->frame,
		c->wm->theme->fg_gc, 0, 0, c->wm->dpy_width,
		size, 0, 0 );
      dbg("using pixmap cache for main title %s", c->name);
      return;
   }

   
   if (c->backing == (Pixmap)NULL)
      client_init_backing(c, c->wm->dpy_width, size);
   
   theme_render_area(c->wm, c->backing, 0,0,
		     c->wm->dpy_width, size );
   if (client_get_state(c) == NormalState)
      theme_paint_pxm(c->wm, c->backing, cx, cy,
		      &c->wm->theme->buttons[BUTTON_TOOLBAR]);

   XCopyArea(c->wm->dpy, c->backing, c->frame,
	     c->wm->theme->fg_gc, 0, 0, c->wm->dpy_width,
	     size, 0, 0 );

}


