/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

/*
  $Id: main_client.c,v 1.43 2002/03/13 00:20:40 mallum Exp $
*/

#include "main_client.h"


client*
main_client_new(wm *w, Window win)
{
   Client *c = base_client_new(w, win); 
   
   c->type = mainwin;
   c->reparent     = &main_client_reparent;
   c->redraw       = &main_client_redraw;
   c->button_press = &main_client_button_press;
   c->move_resize  = &main_client_move_resize;
   c->get_coverage = &main_client_get_coverage;
   c->hide         = &main_client_hide;
   c->show         = &main_client_show;
   c->configure    = &main_client_configure;
   c->destroy      = &main_client_destroy;

   if (w->flags & SINGLE_FLAG)
   {
      /* There was only one client till this came along */
      w->flags ^= SINGLE_FLAG; /* turn off single flag */      
      main_client_redraw(w->main_client, False);

   } else if (!w->main_client) /* This must be the old client*/
      w->flags ^= SINGLE_FLAG; /* turn on single flag */      
 
   return c;
}

void
main_client_configure(Client *c)
{
   int b = c->wm->theme->win_border_width;
   int h = wm_total_toolbar_height(c);
   /* force size */
   c->y = main_client_title_height(c);
   c->x = 0;
       
   c->width  = c->wm->dpy_width - ( 2*b );
   c->height = c->wm->dpy_height - c->y - h - ( 2*b );

   c->wm->main_client = c;
}

int
main_client_title_height(Client *c) /* was _theight(c) */ 
{
   if (!c->wm->config->use_title) return 0;
   
#ifdef USE_XFT
    return c->wm->theme->xftfont->ascent + c->wm->theme->xftfont->descent
       + (c->wm->theme->win_border_width*2) + (c->wm->theme->padding*2) ;
#else
    return c->wm->theme->font->ascent + c->wm->theme->font->descent
        + (c->wm->theme->padding*2) + (c->wm->theme->win_border_width*2);
#endif

}

void /* TOGO ? Not used ? */
main_client_get_coverage(Client *c, int *x, int *y, int *w, int *h)
{
   *x = 0; *y = 0;
   *w = c->width + WBW(c);
   *h = c->height + main_client_title_height(c);
}


void
main_client_reparent(Client *c)
{
    XSetWindowAttributes pattr;
    Theme *t = c->wm->theme;
   
    pattr.override_redirect = True;
    pattr.border_pixel = (t->bg_col).pixel;
    pattr.background_pixel = (t->bg_col).pixel;
    pattr.event_mask = ChildMask|ButtonPressMask|ExposureMask;

    c->frame = XCreateWindow(c->wm->dpy, c->wm->root,
			     0, 0,
			     c->width,
			     c->height + main_client_title_height(c),
			     c->wm->theme->win_border_width,
			     DefaultDepth(c->wm->dpy, c->wm->screen),
			     CopyFromParent,
			     DefaultVisual(c->wm->dpy, c->wm->screen),
			     CWOverrideRedirect|CWBorderPixel|
			     CWEventMask|CWBackPixel,
			     &pattr);

    // Broke for now
    // wu_allow_xstroke(c->wm, c->frame);
    
    // removed CWBackPixel| from above
    XAddToSaveSet(c->wm->dpy, c->window);
    XSelectInput(c->wm->dpy, c->window, ButtonPressMask|ColormapChangeMask|PropertyChangeMask);

    /* get rid of child borders */
    XSetWindowBorderWidth(c->wm->dpy, c->window, 0);

    XReparentWindow(c->wm->dpy, c->window, c->frame,
		    0, main_client_title_height(c));


}

void
main_client_move_resize(Client *c)
{
   base_client_move_resize(c);
   XResizeWindow(c->wm->dpy, c->window, c->width, c->height);
   XMoveResizeWindow(c->wm->dpy, c->frame, c->x,
		     c->y - + main_client_title_height(c), c->width,
		     c->height + main_client_title_height(c));
}

/* redraws the frame */
void
main_client_redraw(client *c, Bool use_cache)
{
   int nx,px,cx,ny,py,cy;
   Theme *t = c->wm->theme;

   if (!c->wm->config->use_title) return;
   
   //if (!c->mapped || client_get_state(c) != NormalState) return;
   
   if (use_cache && c->backing != (Pixmap)NULL)
   {
      
      XCopyArea(c->wm->dpy, c->backing, c->frame,
		c->wm->theme->fg_gc, 0, 0, c->width,
		c->height, 0, 0 );
      dbg("using pixmap cache for main title %s", c->name);
      return;
   }

   if (c->backing == (Pixmap)NULL)
      client_init_backing(c, c->width, main_client_title_height(c));
   
   /* Fill the background */
   theme_render_area(c->wm, c->backing, 0, 0,
		     c->width, main_client_title_height(c) );
    if (!(c->wm->flags & SINGLE_FLAG))
       theme_paint_pxm(c->wm, c->backing, t->padding, t->padding+WBW(c),
		       &t->buttons[BUTTON_MENU_SELECT]);
   
   /* Display title */
   if (c->name) {
      int space_avail = c->wm->dpy_width
	 - c->wm->theme->buttons[BUTTON_PREV].width
	 - c->wm->theme->buttons[BUTTON_NEXT].width
	 - c->wm->theme->buttons[BUTTON_CLOSE].width
	 - 3*BUTTON_SPACING
	 - t->buttons[BUTTON_MENU_SELECT].width
	 - (t->padding*2)
	 - 8; 			    
      
#ifdef USE_XFT
      XGlyphInfo       extents;
      XftTextExtents8(c->wm->dpy, t->xftfont,
		      (unsigned char *) c->name, strlen(c->name),
		      &extents);
      XftDrawString8(c->xftdraw, &(t->xft_fg), t->xftfont,
		     (t->padding*2) + t->buttons[BUTTON_MENU_SELECT].width,
		     t->padding + t->xftfont->ascent+WBW(c),
		     (unsigned char *) c->name,
		     wu_trim_title(c->wm, c->name, space_avail));
#else
      XDrawString(c->wm->dpy, c->backing, t->text_gc,
		  (t->padding*2) + t->buttons[BUTTON_MENU_SELECT].width,
		  t->padding+t->font->ascent+WBW(c),
		  c->name, wu_trim_title(c->wm, c->name, space_avail));
#endif

    }

    /* Buttons */
    px = c->width
       - c->wm->theme->buttons[BUTTON_PREV].width
       - c->wm->theme->buttons[BUTTON_NEXT].width
       - c->wm->theme->buttons[BUTTON_CLOSE].width
       - 3*BUTTON_SPACING;

    nx = c->width
       - c->wm->theme->buttons[BUTTON_NEXT].width
       - c->wm->theme->buttons[BUTTON_CLOSE].width
       - 2*BUTTON_SPACING;

    cx = c->width
       - c->wm->theme->buttons[BUTTON_CLOSE].width
       - BUTTON_SPACING;
    if (!(c->wm->flags & SINGLE_FLAG))
    {
       py = ((main_client_title_height(c) - t->buttons[BUTTON_PREV].height)/2);
       theme_paint_pxm(c->wm, c->backing, px, py, &t->buttons[BUTTON_PREV]);
    
       ny = (main_client_title_height(c) - t->buttons[BUTTON_NEXT].height)/2;
       theme_paint_pxm(c->wm, c->backing, nx, ny, &t->buttons[BUTTON_NEXT]);
    }
    cy = ((main_client_title_height(c) - t->buttons[BUTTON_CLOSE].height)/2);
    theme_paint_pxm(c->wm, c->backing, cx, cy, &t->buttons[BUTTON_CLOSE]);
    
    /* Border - at bottom of title */
    if (c->wm->theme->win_border_width)
       XDrawLine(c->wm->dpy, c->backing, t->bg_gc,
	          0,
		  main_client_title_height(c)-1,
		  c->width,
		 main_client_title_height(c)-1
		  );

    XCopyArea(c->wm->dpy, c->backing, c->frame,
	      c->wm->theme->fg_gc, 0, 0, c->width,
	      c->height, 0, 0 );

}

void main_client_button_press(client *c, XButtonEvent *e)
{
   client *p = NULL;
   Theme *t = c->wm->theme;
   XEvent ev;
   XGCValues gc_vals;
   unsigned long valuemask = 0;
          
   int px,nx,cx,tx,py,cy,ny,state;

#ifdef USE_XFT
   XGlyphInfo       extents;
   XftTextExtents8(c->wm->dpy, c->wm->theme->xftfont,
		   (unsigned char *) c->name, strlen(c->name),
		   &extents);
   tx = extents.width;
#else
   tx = XTextWidth(c->wm->theme->font, c->name, strlen(c->name)) ;
#endif

   if (!c->wm->config->use_title) return;
   
   tx += c->wm->theme->padding;
   
   px = c->width
      - c->wm->theme->buttons[BUTTON_PREV].width
      - c->wm->theme->buttons[BUTTON_NEXT].width
      - c->wm->theme->buttons[BUTTON_CLOSE].width
      - 3*BUTTON_SPACING;
   py = ((main_client_title_height(c) - t->buttons[BUTTON_PREV].height)/2);
   
   nx = c->width
      - c->wm->theme->buttons[BUTTON_NEXT].width
      - c->wm->theme->buttons[BUTTON_CLOSE].width
      - 2*BUTTON_SPACING;
   ny = (main_client_title_height(c) - t->buttons[BUTTON_NEXT].height)/2;
    
   cx = c->width
      - c->wm->theme->buttons[BUTTON_CLOSE].width
      - BUTTON_SPACING;
   cy = ((main_client_title_height(c) - t->buttons[BUTTON_CLOSE].height)/2);


   if (e->y <= main_client_title_height(c))
   {
      if (e->x >= px) 				      /* A button */
      {
	 if (XGrabPointer(c->wm->dpy, c->wm->root, False,
			  (ButtonPressMask|ButtonReleaseMask),
			  GrabModeAsync,
			  GrabModeAsync, None, c->wm->curs, CurrentTime)
	     != GrabSuccess)
	    return;

	 gc_vals.function = GXnand;
	 valuemask =  GCFunction;
	 XChangeGC(c->wm->dpy, t->mask_gc, valuemask, &gc_vals);

	 
	 if (e->x < cx)
	 {
	    /* check there are other main clients */
	    if (c->wm->flags & SINGLE_FLAG)
	    {
	       XUngrabPointer(c->wm->dpy, CurrentTime);
	       return;
	    }
	    if (e->x < nx)
	    {
	       state = BUTTON_PREV_PRESSED;
	       theme_paint_pxm(c->wm, c->frame, px, py,
			       &t->buttons[BUTTON_PREV]);
	    } else {
	       state = BUTTON_NEXT_PRESSED;
	       theme_paint_pxm(c->wm, c->frame, nx, ny,
			       &t->buttons[BUTTON_NEXT]);
	    }
	 } else { 				     
	    state = BUTTON_CLOSE_PRESSED;	 
	    theme_paint_pxm(c->wm, c->frame, cx, cy,
			    &t->buttons[BUTTON_CLOSE]);
	 }
	 
	 XSync(c->wm->dpy, False);	    
	 
	 for (;;)
	 {
	    dbg("looping for release");
	    XMaskEvent(c->wm->dpy,
		       (ButtonPressMask|ButtonReleaseMask), &ev);
	    
	    switch (ev.type)
	    {
	       case ButtonRelease:
		  dbg("Got release");
		  gc_vals.function = GXcopy;
		  valuemask =  GCFunction;
		  XChangeGC(c->wm->dpy, t->mask_gc, valuemask, &gc_vals);

		  if (ev.xbutton.y <= main_client_title_height(c)
		      && (ev.xbutton.x >= px))
		  {
		     if (ev.xbutton.x < cx )
		     {
			XGrabServer(c->wm->dpy);
			XSync(c->wm->dpy, False);	    
			if (ev.xbutton.x <= nx && state == BUTTON_PREV_PRESSED)
			{
			   dbg("raising prev main");
			   p = client_get_prev(c, mainwin);
			} else if (state == BUTTON_NEXT_PRESSED)
			{
			      p = client_get_next(c, mainwin);
			}
			if (p)
			{
			   c->hide(c); p->show(p);
			}
			XUngrabServer(c->wm->dpy);
		     } else { 				  
			if (state == BUTTON_CLOSE_PRESSED)
			{
			   client_deliver_delete(c);
			   XUngrabPointer(c->wm->dpy, CurrentTime);
			   return;
			}
		     }
		  }
		  XUngrabPointer(c->wm->dpy, CurrentTime);
		  c->redraw(c, True);
		  return;
	    }
	 }
      } else { /* drop down window */
	 
	 if ((c->wm->flags & MENU_FLAG) && (client_get_prev(c, mainwin) != c))
	 {
	    for(p = c->prev; p != c ; p = p->prev)
	    {
	       if (p->type == menu)
	       { select_client_destroy(p); return; }
	    }
	 } else {
	    if (e->x <= tx)
	       select_client_new(c->wm);
	 }
      }
   }
}


void
main_client_hide(Client *c)
{
   base_client_hide(c);

   /* lower window to bottom of stack, and set state to iconic */
   XLowerWindow(c->wm->dpy, c->frame);
   client_set_state(c, IconicState);

}

void
main_client_show(Client *c)
{

   dbg("main_client_show: called");
   
   client_set_state(c, NormalState);
   c->wm->main_client = c;
   c->mapped = True;
   
   XMapWindow(c->wm->dpy, c->window); 
   XMapRaised(c->wm->dpy, c->frame);

   /* check input focus */
   if (client_want_focus(c))
   {
      XSetInputFocus(c->wm->dpy, c->window,
		     RevertToPointerRoot, CurrentTime);
      c->wm->focused_client = c;
   }

   /* deal with transients etc */
   base_client_show(c);

}

void
main_client_destroy(Client *c)
{
   Wm *w = c->wm;

   /* What main clients are left?, need to decide what nav buttons appear */
   
   w->main_client = client_get_prev(c, mainwin);

   if(w->main_client == c)  /* Is this the only main client left? */
   {
      w->main_client = NULL;
      if (c->wm->flags & SINGLE_FLAG)
	 c->wm->flags ^= SINGLE_FLAG; /* turn off single flag */
      base_client_destroy(c);
      return;
   }
   
   base_client_destroy(c); 

   if (w->main_client == client_get_prev(w->main_client, mainwin))
   {  /* Is only 1 main_client left  */
      w->flags ^= SINGLE_FLAG; /* turn on single flag */      
      main_client_redraw(w->main_client, False);
   }

   w->main_client->show(w->main_client); /* show previous main client */
 
}




