/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mboundingbox.cpp,v 1.1 2000/10/09 17:55:51 jeh Exp $
 */
#define _MATH3D_EXPORT
#include <math3d/mboundingbox.h>
#include <math3d/m4x4.h>


/*!
 *
 */
Math3d::MBoundingBox::MBoundingBox()
{
}


/*!
 *
 */
Math3d::MBoundingBox::MBoundingBox(const M3d& p)
  : d_min(p), d_max(p)
{
}


/*!
 *
 */
Math3d::MBoundingBox::MBoundingBox(const M3d& p, const M3d& q)
{
  set(p,q);
}



/*!
 *
 */
Math3d::MBoundingBox::MBoundingBox(const MBoundingBox& b)
  : d_min(b.d_min), d_max(b.d_max)
{
}


/*!
 *
 */
const Math3d::MBoundingBox& 
Math3d::MBoundingBox::operator=(const MBoundingBox& b)
{
  d_min=b.d_min;
  d_max=b.d_max;
  return(*this);
}


/*!
 *
 */
void 
Math3d::MBoundingBox::copy(const MBoundingBox& b)
{
  d_min=b.d_min;
  d_max=b.d_max;
}


/*!
 *
 */
void
Math3d::MBoundingBox::set(const M3d& p, const M3d& q)
{
  d_min=p;
  d_min.min(q);
  d_max=p;
  d_max.max(q);
}


/*!
 *
 */
void
Math3d::MBoundingBox::extend(const M3d& p)
{
  d_min.min(p);
  d_max.max(p);
}


/*!
 *
 */
void
Math3d::MBoundingBox::extend(const MBoundingBox& box)
{
  d_min.min(box.d_min);
  d_max.max(box.d_max);
}


/*!
 *
 */
void
Math3d::MBoundingBox::transform(const M4x4& M)
{
  M3d p[8];
  p[0].set(d_min[0],d_min[1],d_min[2]);
  p[1].set(d_max[0],d_min[1],d_min[2]);
  p[2].set(d_max[0],d_max[1],d_min[2]);
  p[3].set(d_min[0],d_max[1],d_min[2]);
  p[4].set(d_min[0],d_min[1],d_max[2]);
  p[5].set(d_max[0],d_min[1],d_max[2]);
  p[6].set(d_max[0],d_max[1],d_max[2]);
  p[7].set(d_min[0],d_max[1],d_max[2]);
  int i;
  M.transform(p[0]);
  d_min=d_max=p[0];
  for (i=1; i<8; i++) {
    M.transform(p[i]);
    extend(p[i]);
  }
}


/*!
 *
 */
Math3d::M3d
Math3d::MBoundingBox::getCenter() const
{
  M3d r;
  r.add(d_max,d_min);
  r*=0.5;
  return(r);
}


/*!
 *
 */
Math3d::M3d
Math3d::MBoundingBox::getSize() const
{
  M3d r;
  r.sub(d_max,d_min);
  return(r);
}


/*!
 *
 */
bool
Math3d::MBoundingBox::isInside(const M3d& p) const
{
  int i;
  for (i=0; i<3; i++) {
    if ((d_min[i]>p[i]) && (p[i]>d_max[i])) return(FALSE);
  }
  return(TRUE);
}


/*!
 *
 */
bool 
Math3d::MBoundingBox::operator==(const MBoundingBox& b) const
{
  return(cmp(b));
}


/*!
 *
 */
bool 
Math3d::MBoundingBox::operator!=(const MBoundingBox& b) const
{
  return(!cmp(b));
}


/*!
 *
 */
bool 
Math3d::MBoundingBox::cmp(const MBoundingBox& b, double epsilon) const
{
  return(
    d_max.cmp(b.d_max, epsilon) &&
    d_min.cmp(b.d_min, epsilon)
  );
}
