#ifndef __GAME_ENGINE_H
#define __GAME_ENGINE_H
/*-------------------------------------------------------------------------
 * Copyright (c) 2000 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * game_engine
 *
 * Synopsis:
 *    The main engine of the MathWar game
 *
 * $Id: game_engine.h,v 1.1 2000/10/15 15:36:07 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to 
 * Free Software Foundation, Inc. 
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
/*! \file game_engine.h
 
  \brief The main engine of the MathWar game

  This engine is basically a state machine controlled by a couple of
  timeouts.

  \author Kenneth W. Sodemann <stufflehead@bigfoot.com>
  $Revision: 1.1 $
  $Date: 2000/10/15 15:36:07 $
*/

#ifndef __GAME_ENGINE_BODY
/*! \typedef game_engine
 *
 * The game_engine structure defines the current game parameters.  The
 * individual fields within this structure are not intended for use
 * outside the game engine.  Thus the definition of the structure is not
 * public.
 */
typedef struct game_engine game_engine;
#endif

/*! \fn game_engine *create_game_engine (GtkWidget  *parent)
 
  \brief Create and intialize a game engine instance 

  This function reads the user defined paramaters (if available), and
  initializes the games control structure accordingly.

  \param parent
  The toplevel parent widget of the game's user interface.

  \return
  The newly created and intialized game engine.
*/
game_engine *create_game_engine (GtkWidget  *parent);


/*! \fn void begin_game (game_engine *game)
 
  \brief Perform initialization required before each game.

  A "game" is defined as a full set of "rounds".  A "round" is
  two cards, and an operation.  The number of rounds in a game is
  settable by the user.

  This function should be called before the start of each game.

  \param game
  A pointer to the game instance.

  \return
  Nothing.
*/
void begin_game (game_engine *game);


/*! \fn void begin_round (game_engine *game)
   
  \brief Start a round

  Picks the cards and operator.  Determines if the computer will guess,
  and if so, when.  Starts the timeout to the actual start of the round.

  \param game
  A pointer to the game instance.

  \return
  Nothing.
*/
void begin_round (game_engine *game);


/*! \fn void destroy_game_engine (game_engine *game)
 
  \brief Destroy the game engine

  Stop all timers, free any memory used by the engine.

  \param game
  A pointer to the game instance.

  \return
  Nothing.
*/
void destroy_game_engine (game_engine *game);


/*! \fn void handle_player_answer (game_engine *game, const gchar *answer)
 
  \brief Handle the even of a player providing an answer

  \param game
  A pointer to the game instance.

  \param answer
  The player's answer.

  \return
  Nothing.
*/
void handle_player_answer (game_engine *game, const gchar *answer);

#endif
