/*
 * misc.c
 *
 * Copyright 1998, 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, version 2.
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "mconfig.h"



/*
 * malloc memory and check the result
 */

void           *
check_malloc(size_t size)
{
	void           *ptr = malloc(size);
	if (ptr == NULL)
		error_memory_exhausted();
	return ptr;
}



/*
 * realloc memory and check the result
 */

void           *
check_realloc(void *ptr, size_t size)
{
	ptr = realloc(ptr, size);
	if (ptr == NULL)
		error_memory_exhausted();
	return ptr;
}



/*
 * report a bogus enum
 */

void 
error_enum_bogus()
{
	error_internal("bogus enum encountered");
}



/*
 * report a generic error
 */

void 
error_exit(const char *message)
{
	const char     *cmdname = argument.cmdname;
	if (cmdname == NULL)
		cmdname = "unknown-program";

	fprintf(stderr, "%s: %s\n", cmdname, message);
	exit(1);
}



/*
 * report an internal error
 */

void 
error_internal(const char *message)
{
	const char     *cmdname = argument.cmdname;
	if (cmdname == NULL)
		cmdname = "unknown-program";

	fprintf(stderr, "%s: *** internal error: %s\n", cmdname, message);
	exit(1);
}



/*
 * report out of memory
 */

void 
error_memory_exhausted()
{
	const char     *cmdname = argument.cmdname;
	if (cmdname == NULL)
		cmdname = "unknown-program";

	write(2, cmdname, strlen(cmdname));
	write(2, ": *** out of memory ***\n", 24);
	_exit(1);
}



/*
 * report null pointer
 */

void 
error_pointer_null()
{
	error_internal("null pointer detected");
}



/*
 * report an error from a system call
 */

void 
error_system_call(const char *message)
{
	const int       errno_save = errno;
	const char     *cmdname = argument.cmdname;
	if (cmdname == NULL)
		cmdname = "unknown-program";

	fprintf(stderr, "%s: ", cmdname);
	errno = errno_save;
	perror(message);
	exit(1);
}



/*
 * Format an error string for an error returned by a system call,
 * such as "file not found."  This function returns a pointer to
 * a static buffer which is overwritten with each call.
 */

const char     *
format_system_error(int errno_save,
		    const char *name_1, const char *name_2)
{
	static stretch_type stretch;

	if (stretch.buf == NULL)
		stretch_ctor(&stretch, 256);
	stretch.len = 0;

	stretch_append(&stretch, name_1, strlen(name_1));

	if (name_2 != NULL) {
		stretch_append(&stretch, ", ", 2);
		stretch_append(&stretch, name_2, strlen(name_2));
	}
	stretch_append(&stretch, ": ", 2);

	{
		const char     *ptr = "unknown error";
		if (errno_save >= 0 || errno_save < sys_nerr)
			ptr = sys_errlist[errno_save];
		stretch_append(&stretch, ptr, strlen(ptr));
	}

	{
		char            buf[64];
		sprintf(buf, " (error %d)", errno_save);
		stretch_append(&stretch, buf, strlen(buf));
	}

	stretch_append(&stretch, "", 1);
	return stretch.buf;
}



/*
 * Fast memory grabbing.
 * This is dynamically allocated memory, never freed, never resized.
 * If I run out of grab-buffer then I call malloc (so the program runs
 *   correctly, just more slowly).
 */

static char     grab_buffer[512 * 1024];
char           *grab_start = grab_buffer;
char           *grab_end = grab_buffer + sizeof(grab_buffer);

void           *
grab_memory_hard(size_t size)
{
	void           *ptr = check_malloc(size);
	memset(ptr, '\0', size);
	return ptr;
}

const char     *
grab_string(const char *str, size_t len)
{
	char           *ptr = grab_memory(len + 1);
	memcpy(ptr, str, len + 1);
	return ptr;
}



/*
 * Method functions for stretch type.
 */

void 
stretch_append(stretch_type * stretch, const char *ptr, int len)
{
	while (stretch->len + len > stretch->size) {
		stretch->buf = check_realloc(stretch->buf,
			      (stretch->size *= 2) * sizeof(*stretch->buf));
	}

	memcpy(stretch->buf + stretch->len, ptr, len);
	stretch->len += len;
}

void 
stretch_ctor(stretch_type * stretch, int size)
{
	stretch->buf = check_malloc(size * sizeof(*stretch->buf));
	stretch->len = 0;
	stretch->size = size;
}

void 
stretch_dtor(stretch_type * stretch)
{
	free(stretch->buf);
	stretch->buf = NULL;
}
