#!/usr/bin/tclsh
# Part of MCU 8051 IDE ( http://mcu8051ide.sf.net )

############################################################################
#    Copyright (C) 2007-2009 by Martin Ošmera                              #
#    martin.osmera@gmail.com                                               #
#                                                                          #
#    This program is free software; you can redistribute it and#or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

# --------------------------------------------------------------------------
# DESCRIPTION
# Provides facility to show tips on startup
#	* Tips are readed from file deindef in NS variable "tips_file"
#	* Format of definition file is XML and it supports mutiple languages
#	* Usage is simple: execute procedure "::Tips::show_tip_of_the_day_win"
#	* It requires NS ConfigDialogs (see ${::GLOBAL_CONFIG(tips)})
# --------------------------------------------------------------------------

namespace eval Tips {
	variable tip_of_the_day_win		;# ID of window "Tip of the day"
	variable tip_of_the_day_text		;# ID of text widget in "Tip of the day"
	variable tip_of_the_day_show_again	;# Bool: Show "Tip of the day"

	variable tips_data			;# List containing tips data
	variable number_of_tips			;# Number of tips avaliable
	variable current_tip			;# Number of the currently displayed tip
	variable expected			;# Expeceted element
	variable take_data			;# Bool: Append data section to $tips_data
	# File containing tips data
	variable tips_file			"${::LIB_DIRNAME}/../data/tips.xml"

	## Invoke dialog "Tip on startup"
	 # @return void
	proc show_tip_of_the_day_win {} {
		variable tip_of_the_day_win		;# ID of window "Tip of the day"
		variable tip_of_the_day_text		;# ID of text widget in "Tip of the day"
		variable tip_of_the_day_show_again	;# Bool: Show "Tip of the day"
		variable number_of_tips			;# Number of tips avaliable
		variable tip_of_the_day_show_again	;# Bool: Show "Tip of the day"

		# Set value of checkbox "Show again"
		set tip_of_the_day_show_again ${::GLOBAL_CONFIG(tips)}
		# Load tips definition file
		load_tips_file

		# Create toplevel window
		set win [toplevel .tip_of_the_day -class {Tip of the day} -bg {#EEEEEE}]
		set tip_of_the_day_win $win

		# Create window header
		pack [label $win.header			\
			-text [mc "Did you know ... "]	\
			-font [font create	\
				-family {times}	\
				-size -25	\
				-weight bold	\
			]	\
			-compound right			\
			-image ::ICONS::32::help	\
		] -pady 5

		# Create middle frame (text windget and scrollbar)
		set middle_frame [frame $win.middle_frame]
		set text [text $middle_frame.text	\
			-width 0 -height 0 -bg white	\
			-wrap word			\
			-yscrollcommand "$middle_frame.scrollbar set"	\
			-font [font create		\
				-family {helvetica}	\
				-size -17		\
				-weight normal		\
			]	\
		]
		pack $text -side left -fill both -expand 1
		pack [ttk::scrollbar $middle_frame.scrollbar	\
			-orient vertical			\
			-command [list $text yview]		\
		] -side left -fill y -after $text
		set tip_of_the_day_text $text

		## Create bottom frame
		set bottom_frame [frame $win.bottom_frame]
		# - CheckButton "Show tips on startup"
		pack [checkbutton $bottom_frame.chbutton		\
			-variable ::Tips::tip_of_the_day_show_again	\
			-command {::Tips::tip_otd_show_again}		\
			-text [mc "Show tips on startup"]		\
		] -side left -anchor e
		# - Button "Close"
		pack [ttk::button $bottom_frame.close_but	\
			-compound left				\
			-text [mc "Close"]			\
			-image ::ICONS::16::button_cancel	\
			-command {::Tips::tip_otd_CLOSE}	\
			-width 8				\
		] -side right -anchor w
		# - Button "Next"
		pack [ttk::button $bottom_frame.next_but	\
			-compound left				\
			-text [mc "Next"]			\
			-image ::ICONS::16::right		\
			-command {::Tips::tip_otd_NEXT}		\
			-width 8				\
		] -side right -anchor w
		# - Button "Previous"
		pack [ttk::button $bottom_frame.prev_but	\
			-compound left				\
			-text [mc "Previous"]			\
			-image ::ICONS::16::left		\
			-command {::Tips::tip_otd_PREV}		\
			-width 8				\
		] -side right -anchor w

		# Pack window frames
		pack $middle_frame -side top -fill both -expand 1 -padx 10 -pady 5
		pack $bottom_frame -side bottom -fill x -after $middle_frame -padx 10 -pady 5

		# Configure text tags
		$text tag configure tag_bold -font [font create	\
			-family {helvetica}	\
			-size -17		\
			-weight bold		\
		]

		# Determinate random number of tip to show
		expr {srand([clock seconds])}
		display_tip [expr {int(rand() * $number_of_tips)}]

		# Configure dialog window
		wm iconphoto $win ::ICONS::16::info
		wm title $win [mc "Tip of the day - MCU 8051 IDE"]
		wm minsize $win 520 250
		wm protocol $win WM_DELETE_WINDOW {
			::Tips::tip_otd_CLOSE
		}
		wm transient $win .
		raise $win
		catch {
			grab $win
		}
	}

	## Load definition of tips
	 # @return void
	proc load_tips_file {} {
		variable tips_data	;# List containing tips data
		variable number_of_tips	;# Number of tips avaliable
		variable tips_file	;# File containing tips data
		variable expected	;# Expeceted element
		variable take_data	;# Bool: Append data section to $tips_data

		# Initialize NS variables
		set take_data		0
		set number_of_tips	0
		set expected		{tips}
		set tips_data		{}

		# Open file
		if {[catch {
			set file [open $tips_file {r}]
		}]} {
			tk_messageBox		\
				-parent .	\
				-type ok	\
				-icon warning	\
				-title "tips.xml"	\
				-message [mc "Unable to open file containing tips,\nplease check your installation"]
			return
		}

		# Create XML parser
		set parser [::xml::parser -final 1 -ignorewhitespace 1		\
			-elementstartcommand ::Tips::xml_data_parser_element	\
			-characterdatacommand ::Tips::xml_data_parser_data	\
		]

		# Start XML parser
		if {[catch {
			$parser parse [read $file]
		} result]} {
			set number_of_tips 0
			set tips_data {}
			tk_messageBox		\
				-parent .	\
				-type ok	\
				-icon warning	\
				-title [mc "Unable to parse tips.xml"]	\
				-message [mc "File tips.xml is corrupted,\nplease check your installation"]
			puts stderr $result
			return
		}

		# Close file and free parser
		close $file
		$parser free
	}

	## Universal parser handler - handles XML tags and data
	 # @parm String arg1 - content of the element
	 # @return void
	proc xml_data_parser_data {arg1} {
		variable tips_data	;# List containing tips data
		variable number_of_tips	;# Number of tips avaliable
		variable current_tip	;# Number of the currently displayed tip
		variable expected	;# Expeceted element
		variable take_data	;# Bool: Append data section to $tips_data

		if {!$take_data} {
			return
		}

		set take_data 0
		incr number_of_tips

		regsub -all {^\s+} $arg1 {} arg1
		regsub -all {\s+$} $arg1 {} arg1
		lappend tips_data [regsub -all -line {^\t+} $arg1 {}]
	}

	## XML parser handler - handles XML tags
	 # @parm String arg1	- name of the element
	 # @parm List attrs	- list of attributes '{attr0 val0 attr1 val1 ...}'
	 # @return void
	proc xml_data_parser_element {arg1 attrs} {
		variable tips_data	;# List containing tips data
		variable number_of_tips	;# Number of tips avaliable
		variable current_tip	;# Number of the currently displayed tip
		variable expected	;# Expeceted element
		variable take_data	;# Bool: Append data section to $tips_data

		if {$arg1 != $expected} {
			error "Bad element `$arg1'"
		}
		if {$arg1 == {tips}} {
			set expected {tip}
		}

		# Iterate over element attributes
		for {set i 0} {$i < [llength $attrs]} {incr i} {
			if {[lindex $attrs $i] != {lang}} {
				incr i
				continue
			}
			incr i

			## Take data if ...
			 # No translation has been loaded and current text is in English
			if {!${::TRANSLATION_LOADED} || [lindex $attrs $i] == {en}} {
				set take_data 1
			 # Or if some translation has been loaded and it conforms with the text
			} elseif {[lindex $attrs $i] == ${::GLOBAL_CONFIG(language)}} {
				set take_data 1
			}
		}
	}

	## Close dialog
	 # @return void
	proc tip_otd_CLOSE {} {
		variable tips_data		;# List containing tips data
		variable number_of_tips		;# Number of tips avaliable
		variable current_tip		;# Number of the currently displayed tip
		variable tip_of_the_day_win	;# ID of window "Tip of the day"

		# Remove dialog
		grab release $tip_of_the_day_win
		destroy $tip_of_the_day_win

		# Free dialog resources
		set tips_data		{}
		set number_of_tips	{}
		set current_tip		{}
	}

	## Display tip with the given number in the window
	 # @parm Int tip_number - number of the tip to show (can overlap allowed range)
	 # @return void
	proc display_tip {tip_number} {
		variable tips_data		;# List containing tips data
		variable number_of_tips		;# Number of tips avaliable
		variable current_tip		;# Number of the currently displayed tip
		variable tip_of_the_day_text	;# ID of text widget in "Tip of the day"

		set current_tip $tip_number

		# Clear text widget
		$tip_of_the_day_text configure -state normal
		$tip_of_the_day_text delete 1.0 end

		# Validate tip number
		if {!$number_of_tips} {
			$tip_of_the_day_text configure -state disabled
			return
		}
		if {$tip_number >= $number_of_tips} {
			set current_tip $number_of_tips
			incr current_tip -1
		}

		# Create map of bold font tags
		set bold_tag_map {}
		set content [lindex $tips_data $current_tip]
		while 1 {
			set tag_pair {}

			set idx [string first {<b>} $content]
			if {$idx == -1} {break}
			regsub {<b>} $content {} content
			lappend tag_pair $idx

			set idx [string first {</b>} $content]
			if {$idx == -1} {break}
			regsub {</b>} $content {} content
			lappend tag_pair $idx

			lappend bold_tag_map $tag_pair
		}

		# Fill text widget
		set start [$tip_of_the_day_text index insert]
		$tip_of_the_day_text insert end $content
		foreach pair $bold_tag_map {
			$tip_of_the_day_text tag add tag_bold $start+[lindex $pair 0]c $start+[lindex $pair 1]c
		}
		$tip_of_the_day_text configure -state disabled
	}

	## Show next tip
	 # @return void
	proc tip_otd_NEXT {} {
		variable number_of_tips	;# Number of tips avaliable
		variable current_tip	;# Number of the currently displayed tip

		incr current_tip
		if {$current_tip >= $number_of_tips} {
			set current_tip 0
		}
		display_tip $current_tip
	}

	## Show previous tip
	 # @return void
	proc tip_otd_PREV {} {
		variable number_of_tips	;# Number of tips avaliable
		variable current_tip	;# Number of the currently displayed tip

		incr current_tip -1
		if {$current_tip < 0} {
			set current_tip [expr {$number_of_tips - 1}]
		}
		display_tip $current_tip
	}

	## Adjust base configuration file to variable "tip_of_the_day_show_again"
	 # @return void
	proc tip_otd_show_again {} {
		variable tip_of_the_day_show_again	;# Bool: Show "Tip of the day"

		::configDialogs::global::set_variable tips $tip_of_the_day_show_again
	}
}
