/*************************************************************************
* COPYRIGHT (C) 1999 - 2006  EDF R&D
* THIS LIBRARY IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
* IT UNDER THE TERMS OF THE GNU LESSER GENERAL PUBLIC LICENSE 
* AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
* EITHER VERSION 2.1 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
*  
* THIS LIBRARY IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
* WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
* MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
* LESSER GENERAL PUBLIC LICENSE FOR MORE DETAILS.
*
* YOU SHOULD HAVE RECEIVED A COPY OF THE GNU LESSER GENERAL PUBLIC LICENSE
* ALONG WITH THIS LIBRARY; IF NOT, WRITE TO THE FREE SOFTWARE FOUNDATION,
* INC., 59 TEMPLE PLACE, SUITE 330, BOSTON, MA 02111-1307 USA
*
*************************************************************************/

#include <med.h>
#include <med_config.h>
#include <med_outils.h> 
#include <string.h>
#include <stdlib.h>

/*
 * - Nom de la fonction : MEDchampEcr
 * - Description : ecriture d'un Champ Rsultat
 * - Parametres :
 *     - fid      (IN)  : ID du fichier HDF courant
 *     - maa      (IN)  : le nom du maillage sur lequel s'applique le champ (eventuellement distant)
 *     - cha      (IN)  : le nom du champ 
 *     - val      (IN)  : valeurs du champ  stocker
 *     - interlace(IN)  : entrelacement utilis en mmoire {MED_FULL_INTERLACE,MED_NO_INTERLACE} 
 *     - nbelem   (IN)  : nombre d'lments (prend en compte le nbre 
 *                         de points de Gauss (c'est demand  l'utilisateur ds la doc) 
                           mais pas le nbre de composantes)
 *     - locname  (IN)  : cl utilise pour la dfinition de la localisation 
                          des points de GAUSS (MED_NOGAUSS si aucun, MED_GAUSS_ELNO si les points de Gauss
 *                        portent sur les noeuds de l'element). La localisation doit exister 
 *                        avant l'appel  MEDchampEcr.
 *     - numco    (IN)  : n de la composante  stocker (MED_ALL si toutes)
 *     - profil   (IN)  : nom du profil utilis (MED_NOPFL si inutilis)
 *     - pflmod   (IN)  : Indique comment lire les informations en mmoire { MED_COMPACT, MED_GLOBAL }. 
 *     - type_ent (IN)  : entit concern par le champ {MED_NOEUD,MED_ARETE,MED_FACE,MED_MAILLE}
 *     - type_geo (IN)  : type gomtrique de l'entit concern {MED_POINT,MED_SEG2 ......}
 *     - numdt    (IN)  : n du pas de temps (MED_NOPDT si aucun)
 *     - dt_unit  (IN)  : chaine de taille MED_NOMP indiquant l'unit du champ
 *     - dt       (IN)  : valeur du pas de temps 
 *     - numo     (IN)  : n d'ordre utilis MED_NONOR si inutile
 * - Resultat : 0 en cas de succes, -1 sinon
 */ 

med_err MEDchampEcr(med_idt fid, char *maa, char *cha,
		    unsigned char *val, med_mode_switch interlace, med_int nbelem, char * locname, 
		    med_int numco, char * profil, med_mode_profil pflmod, 
		    med_entite_maillage type_ent, med_geometrie_element type_geo, 
		    med_int numdt,char * dt_unit, med_float dt, med_int numo)
{
  med_err ret=-1;
  med_idt gid=0,datagroup1=0,datagroup2=0,datagroup3=0,attr=0,gid_loc=0;
  med_int ncomp=0, chtype=0, i=0, ngauss=0, pfluse=0;
  char nomdatagroup1[2*MED_TAILLE_NOM_ENTITE+2]="";
  char nomdatagroup2[2*MED_MAX_PARA+1]="";
  char pflname   [MED_TAILLE_NOM+1]="";
  char locname_i [MED_TAILLE_NOM+1]="";
  char maillage[MED_TAILLE_NOM+1]="";
  char tmp1[MED_TAILLE_NOM_ENTITE+1]="";
  med_size dimd[1],psize=0,lsize=0;
  med_int   *pfltabtmp=0;
  med_size *pfltab=0;
  char chemin[MED_TAILLE_CHA+MED_TAILLE_NOM+1]="";
  char chemin_loc[MED_TAILLE_GAUSS+MED_TAILLE_NOM+1]=""; 
  med_geometrie_element type_geo_g;

  /*
   * On inhibe le gestionnaire d'erreur HDF 5
   */
  _MEDmodeErreurVerrouiller();

  if (MED_MODE_ACCES == MED_LECTURE) {
    MESSAGE("Impossible d'crire un champ en mode MED_LECTURE : ");
    goto ERROR;
  };


  /* 
   * Si le Data Group CHA n'existe pas => erreur
   */
  strcpy(chemin,MED_CHA);
  strcat(chemin,cha);
  if ((gid = _MEDdatagroupOuvrir(fid,chemin)) < 0)
    goto ERROR;

  /* Lecture de l'attribut MED_NOM_NCO */
  if (_MEDattrEntierLire(gid,MED_NOM_NCO,&ncomp) < 0)
    goto ERROR;

  /* Lecture de l'attribut MED_NOM_TYP */
  if ( _MEDattrEntierLire(gid,MED_NOM_TYP,&chtype) < 0)
    goto ERROR;
 

  
  /* 
   *  Creation/Ouverture du datagroup de niveau 1 <type_ent>[.<type_geo>] 
   */
  
  if (_MEDnomEntite(nomdatagroup1,type_ent) < 0)
    goto ERROR;
  if ((type_ent != MED_NOEUD))
    {
      if ( _MEDnomGeometrie(tmp1,type_geo) < 0)
	goto ERROR;
      strcat(nomdatagroup1,".");
      strcat(nomdatagroup1,tmp1);
    }
  datagroup1 = 0;
  if ( (datagroup1 = _MEDdatagroupOuvrir(gid,nomdatagroup1)) < 0) 
    if ((datagroup1 = _MEDdatagroupCreer(gid,nomdatagroup1)) < 0)
      goto ERROR;
  

  /* 
   * Creation/Ouverture du datagroup de niveau 2 <numdt>.<numoo> 
   */
 
  sprintf(nomdatagroup2,"%*li%*li",MED_MAX_PARA,(long ) numdt,MED_MAX_PARA,(long ) numo);

  datagroup2 = 0;   
  if ( (datagroup2 = _MEDdatagroupOuvrir(datagroup1,nomdatagroup2)) < 0 )    
    if ((datagroup2 = _MEDdatagroupCreer(datagroup1,nomdatagroup2)) < 0 )   
      goto ERROR;   
  
  /*Cree ou ouvre l'attribut MED_NOM_NDT pour criture */
  if ( _MEDattrEntierEcrire(datagroup2,MED_NOM_NDT,&numdt) < 0)
    goto ERROR;
  
  /*Cree ou ouvre l'attribut MED_NOM_PDT pour criture */
  if ( _MEDattrFloatEcrire(datagroup2,MED_NOM_PDT,&dt) < 0)
    goto ERROR;
    
  /*Cree ou ouvre l'attribut MED_NOM_NOR pour criture */
  if (_MEDattrEntierEcrire(datagroup2,MED_NOM_NOR,&numo) < 0)
    goto ERROR;
  
  /*Cree ou ouvre l'attribut  MED_NOM_UNI pour criture */ 
  if ( numdt == MED_NOPDT ) {
    if ( _MEDattrStringEcrire(datagroup2,MED_NOM_UNI,MED_TAILLE_PNOM,MED_PNOM_BLANC) < 0)
      goto ERROR;
  } else
    if ( _MEDattrStringEcrire(datagroup2,MED_NOM_UNI,MED_TAILLE_PNOM,dt_unit) < 0)
      goto ERROR;  


  /*
   * Si c'est la premire rfrence  un maillage, initialise l'attribut MED_MAA  ce maillage
   */
  if ((attr = _MEDattrOuvrir(datagroup2,MED_NOM_MAI)) < 0 ) {
    if (_MEDattrStringEcrire(datagroup2,MED_NOM_MAI,MED_TAILLE_NOM,maa) < 0)
      goto ERROR;
  } else {   
    if ( _MEDattrFermer(attr) < 0) goto ERROR;
  }
      
  

  /* 
   * Cree ou ouvre le datagroup de niveau 3 <maa> 
   */
  datagroup3 = 0;   
  if (((datagroup3 = _MEDdatagroupOuvrir(datagroup2,maa)) > 0)    
      && ( MED_MODE_ACCES == MED_LECTURE_AJOUT )) {
    MESSAGE("Impossible de crer une reference  un maillage existant en mode MED_LECTURE_AJOUT :");
    goto ERROR;   
  } else   
    if (datagroup3 < 0)
      if ((datagroup3 = _MEDdatagroupCreer(datagroup2,maa)) < 0)   
	goto ERROR;   
  
  /*Cree ou ouvre l'attribut MED_NOM_NBR */
  if ( _MEDattrEntierEcrire(datagroup3,MED_NOM_NBR,&nbelem) < 0)
    goto ERROR;

  /*Cree ou ouvre  l'attribut MED_NOM_PFL   */ 
  pfluse = 0;
  if ( strlen(profil) == 0)   /* idem MED_NOPFL*/
    /* Ecriture de MED_NOPFLi car MED_NOM_PFL est forcment de taille MED_TAILLE_NOM*/
    strncpy(pflname,MED_NOPFLi,MED_TAILLE_NOM+1);
  else {
    strncpy(pflname,profil,MED_TAILLE_NOM);
    pflname[MED_TAILLE_NOM]='\0';
    pfluse = 1;
  }
  if ( _MEDattrStringEcrire(datagroup3,MED_NOM_PFL,MED_TAILLE_NOM,pflname) < 0){
    MESSAGE("Erreur d'criture de l'attribut pflname : ");
    SSCRUTE(chemin); goto ERROR;
  }

  /* Lit le nombre de points de gauss et vrifie   */
  /* que la localisation porte sur le meme type gomtrique  */
  
  if ( strlen(locname) == 0) {
    /* Ecriture de MED_NOGAUSSi car MED_NOM_GAUSS est forcment de taille MED_TAILLE_NOM*/
    strcpy(locname_i,MED_NOGAUSSi);
    ngauss = MED_NOPG ;
  } else if (! strcmp(locname,MED_GAUSS_ELNO)) { 
    /* Les points de Gauss sont dfinis sur les noeuds de l'element (mot cle) */
    /* le nombre de points de Gauss est egal au nombre de noeuds de l'element */
    ngauss = type_geo % 100;
    strcpy(locname_i,locname);

  } else { 
    strcpy(locname_i,locname);

    strcpy(chemin_loc,MED_GAUSS);
    strcat(chemin_loc,locname_i);
    
    if ((gid_loc = _MEDdatagroupOuvrir(fid,chemin_loc)) < 0) {
      MESSAGE("Ouverture du datagroup : ");
      SSCRUTE(chemin_loc); goto ERROR;
    }
    
    if (_MEDattrEntierLire(gid_loc,MED_NOM_NBR,&ngauss) < 0) {
      MESSAGE("Erreur  la lecture de l'attribut MED_NOM_NBR : ");
      ISCRUTE(ngauss);goto ERROR;
    };
    
    
    if (_MEDattrEntierLire(gid_loc,MED_NOM_GEO,&type_geo_g) < 0) {
      MESSAGE("Erreur  la lecture de l'attribut MED_NOM_GEO : ");
      ISCRUTE(type_geo);goto ERROR;
    };
    
    if ( type_geo_g != type_geo ) {
      MESSAGE("Erreur, la localisation ne porte pas sur le meme type gomtrique : ");
      SSCRUTE(locname);ISCRUTE(type_geo);ISCRUTE(type_geo_g);goto ERROR;
    };
    
  }
  
  /* Cree ou ouvre l'attribut MED_NOM_GAU         */ 
  /* Ecriture de la localisation des pts de gauss  */
  if ( _MEDattrStringEcrire(datagroup3,MED_NOM_GAU,MED_TAILLE_NOM,locname_i) < 0) {
    MESSAGE("Erreur d'criture de l'attribut MED_NOM_GAU : ");
    SSCRUTE(locname); goto ERROR;
  }

  /* Cree ou ouvre l'attribut MED_NOM_NGA         */ 
  /* Ecriture de l'attribut portant le nombre de points de gauss  */
  if ( _MEDattrEntierEcrire(datagroup3,MED_NOM_NGA,&ngauss) < 0) {
    MESSAGE("Erreur d'criture de l'attribut MED_NOM_NGA : ");
    SSCRUTE(ngauss); goto ERROR;
  }

  /*Determination de la taille dimd[0] du dataset  stocker*/
  dimd[0] = nbelem*ncomp;

  /* Gestion des profils*/
  if ( pfluse ) {
   
    if ( ( i = MEDnValProfil(fid,pflname) ) < 0 ) {
      MESSAGE("Erreur  la lecture du nombre de valeurs sur le profil : ");
      SSCRUTE(pflname);
      goto ERROR;
    } else
      psize = i;
    
    pfltabtmp = (med_int *)   malloc (sizeof(med_int)*psize);
    pfltab    = (med_size *) malloc (sizeof(med_size)*psize);
    if (MEDprofilLire(fid,pfltabtmp,pflname) < 0) {
      MESSAGE("Erreur  la lecture du profil : ");
      SSCRUTE(pflname);goto ERROR;
    };
    for (i=0;i<psize;i++)
      pfltab[i] = (med_size) pfltabtmp[i];
    
  }
  else
    psize = MED_NOPF;
  
  
  /*
   * Ecriture du champ
   */
 
  switch(chtype)
    {
    case MED_FLOAT64 :
      if ( _MEDdatasetNumEcrire(datagroup3,MED_NOM_CO,MED_FLOAT64,interlace,
				ncomp,numco,psize,pflmod,pfltab,ngauss,dimd,val) < 0) {
	MESSAGE("Impossible d'ecrire le dataset : ");
	SSCRUTE(MED_NOM_CO);ISCRUTE(dimd); goto ERROR;
      }
      break;
      
    case MED_INT32 :
#if defined(HAVE_F77INT64) 
      if ( _MEDdatasetNumEcrire(datagroup3,MED_NOM_CO,MED_INT64,interlace,
				ncomp,numco,psize,pflmod,pfltab,ngauss,dimd,val) < 0) {
	MESSAGE("Impossible d'ecrire le dataset : ");
	SSCRUTE(MED_NOM_CO);ISCRUTE(dimd); goto ERROR;
      }
#else
      if ( _MEDdatasetNumEcrire(datagroup3,MED_NOM_CO,MED_INT32,interlace,
				ncomp,numco,psize,pflmod,pfltab,ngauss,dimd,val) < 0){
	MESSAGE("Impossible d'ecrire le dataset : ");
	SSCRUTE(MED_NOM_CO);ISCRUTE(dimd); goto ERROR;
      }
#endif
      break;

    case MED_INT64 :
#if defined(HAVE_F77INT64) 
      if ( _MEDdatasetNumEcrire(datagroup3,MED_NOM_CO,MED_INT64,interlace,
				ncomp,numco,psize,pflmod,pfltab,ngauss,dimd,val) < 0){
	MESSAGE("Impossible d'ecrire le dataset : ");
	SSCRUTE(MED_NOM_CO);ISCRUTE(dimd); goto ERROR;
      }
#else
      MESSAGE("Impossible d'ecrire le dataset de type MED_INT64 sur une plateforme autre que IRIX64 et OSF1 !");
      goto ERROR;
#endif
      break;   

    default :
      goto ERROR;
    }

  /*
   * On ferme tout 
   */

  ret = 0;

 ERROR:
  
  if ( pfluse ) { free(pfltab); free(pfltabtmp);}
  
  if (datagroup3>0)     if (_MEDdatagroupFermer(datagroup3) < 0) {
    MESSAGE("Impossible de fermer le datagroup : ");
    ISCRUTE(datagroup3); ret = -1; 
  }
  
  if (datagroup2>0)     if (_MEDdatagroupFermer(datagroup2) < 0) {
    MESSAGE("Impossible de fermer le datagroup : ");
    ISCRUTE(datagroup2); ret = -1; 
  }

  if (datagroup1>0)     if (_MEDdatagroupFermer(datagroup1) < 0) {
    MESSAGE("Impossible de fermer le datagroup : ");
    ISCRUTE(datagroup1); ret = -1; 
  }
  
  if (gid>0)     if (_MEDdatagroupFermer(gid) < 0) {
    MESSAGE("Impossible de fermer le datagroup : ");
    ISCRUTE(gid); ret = -1; 
  }
  
  if (gid_loc>0)     if (_MEDdatagroupFermer(gid_loc) < 0) {
    MESSAGE("Impossible de fermer le datagroup : ");
    ISCRUTE(gid_loc); ret = -1; 
  }
  return ret; 
}

