/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: dictionary.c,v 1.2 2003/02/05 04:38:37 kos Exp $
 *
 */

#include "config.h"

#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <stdlib.h>

#include "array.h"
#include "common.h"
#include "keyspace.h"
#include "rules.h"
#include "xmalloc.h"
#include "llog.h"
#include "net.h"
#include "generator.h"
#include "dictionary.h"

static void
makesafe_filename(char *f, int len) {
  char *c;

  for(c=f; *c && c<f+len; c++)
    if(*c == PATH_SEPARATOR)
      *c = '_';  
}

int
dictionary_destroy(dictionary_t *d) {
  array_nuke(d->rules);
  array_nuke(d->words);
  keyspace_destroy(d->index);
  keyspace_destroy(d->minindex);
  keyspace_destroy(d->maxindex);
  xfree(d);
  return 0;
}

static int
filefetch_conventional(dictionary_t *d, char *file, array *a) {
  FILE *fp;
  char line[DICT_LINELEN];

  if(!(fp = fopen(file, "r")))
    return 1;

  llog(4, "filefetch_conventional: loading %s\n", file);
  
  memset(line, '\0', WORD_LINELEN);
  while(fgets(line, WORD_LINELEN, fp)) {
    zap_whitespace(line, WORD_LINELEN, ZAP_LEFT|ZAP_RIGHT);
    if(*line == '#')
      continue;
    line[d->maxlen+1] = '\0';
    array_add(a, line);
    memset(line, '\0', WORD_LINELEN);
  }
  fclose(fp);
  return 0;
}

static int
filefetch_network(dictionary_t *d, char *file, array *a) {
  net_t *n;
  net_conn_t *c;
  msg *p;
  FILE *fp;
  unsigned long len;
  unsigned long elen;
  unsigned long tlen;
  char buffer[DICT_LINELEN];

  if(!(n = net_init(NET_CLIENT, d->fileserver, atoi(d->fileport)))) {
    return 1;
  }
  if(!(c = net_connect(n))) {
    net_destroy(n);
    return 2;
  }
  if(!(p = net_recv(c))) {
  ff_err:
    net_conn_destroy(c);
    net_destroy(n);
    return 3;
  }
  msg_destroy(p);
  p = msg_new();
  msg_add(p, "fetch", 0);
  msg_add(p, file, 0);
  net_send(c, p);
  if(!(p = net_recv(c)))
    goto ff_err;
  if(atoi(msg_get(p, 0)) != P_MORE)
    goto ff_err;
  
  makesafe_filename(file, DICT_LINELEN);
  if(!(fp = fopen(file, "w")))
    goto ff_err;

  elen = atol(msg_get(p, 1));
  msg_destroy(p);

  llog(4, "filefetch_network: downloading %s\n", file);
  len = 0;
  do {
    if((tlen = fread(buffer, 1, MIN(elen-len, DICT_LINELEN), net_rfp(c))) <= 0)
      goto ff_err;    
    fwrite(buffer, 1, tlen, fp);
    len += tlen;

    zap_whitespace(buffer, DICT_LINELEN, ZAP_LEFT|ZAP_RIGHT);
    buffer[d->maxlen+1] = '\0';
    array_add(a, buffer);
    memset(buffer, '\0', DICT_LINELEN);

  } while(len < elen);
  fclose(fp);

  net_conn_setflags(c, NET_F_DEFAULT);
  p = msg_new();
  msg_add(p, "quit", 0);
  net_send(c, p);
  msg_destroy(p);
  net_conn_destroy(c);
  net_destroy(n);
  
  return 0;
}

static int
filefetch(dictionary_t *d, int which) {
  char *file;
  array *a;
  char file2[DICT_LINELEN];
  
  switch(which) {
  case DICT_F_WORDS:
    file = d->wordsfile;
    a = d->words;
    break;
  case DICT_F_RULES:
    file = d->rulesfile;
    a = d->rules;
    break;
  default:
    return 4;
  }
  
  strncpy(file2, file, DICT_LINELEN);
  makesafe_filename(file2, DICT_LINELEN);
  array_zero(a);

  if(!filefetch_conventional(d, file, a))
    return 0;

  if(!filefetch_conventional(d, file2, a))
    return 0;

  if(!filefetch_network(d, file, a))
    return 0;
  
  return 1;
}


dictionary_t *
dictionary_init(char *opts) {
  dictionary_t *d;
  
  d = (dictionary_t *)calloc(1, sizeof(dictionary_t));
  
  param_get_int(opts, "minlen", &d->minlen, DICT_MINLEN);
  param_get_int(opts, "maxlen", &d->maxlen, DICT_MAXLEN);
  param_get_str(opts, "words", d->wordsfile, DICT_WORDSFILE);
  param_get_str(opts, "rules", d->rulesfile, DICT_RULESFILE);
  param_get_int(opts, "killdupes", &d->killdupes, DICT_KILLDUPES);
  param_get_str(opts, "fileserver", d->fileserver, DICT_FILESERVER);
  param_get_str(opts, "fileport", d->fileport, DICT_FILEPORT);
  param_get_str(opts, "tempdir", d->tempdir, DICT_TEMPDIR);

  if(d->maxlen+1 >= DICT_LINELEN) {
    xfree(d);
    return (dictionary_t *)NULL;
  }

  d->rules = array_init(d->maxlen+1);
  d->words = array_init(d->maxlen+1);

  if(filefetch(d, DICT_F_RULES) || filefetch(d, DICT_F_WORDS)) {
    array_nuke(d->rules);
    array_nuke(d->words);
    xfree(d);
    return (dictionary_t *)NULL;
  }

  rules_init(d->maxlen);

  keyspace_init(d->minindex);
  keyspace_init(d->maxindex);
  keyspace_init(d->index);

  keyspace_fromint(array_nelems(d->words), d->minindex);
  keyspace_fromint(array_nelems(d->rules), d->maxindex);
  keyspace_mul(d->maxindex, d->maxindex, d->minindex);
  keyspace_fromint(0, d->minindex);
  keyspace_fromint(0, d->index);
  d->rule_index = 0;

  dictionary_set(d, d->minindex);
  return d;
}

int
dictionary_set(dictionary_t *d, key_index_t where) {
  key_index_t j;
  key_index_t i;

  if(keyspace_cmp(where, d->minindex) < 0 || keyspace_cmp(where, d->maxindex) > 0)
    return 1;

  keyspace_init(j);
  keyspace_init(i);
  keyspace_fromint(array_nelems(d->words), i);
  keyspace_div(j, where, i);
  d->rule_index = keyspace_toint(j);
  keyspace_mod(j, where, i);
  d->word_index = keyspace_toint(j);
  keyspace_set(d->index, where);
  keyspace_destroy(j);
  keyspace_destroy(i);

  return 0;
}

int
dictionary_fetch(dictionary_t *d, kchar *buf, int userlen, int *len) {  
  char *rword;
  char *word;
  char *rule;
    
  if(dictionary_done(d))
    return 1;
  
  if(d->word_index == array_nelems(d->words)) {
    d->rule_index++;
    d->word_index = 0;
    return dictionary_fetch(d, buf, userlen, len);
  }
  
  rule = array_get(d->rules, d->rule_index);
  word = array_get(d->words, d->word_index);

  if(!(rword = rules_apply(word, rule, 0))) {
    keyspace_inc(d->index);
    d->word_index++;
    return dictionary_fetch(d, buf, userlen, len);
  }

  if(d->killdupes) {
    if(!strncmp(rword, d->prevword, DICT_LINELEN)) {
      keyspace_inc(d->index);
      d->word_index++;
      return dictionary_fetch(d, buf, userlen, len);
    }
    strncpy(d->prevword, rword, DICT_LINELEN);
  }

  strncpy(buf, rword, userlen);
  *len = strlen(buf);
  keyspace_inc(d->index);
  d->word_index++;
  return 0;
}

int
dictionary_done(dictionary_t *d) {  
    
  if(!keyspace_cmp(d->index, d->maxindex))
    return 1;
  return 0;
}

key_index_t *
dictionary_minindex(dictionary_t *b) {
  return &b->minindex;
}

key_index_t *
dictionary_maxindex(dictionary_t *b) {
  return &b->maxindex;
}

key_index_t *
dictionary_curindex(dictionary_t *b) {
  return &b->index;
}

int
dictionary_minlen(dictionary_t *b) {
  return b->minlen;
}

int
dictionary_maxlen(dictionary_t *b) {
  return b->maxlen;
}

generator_impl_t dictionary_impl = {
  "dictionary",
  (gen_init)dictionary_init,
  (gen_set)dictionary_set,
  (gen_fetch)dictionary_fetch,
  (gen_done)dictionary_done,
  (gen_destroy)dictionary_destroy,
  (gen_minindex)dictionary_minindex,
  (gen_maxindex)dictionary_maxindex,
  (gen_curindex)dictionary_curindex,
  (gen_minlen)dictionary_minlen,
  (gen_maxlen)dictionary_maxlen
};

#ifdef DICTIONARY_DEBUG
int
main(int argc, char **argv) {
  dictionary_t *k;
  char buffer[1024];
  char index[DICT_LINELEN];
  char minindex[DICT_LINELEN];
  char maxindex[DICT_LINELEN];
  char cur_min[DICT_LINELEN];
  char cur_max[DICT_LINELEN];
  
  k = dictionary_init(1, 8, "/usr/dict/words", "rules");
  if(!k) {
    printf("foo\n");
    exit(2);
  }

  while(!dictionary_done(k)) {
    dictionary_fetch(k, buffer, 1024);
    keyspace_tostr(index, DICT_LINELEN, k->index);
    keyspace_tostr(maxindex, DICT_LINELEN, k->maxindex);
    keyspace_tostr(minindex, DICT_LINELEN, k->minindex);
    keyspace_tostr(cur_min, DICT_LINELEN, k->cur_min);
    keyspace_tostr(cur_max, DICT_LINELEN, k->cur_max);
    printf("%s %s %s %s %s %s %s: %s\n", index, minindex, maxindex, cur_min, cur_max, k->rule, k->word, buffer);
  }
  exit(0);
}

#endif
