/* mg-data-cell-renderer-info.c
 * Copyright (C) 2000  Red Hat, Inc.,  Jonathan Blandford <jrb@redhat.com>
 * Copyright (C) 2003 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include "mg-data-cell-renderer-info.h"
#include "marshal.h"
#include <libgda/libgda.h>
#include <gdk/gdkkeysyms.h>
#include "utility.h"
#include <libmergeant/mg-context.h>
#include <libmergeant/mg-work-core.h>

static void mg_data_cell_renderer_info_get_property  (GObject                    *object,
						      guint                       param_id,
						      GValue                     *value,
						      GParamSpec                 *pspec);
static void mg_data_cell_renderer_info_set_property  (GObject                    *object,
						      guint                       param_id,
						      const GValue               *value,
						      GParamSpec                 *pspec);
static void mg_data_cell_renderer_info_init       (MgDataCellRendererInfo      *celltext);
static void mg_data_cell_renderer_info_class_init (MgDataCellRendererInfoClass *class);
static void mg_data_cell_renderer_info_dispose    (GObject                  *object);
static void mg_data_cell_renderer_info_finalize   (GObject                  *object);

static void mg_data_cell_renderer_info_get_size   (GtkCellRenderer            *cell,
						   GtkWidget                  *widget,
						   GdkRectangle               *cell_area,
						   gint                       *x_offset,
						   gint                       *y_offset,
						   gint                       *width,
						   gint                       *height);
static void mg_data_cell_renderer_info_render     (GtkCellRenderer            *cell,
						   GdkWindow                  *window,
						   GtkWidget                  *widget,
						   GdkRectangle               *background_area,
						   GdkRectangle               *cell_area,
						   GdkRectangle               *expose_area,
						   GtkCellRendererState        flags);
static gboolean mg_data_cell_renderer_info_activate  (GtkCellRenderer            *cell,
						      GdkEvent                   *event,
						      GtkWidget                  *widget,
						      const gchar                *path,
						      GdkRectangle               *background_area,
						      GdkRectangle               *cell_area,
						      GtkCellRendererState        flags);


enum {
	STATUS_CHANGED,
	LAST_SIGNAL
};

enum {
	PROP_ZERO,
	PROP_VALUE,
	PROP_VALUES,
	PROP_VALUES_COMPLETE,
	PROP_VALUE_ATTRIBUTES,
	PROP_EDITABLE,
	PROP_TO_BE_DELETED
};

struct _MgDataCellRendererInfoPriv {
	gboolean  active;
	guint     attributes;
};

#define INFO_WIDTH 6
#define INFO_HEIGHT 14
static GObjectClass *parent_class = NULL;
static guint info_cell_signals[LAST_SIGNAL] = { 0 };


GType
mg_data_cell_renderer_info_get_type (void)
{
	static GType cell_info_type = 0;

	if (!cell_info_type) {
		static const GTypeInfo cell_info_info = {
			sizeof (MgDataCellRendererInfoClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) mg_data_cell_renderer_info_class_init,
					NULL,		/* class_finalize */
			NULL,		/* class_data */
					sizeof (MgDataCellRendererInfo),
			0,              /* n_preallocs */
			(GInstanceInitFunc) mg_data_cell_renderer_info_init,
		};
		
		cell_info_type =
			g_type_register_static (GTK_TYPE_CELL_RENDERER, "MgDataCellRendererInfo",
						&cell_info_info, 0);
	}

	return cell_info_type;
}

static void
mg_data_cell_renderer_info_init (MgDataCellRendererInfo *cellinfo)
{
	cellinfo->priv = g_new0 (MgDataCellRendererInfoPriv, 1);

	GTK_CELL_RENDERER (cellinfo)->mode = GTK_CELL_RENDERER_MODE_ACTIVATABLE;
	GTK_CELL_RENDERER (cellinfo)->xpad = 1;
	GTK_CELL_RENDERER (cellinfo)->ypad = 1;
}

static void
mg_data_cell_renderer_info_class_init (MgDataCellRendererInfoClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

    	object_class->dispose = mg_data_cell_renderer_info_dispose;
	object_class->finalize = mg_data_cell_renderer_info_finalize;

	object_class->get_property = mg_data_cell_renderer_info_get_property;
	object_class->set_property = mg_data_cell_renderer_info_set_property;

	cell_class->get_size = mg_data_cell_renderer_info_get_size;
	cell_class->render = mg_data_cell_renderer_info_render;
	cell_class->activate = mg_data_cell_renderer_info_activate;
  
	g_object_class_install_property (object_class,
					 PROP_VALUE,
					 g_param_spec_pointer ("value",
							       _("Value"),
							       _("GdaValue to display information for"),
							       G_PARAM_WRITABLE));
	g_object_class_install_property (object_class,
					 PROP_VALUES_COMPLETE,
					 g_param_spec_pointer ("values_complete",
							       _("Values limited to PK fields"),
							       _("GList of GdaValue to render, limited to PK fields"),
							       G_PARAM_WRITABLE));

	g_object_class_install_property (object_class,
					 PROP_VALUES,
					 g_param_spec_pointer ("values",
							       _("Values"),
							       _("GList of the GdaValues to display information for"),
							       G_PARAM_WRITABLE));

	g_object_class_install_property (object_class,
					 PROP_VALUE_ATTRIBUTES,
					 g_param_spec_uint ("value_attributes", NULL, NULL,
                                                            0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_EDITABLE,
					 g_param_spec_boolean ("editable",
							       _("Editable"),
							       _("The information and status changer can be activated"),
							       TRUE,G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_TO_BE_DELETED,
					 g_param_spec_boolean ("to_be_deleted", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));

	info_cell_signals[STATUS_CHANGED] =
		g_signal_new ("status_changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (MgDataCellRendererInfoClass, status_changed),
			      NULL, NULL,
			      marshal_VOID__STRING_UINT,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_UINT);
}

static void
mg_data_cell_renderer_info_dispose (GObject *object)
{
	MgDataCellRendererInfo *cellinfo = MG_DATA_CELL_RENDERER_INFO (object);


	/* parent class */
	parent_class->dispose (object);
}

static void
mg_data_cell_renderer_info_finalize (GObject *object)
{
	MgDataCellRendererInfo *cellinfo = MG_DATA_CELL_RENDERER_INFO (object);

	if (cellinfo->priv) {
		g_free (cellinfo->priv);
		cellinfo->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void
mg_data_cell_renderer_info_get_property (GObject     *object,
					 guint        param_id,
					 GValue      *value,
					 GParamSpec  *pspec)
{
	MgDataCellRendererInfo *cellinfo = MG_DATA_CELL_RENDERER_INFO (object);
  
	switch (param_id) {
	case PROP_VALUE_ATTRIBUTES:
		g_value_set_uint (value, cellinfo->priv->attributes);
		break;
	case PROP_EDITABLE:
		g_value_set_boolean (value, cellinfo->priv->active);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}


static void
mg_data_cell_renderer_info_set_property (GObject      *object,
					 guint         param_id,
					 const GValue *value,
					 GParamSpec   *pspec)
{
	MgDataCellRendererInfo *cellinfo = MG_DATA_CELL_RENDERER_INFO (object);
  
	switch (param_id) {
	case PROP_VALUE:
		g_object_notify (object, "value");
		break;
	case PROP_VALUES:
		g_object_notify (object, "values");
		break;
	case PROP_VALUES_COMPLETE:
		g_object_notify (object, "values_complete");
		break;
	case PROP_VALUE_ATTRIBUTES:
		cellinfo->priv->attributes = g_value_get_uint (value);
		break;
	case PROP_EDITABLE:
		cellinfo->priv->active = g_value_get_boolean (value);
		g_object_notify (G_OBJECT(object), "editable");
		break;
	case PROP_TO_BE_DELETED:
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

/**
 * mg_data_cell_renderer_info_new:
 * 
 * Creates a new #MgDataCellRendererInfo. Adjust rendering
 * parameters using object properties. Object properties can be set
 * globally (with g_object_set()). Also, with #GtkTreeViewColumn, you
 * can bind a property to a value in a #GtkTreeModel. For example, you
 * can bind the "active" property on the cell renderer to a boolean value
 * in the model, thus causing the check button to reflect the state of
 * the model.
 * 
 * Return value: the new cell renderer
 **/
GtkCellRenderer *
mg_data_cell_renderer_info_new (void)
{
	return g_object_new (MG_DATA_CELL_RENDERER_INFO_TYPE, NULL);
}

static void
mg_data_cell_renderer_info_get_size (GtkCellRenderer *cell,
				     GtkWidget       *widget,
				     GdkRectangle    *cell_area,
				     gint            *x_offset,
				     gint            *y_offset,
				     gint            *width,
				     gint            *height)
{
	gint calc_width;
	gint calc_height;

	calc_width = (gint) cell->xpad * 2 + INFO_WIDTH;
	calc_height = (gint) cell->ypad * 2 + INFO_HEIGHT;

	if (width)
		*width = calc_width;

	if (height)
		*height = calc_height;

	if (cell_area) {
		if (x_offset) {
			*x_offset = cell->xalign * (cell_area->width - calc_width);
			*x_offset = MAX (*x_offset, 0);
		}
		if (y_offset) {
			*y_offset = cell->yalign * (cell_area->height - calc_height);
			*y_offset = MAX (*y_offset, 0);
		}
	}
}


static void
mg_data_cell_renderer_info_render (GtkCellRenderer      *cell,
				   GdkWindow            *window,
				   GtkWidget            *widget,
				   GdkRectangle         *background_area,
				   GdkRectangle         *cell_area,
				   GdkRectangle         *expose_area,
				   GtkCellRendererState  flags)
{
	MgDataCellRendererInfo *cellinfo = (MgDataCellRendererInfo *) cell;
	gint width, height;
	gint x_offset, y_offset;
	GtkStateType state = 0;

	static GdkColor **colors = NULL;
	GdkColor *normal = NULL, *prelight = NULL;
	GdkColor *orig_normal, *orig_prelight;
	GtkStyle *style;

	if (!colors)
		colors = utility_entry_build_info_colors_array ();

	style = gtk_style_copy (widget->style);
	orig_normal = & (style->bg[GTK_STATE_NORMAL]);
	orig_prelight = & (style->bg[GTK_STATE_PRELIGHT]);
	if (cellinfo->priv->attributes & MG_DATA_ENTRY_IS_NULL) {
		normal = colors[0];
		prelight = colors[1];
	}

	if (cellinfo->priv->attributes & MG_DATA_ENTRY_IS_DEFAULT) {
		normal = colors[2];
		prelight = colors[3];
	}

	if (cellinfo->priv->attributes & MG_DATA_ENTRY_DATA_NON_VALID) {
		normal = colors[4];
		prelight = colors[5];
	}

	if (!normal)
		normal = orig_normal;
	if (!prelight)
		prelight = orig_prelight;

	style->bg[GTK_STATE_NORMAL] = *normal;
	style->bg[GTK_STATE_ACTIVE] = *normal;
	style->bg[GTK_STATE_PRELIGHT] = *prelight;
	gtk_style_attach (style, window);
	mg_data_cell_renderer_info_get_size (cell, widget, cell_area,
					     &x_offset, &y_offset,
					     &width, &height);
	width -= cell->xpad*2;
	height -= cell->ypad*2;

	if (width <= 0 || height <= 0)
		return;

	state = GTK_STATE_NORMAL;

	gtk_paint_box (style,
		       window,
		       state, GTK_SHADOW_NONE,
		       cell_area, widget, "cellcheck",
		       cell_area->x + x_offset + cell->xpad,
		       cell_area->y + y_offset + cell->ypad,
		       width - 1, height - 1);
	gtk_style_detach (style);
	g_object_unref (G_OBJECT (style));
}


static void mitem_activated_cb (GtkWidget *mitem, MgDataCellRendererInfo *cellinfo);
static gint
mg_data_cell_renderer_info_activate (GtkCellRenderer      *cell,
				     GdkEvent             *event,
				     GtkWidget            *widget,
				     const gchar          *path,
				     GdkRectangle         *background_area,
				     GdkRectangle         *cell_area,
				     GtkCellRendererState  flags)
{
	MgDataCellRendererInfo *cellinfo;
	gchar *tmp;

	cellinfo = MG_DATA_CELL_RENDERER_INFO (cell);

	/* free any pre-allocated path */
	if ((tmp = g_object_get_data (G_OBJECT (cellinfo), "path"))) {
		g_free (tmp);
		g_object_set_data (G_OBJECT (cellinfo), "path", NULL);
	}

	if (cellinfo->priv->active) {
		GtkWidget *menu;
		MgContextNode *node;
		guint attributes = 0;

		/* Dirty hack:
		 * use the 'path' information and the "node" attribute (g_object_get_data ()...) to calculate
		 * the new attributes first.
		 */
		GtkTreeIter iter;
		GtkTreePath *treepath;
		GtkTreeModel *model;
		MgWorkCore *core;

		core = g_object_get_data (G_OBJECT (cell), "work-core");
		g_return_val_if_fail (core && IS_MG_WORK_CORE (core), FALSE);

		model = g_object_get_data (G_OBJECT (cell), "tree-model");
		g_return_val_if_fail (model && GTK_IS_TREE_MODEL (model), FALSE);

		node = g_object_get_data (G_OBJECT (cell), "context-node");
		g_assert (node);

		treepath = gtk_tree_path_new_from_string (path);
		if (! gtk_tree_model_get_iter (model, &iter, treepath)) {
			g_warning ("Can't set iter on model from path %s", path);
			gtk_tree_path_free (treepath);
			return FALSE;
		}
		gtk_tree_path_free (treepath);

		/* we want the attributes */
		gda_value_free (utility_grid_model_get_value (model, &iter, core, node, TRUE, &attributes));
		
		/* build the popup menu */
		menu = utility_entry_build_actions_menu (G_OBJECT (cellinfo), attributes, 
							 G_CALLBACK (mitem_activated_cb));
		g_object_set_data (G_OBJECT (cellinfo), "path", g_strdup (path));
		gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
				0, gtk_get_current_event_time ());
		return TRUE;		
	}

	return FALSE;
}

static void
mitem_activated_cb (GtkWidget *mitem, MgDataCellRendererInfo *cellinfo)
{
	guint action;
	gchar *path;

	action = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (mitem), "action"));
	path = g_object_get_data (G_OBJECT (cellinfo), "path");
#ifdef debug_signal
        g_print (">> 'STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (cellinfo, info_cell_signals[STATUS_CHANGED], 0, path, action); 
#ifdef debug_signal
        g_print ("<< 'STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_free (path);
	g_object_set_data (G_OBJECT (cellinfo), "path", NULL);
}
