(* Copyright (C) 2000, Luca Padovani <luca.padovani@cs.unibo.it>.
 *
 * This file is part of mlminidom, the Ocaml binding for minidom.
 * 
 * mlminidom is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * mlminidom is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mlminidom; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * For details, send a mail to the author.
 *)

exception Node_has_no_parent;;
exception Node_has_no_sibling of string;;
exception Node_has_no_children;;
exception Node_has_no_attributes;;
exception Attribute_has_no_sibling of string;;
exception Attribute_has_no_parent;;
exception Undefined_entity;;

let option_to_exception v e =
  match v with
    Some x -> x
  | None   -> raise e
;;

class o_mDOMString (str: Minidom.mDOMString) =
  object
    method get_dom_string = str
    method get_string = Minidom.string_of_mDOMString str
  end;;
  
let o_mDOMString_of_string str =
  new o_mDOMString (Minidom.mDOMString_of_string str)

class o_mDOMEntity (ent : Minidom.mDOMEntity) =
  object
    method get_dom_entity = ent
    method get_content =
      new o_mDOMString (Minidom.entity_get_content ent)
  end
;;

class o_mDOMDoc (doc : Minidom.mDOMDoc) =
  object
    method get_dom_doc = doc

    method get_root_node =
      new o_mDOMNode (Minidom.doc_get_root_node doc)
    method add_entity (name : o_mDOMString) (value : o_mDOMString) =
      new o_mDOMEntity
        (Minidom.doc_add_entity doc
	  (name#get_dom_string) (value#get_dom_string)
	)
    method get_entity (name : o_mDOMString) =
      match Minidom.doc_get_entity doc (name#get_dom_string) with
      | Some x -> new o_mDOMEntity x
      | None -> raise Undefined_entity
    method get_predefined_entity (name : o_mDOMString) =
      match Minidom.doc_get_predefined_entity doc (name#get_dom_string) with
      | Some x -> new o_mDOMEntity x
      | None -> raise Undefined_entity
  end
and o_mDOMNode (node : Minidom.mDOMNode) =
  object
    method get_dom_node = node

    method is_text = Minidom.node_is_text node
    method is_element = Minidom.node_is_element node
    method is_blank = Minidom.node_is_blank node
    method is_entity_ref = Minidom.node_is_entity_ref node

    method get_type = Minidom.node_get_type node
    method get_name = 
      match Minidom.node_get_name node with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_ns_uri =
      match Minidom.node_get_ns_uri node with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_attribute (name : o_mDOMString) =
      match Minidom.node_get_attribute node (name#get_dom_string) with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_attribute_ns (name : o_mDOMString) (uri : o_mDOMString) =
      match 
        Minidom.node_get_attribute_ns node
	  (name#get_dom_string) (uri#get_dom_string)
      with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method has_attribute (name : o_mDOMString) =
     Minidom.node_has_attribute node (name#get_dom_string)
    method has_attribute_ns (name : o_mDOMString) (uri : o_mDOMString) =
     Minidom.node_has_attribute_ns node (name#get_dom_string) (uri#get_dom_string)
    method get_content =
      match Minidom.node_get_content node with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_parent =
      new o_mDOMNode
      	(option_to_exception (Minidom.node_get_parent node) Node_has_no_parent)
    method get_prev_sibling =
      new o_mDOMNode
      	(option_to_exception
	 (Minidom.node_get_prev_sibling node)
	 (Node_has_no_sibling "left")
	)
    method get_next_sibling =
      new o_mDOMNode
      	(option_to_exception
	 (Minidom.node_get_next_sibling node)
	 (Node_has_no_sibling "right")
	)
    method get_first_child =
      new o_mDOMNode
      	(option_to_exception
	 (Minidom.node_get_first_child node)
	 (Node_has_no_children)
	)
    method get_first_attribute =
      new o_mDOMAttr
      	(option_to_exception
	  (Minidom.node_get_first_attribute node)
	  (Node_has_no_attributes)
	)
    method is_first = Minidom.node_is_first node
    method is_last = Minidom.node_is_last node

    method get_children =
      List.map (function x -> new o_mDOMNode x) (Minidom.node_get_children node)
    method get_attributes = List.map
      (function x -> new o_mDOMAttr x) (Minidom.node_get_attributes node)
  end
and o_mDOMAttr (attr : Minidom.mDOMAttr) =
  object
    method get_dom_attr = attr

    method get_name =
      match Minidom.attr_get_name attr with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_ns_uri =
      match Minidom.attr_get_ns_uri attr with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_value =
      match Minidom.attr_get_value attr with
      | Some x -> Some (new o_mDOMString x)
      | None   -> None
    method get_prev_sibling =
      new o_mDOMAttr
        (option_to_exception
	  (Minidom.attr_get_prev_sibling attr)
	  (Attribute_has_no_sibling "left")
	)
    method get_next_sibling =
      new o_mDOMAttr
        (option_to_exception
	  (Minidom.attr_get_next_sibling attr)
	  (Attribute_has_no_sibling "right")
	)
    method get_parent =
      new o_mDOMNode
        (option_to_exception
	  (Minidom.attr_get_parent attr) Attribute_has_no_parent
	)
  end
;;
    
