# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Monitoring of the document modules state.

# load modules
from nucleus.ids import *


class monitor:
    """ Document modules state monitor. """

    # ----
    def __init__(self):
        self.status = {}

        # common
        self.status['common'] = {}
        self.status['common']['changed'] = False
        self.status['common']['processed'] = False
        self.status['common']['path'] = ''
        self.status['common']['docType'] = ''
        self.status['common']['scanID'] = ''

        # mass params
        self.status['mass'] = {}
        self.status['mass']['masstype'] = 'mmass'
        self.status['mass']['errortype'] = 'Da'
        self.status['mass']['tolerance'] = 0.5

        # make variables for modules
        for module in ('mPeak', 'mSpec', 'mSeq', 'mCut', 'mFrag', 'mDiff', 'mComp'):
            self.status[module] = {}
            self.status[module]['data'] = False
            self.status[module]['matched'] = False
    # ----


    # ----
    def resetDataStatus(self):
        """ Set all data status to False. """
        for module in ('mPeak', 'mSpec', 'mSeq', 'mCut', 'mFrag', 'mDiff', 'mComp'):
            self.status[module]['data'] = False
    # ----


    # ----
    def resetMatchStatus(self):
        """ Set all match status to False. """
        for module in ('mPeak', 'mSpec', 'mSeq', 'mCut', 'mFrag', 'mDiff', 'mComp'):
            self.status[module]['matched'] = False
    # ----


    # ----
    def setMassParam(self, param, value):
        """ Set masstype, errortype, tolerance. """
        self.status['mass'][param] = value
    # ----


    # ----
    def setPath(self, path):
        """ Set document file path. """
        self.status['common']['path'] = path
    # ----


    # ----
    def setDocType(self, docType):
        """ Set document type. """
        self.status['common']['docType'] = docType
    # ----


    # ----
    def setScanID(self, scanID):
        """ Set spectrum scan ID. """
        self.status['common']['scanID'] = scanID
    # ----


    # ----
    def setChangeStatus(self, status):
        """ Set document change status. """
        self.status['common']['changed'] = status
    # ----


    # ----
    def setProcessingStatus(self, status):
        """ Set spectrum processing status. """
        self.status['common']['processed'] = status
    # ----


    # ----
    def setDataStatus(self, status, module):
        """ Set data status for current or selected module. """

        # set data status
        if not status:
            self.status[module]['data'] = False
            self.status[module]['matched'] = False
        else:
            self.status[module]['data'] = True
            self.status[module]['matched'] = False
    # ----


    # ----
    def setMatchStatus(self, status, module):
        """ Set matching status for current or selected module. """
        self.status[module]['matched'] = status
    # ----


    # ----
    def getMassParam(self, param=None):
        """ Get masstype, errortype, tolerance. """
        if not param:
            return self.status['mass']
        else:
            return self.status['mass'][param]
    # ----


    # ----
    def getPath(self):
        """ Get document file path. """
        return self.status['common']['path']
    # ----


    # ----
    def getDocType(self):
        """ Get document type. """
        return self.status['common']['docType']
    # ----


    # ----
    def getScanID(self):
        """ Get spectrum scan ID. """
        return self.status['common']['scanID']
    # ----


    # ----
    def getChangeStatus(self):
        """ Get document change status. """
        return self.status['common']['changed']
    # ----


    # ----
    def getProcessingStatus(self):
        """ Get spectrum processing status. """
        return self.status['common']['processed']
    # ----


    # ----
    def getDataStatus(self, module):
        """ Get data status of current or selected module. """
        return self.status[module]['data']
    # ----


    # ----
    def getMatchStatus(self, module):
        """ Get matching status of current or selected module. """
        return self.status[module]['matched']
    # ----
