# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Edit sequence modifications.

# load libs
import wx

# load modules
from nucleus import mwx


class dlgEditModifications(wx.Dialog):
    """Edit sequence modification"""

    # ----
    def __init__(self, parent, config, sequence, modifications, currentAmino):
        wx.Dialog.__init__(self, parent, -1, "Modifications", style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.config = config
        self.sequence = sequence
        self.modifications = modifications

        # init selected items
        self.selectedResidue = [self.sequence[currentAmino], currentAmino]
        self.selectedMod = None

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.makeSelectionBox(), 0, wx.EXPAND|wx.ALL, 20)
            mainSizer.Add(self.makeEditModBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 20)
        else:    
            mainSizer.Add(self.makeSelectionBox(), 0, wx.EXPAND|wx.ALL, 5)
            mainSizer.Add(self.makeEditModBox(), 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 5)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 3)

        # select default amino
        self.selectDefaultAmino()

        # check and show current modifications
        self.checkAppliedMod()
        self.updateAppliedMod()

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeSelectionBox(self):
        """ Make box for residue selection items. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Select Residue and Modification"), wx.HORIZONTAL)
        residueBox = wx.BoxSizer(wx.VERTICAL)
        availableModBox = wx.BoxSizer(wx.VERTICAL)
        modInfoBox = wx.BoxSizer(wx.VERTICAL)

        selectionResidueType_choices = self.getResidueTypes()
        selectionResidueType_label = wx.StaticText(self, -1, "Residue type:")
        self.selectionResidueType_combo = wx.ComboBox(self, -1, size=(95, -1), choices=selectionResidueType_choices, style=wx.CB_READONLY)

        selectionResidueAvailable_label = wx.StaticText(self, -1, "Available residues:")
        self.selectionResidueAvailable_list = wx.ListBox(self, -1, size=(95, 130), style=wx.LB_SINGLE|wx.LB_ALWAYS_SB)

        selectionModGroups_choices = ('all', 'ptm', 'glyco', 'chemical', 'artefact', 'other')
        selectionModGroups_label = wx.StaticText(self, -1, "Filter by group:")
        self.selectionModGroups_combo = wx.ComboBox(self, -1, size=(170, -1), choices=selectionModGroups_choices, style=wx.CB_READONLY)

        selectionModAvailable_label = wx.StaticText(self, -1, "Available modifications:")
        self.selectionModAvailable_list = wx.ListBox(self, -1, size=(190, 146), style=wx.LB_SINGLE|wx.LB_ALWAYS_SB)

        self.selectionModIgnoreSpec_check = wx.CheckBox(self, -1, "Ignore specifity")

        selectionModDescription_label = wx.StaticText(self, -1, "Description:")
        self.selectionModDescription_value = wx.TextCtrl(self, -1, '', size=(190, -1))
        self.selectionModDescription_value.Enable(False)

        selectionModFormula_label = wx.StaticText(self, -1, "Formula:")
        self.selectionModFormula_value = wx.TextCtrl(self, -1, '', size=(190, -1))
        self.selectionModFormula_value.Enable(False)

        selectionModMass_label = wx.StaticText(self, -1, "Mass: (Mo./Av.)")
        self.selectionModMass_value = wx.TextCtrl(self, -1, '', size=(190, -1))
        self.selectionModMass_value.Enable(False)

        selectionModSpecifity_label = wx.StaticText(self, -1, "Specifity:")
        self.selectionModSpecifity_value = wx.TextCtrl(self, -1, '', size=(190, -1))
        self.selectionModSpecifity_value.Enable(False)

        # pack items
        residueBox.Add(selectionResidueType_label, 0, wx.LEFT, 5)
        residueBox.Add(self.selectionResidueType_combo, 0, wx.TOP|wx.LEFT, 5)

        residueBox.Add(selectionResidueAvailable_label, 0, wx.TOP|wx.LEFT, 5)
        residueBox.Add(self.selectionResidueAvailable_list, 1, wx.TOP|wx.LEFT, 5)

        availableModBox.Add(selectionModGroups_label, 0)
        availableModBox.Add(self.selectionModGroups_combo, 0, wx.EXPAND|wx.TOP, 5)

        availableModBox.Add(selectionModAvailable_label, 0, wx.TOP, 5)
        availableModBox.Add(self.selectionModAvailable_list, 1, wx.EXPAND|wx.TOP, 5)

        modInfoBox.Add(selectionModDescription_label, 0)
        modInfoBox.Add(self.selectionModDescription_value, 0, wx.TOP, 5)

        modInfoBox.Add(selectionModFormula_label, 0, wx.TOP, 5)
        modInfoBox.Add(self.selectionModFormula_value, 0, wx.TOP, 5)

        modInfoBox.Add(selectionModMass_label, 0, wx.TOP, 5)
        modInfoBox.Add(self.selectionModMass_value, 0, wx.TOP, 5)

        modInfoBox.Add(selectionModSpecifity_label, 0, wx.TOP, 5)
        modInfoBox.Add(self.selectionModSpecifity_value, 0, wx.TOP, 5)

        modInfoBox.Add(self.selectionModIgnoreSpec_check, 0, wx.TOP, 10)

        mainBox.Add(residueBox, 0, wx.EXPAND|wx.TOP|wx.RIGHT|wx.BOTTOM, 5)
        mainBox.Add(availableModBox, 1, wx.EXPAND|wx.ALL, 5)
        mainBox.Add(modInfoBox, 0, wx.ALL, 5)

        # set events
        self.selectionResidueType_combo.Bind(wx.EVT_COMBOBOX, self.onResidueType)
        self.selectionResidueAvailable_list.Bind(wx.EVT_LISTBOX, self.onResidueSelected)
        self.selectionModAvailable_list.Bind(wx.EVT_LISTBOX, self.onModSelected)
        self.selectionModGroups_combo.Bind(wx.EVT_COMBOBOX, self.onModFilter)
        self.selectionModIgnoreSpec_check.Bind(wx.EVT_CHECKBOX, self.onIgnoreSpec)

        # set defaults
        self.selectionModGroups_combo.Select(0)

        return mainBox
    # ----


    # ----
    def makeEditModBox(self):
        """ Make add/remove buttons and modification type (residue/global). """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Add / Remove Modification"), wx.HORIZONTAL)
        buttonBox = wx.BoxSizer(wx.VERTICAL)
        currentModBox = wx.BoxSizer(wx.VERTICAL)

        self.editModSelected_radio = wx.RadioButton(self, -1, "Selected residue only", style=wx.RB_GROUP)
        self.editModGlobal_radio = wx.RadioButton(self, -1, "All residues of this type")

        self.editModAdd_button = wx.Button(self, -1, "Add")
        self.editModRemove_button = wx.Button(self, -1, "Remove")

        self.editModIgnorePrevious_check = wx.CheckBox(self, -1, "Ignore multiple mod.")
        self.editModIgnorePrevious_check.SetToolTip(wx.ToolTip("Ignore multiple modifcations on the same residue"))

        modCurrent_label = wx.StaticText(self, -1, "Current modifications:")
        self.modApplied_list = mwx.ListCtrl(self, -1, size=(330, 130))
        self.modApplied_list.InsertColumn(0, "Residue")
        self.modApplied_list.InsertColumn(1, "Modification", wx.LIST_FORMAT_LEFT)
        self.modApplied_list.InsertColumn(2, "Mo. mass", wx.LIST_FORMAT_RIGHT)
        self.modApplied_list.InsertColumn(3, "Av. mass", wx.LIST_FORMAT_RIGHT)
        self.modApplied_list.InsertColumn(4, "Formula", wx.LIST_FORMAT_LEFT)

        # pack items
        buttonBox.Add(self.editModSelected_radio, 0, wx.TOP|wx.RIGHT, 5)
        buttonBox.Add(self.editModGlobal_radio, 0, wx.TOP|wx.RIGHT|wx.BOTTOM, 5)
        buttonBox.Add(self.editModAdd_button, 0, wx.ALL|wx.ALIGN_CENTER, 5)
        buttonBox.Add(self.editModRemove_button, 0, wx.ALL|wx.ALIGN_CENTER, 5)
        buttonBox.Add(self.editModIgnorePrevious_check, 0, wx.TOP|wx.RIGHT|wx.BOTTOM, 5)

        currentModBox.Add(modCurrent_label, 0, wx.TOP, 5)
        currentModBox.Add(self.modApplied_list, 1, wx.EXPAND|wx.TOP, 5)

        mainBox.Add(buttonBox, 0, wx.LEFT|wx.RIGHT|wx.BOTTOM, 5)
        mainBox.Add(currentModBox, 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 5)

        # set defaults
        self.editModSelected_radio.SetValue(True)

        # set events
        self.modApplied_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onAppliedModSelected)
        self.modApplied_list.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.onAppliedModDeselected)
        self.editModAdd_button.Bind(wx.EVT_BUTTON, self.onAddMod)
        self.editModRemove_button.Bind(wx.EVT_BUTTON, self.onRemoveMod)

        return mainBox
    # ----


    # ----
    def makeButtonBox(self):
        """ Make main dialog buttons. """

        # make items
        self.OK_button = wx.Button(self, wx.ID_OK, "Apply")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(self.OK_button, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set defaults
        self.OK_button.Enable(False)

        return mainBox
    # ----


    # ----
    def updateAvailableResidues(self):
        """ Update available residues by type. """

        # clear list
        self.selectionResidueAvailable_list.Clear()

        # get available residues
        residues = self.getAvailableResidues()

        # update list
        for residue in residues:
            self.selectionResidueAvailable_list.Append(residue)

        # select first residue (default)
        self.selectedResidue[1] = int(residues[0][2:])-1
        self.selectionResidueAvailable_list.Select(0)
    # ----


    # ----
    def updateAvailableMod(self):
        """ Update available modifications by type. """

        self.selectedMod = None

        # disable Add button
        self.editModAdd_button.Enable(False)

        # clear list
        self.selectionModAvailable_list.Clear()

        # get modifications
        modifications = self.getAvailableModifications()

        # update list
        for mod in modifications:
            self.selectionModAvailable_list.Append(mod)

        # clear modification info
        self.updateModInfo()
    # ----


    # ----
    def updateAppliedMod(self):
        """ Update applied modifications. """

        # disable remove button
        self.editModRemove_button.Enable(False)

        # clear list
        self.modApplied_list.DeleteAllItems()

        # paste data
        for row in range(len(self.modifications)):

            # get residue
            position = self.modifications[row][0]
            if type(position) == int:
                residue = self.sequence[position] + ' ' + str(position+1)
            else:
                residue = 'all' + ' ' + position

            # get name and formula
            formula = self.modifications[row][1]
            name = self.modifications[row][2]

            # get masses
            mmass = round(self.config.mod[name]['mmass'], self.config.cfg['common']['digits'])
            amass = round(self.config.mod[name]['amass'], self.config.cfg['common']['digits'])

            # paste data to list
            index = self.modApplied_list.InsertStringItem(row, str(residue))
            self.modApplied_list.SetStringItem(index, 1, str(name))
            self.modApplied_list.SetStringItem(index, 2, str(mmass))
            self.modApplied_list.SetStringItem(index, 3, str(amass))
            self.modApplied_list.SetStringItem(index, 4, str(formula))
            self.modApplied_list.SetItemData(index, row)

        # set columns width
        if self.modifications:
            self.modApplied_list.SetColumnWidth(0, wx.LIST_AUTOSIZE_USEHEADER)
            self.modApplied_list.SetColumnWidth(1, wx.LIST_AUTOSIZE)
            self.modApplied_list.SetColumnWidth(2, wx.LIST_AUTOSIZE_USEHEADER)
            self.modApplied_list.SetColumnWidth(3, wx.LIST_AUTOSIZE_USEHEADER)
            self.modApplied_list.SetColumnWidth(4, wx.LIST_AUTOSIZE)
        else:
            for col in range(5):
                self.modApplied_list.SetColumnWidth(col, wx.LIST_AUTOSIZE_USEHEADER)

        self.modApplied_list.updateLastCol()
    # ----


    # ----
    def updateModInfo(self):
        """ Update information about selected modification. """

        # update info
        if self.selectedMod != None:
          mass = '%0.5f / %0.5f' % (self.config.mod[self.selectedMod]['mmass'], self.config.mod[self.selectedMod]['amass'])
          self.selectionModDescription_value.SetValue(self.config.mod[self.selectedMod]['descr'])
          self.selectionModFormula_value.SetValue(self.config.mod[self.selectedMod]['formula'])
          self.selectionModMass_value.SetValue(mass)
          self.selectionModSpecifity_value.SetValue(self.config.mod[self.selectedMod]['specifity'])
        else:
          self.selectionModDescription_value.SetValue('')
          self.selectionModFormula_value.SetValue('')
          self.selectionModMass_value.SetValue('')
          self.selectionModSpecifity_value.SetValue('')
    # ----


    # ----
    def selectDefaultAmino(self):
        """ Select default amino acid. """

        # select items
        if not self.selectedResidue:
            self.selectionResidueType_combo.Select(0)
            self.onResidueType(0)
        else:
            # select residue type
            name = self.config.amino[self.selectedResidue[0]]['name']
            residueTypes = self.getResidueTypes()
            self.selectionResidueType_combo.Select(residueTypes.index(name))

            # update residue list
            availableResidues = self.getAvailableResidues()
            for residue in availableResidues:
                self.selectionResidueAvailable_list.Append(residue)

            # select residue
            residue = self.selectedResidue[0] + ' ' + str(self.selectedResidue[1]+1)
            self.selectionResidueAvailable_list.Select(availableResidues.index(residue))

            # update modification list
            self.updateAvailableMod()
    # ----


    # ----
    def onResidueType(self, evt):
        """ Update available residues when type has changed. """

        # get residue name
        residueName = self.selectionResidueType_combo.GetValue()

        # get residue letter
        for amino in self.config.amino:
            if self.config.amino[amino]['name'] == residueName:
                self.selectedResidue = [amino, False]
                break

        # update available residues
        self.updateAvailableResidues()

        # update available modifications
        self.updateAvailableMod()
    # ----


    # ----
    def onResidueSelected(self, evt):
        """ Update available modifications when residue has changed. """

        # get selected residue
        residue = evt.GetString()
        self.selectedResidue[1] = int(residue[2:])-1

        # update available modifications
        self.updateAvailableMod()
    # ----


    # ----
    def onModSelected(self, evt):
        """ Show info about selected modification and enable 'Add' button. """

        # get selected modification
        self.selectedMod = evt.GetString()

        # enable Add button
        self.editModAdd_button.Enable(True)

        # update modification info
        self.updateModInfo()
    # ----


    # ----
    def onModFilter(self, evt):
        """ Apply selected modification filter. """
        self.updateAvailableMod()
    # ----


    # ----
    def onIgnoreSpec(self, evt):
        """ Apply selected specifity filter. """
        self.updateAvailableMod()
    # ----


    # ----
    def onAppliedModSelected(self, evt):
        """ Enable 'Remove' button if applied modification is selected. """
        self.editModRemove_button.Enable(True)
    # ----


    # ----
    def onAppliedModDeselected(self, evt):
        """ Disable 'Remove' button if applied modification is deselected. """
        self.editModRemove_button.Enable(False)
    # ----


    # ----
    def onAddMod(self, evt):
        """ Apply selected modification on sequece. """

        # get position
        if self.editModSelected_radio.GetValue():
            position = self.selectedResidue[1]
        elif self.editModGlobal_radio.GetValue():
            position = self.selectedResidue[0]

        # get formula
        formula = self.config.mod[self.selectedMod]['formula']

        # check if selected residue has modification already
        if not self.editModIgnorePrevious_check.IsChecked() and self.checkPreviousMod(position):
            dlg = wx.MessageDialog(self, "You are trying to modify residue which is already modified.\nDo you really want to add this modification?", "Multi-modification Warning", wx.YES_NO|wx.ICON_EXCLAMATION|wx.YES_DEFAULT)
            pressed = dlg.ShowModal()
            dlg.Destroy()
            if pressed == wx.ID_NO:
                return

        # add modification
        self.modifications.append([position, formula, self.selectedMod])

        # sort data
        self.modifications.sort(self.sortModifications)

        # update current modifications
        self.updateAppliedMod()

        # enable OK button
        self.OK_button.Enable(True)
    # ----


    # ----
    def onRemoveMod(self, evt):
        """ Remove selected modifications. """

        # get selected items
        indexes = mwx.getSelectedListItems(self.modApplied_list, reverse=True)

        # delete selected
        for index in indexes:
            del self.modifications[index]

        # update current modifications
        self.updateAppliedMod()

        # enable OK button
        self.OK_button.Enable(True)

    # ----


    # ----
    def getResidueTypes(self):
        """ Get residue types from sequence. """


        # search sequence for types
        buff = ''
        for amino in self.sequence:
            if amino not in buff:
                buff += amino

        # get names
        names = []
        for code in buff:
            names.append(self.config.amino[code]['name'])

        # sort names
        names.sort()

        return names
    # ----


    # ----
    def getAvailableResidues(self):
        """ Get residue all residues of selected type. """

        # search sequence
        residues = []
        for index in range(len(self.sequence)):
            if self.sequence[index] == self.selectedResidue[0]:
                residues.append(self.selectedResidue[0] + ' ' + str(index+1))

        return residues
    # ----


    # ----
    def getAvailableModifications(self):
        """ Get available modifications. """

        # get filter
        groupFilter = self.selectionModGroups_combo.GetValue()

        # get modifications
        modifications = []
        for mod in self.config.mod:
            if (
                (self.selectedResidue[0] in self.config.mod[mod]['specifity']) \
                or (self.config.mod[mod]['specifity'] == '' and not self.config.mod[mod]['n-term'] and not self.config.mod[mod]['c-term'])
                or (self.selectedResidue[1] == 0 and self.config.mod[mod]['n-term']) \
                or (self.selectedResidue[1] == len(self.sequence)-1 and self.config.mod[mod]['c-term']) \
                or (self.selectionModIgnoreSpec_check.IsChecked()) \
                ) \
                and (groupFilter == 'all' or groupFilter in self.config.mod[mod]['group']):

                modifications.append(mod)

        # sort names
        modifications.sort()

        return modifications
    # ----


    # ----
    def checkAppliedMod(self):
        """ Check if all applied modifications are available in config. """

        # get indexes to delete
        indexes = []
        for x, mod in enumerate(self.modifications):
            if not mod[2] in self.config.mod:
                indexes.append(x)

        # delete unavailable modifications
        if indexes:
            indexes.sort()
            indexes.reverse()
            for index in indexes:
                del self.modifications[x]
    # ----


    # ----
    def checkPreviousMod(self, position):
        """ Check if any modification applied on selected position. """

        # check residue modification or same global
        globalMods = ''
        for mod in self.modifications:
            if mod[0] == position:
                return True
            elif type(mod[0]) == int and type(position) != int:
                if self.sequence[mod[0]] == position:
                    return True
            elif type(mod[0]) != int:
                globalMods += mod[0]

        # check global
        if type(position) == int and self.sequence[position] in globalMods:
            return True
    # ----


    # ----
    def sortModifications(self, x, y):
        """ Special sorting fce for modification list. """

        if type(x[0]) != int:
            one = ' %s%s' % (x[0], x[2])
        else:
            one = '%s%6s' % (x[0], x[2])

        if type(y[0]) != int:
            two = ' %s%s' % (y[0], y[2])
        else:
            two = '%s%6s' % (y[0], y[2])

        return cmp(one, two)
    # ----

