# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Mass spectrum analyser.

# load libs
import wx
import os

# load modules
import spectrum
import count
from nucleus import mwx
from nucleus import images
from dlg_specparams import dlgEditSpecParams
from dlg_viewoptions import dlgViewOptions
from modules.mimport.document import mDocumentImporter


# ----
class mSpec(wx.Panel):
    """ Show and analyse mass spectrum. """

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        self.parent = parent
        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document

        self.objects = []
        self.container = []
        self.loadedSpectra = []
        self.selectedSpectrum = -1
        self.cursorImages = {'none':wx.StockCursor(wx.CURSOR_ARROW)}

        self.mouseFce = 'none'
        self.peakHeight = self.config.cfg['mspec']['peakheight']

        # init view parameters
        self.viewParams = {}
        self.viewParams['showlabels'] = self.config.cfg['mspec']['showlabels']
        self.viewParams['showannots'] = self.config.cfg['mspec']['showannots']
        self.viewParams['showticks'] = self.config.cfg['mspec']['showticks']
        self.viewParams['labelangle'] = self.config.cfg['mspec']['labelangle']
        self.viewParams['showtracker'] = self.config.cfg['mspec']['showtracker']
        self.viewParams['showposition'] = self.config.cfg['mspec']['showposition']
        self.viewParams['showdistance'] = self.config.cfg['mspec']['showdistance']
        self.viewParams['showgrid'] = self.config.cfg['mspec']['showgrid']
        self.viewParams['showlegend'] = self.config.cfg['mspec']['showlegend']
        self.viewParams['showposbar'] = self.config.cfg['mspec']['showposbar']
        self.viewParams['showgelview'] = self.config.cfg['mspec']['showgelview']
        self.viewParams['gelviewheight'] = self.config.cfg['mspec']['gelviewheight']

        # init spectra colours
        self.specColours = []
        self.specColours.append(self.config.cfg['colours']['spectrum0'])
        self.specColours.append(self.config.cfg['colours']['spectrum1'])
        self.specColours.append(self.config.cfg['colours']['spectrum2'])
        self.specColours.append(self.config.cfg['colours']['spectrum3'])
        self.specColours.append(self.config.cfg['colours']['spectrum4'])
        self.specColours.append(self.config.cfg['colours']['spectrum5'])
        self.specColours.append(self.config.cfg['colours']['spectrum6'])
        self.specColours.append(self.config.cfg['colours']['spectrum7'])
        self.specColours.append(self.config.cfg['colours']['spectrum8'])
        self.specColours.append(self.config.cfg['colours']['spectrum9'])

        # load cursor images
        self.makeCursors()

        # make gui items
        self.makeSpectrumCanvas()
        cursorBox = self.makeCursorBox()
        findPoint = self.makeFindPointBox()
        peakPicking = self.makePeakPickingBox()
        spectraList = self.makeSpectraList()
        self.makeCompareAddButt()

        # init spectrum
        self.resetModule()

        # pack control elements
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(cursorBox, 0, wx.EXPAND|wx.ALL, 10)
        controls.Add(findPoint, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(peakPicking, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(spectraList, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(self.compareAdd_button, 0, wx.ALIGN_CENTRE|wx.ALL, 5)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(self.spectrumCanvas, 1, wx.EXPAND)
        mainSizer.Add(controls, 0, wx.EXPAND)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def makeCursors(self):
        """ Set cursor images for main spectrum area. """

        # default cursors for non-supporting systems
        if not wx.Platform in ('__WXMSW__', '__WXMAC__'):
            self.cursorImages['normal'] = wx.StockCursor(wx.CURSOR_ARROW)
            self.cursorImages['point'] = wx.StockCursor(wx.CURSOR_CROSS)
            self.cursorImages['peak'] = wx.StockCursor(wx.CURSOR_ARROW)
            self.cursorImages['delete'] = wx.StockCursor(wx.CURSOR_ARROW)
            self.cursorImages['distance'] = wx.StockCursor(wx.CURSOR_CROSS)
            return

        # get images
        cursorImages = images.getCursorsBitmap()
        if  wx.Platform == '__WXMSW__':
            fce = { 'normal':(0, 0, 20, 22),
                    'point':(20, 0, 20, 22),
                    'peak':(40, 0, 20, 22),
                    'delete':(60, 0, 20, 22),
                    'distance':(80, 0, 22, 22)}
        elif wx.Platform == '__WXMAC__':
            fce = { 'normal':(0, 22, 16, 16),
                    'point':(16, 22, 16, 16),
                    'peak':(32, 22, 16, 16),
                    'delete':(48, 22, 16, 16),
                    'distance':(64, 22, 16, 16)}

        # set cursors
        for item in fce:
            cursor = cursorImages.GetSubBitmap(wx.Rect(fce[item][0], fce[item][1], fce[item][2], fce[item][3]))
            cursor = cursor.ConvertToImage()
            if item in ('point', 'distance') and wx.Platform == '__WXMSW__':
                cursor.SetOptionInt(wx.IMAGE_OPTION_CUR_HOTSPOT_X, 9)
                cursor.SetOptionInt(wx.IMAGE_OPTION_CUR_HOTSPOT_Y, 10)
            elif item in ('point', 'distance') and wx.Platform == '__WXMAC__':
                cursor.SetOptionInt(wx.IMAGE_OPTION_CUR_HOTSPOT_X, 7)
                cursor.SetOptionInt(wx.IMAGE_OPTION_CUR_HOTSPOT_Y, 8)
            else:
                cursor.SetOptionInt(wx.IMAGE_OPTION_CUR_HOTSPOT_X, 0)
                cursor.SetOptionInt(wx.IMAGE_OPTION_CUR_HOTSPOT_Y, 0)
            self.cursorImages[item] = wx.CursorFromImage(cursor)
    # ----


    # ----
    def makeSpectrumCanvas(self):
        """ Init blank spectrum. """

        # set style
        if wx.Platform == '__WXMAC__':
            style=wx.SIMPLE_BORDER|wx.WANTS_CHARS
        else:
            style=wx.SUNKEN_BORDER|wx.WANTS_CHARS

        self.spectrumCanvas = spectrum.plotCanvas(self, style=style)
        self.spectrumCanvas.setXLabel('m/z')
        self.spectrumCanvas.setYLabel('a. i.')
        self.spectrumCanvas.setXDigits(self.config.cfg['common']['digits'])
        self.spectrumCanvas.setReverseDraw(True)
        self.setMouseFce(self.mouseFce)
        self.spectrumCanvas.setEnableCursorImage(self.config.cfg['mspec']['cursorimage'])

        self.spectrumCanvas.setScrollFactor(self.config.cfg['mspec']['scrollx'])
        self.spectrumCanvas.setMoveFactor(self.config.cfg['mspec']['movex'])
        self.spectrumCanvas.setScaleFactor(self.config.cfg['mspec']['scalex'], self.config.cfg['mspec']['scaley'])

        self.spectrumCanvas.setEnableCursorTracker(self.viewParams['showtracker'])
        self.spectrumCanvas.setEnableCursorValue(self.viewParams['showposition'])
        self.spectrumCanvas.setEnableDistanceValue(self.viewParams['showdistance'])
        self.spectrumCanvas.setEnableGrid(self.viewParams['showgrid'])
        self.spectrumCanvas.setEnableLegend(self.viewParams['showlegend'])
        self.spectrumCanvas.setEnablePositionBar(self.viewParams['showposbar'])
        self.spectrumCanvas.setEnableGelView(self.viewParams['showgelview'])
        self.spectrumCanvas.setGelViewHeight(self.viewParams['gelviewheight'])

        # set events
        self.spectrumCanvas.Bind(wx.EVT_LEFT_UP, self.onLMU)
        self.spectrumCanvas.Bind(wx.EVT_MOTION, self.onMMotion)
    # ----


    # ----
    def makeCursorBox(self):
        """ Make box with cursor info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Cursor Info"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        cursorPos_label = wx.StaticText(self, -1, "Mass: ")
        self.cursorPos_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        cursorPos_units = wx.StaticText(self, -1, ' m/z')
        self.cursorPos_value.SetToolTip(wx.ToolTip("Cursor m/z"))

        cursorIntens_label = wx.StaticText(self, -1, "Int.: ")
        self.cursorIntens_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.cursorIntens_value.SetToolTip(wx.ToolTip("Cursor intensity"))

        cursorDist_label = wx.StaticText(self, -1, "Dist.: ")
        self.cursorDist_value = wx.TextCtrl(self, -1, '---', size=(80, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        cursorDist_units = wx.StaticText(self, -1, ' m/z')
        self.cursorDist_value.SetToolTip(wx.ToolTip("Points distance"))

        # pack items
        grid.Add(cursorPos_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.cursorPos_value, (0, 1))
        grid.Add(cursorPos_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(cursorIntens_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.cursorIntens_value, (1, 1))

        grid.Add(cursorDist_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.cursorDist_value, (2, 1))
        grid.Add(cursorDist_units, (2, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeFindPointBox(self):
        """ Make box for peak finding. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Find Point"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        findPoint_label = wx.StaticText(self, -1, "Mass: ")
        self.findPoint_value = wx.TextCtrl(self, -1, '', size=(80, -1), style=wx.TE_RIGHT|wx.TE_PROCESS_ENTER, validator=mwx.txtValidator('float'))
        findPoint_units = wx.StaticText(self, -1, ' m/z')
        self.findPoint_value.SetToolTip(wx.ToolTip("Press enter to find"))

        findZoom_choices = ['Current', '0.1', '0.5', '1', '5', '10', '20', '50', '100']
        findZoom_label = wx.StaticText(self, -1, "Zoom: ")
        self.findZoom_combo = wx.ComboBox(self, -1, 'Current', size=(80, -1), choices=findZoom_choices, style=wx.CB_READONLY)
        findZoom_units = wx.StaticText(self, -1, ' %')
        self.findZoom_combo.SetToolTip(wx.ToolTip("Zoom spectrum"))

        # pack items
        grid.Add(findPoint_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.findPoint_value, (0, 1))
        grid.Add(findPoint_units, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(findZoom_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.findZoom_combo, (1, 1))
        grid.Add(findZoom_units, (1, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        # set events
        self.findPoint_value.Bind(wx.EVT_TEXT_ENTER, self.onFindPoint)
        self.findZoom_combo.Bind(wx.EVT_COMBOBOX, self.onFindPoint)

        return mainBox
    # ----


    # ----
    def makePeakPickingBox(self):
        """ Make box with peak labeling params. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Peak Picking"), wx.VERTICAL)
        heightBox = wx.BoxSizer(wx.HORIZONTAL)

        peakPickingHeight_label = wx.StaticText(self, -1, "Height: ")
        self.peakPickingHeight_value = wx.StaticText(self, -1, ' '+str(self.peakHeight)+' %')
        self.peakPickingHeight_slider = wx.Slider(self, -1, self.peakHeight, 1, 100, size=(75, -1), style=wx.SL_HORIZONTAL)

        # pack items
        heightBox.Add(peakPickingHeight_label, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        heightBox.Add(self.peakPickingHeight_slider, 0, 0)
        heightBox.Add(self.peakPickingHeight_value, 0, wx.ALIGN_CENTER_VERTICAL, 0)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(heightBox, 0, wx.ALL, 0)
        else:
            mainBox.Add(heightBox, 0, wx.ALL, 5)

        # set events
        self.peakPickingHeight_slider.Bind(wx.EVT_SCROLL, self.onHeightChanged)

        return mainBox
    # ----


    # ----
    def makeSpectraList(self):
        """ Make box with all loaded spectra. """

        # make items
        box_label = wx.StaticText(self, -1, "Loaded Spectra: ")
        self.loadedSpectra_list = mwx.ListCtrl(self, -1, size=(155, 130))
        self.loadedSpectra_list.InsertColumn(0, "Title", wx.LIST_FORMAT_LEFT)
        self.loadedSpectra_list.InsertColumn(1, "Offset", wx.LIST_FORMAT_RIGHT)

        # set DnD
        dropTarget = mwx.FileDropTarget(self)
        dropTarget.SetDropFce('onDropFiles')
        self.loadedSpectra_list.SetDropTarget(dropTarget)

        # pack items
        mainBox = wx.BoxSizer(wx.VERTICAL)
        mainBox.Add(box_label, 0, wx.LEFT, 5)
        mainBox.Add(self.loadedSpectra_list, 0, wx.EXPAND|wx.TOP, 5)

        # set events
        self.loadedSpectra_list.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onCompareEdit)
        self.loadedSpectra_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onCompareItemSelected)
        self.loadedSpectra_list.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onSpectraListRMU)
        self.loadedSpectra_list.Bind(wx.EVT_RIGHT_UP, self.onSpectraListRMU)
        self.loadedSpectra_list.Bind(wx.EVT_COMMAND_RIGHT_CLICK, self.onSpectraListRMU)

        # set columns width
        self.loadedSpectra_list.SetColumnWidth(0, wx.LIST_AUTOSIZE_USEHEADER)
        self.loadedSpectra_list.SetColumnWidth(1, wx.LIST_AUTOSIZE_USEHEADER)

        return mainBox
    # ----


    # ----
    def makeCompareAddButt(self):
        """ Make button to add spectrum. """

        self.compareAdd_button = wx.Button(self, -1, "Add Spectrum")
        self.compareAdd_button.SetToolTip(wx.ToolTip("Add spectrum to compare"))
        self.compareAdd_button.Bind(wx.EVT_BUTTON, self.onCompareAdd)
        self.compareAdd_button.Enable(False)
    # ----


    # ----
    def resetModule(self):
        """ Set new data and refresh spectrum canvas. """

        # set application working
        self.docMonitor('setAppStatus', "Initialising spectrum...")

        # get data from document
        title = os.path.basename(self.docData.getPath())
        if not title:
            title = 'Untitled'
        spectrumData = self.docData.getSpectrum()
        peaklistData = self.docData.getPeaks()
        self.objects = []

        # get show params
        digits = self.config.cfg['common']['digits']
        labelangle = self.viewParams['labelangle']
        showlabels = self.viewParams['showlabels']
        showannots = self.viewParams['showannots']
        showticks = self.viewParams['showticks']
        if not self.docData.getSpectrum():
            showticks = True

        # set module according to spectrum
        self.spectrumCanvas.setMainObject(1)
        self.loadedSpectra = [[title, 0, self.specColours[0], True]]
        self.compareAdd_button.Enable(True)

        # update list of loaded spectra
        self.updateSpectraList()

        # make objects and draw spectrum
        self.objects.append(spectrum.peaklistObject(peaklistData, digits=digits, showlabels=showlabels, showannots=showannots, showticks=showticks, angle=labelangle))
        self.objects.append(spectrum.spectrumObject(spectrumData, colour=self.specColours[0], legend=title))
        self.container = spectrum.objectsContainer(self.objects)
        self.spectrumCanvas.draw(self.container)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def updateCursorInfo(self):
        """ Get cursor positions and update cursor info. """

        # get values
        position = self.spectrumCanvas.getCursorPosition()
        distance = self.spectrumCanvas.getDistance()
        format = '%0.'+`self.config.cfg['common']['digits']`+'f'

        # update XY position
        if position:
            mass = format % position[0]
            intens = format % position[1]
            self.cursorPos_value.SetValue(str(mass))
            self.cursorIntens_value.SetValue(str(intens))
        else:
            self.cursorPos_value.SetValue('---')
            self.cursorIntens_value.SetValue('---')

        # update distance
        if distance:
            distance = format % distance
            self.cursorDist_value.SetValue(str(distance))
        else:
            self.cursorDist_value.SetValue('---')
    # ----


    # ----
    def updateSpectraList(self):
        """ Update list of loaded spectra. """

        # clear list
        self.loadedSpectra_list.DeleteAllItems()
        self.selectedSpectrum = -1

        # paste new data
        for x, spectrum in enumerate(self.loadedSpectra):
            self.loadedSpectra_list.InsertStringItem(x, spectrum[0])
            self.loadedSpectra_list.SetStringItem(x, 1, str(spectrum[1]))
            if spectrum[3]:
                self.loadedSpectra_list.SetItemTextColour(x, spectrum[2])
            else:
                self.loadedSpectra_list.SetItemTextColour(x, (150,150,150))

        # set columns width
        if self.loadedSpectra_list.GetItemCount():
            autosize = wx.LIST_AUTOSIZE
        else:
            autosize = wx.LIST_AUTOSIZE_USEHEADER
        self.loadedSpectra_list.SetColumnWidth(0, autosize)
        self.loadedSpectra_list.SetColumnWidth(1, autosize)
        self.loadedSpectra_list.updateLastCol()
    # ----


    # ----
    def updateSpectrum(self):
        """ Set new spectrum data and refresh spectrum canvas. """

        # update spectrum data
        self.updateSpectrumData()

        # redraw plot
        self.refreshCanvas(rescale=False)
    # ----


    # ----
    def updatePeaklist(self):
        """ Set new peaklist data and refresh spectrum canvas. """

        # update peaklist data
        self.updatePeaklistData()

        # redraw plot - do not rescale if spectrum loaded, do it if peaklist only
        if self.loadedSpectra:
            self.refreshCanvas(rescale=False)
        else:
            self.refreshCanvas(rescale=True)
    # ----


    # ----
    def updateSpectrumData(self):
        """ Set new spectrum data. """

        # set application working
        self.docMonitor('setAppStatus', "Updating spectrum data...")

        # make spectrum lines
        legend = self.objects[1].getLegend()
        spectrumData = self.docData.getSpectrum()
        self.objects[1] = spectrum.spectrumObject(spectrumData, colour=self.specColours[0], legend=legend[0])

        # make spectrum container
        self.container = spectrum.objectsContainer(self.objects)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def updatePeaklistData(self):
        """ Set new peaklist data. """

        # set application working
        self.docMonitor('setAppStatus', "Updating peaklist data...")

        # make peaklist lines
        peaklistData = self.docData.getPeaks()
        self.objects[0] = spectrum.peaklistObject(peaklistData)

        # make spectrum container
        self.container = spectrum.objectsContainer(self.objects)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onShow(self):
        """ Show panel and set focus to main item. """

        self.Show(True)
        self.spectrumCanvas.SetFocus()
    # ----


    # ----
    def onLMU(self, evt):
        """ Do selected fce for left-mouse-click. """

        refresh = False

        # label current point
        if self.mouseFce in ('point', 'peak'):
            refresh = self.labelPeak()

        # delete labels
        elif self.mouseFce == 'delete':
            refresh = self.deletePeaks()

        evt.Skip()

        # refresh peaklist and spectrum
        if refresh:
            wx.CallAfter(self.updatePeaklist)
    # ----


    # ----
    def onMMotion(self, evt):
        """ Show cursor info on mouse motion. """

        evt.Skip()
        wx.CallAfter(self.updateCursorInfo)
    # ----


    # ----
    def onDropFiles(self, paths):
        """ Add dropped files to compare. """
        self.onCompareAdd(paths=paths)
    # ----


    # ----
    def onFindPoint(self, evt):
        """ Set view to selected point. """

        # get point
        point = self.findPoint_value.GetValue()
        point = point.replace(',', '.')
        self.findPoint_value.SetSelection(-1, -1)

        # check point
        if not point:
            return

        # get zoom
        zoom = self.findZoom_combo.GetValue()
        if zoom == 'Current':
            zoom = None
        else:
            zoom = float(zoom)

        # parse values
        try:
            point = float(point)
        except ValueError:
            errorMessage = "Mass must be a number!"
            errorDlg = wx.MessageDialog(self, errorMessage, "Value Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
            return

        # check point
        if not point:
            return

        # highlight point
        self.highlightPoint(point, zoom)
    # ----


    # ----
    def onHeightChanged(self, evt):
        """ Get new value for peakpicking height. """

        value = self.peakPickingHeight_slider.GetValue()
        self.peakHeight = value
        self.peakPickingHeight_value.SetLabel(' '+str(value)+' %')
    # ----


    # ----
    def onSpectraListRMU(self, evt):
        """ Raise popup menu when right-mouse click on spectra list. """

        # create menu items
        menuAddID = wx.NewId()
        menuEditID = wx.NewId()
        menuDeleteID = wx.NewId()
        menuShowID = wx.NewId()
        menuHideID = wx.NewId()

        # bind events to menu
        menu = wx.Menu()
        menu.Bind(wx.EVT_MENU, self.onCompareAdd, id=menuAddID)
        menu.Bind(wx.EVT_MENU, self.onCompareEdit, id=menuEditID)
        menu.Bind(wx.EVT_MENU, self.onCompareDelete, id=menuDeleteID)
        menu.Bind(wx.EVT_MENU, self.onCompareShow, id=menuShowID)
        menu.Bind(wx.EVT_MENU, self.onCompareHide, id=menuHideID)

        # add items to menu
        menu.Append(menuAddID, "Add spectrum...")
        menu.Append(menuEditID, "Edit title or offset...")
        menu.Append(menuDeleteID, "Delete selected")
        menu.AppendSeparator()
        menu.Append(menuShowID, "Show selected")
        menu.Append(menuHideID, "Hide selected")

        # disable all if main spectrum not loaded
        if not self.loadedSpectra:
            menu.Enable(menuAddID, False)
            menu.Enable(menuEditID, False)
            menu.Enable(menuDeleteID, False)
            menu.Enable(menuShowID, False)
            menu.Enable(menuHideID, False)

        # disable if no item selected
        if self.selectedSpectrum == -1:
            menu.Enable(menuEditID, False)
            menu.Enable(menuDeleteID, False)
            menu.Enable(menuShowID, False)
            menu.Enable(menuHideID, False)

        # disable if main spectrum (first item) selected
        if self.selectedSpectrum == 0:
            menu.Enable(menuDeleteID, False)

        # disable if limit reached
        if len(self.loadedSpectra) >= 10:
            menu.Enable(menuAddID, False)

        self.loadedSpectra_list.PopupMenu(menu)
        menu.Destroy()
    # ----


    # ----
    def onCompareItemSelected(self, evt):
        """ Remember selected spectrum. """
        self.selectedSpectrum = evt.m_itemIndex
    # ----


    # ----
    def onViewOptions(self, evt=None):
        """ Show view option dialog. """

        dlg = dlgViewOptions(self, self.viewParams)
        if dlg.ShowModal() != wx.ID_OK:
            dlg.Destroy()
            return False
        else:
            self.viewParams = dlg.getData()
            dlg.Destroy()
            self.refreshCanvas()
    # ----


    # ----
    def onCompareAdd(self, evt=None, paths=''):
        """ Load spectrum to compare with current. """

        # set application working
        self.docMonitor('setAppStatus', "Importing spectrum...")

        # get used colours
        usedColours = []
        for col in self.loadedSpectra:
            usedColours.append(col[2])

        # init importer
        importer = mDocumentImporter(self)

        # show dialog to get document path
        if not paths:
            paths = importer.showDialog(self.config.cfg['common']['lastspecdir'], multiple=True)

        if not paths:
            self.docMonitor('setAppStatus', 0)
            return

        # import data
        count = 0
        for path in paths:
            data = importer.importData(path)

            # stop if no valid data loaded
            if not data or not data['spectrum']:
                if data != None:
                    message = "Document contains no spectrum data."
                    dlg = wx.MessageDialog(self, message, "Import Error", wx.OK|wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()
                continue
            else:
                count += 1

            # stop if maximum count reached
            if len(self.loadedSpectra) >= 10:
                message = "Maximum number of loaded spectra reached!"
                dlg = wx.MessageDialog(self, message, "Maximum Reached", wx.OK|wx.ICON_INFORMATION)
                dlg.ShowModal()
                dlg.Destroy()
                break

            # set spectrum colour
            for col in self.specColours[1:]:
                if not col in usedColours:
                    colour = col
                    usedColours.append(col)
                    break

            # add spectrum lines to container
            title = os.path.basename(data['path'])
            self.loadedSpectra.append([title, 0, colour, True])
            self.objects.append(spectrum.spectrumObject(data['spectrum'], colour=colour, legend=title))
            self.container = spectrum.objectsContainer(self.objects)

            # add peaklist to mComp
            if self.config.cfg['mspec']['importpeaklist']:
                self.docMonitor('sendPeaklist', (data['peaklist'], title))

        # set application ready
        self.docMonitor('setAppStatus', 0)

        if count == 0:
            return

        # refresh spectrum and update spectrum list
        self.updateSpectraList()
        self.refreshCanvas()

        # update Add button
        if len(self.loadedSpectra) >= 10:
            self.compareAdd_button.Enable(False)
    # ----


    # ----
    def onCompareEdit(self, evt):
        """ Change spectrum name and offset. """

        # get title and offset for selected spetrum
        title = self.loadedSpectra[self.selectedSpectrum][0]
        offset = self.loadedSpectra[self.selectedSpectrum][1]
        colour = self.loadedSpectra[self.selectedSpectrum][2]

        # rise spectrum params dialog
        dlg = dlgEditSpecParams(self, title, offset)
        if dlg.ShowModal() != wx.ID_OK:
            dlg.Destroy()
            return False
        else:
            title, offset = dlg.getParams()
            dlg.Destroy()

        # set application working
        self.docMonitor('setAppStatus', "Updating Spectrum...")

        # offset spectrum data
        if self.selectedSpectrum > 0:
            self.objects[self.selectedSpectrum+1].offsetPoints(0, offset)
        else:
            offset = 0

        # set title
        self.objects[self.selectedSpectrum+1].setParameters(show=True, legend=title)

        # update container
        self.container = spectrum.objectsContainer(self.objects)

        # set application ready
        self.docMonitor('setAppStatus', 0)

        # refresh spectrum and update spectrum list
        self.loadedSpectra[self.selectedSpectrum] = [title, offset, colour, True]
        self.updateSpectraList()
        self.refreshCanvas()
    # ----


    # ----
    def onCompareDelete(self, evt):
        """ Delete selected comparing spectra. """

        # raise delete warning
        dlg = wx.MessageDialog(self, "Delete selected spectra?", "Delete Spectra", wx.YES_NO|wx.ICON_QUESTION)
        if dlg.ShowModal() != wx.ID_YES:
            dlg.Destroy()
            return False
        else:
            dlg.Destroy()

        # get selected items
        items = mwx.getSelectedListItems(self.loadedSpectra_list, reverse=True)

        # deletd selected spectra
        for item in items:
            if item != 0:
                del self.loadedSpectra[item]
                del self.container[item+1]

        # refresh spectrum and update spectrum list
        self.updateSpectraList()
        self.refreshCanvas()

        # update Add button
        self.compareAdd_button.Enable(True)
    # ----


    # ----
    def onCompareShow(self, evt):
        """ Set selected spectra visible. """

        items = mwx.getSelectedListItems(self.loadedSpectra_list, reverse=True)
        self.setVisibility(items, True)
    # ----


    # ----
    def onCompareHide(self, evt):
        """ Set selected spectra invisible. """

        items = mwx.getSelectedListItems(self.loadedSpectra_list, reverse=True)
        self.setVisibility(items, False)
    # ----


    # ----
    def setVisibility(self, items, visible=True):
        """ Set visibility of selected spectra. """

        # set visibility
        for item in items:
            self.loadedSpectra[item][3] = visible
            self.objects[item+1].setParameters(show=visible)
            self.container = spectrum.objectsContainer(self.objects)

        # refresh spectrum and update spectrum list
        self.updateSpectraList()
        self.refreshCanvas()
    # ----


    # ----
    def setMouseFce(self, fce):
        """ Set fce for left-mouse-button and cursor type. """

        # check spectrum data
        if not self.docData.getDataStatus('mSpec') and fce in ('point', 'peak'):
            fce = 'none'

        # set leftmouse button fce
        self.mouseFce = fce

        # set mouse drawing type
        if fce == 'distance':
            self.spectrumCanvas.setLMBFunction('distance')
            self.spectrumCanvas.setEnableCursorTracker(self.viewParams['showtracker'])
        elif fce == 'point':
            self.spectrumCanvas.setLMBFunction('point')
            self.spectrumCanvas.setEnableCursorTracker(self.viewParams['showtracker'])
        elif fce in ('peak', 'delete'):
            self.spectrumCanvas.setLMBFunction('rectangle')
            self.spectrumCanvas.setEnableCursorTracker(False)
        else:
            self.spectrumCanvas.setLMBFunction('none')
            self.spectrumCanvas.setEnableCursorTracker(False)

        # set main cursor
        self.spectrumCanvas.setMainCursor(self.cursorImages[fce])
    # ----


    # ----
    def setViewParam(self, param, value):
        """ Set value to selected view parameter. """
        self.viewParams[param] = value
    # ----


    # ----
    def setRange(self):
        """ Set spectrum range. """
        self.spectrumCanvas.setRange()
    # ----


    # ----
    def getPrintOut(self, pageSetup, title):
        """ Get printout of current plot. """
        return self.spectrumCanvas.getPrintOut(pageSetup, self.config.cfg['common']['printfilter'], title)
    # ----


    # ----
    def getBitmap(self, width, height):
        """ Get bitmap of current plot. """
        return self.spectrumCanvas.getBitmap(width, height, self.config.cfg['common']['exportfilter'])
    # ----


    # ----
    def refreshCanvas(self, rescale=False):
        """ Get and set canvas params and redraw spectrum. """

        # set application working
        self.docMonitor('setAppStatus', "Refreshing spectrum...")

        # get show params
        digits = self.config.cfg['common']['digits']
        showlabels = self.viewParams['showlabels']
        showannots = self.viewParams['showannots']
        showticks = self.viewParams['showticks']
        labelangle = self.viewParams['labelangle']
        showtracker = self.viewParams['showtracker']
        showposition = self.viewParams['showposition']
        showdistance = self.viewParams['showdistance']
        showgrid = self.viewParams['showgrid']
        showlegend = self.viewParams['showlegend']
        showposbar = self.viewParams['showposbar']
        showgelview = self.viewParams['showgelview']
        gelviewheight = self.viewParams['gelviewheight']
        if not self.docData.getSpectrum():
            showticks = True

        # reset peaklist params
        self.objects[0].setParameters(digits=digits, showlabels=showlabels, showannots=showannots, showticks=showticks, angle=labelangle)

        # reset canvas params
        self.spectrumCanvas.setXDigits(digits)
        self.spectrumCanvas.setEnableGrid(showgrid)
        self.spectrumCanvas.setEnableLegend(showlegend)
        self.spectrumCanvas.setEnablePositionBar(showposbar)
        self.spectrumCanvas.setEnableGelView(showgelview)
        self.spectrumCanvas.setGelViewHeight(gelviewheight)
        self.spectrumCanvas.setEnableCursorValue(showposition)
        self.spectrumCanvas.setEnableDistanceValue(showdistance)
        if self.mouseFce not in ('peak', 'delete'):
            self.spectrumCanvas.setEnableCursorTracker(showtracker)

        # refresh spectrum
        self.spectrumCanvas.refresh(rescale)

        # set focus on spectrum canvas
        self.spectrumCanvas.SetFocus()

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def labelPeak(self):
        """ Get peak coords and add them to peaklist. """

        peak = None
        spectrumData = self.docData.getSpectrum()

        # label point
        if self.mouseFce == 'point':
            cursorPos = self.spectrumCanvas.getMouseClick()
            if cursorPos:
                peak = count.getPoint(spectrumData, cursorPos[0])

        # label peak
        elif self.mouseFce == 'peak':
            selection = self.spectrumCanvas.getSelectionBox()
            peak = count.getCentroid(spectrumData, selection, self.peakHeight)

        # add peak to peaklist
        if peak:
            self.docData.addPeak(peak)
            self.docMonitor('onPeaklistChanged', 'specAdd')
            return True
        else:
            return False
    # ----


    # ----
    def deletePeaks(self):
        """ Delete peaks within selected area. """

        # get rectangle coords
        selection = self.spectrumCanvas.getSelectionBox()
        if selection:

            # get indexes of deleted peaks
            indexes = []
            peaklist = self.docData.getPeaks()
            for i, peak in enumerate(peaklist):
                if (selection[0] <= peak[0] <= selection[2]) and (selection[1] <= peak[1] <= selection[3]):
                    indexes.append(i)

            # delete peaks
            if indexes:
                self.docData.deletePeaks(indexes)
                self.docMonitor('onPeaklistChanged', 'specDelete')
                return True

        return False
    # ----


    # ----
    def highlightPoint(self, x, zoom=False):
        """ Ensure X point visible and show point arrow. """
        self.spectrumCanvas.highlighXPoint(x, zoom)
    # ----


    # ----
    def copySpectrumToClipboard(self):
        """ Copy current spectrum view to clipboard. """

        # get bitmap
        bitmap = self.spectrumCanvas.getBitmap()
        data = wx.BitmapDataObject(bitmap)

        # paste to clipboard
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(data)
            wx.TheClipboard.Close()
        else:
            message = "Clipboard error! Cannot copy data to clipboard."
            dlg = wx.MessageDialog(self, message, "Copy Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
    # ----
