/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <errno.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_proto.h"
#include "udm_services.h"
#include "udm_agent.h"
#include "udm_db.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_sdp.h"
#include "udm_xmalloc.h"
#include "udm_searchtool.h"
#include "udm_vars.h"
#include "udm_word.h"
#include "udm_db_int.h"
#include "udm_log.h"


/*
#define DEBUG_SDP
*/

/*
#define DEBUG_SEARCH
*/

static ssize_t UdmSearchdSendPacket(int fd,const UDM_SEARCHD_PACKET_HEADER *hdr,const void *data){
	ssize_t nsent;
	
	nsent=UdmSend(fd,hdr,sizeof(*hdr),0);
	if(nsent!=sizeof(*hdr))
		return(nsent);
	if(data){
		nsent+=UdmSend(fd,data,hdr->len,0);
	}
	return nsent;
}


#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

static int open_host(const char *hostname,int port, int timeout)
{
	int net;
	struct hostent *host;
	struct sockaddr_in sa_in;

	bzero((char*)&sa_in,sizeof(sa_in));

	if (port){
		sa_in.sin_port= htons((u_short)port);
	}else{
		return(UDM_NET_ERROR);
	}

	if ((sa_in.sin_addr.s_addr=inet_addr(hostname)) != INADDR_NONE){
		sa_in.sin_family=AF_INET;
	}else{
		host=gethostbyname(hostname);
		if (host){
			sa_in.sin_family=host->h_addrtype;
			memcpy(&sa_in.sin_addr, host->h_addr, (size_t)host->h_length);
		}else{
			return(UDM_NET_CANT_RESOLVE);
		}
	}
	net=socket(AF_INET, SOCK_STREAM, 0);

	if(connect(net, (struct sockaddr *)&sa_in, sizeof (sa_in)))
		return(UDM_NET_CANT_CONNECT);

	return(net);
}

int UdmSearchdConnect(UDM_DB *cl){
	int res=UDM_OK;
	const char* hostname= UdmVarListFindStr(&cl->Vars, "DBHost", "localhost");
	int port= UdmVarListFindInt(&cl->Vars, "DBPort", UDM_SEARCHD_PORT);
	cl->searchd = open_host(hostname, port, 0);
	if(cl->searchd<=0)
	{
		cl->searchd=0;
/*		UdmLog(query, UDM_LOG_ERROR, "Can't connect to searchd at '%s:%d'", host, port);*/
		res=UDM_ERROR;
	}
	return res;
}

void UdmSearchdClose(UDM_DB *cl) {
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t nsent;
	
	if(cl->searchd>0){
		/* Send goodbye */
		hdr.cmd=UDM_SEARCHD_CMD_GOODBYE;
		hdr.len=0;
		nsent=UdmSearchdSendPacket(cl->searchd,&hdr,NULL);
		closesocket(cl->searchd);
		cl->searchd=0;
	}
}

int __UDMCALL UdmResAddDocInfoSearchd(UDM_AGENT * query,UDM_DB *cl,UDM_RESULT * Res,size_t clnum){
	UDM_SEARCHD_PACKET_HEADER hdr;
	char * msg=NULL;
	size_t i /* ,num=0,curnum=0 */;
	int done = 0;
	ssize_t nsent,nrecv;
	char * dinfo=NULL;
	int	rc=UDM_OK;
	char		textbuf[10*1024];
	size_t dlen = 0;
	
	if(!Res->num_rows) return(UDM_OK);
	
	for(i=0;i<Res->num_rows;i++){
	  size_t		ulen;
	  size_t		olen;
	  size_t		nsec;
	  UDM_DOCUMENT	*D=&Res->Doc[i];

	  for(nsec=0;nsec<D->Sections.nvars;nsec++)
	    if (strcasecmp(D->Sections.Var[nsec].name, "Score") == 0) D->Sections.Var[nsec].section=1;

	  UdmDocToTextBuf(D, textbuf, sizeof(textbuf)-1);
	  textbuf[sizeof(textbuf)-1]='\0';
					
	  ulen = strlen(textbuf)+2;
	  olen = dlen;
	  dlen = dlen + ulen;
	  dinfo = (char*)UdmRealloc(dinfo, dlen + 1);
	  dinfo[olen] = '\0';
	  sprintf(dinfo + olen, "%s\r\n", textbuf);
	}
	hdr.cmd=UDM_SEARCHD_CMD_DOCINFO;
	hdr.len = strlen(dinfo);
	
	nsent = UdmSearchdSendPacket(cl->searchd, &hdr, dinfo);
	
	while(!done){
		char * tok, * lt;
		nrecv=UdmRecvall(cl->searchd,&hdr,sizeof(hdr));
		
		if(nrecv!=sizeof(hdr)){
			UdmLog(query, UDM_LOG_ERROR, "Received incomplete header from searchd (%d bytes)", (int)nrecv);
			return(UDM_ERROR);
		}else{
#ifdef DEBUG_SDP
			fprintf(stderr, "Received header cmd=%d len=%d\n",hdr.cmd,hdr.len);
#endif
		}
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg=(char*)UdmMalloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
				sprintf(query->Conf->errstr,"Searchd error: '%s'",msg);
				rc=UDM_ERROR;
				UDM_FREE(msg);
				done=1;
				break;
			case UDM_SEARCHD_CMD_MESSAGE:
				msg=(char*)UdmMalloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Message from searchd: '%s'\n",msg);
#endif
				UDM_FREE(msg);
				break;
			case UDM_SEARCHD_CMD_DOCINFO:
				dinfo = (char*)UdmRealloc(dinfo, hdr.len + 1);
				nrecv=UdmRecvall(cl->searchd,dinfo,hdr.len);
				dinfo[hdr.len]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Received DOCINFO size=%d buf=%s\n",hdr.len,dinfo);
#endif				
				tok = udm_strtok_r(dinfo, "\r\n", &lt);
				
				while(tok){
					UDM_DOCUMENT Doc;
					
					UdmDocInit(&Doc);
					UdmDocFromTextBuf(&Doc,tok);
					
					for(i=0;i<Res->num_rows;i++){
						urlid_t	Res_Doc_url_id = (urlid_t)UdmVarListFindInt(&Res->Doc[i].Sections, "ID", 0);
						urlid_t	Doc_url_id = (urlid_t)UdmVarListFindInt(&Doc.Sections, "ID", 0);
						
						if(Res_Doc_url_id!=Doc_url_id)
							continue;
						UdmDocFromTextBuf(&Res->Doc[i],tok);
						break;
					}
					tok = udm_strtok_r(NULL, "\r\n", &lt);
					UdmDocFree(&Doc);
				}
				UDM_FREE(dinfo);
				done=1;
				break;
			default:
				sprintf(query->Conf->errstr,"Unknown searchd response: cmd=%d len=%d",hdr.cmd,hdr.len);
				rc=UDM_ERROR;
				done=1;
				break;
		}
	}
	return rc;
}

static int UdmSearchdSendWordRequest(UDM_AGENT * query,UDM_DB *cl,const char * q){
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t	nsent;
	
	hdr.cmd=UDM_SEARCHD_CMD_WORDS;
	hdr.len=strlen(q);
	nsent=UdmSearchdSendPacket(cl->searchd,&hdr,q);
	
	return UDM_OK;
}


static UDM_URL_CRD *UdmSearchdGetWordResponse(UDM_AGENT *query,UDM_RESULT *Res,UDM_DB *cl,int *rc){
	UDM_URL_CRD *wrd=NULL;
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t	nrecv;
	char	*msg;
	int	done=0;
	char *wbuf, *p;
	UDM_WIDEWORDLIST *wwl;
	UDM_WIDEWORD *ww;
	size_t i;
	
	Res->total_found=0;
	rc[0]=UDM_OK;
	
	while(!done){
		nrecv=UdmRecvall(cl->searchd,&hdr,sizeof(hdr));
		if(nrecv!=sizeof(hdr)){
			sprintf(query->Conf->errstr,"Received incomplete header from searchd (%d bytes)",(int)nrecv);
			rc[0]=UDM_ERROR;
			return(NULL);
		}
#ifdef DEBUG_SDP
		fprintf(stderr, "Received header cmd=%d len=%d\n",hdr.cmd,hdr.len);
#endif
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg=(char*)UdmMalloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
				sprintf(query->Conf->errstr,"Searchd error: '%s'",msg);
				rc[0]=UDM_ERROR;
				UDM_FREE(msg);
				done=1;
				break;
			case UDM_SEARCHD_CMD_MESSAGE:
				msg=(char*)UdmMalloc(hdr.len+1); /* Add checking */
				nrecv=UdmRecvall(cl->searchd,msg,hdr.len);
				msg[nrecv]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Message from searchd: '%s'\n",msg);
#endif
				UDM_FREE(msg);
				break;
			case UDM_SEARCHD_CMD_WORDS:
				UDM_FREE(wrd);
				wrd=(UDM_URL_CRD*)UdmMalloc(hdr.len + 1); /* FIXME: add checking */
				nrecv=UdmRecvall(cl->searchd,wrd,hdr.len);
				Res->total_found=hdr.len/sizeof(*wrd);
#ifdef DEBUG_SDP
				fprintf(stderr, "Received words size=%d nwrd=%d\n",hdr.len,Res->total_found);
#endif
				done=1;
				break;
		        case UDM_SEARCHD_CMD_PERSITE:
			        Res->PerSite = (size_t*)UdmMalloc(hdr.len + 1);
				nrecv = UdmRecvall(cl->searchd, Res->PerSite, hdr.len);
#ifdef DEBUG_SDP
				fprintf(stderr, "Received PerSite size=%d nwrd=%d\n",hdr.len,Res->total_found);
#endif
				break;
		        case UDM_SEARCHD_CMD_DATA:
			        Res->CoordList.Data = (UDM_URLDATA*)UdmMalloc(hdr.len + 1);
				nrecv = UdmRecvall(cl->searchd, Res->CoordList.Data, hdr.len);
#ifdef DEBUG_SDP
				fprintf(stderr, "Received URLDATA size=%d nwrd=%d\n", hdr.len, Res->total_found);
#endif
				break;
		        case UDM_SEARCHD_CMD_WITHOFFSET:
/*				Res->offset = 1;*/
				break;
		        case UDM_SEARCHD_CMD_WWL:
				Res->PerSite = NULL;
			        if ((wbuf = p = (char *)UdmXmalloc(hdr.len)) != NULL) 
				  if (UdmRecvall(cl->searchd, wbuf, hdr.len))  {
				    wwl = (UDM_WIDEWORDLIST *)p;
				    p += sizeof(UDM_WIDEWORDLIST);
#ifdef DEBUG_SDP
				    fprintf(stderr, "wbuf :%x, wwl: %x, p: %x hdr.len:%d\n", wbuf, wwl, p, hdr.len);
				    fprintf(stderr, "Received WWL nwords=%d nuniq=%d\n", wwl->nwords, wwl->nuniq);
#endif
				    for(i = 0; i < wwl->nwords; i++) {
				      ww = (UDM_WIDEWORD *)p;
				      p += sizeof(UDM_WIDEWORD);
				      ww->word = p;
#ifdef DEBUG_SDP
				      fprintf(stderr, "Word: %s\n", ww->word);
#endif
				      p += ww->len + 1;
				      p += sizeof(int) - ((SDPALIGN)p % sizeof(int));
				      ww->uword = (int*)p;
				      p += sizeof(int) * (ww->len + 1);
				      UdmWideWordListAdd(&Res->WWList, ww);
				    }
				    Res->WWList.nuniq = wwl->nuniq;
				    UDM_FREE(wbuf);
				  }
				break;
			default:
				sprintf(query->Conf->errstr,"Unknown searchd response: cmd=%d len=%d",hdr.cmd,hdr.len);
				rc[0]=UDM_ERROR;
				done=1;
				break;
		}
	}
	return wrd;
}

int __UDMCALL UdmFindWordsSearchd(UDM_AGENT *query,UDM_RESULT *Res, UDM_DB *searchd){
	size_t		maxlen = 16384;
	char		*request, *edf = NULL;
	const char *df = UdmVarListFindStr(&query->Conf->Vars, "DateFormat", NULL);
	int		res=UDM_OK;

	if (NULL==(request=(char*)UdmMalloc(maxlen))) {
		sprintf(query->Conf->errstr,"Can't allocate memory");
		return UDM_ERROR;
	}
	
	if (df) {
	  edf = (char*)UdmMalloc(strlen(df) * 10);
	  UdmEscapeURL(edf, df);
	}

	udm_snprintf(request, maxlen, "%s&BrowserCharset=%s&IP=%s&tmplt=%s%s%s",
		 UdmVarListFindStr(&query->Conf->Vars, "QUERY_STRING", ""),
		 UdmVarListFindStr(&query->Conf->Vars, "BrowserCharset", "iso-8859-1"),
		 UdmVarListFindStr(&query->Conf->Vars, "IP", ""),
		 UdmVarListFindStr(&query->Conf->Vars, "tmplt", ""),
		 (edf) ? "&DateFormat=" : "", (edf) ? edf : ""
		 );
	UDM_FREE(edf);

	request[maxlen-1]='\0';
	if(UDM_OK != (res = UdmSearchdSendWordRequest(query, searchd, request))){
	  return res;
	}
	UDM_FREE(request);

	Res->CoordList.Coords = UdmSearchdGetWordResponse(query, Res, searchd, &res);
	
	return res;
}


int __UDMCALL UdmSearchdCatAction(UDM_AGENT *A, UDM_CATEGORY *C, int cmd, void *db) {
	UDM_DB		*searchd = db;
	UDM_SEARCHD_PACKET_HEADER hdr;
	char *buf;
	ssize_t nsent, nrecv;
	int done = 0;
	int rc=UDM_OK;
	char *msg = NULL;
	char *dinfo = NULL;

	hdr.cmd = UDM_SEARCHD_CMD_CATINFO;
	hdr.len = sizeof(int) + strlen(C->addr) + 1;
	
	if ((buf = (char*)UdmMalloc(hdr.len)) == NULL) {
	  UdmLog(A, UDM_LOG_ERROR, "Out of memory");
	  return UDM_ERROR;
	}

	*((int*)buf) = cmd;
	strcpy(buf + sizeof(int), C->addr);

	nsent = UdmSearchdSendPacket(searchd->searchd, &hdr, buf);

	UDM_FREE(buf);

	while(!done) {
		char * tok, * lt;
		nrecv = UdmRecvall(searchd->searchd, &hdr, sizeof(hdr));
		
		if(nrecv != sizeof(hdr)){
			UdmLog(A, UDM_LOG_ERROR, "Received incomplete header from searchd (%d bytes)", (int)nrecv);
			return(UDM_ERROR);
		}else{
#ifdef DEBUG_SDP
			fprintf(stderr, "Received header cmd=%d len=%d\n", hdr.cmd, hdr.len);
#endif
		}
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg = (char*)UdmMalloc(hdr.len + 1); /* Add checking */
				nrecv = UdmRecvall(searchd->searchd, msg, hdr.len);
				msg[nrecv] = '\0';
				sprintf(A->Conf->errstr, "Searchd error: '%s'", msg);
				rc=UDM_ERROR;
				UDM_FREE(msg);
				done=1;
				break;
			case UDM_SEARCHD_CMD_MESSAGE:
				msg=(char*)UdmMalloc(hdr.len+1); /* Add checking */
				nrecv = UdmRecvall(searchd->searchd, msg, hdr.len);
				msg[nrecv] = '\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Message from searchd: '%s'\n",msg);
#endif
				UDM_FREE(msg);
				break;
			case UDM_SEARCHD_CMD_CATINFO:
			        dinfo=(char*)UdmMalloc(hdr.len+1);
				nrecv=UdmRecvall(searchd->searchd, dinfo, hdr.len);
				dinfo[hdr.len]='\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Received CATINFO size=%d buf=%s\n",hdr.len,dinfo);
#endif				

				C->ncategories = 0;
				tok = udm_strtok_r(dinfo, "\r\n", &lt);
				
				while(tok){
					UdmCatFromTextBuf(C, tok);
					
					tok = udm_strtok_r(NULL, "\r\n", &lt);
				}
				UDM_FREE(dinfo);
				done=1;
				break;
			default:
				sprintf(A->Conf->errstr, "Unknown searchd response: cmd=%d len=%d", hdr.cmd, hdr.len);
				rc=UDM_ERROR;
				done = 1;
				break;
		}
	}
	return rc;
}

int __UDMCALL UdmSearchdURLAction(UDM_AGENT *A, UDM_DOCUMENT *D, int cmd, void *db) {
	UDM_DB		*searchd = db;

	UDM_SEARCHD_PACKET_HEADER hdr;
	char *buf;
	ssize_t nsent, nrecv;
	int done = 0;
	char *msg = NULL;
	char *dinfo = NULL;
	int	rc=UDM_OK;

	if (cmd != UDM_URL_ACTION_DOCCOUNT) {
	  UdmLog(A, UDM_LOG_ERROR, "searchd: unsupported URL action");
	  return UDM_ERROR;
	}
	
	hdr.cmd = UDM_SEARCHD_CMD_URLACTION;
	hdr.len = sizeof(int);
	
	if ((buf = (char*)UdmMalloc(hdr.len)) == NULL) {
	  UdmLog(A, UDM_LOG_ERROR, "Out of memory");
	  return UDM_ERROR;
	}

	*((int*)buf) = cmd;

	nsent = UdmSearchdSendPacket(searchd->searchd, &hdr, buf);

	UDM_FREE(buf);

	while(!done) {

		nrecv = UdmRecvall(searchd->searchd, &hdr, sizeof(hdr));
		
		if(nrecv != sizeof(hdr)){
			UdmLog(A, UDM_LOG_ERROR, "Received incomplete header from searchd (%d bytes)", (int)nrecv);
			return(UDM_ERROR);
		}else{
#ifdef DEBUG_SDP
			fprintf(stderr, "Received header cmd=%d len=%d\n", hdr.cmd, hdr.len);
#endif
		}
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg = (char*)UdmMalloc(hdr.len + 1); /* Add checking */
				nrecv = UdmRecvall(searchd->searchd, msg, hdr.len);
				msg[nrecv] = '\0';
				sprintf(A->Conf->errstr, "Searchd error: '%s'", msg);
				rc=UDM_OK;
				UDM_FREE(msg);
				done=1;
				break;
			case UDM_SEARCHD_CMD_MESSAGE:
				msg=(char*)UdmMalloc(hdr.len+1); /* Add checking */
				nrecv = UdmRecvall(searchd->searchd, msg, hdr.len);
				msg[nrecv] = '\0';
#ifdef DEBUG_SDP
				fprintf(stderr, "Message from searchd: '%s'\n",msg);
#endif
				UDM_FREE(msg);
				break;
			case UDM_SEARCHD_CMD_DOCCOUNT:
			        dinfo=(char*)UdmMalloc(hdr.len+1);
				nrecv=UdmRecvall(searchd->searchd, dinfo, hdr.len);
				dinfo[hdr.len]='\0';

				A->doccount += *((int *)dinfo);
#ifdef DEBUG_SDP
				UdmLog(A, UDM_LOG_DEBUG, "Received DOCCOUNT size=%d doccount=%d(+%s)\n", hdr.len, A->doccount, dinfo);
#endif				
				UDM_FREE(dinfo);
				done=1;
				break;
			default:
				sprintf(A->Conf->errstr, "Unknown searchd response: cmd=%d len=%d", hdr.cmd, hdr.len);
				rc=UDM_ERROR;
				done = 1;
				break;
		}
	}
	return rc;
}

int UdmCloneListSearchd(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_RESULT *Res, UDM_DB *db) {
	UDM_SEARCHD_PACKET_HEADER hdr;
	ssize_t	nsent,nrecv;
	char *msg = NULL, *dinfo = NULL;
	char *tok, *lt;
	char buf[128];
	int done = 0;
	int	rc = UDM_OK;
	
	udm_snprintf(buf, 128, "%s %s", UdmVarListFindStr(&Doc->Sections, "crc32", "0"), UdmVarListFindStr(&Doc->Sections, "ID", "0"));
	hdr.cmd = UDM_SEARCHD_CMD_CLONES;
	hdr.len = strlen(buf);
	nsent = UdmSearchdSendPacket(db->searchd, &hdr, buf);
	while(!done){
		nrecv = UdmRecvall(db->searchd, &hdr, sizeof(hdr));
		
		if(nrecv != sizeof(hdr)){
			UdmLog(Indexer, UDM_LOG_ERROR, "Received incomplete header from searchd (%d bytes)", (int)nrecv);
			return(UDM_ERROR);
		}else{
#ifdef DEBUG_SDP
			UdmLog(Indexer, UDM_LOG_DEBUG, "Received header cmd=%d len=%d\n", hdr.cmd, hdr.len);
#endif
		}
		switch(hdr.cmd){
			case UDM_SEARCHD_CMD_ERROR:
				msg = (char*)UdmMalloc(hdr.len + 1); /* Add checking */
				nrecv = UdmRecvall(db->searchd, msg, hdr.len);
				msg[nrecv] = '\0';
				sprintf(Indexer->Conf->errstr, "Searchd error: '%s'", msg);
				rc = UDM_ERROR;
				UDM_FREE(msg);
				done = 1;
				break;
			case UDM_SEARCHD_CMD_DOCINFO:
				dinfo = (char*)UdmMalloc(hdr.len + 1);
				nrecv = UdmRecvall(db->searchd, dinfo, hdr.len);
				dinfo[hdr.len] = '\0';
#ifdef DEBUG_SDP
				UdmLog(Indexer, UDM_LOG_DEBUG, "Received DOCINFO size=%d buf=%s\n", hdr.len, dinfo);
#endif				
				if (strcmp(dinfo, "nocloneinfo")) tok = udm_strtok_r(dinfo, "\r\n", &lt);
				else tok = NULL;
				
				while(tok){
					UDM_DOCUMENT *D;
					size_t nd = Res->num_rows++;

					Res->Doc = (UDM_DOCUMENT*)UdmRealloc(Res->Doc, (Res->num_rows) * sizeof(UDM_DOCUMENT));
					if (Res->Doc == NULL) {
					  sprintf(Indexer->Conf->errstr, "Realloc error");
					  rc = UDM_ERROR;
					  break;
					}
					D = &Res->Doc[nd];
					UdmDocInit(D);
					UdmDocFromTextBuf(D, tok);
					tok = udm_strtok_r(NULL, "\r\n", &lt);
				}
				UDM_FREE(dinfo);
				done = 1;
				break;
			default:
				sprintf(Indexer->Conf->errstr, "Unknown searchd response: cmd=%d len=%d", hdr.cmd, hdr.len);
				rc = UDM_ERROR;
				done = 1;
				break;
		}
	}
	return rc;
}
