/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <ctype.h>

#include "udmsearch.h"
#include "udm_db.h"
#include "udm_http.h"
#include "udm_parsehtml.h"
#include "udm_host.h"
#include "udm_contentencoding.h"
#include "udm_utils.h"
#include "udm_wild.h"

/******************* Template functions ********************/


static size_t out_string(FILE * stream, char * dst, size_t dst_len, const char * src)
{
  if(src)
  {
    if(stream)
      fputs(src,stream);
    if(dst)
    {
      strncat(dst, src, dst_len - 1);
      return(strlen(src));
    }
  }
  return(0);
}

static char * HiLightDup(const char * src,const char * beg, const char * end)
{
  size_t len=1;
  size_t blen=strlen(beg);
  size_t elen=strlen(end);
  const char * s;
  char  * res, *d;
  
  for(s=src;*s;s++)
  {
    switch(*s)
    {
      case '\2':
        len+=blen;
        break;
      case '\3':
        len+=elen;
        break;
      default:
        len++;
    }
  }
  res = (char*)UdmMalloc(len);
  for(s=src,d=res;*s;s++)
  {
    switch(*s)
    {
      case '\2':
        strcpy(d,beg);
        d+=blen;
        break;
      case '\3':
        strcpy(d,end);
        d+=elen;
        break;
      default:
        *d=*s;
        d++;
    }
  }
  *d='\0';
  return(res);
}


static char *UdmConvDup(const char *src,
                        UDM_CHARSET *from, 
                        UDM_CHARSET *to, int fl)
{
  UDM_CONV cnv;
  size_t len;
  char *dst;
  len= strlen(src);
  dst= (char*) UdmMalloc(len * 12 + 1);
  UdmConvInit(&cnv, from, to, fl);
  UdmConv(&cnv, dst, len * 12 + 1, src, len + 1);
  return dst;
}


size_t UdmTemplatePrintVar(UDM_ENV *Env, 
                           FILE *stream, char *dst, size_t dstlen,
                           const char *value, int type,
                           const char *HlBeg,
                           const char *HlEnd)
{
  char *eval= NULL;
  size_t res= 0;
  
  switch(type)
  {
    case '(': 
      eval = UdmRemoveHiLightDup(value);
      res= out_string(stream, dst, dstlen, eval);
      break;
    case '&':
      {
        char *cval= UdmConvDup(value, Env->bcs, Env->bcs, UDM_RECODE_HTML);
        eval= HiLightDup(cval,HlBeg,HlEnd);
        UDM_FREE(cval);
      }
      res= out_string(stream, dst, dstlen, eval);
      break;
    case '^':
      eval=HiLightDup(value,HlBeg,HlEnd);
      res= out_string(stream, dst, dstlen, eval);
      break;
    case '%':
      eval= (char*)UdmMalloc(strlen(value) * 3 + 1);
      UdmEscapeURL(eval,value);
      res= out_string(stream, dst, dstlen, eval);
      break;
  }
  UDM_FREE(eval);
  return res;
}

static size_t PrintTextTemplate(UDM_AGENT *A, FILE *stream, char *dst, 
                                size_t dst_len, UDM_VARLIST *vars,
                                const char *templ,
                                const char *HlBeg,
                                const char *HlEnd)
{
  const char * s;
  size_t dlen=0;
  
  for(s=templ; (*s) && ((stream) || (dlen < dst_len)); s++)
  {
    int type=0;
    char *value=NULL, *newvalue = NULL;
    char empty[]="";
    size_t maxlen=0;
    size_t curlen=0;

    if(*s=='$')
    {
      const char * vbeg=NULL, * vend;
      int pcount=0;
      
      if(!strncmp(s,"$(",2))
      {
        vbeg=s+2;
        type='(';
      }
      else if(!strncmp(s,"$%(",3))
      {
        vbeg=s+3;
        type='%';
      }
      else if(!strncmp(s,"$&(",3))
      {
        vbeg=s+3;
        type='&';
      }
      else if(!strncmp(s,"$^(",3))
      {
        vbeg=s+3;
        type='^';
      }
      
      for (vend=s; vend[0]; vend++)
      {
        if(vend[0]=='(')
          pcount++;
        if(vend[0]==')' && (--pcount)==0)
          break;
      }
      
      if((type)&&(vend))
      {
        UDM_VAR * var;
        size_t len;
        char name[100]="";
        char * sem;
        
        len=(vend-vbeg);
        if(len>=sizeof(name))len=sizeof(name)-1;
        strncpy(name,vbeg,len);name[len]='\0';
        if((sem=strchr(name,':')))
        {
          *sem=0;
          maxlen=atoi(sem+1);
        }
        
        if ((A->doccount == 0) && !strcasecmp(name, "ndocs"))
        {
          UdmURLAction(A, NULL, UDM_URL_ACTION_DOCCOUNT);
          UdmVarListReplaceInt(vars, "ndocs", A->doccount);
        }

        if((var=UdmVarListFind(vars,name)))
        {
          switch(type)
          {
            case '&':
            case '^': 
              value= UdmStrdup(var->val);
              break;
            default:  value= var->val ? UdmRemoveHiLightDup(var->val) : NULL;
          }
          if(!value)
            value= empty;
        }
        else
        {
          value= empty;
        }
        s=vend;
      }else
      {
        type=0;
      }
    }
    if(!value)
      value=empty;
    curlen=strlen(value);
    
    if((curlen>maxlen)&&(maxlen>0))
    {
      char *p = value + maxlen, *S, *e;
      S = e = p;
      if (isdigit(*e)) {
        while(isdigit(*S) && S > (value + 1)) S--;
        S--;
        if (*S == '&' && S[1] == '#')
        {
          while(isdigit(*e)) e++;
          if (*e == ';') p = e + 1;
        } 
      }
      else if (isalpha(*e))
      {
        while(isalpha(*S) && S > value) S--;
        if (*S == '&')
        {
          while(isalpha(*e)) e++;
          if (*e == ';') p = e + 1;
        }
      }
      else if (*e == ';' && S < value )
      {
        S--;
        if (isdigit(*S))
        {
          while(isdigit(*S) && S > (value + 1)) S--;
          S--;
          if (*S == '&' && S[1] == '#')
          {
            p = e + 1;
          }
        }
        else if (isalpha(*S))
        {
          while(isalpha(*S) && S > value) S--;
          if (*S == '&')
          {
            p = e + 1;
          }
        }
      }
      if ((newvalue = (char*)UdmMalloc((size_t)(p - value) + 8)))
      {
        char *c2, *c3;
        strncpy(newvalue, value, (size_t)(p - value));
        newvalue[(p - value)] = '\0';
        c2 = strrchr(newvalue, '\2');
        c3 = strrchr(newvalue, '\3');
        if ((c2 != NULL) && ((c3 == NULL) || (c2 > c3)))
        {
          strcpy(newvalue + (p - value), "\3...");
        }
        else
        {
          strcpy(newvalue + (p - value), "...");
        }
        if (value != empty) UDM_FREE(value);
        value = newvalue;
      }
    }
    
    if (type)
    {
      dlen+= UdmTemplatePrintVar(A->Conf, stream, dst + dlen, dst_len - dlen, 
                                 value, type, HlBeg, HlEnd);
    }
    else
    {
      if((stream)&&(*s))fputc(*s,stream);
      if(dst)
      {
        dst[dlen++]= *s;
        dst[dlen]= '\0';
      }
    }
    
    if (value != empty) UDM_FREE(value);
  }
  return dlen;
}

static char * GetHtmlTok(const char * src,const char ** lt)
{
  char * res;
  size_t len;
  if((!src)&&!(src=*lt))return(NULL);
  if(*src=='<')
  {
    /* Find ">" and skip "<" */
    if((*lt=strchr(src,'>')))(*lt)++;
  }
  else
  {
    /* Find tag beginning */
    *lt=strchr(src,'<');
  }
  
  if(!(*lt))
  {
    /* Last token */
    res = (char*)UdmStrdup(src);
  }
  else
  {
    /* Token in the middle */
    len=(*lt)-src;
    res = (char*)UdmMalloc(len + 2);
    strncpy(res,src,len);
    res[len]='\0';
  }
  return(res);
}

/* 
  FIXME: add support to include into buffer
  FIXME: currently stream only is supported
*/

static void TemplateInclude(UDM_AGENT *Agent,FILE *stream,
                            UDM_VARLIST *vars,const char *tag_content,
                            const char *HlBeg,
                            const char *HlEnd)
{
  UDM_DOCUMENT * Inc=UdmDocInit(NULL);
  size_t max_doc_size= (size_t)UdmVarListFindInt(vars,"MaxDocSize",UDM_MAXDOCSIZE);

  if(!Inc->Buf.buf) Inc->Buf.buf = (char*)UdmMalloc(max_doc_size);
  Inc->Buf.maxsize=max_doc_size;

  if(tag_content)
  {
    const char *ce;
    size_t vurlen = 256 + 4 * strlen(tag_content);
    char  *vurl = (char*)UdmMalloc(vurlen);
          
    PrintTextTemplate(Agent, NULL, vurl, vurlen, vars, tag_content, HlBeg, HlEnd);
    UdmURLParse(&Inc->CurURL,vurl);
    UDM_FREE(vurl);
    UdmVarListReplaceStr(&Inc->RequestHeaders, "Host", UDM_NULL2EMPTY(Inc->CurURL.hostname));
    Inc->connp.hostname = (char*)UdmStrdup(UDM_NULL2EMPTY(Inc->CurURL.hostname));
    Inc->connp.port = Inc->CurURL.port ? Inc->CurURL.port : Inc->CurURL.default_port;
    
    if(UdmHostLookup(&Agent->Conf->Hosts, &Inc->connp))
    {
    }
          
    if(UdmGetURL(Agent,Inc)==UDM_OK)
    {
      UdmParseHTTPResponse(Agent,Inc);
      if(Inc->Buf.content)
      {
        ce=UdmVarListFindStr(&Inc->Sections,"Content-Encoding","");
#ifdef HAVE_ZLIB
        if(!strcasecmp(ce,"gzip") || !strcasecmp(ce,"x-gzip")){
          UdmUnGzip(Inc);
        }
        else if(!strcasecmp(ce,"deflate"))
        {
          UdmInflate(Inc);
        }
        else if(!strcasecmp(ce,"compress") || !strcasecmp(ce,"x-compress"))
        {
          UdmUncompress(Inc);
        }
#endif
        if(stream)
        {
          fprintf(stream,"%s",Inc->Buf.content);
        }
        else
        {
          /* FIXME: add printing to string */
        }
      }
    }
  }
  UdmDocFree(Inc);
}

static size_t TemplateTag(UDM_AGENT *Agent,FILE *stream,
                          char *dst,size_t dst_len,
                          UDM_VARLIST *vars,const char *tok,
                          const char *HlBeg, 
                          const char *HlEnd)
{
  char * opt;
  UDM_HTMLTOK ltag, *tag = &ltag;
  const char *last;
  UDM_VAR * var=NULL;
  char * vname = NULL, *value = NULL;
  size_t i, res = 0;
  
  opt = (char*)UdmMalloc(strlen(tok) + 200);
  UdmHTMLTOKInit(tag);
  UdmHTMLToken(tok, &last, tag);
  sprintf(opt, "<");

  for (i = 0; i < ltag.ntoks; i++)
  {
    const char *space= (i == 0) ? "" : " ";
    if (ISTAG(i, "selected") && ltag.toks[i].vlen)
    {
      vname = UdmStrndup(ltag.toks[i].val, ltag.toks[i].vlen);
    }
    else if (ISTAG(i, "value"))
    {
      value = UdmStrndup(ltag.toks[i].val, ltag.toks[i].vlen);
      sprintf(UDM_STREND(opt), "%svalue=\"%s\"", space, value);
    }
    else
    {
      char *tname = UdmStrndup(ltag.toks[i].name, ltag.toks[i].nlen);
      if (ltag.toks[i].vlen)
      {
        char *tval = UdmStrndup(ltag.toks[i].val, ltag.toks[i].vlen);
        sprintf(UDM_STREND(opt), "%s%s=\"%s\"", space, tname, tval);
        UDM_FREE(tval);
      }
      else
      {
         sprintf(UDM_STREND(opt), "%s%s", space, tname);
      }
      UDM_FREE(tname);
    }
  }

  if(vname)
  {
    var = UdmVarListFindWithValue(vars, UdmTrim(vname, "$()"), value ? value:"");
  }

  sprintf(UDM_STREND(opt), "%s>", var ? " selected":"");

  if (vname) { UDM_FREE(vname); }
  if (value) { UDM_FREE(value); }

  res = PrintTextTemplate(Agent, stream, dst, dst_len, vars, opt, HlBeg, HlEnd);
  UDM_FREE(opt);
  return res;
}


static void HTMLTokToVarList(UDM_VARLIST *vars,UDM_HTMLTOK *tag)
{
  size_t  toks;
  
  for(toks=0;toks<tag->ntoks;toks++)
  {
    char *vr=tag->toks[toks].name ? UdmStrndup(tag->toks[toks].name,tag->toks[toks].nlen) : (char*)UdmStrdup("");
    char *vl=tag->toks[toks].val ? UdmStrndup(tag->toks[toks].val,tag->toks[toks].vlen) : (char*)UdmStrdup("");
    UdmVarListAddStr(vars,vr,vl);
    UDM_FREE(vr);
    UDM_FREE(vl);
  }
}


#define UDM_TMPL_IF		1
#define UDM_TMPL_IFLIKE		2
#define UDM_TMPL_ELSEIF		3
#define UDM_TMPL_ELSELIKE	4
#define UDM_TMPL_ELSE		5
#define UDM_TMPL_ENDIF		6
#define UDM_TMPL_COPY		7
#define UDM_TMPL_SET		8
#define UDM_TMPL_INCLUDE	9
#define UDM_TMPL_CMP		10
#define UDM_TMPL_WLD		11
#define UDM_TMPL_JE		12
#define UDM_TMPL_JNE		13
#define UDM_TMPL_JMP		14
#define UDM_TMPL_PRINT		15
#define UDM_TMPL_TAG		16
#define UDM_TMPL_IFNOT		17
#define UDM_TMPL_INC		18
#define UDM_TMPL_DEC		19
#define UDM_TMPL_WHILENOT	20
#define UDM_TMPL_WHILE		21
#define UDM_TMPL_ENDWHILE	22
#define UDM_TMPL_PUT		23
#define UDM_TMPL_STRBCUT	24
#define UDM_TMPL_STRPCASE	25
#define UDM_TMPL_STRLEFT	26
#define UDM_TMPL_IFCS		27
#define UDM_TMPL_CMPCS		28
#define UDM_TMPL_URLDECODE	29
#define UDM_TMPL_HTMLENCODE	30

typedef struct udm_tmpl_names_st
{
  const char *name;
  size_t len;
  int num;
} UDM_TMPL_NAME;


UDM_TMPL_NAME tnames[]=
{
  {"IFLIKE",   6, UDM_TMPL_IFLIKE   },
  {"IFNOT",    5, UDM_TMPL_IFNOT    },
  {"IFCS",     4, UDM_TMPL_IFCS     },
  {"IF",       2, UDM_TMPL_IF       },
  {"ELSEIF",   6, UDM_TMPL_ELSEIF   },
  {"ELIF",     4, UDM_TMPL_ELSEIF   },
  {"ELIKE",    5, UDM_TMPL_ELSELIKE },
  {"ELSELIKE", 8, UDM_TMPL_ELSELIKE },
  {"ELSE",     4, UDM_TMPL_ELSE     },
  {"ENDIF",    5, UDM_TMPL_ENDIF    },
  {"/IF",      3, UDM_TMPL_ENDIF    },
  {"COPY",     4, UDM_TMPL_COPY     },
  {"SET",      3, UDM_TMPL_SET      },
  {"PUT",      3, UDM_TMPL_PUT      },
  {"INCLUDE",  7, UDM_TMPL_INCLUDE  },
  {"INC",      3, UDM_TMPL_INC      },
  {"DEC",      3, UDM_TMPL_DEC      },
  {"WHILENOT", 8, UDM_TMPL_WHILENOT },
  {"WHILE",    5, UDM_TMPL_WHILE    },
  {"ENDWHILE", 8, UDM_TMPL_ENDWHILE },
  {"STRBCUT",  7, UDM_TMPL_STRBCUT  },
  {"STRPCASE", 8, UDM_TMPL_STRPCASE }, 
  {"STRLEFT",  7, UDM_TMPL_STRLEFT  },
  {"URLDECODE",9, UDM_TMPL_URLDECODE},
  {"HTMLENCODE",10, UDM_TMPL_HTMLENCODE},
  { NULL,      0, 0                 }
};


static int HtmlTemplateName(const char *str)
{
  UDM_TMPL_NAME *t;
  for (t= tnames; t->name; t++)
  {
    if (!strncasecmp(str, t->name, t->len))
      return t->num;
  }
  return 0;
}


typedef struct udm_tmpl_compile_stack_item_st
{
  size_t beg;
  size_t jmp;
} UDM_TMPL_COMPILE_STACK_ITEM;


typedef struct udm_tmpl_compile_stack_st
{
  size_t nitems;
  size_t mitems;
  UDM_TMPL_COMPILE_STACK_ITEM *Items;
} UDM_TMPL_COMPILE_STACK;


static void CompileStackInit(UDM_TMPL_COMPILE_STACK *st)
{
  bzero((void*) st, sizeof(*st));
}


static int CompileStackPush(UDM_TMPL_COMPILE_STACK *st, 
                            UDM_TMPL_COMPILE_STACK_ITEM *item)
{
  if (st->nitems >= st->mitems)
  {
    size_t sz;
    st->mitems+= 16;
    sz= st->mitems * sizeof(UDM_TMPL_COMPILE_STACK_ITEM);
    st->Items= (UDM_TMPL_COMPILE_STACK_ITEM *) UdmRealloc(st->Items, sz);
    if (!st->Items)
      return UDM_ERROR;
  }
  st->Items[st->nitems]= item[0];
  st->nitems++;
  return UDM_OK;
}


static int CompileStackPop(UDM_TMPL_COMPILE_STACK *st)
{
  if (!st->nitems)
    return UDM_ERROR;
  st->nitems--;
  return UDM_OK;
}


static void CompileStackFree(UDM_TMPL_COMPILE_STACK *st)
{
  UDM_FREE(st->Items);
}


typedef struct udm_tmpl_cmd_st
{
  int cmd;
  char *arg1;
  char *arg2;
  size_t jump;
} UDM_TMPL_PRGITEM;


typedef struct udm_tmpl_prg_st
{
  size_t nitems;
  size_t mitems;
  size_t curr;
  int reg;
  UDM_TMPL_PRGITEM *Items;
  char errstr[128];
} UDM_TMPL_PRG;


static void HtmlTemplatePrgInit(UDM_TMPL_PRG *prg)
{
  bzero((void*)prg, sizeof(*prg));
}


static int HtmlTemplatePrgAdd(UDM_TMPL_PRG *prg, UDM_TMPL_PRGITEM *item)
{
  if (prg->nitems >= prg->mitems)
  {
    prg->mitems+= 64;
    prg->Items= (UDM_TMPL_PRGITEM*) UdmRealloc(prg->Items,
                                               prg->mitems*sizeof(*item));
    if (!prg->Items)
      return UDM_ERROR;
  }
  prg->Items[prg->nitems]= item[0];
  prg->nitems++;
  return UDM_OK;
}


static void HtmlTemplatePrgFree(UDM_TMPL_PRG *prg)
{
  size_t i;
  for (i=0 ; i < prg->nitems; i++)
  {
    UDM_FREE(prg->Items[i].arg1);
    UDM_FREE(prg->Items[i].arg2);
  }
  UDM_FREE(prg->Items);
}


static int HtmlTemplatePrgAdd2Arg(UDM_TMPL_PRG *prg, int cmdnum,
                                  const char *arg1, const char *arg2)
{
  UDM_TMPL_PRGITEM i;
  i.cmd= cmdnum;
  i.arg1= arg1 ? UdmStrdup(arg1) : NULL;
  i.arg2= arg2 ? UdmStrdup(arg2) : NULL; 
  i.jump= 0;
  return HtmlTemplatePrgAdd(prg, &i);
}


static int HtmlTemplatePrgAdd1Arg(UDM_TMPL_PRG *prg, 
                                  int cmdnum, const char *arg1)
{
  UDM_TMPL_PRGITEM i;
  i.cmd= cmdnum;
  i.arg1= arg1 ? UdmStrdup(arg1) : NULL;
  i.arg2= NULL;
  i.jump= 0;
  return HtmlTemplatePrgAdd(prg, &i);
}


static int HtmlTemplatePrgAddJmp(UDM_TMPL_PRG *prg, int cmdnum, int addr)
{
  UDM_TMPL_PRGITEM i;
  i.cmd= cmdnum;
  i.arg1= NULL;
  i.arg2= NULL;
  i.jump= addr;
  return HtmlTemplatePrgAdd(prg, &i);
}

static void HtmlTemplateErrorUnexpectedSym(UDM_TMPL_PRG *prg, const char *sym)
{
  udm_snprintf(prg->errstr, sizeof(prg->errstr), "Unexpected '%s'", sym);
}


static int HtmlTemplateCompileCommand(UDM_TMPL_PRG *prg,
                                      UDM_TMPL_COMPILE_STACK *st,
                                      const char *template, int cmdnum)
{
  UDM_HTMLTOK  tag;
  UDM_VARLIST  attr;
  UDM_TMPL_COMPILE_STACK_ITEM sitem;
  const char  *var,*val,*hlast=NULL;
  int rc= UDM_OK;
  
  UdmHTMLTOKInit(&tag);
  UdmHTMLToken(template,&hlast,&tag);
  UdmVarListInit(&attr);
  HTMLTokToVarList(&attr,&tag);
  
  var= UdmVarListFindStr(&attr,"Name","");
  val= UdmVarListFindStr(&attr,"Content","");
  
  switch (cmdnum)
  {
    case UDM_TMPL_WHILE:
    case UDM_TMPL_IF:
    case UDM_TMPL_IFCS:
      sitem.beg= prg->nitems;
      sitem.jmp= prg->nitems + 1;
      CompileStackPush(st, &sitem);
      HtmlTemplatePrgAdd2Arg(prg, cmdnum == UDM_TMPL_IFCS ? 
                                  UDM_TMPL_CMPCS : UDM_TMPL_CMP ,  var, val);
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JNE, 0);
      break;
    
    case UDM_TMPL_WHILENOT:
    case UDM_TMPL_IFNOT:
      sitem.beg= prg->nitems;
      sitem.jmp= prg->nitems + 1;
      CompileStackPush(st, &sitem);
      HtmlTemplatePrgAdd2Arg(prg, UDM_TMPL_CMP, var, val);
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JE, 0);
      break;
    
    case UDM_TMPL_IFLIKE:
      sitem.beg= prg->nitems;
      sitem.jmp= prg->nitems + 1;
      CompileStackPush(st, &sitem);
      HtmlTemplatePrgAdd2Arg(prg, UDM_TMPL_WLD, var, val);
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JNE, 0);
      break;
    
    case UDM_TMPL_ELSEIF:
      if (!st->nitems)
      {
        HtmlTemplateErrorUnexpectedSym(prg, "<!ELSEIF>");
        rc= UDM_ERROR; 
        goto ret;
      }
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JMP, 0);
      prg->Items[st->Items[st->nitems-1].jmp].jump= prg->nitems;
      st->Items[st->nitems-1].jmp= prg->nitems + 1;
      HtmlTemplatePrgAdd2Arg(prg, UDM_TMPL_CMP, var, val);
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JNE, 0);
      break;
    
    case UDM_TMPL_ELSELIKE:
      if (!st->nitems)
      {
        HtmlTemplateErrorUnexpectedSym(prg, "<!ELSELIKE>");
        rc= UDM_ERROR; 
        goto ret;
      }
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JMP, 0);
      prg->Items[st->Items[st->nitems-1].jmp].jump= prg->nitems;
      st->Items[st->nitems-1].jmp= prg->nitems + 1;
      HtmlTemplatePrgAdd2Arg(prg, UDM_TMPL_WLD, var, val);
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JNE, 0);
      break;
    
    case UDM_TMPL_ELSE:
      if (!st->nitems)
      {
        HtmlTemplateErrorUnexpectedSym(prg, "<!ELSE>");
        rc= UDM_ERROR; 
        goto ret;
      }
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JMP, 0);
      prg->Items[st->Items[st->nitems-1].jmp].jump= prg->nitems;
      break;
    
    case UDM_TMPL_ENDWHILE:
      if (!st->nitems)
      {
        HtmlTemplateErrorUnexpectedSym(prg, "<!ENDWHILE>");
        rc= UDM_ERROR; 
        goto ret;
      }
      HtmlTemplatePrgAddJmp(prg, UDM_TMPL_JMP, st->Items[st->nitems-1].beg);
      prg->Items[st->Items[st->nitems-1].jmp].jump= prg->nitems;
      CompileStackPop(st);
      break;
    
    case UDM_TMPL_ENDIF:
      if (!st->nitems)
      {
        HtmlTemplateErrorUnexpectedSym(prg, "<!ENDIF>");
        rc= UDM_ERROR; 
        goto ret;
      }
      /*
        Now we know the address if <!ENDIF>
        We should pass through <!IF> and all <!ELSEIF>
        and set correct JMP addresses.
      */
      {
        size_t i;
        for (i= st->Items[st->nitems-1].beg; i < prg->nitems; i++)
        {
          if (prg->Items[i].jump == 0 &&
              (prg->Items[i].cmd == UDM_TMPL_JMP ||
               prg->Items[i].cmd == UDM_TMPL_JE  ||
               prg->Items[i].cmd == UDM_TMPL_JNE))
            prg->Items[i].jump= prg->nitems; 
        }
      }
      CompileStackPop(st);
      break;
    
    case UDM_TMPL_COPY:
    case UDM_TMPL_SET:
    case UDM_TMPL_PUT:
    case UDM_TMPL_STRBCUT:
    case UDM_TMPL_STRPCASE:
    case UDM_TMPL_STRLEFT:
    case UDM_TMPL_URLDECODE:
    case UDM_TMPL_HTMLENCODE:
      HtmlTemplatePrgAdd2Arg(prg, cmdnum, var, val);
      break;
    
    case UDM_TMPL_INCLUDE:
      HtmlTemplatePrgAdd1Arg(prg, cmdnum, val);
      break;
      
    case UDM_TMPL_INC:
    case UDM_TMPL_DEC:
      HtmlTemplatePrgAdd1Arg(prg, cmdnum, var);
      break;
      
    default:
      HtmlTemplatePrgAdd1Arg(prg, UDM_TMPL_PRINT, val);
      break;
  }
  
ret:
  UdmVarListFree(&attr);
  return rc;
}


static int HtmlTemplateCompile(UDM_TMPL_PRG *prg, const char *template)
{
  const char *lt;
  char *tok;
  UDM_TMPL_COMPILE_STACK st;
  int rc= UDM_OK;
  
  CompileStackInit(&st);
  
  tok= GetHtmlTok(template, &lt);
  while (tok)
  {
    int cmdnum;
    
    if (tok[0]=='<' && tok[1] == '!' && (cmdnum= HtmlTemplateName(tok+2)))
    {
      rc= HtmlTemplateCompileCommand(prg, &st, tok, cmdnum);
    }
    else if(!(strncasecmp(tok,"<OPTION",7)))
    {
      rc= HtmlTemplatePrgAdd1Arg(prg, UDM_TMPL_TAG, tok);
    }
    else if(!(strncasecmp(tok,"<INPUT",6)))
    {
      rc= HtmlTemplatePrgAdd1Arg(prg, UDM_TMPL_TAG, tok);
    }
    else
    {
      rc= HtmlTemplatePrgAdd1Arg(prg, UDM_TMPL_PRINT, tok);
    }
    UDM_FREE(tok);
    if (rc != UDM_OK)
      break;
    tok=GetHtmlTok(NULL,&lt);
  }
  CompileStackFree(&st);
  return rc;
}


static void HtmlTemplateProgItemPrint(UDM_TMPL_PRGITEM *it)
{
  switch (it->cmd)
  {
    case UDM_TMPL_CMP:
      printf("CMP '%s' '%s'\n", it->arg1, it->arg2);
      break;
    case UDM_TMPL_CMPCS:
      printf("CMPCS '%s' '%s'\n", it->arg1, it->arg2);
      break;
    case UDM_TMPL_WLD:
      printf("WLD '%s' '%s'\n", it->arg1, it->arg2);
      break;
    case UDM_TMPL_SET:
      printf("SET '%s' '%s'\n", it->arg1, it->arg2);
      break;
    case UDM_TMPL_PUT:
      printf("PUT '%s' '%s'\n", it->arg1, it->arg2);
      break;
    case UDM_TMPL_JNE:
      printf("JNE %04X\n", it->jump);
      break;
    case UDM_TMPL_JE:
      printf("JE  %04X\n", it->jump);
      break;
    case UDM_TMPL_JMP:
      printf("JMP %04X\n", it->jump);
      break;
    case UDM_TMPL_PRINT:
      printf("PRN '%s'\n", it->arg1);
      break;
    case UDM_TMPL_TAG:
      printf("TAG '%s'\n", it->arg1);
      break;
    case UDM_TMPL_INCLUDE:
      printf("EXT '%s'\n", it->arg1);
      break;
    case UDM_TMPL_INC:
      printf("INC '%s'\n", it->arg1);
      break;
    case UDM_TMPL_DEC:
      printf("DEC '%s'\n", it->arg1);
      break;
    case UDM_TMPL_STRBCUT:
      printf("STRBCUT '%s'\n", it->arg1);
      break;
    case UDM_TMPL_URLDECODE:
      printf("URLDECODE '%s'\n", it->arg1);
      break;
    case UDM_TMPL_HTMLENCODE:
      printf("HTMLENCODE '%s'\n", it->arg1);
      break;
    case UDM_TMPL_STRLEFT:
      printf("STRLEFT '%s'\n", it->arg1);
      break;
    case UDM_TMPL_STRPCASE:
      printf("STRPCASE '%s'\n", it->arg1);
      break;
    default:
      printf("Unknown command %d '%s' '%s' %d\n", it->cmd, it->arg1, it->arg2, it->jump);
  }
}


#if 1
static int HtmlTemplateProgPrint(UDM_TMPL_PRG *p)
{
  size_t i;
  
  for (i=0 ; i < p->nitems ; i++)
  {
    UDM_TMPL_PRGITEM *it= &p->Items[i];
    printf("%04X ", i);
    HtmlTemplateProgItemPrint(it);
  }
  return UDM_OK;
}
#endif



static void PrintHtmlTemplate(UDM_AGENT * Agent,
                              FILE * stream, char * dst, size_t dst_len,
                              UDM_VARLIST * vars, const char * template)
{
  size_t    dlen=0;
  const char *HlBeg=UdmVarListFindStr(vars,"HlBeg","");
  const char *HlEnd=UdmVarListFindStr(vars,"HlEnd","");
  char *hlbeg= UdmRemoveHiLightDup(HlBeg);
  char *hlend= UdmRemoveHiLightDup(HlEnd);
  UDM_TMPL_PRG prg;
  UDM_TMPL_PRG *p= &prg;
  
  HtmlTemplatePrgInit(&prg);
  if (UDM_OK != HtmlTemplateCompile(&prg, template))
  {
    if (stream)
      fprintf(stream, "Template error: %s\n", prg.errstr);
    goto ret;
  }
  
  if (0)
    HtmlTemplateProgPrint(&prg);
  
  for (p->curr= 0; p->curr < p->nitems; )
  {
    UDM_TMPL_PRGITEM *it= &p->Items[p->curr];
    
    switch (it->cmd)
    {
      case UDM_TMPL_JNE:
        if (p->reg)
          p->curr= it->jump;
        else
          p->curr++;
        break;
      
      case UDM_TMPL_JE:
        if (!p->reg)
          p->curr= it->jump;
        else
          p->curr++;
        break;
      
      case UDM_TMPL_JMP:
        p->curr= it->jump;
        break;
      
      case UDM_TMPL_CMP:
      case UDM_TMPL_CMPCS:
      case UDM_TMPL_WLD:
        {
          char *nhvar=UdmRemoveHiLightDup(UdmVarListFindStr(vars,it->arg1,""));
          size_t vurlen = 256 + 4 * strlen(it->arg2);
          char  *vurl = (char*)UdmMalloc(vurlen);
          vurl[0]= '\0';
          PrintTextTemplate(Agent, NULL, vurl, vurlen, vars, it->arg2, HlBeg, HlEnd);
          if (it->cmd == UDM_TMPL_CMP)
            p->reg= strcasecmp(nhvar, vurl);
          else if (it->cmd == UDM_TMPL_CMPCS)
            p->reg= strcmp(nhvar, vurl);
          else
            p->reg= UdmWildCaseCmp(nhvar, vurl);
          UdmFree(nhvar);
          UdmFree(vurl);
        }
        p->curr++;  
        break;
      
      case UDM_TMPL_PUT:
      case UDM_TMPL_SET:
        {
          size_t vurlen = 256 + 4 * strlen(it->arg2);
          char  *vurl = (char*)UdmMalloc(vurlen);
          vurl[0]= '\0';
          PrintTextTemplate(Agent, NULL, vurl, vurlen, vars, it->arg2, HlBeg, HlEnd);
          if (it->cmd == UDM_TMPL_SET)
            UdmVarListReplaceStr(vars,it->arg1, vurl);
          else
            UdmVarListAddStr(vars,it->arg1, vurl);
          UdmFree(vurl);
        }
        p->curr++;
        break;
      
      case UDM_TMPL_STRPCASE:
      case UDM_TMPL_STRBCUT:
      case UDM_TMPL_STRLEFT:
      case UDM_TMPL_URLDECODE:
      case UDM_TMPL_HTMLENCODE:
        {
          size_t vurlen = 256 + 4 * strlen(it->arg2);
          char  *vurl = (char*)UdmMalloc(vurlen);
          vurl[0]= '\0';
          PrintTextTemplate(Agent, NULL, vurl, vurlen, vars, it->arg2, HlBeg, HlEnd);
          
          if (it->cmd == UDM_TMPL_STRPCASE)
          {
            char *v;
            for (v= vurl; v[0]; v++)
            {
              /* This will probably need full Unicode support in the future */
              v[0]= v == vurl ? toupper(v[0]) : tolower(v[0]);
            }
          }
          else if (it->cmd == UDM_TMPL_STRBCUT)
          {
            if ((vurlen= strlen(vurl)) > 1)
              memmove(vurl, vurl+1, vurlen); /* including trailing null */
            else
              vurl[0]= '\0';
          }
          else if (it->cmd == UDM_TMPL_STRLEFT)
          {
            vurl[1]= '\0';
          }
          else if (it->cmd == UDM_TMPL_URLDECODE)
          {
            UdmUnescapeCGIQuery(vurl, vurl);
          }
          else if (it->cmd == UDM_TMPL_HTMLENCODE)
          {
            size_t len= strlen(vurl);
            size_t enclen= len * 6 + 1;
            char *vurl_encoded= (char*) UdmMalloc(enclen); 
            UdmHTMLEncode(vurl_encoded, enclen, vurl, len + 1);
            UdmFree(vurl);
            vurl= vurl_encoded;
          }
          
          UdmVarListReplaceStr(vars,it->arg1, vurl);
          UdmFree(vurl);
        }
        p->curr++;
        break;
      
      case UDM_TMPL_COPY:
        {
          const char *val;
          size_t vurlen = 256 + 4 * strlen(it->arg2);
          char  *vurl = (char*)UdmMalloc(vurlen);
          vurl[0]= '\0';
          PrintTextTemplate(Agent, NULL, vurl, vurlen, vars, it->arg2, HlBeg, HlEnd);
          val= UdmVarListFindStr(vars, vurl, "");
          UdmVarListReplaceStr(vars,it->arg1,val);
          UdmFree(vurl);
        }
        p->curr++;
        break;
      
      case UDM_TMPL_PRINT:
        dlen += PrintTextTemplate(Agent, stream, dst + dlen, dst_len - dlen, 
                                  vars, it->arg1, hlbeg,hlend);
        p->curr++;
        break;
      
      case UDM_TMPL_TAG:
        dlen += TemplateTag(Agent, stream, dst + dlen, dst_len - dlen,
                            vars, it->arg1, hlbeg, hlend);
        p->curr++;
        break;

      case UDM_TMPL_INCLUDE:
        if(Agent)TemplateInclude(Agent, stream, vars, it->arg1, hlbeg, hlend);
        p->curr++;
        break;
      
      case UDM_TMPL_INC:
        {
          int val= UdmVarListFindInt(vars, it->arg1, 0) + 1;
          UdmVarListReplaceInt(vars, it->arg1, val);
        }
        p->curr++;
        break;
      
      case UDM_TMPL_DEC:
        {
          int val= UdmVarListFindInt(vars, it->arg1, 0) - 1;
          UdmVarListReplaceInt(vars, it->arg1, val);
        }
        p->curr++;
        break;
      
      default:
        printf("Unknown command: \n");
        HtmlTemplateProgItemPrint(it);
        exit(1);
    }
  }

ret:
  HtmlTemplatePrgFree(&prg);
  UDM_FREE(hlbeg);
  UDM_FREE(hlend);
}


void __UDMCALL UdmTemplatePrint(UDM_AGENT * Agent,
                                FILE *stream, char *dst, size_t dst_len,
                                UDM_VARLIST *vars, UDM_VARLIST *tm,
                                const char *w)
{
  size_t  t;
  size_t  matches=0;
  size_t  format=(size_t)UdmVarListFindInt(vars,"o",0);
  UDM_VAR  *First=NULL;
  
  if(dst)*dst='\0';
  for(t=0;t<tm->nvars;t++)
  {
    if(!strcasecmp(w,tm->Var[t].name))
    {
      if(!First)First=&tm->Var[t];
      if(matches==format)
      {
        PrintHtmlTemplate(Agent, stream, dst, dst_len, vars, tm->Var[t].val);
        return;
      }
      matches++;
    }
  }
  if (First) PrintHtmlTemplate(Agent, stream, dst, dst_len, vars, First->val);
  return;
}

#if 0
static int ParseVariable(UDM_ENV *Env,UDM_VARLIST *vars,char *str)
{
  char *tok,*lt;
  
  if((tok = udm_strtok_r(str, " \t\r\n", &lt)))
  {
    char * arg=NULL;
          
/* + */    if(!strcasecmp(str,"Affix"))
    {
      char aname[1024];
      char *args[5], *p;
      size_t narg=0;
          
      while((tok)&&(narg<5))
      {
        args[narg++]=tok;
        tok = udm_strtok_r(NULL, " \t", &lt);
      }
      if(narg!=4 || !(p= UdmParseEnvVar(Env,args[3])))
      {
        sprintf(Env->errstr,"Bad Affix command");
        return UDM_ERROR;
      }
      if(p[0]=='/')strncpy(aname,p,sizeof(aname)-1);
      else  udm_snprintf(aname,sizeof(aname)-1,"%s/%s",UDM_CONF_DIR,p);
      UdmFree(p);
      if(UdmAffixListListAdd(&Env->Affixes,args[1],args[2],aname))
      {
        udm_snprintf(Env->errstr,sizeof(Env->errstr)-1,"Can't add affix :%s",aname);
        return UDM_ERROR;
      }
    }
/* + */    else if(!strcasecmp(str,"Spell"))
    {
      char aname[1024];
      char *args[5], *p;
      size_t narg=0;

      while((tok)&&(narg<5))
      {
        args[narg++]=tok;
        tok = udm_strtok_r(NULL, " \t", &lt);
      }
      if(narg != 4 || !(p= UdmParseEnvVar(Env,args[3])))
      {
        sprintf(Env->errstr,"Bad Spell command");
        return UDM_ERROR;
      }
      if(p[0]=='/')strncpy(aname,p,sizeof(aname)-1);
      else  udm_snprintf(aname,sizeof(aname)-1,"%s/%s",UDM_CONF_DIR,p);
      UdmFree(p);
      if(UdmSpellListListAdd(&Env->Spells,args[1],args[2],aname))
      {
        udm_snprintf(Env->errstr,sizeof(Env->errstr)-1,"Can't load dictionary :%s",aname);
        return UDM_ERROR;
      }
    }
/* + */    else if(!strcasecmp(str, "IspellUsePrefixes"))
    {
      char sel[8];
      int val = 1;
      sscanf(str+17, "%4s", sel);
      if (!strncasecmp(sel, "no", 2))
      {
        val = 0;
      }
      UdmVarListReplaceInt(&Env->Vars, "IspellUsePrefixes", val);
    }
/* + */    else if(!strcasecmp(str,"StopwordFile"))
    {
      char aname[1024];      
#ifdef WIN32  /* Space is valid symbol for WIN32 versions */
      arg = udm_strtok_r(NULL, "\t\r\n", &lt);
#else
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
#endif
      if(arg)
      {
        if(arg[0]=='/')
          strncpy(aname,arg,sizeof(aname)-1);
        else
#ifdef WIN32
          udm_snprintf(aname,sizeof(aname)-1,"%s", arg);
#else
          udm_snprintf(aname,sizeof(aname)-1,"%s/%s",UDM_CONF_DIR,arg);
#endif
        if(UdmStopListLoad(Env,aname))
        {
          return UDM_ERROR;
        }
      }
      else
      {
        sprintf(Env->errstr,"Bad StopwordFile command");
        return UDM_ERROR;
      }
    }
/* + */    else if(!strcasecmp(str,"Synonym"))
    {
      char aname[1024];
      char *p;
#ifdef WIN32  /* Space is valid symbol for WIN32 versions */
      arg = udm_strtok_r(NULL, "\t\r\n", &lt);
#else
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
#endif
      if(arg && (p= UdmParseEnvVar(Env, arg)))
      {
        if(p[0]=='/')
          strncpy(aname,p,sizeof(aname)-1);
        else
#ifdef WIN32
          udm_snprintf(aname,sizeof(aname)-1,"%s",p);
#else
          udm_snprintf(aname,sizeof(aname)-1,"%s/%s",UDM_CONF_DIR,p);
#endif
        if(UdmSynonymListLoad(Env,aname)){
          return UDM_ERROR;
        }
        UdmFree(p);
      }
      else
      {
        sprintf(Env->errstr,"Bad Synonym command");
        return UDM_ERROR;
      }
    }
/* + */    else if(!strcasecmp(str,"ImportEnv"))
    {
      const char *val;
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if ((val=getenv(arg)))
        UdmVarListReplaceStr(vars,arg,val);
    }
/* + */    else if(!strcasecmp(str,"DBAddr"))
    {
      char *p;
      if((arg = udm_strtok_r(NULL, " \t\r\n", &lt)) &&
          (p = UdmParseEnvVar(Env, arg)))
      {
        UdmDBListAdd(&Env->dbl, p, UDM_OPEN_MODE_READ);
/*      UdmVarListReplaceStr(vars,tok,arg);*/
        UdmFree(p);
      }
    }
/* - */    else if((str[0]=='R'||str[0]=='r')&&(str[1]>='0')&&(str[1]<='9'))
    {
      float r;
      int ir;
      arg = udm_strtok_r(NULL, " =\t\r\n", &lt); 
      if(arg)
      {
        r = (float)UDM_ATOF(arg);
        srand((unsigned)time(0));
        r = r * rand() / RAND_MAX; ir = (int)r;
        UdmVarListReplaceInt(vars,str,ir);
      }
    }
/* + */    else if(!strcasecmp(str,"HlBeg"))
    {
      UdmVarListReplaceStr(vars,"HlBeg",lt);
    }
/* + */    else if(!strcasecmp(str,"HlEnd"))
    {
      UdmVarListReplaceStr(vars,"HlEnd",lt);
    }
/* + */    else if(!strcasecmp(str,"DateFormat"))
    {
      UdmVarListReplaceStr(vars, "DateFormat", lt);
    }
/* + */    else if(!strcasecmp(str,"Limit"))
    {
      char * sc, * nm;
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if((sc=strchr(arg,':'))){
        *sc='\0';sc++;
        nm=(char*)UdmMalloc(strlen(arg)+8);
        sprintf(nm,"Limit-%s",arg);
        UdmVarListReplaceStr(vars,nm,sc);
        UDM_FREE(nm);
      }
    }
/* + */    else if(!strcasecmp(str,"CrossWords"))
    {
      UdmVarListReplaceStr(vars,"CrossWords",lt);
    }
/* ? */    else if(!strcasecmp(str,"Alias"))
    {
      char * arg1;
      
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      arg1 = udm_strtok_r(NULL, " \t\r\n", &lt);
      if(arg1){
        UDM_MATCH Alias;
        Alias.pattern=arg;
        Alias.arg=arg1;
        Alias.match_type=UDM_MATCH_BEGIN;
        Alias.case_sense=0;
        UdmMatchListAdd(NULL, &Env->Aliases, &Alias, Env->errstr, sizeof(Env->errstr), 0);
      }
    }
/* + */    else if(!strcasecmp(str,"MaxWordLength"))
    {
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if(arg)Env->WordParam.max_word_len=atoi(arg);
    }
/* + */    else if(!strcasecmp(str,"MinWordLength"))
    {
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if(arg)Env->WordParam.min_word_len=atoi(arg);
    }
/* + */    else if(!strcasecmp(str,"ExcerptSize"))
    {
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if (arg) UdmVarListReplaceInt(vars, tok, atoi(arg));
    }
/* + */    else if(!strcasecmp(str,"LocalCharset"))
    {
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if (arg)
      {
        UdmVarListReplaceStr(vars, tok, arg);
        Env->lcs = UdmGetCharSet(arg);
      }
    }
/* + */    else if(!strcasecmp(str,"BrowserCharset"))
    {
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      if (arg)
      {
        UdmVarListReplaceStr(vars, tok, arg);
        Env->bcs = UdmGetCharSet(arg);
      }
    }
/* ? */    else if(!strcasecmp(str,"Mime"))
    {
      char *mime_arg[4];
      int argn = 0;

      tok = udm_strtok_r(NULL, " \t\r\n", &lt);
      while((tok)&&(argn<3))
      {
        mime_arg[argn++]=tok;
        tok = udm_strtok_r(NULL, " \t", &lt);
      }
      if(argn != 3)
      {
        sprintf(Env->errstr, "Error: too %s arguments for Mime command\n", (argn < 3 ? "few" : "many"));
        return UDM_ERROR;
      }
      else
      {
        UDM_PARSER P;
        P.from_mime = mime_arg[0];
        P.to_mime = mime_arg[1];
        P.cmd = mime_arg[2];
        UdmParserAdd(&Env->Parsers, &P);
      }
    }
    else
    {
      arg = udm_strtok_r(NULL, " \t\r\n", &lt);
      UdmVarListReplaceStr(vars,tok,arg);
    }
  }
  return UDM_OK;
}
#endif

static int ParseHl (UDM_VARLIST *v, const char *str)
{
  const char *p;
  const char *e;
  char *s;

  for (p = str; *p; p++) if (! isspace(*p)) break;

  if (! strncasecmp(p, "HlBeg", 5))
  {
    p += 5;
    while (*p && isspace(*p)) p++;
    if ((*p == '\'' || *p == '"') && (e = strrchr(p + 1, *p))) s = UdmStrndup(p + 1, e - p - 1);
    else s = UdmStrdup(p);
    UdmVarListReplaceStr(v, "HlBeg", s);
    UdmFree(s);
  }
  else if (! strncasecmp(p, "HlEnd", 5))
  {
    p += 5;
    while (*p && isspace(*p)) p++;
    if ((*p == '\'' || *p == '"') && (e = strrchr(p + 1, *p))) s = UdmStrndup(p + 1, e - p - 1);
    else s = UdmStrdup(p);
    UdmVarListReplaceStr(v, "HlEnd", s);
    UdmFree(s);
  }
  else
  {
    return UDM_ERROR;
  }

  return UDM_OK;
}

/* Load template  */
int UdmTemplateLoad(UDM_AGENT *Agent,
                    const char * tname,
                    UDM_VARLIST *tmpl)
{
  UDM_ENV *Env = Agent->Conf;
  UDM_CFG Cfg;
  UDM_SERVER Srv;
  FILE    *file;
  char    str[1024];
  char    ostr[1024];
  const char  *dbaddr=NULL;
  int    variables=0;
  char    cursection[128]="";
  char    *cur=NULL;
  char    nameletter[]=
        "abcdefghijklmnopqrstuvwxyz"
        "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
        "0123456789._";
  
  UdmServerInit(&Srv);
  bzero((void*)&Cfg, sizeof(Cfg));
  Cfg.Indexer = Agent;
  Cfg.level = 0;
  Cfg.flags = UDM_FLAG_LOAD_LANGMAP + UDM_FLAG_SPELL;
  Cfg.Srv = &Srv;

  if(!(file=fopen(tname,"r")))
  {
    udm_snprintf(Env->errstr,sizeof(Env->errstr)-1,"Unable to open template '%s': %s",tname,strerror(errno));
    return(1);
  }
  
  while(fgets(str,sizeof(str)-1,file))
  {
    char  *s;
    
    str[sizeof(str)-1]='\0';
    strcpy(ostr,str);
    
    s=UdmTrim(str," \r\n");
    
    if(!strcasecmp(s,"<!--variables"))
    {
      variables=1;
      continue;
    }
    
    if(!strcmp(s,"-->") && variables)
    {
      variables=0;
      continue;
    }
    
    if(variables)
    {
      int r;
      if(!*s)continue;
      if(*s=='#')continue;

      if(UDM_OK!=(r=ParseHl(&Env->Vars, s)) && UDM_OK!=(r=UdmEnvAddLine(&Cfg,s)))
        return r;
/*      if(UDM_OK!=(r=ParseVariable(Env,vars,s)))
        return r; */
      continue;
    }
    
    if(!memcmp(s,"<!--",4))
    {
      char *e;
      
      for(e=s+4;(*e)&&(strchr(nameletter,*e)||(*e=='/'));e++);
      
      if(!strcmp(e,"-->"))
      {
        *e='\0';
        s+=4;
        
        if(s[0]=='/')
        {
          if(!strcasecmp(s+1,cursection) && cursection[0])
          {
            UDM_VAR *I;
            tmpl->Var=(UDM_VAR*)UdmRealloc(tmpl->Var,(tmpl->nvars+1)*sizeof(UDM_VAR));
            I=&tmpl->Var[tmpl->nvars];
            I->name = (char*)UdmStrdup(cursection);
            I->val = (char*)UdmStrdup(cur?cur:"");
            tmpl->nvars++;
            cursection[0]='\0';
            UDM_FREE(cur);
            continue;
          }
        }
        else if(s[1])
        {
          strncpy(cursection,s,sizeof(cursection));
          cursection[sizeof(cursection)-1]='\0';
          continue;
        }
      }
    }
    
    if(!cursection[0])
      continue;
    
    if(!cur)
    {
      cur = (char*)UdmStrdup(ostr);
    }
    else
    {
      cur=(char*)UdmRealloc(cur,strlen(cur)+strlen(ostr)+1);
      strcat(cur,ostr);
    }
  }
  fclose(file);
  UdmVarListReplaceLst(&Env->Vars, &Srv.Vars, NULL, "*");
  UdmServerFree(&Srv);
  UDM_FREE(cur);
  
  if(Env->Spells.nitems && Env->Affixes.nitems)
  {
    char *err= Env->errstr;
    int flags= strcasecmp(UdmVarListFindStr(&Env->Vars,
                                            "IspellUsePrefixes","no"),"no");
    flags= flags ? 0 : UDM_SPELL_NOPREFIX;
    if (UdmSpellListListLoad(&Env->Spells, err, 128) ||
        UdmAffixListListLoad(&Env->Affixes, flags, err, 128))
    return UDM_ERROR;
  }
  UdmSynonymListSort(&Env->Synonyms);
  
  
#ifdef HAVE_SQL
  if(Env->dbl.nitems == 0)
  {
    if (UDM_OK!=UdmDBListAdd(&Env->dbl, dbaddr = "mysql://localhost/mnogosearch", UDM_OPEN_MODE_READ))
    {
      sprintf(Env->errstr,"Invalid DBAddr: '%s'", (!dbaddr)?"NULL":dbaddr);
      return UDM_ERROR;
    }
  }
#endif
  if(Env->dbl.nitems == 0)
  {
    if (UDM_OK!=UdmDBListAdd(&Env->dbl, dbaddr =  "searchd://localhost/", UDM_OPEN_MODE_READ))
    {
      sprintf(Env->errstr,"Invalid DBAddr: '%s'", (!dbaddr)?"NULL":dbaddr);
      return UDM_ERROR;
    }
  }
  
  return UDM_OK;
}
