/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_wild.h"
#include "udm_vars.h"


static int varcmp(void *v1, void *v2) {
  register UDM_VAR *uv1 = v1, *uv2 = v2;
  if (uv1->name == NULL) {
    if (uv2->name == NULL) return 0;
    return 1;
  }
  if (uv2->name == NULL) return -1;
  return strcasecmp(uv1->name, uv2->name);
}

void UdmVarFree(UDM_VAR *S){
	UDM_FREE(S->name);
	UDM_FREE(S->val);
	return;
}

static int UdmVarCopy(UDM_VAR *D, UDM_VAR *S)
{
  D->section= S->section;
  D->maxlen= S->maxlen;
  D->curlen= S->curlen;
  D->name= (char*)UdmStrdup(S->name);
  D->flags= S->flags;
  if (S->maxlen == 0)
  {
    D->val= S->val ? (char*)UdmStrdup(S->val) : NULL;
  } 
  else
  {
    if (S->val == NULL)
    {
      D->val = NULL;
    }
    else
    {
      D->val = (char*)UdmMalloc(S->maxlen + 4);
      strncpy(D->val, S->val, S->curlen + 1);
    }
  }
  return UDM_OK;
}

static int UdmVarCopyNamed(UDM_VAR *D, UDM_VAR *S, const char *name){
  D->section= S->section;
  D->maxlen= S->maxlen;
  D->curlen= S->curlen;
  D->flags= S->flags;
  if(name)
  {
    size_t len = strlen(name) + strlen(S->name) + 3;
    D->name = (char*)UdmMalloc(len);
    udm_snprintf(D->name, len, "%s.%s", name, S->name);
  }
  else
  {
    D->name = (char*)UdmStrdup(S->name);
  }
  
  if (S->maxlen == 0)
  {
    D->val= S->val ? (char*)UdmStrdup(S->val) : NULL;
  }
  else
  {
    if (S->val == NULL) {
      D->val = NULL;
    }
    else
    {
      D->val = (char*)UdmMalloc(S->maxlen + 4);
      strncpy(D->val, S->val, S->curlen + 1);
    }
  }
  return UDM_OK;
}

int UdmVarListAdd(UDM_VARLIST * Lst,UDM_VAR * S){
	Lst->Var=(UDM_VAR*)UdmRealloc(Lst->Var,(Lst->nvars+1)*sizeof(*Lst->Var));
	if(S) UdmVarCopy(&Lst->Var[Lst->nvars],S);
	else bzero((void*)&Lst->Var[Lst->nvars], sizeof(UDM_VAR));
	Lst->nvars++;
	if(S) UdmSort(Lst->Var, Lst->nvars, sizeof(UDM_VAR), (qsort_cmp)varcmp);
	return UDM_OK;
}

static int UdmVarListAddNamed(UDM_VARLIST *Lst,UDM_VAR *S, const char *name){
	UDM_VAR	*v;
	UdmVarListAdd(Lst,NULL);
	v=&Lst->Var[Lst->nvars-1];
	UdmVarCopyNamed(v,S,name);
	UdmSort(Lst->Var, Lst->nvars, sizeof(UDM_VAR), (qsort_cmp)varcmp);
	return UDM_OK;
}

UDM_VAR * UdmVarListFind(UDM_VARLIST * vars,const char * name){
	UDM_VAR key;
	
	if (!vars->nvars) return NULL;
	key.name= (char*)name;
	return (UDM_VAR*)bsearch(&key, vars->Var, vars->nvars, sizeof(UDM_VAR), (qsort_cmp)varcmp);
}

int UdmVarListDel(UDM_VARLIST *vars,const char *name){
	UDM_VAR	*v = UdmVarListFind(vars, name);
	size_t nvars = vars->nvars - (v - vars->Var) - 1;

	if (v) {
	  UdmVarFree(v);
	  if (nvars > 0) memmove(v, v + 1, nvars * sizeof(*v));
	  vars->nvars--;
	}
	
	return UDM_OK;
}

int __UDMCALL UdmVarListReplace(UDM_VARLIST * Lst,UDM_VAR * S) {
	UDM_VAR	*v = UdmVarListFind(Lst, S->name);
	if (v == NULL) {
		return UdmVarListAdd(Lst, S);
	}else{
		UdmVarFree(v);
		UdmVarCopy(v,S);
	}
	return UDM_OK;
}

static int UdmVarListReplaceNamed(UDM_VARLIST *Lst,UDM_VAR *S, const char *name) {
	int	rc;
	UDM_VAR	*v=UdmVarListFind(Lst, S->name);
	if (v){
		UdmVarFree(v);
		rc=UdmVarCopyNamed(v,S,name);
	}else{
		UdmVarListAdd(Lst,NULL);
		v=&Lst->Var[Lst->nvars-1];
		rc=UdmVarCopyNamed(v,S,name);
		UdmSort(Lst->Var, Lst->nvars, sizeof(UDM_VAR), (qsort_cmp)varcmp);
	}
	return rc;
}

static int UdmVarListInsNamed(UDM_VARLIST *Lst,UDM_VAR *S, const char *name) {
	int	rc;
	UDM_VAR	*v=UdmVarListFind(Lst, S->name);
	if (!v){
		UdmVarListAdd(Lst,NULL);
		v=&Lst->Var[Lst->nvars-1];
		rc=UdmVarCopyNamed(v,S,name);
		UdmSort(Lst->Var, Lst->nvars, sizeof(UDM_VAR), (qsort_cmp)varcmp);
	}
	return rc;
}

/*
const char *UdmVarListFindStrTxt(UDM_VARLIST * vars,const char * name,const char * defval){
	UDM_VAR * var;
	if((var=UdmVarListFind(vars,name)) != NULL)
		return((var->txt_val != NULL) ? var->txt_val : ((var->val != NULL) ? var->val : defval));
	else
		return(defval);
}
*/

__C_LINK const char * __UDMCALL UdmVarListFindStr(UDM_VARLIST * vars, const char * name, const char * defval){
	UDM_VAR * var;
	if((var=UdmVarListFind(vars,name)) != NULL)
		return((var->val != NULL)?var->val:defval);
	else
		return(defval);
}

UDM_VARLIST * UdmVarListInit(UDM_VARLIST *l){
	if(!l){
		l=(UDM_VARLIST*)UdmMalloc(sizeof(*l));
		bzero((void*)l,sizeof(*l));
		l->freeme=1;
	} else {
		bzero((void*)l,sizeof(*l));
	}
	return l;
}


void UdmVarListFree(UDM_VARLIST * vars){
	size_t i;
	for(i=0;i<vars->nvars;i++){
		UDM_FREE(vars->Var[i].name);
		UDM_FREE(vars->Var[i].val);
	}
	UDM_FREE(vars->Var);
	vars->nvars = 0;
	if(vars->freeme)
		UDM_FREE(vars);
}

int UdmVarListAddStr(UDM_VARLIST * vars,const char * name, const char * val){
	vars->Var=(UDM_VAR *)UdmRealloc(vars->Var,(vars->nvars+1)*sizeof(UDM_VAR));
	vars->Var[vars->nvars].flags= 0;
	vars->Var[vars->nvars].section=0;
	vars->Var[vars->nvars].maxlen=0;
	vars->Var[vars->nvars].curlen = (val != NULL) ? strlen(val) : 0;	
	vars->Var[vars->nvars].name = name ? (char*)UdmStrdup(name) : NULL;
	vars->Var[vars->nvars].val = val ? (char*)UdmStrdup(val) : NULL;
	vars->nvars++;
	UdmSort(vars->Var, vars->nvars, sizeof(UDM_VAR), (qsort_cmp)varcmp);
	return vars->nvars;
}

int UdmVarListAddInt(UDM_VARLIST * vars,const char * name, int val){
	char num[64];
	udm_snprintf(num, 64, "%d", val);
	return UdmVarListAddStr(vars,name,num);
}

int UdmVarListAddUnsigned(UDM_VARLIST * vars,const char * name, uint4 val){
	char num[64];
	udm_snprintf(num, 64, "%u", val);
	return UdmVarListAddStr(vars, name, num);
}

int UdmVarListAddDouble(UDM_VARLIST * vars,const char * name, double val) {
	char num[128];
	udm_snprintf(num, 128, "%f", val);
	return UdmVarListAddStr(vars, name, num);
}

int UdmVarListInsStr(UDM_VARLIST * vars,const char * name,const char * val){
	return UdmVarListFind(vars,name) ? UDM_OK : UdmVarListAddStr(vars,name,val);
}

int UdmVarListInsInt(UDM_VARLIST * vars,const char * name,int val){
	return UdmVarListFind(vars,name) ? UDM_OK : UdmVarListAddInt(vars,name,val);
}


__C_LINK int __UDMCALL UdmVarListReplaceStr(UDM_VARLIST * vars,const char * name,const char * val){
  UDM_VAR * var;
	
  if((var=UdmVarListFind(vars,name)) != NULL)
  {
    UDM_FREE(var->val);
    if (var->maxlen == 0)
    {
      var->val= (val != NULL) ? (char*)UdmStrdup(val) : NULL;
    }
    else
    {
      if (val == NULL)
      {
        var->val = NULL;
      }
      else
      {
        var->val= (char*)UdmMalloc(var->maxlen + 4);
        strncpy(var->val, val, var->maxlen + 1);
      }
    }
    var->curlen= (val != NULL) ? strlen(val) : 0;
  }
  else
  {
    UdmVarListAddStr(vars,name,val);
  }
  return vars->nvars;
}

__C_LINK int __UDMCALL UdmVarListReplaceInt(UDM_VARLIST * vars,const char * name,int val){
	UDM_VAR * var;
	char num[64];
	if((var=UdmVarListFind(vars,name)) != NULL){
	        udm_snprintf(num, 64, "%d", val);
	        UdmVarListReplaceStr(vars, name, num);
	}else{
		UdmVarListAddInt(vars,name,val);
	}
	return vars->nvars;
}

int UdmVarListReplaceUnsigned(UDM_VARLIST * vars, const char * name, uint4 val){
	UDM_VAR * var;
	char num[64];
	if((var = UdmVarListFind(vars, name)) != NULL){
	        udm_snprintf(num, 64, "%u", val);
	        UdmVarListReplaceStr(vars, name, num);
	}else{
		UdmVarListAddUnsigned(vars, name, val);
	}
	return vars->nvars;
}

int UdmVarListReplaceDouble(UDM_VARLIST * vars, const char * name, double val) {
	UDM_VAR * var;
	char num[128];
	if((var = UdmVarListFind(vars, name)) != NULL){
	        udm_snprintf(num, 128, "%lf", val);
	        UdmVarListReplaceStr(vars, name, num);
	}else{
		UdmVarListAddDouble(vars, name, val);
	}
	return vars->nvars;
}


UDM_VAR * UdmVarListFindWithValue(UDM_VARLIST * vars,const char * name,const char * val){
	size_t i;
	
	for(i=0;i<vars->nvars;i++)
		if(!strcasecmp(name,vars->Var[i].name)&&!strcasecmp(val,vars->Var[i].val))
			return(&vars->Var[i]);
	return(NULL);
}

int UdmVarListFindInt(UDM_VARLIST * vars,const char * name,int defval){
	UDM_VAR * var;
	if((var=UdmVarListFind(vars,name)) != NULL)
		return((var->val != NULL)?atoi(var->val):defval);
	else
		return(defval);
}

unsigned UdmVarListFindUnsigned(UDM_VARLIST * vars, const char * name, unsigned defval){
	UDM_VAR * var;
	if((var=UdmVarListFind(vars,name)) != NULL)
		return((var->val != NULL) ? (unsigned)strtoul(var->val, (char**)NULL, 10) : defval);
	else
		return(defval);
}

double UdmVarListFindDouble(UDM_VARLIST * vars, const char * name, double defval) {
	UDM_VAR * var;
	if((var=UdmVarListFind(vars,name)) != NULL)
		return((var->val != NULL) ? strtod(var->val, (char**)NULL) : defval);
	else
		return(defval);
}

int UdmVarListReplaceLst(UDM_VARLIST *D, UDM_VARLIST *S,const char *name, const char *mask){
	size_t i;
	
	for(i=0;i<S->nvars;i++){
		UDM_VAR	*v=&S->Var[i];
		if(!UdmWildCaseCmp(v->name,mask)){
			UdmVarListReplaceNamed(D,v,name);
		}
	}
	return UDM_OK;
}

int UdmVarListAddLst(UDM_VARLIST *D, UDM_VARLIST *S,const char *name,const char *mask){
	size_t i;
	
	for(i=0;i<S->nvars;i++){
		UDM_VAR	*v=&S->Var[i];
		if(!UdmWildCaseCmp(v->name,mask)){
			UdmVarListAddNamed(D,v,name);
		}
	}
	return UDM_OK;
}

int UdmVarListInsLst(UDM_VARLIST *D, UDM_VARLIST *S,const char *name,const char *mask){
	size_t i;
	
	for(i=0;i<S->nvars;i++){
		UDM_VAR	*v=&S->Var[i];
		if(!UdmWildCaseCmp(v->name,mask)){
			UdmVarListInsNamed(D,v,name);
		}
	}
	return UDM_OK;
}


/* 
  Add environment variables 
  into a varlist
*/

extern char **environ;

int UdmVarListAddEnviron(UDM_VARLIST *Vars, const char *name){
	char	**e, *val;
	char	*str;
	size_t  lenstr = 1024;
	
	if ((str = (char*)UdmMalloc(1024)) == NULL) return UDM_ERROR;

	for ( e=environ; e[0] ; e++){
		size_t len = strlen(e[0]);
		if (len > lenstr) {
		  if ((str = (char*)UdmRealloc(str, lenstr = len + 64)) == NULL) return UDM_ERROR;
		}
		len = udm_snprintf(str, lenstr - 1, "%s%s%s", name ? name : "", name ? "." : "", e[0]);
		str[len]='\0';
		
		if((val=strchr(str,'='))){
			*val++='\0';
			UdmVarListAddStr(Vars,str,val);
		}
	}
	UDM_FREE(str);
	return UDM_OK;
}


