#ifndef BARRIERS_H
#define BARRIERS_H

#include <map>

#include "Tools.h"
#include "StaticDecorationBase.h"

//----------------------------------------------------------------------------
class BarrierSurfaces : public SurfacesBase
{
    SURFACES_SINGLETON_OBJECT(BarrierSurfaces);

    //------------------------------------------------------------------------
    class SurfaceKey
    {
      public:
        SurfaceKey(HVDecorationBase::Orientation o,
                   bool activated,
                   unsigned size)
            : m_o(o), m_activated(activated), m_size(size) {}
        ~SurfaceKey() {}

        friend bool operator<(const SurfaceKey &k1, const SurfaceKey &k2)
        {
            COMPARE_SURFACE_KEY_MEMBER(m_o);
            if (!k1.m_activated && k2.m_activated)  return true;
            if (k1.m_activated && !k2.m_activated)  return false;
            COMPARE_SURFACE_KEY_MEMBER(m_size);
            return false;
        }

      private:
        HVDecorationBase::Orientation m_o;
        bool m_activated;
        unsigned m_size;
    };

    typedef std::map<SurfaceKey, SDL_Surface*> SurfaceMap;
    typedef SurfaceMap::iterator SurfaceIter;
    typedef SurfaceMap::const_iterator SurfaceCIter;

  public:

    const SDL_Surface *getSurface(HVDecorationBase::Orientation o,
                                  bool activated, unsigned size);

  private:

    SDL_Surface *createSurface(HVDecorationBase::Orientation o,
                               bool activated, unsigned size) const;


    /// All barrier frames for the left bound.
    SDL_Surface *m_left[2];

    /// All barrier frames for the beam.
    SDL_Surface *m_beam[2];

    /// The surface for the deactivated beam.
    SDL_Surface *m_deactivated;

    /// All barrier frames for the right bound.
    SDL_Surface *m_right[2];


    /// A map for all dynamically created barrier surfaces via getSurface().
    SurfaceMap m_surfaceMap;
};



//----------------------------------------------------------------------------
#undef DECLARE_UPDATE_STATE
#define DECLARE_UPDATE_STATE(c) \
    class c; \
    friend class c; \
    class c : public UpdateState \
    { \
        STATE_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        void onEntry(Barrier *barrier); \
        void update(Barrier *barrier);  \
    }


//----------------------------------------------------------------------------
class Barrier : public StaticDecorationBase,
                public AnimatedDecorationBase,
                public HVDecorationBase
{
    //------------------------------------------------------------------------
    class UpdateState
    {
      protected:
        UpdateState() {}
      public:
        virtual ~UpdateState() {}

        /**
         * This method is always called when changing to a new state.
         * Its purpose is to change the barrier's m_activated member.
         */
        virtual void onEntry(Barrier *barrier) = 0;

        virtual void update(Barrier *barrier) = 0;
    };

    //------------------------------------------------------------------------
    DECLARE_UPDATE_STATE(Initial);
    DECLARE_UPDATE_STATE(InitialDelay);
    DECLARE_UPDATE_STATE(Activated);
    DECLARE_UPDATE_STATE(Deactivated);
    DECLARE_UPDATE_STATE(DoNothing);

  public:

    //------------------------------------------------------------------------
    class InitializationData
        : public StaticDecorationBase::InitializationData,
          public AnimatedDecorationBase::InitializationData,
          public HVDecorationBase::InitializationData
    {
      public:
        InitializationData();
        InitializationData(const XMLNode *magnetNode);
        ~InitializationData();

        bool activatedPresent;
        bool activated;
    };

    //------------------------------------------------------------------------
    Barrier(const InitializationData &init);
    ~Barrier();

    //------------------------------------------------------------------------
    inline void activate()
    {
        m_activated = true;
    }

    inline void deactivate()
    {
        m_activated = false;
    }

    inline void toggle()
    {
        m_activated = !m_activated;
    }

    //------------------------------------------------------------------------
    inline bool isActivated() const
    {
        return m_activated;
    }

    //------------------------------------------------------------------------
    inline void setUpdateState(UpdateState *state)
    {
        if (state)  state->onEntry(this);
        m_updateState = state;
    }

    //------------------------------------------------------------------------
    void update();

    //------------------------------------------------------------------------
    void updateSurface();

    //------------------------------------------------------------------------
    static Barrier *create(const XMLNode *barrierNode);

  private:
    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    //------------------------------------------------------------------------
    bool m_activated;

    //------------------------------------------------------------------------
    unsigned m_frameCounter;

    UpdateState *m_updateState;
};

#endif //BARRIERS_H
