///////////////////////////////////////////////////////////////////////////////
// $Id: maptoolline.cpp,v 1.2 2004/10/03 19:30:45 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    maptoolline.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the line tool class for maps
*/

// local includes
#include "mapdocument.h"
#include "maptoolline.h"

///////////////////////////////////////////////////////////////////////////////

MapToolLine::MapToolLine(MapDocument* doc) : MapTool(doc)
{
}

///////////////////////////////////////////////////////////////////////////////

MapToolLine::~MapToolLine()
{
}

///////////////////////////////////////////////////////////////////////////////

void MapToolLine::activate()
{
	// clear rect
	m_rect  = QRect();
	m_start = QPoint();
	m_end   = QPoint();
}

///////////////////////////////////////////////////////////////////////////////

bool MapToolLine::usedMouseRelease(uint row, uint col, int button)
{
	if (button == Qt::RightButton)
	{
		activate();
		return true;
	}
	
	if (button != Qt::LeftButton) return false;

	// sanity checks
	if (m_doc == 0) return false;
	if (static_cast<int>(row) >= m_doc->numRows() ||
	    static_cast<int>(col) >= m_doc->numCols()) return false;

	// Do not draw on invisible layers
	if (!m_doc->isLayerVisible(m_doc->currentLayer())) return false;
	
	// if the rectangle is already valid, this is the second click
	// which finishes it
	if (m_rect.isNull())
	{
		m_rect = QRect(col, row, 1, 1);
		m_start = QPoint(col, row);
		calculatePoints();
	}
	else
	{
		m_end = QPoint(col, row);
		m_rect = QRect(m_start, m_end).normalize();
		m_rect.setWidth(m_rect.width() + 1);
		m_rect.setHeight(m_rect.height() + 1);
		calculatePoints();

		for (uint p = 0; p < m_points.count(); ++p)
		{
			m_doc->drawCurrentTileAt(m_points[p].y(), m_points[p].x());
		}
		m_rect = QRect();
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapToolLine::usedMouseMove(uint row, uint col, int)
{
	// sanity checks
	if (m_doc == 0) return false;
	
	// Do not draw on invisible layers
	if (!m_doc->isLayerVisible(m_doc->currentLayer())) return false;
	
	if (m_rect.isNull()) return false;

	m_end = QPoint(col, row);
	m_rect = QRect(m_start, m_end).normalize();
	calculatePoints();

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapToolLine::coversTileAt(uint row, uint col) const
{
	if (!m_rect.contains(col, row)) return false;

	return m_points.find(QPoint(col, row)) != -1;
}

///////////////////////////////////////////////////////////////////////////////

QRect MapToolLine::coverArea() const
{
	return m_rect;
}

///////////////////////////////////////////////////////////////////////////////

void MapToolLine::calculatePoints()
{
	if (m_rect.isNull()) return;

	QPoint p1 = m_start;
	QPoint p2 = m_end;

	if (m_start.x() > m_end.x() && m_start.y() > m_end.y())
	{
		p1 = m_end;
		p2 = m_start;
	}

	int x1 = QMIN(p1.x(), p2.x());
	int y1 = QMIN(p1.y(), p2.y());
	int x2 = QMAX(p1.x(), p2.x());
	int y2 = QMAX(p1.y(), p2.y());

	int dx = x2 - x1;
	int dy = y2 - y1;

	if (dx > dy)
	{
		m_points.resize(dx + 1);
		if (dy == 0)
		{
			for (uint p = 0; p < m_points.count(); ++p)
			{
				m_points.setPoint(p, QPoint(x1 + p, y1));
			}
		}
		else
		{
			double delta = static_cast<double>(dx) / static_cast<double>(dy);

			int x = x1;
			uint p = 0;
			for (int y = y1; y <= y2; ++y)
			{
				dy = y - y1;
				x2 = x1 + static_cast<int>(delta*dy);
				for (; x <= x2; ++x)
				{
					dx = x - x1;

					int col = x;
					int row = y;
					if (p1.x() > p2.x())
						row = p2.y() - dy;
					if (p1.y() > p2.y())
						col = p2.x() - dx;

					m_points.setPoint(p, QPoint(col, row));
					p++;
				}
			}
		}
	}
	else
	{
		m_points.resize(dy + 1);

		if (dx == 0)
		{
			for (uint p = 0; p < m_points.count(); ++p)
			{
				m_points.setPoint(p, QPoint(x1, y1 + p));
			}
		}
		else
		{
			double delta = static_cast<double>(dy) / static_cast<double>(dx);

			int y = y1;
			uint p = 0;
			for (int x = x1; x <= x2; ++x)
			{
				dx = x - x1;
				y2 = y1 + static_cast<int>(delta*dx);
				for (; y <= y2; ++y)
				{
					dy = y - y1;

					int col = x;
					int row = y;
					if (p1.x() > p2.x())
						row = p2.y() - dy;
					if (p1.y() > p2.y())
						col = p2.x() - dx;

					m_points.setPoint(p, QPoint(col, row));
					p++;
				}
			}
		}
	}
}

// End of file
