/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Classes for handling child processes
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         Mike Butler,    mgb@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 * $Id$
 */


#ifndef _UtChild_h
#define _UtChild_h

#include <errno.h>
#include <fcntl.h>		// For O_RDONLY
#include <signal.h>
#include <sys/wait.h>		// for WNOHANG
#include <unistd.h>
#include <UtReport.h>
#include <UtSequence.h>
#include <UtTime.h>

class Child {

protected:
  pid_t cPid;			// Child's pid...
  const String cCommand;	// Command child is to run
  int cStat;			// Child exit status here...
  int cReadFd;
  int cWriteFd;

public:

  int ReadFd() { return cReadFd; }
  int WriteFd() { return cWriteFd; }

  /// Open() - Initialize child
  // This is a tad long but straight forward...
  bool Open() {
    /* Split command into argv vector for exec... */
    Sequence seq;
    seq.FieldSep() = " ";
    seq.Split(cCommand);
    
    /* Convert to argvZ for exec... */
    const char **argv = new const char *[seq.Size()+2];
    seq.ArgvZ(argv);
    
    // Temporary FD's for pipes...
    int parentToChild[2];
    int childToParent[2];
    
    if(pipe(parentToChild)) return(false);
    if(pipe(childToParent)) {
      close(parentToChild[0]);
      close(parentToChild[1]);
      return(false);
    }
    
    /* Then make child and hook up comm */
    if(!(cPid = fork())) {
      /* We're the child...  Use stdin/stdout for pipe */
      dup2(parentToChild[0], STDIN_FILENO);
      dup2(childToParent[1], STDOUT_FILENO);
      // #### Option - somtimes capture stderr output too!
      // Perhaps to a seperate port
      //dup2(childToParent[1], STDERR_FILENO);
      close(parentToChild[0]);
      close(parentToChild[1]);
      close(childToParent[0]);
      close(childToParent[1]);
      
      execvp(argv[0], const_cast<char**>(argv));
      Report::Error("Child::Open() -exec failed!");
    }
    
    /* We're parent... Clean up a bit... */
    close(parentToChild[0]);	// Other end is child's
    close(childToParent[1]);
    fcntl(parentToChild[1], F_SETFL, O_NONBLOCK);
    
    delete [] argv;

    // Clean up alot if things went badly...
    if(cPid < 0) {
      close(childToParent[0]);
      close(parentToChild[1]);
      return(false);
    }

    // Otherwise, prep to run...
    
    cWriteFd = parentToChild[1];
    cReadFd  = childToParent[0];
    return(true);
  }

  virtual void Close() {
    
    timeval tv = Time(0.01); 
    select(0,0,0,0,&tv);	// Wait 10mS for child to see it...
    
    int timeout = 10;
    if(cPid > 0) {
      kill(cPid, SIGHUP);
      kill(cPid, SIGINT);
      // Wait a little while to reap child...
      while(--timeout) {
	tv = Time(0.1);
	if(IsRunning()) select(0,0,0,0,&tv);
	else break;
      }
      
      // Child didn't stop?  Time to get serious!
      if(!timeout) {
	kill(cPid, SIGKILL);
	waitpid(cPid, &cStat, 0);
      }
    close(cReadFd);
    close(cWriteFd);
    }
    
    cPid = 0;
    cReadFd = -1;
    cWriteFd = -1;
  }


  // Check on child's health...
  // Record exit status and clear cPid as side effects...
  bool IsRunning() {
    bool done = waitpid(cPid, &cStat, WNOHANG);
    if(done) cPid = 0;
    return(!done);
  }

  int ExitStatus() {
    return(cStat);
  }

  Child(const String &command)  :
    cPid(0), cCommand(command), cStat(-1), cReadFd(-1), cWriteFd(-1)
    { 
    }
  

  virtual ~Child() { 
    Close();
  }
  
  /// System() - Make a synchronous system call with timeout, capture result...
  //  This is a convenience routine and is only tenuously related to C.I.
  //  Args:
  //     cmd    -  is the command line
  //     result -  pointer to result string  0 ==> don't capture result
  //     limit  -  pointer to time limit  
  //     data   -  pointer to data to pipe out to child on stdin
  static bool System(const String &cmd, 
		     const String  *input = 0,
		     String *output = 0, 
		     String *error = 0, 
		     const Time &limit = Time(0)
		     ) {
    String childInput;
    if(input) childInput = *input;
    if(output) *output = "";
    if(error)  *error = "";

    /* Make communication paths between parent and child... */
    enum { in  = 0, inR =  in,  inW  = inR+1,
	   out = 2, outR = out, outW = outR+1,
	   err = 4, errR = err, errW = errR+1,
    };

    int pipes[6] = { -1, -1, -1, -1, -1, -1, };
    bool ok = false;
    do {
      if((input)  && (pipe(&pipes[in])))  break;
      if((output) && (pipe(&pipes[out]))) break;
      if((error)  && (pipe(&pipes[err]))) break;
      ok = true;
    } while(false);
    if(!ok) {
      for(size_t i = 0; i < sizeof(pipes)/sizeof(pipes[0]); i++) 
	close(pipes[i]);
      if(error) *error = String("Can't make pipes - ") + strerror(errno);
      return(false);
    }
    
    /* Then birth child and hook up tubes */
    int child;
    if(!(child = fork())) {
      /* We're child...  Use stdin/stdout for pipe */
      dup2(pipes[ inR],  STDIN_FILENO);
      dup2(pipes[outW], STDOUT_FILENO);
      dup2(pipes[errW], STDERR_FILENO);
      for(size_t i = 0; i < sizeof(pipes)/sizeof(pipes[0]); i++) 
	close(pipes[i]);
      
      execlp("sh", "sh", "-c", cmd.c_str(), 0);

      /* Shouldn't get here! */
      Report::Error(String("Child::System() - '") + cmd + "' failed -" + strerror(errno));
    }
    
    if(child < 0) {
      if(error) *error = String("Can't exec - ") + strerror(errno);
      return(false);
    }
    
    /* We're parent... Clean up a bit... */
    close(pipes[inR]);
    close(pipes[outW]);
    close(pipes[errW]);
    fcntl(pipes[inW], F_SETFL, O_NONBLOCK);
    
    /* Run command, send data, collect output... */
    int got, put;
    char buf[512];
    
    Time timeout = Time::Eternity();
    if(limit > 0) timeout = Time::Now() + limit;
    
    // Keep cranking as long as there's something to do...
    while(input || output || error) {
      // Any input to send to child?
      if(input) {
	if(childInput.length()) {
	  put = write(pipes[inW], childInput.c_str(), childInput.length());
	  if(put >= 0) childInput = childInput.substr(put);
	} 
	if(!childInput.length()) {
	  // Need close() to send child EOF...
	  close(pipes[inW]);
	  pipes[inW] = -1;
	  input = 0;
	}
      }

      /* Run out of time?  Kill child and give up... */
      if(Time::Now() > timeout) {
	kill(child, SIGKILL);
	break;
      }

      // Collecting output?  If not, don't even check...
      if(!(error || output)) {
	if(pipes[inW] == -1) {
	  kill(child, SIGKILL);
	  break;
	}
	continue;
      }

      /* Gathering output... need to spin loop to allow overlapped I/O... 
       * If there's more output, then don't sleep, otherwise sleep for input
       */
      fd_set rfds;
      FD_ZERO(&rfds);
      if(output) FD_SET(pipes[outR], &rfds);
      if(error)  FD_SET(pipes[errR], &rfds);
      int tmp = pipes[outR];
      if(tmp < pipes[errR]) tmp = pipes[errR];

      timeval wake = { 0, 10000 };
      // if(childInput.length()) wake.tv_usec = 0;

      switch(select(tmp+1, &rfds, 0, 0, &wake)) {
      case -1:
	/* Error! */
	kill(child, SIGKILL);
	input = output = error = 0;
	break;
      case 0:
	/* Timeout, make another pass... */
	break;
      default:
	/* Have input from child; accumulate it... */
	if((pipes[outR] > -1) && FD_ISSET(pipes[outR], &rfds)) {
	  got = read(pipes[outR], buf, sizeof(buf));
	  if(got == 0) output = 0;
	  else if((got < 0) && (errno == EWOULDBLOCK)) ; // nop
	  else if(got < 0) output = 0;
	  else if(output) *output = *output + String(buf, 0, got);
	}
	// Maybe child wrote to STDERR...
	if((pipes[errR] > -1) && FD_ISSET(pipes[errR], &rfds)) {
	  got = read(pipes[errR], buf, sizeof(buf));
	  if(got == 0) error = 0;
	  else if((got < 0) && (errno == EWOULDBLOCK)) ; // nop
	  else if(got < 0) error = 0;
	  else if(error) *error = *error + String(buf, 0, got);
	}
      }
    }
    
    /* Here on EOF or error... */
    int stat;
    waitpid(child, &stat, 0);
    close(pipes[inW]);
    close(pipes[outR]);
    close(pipes[errR]);
    
    /* Failure if stat is non-zero */
    return((stat) ? false : true);
  }
};

#endif
