# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.box import HBox
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image


from elisa.core.resource_manager import NoMatchingResourceProvider
from elisa.core.media_uri import MediaUri
from elisa.core import common

# i18n
from elisa.core.utils.i18n import install_translation
_ = install_translation('poblesec')

from twisted.internet import defer


class ScannerWidget(Widget):
    def __init__(self):
        super(ScannerWidget, self).__init__()
        # to be able to be more verbose on statistics
        self._currently_scanning = None
        self._last_count = 0

        self._create_widgets()
        self.update_style_properties(self.style.get_items())

    def _create_widgets(self):
        self.background = Image()
        self.background.visible = True
        self.add(self.background)

        self.horizontal_layout = HBox()
        self.horizontal_layout.visible = True
        self.add(self.horizontal_layout)

        self.icon = Image()
        self.icon.visible = True
        self.horizontal_layout.pack_start(self.icon)

        self.labels = Widget()
        self.labels.visible = True
        self.horizontal_layout.pack_start(self.labels)

        self.text = Text()
        self.text.visible = True
        self.labels.add(self.text)

        self.subtext = Text()
        self.subtext.visible = True
        self.labels.add(self.subtext)

    def running__set(self, running):
        if running:
            self.opacity = 255
        else:
            self.opacity = 0

    running = property(fset=running__set)

    def currently_scanning__set(self, scan_resource):
        if self._currently_scanning:
            # we have to unbind from the old one
            self.files_scanned = 0
            self._last_count = 0
            self._currently_scanning.unbind_object(self)
            self._currently_scanning = None
            self.text.label = ''
            self.subtext.label = ''

        if scan_resource is None:
            # scanning stopped
            self.running = False
        else:
            self.running = True
            self._currently_scanning = scan_resource

            self.text.label = _('Scanning %(resource)s') % {'resource': scan_resource.name}
            self.subtext.label = _('estimating')
            scan_resource.bind('files_scanned', self, 'files_scanned')

    currently_scanning = property(fset=currently_scanning__set)

    def files_scanned__set(self, counter):
        total = self._currently_scanning.files_total
        if not total:
            # there seem to be no total amount. we can't calculate
            return

        # update the display every 5 files
        if counter - self._last_count < 5:
            return

        self._last_count = counter
        self.subtext.label = _('File %(index)d of %(count)d') % {'index': counter,
                                                                 'count': total}

    files_scanned = property(fset=files_scanned__set)

    def set_focus(self):
        return False

    def clean(self):
        self.subtext.clean()
        self.subtext = None
        self.text.clean()
        self.text = None
        return super(ScannerWidget, self).clean()


def scanning_decorator(controller):
    uri = MediaUri("media_scanner://localhost/statistic")
    try:
        model, dfr = common.application.resource_manager.get(uri)
    except NoMatchingResourceProvider, e:
        # Media Scanner not installed
        return defer.fail(e)

    widget = ScannerWidget()
    widget.visible = True
    controller.topbar.button_bar.pack_start(widget)

    model.bind('currently_scanning', widget, 'currently_scanning')

    return defer.succeed(widget)
