/* build layout

   Written by Matthias Hensler
   Copyright WSPse 1999+2000
   eMail: wsp@gmx.de

Created: 1999/06/06
Updated: 2000/07/27
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
   */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ncurses.h>
#include "mp3creat.h"

extern char *copy_char_str(char *old);
extern char *mp3_genre[];
extern void wuuush(int);
extern int  track_last;           /* number of last track of currently inserted cd    */
extern int  mp3_frame_mult;       /* frame multiplikator to calculate size of file    */
extern int  num_ripped_tracks;    /* number of already ripped tracks (in second list) */
extern BOOL config_cddb_enbl;
extern char *def_unknown_gen;

extern struct {
  int min;
  int sec;
  int frame;
} cdtoc[100];

#define TOP_STR "WSPse MP3-Creator"

/* global anchors for left and right side of main-window:           */
song_typ *lay_global_anchor[2];   /* start of linear list           */
song_typ *lay_top_anchor[2];      /* entry on top of the screen     */
song_typ *lay_curr_anchor[2];     /* entry currently selected       */
int lay_select_line[2];           /* row of selected entry          */
unsigned long int lay_tot_frm[2]; /* total frames on both sides     */
int lay_divider;                  /* size of left side (prozentual) */
int lay_act_side;                 /* 0=left, 1=right side of layout */

void fill_in_songs(WINDOW *win);

long int sub_frame_calc(song_typ *anchor)
{
  song_typ *curr;
  long int frames;

  frames = 0L;
  curr   = anchor;
  while(curr) {
    if(curr->convert) {
      frames += curr->frame_len;
    }
    curr = curr->next;
  }
  
  return frames;
}
	
void calc_tot_frm()
{
  lay_tot_frm[0] = sub_frame_calc(lay_global_anchor[0]);
  lay_tot_frm[1] = sub_frame_calc(lay_global_anchor[1]);
}
  
int calc_divider_pos(int maxx)
{
  if(lay_divider == 0 || lay_divider == 100) return -1;
  else                                       return ((lay_divider * maxx) / 100);
}
  
void draw_main_win(WINDOW *win, int dummy1, int dummy2)
{
  int maxx, maxy;
  int divider;
  char help_str[50];
  
  curs_set(0);
  wclear(win);
  scrollok(win, FALSE);
  keypad(win, TRUE);

  wbkgd(win, COLOR_PAIR(1));
  getmaxyx(stdscr, maxy, maxx);

  divider = calc_divider_pos(maxx);

  box(win, 0, 0);

  wmove(win, maxy-8, 0);
  waddch(win, ACS_LTEE);
  whline(win, 0, maxx-2);
  wmove(win, maxy-8, maxx-1);
  waddch(win, ACS_RTEE);

  strcpy(help_str, _("press F1 or 'H' for help"));

  if(divider != -1) {
    mvwaddch(win, 0, divider, ACS_TTEE);
    mvwvline(win, 1, divider, ACS_VLINE, maxy-9);
    mvwaddch(win, maxy-8, divider, ACS_BTEE);
  }
  
  mvwaddstr(win, maxy-7, 1, _("Track  :   /   [  :  ,~     KB] - Total:   :   [~      KB]"));
  mvwaddstr(win, maxy-6, 1, _("Title  :"));
  mvwaddstr(win, maxy-5, 1, _("Artist :"));
  mvwaddstr(win, maxy-4, 1, _("Album  :"));
  mvwaddstr(win, maxy-3, 1, _("File   :"));
  mvwaddstr(win, maxy-2, 1, _("Year   :       Genre:"));

  if(maxx > strlen(help_str))
    mvwaddstr(win, maxy-1, maxx-strlen(help_str)-1, help_str);
  if(maxx > strlen(TOP_STR))
    mvwaddstr(win, 0, (maxx-strlen(TOP_STR))>>1, TOP_STR);
	    
  fill_in_songs(win);
}

void draw_song_status(WINDOW *win, song_typ *song)
{
  int maxx, maxy;
  chtype old_ch;
  char year[10];
  int i;
  long int j;
  char *dirname;

  getmaxyx(stdscr, maxy, maxx);
  if(lay_act_side == 0)
    mvwaddstr(win, maxy-7, 1, _("Track  :   /   [  :  ,~     KB] - Total:   :   [~      KB]"));
  else
    mvwaddstr(win, maxy-7, 1, _("Tracks :       [  :  ,~     KB] - Total:   :   [~      KB]"));

  old_ch = getbkgd(win);
  wbkgdset(win, COLOR_PAIR(2) | A_BOLD);

  if(lay_act_side == 0) {
    sprintf(year, "%02d", (song->toc)+1);
    mvwaddstr(win, maxy-7, 10, year);
    sprintf(year, "%02d", track_last);
    mvwaddstr(win, maxy-7, 13, year);
  } else {
    sprintf(year, "%05d", num_ripped_tracks);
    mvwaddstr(win, maxy-7, 10, year);
  }

  j = song->frame_len;
  sprintf(year, "%02ld", (j / 4500));
  mvwaddstr(win, maxy-7, 17, year);
  sprintf(year, "%02ld", (j / 75) % 60);
  mvwaddstr(win, maxy-7, 20, year);
  sprintf(year, "%-5ld", (j * mp3_frame_mult) / 1024);
  mvwaddstr(win, maxy-7, 24, year);

  sprintf(year, "%02ld", (lay_tot_frm[lay_act_side] / 4500));
  mvwaddstr(win, maxy-7, 42, year);
  sprintf(year, "%02ld", (lay_tot_frm[lay_act_side] / 75) % 60);
  mvwaddstr(win, maxy-7, 45, year);
  sprintf(year, "%-6ld", (lay_tot_frm[lay_act_side] * mp3_frame_mult) / 1024);
  mvwaddstr(win, maxy-7, 50, year);
  
  sprintf(year, "%04d", song->year);
  for(i=maxy-6;i<maxy-2;i++) {
    wmove(win, i, 10);
    whline(win, ' ', maxx-11);
  }
  wmove(win, maxy-2, 23);
  whline(win, ' ', maxx-26);
  mvwaddnstr(win, maxy-6, 10, song->title,    maxx-11);
  if(maxx-11 < strlen(song->title)) mvwaddch(win, maxy-6, maxx-2, '~');
  mvwaddnstr(win, maxy-5, 10, song->artist,   maxx-11);
  if(maxx-11 < strlen(song->artist)) mvwaddch(win, maxy-5, maxx-2, '~');
  mvwaddnstr(win, maxy-4, 10, song->album,    maxx-11);
  if(maxx-11 < strlen(song->album)) mvwaddch(win, maxy-4, maxx-2, '~');
  
  /* draw filename: at least show beginning of the filename.
   * if there is more space than the filename needs, show the end of the directoryname
   */
  wmove(win, maxy-3, 10);
  i = maxx-11;     /* total space we have here */
  if(strlen(song->filename)+1 >= i) {
    /* we can only display the filename */
    waddch(win, '~');
    i--;
  } else if(song->dirname) {
    /* there is some space for the directoryname */
    if(*(song->dirname + strlen(song->dirname) -1) != '/') {
      dirname = (char *) malloc(sizeof(char) * (strlen(song->dirname)+2));
      if(! dirname) {
	wuuush(1);
      }
      sprintf(dirname, "%s/", song->dirname);
    } else {
      dirname = copy_char_str(song->dirname);
    }
    /* dirname now contains the path with a slash at the end */
    j = i - strlen(song->filename);
    if(j < strlen(dirname)) {
      waddch(win, '~');
      j--;
      i -= j;
      waddstr(win, (dirname + (strlen(dirname)-j)));
    } else {
      /* enough space to show everything */
      waddstr(win, dirname);
      /* we do not really need to reset i, since there is enough space */
      i -= strlen(dirname);
    }

    free(dirname);
  }
  
  /* show filename */
  if(i < strlen(song->filename)) {
    waddnstr(win, song->filename, i-1);
    waddch(win, '~');
  } else {
    waddstr(win, song->filename);
  }
    
  mvwaddnstr(win, maxy-2, 10, year, 4);
  mvwaddnstr(win, maxy-2, 23, (song->genre != TOT_GENRES ? mp3_genre[song->genre] : def_unknown_gen), 20);

  waddstr(win, "  ");
  if(! song->on_fly) wbkgdset(win, old_ch);
  else               wbkgdset(win, COLOR_PAIR(4));
  if(lay_act_side == 0) waddstr(win, _("[on fly]"));
  else                  waddstr(win, _("[delete]"));

  wbkgdset(win, old_ch);
  waddstr(win, " ");
  if(config_cddb_enbl) wbkgdset(win, COLOR_PAIR(4));
  waddstr(win, _("[CDDB]"));

  wbkgdset(win, old_ch);
  waddstr(win, " ");
  if(song->sampler) wbkgdset(win, COLOR_PAIR(4));
  waddstr(win, _("[Sampler]"));

  wbkgdset(win, old_ch);
}

void draw_song_line(WINDOW *win, song_typ *song, BOOL marked, int line)
{
  int maxx, maxy;
  chtype old_ch, new_ch;
  char *buf;
  int i;
  int divider, start, end;

  getmaxyx(stdscr, maxy, maxx);
  old_ch = getbkgd(win);
  i=0;
  if(marked) i |= 2;
  if(song->convert) i |= 1;
  new_ch = COLOR_PAIR(i+1);
  if(song->convert) new_ch |= A_BOLD;
  wbkgdset(win, new_ch);

  divider = calc_divider_pos(maxx);

  start = 1;
  end = maxx-2;
  if(divider != -1) {
    if(lay_act_side == 0) end   = divider-1;
    else                  start = divider+1;
  }
  
  wmove(win, line, start);
  whline(win, ' ', (end-start)+1);

  buf = (char *) malloc(sizeof(char) * (strlen(song->title) +
					strlen(song->artist) + 4));
  if(buf == NULL) {
    perror("malloc");
    wuuush(1);
  }
  
  sprintf(buf, "%s (%s)", song->title, song->artist);
  waddnstr(win, buf, (end-start)+1);
  free(buf);

  wbkgdset(win, old_ch);
}

void lay_fill_left(WINDOW *win, BOOL do_mark)
{
  int i;
  int maxx, maxy;
  song_typ *curr;
  int divider, end;

  lay_curr_anchor[0] = NULL;

  if(lay_divider == 0) return;
  
  getmaxyx(stdscr, maxy, maxx);
  divider = calc_divider_pos(maxx);

  if(divider != -1) end = divider-1;
  else              end = maxx-1;
  
  for(i=1;i<=maxy-9;i++) mvwhline(win, i, 1, ' ', end-1);

  i    = 1;
  curr = lay_top_anchor[0];
  
  /* print silly message, if no cd is inserted, or device was illegal */
  if(! curr) {
    mvwaddnstr(win, 1, 1, _("no cd-volume or illegal device"), end-1);
    mvwaddnstr(win, 2, 2, _("press \"F2\" for configmenu"), end-2);
    mvwaddnstr(win, 3, 2, _("or insert cd an press \"V\""), end-2);
    return;
  }
  
  while(curr != NULL) {
    if(i == lay_select_line[0]) {
      draw_song_line(win, curr, do_mark, i);
      draw_song_status(win, curr);
      lay_curr_anchor[0] = curr;
    } else {
      draw_song_line(win, curr, FALSE, i);
    }
    if(i == maxy-9) break;
    i++;
    curr = curr->next;
  }
}

void lay_fill_right(WINDOW *win, BOOL do_mark)
{
  int i;
  int maxx, maxy;
  song_typ *curr;
  int divider, start;
 
  lay_curr_anchor[1] = NULL;
  
  if(lay_divider == 100) return;
  
  getmaxyx(stdscr, maxy, maxx);
  divider = calc_divider_pos(maxx);
  
  if(divider != -1) start = divider+1;
  else              start = 1;
  
  for(i=1;i<=maxy-9;i++) mvwhline(win, i, start, ' ', (maxx-1)-start);
  
  i    = 1;
  curr = lay_top_anchor[1];
  
  while(curr != NULL) {
    if(i == lay_select_line[1]) {
      draw_song_line(win, curr, do_mark, i);
      draw_song_status(win, curr);
      lay_curr_anchor[1] = curr;
    } else {
      draw_song_line(win, curr, FALSE, i);
    }
    if(i == maxy-9) break;
    i++;
    curr = curr->next;
  }
}

void fill_in_songs(WINDOW *win)
{
  if(lay_act_side == 1) {
    lay_act_side = 0;
    lay_fill_left(win, FALSE);
    lay_act_side = 1;
    lay_fill_right(win, TRUE);
  } else {
    lay_act_side = 1;
    lay_fill_right(win, FALSE);
    lay_act_side = 0;
    lay_fill_left(win, TRUE);
  }
}

void fill_in_side(WINDOW *win, int side)
{
  if(side == 0) lay_fill_left(win, TRUE);
  else          lay_fill_right(win, TRUE);
}

void layout_update_cddb_flag(WINDOW *win)
{
  if(lay_global_anchor[lay_act_side] && lay_curr_anchor[lay_act_side]) {
    draw_song_status(win, lay_curr_anchor[lay_act_side]);
  } else {
    fill_in_songs(win);
  }

}
