/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001 Richard Hult     <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 *         Richard Hult     <rhult@codefactory.se> 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <time.h>

#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "libmrproject/GNOME_MrProject.h"
#include "network-canvas.h"
#include "network-item.h"
#include "task-box.h"
#include "arrow-item.h"

#define DEBUG 0
#include "util/debug.h"

static void  network_canvas_init       (NetworkCanvas      *network_canvas);
static void  network_canvas_class_init (NetworkCanvasClass *klass);
static void  nc_destroy                (GtkObject          *object);
static void  nc_realize                (GtkWidget          *widget);
static void  nc_size_allocate          (GtkWidget          *widget,
					GtkAllocation      *allocation);
static void  nc_reflow                 (NetworkCanvas      *network_canvas);


struct _NetworkCanvasPriv {
        GnomeCanvasGroup *orphaned_boxes;
	IdMap            *tasks;
	IdMap            *arrows;
	TaskBox          *root_box;
	guint             reflow_idle_id;
};

GNOME_CLASS_BOILERPLATE (NetworkCanvas, network_canvas, 
                         GnomeCanvas,   gnome_canvas);


static void
network_canvas_init (NetworkCanvas *network_canvas)
{
        NetworkCanvasPriv *priv;
        
        priv = g_new0 (NetworkCanvasPriv, 1);
        
        priv->orphaned_boxes = GNOME_CANVAS_GROUP (
                gnome_canvas_item_new (
                        gnome_canvas_root (GNOME_CANVAS (network_canvas)),
                        GNOME_TYPE_CANVAS_GROUP,
			"x", 0.0,
			"y", 0.0,
                        NULL));

        gnome_canvas_item_hide (GNOME_CANVAS_ITEM (priv->orphaned_boxes));
                                                 
	priv->tasks = id_map_new (0);
	priv->arrows = id_map_new (0);
        
        network_canvas->priv = priv;

}

static void
network_canvas_class_init (NetworkCanvasClass *klass)
{
        GtkObjectClass   *object_class;
        GtkWidgetClass   *widget_class;

        object_class = (GtkObjectClass *) klass;
        widget_class = (GtkWidgetClass *) klass;

        object_class->destroy = nc_destroy;
        
        widget_class->realize = nc_realize;
	widget_class->size_allocate = nc_size_allocate;
}

static void
nc_destroy (GtkObject *object)
{
	NetworkCanvas *canvas;
	
	canvas = NETWORK_CANVAS (object);
	
	if (canvas->priv->reflow_idle_id) {
		g_source_remove (canvas->priv->reflow_idle_id);
		canvas->priv->reflow_idle_id = 0;
	}

	if (canvas->priv) {
		gtk_object_unref (GTK_OBJECT (canvas->priv->tasks));
		gtk_object_unref (GTK_OBJECT (canvas->priv->arrows));
	
		g_free (canvas->priv);
		canvas->priv = NULL;
	}
	
        GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
nc_realize (GtkWidget *widget)
{
	GdkColormap     *colormap;
	GtkStyle        *style;

	d(puts (__FUNCTION__));

	g_return_if_fail (widget != NULL);
        g_return_if_fail (IS_NETWORK_CANVAS (widget));

        GNOME_CALL_PARENT_HANDLER (GTK_WIDGET_CLASS, realize, (widget));

	/* We set the background pixmap to NULL so that X won't clear 
	 * exposed areas and thus be faster.
	 */
	gdk_window_set_back_pixmap (GTK_LAYOUT (widget)->bin_window,
				    NULL,
				    FALSE);

	/* Set the background to white. */
	style = gtk_style_copy (widget->style);
	colormap = gtk_widget_get_colormap (widget);
	gdk_color_white (colormap, &style->bg[GTK_STATE_NORMAL]);
	gtk_widget_set_style (widget, style);
	gtk_style_unref (style);
}

static void
nc_size_allocate (GtkWidget     *widget,
		  GtkAllocation *allocation)
{
        GNOME_CALL_PARENT_HANDLER (GTK_WIDGET_CLASS,
				   size_allocate,
				   (widget, allocation));
	
	nc_reflow (NETWORK_CANVAS (widget));
}

GtkWidget *
network_canvas_new (void)
{
        NetworkCanvas     *network_canvas;
	NetworkCanvasPriv *priv;

        network_canvas = gtk_type_new (NETWORK_CANVAS_TYPE);
        priv           = network_canvas->priv;
	
	priv->root_box = TASK_BOX (
		gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (network_canvas)),
				       TYPE_TASK_BOX,
				       "x", 0.0,
				       "y", 0.0,
				       NULL));

	id_map_insert_id (priv->tasks, 0, priv->root_box);

        return GTK_WIDGET (network_canvas);
}

static void
item_destroyed (NetworkItem *item, NetworkCanvas *canvas)
{
	NetworkCanvasPriv *priv;
	GM_Id              id;

	g_return_if_fail (canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (canvas));

	priv = canvas->priv;

	if (!item->task) {
		/* Special case the root box. */
		id = 0;
	} else {
		id = item->task->taskId;
	}

	id_map_remove (priv->tasks, id);
}	

void
network_canvas_insert_task (NetworkCanvas *network_canvas,
                            GM_Task       *task)
{
	NetworkCanvasPriv *priv;
	NetworkItem       *item, *parent_item;

	d(puts (__FUNCTION__));
	
	g_return_if_fail (network_canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (network_canvas));

	priv = network_canvas->priv;

	item = NETWORK_ITEM (gnome_canvas_item_new (priv->orphaned_boxes,
						    TYPE_TASK_BOX,
						    "x", 10.0, 
						    "y", 10.0,
						    "task", task,
						    NULL));
	gtk_signal_connect (GTK_OBJECT (item),
			    "destroy",
			    item_destroyed,
			    network_canvas);
	
	id_map_insert_id (priv->tasks, task->taskId, item);

	d(g_print ("NC: inserting id %d\n", task->taskId));
	
	parent_item = id_map_lookup (priv->tasks, task->parentId);
	if (!parent_item) {
		d(g_print ("Could not find parent with id %d (for %d).\n", task->parentId, task->taskId));
		return;
	}

	task_box_add_child (TASK_BOX (parent_item), item, -1);
	
	nc_reflow (network_canvas);
}

void
network_canvas_remove_tasks (NetworkCanvas *network_canvas,
			     GSList        *tasks)
{
	NetworkCanvasPriv *priv;
	GSList            *node;
	GM_Id              id;
	NetworkItem       *item;
	
	g_return_if_fail (network_canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (network_canvas));
	
	d(puts(__FUNCTION__));

	priv = network_canvas->priv;
	
	for (node = tasks; node; node = node->next) {
 		id = GPOINTER_TO_INT (node->data);

 		item = id_map_lookup (priv->tasks, id);
		
 		if (item) {
 			id_map_remove (priv->tasks, id);
			task_box_remove_child (TASK_BOX (item->parent_item), item);
			gtk_object_destroy (GTK_OBJECT (item));
 		}
	}
	
	nc_reflow (network_canvas);
}

void
network_canvas_update_task (NetworkCanvas *network_canvas,
			    GM_Task       *task)
{
	NetworkCanvasPriv *priv;
	NetworkItem       *item;
	
	g_return_if_fail (network_canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (network_canvas));
	g_return_if_fail (task != NULL);

	d(puts(__FUNCTION__));

	priv = network_canvas->priv;
	
	item = id_map_lookup (priv->tasks, task->taskId);
	if (!item) {
		g_warning ("Couldn't find task to update");
		return;
	}
	
	network_item_update_task (item, task);
}

void
network_canvas_reparent_task (NetworkCanvas *network_canvas,
                              GM_Id          task_id,
                              GM_Id          new_parent_id)
{
	NetworkCanvasPriv *priv;
	NetworkItem       *item, *new_parent_item;

	d(puts (__FUNCTION__));
	
	g_return_if_fail (network_canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (network_canvas));

	priv = network_canvas->priv;

	item = id_map_lookup (priv->tasks, task_id);
	if (!item) {
		d(g_print ("Could not find item with id %d.\n", task_id));
		return;
	}
	
	new_parent_item = id_map_lookup (priv->tasks, new_parent_id);
	if (!new_parent_item) {
		d(g_print ("Could not find parent with id %d (for %d).\n", new_parent_id, task_id));
		return;
	}

	task_box_reparent (TASK_BOX (new_parent_item), TASK_BOX (item));
	
	nc_reflow (network_canvas);
}

void
network_canvas_link (NetworkCanvas *network_canvas,
		     GM_Dependency *dependency)
{
	NetworkCanvasPriv *priv;
	NetworkItem       *item, *predecessor;
	ArrowItem         *arrow;
		
	g_return_if_fail (network_canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (network_canvas));
	
	d(puts(__FUNCTION__));

	priv = network_canvas->priv;

	item = id_map_lookup (priv->tasks, dependency->taskId);
	predecessor = id_map_lookup (priv->tasks, dependency->predecessorId);

	g_return_if_fail (item != NULL);
	g_return_if_fail (predecessor != NULL);

	arrow = arrow_item_new (item, predecessor);
	id_map_insert_id (priv->arrows, dependency->depId, arrow);
	
	network_item_link (item, predecessor);

	nc_reflow (network_canvas);
}

void
network_canvas_unlink (NetworkCanvas *network_canvas,
                       GM_Dependency *dependency)
{
	NetworkCanvasPriv *priv;
	NetworkItem       *item, *predecessor;
	ArrowItem         *arrow;
		
	g_return_if_fail (network_canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (network_canvas));
	
	d(puts(__FUNCTION__));

	priv = network_canvas->priv;

	item = id_map_lookup (priv->tasks, dependency->taskId);
	predecessor = id_map_lookup (priv->tasks, dependency->predecessorId);
	arrow = id_map_lookup (priv->arrows, dependency->depId);
		
	g_return_if_fail (item != NULL);
	g_return_if_fail (predecessor != NULL);

	gtk_object_destroy (GTK_OBJECT (arrow));
	id_map_remove (priv->arrows, dependency->depId);

	network_item_unlink (item, predecessor);

	nc_reflow (network_canvas);
}

void
network_canvas_clear (NetworkCanvas *canvas)
{
	NetworkCanvasPriv *priv;
		
	g_return_if_fail (canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (canvas));
	
	d(puts(__FUNCTION__));

	priv = canvas->priv;

	if (priv->reflow_idle_id) {
		g_source_remove (priv->reflow_idle_id);
		priv->reflow_idle_id = 0;
	}

	gtk_object_destroy (GTK_OBJECT (priv->root_box));

	priv->root_box = TASK_BOX (
		gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (canvas)),
				       TYPE_TASK_BOX,
				       "x", 0.0,
				       "y", 0.0,
				       NULL));

	id_map_remove (priv->tasks, 0);
	id_map_insert_id (priv->tasks, 0, priv->root_box);
}

static gboolean
nc_reflow_idle (NetworkCanvas *network_canvas)
{
	GnomeCanvas   *canvas;
	GtkAllocation *allocation;
	gdouble        sx2, sy2;
	gdouble        width, height;


	d(puts (__FUNCTION__));
	TRACE ();
 
	canvas = GNOME_CANVAS (network_canvas);
	
	allocation = &GTK_WIDGET (canvas)->allocation;
	width = allocation->width;
	height = allocation->height;
	
	gnome_canvas_item_get_bounds (
		GNOME_CANVAS_ITEM (gnome_canvas_root (canvas)),
		NULL, NULL, &sx2, &sy2);

	width = MAX ((int) width - 1, sx2 + PAD);
	height = MAX ((int) height - 1, sy2 + PAD);

	gnome_canvas_set_scroll_region (canvas,
					0,
					0,
					width,
					height);

	network_canvas->priv->reflow_idle_id = 0;
	
	return FALSE;
}

static void
nc_reflow (NetworkCanvas *canvas)
{
	g_return_if_fail (canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (canvas));
	
	if (!canvas->priv->reflow_idle_id) {
		canvas->priv->reflow_idle_id =
			g_idle_add ((GSourceFunc) nc_reflow_idle, canvas);
	}
}	

void
network_canvas_sort (NetworkCanvas *canvas, GM_IdSeq *ids)
{
	NetworkCanvasPriv *priv;
	GSList            *list, *l;
	GList             *queue, *q;
	GM_Id              id;
	NetworkItem       *item;
				
	g_return_if_fail (canvas != NULL);
	g_return_if_fail (IS_NETWORK_CANVAS (canvas));
	g_return_if_fail (ids != NULL);

	priv = canvas->priv;

	queue = NULL;
	list = corba_util_id_seq_to_list (ids);
	for (l = list; l; l = l->next) {
		id = GPOINTER_TO_INT (l->data);

		item = id_map_lookup (priv->tasks, id);
		if (!item) {
			/* If we try to sort before the canvas is realized,
			 * we don't have any items.
			 */
			continue;
		}

		queue = g_list_prepend (queue, item);
	}

	g_slist_free (list);
	
	if (queue == NULL) {
		/* Don't bother. */
		return;
	}
	
	queue = g_list_reverse (queue);

	for (q = queue; q; q = q->next) {
		item = q->data;
		
		item->col = 0;
	}

	/* Start by sorting out which column each box needs to
	 * be put in (with regards to predecessors).
	 */
	for (q = queue; q; q = q->next) {
		item = q->data;

		network_item_sort_predecessors (item);
	}

	/* Then do the layout, start from the end of the list,
	 * i.e the youngest children and the latest successors.
	 */
	queue = g_list_reverse (queue);
	for (q = queue; q; q = q->next) {
		item = q->data;

		network_item_layout (item);
	}

	/* Finally layout the root box. */
	network_item_layout (NETWORK_ITEM (priv->root_box));

	nc_reflow (canvas);

	g_list_free (queue);
}

