/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <bonobo/bonobo-ui-component.h>
#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-main.h>
#include <bonobo/bonobo-context.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>

#include "util/corba-utils.h"
#include "util/file-sel.h"
#include "client/widgets/mr-message-box.h"
#include "libmrproject/GNOME_MrProject.h"
#include "python-component.h"
#include "python.h"

#define OAFIID "OAFIID:GNOME_MrProject_PythonComponent"
#define FACTORY_OAFIID "OAFIID:GNOME_MrProject_PythonComponentFactory"

static void python_component_init       (PythonComponent       *component);

static void python_component_class_init (PythonComponentClass  *klass);

static void open_script_cb              (BonoboUIComponent     *component,
					 gpointer               data,
					 const char            *cname);

static void debug_console_cb            (BonoboUIComponent     *componeant,
					 gpointer               data,
					 const char            *cname);

#define PARENT_TYPE SHELL_COMPONENT_TYPE
static GtkObjectClass *parent_class;

static gint number_of_running_components = 0;

/* Private members. */
struct _PythonComponentPriv {
};

static BonoboUIVerb verbs[] = {
	BONOBO_UI_VERB ("FileOpenScript", open_script_cb),
	BONOBO_UI_VERB ("DebugConsole", debug_console_cb),	
	BONOBO_UI_VERB_END
};

static gboolean
python_component_add_to_shell (ShellComponent      *component, 
			       GM_Shell             shell)
{
	PythonComponent     *python_component;
	PythonComponentPriv *priv;
	CORBA_Environment    ev;
	Bonobo_UIContainer   ui_container;
	BonoboUIComponent   *ui_component;
	
	g_return_val_if_fail (component != NULL, FALSE);
	g_return_val_if_fail (IS_PYTHON_COMPONENT (component), FALSE);

	python_component = PYTHON_COMPONENT (component);
	priv             = python_component->priv;

	if (!init_python_script (shell)) {
		g_print ("Could not init python\n");
		/* FIXME: clean up? */
		return FALSE;
	}

	CORBA_exception_init (&ev);
	
	ui_container = GNOME_MrProject_Shell_getUIContainer (shell, &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Python: could not get ui container.");
		CORBA_exception_free (&ev);

		return FALSE;
	}

	CORBA_exception_free (&ev);

	ui_component = bonobo_ui_component_new ("MrProjectPythonComponent");
	bonobo_ui_component_set_container (ui_component, ui_container);
	bonobo_ui_component_add_verb_list_with_data (ui_component,
						     verbs,
						     component);

	bonobo_ui_util_set_ui (ui_component, GNOME_DATADIR,
			       "GNOME_MrProject_PythonComponent.ui",
			       "mrproject");

	return TRUE;
}

static void
python_component_remove_from_shell (ShellComponent *component)
{
	PythonComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_PYTHON_COMPONENT (component));

	priv = PYTHON_COMPONENT (component)->priv;

	g_print ("remove python from shell\n");
	
	gtk_main_quit ();
}

static void
python_component_destroy (GtkObject *object)
{
	PythonComponent     *component;
	PythonComponentPriv *priv;

	component = PYTHON_COMPONENT (object);
	priv      = component->priv;

	puts (__FUNCTION__);
	
	g_free (priv);

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);

	number_of_running_components--;
	if (number_of_running_components < 1) {
		g_print ("Hey, done with this.\n");
	}
}

static void
python_component_class_init (PythonComponentClass *klass)
{
	GtkObjectClass      *object_class;
	ShellComponentClass *shell_component_class;

	object_class                             = (GtkObjectClass *) klass;
	shell_component_class                    = SHELL_COMPONENT_CLASS (klass);
	parent_class                             = gtk_type_class (PARENT_TYPE);

	object_class->destroy                    = python_component_destroy;

	shell_component_class->add_to_shell      = python_component_add_to_shell;
	shell_component_class->remove_from_shell = python_component_remove_from_shell;
}

static void
python_component_init (PythonComponent *component)
{
	PythonComponentPriv *priv;
	
	priv = g_new0 (PythonComponentPriv, 1);
	
	component->priv = priv;

	number_of_running_components++;
	g_print ("Got %d python components.\n", number_of_running_components);
}

static BonoboObject *
python_component_factory (BonoboGenericFactory   *this,
			  const char             *object_id,
			  void                   *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, OAFIID)) {
		return BONOBO_OBJECT (gtk_type_new (PYTHON_COMPONENT_TYPE));
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

static void
open_script_cb (BonoboUIComponent *component,
		gpointer           data,
		const char        *cname)
{
	GM_Project         project_co;
	CORBA_Environment  ev;
	ShellComponent    *shell_component;
	gchar             *filename;

	shell_component = SHELL_COMPONENT (data);

	CORBA_exception_init (&ev);
		
	project_co = GNOME_MrProject_Shell_getProject (shell_component->shell, &ev);
	if (BONOBO_EX (&ev)) {
		g_log_exception (&ev, "");
		CORBA_exception_free (&ev);
		return;
	}

	filename = file_sel_run ("Open Script");
	if (filename) {
		open_python_script (filename);
		g_free (filename);
	}
}

static void
debug_console_cb (BonoboUIComponent *component,
		  gpointer           data,
		  const char        *cname)
{
	launch_python_shell ();
}


BONOBO_X_TYPE_FUNC (PythonComponent, 
		    PARENT_TYPE,
		    python_component);

BONOBO_OAF_FACTORY_MULTI (FACTORY_OAFIID,
			  "Mr Project python component factory",
			  VERSION,
			  python_component_factory,
			  NULL);
