/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmarshal.h>
#include <stdlib.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "resource-table-model.h"

static void     resource_tm_init             (ResourceTableModel      *model);
static void     resource_tm_class_init       (ResourceTableModelClass *klass);
static void     resource_tm_destroy          (GtkObject               *object);

static gint     resource_tm_col_count        (ETableModel             *etm);
static gint     resource_tm_row_count        (ETableModel             *etm);
static void *   resource_tm_value_at         (ETableModel             *etm,
					      gint                     col,
					      gint                     row);

static void     resource_tm_set_value_at     (ETableModel             *etm,
					      gint                     col,
					      gint                     row,
					      const void              *value);

static gboolean resource_tm_is_cell_editable (ETableModel             *etm,
					      gint                     col,
					      gint                     row);

static void     resource_tm_free_value       (ETableModel             *etm,
					      gint                     col,
					      void                    *value);
static gboolean resource_tm_value_is_empty   (ETableModel             *etm,
					      gint                     col,
					      const void              *value);
static gchar *  resource_tm_value_to_string  (ETableModel             *etm,
					      gint                     col,
					      const void              *value);

static GM_ResourceType get_type_value (gchar *type);
static GM_Id    resource_tm_get_group_id     (ResourceTableModel      *rtm,
					      gchar                   *group_name);

/* -- Private members -- */
struct _ResourceTableModelPriv {
	GArray *resources;

	gint    nr_of_cols;
	gint    nr_of_rows;
	
	GList  *type_strings;

	IdMap   *groups;
};

/* -- Signals -- */
enum {
	RESOURCE_CHANGED,
	LAST_SIGNAL
};

static guint signals [LAST_SIGNAL] = { 0 };

#define NR_OF_TYPES 3
static gchar *resource_type_names[] = {
	N_("Not initialized"),
	N_("Work"),
	N_("Material")
};

static gchar *no_group_string = N_("No group");

GNOME_CLASS_BOILERPLATE (ResourceTableModel, resource_tm, 
			 ETableModel, e_table_model);

static void 
resource_tm_init (ResourceTableModel *rtm)
{
	ResourceTableModelPriv *priv;
	int                     i;
	
	priv               = g_new0 (ResourceTableModelPriv, 1);
	rtm->priv          = priv;
	priv->resources    = g_array_new (FALSE, FALSE, sizeof (GM_Resource*));
	priv->nr_of_cols   = LAST_RT_COL;
	priv->nr_of_rows   = 0;
	priv->type_strings = NULL;
	priv->groups       = id_map_new (0);

	for (i = NR_OF_TYPES - 1; i >= 1; --i) {
		priv->type_strings = 
			g_list_prepend (priv->type_strings,
					_(resource_type_names[i]));
		
	}
}

static void 
resource_tm_class_init (ResourceTableModelClass *klass)
{
	GtkObjectClass   *object_class;
	ETableModelClass *model_class;
	
	object_class = (GtkObjectClass *) klass;
	model_class  = E_TABLE_MODEL_CLASS (klass);
	parent_class = gtk_type_class (E_TABLE_MODEL_TYPE);
	
	/* -- GtkObject functions   -- */
	object_class->destroy         = resource_tm_destroy;
	
	/* -- ETableModel functions -- */
	model_class->column_count     = resource_tm_col_count;
 	model_class->row_count        = resource_tm_row_count; 
	model_class->value_at         = resource_tm_value_at;
	model_class->set_value_at     = resource_tm_set_value_at;
 	model_class->is_cell_editable = resource_tm_is_cell_editable; 
	model_class->free_value       = resource_tm_free_value;
	model_class->value_is_empty   = resource_tm_value_is_empty;
	model_class->value_to_string  = resource_tm_value_to_string;

	/* -- Signals -- */
	signals[RESOURCE_CHANGED] =
		gtk_signal_new ("resource_changed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1,
				GTK_TYPE_POINTER);
	
	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void 
resource_tm_destroy (GtkObject *object)
{
}


static gint     
resource_tm_col_count (ETableModel *etm)
{
	g_return_val_if_fail (etm != NULL, -1);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (etm), -1);
	
	return RESOURCE_TABLE_MODEL(etm)->priv->nr_of_cols;
}

static gint     
resource_tm_row_count (ETableModel *etm)
{
	g_return_val_if_fail (etm != NULL, -1);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (etm), -1);
	
	return RESOURCE_TABLE_MODEL(etm)->priv->nr_of_rows;
}

static void *   
resource_tm_value_at (ETableModel *etm, gint col, gint row)
{
	ResourceTableModel     *rtm;
	ResourceTableModelPriv *priv;
	GM_Resource            *resource;
        GM_ResourceGroup       *group;
	
	g_return_val_if_fail (etm != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (etm), NULL);
	
	rtm  = RESOURCE_TABLE_MODEL (etm);
	priv = rtm->priv;
        
	if (row >= priv->nr_of_rows){
		g_warning ("get_value_at: Requested row too large: %d\n", row);
		return NULL;
	}
	
	resource = g_array_index (priv->resources, GM_Resource *, row);

	if (!resource) {
		g_warning ("get_value_at: No ResourceRow at %d\n", row);
		return NULL;
	}

	switch (col) {
	case RT_COL_ID:
  		return GINT_TO_POINTER (resource->resourceId);
		break;
	case RT_COL_NAME:
		return CORBA_string_dup (resource->name);
		break;
	case RT_COL_TYPE:
		return g_strdup (_(resource_type_names[resource->type]));
		break;
	case RT_COL_GROUP:
                group = id_map_lookup (priv->groups, resource->groupId);

                if (group) {
                        return CORBA_string_dup (group->name);
                }

                return g_strdup (_(no_group_string));
		break;
	case RT_COL_UNITS:
		return GINT_TO_POINTER (resource->units);
		break;
	case RT_COL_EMAIL:
		return CORBA_string_dup (resource->email);
		break;
	case RT_COL_RATE_STD:
		return &(resource->stdRate);
		break;

	case RT_COL_RATE_OVT:
		return &(resource->ovtRate);
		break;
	default:
		return NULL;
	}

	return NULL;
}

static void     
resource_tm_set_value_at (ETableModel *etm, 
			  gint         col, 
			  gint         row, 
			  const void  *value)
{
	ResourceTableModel *rtm;
	GM_Resource        *resource;
	GM_Resource        *copy;
	
	g_return_if_fail (etm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (etm));
	g_return_if_fail (value != NULL);
	
	rtm = RESOURCE_TABLE_MODEL (etm);

	if (row >= rtm->priv->nr_of_rows) {
		g_warning ("set_value_at: No ResourceRow for rtm node");
		return;
	}

	resource = g_array_index (rtm->priv->resources, 
				  GM_Resource *,
				  row);

	if (!resource) {
		g_warning ("set_value_at: No ResourceRow for rtm node");
		return;
	}
	
	/* Used for sending to listeners */
	copy = corba_util_resource_duplicate (resource);
	
	switch (col) {
	case RT_COL_ID:
		/* This can't be changed */
		break;
	case RT_COL_NAME:
		CORBA_free (copy->name);
		copy->name    = CORBA_string_dup ((gchar *) value);
		break;
	case RT_COL_TYPE:
 		copy->type    = get_type_value ((gchar *) value);
		break;
	case RT_COL_GROUP:
		copy->groupId = resource_tm_get_group_id (rtm, 
							  (gchar *) value);
		break;
	case RT_COL_UNITS:
  		copy->units   = atoi ((gchar *) value);
		break;
	case RT_COL_EMAIL:
		g_print ("New email: %s\n", (gchar *) value);
		
		if (copy->email) {
			CORBA_free (copy->email);
		}
		copy->email   = CORBA_string_dup ((gchar *) value);
	case RT_COL_RATE_STD:
 		copy->stdRate = atof ((gchar *) value);
		break;
	case RT_COL_RATE_OVT:
 		copy->ovtRate = atof ((gchar *) value);
		break;
	default:
		return;
	}

	/* Notify the resource component */
	gtk_signal_emit (GTK_OBJECT (rtm), 
			 signals[RESOURCE_CHANGED],
			 copy);

	CORBA_free (copy);
}

static gboolean 
resource_tm_is_cell_editable (ETableModel *etm, gint col, gint row)
{
	ResourceTableModel *rtm;

	g_return_val_if_fail (etm != NULL, FALSE);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (etm), FALSE);
	
	rtm = RESOURCE_TABLE_MODEL (etm);

 	if (col == RT_COL_ID) { 
 		return FALSE; 
 	} 
	
	return TRUE;
}

static void     
resource_tm_free_value (ETableModel *etm, gint col, void *value)
{
	ResourceTableModel *rtm;

	g_return_if_fail (etm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (etm));
	
	rtm = RESOURCE_TABLE_MODEL (etm);
}

static gboolean 
resource_tm_value_is_empty (ETableModel *etm, gint col, const void *value)
{
	return TRUE;
}

static gchar *
resource_tm_value_to_string (ETableModel *etm, gint col, const void *value)
{
	return g_strdup ("");
}

ETableModel *
resource_tm_new (void)
{
	ResourceTableModel *rtm;
	
	rtm = gtk_type_new (resource_tm_get_type ());

	return E_TABLE_MODEL (rtm);
}

guint
resource_tm_add_resource  (ResourceTableModel *rtm, GM_Resource *res)
{
	ResourceTableModelPriv *priv;
	GM_Resource            *resource;
	
	g_return_val_if_fail (rtm != NULL, 0);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (rtm), 0);
	
	priv     = rtm->priv;
	resource = corba_util_resource_duplicate(res);

 	g_array_append_val (priv->resources, resource); 

	priv->nr_of_rows++;

	e_table_model_row_inserted (E_TABLE_MODEL (rtm), priv->nr_of_rows - 1);
  	e_table_model_changed (E_TABLE_MODEL (rtm));  

	return priv->nr_of_rows - 1;
}

void
resource_tm_remove_all_resources (ResourceTableModel *rtm)
{
	gint        i;
	GM_Resource *resource;

	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (rtm));
	
	/* we remove from the root until we are done */
	for (i = 0; i < rtm->priv->nr_of_rows; i++) {
		resource = g_array_index (rtm->priv->resources,
					  GM_Resource *,
					  0);

		if (resource) {
			CORBA_free (resource);
		}

		rtm->priv->resources = g_array_remove_index (rtm->priv->resources, 0); /* remove them all */
		e_table_model_row_deleted (E_TABLE_MODEL (rtm), rtm->priv->nr_of_rows - i - 1);
	}
	rtm->priv->nr_of_rows = 0;
	e_table_model_changed (E_TABLE_MODEL (rtm));
}


void
resource_tm_remove_resource (ResourceTableModel *rtm, gint id)
{
	GM_Resource *resource;
	gint         row;
	
	g_return_if_fail (rtm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (rtm));
	
	row      = resource_tm_get_row (rtm, id);
	resource = g_array_index (rtm->priv->resources, 
				  GM_Resource *,
				  row);
	
	if (!resource) {
		return;
	}

	rtm->priv->resources = g_array_remove_index (rtm->priv->resources, 
						     row);

	CORBA_free (resource);

	rtm->priv->nr_of_rows--;

	e_table_model_row_deleted (E_TABLE_MODEL (rtm), rtm->priv->nr_of_rows);
	e_table_model_changed (E_TABLE_MODEL (rtm));
}

void
resource_tm_add_group (ResourceTableModel *rtm,
		       GM_ResourceGroup   *g)
{
	ResourceTableModelPriv *priv;
	GM_ResourceGroup       *group;
	
	g_return_if_fail (rtm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (rtm));
	g_return_if_fail (g != NULL);
	
	priv  = rtm->priv;
	group = corba_util_resource_group_duplicate (g);

	id_map_insert_id (priv->groups, group->groupId, group);
}

void
resource_tm_update_group (ResourceTableModel *rtm, GM_ResourceGroup *group)
{
	ResourceTableModelPriv *priv;
	GM_ResourceGroup       *original;
	
	g_return_if_fail (rtm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (rtm));
	g_return_if_fail (group != NULL);
	
	priv     = rtm->priv;
	original = id_map_lookup (priv->groups, group->groupId);
	
	if (!original) {
		g_warning ("Trying to update a nonexisting group");
		return;
	}

	corba_util_resource_group_update (original, group);
}

void
resource_tm_remove_group (ResourceTableModel *rtm, gint gid)
{
	ResourceTableModelPriv *priv;
	GM_ResourceGroup       *group;
	
	g_return_if_fail (rtm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (rtm));
	
	priv  = rtm->priv;
	group = id_map_lookup (priv->groups, gid);
	
	if (!group) {
		g_warning ("Trying to delete nonexisting group");
		return;
	}
	
	id_map_remove (priv->groups, gid);
	
	CORBA_free (group);
}

void
resource_tm_remove_all_groups (ResourceTableModel *rtm)
{
	GSList           *group_list;
	GSList           *node;
	GM_ResourceGroup *group;

	group_list = id_map_get_objects (rtm->priv->groups);

	for (node = group_list; node; node = node->next) {
		g_assert (node->data != NULL);

		group = (GM_ResourceGroup *)node->data;

		id_map_remove (rtm->priv->groups, group->groupId);

		CORBA_free (group);
	}
	g_slist_free (group_list);
}

GM_Resource *
resource_tm_get_resource_on_row (ResourceTableModel *rtm, gint row)
{
	GM_Resource *resource;
	
	g_return_val_if_fail (rtm != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (rtm), NULL);
	
	resource = g_array_index (rtm->priv->resources, GM_Resource *, row);
	
	if (!resource) {
		g_warning ("No ResourceRow for rtm node.");
		return NULL;
	}
	
	return resource;
}

void
resource_tm_update_resource (ResourceTableModel *rtm, 
			     GM_Resource        *upd_resource)
{
	ResourceTableModelPriv *priv;
	GM_Resource            *resource;
	gint                    row;
	gboolean                changed;
	
	g_return_if_fail (rtm != NULL);
	g_return_if_fail (IS_RESOURCE_TABLE_MODEL (rtm));

	priv = rtm->priv;
	row  = resource_tm_get_row (rtm, upd_resource->resourceId);
	
	if (row == -1) {
		g_warning ("Trying to update non-existing row");
		return;
	}

	resource = g_array_index (priv->resources, GM_Resource *, row);
	changed  = corba_util_resource_update (resource, upd_resource);

	if (changed) {
		e_table_model_row_changed (E_TABLE_MODEL (rtm), row); 
	}
}

gint
resource_tm_get_row   (ResourceTableModel *rtm, gint id)
{
	ResourceTableModelPriv *priv;
	GM_Resource            *resource;
	gint                    i;
	
	g_return_val_if_fail (rtm != NULL, -1);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (rtm), -1);
	
	priv = rtm->priv;
	
	for (i = 0; i < priv->nr_of_rows; ++i) {
		resource = g_array_index (priv->resources, GM_Resource *, i);

		if (resource->resourceId == id) {
			return i;
		}
	}

	return -1;
}

GSList *
resource_tm_get_resources_with_group (ResourceTableModel *rtm, GM_Id gid)
{
	ResourceTableModelPriv *priv;
	GM_Resource            *resource;
	GSList                 *list = NULL;
	gint                    index, len;
	
	g_return_val_if_fail (rtm != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_TABLE_MODEL (rtm), NULL);
	
	priv = rtm->priv;
	len  = priv->resources->len;

	for (index = 0; index < len; ++index) {
		resource = g_array_index (priv->resources,
					  GM_Resource *,
					  index);
		
		if (resource->groupId == gid) {
			list = g_slist_prepend (list, resource);
		}
	}
	
	return list;
}

static GM_ResourceType 
get_type_value (gchar *type)
{
	gint i;
	
	for (i = 0; i < NR_OF_TYPES; ++i) {
		if (!strcmp (_(resource_type_names[i]), type)) {
			return i;
		}
	}
	
	return 0;
}

static GM_Id
resource_tm_get_group_id (ResourceTableModel *rtm, gchar *group_name)
{
	GSList           *groups, *tmp_list;
	GM_ResourceGroup *group;
	GM_Id             id = 0;
	
	groups = id_map_get_objects (rtm->priv->groups);

	for (tmp_list = groups; tmp_list; tmp_list = tmp_list->next) {
		group = (GM_ResourceGroup *) tmp_list->data;

		if (!strcmp (group->name, group_name)) {
			id = group->groupId;
			break;
		}
	}
	
	g_slist_free (groups);

	return id;
}

GList *
resource_tm_get_type_strings (ResourceTableModel   *rtm)
{
	return rtm->priv->type_strings;
}

static void
add_group_name_to_list (gpointer key, gpointer value, gpointer user_data)
{
        GM_ResourceGroup *group;
        GList            **list;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (value != NULL);

        group = (GM_ResourceGroup *) value;
        list  = (GList **) user_data;
	
        *list = g_list_prepend (*list, CORBA_string_dup (group->name));
}

static gint
string_compare (gconstpointer a, gconstpointer b)
{
	return strcmp ((gchar *) a, (gchar *) b);
}

GList *
resource_tm_get_group_strings (ResourceTableModel *rtm)
{
        GList  *list;
	
	list = g_list_prepend (NULL, g_strdup (_(no_group_string)));

        id_map_foreach (rtm->priv->groups,
			add_group_name_to_list,
			&list);

	list = g_list_sort (list, string_compare);
	
        return list;
}


