/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Anders Carlsson <andersca@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Anders Carlsson
 */

/* Maybe we want to use the design described here: http://www.iarchitect.com/images/mewredo.gif
 * (see http://www.iarchitect.com/tabs.htm for full article)? Components could add their own category
 * to the list and a control for the action area.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "preferences-dialog.h"

#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <gdk/gdkkeysyms.h> /* for GDK_Escape */
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#define MRPROJECT_ROOT_KEY "/apps/mrproject"

static void
preferences_close_clicked (GtkWidget *button, GtkWidget **dialog)
{
	gtk_widget_destroy (*dialog);

	*dialog = NULL;
}

static gboolean
preferences_key_event (GtkWidget   *widget,
		     GdkEventKey *event,
		     GtkWidget   *close_button)
{
	if (event->keyval == GDK_Escape) {
		gtk_button_clicked (GTK_BUTTON (close_button));
		return TRUE;
	} else {
		return FALSE;
	}
}

static void
preferences_check_button_key_changed (GConfClient* client,
				      guint cnxn_id,
				      GConfEntry *entry,
				      gpointer user_data)
{
	GtkWidget *button;

	button = user_data;

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), gconf_value_get_bool (entry->value));
}

static void
preferences_check_button_toggled (GtkWidget *button, const gchar *key)
{
	gchar *full_key;
	GConfClient *client;

	client = gconf_client_get_default ();

	full_key = g_strdup_printf ("%s/%s", MRPROJECT_ROOT_KEY, key);
	gconf_client_set_bool (client, full_key, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)), NULL);

	g_free (full_key);
}

static void
preferences_widget_destroy (GtkWidget *button, guint connection)
{
	GConfClient *client;

	client = gconf_client_get_default ();

	gconf_client_notify_remove (client, connection);
}

static void
preferences_initialize_check_button (GtkWidget *button, const gchar *key)
{
	GConfClient *client;
	gchar *full_key;
	guint notify;
	
	g_return_if_fail (button != NULL);

	client = gconf_client_get_default ();

	/* First, set our initial value */
	full_key = g_strdup_printf ("%s/%s", MRPROJECT_ROOT_KEY, key);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), gconf_client_get_bool (client, full_key, NULL));

	/* Then, connect for changes in the key */
	notify = gconf_client_notify_add (client, full_key,
					  preferences_check_button_key_changed,
					  button,
					  NULL, NULL);
	
	/* Last, connect for changes in the toggle button */
	gtk_signal_connect (GTK_OBJECT (button), "toggled",
			    GTK_SIGNAL_FUNC (preferences_check_button_toggled), (gchar *)key);
	gtk_signal_connect (GTK_OBJECT (button), "destroy",
			    GTK_SIGNAL_FUNC (preferences_widget_destroy), GINT_TO_POINTER (notify));
	
	g_free (full_key);

}

static void
preferences_entry_changed (GtkWidget *entry, const gchar *key)
{
	GConfClient *client;
	gchar *full_key;

	client = gconf_client_get_default ();

	full_key = g_strdup_printf ("%s/%s", MRPROJECT_ROOT_KEY, key);
	gconf_client_set_string (client, full_key, gtk_entry_get_text (GTK_ENTRY (entry)), NULL);

	g_free (full_key);

}

static void
preferences_entry_key_changed (GConfClient* client,
			       guint cnxn_id,
			       GConfEntry *entry,
			       gpointer user_data)
{
	gchar *key;

	key = gtk_object_get_data (GTK_OBJECT (user_data), "key");
	
	if (gconf_value_get_string (entry->value) != NULL) {
		gtk_signal_handler_block_by_data (GTK_OBJECT (user_data), key);
		gtk_entry_set_text (GTK_ENTRY (user_data), gconf_value_get_string (entry->value));
		gtk_signal_handler_unblock_by_data (GTK_OBJECT (user_data), key);
	}		
}

static void
preferences_initialize_entry (GtkWidget *entry, const gchar *key)
{
	GConfClient *client;
	gchar *full_key;
	gchar *str;
	guint notify = 0;
	
	g_return_if_fail (entry != NULL);

	client = gconf_client_get_default ();

	/* First, set our initial value */
	full_key = g_strdup_printf ("%s/%s", MRPROJECT_ROOT_KEY, key);
	str = gconf_client_get_string (client, full_key, NULL);

	if (str != NULL)
		gtk_entry_set_text (GTK_ENTRY (entry), str);
	
	g_free (str);

	/* Then, connect for changes in the key */
	notify = gconf_client_notify_add (client, full_key,
					  preferences_entry_key_changed,
					  entry,
					  NULL, NULL);

	/* Last, connect for changes in the toggle button */
	gtk_object_set_data (GTK_OBJECT (entry), "key", (gchar *)key);
	gtk_signal_connect (GTK_OBJECT (entry), "changed",
			    GTK_SIGNAL_FUNC (preferences_entry_changed), (gchar *)key);
	gtk_signal_connect (GTK_OBJECT (entry), "destroy",
			    GTK_SIGNAL_FUNC (preferences_widget_destroy), GINT_TO_POINTER (notify));
	
	g_free (full_key);
}

void
preferences_dialog_run (GtkWindow *project_window)
{
	GladeXML *glade;
	GtkWidget *button;
	GConfClient *client;

	static GtkWidget *dialog = NULL;

	if (dialog != NULL) {
		gdk_window_raise (dialog->window);
		return;
	}
	
	glade = glade_xml_new (MRPROJECT_DATADIR
			       "preferences-dialog.glade",
			       NULL);
	
	if (glade == NULL) {
		g_warning (_("Could not create properties dialog."));
		return;
	}
	
	
	dialog = glade_xml_get_widget (glade, "toplevel");
	button = glade_xml_get_widget (glade, "close_button");

	gtk_signal_connect (GTK_OBJECT (dialog), "key_press_event",
			    GTK_SIGNAL_FUNC (preferences_key_event), button);

	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    preferences_close_clicked,
			    &dialog);

	/* Set up GConf callbacks */
	client = gconf_client_get_default ();
	gconf_client_add_dir(client,
			     MRPROJECT_ROOT_KEY,
			     GCONF_CLIENT_PRELOAD_NONE,
			     NULL);


	preferences_initialize_check_button (glade_xml_get_widget (glade, "simple_check_button"), "simple_setting");
	preferences_initialize_entry (glade_xml_get_widget (glade, "simple_entry"), "simple_entry_setting");
	
	gtk_widget_show_all (dialog);
	gtk_object_unref (GTK_OBJECT (glade));
}
