/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <bonobo/bonobo-ui-util.h>
#include <glade/glade.h>
#include "util/corba-utils.h"

#include "project-properties-dialog.h"

typedef struct {
	GtkWindow          *project_window;
	GM_Project          project_co;
	Bonobo_PropertyBag  pb_co;
	Bonobo_EventSource  es_co;
	BonoboListener     *listener;
	gint                listener_id;

	GtkWidget          *name_entry;
	GtkWidget          *company_entry;
	GtkWidget          *manager_entry;
} ProjectPropertiesDialog;

static gboolean setup_corba_glue      (ProjectPropertiesDialog *ppd,
				       GM_Shell                 shell_co);
static void cleanup_corba_glue        (ProjectPropertiesDialog *ppd);
static void disconnect_corba_listener (ProjectPropertiesDialog *ppd);



void
project_properties_dialog_run (GtkWindow *project_window,
			       GM_Shell   shell_co)
{
	GladeXML                *glade;
	GtkWidget               *dialog;
	ProjectPropertiesDialog *ppd;
	time_t                   start, finish;
	gchar                   *name, *company, *manager;
	gchar                   *new_name, *new_company, *new_manager;
	gint                     ret;

	glade = glade_xml_new (MRPROJECT_DATADIR
			       "project-properties-dialog.glade",
			       NULL);
	if (!glade) {
		g_warning (_("Could not create properties dialog."));
		return;
	}

	dialog = glade_xml_get_widget (glade, "toplevel");
	gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);

	/*last_date_edit = glade_xml_get_widget (gui, "dateedit_finish");*/

	ppd = g_new0 (ProjectPropertiesDialog, 1);
	ppd->project_window = project_window;
	
	ppd->name_entry = glade_xml_get_widget (glade, "entry_name");
	ppd->company_entry = glade_xml_get_widget (glade, "entry_company");
	ppd->manager_entry = glade_xml_get_widget (glade, "entry_manager");

	setup_corba_glue (ppd, shell_co);
		 
	name = bonobo_property_bag_client_get_value_string (
		ppd->pb_co, "Name", NULL);

	company = bonobo_property_bag_client_get_value_string (
		ppd->pb_co, "Company", NULL);
	
	manager = bonobo_property_bag_client_get_value_string (
		ppd->pb_co, "Manager", NULL);

	start = bonobo_property_bag_client_get_value_glong (
		ppd->pb_co, "Start", NULL);

	finish = bonobo_property_bag_client_get_value_glong (
		ppd->pb_co, "Finish", NULL);

	gtk_entry_set_text (GTK_ENTRY (ppd->name_entry), name);
	gtk_entry_set_text (GTK_ENTRY (ppd->company_entry), company);
	gtk_entry_set_text (GTK_ENTRY (ppd->manager_entry), manager);
	
	ret = gnome_dialog_run (GNOME_DIALOG (dialog));
	/* We don't want to get event from here... */
	disconnect_corba_listener (ppd);
	if (ret == GNOME_OK) {
		new_name = gtk_editable_get_chars (
			GTK_EDITABLE (ppd->name_entry), 0, -1);
		new_company = gtk_editable_get_chars (
			GTK_EDITABLE (ppd->company_entry), 0, -1);
		new_manager = gtk_editable_get_chars (
			GTK_EDITABLE (ppd->manager_entry), 0, -1);

		if (strcmp (name, new_name)) {
			bonobo_property_bag_client_set_value_string (
				ppd->pb_co, "Name", new_name, NULL);
		}

		if (strcmp (company, new_company)) {
		bonobo_property_bag_client_set_value_string (
			ppd->pb_co, "Company", new_company, NULL);
		}
		
		if (strcmp (manager, new_manager)) {
			bonobo_property_bag_client_set_value_string (
				ppd->pb_co, "Manager", new_manager, NULL);
		}
		
		g_free (name);
		g_free (company);
		g_free (manager);
		g_free (new_name);
		g_free (new_company);
		g_free (new_manager);
	}

	if (ret != -1) {
		gnome_dialog_close (GNOME_DIALOG (dialog));
	}

	gtk_object_unref (GTK_OBJECT (glade));
	cleanup_corba_glue (ppd);
}

#define IS_EVENT(event_name, str) (!strncmp (event_name, str, sizeof (str)-1))

static void
listener_cb (BonoboListener    *listener,
	     char              *event_name, 
	     CORBA_any         *any,
	     CORBA_Environment *ev,
	     gpointer           user_data)
{
	ProjectPropertiesDialog *ppd;
	gchar                   *subtype, *kind;
	gchar                   *str;

	ppd = user_data;

	subtype = bonobo_event_subtype (event_name);
	kind = bonobo_event_kind (event_name);

	if (IS_EVENT (event_name, "Bonobo/Property:")) {
		if (IS_EVENT (kind, "change")) {
			if (!strcmp (subtype, "Name")) {
				str = BONOBO_ARG_GET_STRING (any);
				gtk_entry_set_text (GTK_ENTRY (ppd->name_entry), str);
				//g_free (str);
			}
			else if (!strcmp (subtype, "Company")) {
				str = BONOBO_ARG_GET_STRING (any);
				gtk_entry_set_text (GTK_ENTRY (ppd->company_entry), str);
				//g_free (str);
			}
			else if (!strcmp (subtype, "Manager")) {
				str = BONOBO_ARG_GET_STRING (any);
				gtk_entry_set_text (GTK_ENTRY (ppd->manager_entry), str);
				//g_free (str);
			}
		}
	} else {
		g_print ("ProjectWindow: got unhandled event: %s\n", event_name);
	}

	g_free (subtype);
	g_free (kind);
}

static gboolean setup_corba_glue (ProjectPropertiesDialog *ppd,
				  GM_Shell                 shell_co)
{
	GM_Project         project_co;
	Bonobo_EventSource es_co;
	Bonobo_Listener    listener_co;
	Bonobo_PropertyBag pb_co;
	CORBA_Environment  ev;

	CORBA_exception_init (&ev);

	project_co = GNOME_MrProject_Shell_getProject (shell_co,
						       &ev);
	if (BONOBO_EX (&ev) || !project_co) {
		CORBA_exception_free (&ev);
		return FALSE;
	}

	pb_co = Bonobo_Unknown_queryInterface (
		project_co, "IDL:Bonobo/PropertyBag:1.0", &ev);
	if (BONOBO_EX (&ev) || !pb_co) {
		CORBA_exception_free (&ev);
		CORBA_Object_release (project_co, NULL);
		return FALSE;
	}
	
	es_co = GNOME_MrProject_Shell_getProxyEventSource (
		shell_co,
		&ev);
	if (BONOBO_EX (&ev) || !es_co) {
		CORBA_Object_release (project_co, NULL);
		CORBA_exception_free (&ev);
		return FALSE;
	}

	ppd->project_co = project_co;
	ppd->pb_co = pb_co;
	ppd->es_co = es_co;

	/* Listen to events from the shell. */
	ppd->listener = bonobo_listener_new (listener_cb, ppd);
	listener_co = BONOBO_OBJREF (ppd->listener);
	ppd->listener_id = Bonobo_EventSource_addListenerWithMask (
		es_co,
		listener_co,
		"=Bonobo/Property:change:Start,"	
		"=Bonobo/Property:change:Finish,"
		"=Bonobo/Property:change:Name,"
		"=Bonobo/Property:change:Manager,"
		"=Bonobo/Property:change:Company",
		&ev);
	if (BONOBO_EX (&ev)) {
		bonobo_object_release_unref (es_co, NULL);
		CORBA_exception_free (&ev);
		return FALSE;
	}

	CORBA_exception_free (&ev);

	return TRUE;
}

static void
disconnect_corba_listener (ProjectPropertiesDialog *ppd)
{
	if (ppd->listener_id != 0) {
		Bonobo_EventSource_removeListener (ppd->es_co, ppd->listener_id, NULL);
		ppd->listener_id = 0;
	}
	if (ppd->es_co) {
		bonobo_object_release_unref (ppd->es_co, NULL);
		ppd->es_co = NULL;
	}
}

static void
cleanup_corba_glue (ProjectPropertiesDialog *ppd)
{
	bonobo_object_release_unref (ppd->pb_co, NULL);
	disconnect_corba_listener (ppd);
}

