/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <gtk/gtkobject.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <bonobo/bonobo-xobject.h>
#include <bonobo/bonobo-exception.h>

#include "backends/file-project.h"
#include "engine.h"

#include "util/debug.h"

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass *parent_class;


static void engine_init       (Engine      *engine);
static void engine_class_init (EngineClass *klass);


/* Private members. */
struct _EnginePriv {
	GSList   *projects;
	gboolean  immortal;
};

static void
project_destroyed (Project *project, Engine *engine)
{
	EnginePriv *priv;

	priv = engine->priv;
	
	priv->projects = g_slist_remove (priv->projects, project);

	if (!priv->immortal && g_slist_length (priv->projects) == 0) {
		g_message ("Last project gone, bye.");
		gtk_main_quit ();
	}
}

Project *
engine_create_project (Engine *engine)
{
	Project *project;

	project = file_project_new ();

	gtk_signal_connect (GTK_OBJECT (project),
			    "destroy",
			    project_destroyed,
			    engine);

	engine->priv->projects = g_slist_append (engine->priv->projects, project);

	return project;
}

static GM_ProjectSeq *
impl_Engine_getProjects (PortableServer_Servant  servant,
			 CORBA_Environment      *ev)
{
	Engine        *engine;
	EnginePriv    *priv;
	guint          len, i;
	GSList        *list;
	GM_ProjectSeq *seq;

	engine = ENGINE (bonobo_x_object (servant));
	priv = engine->priv;

	len = g_slist_length (priv->projects);
	
	seq = GNOME_MrProject_ProjectSeq__alloc ();
	seq->_buffer = CORBA_sequence_GNOME_MrProject_Project_allocbuf (len);
	seq->_length = len;
	seq->_maximum = len;
	CORBA_sequence_set_release (seq, CORBA_TRUE);

	list = priv->projects;
	for (i = 0; i < len; i++) {
		GM_Project project;

		project = CORBA_Object_duplicate (BONOBO_OBJREF (list->data), NULL);
		seq->_buffer[i] = project;
		list = list->next;
	}

	return seq;
}

static GNOME_MrProject_Project
impl_Engine_openProject (PortableServer_Servant  servant,
			 const CORBA_char       *uri,
			 CORBA_Environment      *ev)
{
	Engine	*engine;
	Project *project;
	
	engine = ENGINE (bonobo_x_object (servant));

	project = engine_create_project (engine);

	GNOME_MrProject_Project_load (BONOBO_OBJREF (project),
				      uri,
				      ev);

	if (!BONOBO_EX (ev)) {
		return CORBA_Object_duplicate (BONOBO_OBJREF (project), ev);
	} else {
		return NULL;
	}
}

static GNOME_MrProject_Project
impl_Engine_createProject (PortableServer_Servant  servant,
			   CORBA_Environment      *ev)
{
	Engine	*engine;
	Project *project;

	engine = ENGINE (bonobo_x_object (servant));

	project = engine_create_project (engine);

	return CORBA_Object_duplicate (BONOBO_OBJREF (project), ev);
}

static void
impl_Engine_shutDown (PortableServer_Servant  servant,
		      CORBA_boolean           force,
		      CORBA_Environment      *ev)
{
	Engine     *engine;
	EnginePriv *priv;

	engine = ENGINE (bonobo_x_object (servant));
	priv = engine->priv;

	if (force) {
		g_message ("Forced shutdown.");
		gtk_main_quit ();
	}
	else if (!priv->immortal && g_slist_length (priv->projects) == 0) {
		g_message ("Shutdown.");
		gtk_main_quit ();
	}		
}

static void
impl_Engine__set_immortal (PortableServer_Servant  servant,
			   CORBA_boolean           immortal,
			   CORBA_Environment      *ev)
{
	Engine	   *engine;
	EnginePriv *priv;

	engine = ENGINE (bonobo_x_object (servant));
	priv = engine->priv;

	priv->immortal = immortal;

	if (!immortal && g_slist_length (priv->projects) == 0) {
		g_message ("Engine set to mortal and no projects.");
	}	
}

static CORBA_boolean 
impl_Engine__get_immortal (PortableServer_Servant  servant,
			   CORBA_Environment      *ev)
{
	Engine	*engine;

	engine = ENGINE (bonobo_x_object (servant));

	return engine->priv->immortal;
}

static void
engine_destroy (GtkObject *object)
{
	Engine     *engine;
	EnginePriv *priv;

	d(puts (__FUNCTION__));
	
	engine = ENGINE (object);
	priv = engine->priv;

	if (engine->priv) {
		g_free (engine->priv);
		engine->priv = NULL;
	}

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
engine_class_init (EngineClass *klass)
{
	POA_GNOME_MrProject_Engine__epv *epv = &klass->epv;
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class (PARENT_TYPE);

	object_class->destroy = engine_destroy;

	epv->getProjects   = impl_Engine_getProjects;
	epv->openProject   = impl_Engine_openProject;
	epv->createProject = impl_Engine_createProject;
	epv->shutDown      = impl_Engine_shutDown;
	epv->_get_immortal = impl_Engine__get_immortal;
	epv->_set_immortal = impl_Engine__set_immortal;
}

BONOBO_X_TYPE_FUNC_FULL (Engine, 
			 GNOME_MrProject_Engine,
			 PARENT_TYPE,
			 engine);


static void
engine_init (Engine *engine)
{
	EnginePriv *priv;

	g_print ("engine init\n");

	priv = g_new0 (EnginePriv, 1);
	engine->priv = priv;

	priv->immortal = FALSE;
}

Engine *
engine_new (void)
{
	return ENGINE (gtk_type_new (ENGINE_TYPE));
}
