#!/bin/sh
# the next line restarts using wish \
exec wish $0 -- $@
# Time-stamp: <2006-10-09 10:30:46 poser>
#
# Copyright (C) 2005, 2006 William J. Poser (billposer@alum.mit.edu)
# This program is free software; you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.#
# A copy of the GNU General Public License is contained in the
# procedure "License" in this file.
# If it is not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
# or go to the web page:  http://www.gnu.org/licenses/gpl.txt.

set Version "1.19";
set PackageVersion "8.29";
encoding system utf-8
set DebugP 0;				# General debugging
#If you are running this under MS Windows without using cygwin, I am told that
#wish, the Tcl/Tk windowing shell, will not work properly. Try using tclsh.exe
#as the Tcl interpreter and uncomment the following line to import Tk.
#package require Tk
package require msgcat
package require Iwidgets

set UseCheckbuttonP 1;

proc dmsg {msg} {
    if {$::DebugP} {
	puts $msg;
	flush stdout;
    }
}

# Portability stuff

set KeySelHeight 1;
set KeySelWidth 10;
set InitFile  ".msgrc";

#Figure out what system we are running on
if {[string equal $tcl_platform(platform) windows]} {
    set System MSWindows;
    dmsg "Running under MS Windows";
} elseif {[string equal $tcl_platform(platform) unix]} {
    if {[string equal $tcl_platform(os) Darwin]} {
	set System MacOSX;
	dmsg "Running under Mac OS X";
    } else {
	set System Unix;
	dmsg "Running under Unix";
    }
}

#Find out what our base graphics system is
if {[string match X11*  [winfo server .]]} {
    set AquaP 0
    set WindowSystem X11
} else {
    if {[string match $System MSWindows]} {
	set AquaP 0;
	set WindowSystem MSWindows;
    }
    if {[string match $System MacOSX]} {
	set AquaP 1
	set WindowSystem Aqua
    }
}

if {[info exists ::env(TMP)]} {
    set TempDir $::env(TMP);
} elseif {[info exists ::env(TEMP)]} {
    set TempDir $::env(TEMP);
} else {
    if {[string equal $System MSWindows]} {
	set TempDir "C:\tmp";
    } else {
	set TempDir "/tmp";
    }
}

switch $System {
    Unix {
	event add <<B3>> <ButtonPress-3>
	event add <<B3Release>> <ButtonRelease-3>
    }
    MacOSX {
	event add <<B3>> <Control-ButtonPress-1>
	event add <<B3Release>> <Control-ButtonRelease-1>
	if {$AquaP} {
	    set KeySelHeight 2;
	    set KeySelWidth 10;
	}
    }
    MSWindows {
	event add <<B3>> <ButtonPress-3>
	event add <<B3Release>> <ButtonRelease-3>
	set InitFile    "Msgrc"
    }
}

proc _ {s} {return [::msgcat::mc $s]};	# Define shorthand for gettext


set BalloonHelpP 1;
set HPWidth 64;				# Width of help popups.
set HPLines 10;				# Number of lines for help popups.
set xcwidth 5;
set genfxpad 540
set grelief solid
set gborder 1
set LastCommandLineSaveTime 0;		# Time (in sec. since epoch) of last command line save.
set LastExecutionTime 0;		# Time (in sec. since epoch) of last msort execution.	
set SortInProgressP 0;
set SortAbortedP 0;
set SubdHeight 6;
set ReadInitFileP 1;
set CustomChartList "";
set FixedRecordLength 0;
set RecordParseDisplayedP 0;
set MiscOptionsDisplayedP 0;
set UnicodeDisplayedP 0;
set AlgorithmDisplayedP 0;
set InputFileDisplayedP 0;
set OutputFileDisplayedP 0;
set ExclusionsDisplayedP 0;
set KeyFieldIdentificationDisplayedP 0;
set MiscellaneousOptionsDisplayedP 0;
set SortOrderDisplayedP 0;
set SortTypeDisplayedP 0;
set SubstitutionsDisplayedP 0;

set ColorSpecs(BalloonHelp,Background)		"\#FF8888"
set ColorSpecs(BalloonHelp,Foreground)		black
set ColorSpecs(Default,Background)		"light grey"
set ColorSpecs(Backdrop,Background)		black
set ColorSpecs(Menu,ActiveBackground)	        salmon;
set ColorSpecs(Menu,ActiveForeground)	        black
set ColorSpecs(Menu,Background)			azure3
set ColorSpecs(Menu,Foreground)			black
set ColorSpecs(Menu,Select)			coral
set ColorSpecs(Menubar,Background)	 	"\#c36176";
set ColorSpecs(Menubar,Foreground)		"\#fee4a9";
set ColorSpecs(Menubar,ActiveBackground)	"\#fee4a9";
set ColorSpecs(Menubar,ActiveForeground)	"\#c36176";
set ColorSpecs(Messages,Background) 		"\#e9c4dc";
set ColorSpecs(Messages,Foreground) 		"\#000000";
set ColorSpecs(IPAEntry,Background)		"\#FFFFFF";
set ColorSpecs(IPAEntry,Foreground)		"\#000000";
set ColorSpecs(IPAHeadings,Background)		"\#09ffb2";
set ColorSpecs(PageChoice,Background) 		"\#e6b483";
set ColorSpecs(PageChoice,ActiveBackground)	"\#5040FF";
set ColorSpecs(PageChoice,Selected) 	"\#F30086";
set ColorSpecs(PageChoice,ActiveForeground) 		"\#FFFFFF";
set ColorSpecs(PageChoice,UnselectedPageForeground) 		"\#000000";
set ColorSpecs(RadioButton,Background) "\#A0A0FF"
set ColorSpecs(RadioButton,SelectedBackground) green
set ColorSpecs(RadioButton,ActiveBackground) "\#FF8080"
set ColorSpecs(SortOrderDefinition,Background)			"\#FFFFFF";
set ColorSpecs(SortOrderSeparator,Background)			"\#FFFFFF";
set ColorSpecs(Substitutions,Background)		"\#FFFFFF";
set ColorSpecs(UserTextEntry,Background)	"\#FFFFFF";
set ColorSpecs(UserTextEntry,Foreground)	black
set ColorSpecs(Entry,Background)	"navajo white"
set ColorSpecs(Entry,DisabledBackground)		blue
set ColorSpecs(UserText,Foreground)		\#0000FF
set ColorSpecs(RollUpButton,ActiveBackground) NavajoWhite
set ColorSpecs(RollUpButton,ActiveForeground) red


# General defaults
set BMPOnlyP 0;
set GlobalInversionP 0;
set ReservePrivateUseAreasP 0;
set FieldSeparator  ""
set RecordSeparator "";
set KeyCount 0;
set MaxKeyCount $KeyCount;
set WhichKey 0;
set RecordParse 1;
set WholeRecordIsKeyP 0;
set EOLCarriageReturnP 0;
set WhichFontToSet MainFont

#This array maps key numbers to page names, e.g. 0-> .gen, 1->.ks1
array set KeyMap {};
array set InverseKeyMap {};
#This array maps key numbers to key selection buttons, e.g. 0->.top.sel.gen, 1->.top.sel.ks1
array set SelectionButtonMap {};
array set InverseSelectionButtonMap {};
array set CommandGlosses {};
array set PopupList {};
#This array is indexed on piece of information and key name.
array set KeyInfo {};

#Defaults
set SortAlgorithm q
set DefaultBrowser dillo
#For this purpose dillo is in many respects arguably the best choice because
#it is very lightweight and starts up fast. However, it isn't as fully
#featured as the others, and in particular, doesn't handle the full range of
#Unicode characters.
#To select a different default browser - uncomment (delete the initial #-sign from) one of the
#following lines:
#set DefaultBrowser firefox
#set DefaultBrowser epiphany;
#set DefaultBrowser galeon;
#set DefaultBrowser konqueror;
#set DefaultBrowser mozilla;
#set DefaultBrowser netscape;
#set DefaultBrowser opera;
set BrowserList [list firefox mozilla epiphany galeon konqueror dillo netscape]
set BrowserPIDS [list];

proc LoadMessageCatalog {} {
    global DebugP;

    if { $DebugP } {
	if { [::msgcat::mcload "msgs"] == 0} {
	    puts "No message catalog loaded."
	} else {
	    puts "Message catalog loaded."
	}
    } else  {
	::msgcat::mcload [file join [file dirname [info script]] msgs];
    }
}

proc ShutDown {} {
    global BrowserPIDS;
    #Shutdown child browsers
    foreach pid $BrowserPIDS {
	catch {exec kill $pid};
    }
    exit 0;
}

proc OldShutDown {} {
    global BrowserPIDS;
    #Shutdown child browsers
    foreach pid $BrowserPIDS {
	catch {exec kill $pid};
    }
    if {$::LastCommandLineSaveTime < $::LastExecutionTime} {
	ShowMessage [_ "The command line has not been saved recently."]
	SaveCommandLine; 
    }
    exit 0;
}

proc ShowWebPage {url} {
    global BrowserPIDS;
    if {[string equal $::System MacOSX]} {
	lappend BrowserPIDS [exec osascript -e "\"open location $url\""]
	return 
    }
    set BrowserFound 0;
    foreach Browser $::BrowserList {
	if { [string length [auto_execok $Browser]]} {
	    set BrowserFound 1;
	    break ;
	} else {
	    ShowMessage [format \
	     [_ "The browser %s is not available on this machine or not in your path."]\
		 $Browser];
	}
    }
    if {$BrowserFound} {
	lappend BrowserPIDS [exec $Browser $url &]
    } else {
	ShowMessage [_ "No browser on the browser list was located."]
    }
}

proc BindInsertionTarget {v} {
    bind $v <FocusIn> "+SetInsertionTargets $v"
}

#This provides an addition to the default methods of the Scrollbar
#widget. The default method, bound to <B1>, #increments or decrements
#the value by the resolution, by default 1. If you want a larger
#increment, you can use a larger resolution, but sometimes it is
#convenient to have a finer resolutio available while also being
#able to move in larger increments. A solution is to bind to
#<B3>, or some other event if you wish, a larger movement.
#This procedure moves by the specified multiplier k times
#the resolution of the Scrollbar. 

proc ScrollbarMoveBigIncrement {w f x y} {
    set part [$w identify $x $y]
    switch -exact -- $part {
	trough1 {
	    set dir -1;
	}
	arrow1 {
	    set dir -1;
	}
	trough2 {
	    set dir  1;
	}
	arrow2 {
	    set dir  1;
	}
	default {
	    return ;
	}
    }
    set CurrentFraction [lindex [$w get] 0]
    set NewFraction [expr $CurrentFraction + ($dir * $f)]
    eval [concat [$w cget -command] moveto $NewFraction]
}

#Font stuff
#Font system begins here

# We set these outside the namespace since FontInfo has to be global.
# Default defaults

set FontInfo(family) courier
set FontInfo(size) 12
set FontInfo(weight) normal
set FontInfo(slant) roman
set FontInfo(underline) 0
set FontInfo(overstrike) 0

namespace eval fontsel {
    variable FontPropertyList [list family size weight slant underline overstrike]
    variable FontPropertyClass
    variable FSCColors

    foreach p $FontPropertyList {
	set FontPropertyClass($p) generic;
    }
    set FontPropertyClass(underline) boolean
    set FontPropertyClass(overstrike) boolean

    #We check the existence of the variable so that this
    #can safely be called AFTER non-default values are set.
    proc SetFontInfoDefaults {} {
	global FontInfo;
	foreach ft $::FontList {
	    if {![info exist FontInfo($ft,family)]} {
		set FontInfo($ft,family)	$FontInfo(family)
	    }
	    if {![info exist FontInfo($ft,size)]} {
		set FontInfo($ft,size)	$FontInfo(size)
	    }
	    if {![info exist FontInfo($ft,weight)]} {
		set FontInfo($ft,weight)	$FontInfo(weight)
	    }
	    if {![info exist FontInfo($ft,slant)]} {
		set FontInfo($ft,slant)	$FontInfo(slant)
	    }
	    if {![info exist FontInfo($ft,underline)]} {
		set FontInfo($ft,underline)	$FontInfo(underline)
	    }
	    if {![info exist FontInfo($ft,overstrike)]} {
		set FontInfo($ft,overstrike)	$FontInfo(overstrike)
	    }
	    if {![info exist FontInfo($ft,gloss)]} {
		set FontInfo($ft,gloss)	$ft;
	    }
	    if {![info exist FontInfo($ft,help)]} {
		set FontInfo($ft,help)	"?"
	    }
	}
    }

    #For a given font assign font-specific properties from generics.
    proc FontSet {which} {
	global FontInfo
	variable FontPropertyList
	foreach prop $FontPropertyList {
	    set FontInfo($which,$prop) $FontInfo($prop);
	}
    }

    #Assign the proprety values of a specified font to the generics.
    proc InverseFontSet {which} {
	global FontInfo
	variable FontPropertyList
	foreach prop $FontPropertyList {
	    set FontInfo($prop) $FontInfo($which,$prop);
	}
    }

    #Configure a specified font according to the font-specific properties.
    proc ConfigureFont {which} {
	global FontInfo
	variable FontPropertyList
	foreach prop $FontPropertyList {
	    font configure $which -$prop $FontInfo($which,$prop)
	}
    }

    #Record as default values the font-specific values of properties at the
    #time of the call. The recorded values are intended for use by
    #ResetToDefaults 
    proc RecordDefaults {} {
	global FontInfo
	global FontList;
	variable FontPropertyList

	foreach f $FontList {
	    foreach p $FontPropertyList {
		set FontInfo($f,$p,Default) $FontInfo($f,$p);
	    }
	}
    }

    #Create all of the fonts on the font list and configure them
    #according to the generic values of the properties, which are
    #also assigned at this time to the font-specific variables.
    proc CreateFonts {} {
	global FontList;

	foreach ft $FontList {
	    font create $ft;
	    InverseFontSet $ft;
	    ConfigureFont $ft;
	}
    
    }

    #Get the values from the three scales, combine them into a hex RGB spec,
    #assign its value to FontExampleColor, and configure the two examples.
    proc UpdateFontExampleColor {x} {
	variable FSCColors
	global FontExampleColorConfigureWhich

	UpdateFontExampleColorHex;
	.selectFont.curcan.att.r1c1 configure \
	    -$::FontExampleColorConfigureWhich  \#$::FontExampleColor
	.selectFont.curcan.att.r2c1 configure \
	    -$::FontExampleColorConfigureWhich \#$::FontExampleColor
	set FSCColors(red,$FontExampleColorConfigureWhich)   $::FontRed;
	set FSCColors(green,$FontExampleColorConfigureWhich) $::FontGreen;
	set FSCColors(blue,$FontExampleColorConfigureWhich)  $::FontBlue;
    }

    proc UpdateFontExampleColorHex {} {
	set ::FontExampleColor [format "%02X%02X%02X" $::FontRed $::FontGreen $::FontBlue];
    }

    proc RestoreCSColors {} {
	variable FSCColors
	global FontExampleColorConfigureWhich
	set ::FontRed   $FSCColors(red,$FontExampleColorConfigureWhich)
	set ::FontGreen $FSCColors(green,$FontExampleColorConfigureWhich)
	set ::FontBlue  $FSCColors(blue,$FontExampleColorConfigureWhich)
	UpdateFontExampleColorHex;
    }

    proc CloneMainFontAsFontControlPanelFont {} {
	variable FontPropertyList
	global FontInfo

	foreach p $FontPropertyList {
	    font configure FontControlPanelFont -$p $FontInfo(MainFont,$p);
	}
    }
					

    #We create this separately because we don't want it on the FontList.
    font create FontControlPanelFont;

    #This is the main procedure of this package.
    #It creates the font control panel.
    proc CreateFontControlPanel {args} {
	global FontInfo;
	global WhichFontToSet
	global FontList;
	variable FontPropertyList
	variable FSCColors

	set General 1
	set exwid 10

	set w .selectFont
	if {[winfo exists $w]} {
	    wm deiconify $w;
	    raise $w;
	    return ;
	}
	toplevel $w
	wm title $w [_ "Font Selection"]
	BindKeys $w
	wm protocol $w WM_DELETE_WINDOW "wm withdraw $w"
	
	#The idea is that in general we want to use the main
	#font here, but we keep them separate so that we
	#can wait before switching the control panel to
	#the new main font configuration so that
	#the user isn't confronted with an unusable
	#control panel if he or she sets the mainfont to something
	#crazy.
	CloneMainFontAsFontControlPanelFont;

	if {[llength $args] > 0} {
	    set ::WhichFontToSet [lindex $args 0];
	    set General 0;
	} 
	set FontTypes [llength $FontList];
	if {$FontTypes == 1} {
	    set ::WhichFontToSet [lindex $FontList 0];
	    set General 0;
	}

	frame $w.which -relief ridge -border 3
	if {$General} {
	    set msg [_ "For what aspect of the program\ndo you wish to set the font?"]
	    balloonhelp_for $w.which $msg
	    label $w.which.title -text [_ "Use for which to set font"] \
		-anchor w -font FontControlPanelFont
	    frame $w.which.btns;
	    set k 0;
	    set PerRow 4;
	    set ypad 3
	    set xpad 10
	    foreach ft $FontList {
		set bn $w.which.btns.f$ft;
		radiobutton $bn -text [_ $FontInfo($ft,gloss)]  \
		    -variable WhichFontToSet -value $ft \
		    -command fontsel::SetFontSelectionDefaults \
		    -font FontControlPanelFont \
		    -indicatoron 0 -selectcolor $::ColorSpecs(Default,Background)
		balloonhelp_for $w.which.btns.f$ft $FontInfo($ft,help)
		set row [expr 1+ $k/$PerRow]
		set col [expr $k%$PerRow]
		if {$FontTypes >= $PerRow} {
		    grid $w.which.btns.f$ft -row $row -column $col \
			-sticky we -ipadx 3 -padx $xpad -pady $ypad
		}
		incr k
	    }
	    if {$FontTypes < $PerRow} {
		foreach ft $FontList {
		    pack $w.which.btns.f$ft -side left -expand 1 -fill x \
			-ipadx 5 -ipady 3 -padx 40 -pady 3
		}
	    }
	    frame $w.which.pad  -height 4
	    pack $w.which.title -side top -expand 1 -fill both -anchor w
	    pack $w.which.btns  -side top -expand 1 -fill both -anchor w
	    pack $w.which.pad   -side top -expand 0 -fill x    -anchor w

	}

	frame $w.mid 
	frame $w.mid.opts  -relief ridge -border 3
	frame $w.mid.families  -relief ridge -border 3
	label $w.mid.families.lab -text [_ "Family"] -relief ridge \
	    -justify center -font FontControlPanelFont
	listbox $w.mid.families.lb -height 1 -exportselection 0 \
	    -yscrollcommand "$w.mid.families.sbar set" -selectmode single \
	    -font FontControlPanelFont
	scrollbar $w.mid.families.sbar -orient vertical -command "$w.mid.families.lb yview"
	bind $w.mid.families.sbar <<B3>> \
	    "ScrollbarMoveBigIncrement $w.mid.families.sbar 0.20 %x %y"

	bind $w.mid.families.lb <<B3>> ProvideFontDescription

	pack $w.mid.families.lab  -side top -expand 0 -fill x
	pack $w.mid.families.lb   -side left -expand 1 -fill both
	pack $w.mid.families.sbar -side left -expand 1 -fill y

	label $w.mid.opts.lab -text [_ "Attributes"] -relief ridge \
	    -justify center -font FontControlPanelFont
	label $w.mid.opts.sizel -text [_ "Size:"]  -font FontControlPanelFont
	scale $w.mid.opts.size -orient h -digit 1 -from 5 -to 55 \
	    -variable FontInfo(size)  -tickinterval 0 -length 150 \
	    -font FontControlPanelFont
	bind $w.mid.opts.size <<B3>> "ScaleMoveBigIncrement $w.mid.opts.size 5 %x %y"

	label $w.mid.opts.weightl -text [::msgcat::mc "Bold:"] \
	    -font FontControlPanelFont	    
	checkbutton $w.mid.opts.weight -variable FontInfo(weight) \
	    -onvalue bold -offvalue normal  -font FontControlPanelFont

	label $w.mid.opts.slantl -text [::msgcat::mc "Italic:"] \
	    -font FontControlPanelFont
	checkbutton $w.mid.opts.slant -variable FontInfo(slant) \
	    -onvalue italic -offvalue roman -font FontControlPanelFont

	label $w.mid.opts.ulinel -text [::msgcat::mc "Underline:"] \
	    -font FontControlPanelFont
	checkbutton $w.mid.opts.uline -variable FontInfo(underline) \
	    -onvalue 1 -offvalue 0  -font FontControlPanelFont

	label $w.mid.opts.strkovl -text [::msgcat::mc "Overstrike:"] \
	    -font FontControlPanelFont
	checkbutton $w.mid.opts.strkov -variable FontInfo(overstrike) \
	    -onvalue 1 -offvalue 0  -font FontControlPanelFont

	grid $w.mid.opts.lab   -row 0 -column 0 -columnspan 2 -sticky ew
	grid $w.mid.opts.sizel -row 1 -column 0 -sticky w
	grid $w.mid.opts.size -row 1 -column 1 -sticky w
	grid $w.mid.opts.weightl -row 2 -column 0 -sticky w
	grid $w.mid.opts.weight -row 2 -column 1 -sticky w
	grid $w.mid.opts.slantl -row 3 -column 0 -sticky w
	grid $w.mid.opts.slant -row 3 -column 1 -sticky w
	grid $w.mid.opts.ulinel -row 4 -column 0 -sticky w
	grid $w.mid.opts.uline -row 4 -column 1 -sticky w
	grid $w.mid.opts.strkovl -row 5 -column 0 -sticky w
	grid $w.mid.opts.strkov -row 5 -column 1 -sticky w

	pack $w.mid.families -side left -expand 1 -fill both -padx 5 -pady 2
	pack $w.mid.opts     -side left -expand 1 -fill both -padx 5 -pady 2

	eval $w.mid.families.lb insert 0 [lsort [font families]]

	bind $w.mid.families.lb <ButtonPress-1> {fontsel::SelectFontFamily %W %y}

	frame $w.cntls -relief ridge -border 3
	button $w.cntls.sav -text [_ "Apply"]  -command fontsel::FontSave \
	    -font FontControlPanelFont
	button $w.cntls.res -text [_ "Reset"] -command  fontsel::ResetToDefaults \
	    -font FontControlPanelFont
	button $w.cntls.can -text [_ "Dismiss"] -command "wm withdraw $w" \
	    -font FontControlPanelFont
	button $w.cntls.cpanel -text [_ "Here Too?"] \
	    -command fontsel::CloneMainFontAsFontControlPanelFont \
	-font FontControlPanelFont
	pack $w.cntls.cpanel -side left  -expand 1 -fill both -padx 3
	pack $w.cntls.can    -side left  -expand 1 -fill both -padx 3
	pack $w.cntls.res    -side left  -expand 1 -fill both -padx 3
	pack $w.cntls.sav    -side right -expand 1 -fill both -padx 3

	set msg [_ "Press this button to dismiss the popup."];
	balloonhelp_for $w.cntls.can $msg
	set msg [_ "Press this button to reset the font to its defaults."];
	balloonhelp_for $w.cntls.res $msg
	set msg [_ "Press this button to make the changes\nyou have made take effect."]
	balloonhelp_for $w.cntls.sav $msg
	set msg [format \
	     [_ "Press this button to set the fonts\nin this control panel to the %s font."] \
		     [_ $::FontInfo(MainFont,gloss)]]
	balloonhelp_for $w.cntls.cpanel $msg

	frame $w.curcan -relief ridge -border 1
	set ATT [frame $w.curcan.att]
	set Row 0
	foreach r {Header Current Candidate} {
	    set Column 2
	    foreach p $FontPropertyList {
		set v $w.curcan.att.r${Row}c${Column};
		label $v  -font FontControlPanelFont
		incr Column;
	    } 
	    incr Row;
	}

	font create CurrentFontExampleFont \
	    -family $FontInfo($::WhichFontToSet,family) \
	    -size $FontInfo($::WhichFontToSet,size) \
	    -weight $FontInfo($::WhichFontToSet,weight) \
	    -slant $FontInfo($::WhichFontToSet,slant) \
	    -underline $FontInfo($::WhichFontToSet,underline) \
	    -overstrike $FontInfo($::WhichFontToSet,overstrike);

	font create CandidateFontExampleFont \
	    -family $FontInfo($::WhichFontToSet,family) \
	    -size $FontInfo($::WhichFontToSet,size) \
	    -weight $FontInfo($::WhichFontToSet,weight) \
	    -slant $FontInfo($::WhichFontToSet,slant) \
	    -underline $FontInfo($::WhichFontToSet,underline) \
	    -overstrike $FontInfo($::WhichFontToSet,overstrike);

	label $w.curcan.att.r0c0 -font FontControlPanelFont
	label $w.curcan.att.r0c1 -font FontControlPanelFont
	label $w.curcan.att.r1c0  -text [_ "Current"] -font FontControlPanelFont
	entry $w.curcan.att.r1c1 -font CurrentFontExampleFont -relief flat -width $exwid
	$w.curcan.att.r1c1 insert 0 [_ "example"]
	label $w.curcan.att.r2c0  -text [_ "Candidate"] -font FontControlPanelFont
	entry $w.curcan.att.r2c1 -font CandidateFontExampleFont -relief flat  -width $exwid
	$w.curcan.att.r2c1 insert 0 [_ "example"]

	#This is for use in conjunction with my character insertion library.
	#It arranges for the various character insertion widgets to insert
	#characters into the focussed window.
	if {[llength [info commands BindInsertionTarget]]} {
	    BindInsertionTarget $w.curcan.att.r1c1;
	    BindInsertionTarget $w.curcan.att.r2c1;
	}

	set CLR [frame $w.curcan.clr]
	set ln 60;
	set wd 10
	set trc \#D4B8C1
	scale $CLR.red -orient v -digit 1 -from 255 -to 0 -tickinterval 0 \
	    -variable FontRed  -showvalue 0 -length $ln -width $wd\
	    -troughcolor $trc -activebackground \#F0AAAA \
	    -bg red -fg yellow -command fontsel::UpdateFontExampleColor
	scale $CLR.grn -orient v -digit 1 -from 255 -to 0 -tickinterval 0 \
	    -variable FontGreen  -showvalue 0 -length $ln -width $wd\
	    -troughcolor $trc -activebackground \#AAF0AA \
	    -bg green -fg yellow -command fontsel::UpdateFontExampleColor
	scale $CLR.blu -orient v -digit 1 -from 255 -to 0 -tickinterval 0 \
	    -variable FontBlue  -showvalue 0 -length $ln -width $wd\
	    -troughcolor $trc -activebackground \#AAAAF0 \
	    -bg blue -fg yellow -command fontsel::UpdateFontExampleColor
	set FSCColors(red,bg)   0x00
	set FSCColors(green,bg) 0x00
	set FSCColors(blue,bg)  0x00
	set FSCColors(red,fg)   0x00
	set FSCColors(green,fg) 0x00
	set FSCColors(blue,fg)  0x00

	bind $CLR.red <<B3>> "ScaleMoveBigIncrement $CLR.red -20 %x %y"
	bind $CLR.grn <<B3>> "ScaleMoveBigIncrement $CLR.grn -20 %x %y"
	bind $CLR.blu <<B3>> "ScaleMoveBigIncrement $CLR.blu -20 %x %y"

	set ::FontRed   255
	set ::FontGreen 255
	set ::FontBlue  255
	set FontExampleColor \#FFFFFF
	label $CLR.rdt -textvariable FontExampleColor -width 7 -relief raised
	bind $CLR.rdt <Button-1> "pack forget $CLR"
	bind $ATT <Button-1> "pack $w.curcan.clr -before $ATT -side left -expand 1 -fill both"
	set msg [_ "Left click here to remove the color selector."]
	balloonhelp_for $CLR.rdt $msg;
	set msg [_ "Use this to set the colors in the example text boxes."]
	balloonhelp_for $CLR.red $msg;
	balloonhelp_for $CLR.grn $msg;
	balloonhelp_for $CLR.blu $msg;

	frame $CLR.whi
	set ::FontExampleColorConfigureWhich bg;
	radiobutton $CLR.whi.t -variable FontExampleColorConfigureWhich \
	    -value fg -command fontsel::RestoreCSColors -text [_ "fg"] \
	    -indicatoron 0 -activebackground gray -selectcolor gray \
	    -font FontControlPanelFont
	radiobutton $CLR.whi.b -variable FontExampleColorConfigureWhich \
	    -value bg -command fontsel::RestoreCSColors -text [_ "bg"] \
	    -indicatoron 0 -activebackground gray -selectcolor gray \
	    -font FontControlPanelFont
	pack $CLR.whi.t -side left -expand 1 -fill both
	pack $CLR.whi.b -side left -expand 1 -fill both

	pack $CLR.rdt -side top -expand 1  -fill both
	pack $CLR.whi -side bottom -expand 1  -fill both
	pack $CLR.red -side left -expand 1 -fill y
	pack $CLR.grn -side left -expand 1 -fill y
	pack $CLR.blu -side left -expand 1 -fill y

	set msg [_ "Choose the color for the sample text."]
	balloonhelp_for $CLR.whi.t $msg;
	set msg [_ "Choose the background color for the sample text."]
	balloonhelp_for $CLR.whi.b $msg;

	#Now grid them all
	set xp 5
	set yp 3
	set CLimit [expr [llength $FontPropertyList] + 2];
	for {set r 0} {$r < 3} {incr r} {
	    for {set c 0} {$c < $CLimit} {incr c} {
		set v $w.curcan.att.r${r}c${c};
		grid $v -row $r -column $c -sticky w -padx $xp -pady $yp
	    } 
	}

	set xp 3
	set yp 1
	if {!$General} {
	    $w.curcan.att.r0c0 configure -text [string totitle [_ $FontInfo($::WhichFontToSet,gloss)]]
	}

	pack $w.curcan.clr -side left -expand 1 -fill both -padx 9 -pady 5
	pack $w.curcan.att -side left -expand 1 -fill both -padx 3 -pady 5

	#Put it all together
	frame $w.botsep -height 4
	pack $w.which -side top -expand 1 -fill x -padx 8 -pady 6
	pack $w.curcan -side top -expand 1 -fill x -padx 8 -pady 6
	pack $w.mid   -side top -expand 1 -fill x -padx 8 -pady 1
	pack $w.cntls -side top -expand 1 -fill x -padx 8 -pady 2
	pack $w.botsep -side top -expand 1 -fill x -padx 8 -pady 2
	if {$General} {
	    $w.which.btns.f[lindex $FontList 0] invoke;#  Default to first on list
	}
	foreach f $FontPropertyList {
	    trace variable FontInfo($f) w fontsel::FontShow;
	}
	return $w;
    }

    #This resets the properties of the font specified by WhichFontToSet
    #to their default values, where "default" means whatever values
    #are stored in FontInfo(<font>,<property>,Default). The intention
    #is that these values will have been set by RecordDefaults at
    #a suitable point. A suitable point might be program startup or it
    #might be after init files have been read.
    #After resetting the attributes for the current font to their
    #default values, it configures the fonts for both the current and candidate
    #examples in the control panel to match.
    proc ResetToDefaults {} {
	global FontInfo;
	global WhichFontToSet
	variable FontPropertyList

	foreach prop $FontPropertyList {
	    set FontInfo($::WhichFontToSet,$prop) $FontInfo($::WhichFontToSet,$prop,Default);
	    set FontInfo($prop) $FontInfo($::WhichFontToSet,$prop,Default);
	}
	ConfigureFont $::WhichFontToSet;
	ConfigureCurrentExampleFont;
	ConfigureCandidateExampleFont;
	DescribeBothFonts;
    }

    #This is the procedure called from the control panel when the user decides
    #to apply and save the values set on the control panel. It sets the font-specific
    #attributes from the generic ones, configures the target font as well as the
    #current example font, and updates the description of the current font
    #in the control panel.
    proc FontSave {} {
	global FontInfo

	set f $::WhichFontToSet;
	FontSet $f;
	ConfigureFont $f
	ConfigureCurrentExampleFont;
	DescribeBothFonts;
    } 

    #This procedure configures the font used to display the candidate example
    #in the control panel. It is intended to be called whenever a property
    #is changed in the control panel or when the target font is changed.
    proc ConfigureCandidateExampleFont {} {
	global FontInfo
	global WhichFontToSet
	variable FontPropertyList

	foreach prop $FontPropertyList {
	    font configure CandidateFontExampleFont -$prop $FontInfo($prop)
	}
    }

    #This procedure configures the font used to display the current example
    #in the control panel. It is intended to be called when the user
    #applies new values or when the target font is changed.
    proc ConfigureCurrentExampleFont {} {
	global FontInfo
	global WhichFontToSet
	variable FontPropertyList

	foreach prop $FontPropertyList {
	    font configure CurrentFontExampleFont -$prop $FontInfo($::WhichFontToSet,$prop)
	}
    }

    #This procedure configures the candidate example font#and updates the
    #description of the candidate font in the control panel. It is intended
    #to be called whenever a proprty of the candidate font is changed.
    proc FontShow {args} {
	ConfigureCandidateExampleFont;
	DescribeBothFonts;
    }

    #This procedure generates the descriptions of the current and candidate
    #fonts that are shown in the control panel.
    proc DescribeBothFonts {} {
	global FontInfo
	global WhichFontToSet;
	variable FontPropertyList

	set f $::WhichFontToSet;
	set c 2;
	foreach prop $FontPropertyList {
	    .selectFont.curcan.att.r0c$c configure -text  [string totitle [_ $prop]] 
	    set value $FontInfo($f,$prop)
	    if {$value == 1} {set value [_ "Yes"]} else {
		if {$value == 0} {set value [_ "No"]}}
	    .selectFont.curcan.att.r1c$c configure -text $value;
	    set value $FontInfo($prop)
	    if {$value == 1} {set value [_ "Yes"]} else {
		if {$value == 0} {set value [_ "No"]}}
	    .selectFont.curcan.att.r2c$c configure -text $value; 
	    incr c;
	}
    }

#This is to be called when we change font targets.
#It resets the control panel to show the current
#configuration for the newly targeted font.
    proc SetFontSelectionDefaults {} {
	global FontInfo;
	global WhichFontToSet;
	variable FontPropertyList

	foreach prop $FontPropertyList {
	    set FontInfo($prop) $FontInfo($WhichFontToSet,$prop);
	}
	set FontChoices [.selectFont.mid.families.lb size];
	set k 1;
	while {$k <= $FontChoices} {
	    set Family [.selectFont.mid.families.lb get $k];
	    if {[string equal $Family [string tolower $FontInfo($WhichFontToSet,family)]]} {
		.selectFont.mid.families.lb selection clear 0 end;
		.selectFont.mid.families.lb see $k
		.selectFont.mid.families.lb selection set $k $k;
		.selectFont.mid.families.lb activate $k;
		set FontInfo(family) [.selectFont.mid.families.lb get $k]
		break
	    }
	    incr k;
	}
	ConfigureCandidateExampleFont
	ConfigureCurrentExampleFont;
	DescribeBothFonts;
    }

    #This is the callback that extracts a new font family
    #setting from the control panel.
    proc SelectFontFamily {w y} {
	global FontInfo

	set index [$w nearest $y]
	if {$index != ""} {
	    set FontInfo(family) [$w get $index]
	}
    }

    #This is the callback that extracts a new font size
    #from the control panel.
    proc SelectFontSize {w} {
	global WhichFontToSet
	global FontInfo

	set size [$w get]
	if {[catch {expr round($size)} size] == 0} {
	    if { ($size > 0) && ($size <= 60) } {
		set FontInfo(size) $size
		return ;
	    }
	}
	$w delete 0 end
	$w insert 0 $FontInfo($WhichFontToSet,size)
    }

    #This is presently a stub.
    #It is intended to be called on an event, probably <B3>,
    #in the font family listbox in the control panel.
    #Its purpose is to provide the user with information about
    #the particular font. 
    proc ProvideFontDescription {w y} {
	variable FontDescriptions

	set index [$w nearest $y]
	if {$index != ""} {
	    FontDescriptions([$w get $index])
	}
    }

    #Create a command for setting each combination of font and property.
    proc DefineFontSettingProcs {} {
	variable FontPropertyList
	variable FontPropertyClass
	global FontList;
	global FontProcList;

	foreach ft $FontList {
	    foreach prop $fontsel::FontPropertyList {
		if {[string equal $FontPropertyClass($prop) boolean]} {
		    set ProcName  [format "::fontsel::Set%s%sP" $ft [string totitle $prop]];
		    lappend FontProcList $ProcName;
		    set cmd [list "proc"  $ProcName  "\{v\}" \
			 "set ::FontInfo($ft,$prop) \[SlaveBoolean \$v\];\nConfigureFont $ft;"]
		} else {
		    set ProcName  [format "::fontsel::Set%s%s" $ft [string totitle $prop]];
		    lappend FontProcList $ProcName;
		    set cmd [list "proc" $ProcName "\{v\}" \
			 "set ::FontInfo($ft,$prop) \$v;\nConfigureFont $ft;"]
		}
		eval $cmd;
	    }
	}
    }

    #Create aliases in the named daughter interpreter for the font setting commands
    proc AliasFontSettings {interp} {
	variable FontPropertyList
	variable FontPropertyClass
	global FontList;
	
	foreach ft $FontList {
	    foreach prop $FontPropertyList {
		set Tprop [string totitle $prop]
		set Tprop2 $Tprop;
		if {[string equal $FontPropertyClass($prop) boolean]} {
		    set Tprop ${Tprop}P
		}
		eval [split [format "%s alias %s%s ::fontsel::Set%s%s" \
			 $interp $ft $Tprop $ft $Tprop2]]
	    }
	}
    }

    #Returns a list of init file commands representing the current font settings.
    proc SaveFontSettings {} {
	variable FontPropertyList
	variable FontPropertyClass
	global FontInfo
	global FontList;
	
	set cl [list]
	foreach ft $FontList {
	    foreach prop $FontPropertyList {
		set Tprop [string totitle $prop]
		if {[string equal $FontPropertyClass($prop) boolean]} {
		    set Tprop ${Tprop}P
		}
		lappend cl [format "%s%s %s" $ft $Tprop [list $FontInfo($ft,$prop)]]
	    }
	}
	return [lsort $cl];
    }
}

# End of namespace fontsel

# This is the stuff specific to this program.

set FontList { \
MainFont\
HeaderFont\
BalloonHelpFont\
MenuFont\
UserPlainTextFont\
UserTechnicalTextFont\
CharacterEntryFont\
DiacriticFont\
AccentedLetterFont}

set FontInfo(family) code2000

set FontInfo(BalloonHelpFont,family) lucida
set FontInfo(BalloonHelpFont,size) 13
set FontInfo(BalloonHelpFont,gloss) "balloon help"
set FontInfo(BalloonHelpFont,help) "This font is used in help balloons."

set FontInfo(MainFont,family) "courier"
set FontInfo(MainFont,size) 12
set FontInfo(MainFont,gloss) [_ "general"]
set FontInfo(MainFont,help) [_ "This font is used for most things."]

set FontInfo(HeaderFont,family) "courier"
set FontInfo(HeaderFont,size) 12
set FontInfo(HeaderFont,weight) bold
set FontInfo(HeaderFont,gloss) [_ "headings"]
set FontInfo(HeaderFont,help) [_ "This font is used for headings."]

set FontInfo(UserPlainTextFont,gloss) [_ "plain user text"]
set FontInfo(UserPlainTextFont,help) [_ "This font is used for ordinary text entered\nby the user, such as file names."]
set FontInfo(UserPlainTextFont,family) bible
set FontInfo(UserPlainTextFont,size) 14
set FontInfo(UserTechnicalTextFont,gloss) [_ "technical user text"]
set FontInfo(UserTechnicalTextFont,help) [_ "This font is used for text entered by the\nuser that is likely to contain unusual\ncharacters, such as multigraph definitions."]
set FontInfo(UserTechnicalTextFont,family) "sil doulos"
set FontInfo(UserTechnicalTextFont,size) 13

set FontInfo(MenuFont,family) courier
set FontInfo(MenuFont,size) 12
set FontInfo(MenuFont,gloss) [_ "menu"]
set FontInfo(MenuFont,help) [_ "This font is used for menu labels."]

set FontInfo(CharacterEntryFont,gloss) "character entry"
set FontInfo(CharacterEntryFont,help) "This font is used on the labels of\nmost character insertion widgets."
set FontInfo(AccentedLetterFont,gloss) "accented letters"
set FontInfo(AccentedLetterFont,help) "This font is used on the labels of the\naccented character insertion widget."

set FontInfo(DiacriticFont,gloss) "diacritics"
set FontInfo(DiacriticFont,help) "This font is used on the labels of the\ndiacritic insertion widget."

set FontInfo(SmallLabelFont,family) $FontInfo(MainFont,family);
set FontInfo(SmallLabelFont,gloss) "small labels"
set FontInfo(SmallLabelFont,help) "small labels"

set FontInfo(CharacterEntryFont,size) 14
set FontInfo(AccentedLetterFont,size) 14
set FontInfo(DiacriticFont,size)  [expr $FontInfo(CharacterEntryFont,size) + 40];
set FontInfo(SmallLabelFont,size) [expr $FontInfo(AccentedLetterFont,size) -  8];


#Now call these two functions.

fontsel::SetFontInfoDefaults;
fontsel::CreateFonts;

#End of font stuff

option add *selectFont*Font MainFont 100
option add *Label.Font MainFont 100
option add *Menu.Font MenuFont 100
option add *Radiobutton.Font MainFont 100

proc AddKey {} {
    incr ::KeyCount;
    incr ::MaxKeyCount;
    set KeyName [format ".ks%1d" $::MaxKeyCount]
    set KeyPageSelectionButtonLabel [format "Key %d" $::KeyCount] 
    #We start out with a simple correspondance of selection button name to page name.
    #If the button already exists, we increase the index until we find one that is free.
    set sbind $::KeyCount;
    set SelectionButtonName [format ".top.sel.ks%d" $sbind]
    while {[winfo exists $SelectionButtonName]} {
	incr sbind;
	set SelectionButtonName [format ".top.sel.ks%d" $sbind]
    }
    set ::SelectionButtonMap($::KeyCount) $SelectionButtonName;
    UpdateInverseSelectionButtonMap;
    set NewButton [radiobutton $SelectionButtonName \
	       -text $KeyPageSelectionButtonLabel \
	       -variable WhichKey -value $::KeyCount \
	       -width $::KeySelWidth -height $::KeySelHeight]
    $NewButton configure -bg $::ColorSpecs(PageChoice,Background)
    $NewButton configure -activebackground $::ColorSpecs(PageChoice,ActiveBackground)
    $NewButton configure -selectcolor $::ColorSpecs(PageChoice,Selected) 
    $NewButton configure -indicatoron 0
    pack $NewButton -side left
    set ::KeyMap($::KeyCount) $KeyName;
    UpdateInverseKeyMap;
    SetupKey $::KeyMap($::KeyCount);
    balloonhelp_for $NewButton [format [_ "Click on this button to switch to settings for key %d\nPress the right button, drag to another button,\nand release to swap keys."] $::KeyCount];
    BindKeySelectionButton $NewButton;
#    dmsg "Added key $KeyName with button $SelectionButtonName"
    TellKeyMap;
    TellSelectionButtonMap;
    $NewButton invoke
}

proc SwapKeys {m n} {
    set temp $::KeyMap($m);
    set ::KeyMap($m) $::KeyMap($n)
    set ::KeyMap($n) $temp;
    UpdateInverseKeyMap;
    TellKeyMap;
    TellSelectionButtonMap;
    $::SelectionButtonMap($::WhichKey) invoke
    ShowMessage [format [_ "Swapped keys %d and %d"] $m $n]
}

proc UpdateInverseKeyMap {} {
    unset ::InverseKeyMap;
    foreach KeyNumber [array names ::KeyMap] {
	set KeyName $::KeyMap($KeyNumber);
	set ::InverseKeyMap($KeyName) $KeyNumber;
    }
}

proc UpdateInverseSelectionButtonMap {} {
    unset ::InverseSelectionButtonMap;
    foreach KeyNumber [array names ::SelectionButtonMap] {
	set KeyName $::SelectionButtonMap($KeyNumber);
	set ::InverseSelectionButtonMap($KeyName) $KeyNumber;
    }
}

proc BindKeySelectionButton {b} {
    bind $b <<B3>> {SetKeyButtonMoveSource %X %Y}
    bind $b <<B3Release>> {SwapKeysOnEvent %X %Y}
}

proc SetKeyButtonMoveSource {x y} {
    set EventWindow [winfo containing $x $y];
    if {[info exists ::InverseSelectionButtonMap($EventWindow)] && $::InverseSelectionButtonMap($EventWindow)} {
	set ::KeyButtonMoveSource $::InverseSelectionButtonMap($EventWindow);
    } else {
	ShowMessage [_ "That wouldn't make sense. You can only swap key buttons."]
    }
}

proc SwapKeysOnEvent {x y} {
    set EventWindow [winfo containing $x $y];
    if {[info exists ::InverseSelectionButtonMap($EventWindow)] && $::InverseSelectionButtonMap($EventWindow)} {
	SwapKeys $::KeyButtonMoveSource $::InverseSelectionButtonMap($EventWindow);
    } else {
	ShowMessage [_ "That wouldn't make sense. You can only swap key buttons."]
    }
}

proc UpdateKeySelectionButtonLabels {} {
    for {set KeyNumber 1} {$KeyNumber <= $::KeyCount} {incr KeyNumber} {
	set KeyPageName $::KeyMap($KeyNumber);
	set KeyPageSelectionButton [format ".top.sel.ks%d" $KeyNumber];
	set KeyPageSelectionButtonLabel [format [_ "Key %d"] $KeyNumber]
	if {[winfo exists $KeyPageSelectionButton]} {
	    $KeyPageSelectionButton configure -text $KeyPageSelectionButtonLabel;
	}
    } 
}

proc RemoveKey {} {
    if {$::KeyCount == 1} {
	ShowMessage [_ "You can't sort anything without any keys."]
	return
    }
    set LastKeyNumber [lindex [lsort -integer -decreasing [array names ::KeyMap]] 0];
    set LastKeyName $::KeyMap($LastKeyNumber);
    trace vdelete ::KeyInfo(KeySelection,$LastKeyName) w ChooseActiveKeySelection
    trace vdelete ::KeyInfo(ExclusionsP,$LastKeyName) w ToggleExclusionEntryActive
    trace vdelete ::KeyInfo(SubstitutionsP,$LastKeyName) w ToggleSubstitutionEntryActive
    trace vdelete ::KeyInfo(SortOrderSpecType,$LastKeyName) w ToggleCustomSortOrderEntryActive
    trace vdelete ::KeyInfo(SortType,$LastKeyName) w ToggleOnSortType
    trace vdelete ::KeyInfo(OptionalP,$LastKeyName) w ToggleOptComparisonActive
    destroy $LastKeyName;

    set LastKeyPageSelectionButton $::SelectionButtonMap($::KeyCount);
    destroy $LastKeyPageSelectionButton;
    incr ::KeyCount -1;
    unset ::KeyMap($LastKeyNumber);
    unset ::SelectionButtonMap($LastKeyNumber);
    UpdateInverseSelectionButtonMap;
    UpdateInverseKeyMap;
    TellKeyMap;
    TellSelectionButtonMap;
    set ::WhichKey 1;
}

proc TellKeyMap {} {
    if {$::DebugP} {
	puts -nonewline "Keymap:            ";
	foreach n [array names ::KeyMap] {
	    puts -nonewline "$n $::KeyMap($n)   "
	}
	puts "";
	puts -nonewline "Inverse Keymap:    ";
	foreach n [array names ::InverseKeyMap] {
	    puts -nonewline "$n $::InverseKeyMap($n)   "
	}
	puts "";
    }
}

proc TellSelectionButtonMap {} {
    if {$::DebugP} {
	puts -nonewline "Buttonmap:         ";
	foreach n [array names ::SelectionButtonMap] {
	    puts -nonewline "$n $::SelectionButtonMap($n)   "
	}
	puts "";
	puts -nonewline "Inverse Buttonmap: ";
	foreach n [array names ::InverseSelectionButtonMap] {
	    puts -nonewline "$n $::InverseSelectionButtonMap($n)   "
	}
	puts "";
    }
}

proc ShowMessage {msg} {
    if {[winfo exists .top.msg]} {
	.top.msg configure -state normal;
	.top.msg delete 1.0 end;
	.top.msg insert 1.0 $msg;
	.top.msg configure -state disabled;
    } else {
	puts $msg;
    }
}

proc ClearMessageWindow {} {
     .top.msg configure -state normal
     .top.msg delete 1.0 end;
     .top.msg configure -state disabled
}

proc SpecifyExclusionFile {KeyName} {
    set ExclusionFile [tk_getOpenFile];
    if {$ExclusionFile == ""} {
	ShowMessage [_ "File selection aborted."]
	return
    } else { 
	$KeyName.exc.bot.ent delete 0 end;
	$KeyName.exc.bot.ent insert insert $ExclusionFile;
    }
}

proc SpecifySubstitutionFile {KeyName} {
    set SubstitutionFile [tk_getOpenFile];
    if {[string equal $SubstitutionFile ""]} {
	ShowMessage [_ "File selection aborted."]
	return
    } else { 
	$KeyName.sub.bot.ent delete 0 end;
	$KeyName.sub.bot.ent insert insert $ExclusionFile;
    }
}

proc GetFieldSeparator {} {
    return [.gen.ctr.wr.fsent get];
}

proc GetRecordSeparator {} {
    return [.gen.ctr.rp.cent get];
}

proc GetSortOrderSeparatorFile {KeyName} {
    return [$KeyName.sos.bot.ent get]
}

proc GetSortOrderFile {KeyName} {
    return [$KeyName.so.bot.ent get]
}

proc GetLocale {KeyName} {
    return [$KeyName.so.loc.ent get]
}

proc GetExclusionFile {KeyName} {
    return [$KeyName.exc.bot.ent get]
}

proc GetSubstitutionFile {KeyName} {
    return [$KeyName.sub.bot.ent get]
}

proc SpecifySortOrderFile {KeyName} {
    set SortOrderFile [tk_getOpenFile];
    if {$SortOrderFile == ""} {
	ShowMessage [_ "File selection aborted."]
	return
    } else { 
	$KeyName.so.bot.ent delete 0 end
	$KeyName.so.bot.ent insert insert $SortOrderFile;
    }
}

proc SpecifySortOrderSeparatorFile {KeyName} {
    set SortOrderSeparatorFile [tk_getOpenFile];
    if {$SortOrderSeparatorFile == ""} {
	ShowMessage [_ "File selection aborted."]
	return
    } else { 
	$KeyName.sos.bot.ent delete 0 end
	$KeyName.sos.bot.ent insert insert $SortOrderSeparatorFile;
    }
}

proc MsortAvailableP {} {
    if {[string length [auto_execok msort]]} {
	return 1;
    }
    return 0;
}

proc FindoutMsortVersion {} {
    catch {exec msort -v} vline
    return [lindex [split [lindex [split $vline "\n"] 0]] 1]
}


#Because of the strangeness needed in ExecuteMsort, this constructs everything
#except for the diversion of stdout to the output file.
proc ConstructCommandLine {} {
    global KeyMap;
    set cl [list msort];
    #Global flags
    lappend cl "-q";
    lappend cl "-a"
    switch -- $::SortAlgorithm {
	i {
	    lappend cl "i"
	}
	m {
	    lappend cl "m"
	}
	q {
	    lappend cl "q"
	}
	s {
	    lappend cl "s"
	}
    }
    if {$::GlobalInversionP == 1} {
	lappend cl "-I";
    }
    if {$::BMPOnlyP == 1} {
	lappend cl "-B";
    }
    if {$::ReservePrivateUseAreasP == 1} {
	lappend cl "-p";
    }
    if {$::EOLCarriageReturnP} {
	lappend cl "-m";
    }
    if {$::RecordParse == 0} {
	lappend cl "-l";
    } elseif {$::RecordParse ==2} {
	set rs [GetRecordSeparator];
	if {[string equal $rs ""] ==0} {
	    lappend cl "-r"
	    lappend cl $rs;
	}
    } elseif {$::RecordParse ==3} {
	lappend cl "-O"
	lappend cl [.gen.ctr.rp.fl.len get]
    }
    if {$::WholeRecordIsKeyP} {
	lappend cl "-w";
    } else {
	set fs [GetFieldSeparator];
	if {[string equal $fs ""] ==0} {
	    lappend cl "-d"
	    lappend cl $fs
	}
    }
    #Key specific flags
    for {set i 1} {$i <= $::KeyCount} {incr i} {
	set hd $KeyMap($i);
	if {$::WholeRecordIsKeyP == 0} {
	    if {$::KeyInfo(KeySelection,$hd) == 0} {
		set p1 [$hd.key.ki.pos.posent.fldent1 get]
		set p2 [$hd.key.ki.pos.posent.fldent2 get]
		set c1 [$hd.key.ki.pos.posent.chrent1 get]
		set c2 [$hd.key.ki.pos.posent.chrent2 get]
		if {[string equal $p1 ""]} {
		    ShowMessage [format [_ "No start field has been specified for key %d"] $i];
		    return;
		}
		if {[string equal $p2 ""]} {
		    ShowMessage [format [_ "No end field has been specified for key %d"] $i];
		    return;
		}
		lappend cl "-n"
		set keyrange $p1
		if {![string equal $c1 ""]} {
		    append keyrange .$c1
		}
		if {$p1 != $p2} {
		    append keyrange ,$p2
		    if {![string equal $c2 ""]} {
			append keyrange .$c2
		    }
		}
		lappend cl $keyrange
	    } elseif {$::KeyInfo(KeySelection,$hd) == 1} {
		set t [$hd.key.ki.pos.tagent get];
		if {[string equal $t ""] == 0} {
		    lappend cl "-t"
		    lappend cl $t
		} else {
		    ShowMessage [format [_ "No tag has been specified for key %d"] $i];
		    return;
		}
	    } else {
		set first [$hd.key.ki.pos.firstent get];
		set last  [$hd.key.ki.pos.lastent  get];
		if { ([string equal $first ""] == 0) && ([string equal $last "" ] == 0)} {
		    lappend cl "-e"
		    lappend cl [format "%d,%d" $first $last]
		} else {
		    ShowMessage [format [_ "No character range has been specified for key %d"] $i];
		    return;
		}
	    }
	}
	lappend cl "-c";
	switch -- $::KeyInfo(SortType,$hd) {
	    1 {
		lappend cl "l";
	    }
	    2 {
		lappend cl "s";
	    }
	    3 {
		lappend cl "n";
	    }
	    4 {
		lappend cl "t";
	    }
	    5 {
		lappend cl "d";
		lappend cl "-f";
		lappend cl $::KeyInfo(DateFormat,$hd);
	    }
	    6 {
		lappend cl "i";
	    }
	    7 {
		lappend cl "h";
	    }
	    8 {
		lappend cl "r";
	    }
	    9 {
		lappend cl "a";
	    }
	    10 {
		lappend cl "N";
	    }
	    11 {
		lappend cl "m";
	    }
	}
	if { ($::KeyInfo(StripDiacriticsP,$hd)) || \
		 ($::KeyInfo(ConvertEnclosuresP,$hd)) || \
		 ($::KeyInfo(ConvertStylisticP,$hd)) } {
	    lappend cl "-T"
	    set arg "";
	    if {$::KeyInfo(StripDiacriticsP,$hd)} {
		append arg d
	    }
	    if {$::KeyInfo(ConvertEnclosuresP,$hd)} {
		append arg e
	    }
	    if {$::KeyInfo(ConvertStylisticP,$hd)} {
		append arg s
	    }
	    lappend cl $arg
	}
	if {$::KeyInfo(FoldCaseP,$hd)} {
	    set type $::KeyInfo(SortType,$hd)
	    if { ($type == 1) || ($type == 2) || ($type ==7)} {
		lappend cl "-C"
	    } else {
		ShowMessage [_ "Case-folding is not compatible with this comparison type."]
	    }
	}
	if {$::KeyInfo(TurkicFoldCaseP,$hd)} {
	    set type $::KeyInfo(SortType,$hd)
	    if { ($type == 1) || ($type == 2) || ($type ==7)} {
		lappend cl "-z"
	    } else {
		ShowMessage [_ "Case-folding is not compatible with this comparison type."]
	    }
	}
	if {$::KeyInfo(ReverseKeyP,$hd)} {
	    if {$::KeyInfo(SortType,$hd) == 1} {
		lappend cl "-R"
	    }
	}
	if {$::KeyInfo(LocalInversionP,$hd)} {
	    lappend cl "-i"
	}
	if {$::KeyInfo(OptionalP,$hd)} {
	    lappend cl "-o";
	    switch -- $::KeyInfo(OptComp,$hd) {
		-1 {lappend cl "l"}
		 0 {lappend cl "e"}
		 1 {lappend cl "g"}
	    }
	}
	set f [GetSortOrderSeparatorFile $hd];
	if {[string equal $f ""] ==0} {
	    lappend cl "-W"
	    lappend cl $f;
	}
	set f [GetExclusionFile $hd];
	if {[string equal $f ""] ==0} {
	    lappend cl "-x"
	    lappend cl $f;
	}
	set f [GetSubstitutionFile $hd];
	if {[string equal $f ""] ==0} {
	    lappend cl "-S"
	    lappend cl $f;
	}
	switch -exact -- $::KeyInfo(SortOrderSpecType,$hd) {
	    1 {
		set l [GetLocale $hd];
		lappend cl "-s"
		lappend cl $l;
	    }
	    2 {
		set f [GetSortOrderFile $hd];
		lappend cl "-s"
		lappend cl $f;
	    }
	}
    }
    lappend cl "<";
    set inf [.gen.ctr.if.ent get]
    if {[string equal $inf ""]} {
	ShowMessage [_ "No input file has been specified. You must supply one."]
	return "err";
    }
    if {[file exists $inf] == 0} {
	ShowMessage [format [_ "Input file \`%s\' does not exist."] $inf]
	return "err";
    }
    if {[file readable $inf] == 0} {
	ShowMessage [format [_ "Input file \`%s\' is not readable."] $inf]
	return "err";
    }
    lappend cl $inf;
    set outf [.gen.ctr.of.ent get]
    if {[string equal $outf ""]} {
	ShowMessage [_ "No output file has been specified. You must supply one."]
	return "err";
    }
    if {[file exists $outf]} {
	if {[file writable $outf] == 0} {
	    ShowMessage [format [_ "Output file \`%s\' is not writable."] $outf]
	    return "err";
	}
    }
    return [join $cl]; 
}


#Returns a list containing the record count and the number of comparisons.
proc GetLogInfo {} {
    if {[catch {open "msort.log" "r"} mlh ] != 0} {
	ShowMessage [_ "Mysterious failure to open msort log file."];
	return -1;
    }
    set LineCount 0;
    while { [gets $mlh line] >= 0} {
	incr LineCount;
	lappend log $line;
    }
    if {$LineCount < 1} {
	error;
    }
    set Ultima [lindex $log [expr $LineCount -1]];
    if {[scan $Ultima "Records written:[ ]%s" Records] < 1} {
	ShowMessage [_ "Unable to obtain record count from msort log."]
	set Records -1;
    }
    set Penult   [lindex $log [expr $LineCount -2]];
    if {[scan $Penult "\tActual:[ ]%s" Comparisons] < 1} {
	ShowMessage [_ "Unable to obtain comparison count from msort log."]
	set Comparisons -1;
    }

    close $mlh;
    return [list [string trim $Records] [string trim $Comparisons]];
}

set ElapsedTime 0;
set StdoutResult "";
proc MsortPipeEvent {PipeHandle Start} {
    global ElapsedTime
    global StdoutResult
    if {[eof $PipeHandle]} {
	set ElapsedTime [expr {[clock seconds] - $Start}]
	catch {[close $PipeHandle]};
    } else {
	set StdoutResult [read $PipeHandle]
    }
}


set msortpid 0;
proc ExecuteMsort {} {
    set ::SortAbortedP 0;
    ClearMessageWindow;
    file delete msort.log
    set cl [ConstructCommandLine]
    if {[string match msort* $cl] == 0} {
	if {[string equal $cl ""]} {
	    ShowMessage [_ "Obligatory information is missing: cannot sort."];
	}
	return ;
    }
    set cl [split $cl];
    lappend cl "2>"
    lappend cl "/dev/null"
    set ::SortInProgressP 1;
    set StartTime [clock seconds]
    set ::LastExecutionTime  $StartTime;
    catch {open "|$cl" "r"} PipeID;
    fileevent $PipeID readable [list MsortPipeEvent $PipeID $StartTime]
    set ::msortpid [lindex [pid $PipeID] 0]
    tkwait variable ::ElapsedTime;
    set ::SortInProgressP 0;
    if {$::SortAbortedP} {return}
    catch {GetLogInfo} LogInfo;
    if {[info exists LogInfo]} {
	set Records [lindex $LogInfo 0];
	set Comparisons  [lindex $LogInfo 1];
    } else {
	set Records -1;
	set Comparisons -1;
    }
    set msg "";
    if {$Records >= 0} {
	lappend msg [format [_ "Records: %s"] $Records];
	if {$Comparisons >= 0} {
	    lappend msg [format [_ "  Comparisons: %s"] $Comparisons];
	}
    }
    lappend msg [format [_ "  Elapsed time: %d seconds"] $::ElapsedTime];
    if {[string equal $::errorCode "NONE"] == 0} {
	set spec [split $::errorCode]
	switch -exact -- [lindex $spec 0] {
	    CHILDSTATUS {
		set MsortErrorCode [lindex $spec 2];
		lappend msg [_ $::MsortErrors($MsortErrorCode)];
	    }
	    POSIX {
		lappend msg [lindex $spec 2]
	    }
	}
    }
    ShowMessage [join $msg];
    set OutputFile [.gen.ctr.of.ent get];
    if {[catch {open $OutputFile "w"} OutputHandle ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $OutputFile];
	return ;
    }
    puts -nonewline $OutputHandle $::StdoutResult
    close $OutputHandle;
}

proc ExplicateErrorStatus {} {
    global FeatureTestDebugP;
    global FeatureTestLogHandle;
    switch -exact -- [lindex $::errorCode 0] {
	CHILDKILLED {
	    foreach { - pid sigName msg } $::errorCode break
	    if {$FeatureTestDebugP} {
		puts $FeatureTestLogHandle [format "Child process %d died on signal %s" $pid $sigName];
		puts $FeatureTestLogHandle [format "Error message: %s" $msg];
	    }
	}
	CHILDSTATUS {
	    foreach { - pid code } $::errorCode break
	    if {$FeatureTestDebugP} {
		puts $FeatureTestLogHandle [format "Child process %d exited with error code %d" $pid $code];
	    }
	}
	CHILDSUSP {
	    foreach { - pid sigName msg } $::errorCode break
	    if {$FeatureTestDebugP} {
		puts $FeatureTestLogHandle [format "Child process %d has been suspended due to signal %d" $pid $sigName];
		puts $FeatureTestLogHandle [format "Error message: %s" $msg];
	    }
	}
	POSIX {
	    foreach { - errName msg } $::errorCode break
	    if {$FeatureTestDebugP} {
		puts $FeatureTestLogHandle [format "One of the kernel calls to launch the command failed ith error code %d" $code];
		puts $FeatureTestLogHandle [format "Error message: %s" $msg];
	    }
	}
    }
}


array set MsortErrors {
    1 "Failed to open file"\
    2 "I/o error"\
    3 "Not really an error - provided information"\
    4 "Limit exceeded"\
    5 "Command line option not recognized"\
    6 "Invalid argument to command line option"\
    7 "Out of memory"\
    8 "Ill-formed record"\
    9 "Other error"
}

proc AbortSort {} {
    if {$::SortInProgressP} {
	if {[string equal $::System MSWindows] == 0} {
	    catch {[exec kill -9 $::msortpid]}
	    set ::SortAbortedP 1;
	    ShowMessage [_ "Sort aborted."]
	} else {
	    ShowMessage [_ "Sorry, this must be an MS Windows system."];
	}
    } else {
	ShowMessage [_ "No sort is in progress."]
    }
}

proc SaveCommandLine {} {
    set cl [ConstructCommandLine];
    if {[string match msort* $cl] == 0} {
	if {[string equal $cl ""]} {
	    ShowMessage [_ "Failed to construct command line."]
	}
	return ;
    }
    set outf [.gen.ctr.of.ent get]
    set cl [split $cl];
    lappend cl ">"
    lappend cl $outf 
    set cl [join $cl];
    set SaveFile [tk_getSaveFile -initialfile [_ "SavedMsortCommandLine"]];
    if {$SaveFile == ""} {
	ShowMessage [_ "File selection cancelled."]
	return ;
    }
    if {[catch {open $SaveFile "w"} SaveHandle ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $SaveFile];
	return ;
    }
    puts $SaveHandle $cl;
    close $SaveHandle;
    set ::LastCommandLineSaveTime  [clock seconds];
}

proc ShowCommandLine {} {
    set cl [ConstructCommandLine];
    if {[string match msort* $cl] == 0} {
	if {[string equal $cl ""]} {
	    ShowMessage [_ "Failed to construct command line."]
	}
	return ;
    } else {
	set outf [.gen.ctr.of.ent get]
	set cl [split $cl];
	lappend cl ">"
	lappend cl $outf 
	ShowMessage [join $cl];
    }
}

proc InsertClipboardContents {w} {
    set sel "";
    if {[catch {set sel [selection get -selection CLIPBOARD -type UTF8_STRING]}]} {
	ShowMessage [_ "The clipboard is empty."];
	return;
    } else {
	$w insert insert $sel;
    }
}

#If the filename passed as argument is a pathname
#leading to a file in the current working directory,
#return just the basename+extension. Otherwise
#return the argument.
proc MinimizeFileName {s} {
    set cwd [pwd];
    set sdir [file dirname $s]
    if {[string equal $cwd $sdir]} {
	return [file tail $s]
    } else {
	return $s;
    }
}


proc SelectInputFile {} {
    set InputFile [tk_getOpenFile];
    if {$InputFile == ""} {
	ShowMessage [_ "File selection aborted."]
	return
    } else { 
	.gen.ctr.if.ent delete 0 end
	.gen.ctr.if.ent insert 0 [MinimizeFileName $InputFile]
    }
}

proc SelectOutputFile {} {
    set OutputFile [tk_getSaveFile -initialfile [_ "SortOutput"]];
    if {$OutputFile == ""} {
	ShowMessage [_ "File selection aborted."]
	return
    } else { 
	.gen.ctr.of.ent delete 0 end
	.gen.ctr.of.ent insert 0 [MinimizeFileName $OutputFile]
    }
}

array set DefaultKeyBindings {
"<Control-c>" 		  AbortSort
"<Control-d>"             ToggleIPAD
"<Control-b>" 		  ToggleBalloonHelp
"<Control-j>"             ToggleIPAC
"<Control-k>"             DescribeKeyBindings
"<Control-w>"             ToggleIPAV
};

if {[string equal $tcl_platform(platform) "unix"]} {
    set DefaultKeyBindings(<Control-c>) AbortSort;
}

proc SetCommandGlosses {} {
    global CommandGlosses;

    set Pairs [list\
	   AbortSort "[_ "Abort the currently running msort job"]"\
	   ToggleIPAC "[_ "Toggle the IPA consonant entry widget."]"\
	   ToggleIPAV "[_ "Toggle the IPA vowel entry widget."]"\
		   ToggleBalloonHelp "[_ "Toggle Balloon Help"]"\
	   DescribeKeyBindings "[_ "Toggle Key Binding Help"]"\
	   ToggleIPAD "[_ "Toggle the IPA diacritic entry widget."]"]
    array set CommandGlosses $Pairs;
}

proc BindKeys {w {bl DefaultKeyBindings}} {
    upvar \#0 $bl kb;
    foreach KeySeq [array names kb] {
	bind $w $KeySeq $kb($KeySeq);
    }
}

proc DescribeKeyBindings {{bl DefaultKeyBindings}} {
    global CommandGlosses;
    upvar \#0 $bl kb;

    if {[PopupDown KeyBindings] ==1} {return}
    set po [CreateTextDisplay [_ "Key Bindings"] 88 6]
    BindKeys $po;
    set ::PopupList(KeyBindings) $po;

    #Invert array and get maximum length of gloss while we are at it.
    set MaxGlossLength 0;
    foreach KeySeq [array names kb] {
	set Gloss $CommandGlosses($kb($KeySeq));
	set GlossLength [string length $Gloss];
	if {$MaxGlossLength < $GlossLength} {set MaxGlossLength $GlossLength}
	set GlossToKeySeq($Gloss) $KeySeq;
    }
    #Now generate entries in alphabetical order of command glosses
    foreach Gloss [lsort [array names GlossToKeySeq]] {
	AppendToTextDisplay $po [format "%-*s\t%s\n" $MaxGlossLength $Gloss $GlossToKeySeq($Gloss)];
    }
}

set bhInfo(active) 1
proc balloonhelp_control {state} {
     global bhInfo
     if {$state} {
          set bhInfo(active) 1
     } else {
	balloonhelp_cancel
	set bhInfo(active) 0
     }
}

proc balloonhelp_for {win mesg} {
    global bhInfo
    set bhInfo($win) $mesg
    set ::bhOverlapP($win) 1; 
    bind $win <Enter> {+balloonhelp_pending %W}
    bind $win <Leave> {+balloonhelp_cancel}
}

proc balloonhelpd_for {win mesg} {
    global bhInfo
    set ::bhOverlapP($win) 0;
    set bhInfo($win) $mesg
    bind $win <Enter> {+balloonhelp_show %W}
}

proc balloonhelp_pending {win} {
     global bhInfo
     balloonhelp_cancel
     set bhInfo(pending) [after 1000 [list balloonhelp_show $win]]
}

proc balloonhelp_cancel {} {
    global bhInfo
    if { [info exists bhInfo(pending)]} {
	after cancel $bhInfo(pending)
	unset bhInfo(pending)
    }
    if {[winfo exists .balloonhelp]} {
	wm withdraw .balloonhelp
    }
}

proc balloonhelp_show {win} {
    global bhInfo;
    global bhOverlapP;
    if {$bhOverlapP($win)} {
	set Overlap 25;
    } else {
	set Overlap -10;
    }
    if {[winfo exists $win]} {
	if {$bhInfo(active)} {
	    .balloonhelp.info configure -text $bhInfo($win)
	    #Set abcissa
	    set MaxStringWidth 0;
	    foreach line [split $bhInfo($win) "\n"] {
		set StringWidth [font measure BalloonHelpFont -displayof .balloonhelp.info $line]
		if {$StringWidth > $MaxStringWidth} {
		    set MaxStringWidth $StringWidth;
		}
	    }
	    set ScreenWidth [winfo screenwidth $win]
	    set Width [winfo width $win];
	    set LeftEdge  [winfo rootx $win];
	    set RightEdge [expr $LeftEdge + $Width];
	    if {$ScreenWidth - $RightEdge < $MaxStringWidth} { 
		if {$LeftEdge > $MaxStringWidth} {
		    set x [expr $LeftEdge - $MaxStringWidth + $Overlap];
		} else {
		    if {$ScreenWidth - $MaxStringWidth > 0} {
			set x [expr $RightEdge - $MaxStringWidth];
		    } else {
			set x [expr $ScreenWidth - $MaxStringWidth];
		    }
		}
	    } else {
		set x [expr $RightEdge - $Overlap];
	    }
	    #Set ordinate
	    set Height [winfo height $win];
	    set TopEdge [winfo rooty $win];
	    set y [expr $TopEdge + ($Height/2)];
	    wm geometry .balloonhelp +$x+$y
	    wm deiconify .balloonhelp
	    raise .balloonhelp
	}
    }
    if {[info exist bhInfo(pending)]} {
	unset bhInfo(pending)
    }
}

proc ToggleBalloonHelp {} {
    global BalloonHelpP;
    global BalloonHelpIndex;
    global m;

    if {$BalloonHelpP} {
	set BalloonHelpP 0;
	balloonhelp_control 0
	ShowMessage [_ "Irritating Balloon Help Disabled"];
	$m.configure entryconfigure $BalloonHelpIndex -label [_ "Enable Balloon Help"];
    } else {
	set BalloonHelpP 1;
	balloonhelp_control 1
	ShowMessage [_ "Balloon Help Enabled"];
	$m.configure entryconfigure $BalloonHelpIndex -label [_ "Disable Irritating Balloon Help"];
    }
}

proc new_dialog_create {class {win "auto"}} {
    if {$win == "auto"} {
        set count 0
        set win ".ndialog[incr count]"
        while {[winfo exists $win]} {
            set win ".ndialog[incr count]"
        }
    }
    toplevel $win -class $class;
    frame $win.info
    pack $win.info -expand yes -fill both -padx 4 -pady 4
    wm title $win $class
    wm group $win .

    after idle [format {
        update idletasks
        wm minsize %s [winfo reqwidth %s] [winfo reqheight %s]
    } $win $win $win]

    return $win
}

# The following code is taken from the Efftcl library by Mark Harrison and
# Michael McLennan, copyrighted by Mark Harrison and Lucent Technologies, available
# from http://www.awprofessional.com/content/images/0201634740/sourcecode/efftcl.zip.
# As the authors explicitly give permission to "steal the code for your own applications"
# the relevant portions are included here so as not to require the user to install
# to install the library. If you install the library, remove the following and
# uncomment the line "#package require Efftcl" by deleting the crosshatch.

#  Effective Tcl/Tk Programming
#    Mark Harrison, DSC Communications Corp.
#    Michael McLennan, Bell Labs Innovations for Lucent Technologies
#    Addison-Wesley Professional Computing Series
# ======================================================================
#  Copyright (c) 1996-1997  Lucent Technologies Inc. and Mark Harrison
# ======================================================================

proc dialog_create {class {win "auto"}} {
    if {$win == "auto"} {
        set count 0
        set win ".dialog[incr count]"
        while {[winfo exists $win]} {
            set win ".dialog[incr count]"
        }
    }
    toplevel $win -class $class -background \#2200DD;

    frame $win.info
    pack $win.info -expand yes -fill both -padx 4 -pady 4
    frame $win.sep -height 2 -borderwidth 1 -relief sunken
    pack $win.sep -fill x -pady 4
    frame $win.controls
    pack $win.controls -fill x -padx 4 -pady 1

    wm title $win $class
    wm group $win .

    after idle [format {
        update idletasks
        wm minsize %s [winfo reqwidth %s] [winfo reqheight %s]
    } $win $win $win]

    return $win
}

proc dialog_info {win} {
    return "$win.info"
}

proc dialog_controls {win} {
    return "$win.controls"
}

proc dialog_wait {win varName} {
    dialog_safeguard $win

    set x [expr [winfo rootx .]+50]
    set y [expr [winfo rooty .]+50]
    wm geometry $win "+$x+$y"

    wm deiconify $win
    grab set $win

    vwait $varName

    grab release $win
    wm withdraw $win
}

bind modalDialog <ButtonPress> {
    wm deiconify %W
    raise %W
}
proc dialog_safeguard {win} {
    if {[lsearch [bindtags $win] modalDialog] < 0} {
        bindtags $win [linsert [bindtags $win] 0 modalDialog]
    }
}

proc CreateTextDisplay {title width height {bg "\#96C4A3"} {fg "\#000000"} } {
    set top [new_dialog_create Textdisplay]
    wm title $top $title
    set info [dialog_info $top]
    scrollbar $info.sbar -command "$info.text yview" 
    bind $info.sbar <<B3>> "ScrollbarMoveBigIncrement $info.sbar 0.20 %x %y"
    pack $info.sbar -side right -fill y
    text $info.text -height $height -width $width -font UserTechnicalTextFont \
	-wrap word -yscrollcommand "$info.sbar set"  \
	-background $bg -foreground $fg -exportselection 1;
    pack $info.text -side left -expand yes -fill both
    $info.text configure -state disabled
    return $top
}

proc AppendToTextDisplay {top mesg} {
    set info [dialog_info $top]
    $info.text configure -state normal
    $info.text insert end $mesg
    $info.text configure -state disabled
}

set linkNum 0;
proc AppendLinkToTextDisplay {top mesg LinkCode} {
    global linkNum
    set info [dialog_info $top]
    $info.text configure -state normal
    set tag "link[incr linkNum]"
    bind $info.text <ButtonPress> break
    $info.text insert end $mesg [list body $tag]
    $info.text tag configure $tag -foreground red -underline 1
    $info.text tag bind $tag <Enter> \
        "$info.text tag configure $tag -foreground blue"
    $info.text tag bind $tag <Leave> \
        "$info.text tag configure $tag -foreground red"
    $info.text tag bind $tag <ButtonPress> \
        "$LinkCode"
    $info.text configure -state disabled
}

proc About {} {
    global Version;
    global PackageVersion;
    if {[PopupDown About] ==1} {return}
    set po [CreateTextDisplay [_ "About"] 55 8];
    BindKeys $po
    set ::PopupList(About) $po
    AppendToTextDisplay $po [format [_ "This is msg version %s."] $Version];
    AppendToTextDisplay $po [format [_ " It is part of the msort package, version %s."] $PackageVersion];
    AppendToTextDisplay $po " It is a graphical front end to Msort, a sophisticated sorting program. In order for it to be of any use, you must have Msort installed.";
    AppendToTextDisplay $po " You can obtain both Msort and Msg from:\n";
    AppendLinkToTextDisplay $po "http://www.billposer.org/Software/msort.html" {ShowWebPage http://www.billposer.org/Software/msort.html};
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po "Both programs were written by Bill Poser (billposer@alum.mit.edu).";
    AppendToTextDisplay $po "\n";
}

proc BugReports {} {
    global Version;
    global tcl_platform;

    if {[PopupDown BugReports] ==1} {return}
    set po [CreateTextDisplay [_ "Bug Reports"] 50 21];
    BindKeys $po
    set ::PopupList(BugReports) $po
    AppendToTextDisplay $po [_ "Report bugs to: billposer@alum.mit.edu.\n"];
    AppendToTextDisplay $po [_ "Please include the following information:\n\n"];
    AppendToTextDisplay $po [_ "\tWhat version of the program are you using?\n"];
    AppendToTextDisplay $po [format [_ "\t\t(This is version %s.)\n\n"] $Version];
    AppendToTextDisplay $po [_ "\tWhat version of msort are you using?\n"];
    AppendToTextDisplay $po [format [_ "\t\t(Version located: %s.)\n\n"] [FindoutMsortVersion]];
    AppendToTextDisplay $po [_ "\tWhat operating system are you running?.\n"];
    set OS $tcl_platform(os);
    if {$OS == "Linux"} {set OS "GNU/Linux"};
    AppendToTextDisplay $po [format [_ "\t\t(This is %s  %s.)\n\n"] $OS  $tcl_platform(osVersion)];
    AppendToTextDisplay $po [_ "\tWhat window system are you using?.\n"];
    AppendToTextDisplay $po [format [_ "\t\t(This is %s.)\n\n"] $::WindowSystem]
    AppendToTextDisplay $po [_ "\tWhat version of tcl/tk are you using?.\n"];
    AppendToTextDisplay $po [format [_ "\t\t(This is version %s.)\n\n"] [info patchlevel]];
    AppendToTextDisplay $po [_ "Bug reports may be sent in any language that I can read without too much trouble or am trying to learn or improve. These include:\n\n"];
    AppendLinkToTextDisplay $po [_ "\tCatalan"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=cat};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tDakelh (Carrier)"] {ShowWebPage http://ydli.org}
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tDutch"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=nld};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tEnglish"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=eng};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tEsperanto"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=epo};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tFrench"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=fra};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tGerman"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=deu};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tItalian"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=ita};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tJapanese"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=jpn};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tKazakh"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=kaz};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tKorean"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=kor};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tLatin"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=lat};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tPortuguese"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=por};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tSpanish"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=spa};
    AppendToTextDisplay $po "\n";
    AppendLinkToTextDisplay $po [_ "\tTurkish"] {ShowWebPage http://www.ethnologue.com/show_language.asp?code=tur};
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po "Please note that in many cases although I can understand the language my ability to respond in it may be limited.\n";
    AppendToTextDisplay $po "\n";
}

proc HowTo {} {
    if {[PopupDown HowTo] ==1} {return}
    set po [CreateTextDisplay [_ "HowTo"] 55 20];
    BindKeys $po
    set ::PopupList(HowTo) $po
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "This program provides a graphical interface to msort, a sophisticated and flexible sorting program. It helps you to choose the right options to obtain the output that you desire. For full details,
see the Msort Reference Manual, which provides a description of the sorting process as well as details of the program.\n"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "The display consists of a number of regions. The first few are the same on every page. At the top (unless you are using a Macintosh) is the menubar. Next (or first, on a Macintosh) comes the Message REgion. This is where Msg prints messages to you. Immediately below the Message Region is a row of two or more buttons used to select the page to work on. What follows depends on whether you are working on the General page or on a key-specific page.\n"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "Msg provides two kinds of pages. The first page that you see is for general settings, that is, settings that are not particular to a specific key. You can always get back to this page by clicking on the General button in the region just below the message region. The General button is always the leftmost button. Following the General button are one or more buttons labelled with key numbers, e.g. \`Key 1\'. When Msg starts up, one such button is already present since you cannot sort without at least one key. If you need to use more than one key, you can add key pages by clicking on the Add Key button in the menubar. If you decide that you don't need that many keys, you can remove keys by clicking on the Remove Last Key button. You can tell which key you are working on by looking at the label on the highlighted page selection button.\n"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "The General page is used to set parameters that are not key-specific. You must provide the name of the input file (the file whose contents you wish to sort) and the name of the file in which to put the results of the sort. You must also decide how the input is to be divided into records and how each record is to be divided into fields. This page also provides the option of inverting the overall order of the output.\n"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "The key-specific pages allow you to specify how to locate the field that contains the key and to determine various details of the sort, such as what sort order to use and whether to sort lexicographically or numerically. Msort provides a number of useful but unusual options, so you may wish to explore what it provides.\n"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "Information about particular aspects of a page comes in two main forms. First, if you leave the mouse pointer over a region for a short time, a help balloon will pop up. Help balloons provide a short explanation of the function of the region. (Once you are familiar with the program, you may find help balloons irritating. You can disable them by means of a command on the Help menu.) Second, in many cases more detailed information is available by clicking the right mouse button. (On a MacIntosh, clicking while depressing the Control key should be equivalent.) Right-clicking again in the same region will remove the popup.\n"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "When you have set the necessary parameters, you can execute the sort by pressing the sort button on the menubar. If you are familiar with the msort command line, you can see what command Msg will execute by selecting the command Show Command Line on the File menu. If you want to save the command line for future use, you can copy it to a file by selecting the command Save Command Line from the File Menu.\n"];
    AppendToTextDisplay $po "\n";
}

proc ShowGPL {} {
    if {[PopupDown ShowGPL] ==1} {return}
    set po [CreateTextDisplay [_ "License"] 70 24]
    BindKeys $po
    set ::PopupList(ShowGPL) $po
    AppendToTextDisplay $po [format "%s%s" [format "%s\n\t%s\n" [_ "For this license in your language see:"] [_ "http://www.gnu.org/copyleft/gpl.html"]]  "\
\

		    GNU GENERAL PUBLIC LICENSE\

   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION
\

  0. This License applies to any program or other work which contains
a notice placed by the copyright holder saying it may be distributed
under the terms of this General Public License.  The \"Program\", below,
refers to any such program or work, and a \"work based on the Program\"
means either the Program or any derivative work under copyright law:
that is to say, a work containing the Program or a portion of it,
either verbatim or with modifications and/or translated into another
language.  (Hereinafter, translation is included without limitation in
the term \"modification\".)  Each licensee is addressed as \"you\".
\

Activities other than copying, distribution and modification are not
covered by this License; they are outside its scope.  The act of
running the Program is not restricted, and the output from the Program
is covered only if its contents constitute a work based on the
Program (independent of having been made by running the Program).
Whether that is true depends on what the Program does.
\

  1. You may copy and distribute verbatim copies of the Program's
source code as you receive it, in any medium, provided that you
conspicuously and appropriately publish on each copy an appropriate
copyright notice and disclaimer of warranty; keep intact all the
notices that refer to this License and to the absence of any warranty;
and give any other recipients of the Program a copy of this License
along with the Program.
\

You may charge a fee for the physical act of transferring a copy, and
you may at your option offer warranty protection in exchange for a fee.
\

  2. You may modify your copy or copies of the Program or any portion
of it, thus forming a work based on the Program, and copy and
distribute such modifications or work under the terms of Section 1
above, provided that you also meet all of these conditions:
\

    a) You must cause the modified files to carry prominent notices
    stating that you changed the files and the date of any change.
\

    b) You must cause any work that you distribute or publish, that in
    whole or in part contains or is derived from the Program or any
    part thereof, to be licensed as a whole at no charge to all third
    parties under the terms of this License.
\

    c) If the modified program normally reads commands interactively
    when run, you must cause it, when started running for such
    interactive use in the most ordinary way, to print or display an
    announcement including an appropriate copyright notice and a
    notice that there is no warranty (or else, saying that you provide
    a warranty) and that users may redistribute the program under
    these conditions, and telling the user how to view a copy of this
    License.  (Exception: if the Program itself is interactive but
    does not normally print such an announcement, your work based on
    the Program is not required to print an announcement.)
\

These requirements apply to the modified work as a whole.  If
identifiable sections of that work are not derived from the Program,
and can be reasonably considered independent and separate works in
themselves, then this License, and its terms, do not apply to those
sections when you distribute them as separate works.  But when you
distribute the same sections as part of a whole which is a work based
on the Program, the distribution of the whole must be on the terms of
this License, whose permissions for other licensees extend to the
entire whole, and thus to each and every part regardless of who wrote it.
\

Thus, it is not the intent of this section to claim rights or contest
your rights to work written entirely by you; rather, the intent is to
exercise the right to control the distribution of derivative or
collective works based on the Program.
\

In addition, mere aggregation of another work not based on the Program
with the Program (or with a work based on the Program) on a volume of
a storage or distribution medium does not bring the other work under
the scope of this License.
\

  3. You may copy and distribute the Program (or a work based on it,
under Section 2) in object code or executable form under the terms of
Sections 1 and 2 above provided that you also do one of the following:
\

    a) Accompany it with the complete corresponding machine-readable
    source code, which must be distributed under the terms of Sections
    1 and 2 above on a medium customarily used for software interchange; or,
\

    b) Accompany it with a written offer, valid for at least three
    years, to give any third party, for a charge no more than your
    cost of physically performing source distribution, a complete
    machine-readable copy of the corresponding source code, to be
    distributed under the terms of Sections 1 and 2 above on a medium
    customarily used for software interchange; or,
\

    c) Accompany it with the information you received as to the offer
    to distribute corresponding source code.  (This alternative is
    allowed only for noncommercial distribution and only if you
    received the program in object code or executable form with such
    an offer, in accord with Subsection b above.)
\

The source code for a work means the preferred form of the work for
making modifications to it.  For an executable work, complete source
code means all the source code for all modules it contains, plus any
associated interface definition files, plus the scripts used to
control compilation and installation of the executable.  However, as a
special exception, the source code distributed need not include
anything that is normally distributed (in either source or binary
form) with the major components (compiler, kernel, and so on) of the
operating system on which the executable runs, unless that component
itself accompanies the executable.
\

If distribution of executable or object code is made by offering
access to copy from a designated place, then offering equivalent
access to copy the source code from the same place counts as
distribution of the source code, even though third parties are not
compelled to copy the source along with the object code.
\

  4. You may not copy, modify, sublicense, or distribute the Program
except as expressly provided under this License.  Any attempt
otherwise to copy, modify, sublicense or distribute the Program is
void, and will automatically terminate your rights under this License.
However, parties who have received copies, or rights, from you under
this License will not have their licenses terminated so long as such
parties remain in full compliance.
\

  5. You are not required to accept this License, since you have not
signed it.  However, nothing else grants you permission to modify or
distribute the Program or its derivative works.  These actions are
prohibited by law if you do not accept this License.  Therefore, by
modifying or distributing the Program (or any work based on the
Program), you indicate your acceptance of this License to do so, and
all its terms and conditions for copying, distributing or modifying
the Program or works based on it.
\

  6. Each time you redistribute the Program (or any work based on the
Program), the recipient automatically receives a license from the
original licensor to copy, distribute or modify the Program subject to
these terms and conditions.  You may not impose any further
restrictions on the recipients' exercise of the rights granted herein.
You are not responsible for enforcing compliance by third parties to
this License.
\

  7. If, as a consequence of a court judgment or allegation of patent
infringement or for any other reason (not limited to patent issues),
conditions are imposed on you (whether by court order, agreement or
otherwise) that contradict the conditions of this License, they do not
excuse you from the conditions of this License.  If you cannot
distribute so as to satisfy simultaneously your obligations under this
License and any other pertinent obligations, then as a consequence you
may not distribute the Program at all.  For example, if a patent
license would not permit royalty-free redistribution of the Program by
all those who receive copies directly or indirectly through you, then
the only way you could satisfy both it and this License would be to
refrain entirely from distribution of the Program.
\

If any portion of this section is held invalid or unenforceable under
any particular circumstance, the balance of the section is intended to
apply and the section as a whole is intended to apply in other
circumstances.
\

It is not the purpose of this section to induce you to infringe any
patents or other property right claims or to contest validity of any
such claims; this section has the sole purpose of protecting the
integrity of the free software distribution system, which is
implemented by public license practices.  Many people have made
generous contributions to the wide range of software distributed
through that system in reliance on consistent application of that
system; it is up to the author/donor to decide if he or she is willing
to distribute software through any other system and a licensee cannot
impose that choice.
\

This section is intended to make thoroughly clear what is believed to
be a consequence of the rest of this License.
\

  8. If the distribution and/or use of the Program is restricted in
certain countries either by patents or by copyrighted interfaces, the
original copyright holder who places the Program under this License
may add an explicit geographical distribution limitation excluding
those countries, so that distribution is permitted only in or among
countries not thus excluded.  In such case, this License incorporates
the limitation as if written in the body of this License.
\

  9. The Free Software Foundation may publish revised and/or new versions
of the General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.
\

Each version is given a distinguishing version number.  If the Program
specifies a version number of this License which applies to it and \"any
later version\", you have the option of following the terms and conditions
either of that version or of any later version published by the Free
Software Foundation.  If the Program does not specify a version number of
this License, you may choose any version ever published by the Free Software
Foundation.
\

  10. If you wish to incorporate parts of the Program into other free
programs whose distribution conditions are different, write to the author
to ask for permission.  For software which is copyrighted by the Free
Software Foundation, write to the Free Software Foundation; we sometimes
make exceptions for this.  Our decision will be guided by the two goals
of preserving the free status of all derivatives of our free software and
of promoting the sharing and reuse of software generally.
\

			    NO WARRANTY
\

  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.
\

  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGES."]
}

set MainKeyFieldIdentBH [_ "How is the field used for this key identified? One possibility\nis by its position in the record. A second is by tag, that is,\na sequence of characters with which the field begins. The third\nis by a range of characters, that is, the characters in certain\npositions constitute the key."]
set DisabledKeyFieldIdentBH [_ "This area is used to specify how to select the key field.\nIt is currently disabled since you have chosen\nto use the entire record as the sort key."] 

proc DisableKeyFieldID {hd} {
    $hd.key.ki.pos.rdb1 configure -state disabled
    $hd.key.ki.pos.rdb2 configure -state disabled
    $hd.key.ki.pos.rdb3 configure -state disabled
    $hd.key.ki.pos.posent.fldent1 configure -state disabled
    $hd.key.ki.pos.tagent configure -state disabled
    $hd.key.ki.pos.firstent configure -state disabled
    $hd.key.ki.pos.lastent configure -state disabled
    balloonhelp_for $hd.key.ki $::DisabledKeyFieldIdentBH;
    balloonhelp_for $hd.key.ki.pos.posent.fldent1 $::DisabledKeyFieldIdentBH;
    balloonhelp_for $hd.key.ki.pos.tagent $::DisabledKeyFieldIdentBH;
    balloonhelp_for $hd.key.ki.pos.firstent $::DisabledKeyFieldIdentBH;
    balloonhelp_for $hd.key.ki.pos.lastent $::DisabledKeyFieldIdentBH;
}

proc EnableKeyFieldID {hd} {
    $hd.key.ki.pos.rdb1 configure -state normal
    $hd.key.ki.pos.rdb2 configure -state normal
    $hd.key.ki.pos.rdb3 configure -state normal
    balloonhelp_for $hd.key.ki $::MainKeyFieldIdentBH;
    balloonhelp_for $hd.key.ki.pos.rdb1 [_ "A position specification consists of a single position or\na contiguous range of positions specified by their beginning\nand end. A position consists of a field number together with\na character offset. Field numbers, but not character offsets,\nmay be negative, in which case the count is from the right."]
    balloonhelp_for $hd.key.ki.pos.posent.fldent1 [_ "Enter the field number here."]
    balloonhelp_for $hd.key.ki.pos.posent.fldent2 [_ "Enter the field number here."]
    balloonhelp_for $hd.key.ki.pos.posent.chrent1 [_ "Enter the character offset here."]
    balloonhelp_for $hd.key.ki.pos.posent.chrent2 [_ "Enter the character offset here."]
    balloonhelp_for $hd.key.ki.pos.tagent [_ "Enter the tag regular expression here."]
    balloonhelp_for $hd.key.ki.pos.firstent [_ "Enter the index of the first character in the range\n(starting from 1) here. Negative numbers may be used\nto count from the end."]
    balloonhelp_for $hd.key.ki.pos.lastent [_ "Enter the index of the last character in the range\n(starting from 1) here. Negative numbers may be used\nto count from the end."]
 }

proc PackGen {} {

    set genxpadl {12 8}
    set genxpadr {8 12}
    set genypadi 5
    set genypadt {8 5}
    set genypadb {5 8}

    grid forget .gen.ctr.rp
    grid forget .gen.ctr.wr
    grid forget .gen.ctr.parckb
    grid forget .gen.ctr.gi
    grid forget .gen.ctr.eol
    grid forget .gen.ctr.miscckb
    grid forget .gen.ctr.bmp
    grid forget .gen.ctr.pua
    grid forget .gen.ctr.unickb
    grid forget .gen.ctr.al
    grid forget .gen.ctr.alckb
    grid forget .gen.ctr.if
    grid forget .gen.ctr.of

    grid forget .gen.ctr.parf
    grid forget .gen.ctr.miscf
    grid forget .gen.ctr.alf
    grid forget .gen.ctr.unif
    grid forget .gen.ctr.iff
    grid forget .gen.ctr.off

    set Row 0;

    if {$::RecordParseDisplayedP} {
	grid .gen.ctr.rp  -row $Row -column 0 -sticky wens -pady $genypadt -padx $genxpadl
	grid .gen.ctr.wr  -row $Row -column 1 -sticky wens -pady $genypadt -padx $genxpadr
	grid .gen.ctr.parckb -row $Row -column 2 -sticky e -pady $genypadb -padx $genxpadl
    } else {
	grid .gen.ctr.parf -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    }
    incr Row
 
    if {$::UnicodeDisplayedP} {
	grid .gen.ctr.bmp -row $Row -column 0 -sticky wens -pady $genypadi -padx $genxpadl
	grid .gen.ctr.pua -row $Row -column 1 -sticky wens -pady $genypadi -padx $genxpadr
	grid .gen.ctr.unickb -row $Row -column 2 -sticky e -pady $genypadb -padx $genxpadl
    } else {
	grid .gen.ctr.unif -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    }
    incr Row

    if {$::AlgorithmDisplayedP} {
	grid .gen.ctr.al  -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl
	grid .gen.ctr.alckb -row $Row -column 2 -sticky e -pady $genypadb -padx $genxpadl
    } else {
	grid .gen.ctr.alf -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    }
    incr Row

    if {$::MiscOptionsDisplayedP} {
	grid .gen.ctr.gi  -row $Row -column 0 -sticky wens -pady $genypadi -padx $genxpadl
	grid .gen.ctr.eol -row $Row -column 1 -sticky wens -pady $genypadi -padx $genxpadr
	grid .gen.ctr.miscckb -row $Row -column 2 -sticky e -pady $genypadb -padx $genxpadl
    } else {
	grid .gen.ctr.miscf -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    }
    incr Row

    if {$::InputFileDisplayedP} {
	grid .gen.ctr.if  -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    } else {
	grid .gen.ctr.iff -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    }
    incr Row

    if {$::OutputFileDisplayedP} {
	grid .gen.ctr.of  -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    } else {
	grid .gen.ctr.off -row $Row -column 0 -sticky wens -pady $genypadb -padx $genxpadl \
	    -columnspan 3
    }
    pack  .gen.ctr -side top -expand 1 -fill x -padx 6
    ConfigureGenOne
    ConfigureGenTwo
}

proc OldPackGen {} {
    pack forget .gen.ctr
    pack forget .gen.sep3

    pack  .gen.ctr -side top -expand 1 -fill x -padx 6
    pack  .gen.sep3 -side top -expand 1 -fill x -padx 6
}

proc PackKeyPage {hd} {
    set kpxpd 5
    set kpypd 2
    set kpfypd 4

    pack forget $hd.misco
    pack forget $hd.exc
    pack forget $hd.sub
    pack forget $hd.sos
    pack forget $hd.so
    pack forget $hd.st
    pack forget $hd.key

    pack forget $hd.miscof
    pack forget $hd.excf
    pack forget $hd.subf
    pack forget $hd.sosf
    pack forget $hd.sof
    pack forget $hd.stf
    pack forget $hd.keyf

    if {$::KeyInfo(MiscellaneousOptionsDisplayedP,$hd)} {
	pack $hd.misco -side bottom -expand y -fill x -pady $kpypd -padx $kpxpd
    } else {
	pack $hd.miscof -side bottom -expand y -fill x -pady $kpfypd -padx $kpxpd
    }

    if {$::KeyInfo(SubstitutionsDisplayedP,$hd)} {
	pack $hd.sub -side bottom -expand 0 -fill x -pady $kpypd -padx $kpxpd
    } else {
	pack $hd.subf -side bottom -expand 0 -fill x -pady $kpfypd -padx $kpxpd
    }
    if {$::KeyInfo(ExclusionsDisplayedP,$hd)} {
	pack $hd.exc -side bottom -expand 0 -fill x -pady $kpypd -padx $kpxpd
    } else {
	pack $hd.excf -side bottom -expand 0 -fill x -pady $kpfypd -padx $kpxpd
    }

    if {$::KeyInfo(SortOrderSpecType,$hd) == 2} {
	if {$::KeyInfo(SortOrderDisplayedP,$hd)} {
	    pack $hd.sos   -expand y -fill x -side bottom -pady $kpypd -padx $kpxpd
	}
    }

    set SortType $::KeyInfo(SortType,$hd)
    if {$SortType == 1 || $SortType == 2 || $SortType == 7} {
	if {$::KeyInfo(SortOrderDisplayedP,$hd)} {
	    pack $hd.so  -side bottom -expand 0 -fill both -pady $kpypd -padx $kpxpd
	} else {
	    pack $hd.sof  -side bottom -expand 0 -fill both -pady $kpfypd -padx $kpxpd
	}
    }

    if {$::KeyInfo(SortTypeDisplayedP,$hd)} {
	pack $hd.st  -side bottom -expand 0 -fill both -pady $kpypd -padx $kpxpd
    } else {
	pack $hd.stf  -side bottom -expand 0 -fill both -pady $kpfypd -padx $kpxpd
    }
    if {$::KeyInfo(KeyFieldIdentificationDisplayedP,$hd)} {
	pack $hd.key -side bottom -expand 0 -fill both -pady $kpypd -padx $kpxpd
    } else {
	pack $hd.keyf -side bottom -expand 0 -fill both -pady $kpfypd -padx $kpxpd
    }
}

proc ValidatePositionField {w s type action} {
    if {$s == 0} {
	ShowMessage [_ "There is no field 0."]
	focus $w;
	return 0;
    }
    if {[string equal $type focusout]} { 
	if {[string length $s] == 0} {
	    ShowMessage [_ "You must specify a position."]
	    focus $w;
	    return 0;
	}
	if {![regexp {^-?[[:digit:]]+$} $s]} {
	    ShowMessage [_ "The field number must be an integer."]
	    focus $w;
	    return 0
	}
    }
    if {[string equal $type key]} {
	if {![regexp {^-?[[:digit:]]*$} $s]} {
	    ShowMessage [_ "The field number must be an integer."]
	    focus $w;
	    return 0
	}
    }
    ClearMessageWindow
    return 1;
}

proc ValidatePositionChar {w s type action} {
    if {$s == 0} {
	ShowMessage [_ "There is no character 0."]
	focus $w;
	return 0;
    }
    if {[string equal $type focusout]} { 
	if {[string length $s] == 0} {
	    ShowMessage [_ "You must specify a character offset."]
	    focus $w;
	    return 0;
	}
	if {![regexp {^-?[[:digit:]]+$} $s]} {
	    ShowMessage [_ "The character offset must be a positive integer."]
	    focus $w;
	    return 0
	}
    }
    if {[string equal $type key]} {
	if {![regexp {^-?[[:digit:]]*$} $s]} {
	    ShowMessage [_ "The character offset must be a positive integer."]
	    focus $w;
	    return 0
	}
    }
    ClearMessageWindow
    return 1;
}

proc ValidateRange {w s type action} {
    if {[string equal $type focusout]} { 
	if {[string length $s] == 0} {
	    ShowMessage [_ "You must specify a position."]
	    focus $w;
	    return 0;
	}
	if {![regexp {^-?[[:digit:]]+$} $s]} {
	    ShowMessage [_ "The position must be an integer."]
	    focus $w;
	    return 0
	}
    }
    if {[string equal $type key]} {
	if {![regexp {^-?[[:digit:]]*$} $s]} {
	    ShowMessage [_ "The position must be an integer."]
	    focus $w;
	    return 0
	}
    }
    ClearMessageWindow
    return 1;
}


proc SetupKey {hd} {

    set ::KeyInfo(SortOrderSpecType,$hd) 0;
    set ::KeyInfo(ExclusionsP,$hd) 0;
    set ::KeyInfo(SubstitutionsP,$hd) 0;
    set ::KeyInfo(KeySelection,$hd) 0;
    set ::KeyInfo(OptionalP,$hd) 0
    set ::KeyInfo(LocalInversionP,$hd) 0
    set ::KeyInfo(ReverseKeyP,$hd) 0;
    set ::KeyInfo(FoldCaseP,$hd) 0;
    set ::KeyInfo(TurkicFoldCaseP,$hd) 0;
    set ::KeyInfo(StripDiacriticsP,$hd) 0;
    set ::KeyInfo(ConvertEnclosuresP,$hd) 0;
    set ::KeyInfo(ConvertStylisticP,$hd) 0;
    set ::KeyInfo(SortType,$hd) 1;
    set ::KeyInfo(DateFormat,$hd) "y-m-d";

    set ::KeyInfo(ExclusionsDisplayedP,$hd) $::ExclusionsDisplayedP;
    set ::KeyInfo(KeyFieldIdentificationDisplayedP,$hd) $::KeyFieldIdentificationDisplayedP;
    set ::KeyInfo(MiscellaneousOptionsDisplayedP,$hd) $::MiscellaneousOptionsDisplayedP;
    set ::KeyInfo(SortOrderDisplayedP,$hd) $::SortOrderDisplayedP;
    set ::KeyInfo(SortTypeDisplayedP,$hd) $::SortTypeDisplayedP;
    set ::KeyInfo(SubstitutionsDisplayedP,$hd) $::SubstitutionsDisplayedP;

    set FileNameEntryWidth 50

    #How to identify the key field
    set mrelief solid
    set mborder 1
    frame $hd -bg $::ColorSpecs(Backdrop,Background);

    if {$::UseCheckbuttonP} {
	frame       $hd.keyf -relief $mrelief -border $mborder
	label  $hd.keyf.lab  -text [_ "How to Identify the Key Field"]   -anchor w -font HeaderFont
	checkbutton $hd.keyf.ckb -variable KeyInfo(KeyFieldIdentificationDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" 
	pack $hd.keyf.lab -side left -expand 0 -fill y -anchor w
	pack $hd.keyf.ckb -side right -expand 1 -fill y -anchor e
    } else {
	frame       $hd.keyf -relief $mrelief -border $mborder
	checkbutton $hd.keyf.ckb -variable KeyInfo(KeyFieldIdentificationDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" \
	    -text [_ "How to Identify the Key Field"] -anchor w -font HeaderFont -indicatoron 0 \
	    -activeforeground red 	-activebackground navajowhite
	pack $hd.keyf.ckb -side right -expand 1 -fill both -anchor e
    }

    frame	$hd.key -relief $mrelief -border $mborder -bg $::ColorSpecs(Default,Background);
    checkbutton $hd.key.ckb -variable KeyInfo(KeyFieldIdentificationDisplayedP,$hd) \
	-onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
    frame       $hd.key.ki
    frame $hd.key.ki.pos
    label       $hd.key.ki.title  -text [_ "How To Identify the Key Field"] -anchor w \
	 -font HeaderFont
    radiobutton $hd.key.ki.pos.rdb1 \
	-variable KeyInfo(KeySelection,$hd) -value 0 \
	-text [_ "position"]  -anchor w
    radiobutton $hd.key.ki.pos.rdb2 \
	-variable KeyInfo(KeySelection,$hd) -value 1 \
	-text [_ "tag"]  -anchor w
    radiobutton $hd.key.ki.pos.rdb3 \
	-variable KeyInfo(KeySelection,$hd) -value 2 \
	-text [_ "range"]  -anchor w

    frame $hd.key.ki.pos.posent;
    entry       $hd.key.ki.pos.posent.fldent1   -width 2 -font UserPlainTextFont -validate all \
	-validatecommand {ValidatePositionField %W %P %V %d}
    entry       $hd.key.ki.pos.posent.fldent2   -width 2 -font UserPlainTextFont -validate all \
	-validatecommand {ValidatePositionField %W %P %V %d}
    entry       $hd.key.ki.pos.posent.chrent1   -width 2 -font UserPlainTextFont -validate all \
	-validatecommand {ValidatePositionChar %W %P %V %d}
    entry       $hd.key.ki.pos.posent.chrent2   -width 2 -font UserPlainTextFont -validate all \
	-validatecommand {ValidatePositionChar %W %P %V %d}

    pack $hd.key.ki.pos.posent.fldent1 -side left -padx {0 1}
    pack $hd.key.ki.pos.posent.chrent1 -side left -padx {2 6}
    pack $hd.key.ki.pos.posent.fldent2 -side left -padx {6 1}
    pack $hd.key.ki.pos.posent.chrent2 -side left -padx {2 3}

    entry       $hd.key.ki.pos.tagent   -width 8 -font UserTechnicalTextFont
    entry       $hd.key.ki.pos.firstent -width 4 -font UserPlainTextFont -validate all \
	-validatecommand {ValidateRange %W %P %V %d}
    entry       $hd.key.ki.pos.lastent  -width 4 -font UserPlainTextFont -validate all \
	-validatecommand {ValidateRange %W %P %V %d}
    bind $hd.key.ki.pos.tagent <FocusIn> "SetInsertionTargets $hd.key.ki.pos.tagent"

    $hd.key.ki.pos.posent.fldent1 insert end $::KeyCount;
    $hd.key.ki.pos.posent.fldent2 insert end $::KeyCount;
    bind $hd.key.ki.pos.rdb1 <<B3>> ExplainPositionID;
    bind $hd.key.ki.pos.rdb2 <<B3>> ExplainTagID;
    bind $hd.key.ki.pos.rdb3 <<B3>> ExplainRangeID;
    $hd.key.ki.pos.rdb1 select
    $hd.key.ki.pos.posent.fldent1 configure -state normal
    $hd.key.ki.pos.posent.fldent2 configure -state normal
    $hd.key.ki.pos.posent.chrent1 configure -state normal
    $hd.key.ki.pos.posent.chrent2 configure -state normal
    $hd.key.ki.pos.tagent configure -state disabled
    $hd.key.ki.pos.firstent configure -state disabled
    $hd.key.ki.pos.lastent configure -state disabled
    grid $hd.key.ki.pos.rdb1   -row 0 -column 0 -sticky w -pady 3
    grid $hd.key.ki.pos.posent -row 0 -column 1  -columnspan 2 -sticky w -pady 3 -padx {14 0}
    grid $hd.key.ki.pos.rdb2   -row 1 -column 0 -sticky w -pady 3
    grid $hd.key.ki.pos.tagent -row 1 -column 1  -sticky w -pady 3 -padx {14 0}
    grid $hd.key.ki.pos.rdb3   -row 2 -column 0 -sticky w -pady 3
    grid $hd.key.ki.pos.firstent -row 2 -column 1  -sticky w -pady 3 -padx {14 0}
    grid $hd.key.ki.pos.lastent -row 2 -column 2  -sticky w -pady 3
    pack $hd.key.ki.title -side top -anchor w -padx 5
    pack $hd.key.ki.pos -side bottom -anchor w -padx 15
    bind $hd.key.ki.pos.tagent <Control-v> {InsertClipboardContents %W}
    bind $hd.key.ki.title <<B3>> ExplainHowToIdentifyKey;
    bind $hd.keyf <<B3>> ExplainHowToIdentifyKey;
    $hd.key.ki.pos.rdb1 select

#Optional field?
    frame $hd.key.opt
    label       $hd.key.opt.title -text [_ "Is This Key Optional?"] -anchor w -font HeaderFont
    radiobutton $hd.key.opt.rdb1 -variable KeyInfo(OptionalP,$hd) -value -0 \
	-text [_ "no"]  -anchor w 
    radiobutton $hd.key.opt.rdb2 -variable KeyInfo(OptionalP,$hd) -value -1 \
	-text [_ "yes"] -anchor w 
    $hd.key.opt.rdb1 select
    frame $hd.key.opt.oc
    label       $hd.key.opt.oc.clab -text [_ "Comparison"]
    radiobutton $hd.key.opt.oc.rdb3 -variable KeyInfo(OptComp,$hd) \
	-value -1 -text [_ "<"] -state disabled
    radiobutton $hd.key.opt.oc.rdb4 -variable KeyInfo(OptComp,$hd) \
	-value  0 -text [_ "="] -state disabled
    radiobutton $hd.key.opt.oc.rdb5 -variable KeyInfo(OptComp,$hd) \
	-value  1 -text [_ ">"] -state disabled
    $hd.key.opt.oc.rdb4 select
    pack $hd.key.opt.oc.clab -side top
    pack $hd.key.opt.oc.rdb3 -side left  -expand 1 -padx 2
    pack $hd.key.opt.oc.rdb5 -side right -expand 1 -padx 2
    pack $hd.key.opt.oc.rdb4 -side right -expand 1 -padx 2
    pack $hd.key.opt.title -side top -expand 1 -fill both  -anchor w -pady {1 8}
    pack $hd.key.opt.rdb1 -side top  -expand 1 -fill y -anchor w  -padx {15 1}
    pack $hd.key.opt.rdb2 -side left -expand 0 -anchor w -padx {15 4} 
    pack $hd.key.opt.oc   -side left -expand 0 -anchor w -padx 5 -pady {1 18}
    bind $hd.key.opt.title <<B3>> ExplainOptional
    bind $hd.key.opt.oc <<B3>> ExplainOptional
    bind $hd.key.opt.rdb1 <<B3>> ExplainOptional
    bind $hd.key.opt.rdb2 <<B3>> ExplainOptional
    bind $hd.key.opt.oc.rdb3 <<B3>> ExplainComparison
    bind $hd.key.opt.oc.rdb4 <<B3>> ExplainComparison
    bind $hd.key.opt.oc.rdb5 <<B3>> ExplainComparison
    bind $hd.key.opt.oc.clab <<B3>> ExplainComparison

    pack $hd.key.ki  -side left -expand 1 -fill both -pady 4
    pack $hd.key.opt -side left -expand 1 -fill both -pady 4
    pack $hd.key.ckb -side left -expand 0 -fill x -pady 4 -anchor ne

    #Sort type selection widget

    if {$::UseCheckbuttonP} {
	frame       $hd.stf -relief $mrelief -border $mborder
	label  $hd.stf.lab  -text [_ "Sort Type"]   -anchor w -font HeaderFont
	checkbutton $hd.stf.ckb -variable KeyInfo(SortTypeDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" 
	pack $hd.stf.lab -side left -expand 0 -fill y -anchor w
	pack $hd.stf.ckb -side right -expand 1 -fill y -anchor e
    } else {
	frame       $hd.stf -relief $mrelief -border $mborder
	checkbutton $hd.stf.ckb -variable KeyInfo(SortTypeDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" \
	    -text [_ "Sort Type"] -anchor w -font HeaderFont -indicatoron 0 \
	    -activeforeground red 	-activebackground navajowhite
	pack $hd.stf.ckb -side right -expand 1 -fill both -anchor e
    }

    frame $hd.st -relief $mrelief -border $mborder
    frame $hd.st.lab
    frame $hd.st.cnt
    frame $hd.st.buf
    label      $hd.st.lab.title -text [_ "Sort Type"] -anchor w -font HeaderFont
    checkbutton $hd.st.lab.ckb -variable KeyInfo(SortTypeDisplayedP,$hd) \
	-onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
    pack $hd.st.lab.title -side left  -expand 1 -fill both -anchor w
    pack $hd.st.lab.ckb -side right -expand 0 -fill both -anchor e

    radiobutton $hd.st.cnt.rdb1 -variable KeyInfo(SortType,$hd) -value 1 -text [_ "lexicographic"]
    radiobutton $hd.st.cnt.rdb2 -variable KeyInfo(SortType,$hd) -value 2 -text [_ "string length"]
    radiobutton $hd.st.cnt.rdb3 -variable KeyInfo(SortType,$hd) -value 3 -text [_ "numeric"]
    radiobutton $hd.st.cnt.rdb6 -variable KeyInfo(SortType,$hd) -value 6 -text [_ "iso8601"]
    radiobutton $hd.st.cnt.rdb7 -variable KeyInfo(SortType,$hd) -value 7 -text [_ "hybrid"]
    radiobutton $hd.st.cnt.rdb8 -variable KeyInfo(SortType,$hd) -value 8 -text [_ "random"]
    radiobutton $hd.st.cnt.rdb9 -variable KeyInfo(SortType,$hd) -value 9 -text [_ "angle"]
    radiobutton $hd.st.cnt.rdbA -variable KeyInfo(SortType,$hd) -value 10 -text [_ "numeric string"]
    radiobutton $hd.st.cnt.rdb4 -variable KeyInfo(SortType,$hd) -value 4 -text [_ "time"]
    radiobutton $hd.st.cnt.rdb5 -variable KeyInfo(SortType,$hd) -value 5 -text [_ "date"]
    radiobutton $hd.st.cnt.rdbB -variable KeyInfo(SortType,$hd) -value 11 -text [_ "month"]
    button      $hd.st.cnt.dfb  -text [_ "Date Format"] -state disabled \
	-command [list PopupSetDateFormat $hd]
    $hd.st.cnt.rdb1 select
    balloonhelp_for $hd.st.cnt.dfb [_ "Press this button to set the date format\nfor this key. It is only available if you\nhave selected a date key."]

    frame $hd.st.cnt.bsep -height 6
    set stxpd 10
#    grid $hd.st.cnt.title  -row 0 -column 0   -sticky w -pady 3 -columnspan 4  -padx 5
    grid $hd.st.cnt.rdb1   -row 1 -column 0   -sticky w -pady 3 -padx {15 4}
    grid $hd.st.cnt.rdb7   -row 1 -column 1   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdb3   -row 1 -column 2   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdb9   -row 1 -column 3   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdbA   -row 1 -column 4   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdb8   -row 1 -column 5   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdb2   -row 2 -column 0   -sticky w -pady 3 -padx {15 4}
    grid $hd.st.cnt.rdb6   -row 2 -column 1   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdb4   -row 2 -column 2   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdb5   -row 2 -column 3   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.dfb    -row 2 -column 4   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.rdbB   -row 2 -column 5   -sticky w -pady 3 -padx $stxpd
    grid $hd.st.cnt.bsep   -row 3 -column 0   -sticky w -pady 3

    pack $hd.st.lab -side top  -expand 1 -fill both -anchor w
    pack $hd.st.cnt -side left -expand 0 -fill none -anchor w
    pack $hd.st.buf -side left -expand 1 -fill none

    bind $hd.st.lab.title <<B3>> ExplainSortType
    bind $hd.st.cnt.rdb1 <<B3>> ExplainLexicographicComparison;
    bind $hd.st.cnt.rdb2 <<B3>> ExplainStringLength;
    bind $hd.st.cnt.rdb3 <<B3>> ExplainNumericComparison;
    bind $hd.st.cnt.rdb4 <<B3>> ExplainTimeComparison;
    bind $hd.st.cnt.rdb5 <<B3>> ExplainDateComparison;
    bind $hd.st.cnt.rdb6 <<B3>> ExplainISO8601Comparison;
    bind $hd.st.cnt.rdb7 <<B3>> ExplainHybridComparison;
    bind $hd.st.cnt.rdb8 <<B3>> ExplainRandomComparison;
    bind $hd.st.cnt.rdb9 <<B3>> ExplainAngleComparison;

    frame $hd.misco -relief $mrelief -border $mborder;
    frame $hd.misco.labf
    frame $hd.misco.bd
    pack $hd.misco.labf -side top -expand 1 -fill both 
    pack $hd.misco.bd   -side top -expand 1 -fill both 
    label $hd.misco.labf.lab -text "Miscellaneous Options" -font HeaderFont
    checkbutton $hd.misco.labf.ckb -variable KeyInfo(MiscellaneousOptionsDisplayedP,$hd) \
	-onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
    pack $hd.misco.labf.lab -side left -expand 0 -fill y -anchor w
    pack $hd.misco.labf.ckb -side right -expand 1 -fill y -anchor e

    if {$::UseCheckbuttonP} {
	frame       $hd.miscof -relief $mrelief -border $mborder
	label  $hd.miscof.lab  -text [_ "Miscellaneous Options"]   -anchor w -font HeaderFont
	checkbutton $hd.miscof.ckb  -variable KeyInfo(MiscellaneousOptionsDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" 
	pack $hd.miscof.lab -side left -expand 0 -fill y -anchor w
	pack $hd.miscof.ckb -side right -expand 1 -fill y -anchor e
    } else {
	frame       $hd.miscof -relief $mrelief -border $mborder
	checkbutton $hd.miscof.ckb -variable KeyInfo(MiscellaneousOptionsDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" \
	    -text [_ "Miscellaneous Options"] -anchor w -font HeaderFont -indicatoron 0 \
	    -activeforeground red 	-activebackground navajowhite
	pack $hd.miscof.ckb -side right -expand 1 -fill both -anchor e
    }
	
    #Fold case?
    frame       $hd.misco.bd.fc
    frame       $hd.misco.bd.fc.left
    label       $hd.misco.bd.fc.left.lab  -text [_ "Fold Case?"] -anchor w -font HeaderFont
    label       $hd.misco.bd.fc.img;
    frame       $hd.misco.bd.fc.left.rbf
    radiobutton $hd.misco.bd.fc.left.rbf.rdb1 -variable KeyInfo(FoldCaseP,$hd) \
	-value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.fc.left.rbf.rdb2 -variable KeyInfo(FoldCaseP,$hd) \
	-value 1 -text [_ "yes"]
    $hd.misco.bd.fc.left.rbf.rdb1 select
    pack $hd.misco.bd.fc.left.rbf.rdb1 -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.fc.left.rbf.rdb2 -anchor w -side left
    pack $hd.misco.bd.fc.left.lab -side top -expand y -fill x  -anchor w -padx 5
    pack $hd.misco.bd.fc.left.rbf -side bottom -expand y -anchor w
    pack $hd.misco.bd.fc.left -side left
    pack $hd.misco.bd.fc.img -side right -expand 1;
    bind $hd.misco.bd.fc.img <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.fc.left <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.fc.left.lab <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.fc.left.rbf <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.fc.left.rbf.rdb1 <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.fc.left.rbf.rdb2 <<B3>> ExplainFoldCase;
    set bhmsg [_ "Make comparisons insensitive to case?"]
    balloonhelp_for $hd.misco.bd.fc $bhmsg;
    balloonhelp_for $hd.misco.bd.fc.img $bhmsg;
    balloonhelp_for $hd.misco.bd.fc.left $bhmsg;
    balloonhelp_for $hd.misco.bd.fc.left.rbf $bhmsg;
    balloonhelp_for $hd.misco.bd.fc.left.rbf.rdb1 $bhmsg;
    balloonhelp_for $hd.misco.bd.fc.left.rbf.rdb2 $bhmsg;
    balloonhelp_for $hd.misco.bd.fc.left.lab $bhmsg;
    #Turkic fold case?
    frame       $hd.misco.bd.tfc
    frame       $hd.misco.bd.tfc.left
    label       $hd.misco.bd.tfc.left.lab  -text [_ "Turkic Fold Case?"] -anchor w -font HeaderFont
    label       $hd.misco.bd.tfc.img;
    frame       $hd.misco.bd.tfc.left.rbf
    radiobutton $hd.misco.bd.tfc.left.rbf.rdb1 -variable KeyInfo(TurkicFoldCaseP,$hd) \
	-value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.tfc.left.rbf.rdb2 -variable KeyInfo(TurkicFoldCaseP,$hd) \
	-value 1 -text [_ "yes"]
    $hd.misco.bd.tfc.left.rbf.rdb1 select
    pack $hd.misco.bd.tfc.left.rbf.rdb1 -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.tfc.left.rbf.rdb2 -anchor w -side left
    pack $hd.misco.bd.tfc.left.lab -side top -expand y -fill x  -anchor w -padx 5
    pack $hd.misco.bd.tfc.left.rbf -side bottom -expand y -anchor w
    pack $hd.misco.bd.tfc.left -side left
    pack $hd.misco.bd.tfc.img -side right -expand 1;
    bind $hd.misco.bd.tfc.img <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.tfc.left <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.tfc.left.lab <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.tfc.left.rbf <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.tfc.left.rbf.rdb1 <<B3>> ExplainFoldCase;
    bind $hd.misco.bd.tfc.left.rbf.rdb2 <<B3>> ExplainFoldCase;
    set bhmsg [_ "Make comparisons insensitive to case including Turkic\nfolding of \u0049 U+0049 to \u0131 U+0131 and \u0130 U+0130 to \u0069 U+0069 ?"]
    balloonhelp_for $hd.misco.bd.tfc $bhmsg;
    balloonhelp_for $hd.misco.bd.tfc.img $bhmsg;
    balloonhelp_for $hd.misco.bd.tfc.left $bhmsg;
    balloonhelp_for $hd.misco.bd.tfc.left.rbf $bhmsg;
    balloonhelp_for $hd.misco.bd.tfc.left.rbf.rdb1 $bhmsg;
    balloonhelp_for $hd.misco.bd.tfc.left.rbf.rdb2 $bhmsg;
    balloonhelp_for $hd.misco.bd.tfc.left.lab $bhmsg;

    #Reverse key?
    frame       $hd.misco.bd.rk
    frame       $hd.misco.bd.rk.left
    label       $hd.misco.bd.rk.left.lab  -text [_ "Reverse Key?"] -anchor w -font HeaderFont
    label       $hd.misco.bd.rk.img;
    frame       $hd.misco.bd.rk.left.rbf
    radiobutton $hd.misco.bd.rk.left.rbf.rdb1 -variable KeyInfo(ReverseKeyP,$hd) -value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.rk.left.rbf.rdb2 -variable KeyInfo(ReverseKeyP,$hd) -value 1 -text [_ "yes"]
    $hd.misco.bd.rk.left.rbf.rdb1 select
    pack $hd.misco.bd.rk.left.rbf.rdb1  -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.rk.left.rbf.rdb2  -anchor w -side left -expand 0
    pack $hd.misco.bd.rk.left.lab -side top -expand y -fill x  -anchor w
    pack $hd.misco.bd.rk.left.rbf -side bottom -expand y -anchor w
    pack $hd.misco.bd.rk.left -side left
    pack $hd.misco.bd.rk.img -side right -expand 1;
    bind $hd.misco.bd.rk.left <<B3>> ExplainReverseKey;
    bind $hd.misco.bd.rk.img <<B3>> ExplainReverseKey;
    bind $hd.misco.bd.rk.left.lab <<B3>> ExplainReverseKey;
    bind $hd.misco.bd.rk.left.rbf <<B3>> ExplainReverseKey;
    bind $hd.misco.bd.rk.left.rbf.rdb1 <<B3>> ExplainReverseKey;
    bind $hd.misco.bd.rk.left.rbf.rdb2 <<B3>> ExplainReverseKey;
    #Invert comparison?
    frame       $hd.misco.bd.li
    label       $hd.misco.bd.li.lab  -text [_ "Invert Comparison?"] -anchor w -font HeaderFont
    frame       $hd.misco.bd.li.rbf
    radiobutton $hd.misco.bd.li.rbf.rdb1 -variable KeyInfo(LocalInversionP,$hd) -value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.li.rbf.rdb2 -variable KeyInfo(LocalInversionP,$hd) -value 1 -text [_ "yes"]
    $hd.misco.bd.li.rbf.rdb1 select
    pack $hd.misco.bd.li.rbf.rdb1 -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.li.rbf.rdb2 -anchor w -side left -expand 0
    pack $hd.misco.bd.li.lab -side top -expand y -fill x  -anchor w
    pack $hd.misco.bd.li.rbf -side bottom -expand y -anchor w
    bind $hd.misco.bd.li.lab <<B3>> ExplainInvertComparison;
    bind $hd.misco.bd.li.rbf <<B3>> ExplainInvertComparison;
    bind $hd.misco.bd.li.rbf.rdb1 <<B3>> ExplainInvertComparison;
    bind $hd.misco.bd.li.rbf.rdb2 <<B3>> ExplainInvertComparison;

    #Strip diacritics
    frame       $hd.misco.bd.sd
    frame       $hd.misco.bd.sd.left
    label       $hd.misco.bd.sd.left.lab  -text [_ "Strip Diacritics?"] -anchor w -font HeaderFont
    label       $hd.misco.bd.sd.img;
    frame       $hd.misco.bd.sd.left.rbf
    radiobutton $hd.misco.bd.sd.left.rbf.rdb1 -variable KeyInfo(StripDiacriticsP,$hd) \
	-value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.sd.left.rbf.rdb2 -variable KeyInfo(StripDiacriticsP,$hd) \
	-value 1 -text [_ "yes"]
    $hd.misco.bd.sd.left.rbf.rdb1 select
    pack $hd.misco.bd.sd.left.rbf.rdb1 -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.sd.left.rbf.rdb2 -anchor w -side left
    pack $hd.misco.bd.sd.left.lab -side top -expand y -fill x  -anchor w -padx 5
    pack $hd.misco.bd.sd.left.rbf -side bottom -expand y -anchor w
    pack $hd.misco.bd.sd.left -side left
    pack $hd.misco.bd.sd.img -side right -expand 1;
    bind $hd.misco.bd.sd.img <<B3>> ExplainStripDiacritics;
    bind $hd.misco.bd.sd.left <<B3>> ExplainStripDiacritics;
    bind $hd.misco.bd.sd.left.lab <<B3>> ExplainStripDiacritics;
    bind $hd.misco.bd.sd.left.rbf <<B3>> ExplainStripDiacritics;
    bind $hd.misco.bd.sd.left.rbf.rdb1 <<B3>> ExplainStripDiacritics;
    bind $hd.misco.bd.sd.left.rbf.rdb2 <<B3>> ExplainStripDiacritics;
    set bhmsg [_ "Strip diacritics?"]
    balloonhelp_for $hd.misco.bd.sd $bhmsg;
    balloonhelp_for $hd.misco.bd.sd.img $bhmsg;
    balloonhelp_for $hd.misco.bd.sd.left $bhmsg;
    balloonhelp_for $hd.misco.bd.sd.left.rbf $bhmsg;
    balloonhelp_for $hd.misco.bd.sd.left.rbf.rdb1 $bhmsg;
    balloonhelp_for $hd.misco.bd.sd.left.rbf.rdb2 $bhmsg;
    balloonhelp_for $hd.misco.bd.sd.left.lab $bhmsg;

    #Convert enclosures
    frame       $hd.misco.bd.ce
    frame       $hd.misco.bd.ce.left
    label       $hd.misco.bd.ce.left.lab  -text [_ "Strip Enclosures?"] -anchor w -font HeaderFont
    label       $hd.misco.bd.ce.img;
    frame       $hd.misco.bd.ce.left.rbf
    radiobutton $hd.misco.bd.ce.left.rbf.rdb1 -variable KeyInfo(ConvertEnclosuresP,$hd) \
	-value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.ce.left.rbf.rdb2 -variable KeyInfo(ConvertEnclosuresP,$hd) \
	-value 1 -text [_ "yes"]
    $hd.misco.bd.ce.left.rbf.rdb1 select
    pack $hd.misco.bd.ce.left.rbf.rdb1 -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.ce.left.rbf.rdb2 -anchor w -side left
    pack $hd.misco.bd.ce.left.lab -side top -expand y -fill x  -anchor w -padx 5
    pack $hd.misco.bd.ce.left.rbf -side bottom -expand y -anchor w
    pack $hd.misco.bd.ce.left -side left
    pack $hd.misco.bd.ce.img -side right -expand 1;
    bind $hd.misco.bd.ce.img <<B3>> ExplainConvertEnclosures;
    bind $hd.misco.bd.ce.left <<B3>> ExplainConvertEnclosures;
    bind $hd.misco.bd.ce.left.lab <<B3>> ExplainConvertEnclosures;
    bind $hd.misco.bd.ce.left.rbf <<B3>> ExplainConvertEnclosures;
    bind $hd.misco.bd.ce.left.rbf.rdb1 <<B3>> ExplainConvertEnclosures;
    bind $hd.misco.bd.ce.left.rbf.rdb2 <<B3>> ExplainConvertEnclosures;
    set bhmsg [_ "Convert enclosures?"]
    balloonhelp_for $hd.misco.bd.ce $bhmsg;
    balloonhelp_for $hd.misco.bd.ce.img $bhmsg;
    balloonhelp_for $hd.misco.bd.ce.left $bhmsg;
    balloonhelp_for $hd.misco.bd.ce.left.rbf $bhmsg;
    balloonhelp_for $hd.misco.bd.ce.left.rbf.rdb1 $bhmsg;
    balloonhelp_for $hd.misco.bd.ce.left.rbf.rdb2 $bhmsg;
    balloonhelp_for $hd.misco.bd.ce.left.lab $bhmsg;

    #Convert stylistic
    frame       $hd.misco.bd.cs
    frame       $hd.misco.bd.cs.left
    label       $hd.misco.bd.cs.left.lab  -text [_ "Strip Style?"] -anchor w\
	-font HeaderFont
    label       $hd.misco.bd.cs.img;
    frame       $hd.misco.bd.cs.left.rbf
    radiobutton $hd.misco.bd.cs.left.rbf.rdb1 -variable KeyInfo(ConvertStylisticP,$hd) \
	-value 0 -text [_ "no"]
    radiobutton $hd.misco.bd.cs.left.rbf.rdb2 -variable KeyInfo(ConvertStylisticP,$hd) \
	-value 1 -text [_ "yes"]
    $hd.misco.bd.cs.left.rbf.rdb1 select
    pack $hd.misco.bd.cs.left.rbf.rdb1 -anchor w -side left -padx {15 3}
    pack $hd.misco.bd.cs.left.rbf.rdb2 -anchor w -side left
    pack $hd.misco.bd.cs.left.lab -side top -expand y -fill x  -anchor w -padx 5
    pack $hd.misco.bd.cs.left.rbf -side bottom -expand y -anchor w
    pack $hd.misco.bd.cs.left -side left
    pack $hd.misco.bd.cs.img -side right -expand 1;
    bind $hd.misco.bd.cs.img <<B3>> ExplainConvertStylistic;
    bind $hd.misco.bd.cs.left <<B3>> ExplainConvertStylistic;
    bind $hd.misco.bd.cs.left.lab <<B3>> ExplainConvertStylistic;
    bind $hd.misco.bd.cs.left.rbf <<B3>> ExplainConvertStylistic;
    bind $hd.misco.bd.cs.left.rbf.rdb1 <<B3>> ExplainConvertStylistic;
    bind $hd.misco.bd.cs.left.rbf.rdb2 <<B3>> ExplainConvertStylistic;
    set bhmsg [_ "Convert stylistic equivalents?"]
    balloonhelp_for $hd.misco.bd.cs $bhmsg;
    balloonhelp_for $hd.misco.bd.cs.img $bhmsg;
    balloonhelp_for $hd.misco.bd.cs.left $bhmsg;
    balloonhelp_for $hd.misco.bd.cs.left.rbf $bhmsg;
    balloonhelp_for $hd.misco.bd.cs.left.rbf.rdb1 $bhmsg;
    balloonhelp_for $hd.misco.bd.cs.left.rbf.rdb2 $bhmsg;
    balloonhelp_for $hd.misco.bd.cs.left.lab $bhmsg;

    set xp 5
    grid $hd.misco.bd.rk  -row 1 -column 1 -sticky w -pady {5  5} -padx {5 5}
    grid $hd.misco.bd.li  -row 1 -column 2 -sticky w -pady {10 5} -padx $xp
    grid $hd.misco.bd.cs  -row 1 -column 3 -sticky w -pady {10 5} -padx $xp
    grid $hd.misco.bd.ce  -row 2 -column 1 -sticky w -pady {5 12} -padx {5 5}
    grid $hd.misco.bd.sd  -row 2 -column 2 -sticky w -pady {5 12} -padx $xp
    grid $hd.misco.bd.fc  -row 2 -column 3 -sticky w -pady {5 12} -padx $xp
    grid $hd.misco.bd.tfc -row 2 -column 4 -sticky w -pady {5 12} -padx $xp

    # Sort order/multigraphs
    frame       $hd.so -relief $mrelief -border $mborder
    frame       $hd.so.labf -relief $mrelief -border $mborder
    label       $hd.so.labf.lab  -text [_ "Sort Order"] -anchor w -font HeaderFont
    checkbutton $hd.so.labf.ckb -variable KeyInfo(SortOrderDisplayedP,$hd) \
	-onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
    pack $hd.so.labf.lab -side left  -expand 1 -fill both -anchor w
    pack $hd.so.labf.ckb -side right -expand 0 -fill both -anchor e
    radiobutton $hd.so.rdb0 -variable KeyInfo(SortOrderSpecType,$hd) \
	-value 0 -text [_ "Unicode Order"] -anchor w

    if {$::UseCheckbuttonP} {
	frame       $hd.sof -relief $mrelief -border $mborder
	label  $hd.sof.lab  -text [_ "Sort Order"]   -anchor w -font HeaderFont
	checkbutton $hd.sof.ckb -variable KeyInfo(SortOrderDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" 
	pack $hd.sof.lab -side left -expand 0 -fill y -anchor w
	pack $hd.sof.ckb -side right -expand 1 -fill y -anchor e
    } else {
	frame       $hd.sof -relief $mrelief -border $mborder
	checkbutton $hd.sof.ckb -variable KeyInfo(SortOrderDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" \
	    -text [_ "Sort Order"] -anchor w -font HeaderFont -indicatoron 0 \
	    -activeforeground red 	-activebackground navajowhite
	pack $hd.sof.ckb -side right -expand 1 -fill both -anchor e
    }


    frame       $hd.so.loc 
    radiobutton $hd.so.loc.rdb1 -variable KeyInfo(SortOrderSpecType,$hd) \
	-value 1 -text [_ "Locale Order"] -anchor w
    entry  $hd.so.loc.ent -width $FileNameEntryWidth  -font UserPlainTextFont
    $hd.so.loc.ent insert end "current"
    $hd.so.loc.ent configure -state disabled
    bind $hd.so.loc.ent <FocusIn> "SetInsertionTargets $hd.so.loc.ent"
    frame       $hd.so.bot 
    radiobutton $hd.so.bot.rdb2 -variable KeyInfo(SortOrderSpecType,$hd) \
	-value 2 -text [_ "Custom Order"] -anchor w
    $hd.so.rdb0 select
    button      $hd.so.bot.b1  -text [_ "Specify File"] \
	-state disabled -command [list SpecifySortOrderFile $hd]
    button      $hd.so.bot.b2  -text [_ "Define"]  -state disabled -command "EnterSortOrder $hd"
    entry  $hd.so.bot.ent -width $FileNameEntryWidth -state disabled  -font UserPlainTextFont
    bind $hd.so.bot.ent <FocusIn> "SetInsertionTargets $hd.so.bot.ent"
    frame $hd.so.bsep -height 5
    frame $hd.so.tsep -height 5

    pack $hd.so.labf -side top -expand 1 -fill both  -padx 5
    pack $hd.so.bot.rdb2 -side left -expand 0 -fill x  -anchor w -padx {0 4}
    pack $hd.so.bot.b2 -side left -expand 0  -anchor w -padx {2 4}
    pack $hd.so.bot.b1 -side left -expand 0  -anchor w
    pack $hd.so.bot.ent  -expand 0 -fill none  -anchor w -side left -padx 3
    pack $hd.so.loc.rdb1 -side left -expand 0 -fill x  -anchor w -padx {0 4}
    pack $hd.so.loc.ent  -expand 0 -fill none  -anchor w -side left -padx 3
    pack $hd.so.bsep -side bottom -expand 0 -fill both
    pack $hd.so.bot  -side bottom -expand 0 -fill x  -anchor w -padx {15 4}
    pack $hd.so.tsep -side top -expand 0 -fill both
    pack $hd.so.rdb0 -side top -expand 0 -fill y  -anchor w -padx {15 4}
    pack $hd.so.loc -side top -expand 0 -fill y  -anchor w -padx {15 4}
    bind $hd.so.labf <<B3>> ExplainSortOrder;
    balloonhelp_for $hd.so.rdb0 [_ "Choose this option if you want to sort on this key using the order of\nUnicode codepoints as the sort order. For ASCII characters this is the\nsame as ASCII order."]
    balloonhelp_for $hd.so.bot.b1 [_ "The sort order definition will be read from the file shown\nin the entry box to the right. You can type the name of the\nfile in the entry box if you like, or you can press this button\nto bring up a file selection dialogue."]
    balloonhelp_for $hd.so.bot.b2 [_ "Press this button if you want to define a sort order interactively.\nThe result will be written out to a file,\nwhose name will appear in the entry box to the right."]
    balloonhelp_for $hd.so.loc.rdb1 [_ "Choose this option if you want to sort on this key using\nthe collation rules for a locale."]
    balloonhelp_for $hd.so.bot.rdb2 [_ "Choose this option if you want to use a custom sort order."]
    balloonhelp_for $hd.so.loc.ent [_ "The sort order will be determined by the collation rules for\nthe locale specified here. You may enter either a locale name\nor the word \"current\". In the latter case, your current locale\nwill be used. Note that this only works on systems using glibc\nand only if the locale you choose has been installed."]
    balloonhelp_for $hd.so.bot.ent [_ "The sort order will be read from the file whose name\nis shown in this entry box. You may select an\nexisting file by entering its name directly or by using\nthe file selection dialogue. You may also\ndefine the sort order interactively, after which\nit will be written to a file, whose name\nwill appear in the entry box."]

     #Sort order file separators
    frame $hd.sos -relief $mrelief -border $mborder
    label       $hd.sos.tit  -text [_ "Custom Sort Order File Separators?"] \
	-anchor w -font HeaderFont
    radiobutton $hd.sos.rdb1 -variable KeyInfo(CustomSortOrderSeparatorsP,$hd) -value 0 -text [_ "no"] -anchor w
    frame       $hd.sos.bot 
    radiobutton $hd.sos.bot.rdb2 -variable KeyInfo(CustomSortOrderSeparatorsP,$hd) -value 1 -text [_ "yes"] -anchor w
    $hd.sos.rdb1 select
    button      $hd.sos.bot.b1  -text [_ "Specify File"] -state disabled -command [list SpecifySortOrderSeparatorFile $hd]
    button      $hd.sos.bot.b2  -text [_ "Define"]  -state disabled -command "EnterSortOrderSeparators $hd"
    entry  $hd.sos.bot.ent -width $FileNameEntryWidth -state disabled  -font UserPlainTextFont
    bind $hd.sos.bot.ent <FocusIn> "SetInsertionTargets $hd.sos.bot.ent"
    frame $hd.sos.bsep -height 5
    frame $hd.sos.tsep -height 5
    pack $hd.sos.bot.rdb2 -side left -expand 0 -fill x  -anchor w -padx {0 4}
    pack $hd.sos.bot.b2 -side left -expand 0  -anchor w -padx {2 4}
    pack $hd.sos.bot.b1 -side left -expand 0  -anchor w
    pack $hd.sos.bot.ent  -expand 0 -fill none  -anchor w -side left -padx 3
    pack $hd.sos.tsep -side top -expand 1 -fill both
    pack $hd.sos.tit -side top -expand 1 -fill both  -padx 5
    pack $hd.sos.bsep -side bottom -expand 1 -fill both
    pack $hd.sos.bot -side bottom -expand 1 -fill x  -anchor w -padx {15 4}
    pack $hd.sos.rdb1 -side top -expand 1 -fill y  -anchor w -padx {15 4}
    bind $hd.sos.tit <<B3>> ExplainSortOrderSeparators;
    set msg [_ "Redefine sort order separators so as to\ninclude whitespace in multigraphs?"];
    balloonhelp_for $hd.sos $msg;
    balloonhelp_for $hd.sos.tit $msg;
    balloonhelp_for $hd.sos.bot $msg;
    balloonhelp_for $hd.sos.rdb1 $msg;
    balloonhelp_for $hd.sos.bot.rdb2 $msg;
    balloonhelp_for $hd.sos.bot.b1 [_ "The sort order separators will be read from the file shown\nin the entry box to the right. You can type the name of the\nfile in the entry box if you like, or you can press this button\nto bring up a file selection dialogue."]
    balloonhelp_for $hd.sos.bot.b2 [_ "Press this button if you want to enter the separators interactively.\nThe result will be written out to a file,\nwhose name will appear in the entry box to the right."]
    balloonhelp_for $hd.sos.bot.ent [_ "The sort order separators will be read from the file whose name\n is shown in this entry box. You may select an\nexisting file by entering its name directly or by using\nthe file selection dialogue. You may also\ndefine the sort order interactively, after which\nthey it will be written to a file, whose name\nwill appear in the entry box."]

     #Exclusions
    frame       $hd.exc -relief $mrelief -border $mborder
    frame       $hd.exc.labf -relief $mrelief -border $mborder
    label       $hd.exc.labf.lab  -text [_ "Exclusions"] -anchor w -font HeaderFont
    checkbutton $hd.exc.labf.ckb -variable KeyInfo(ExclusionsDisplayedP,$hd) \
	-onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
    pack $hd.exc.labf.lab -side left  -expand 1 -fill both -anchor w
    pack $hd.exc.labf.ckb -side right -expand 0 -fill both -anchor e
    radiobutton $hd.exc.rdb1 -variable KeyInfo(ExclusionsP,$hd) -value 0 -text [_ "no"] -anchor w
    frame       $hd.exc.bot 
    radiobutton $hd.exc.bot.rdb2 -variable KeyInfo(ExclusionsP,$hd) -value 1 -text [_ "yes"] -anchor w
    $hd.exc.rdb1 select
    button      $hd.exc.bot.b1  -text [_ "Specify File"] -state disabled -command [list SpecifyExclusionFile $hd]
    button      $hd.exc.bot.b2  -text [_ "Define"]  -state disabled -command "EnterExclusions $hd"
    entry  $hd.exc.bot.ent -width $FileNameEntryWidth -state disabled -font UserPlainTextFont
    bind $hd.exc.bot.ent <FocusIn> "SetInsertionTargets $hd.exc.bot.ent"
    frame $hd.exc.bsep -height 5
    frame $hd.exc.tsep -height 5
    pack $hd.exc.bot.rdb2 -side left -expand 0 -fill x  -anchor w -padx {0 4}
    pack $hd.exc.bot.b2 -side left -expand 0  -anchor w -padx {2 4}
    pack $hd.exc.bot.b1 -side left -expand 0  -anchor w
    pack $hd.exc.bot.ent  -expand 0 -fill none  -anchor w -side left  -padx 3
    pack $hd.exc.labf -side top -anchor w -expand 1 -fill both -padx 5
    pack $hd.exc.bsep -side bottom -expand 1 -fill both
    pack $hd.exc.bot -side bottom -expand 1 -fill x  -anchor w -padx {15 4}
    pack $hd.exc.tsep -side top -expand 1 -fill both
    pack $hd.exc.rdb1 -side top -expand 1 -fill y  -anchor w -padx {15 4}

    if {$::UseCheckbuttonP} {
	frame       $hd.excf -relief $mrelief -border $mborder
	label  $hd.excf.lab  -text [_ "Exclusions"]   -anchor w -font HeaderFont
	checkbutton $hd.excf.ckb -variable KeyInfo(ExclusionsDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" 
	pack $hd.excf.lab -side left -expand 0 -fill y -anchor w
	pack $hd.excf.ckb -side right -expand 1 -fill y -anchor e
    } else {
	frame       $hd.excf -relief $mrelief -border $mborder
	checkbutton $hd.excf.ckb -variable KeyInfo(ExclusionsDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" \
	    -text [_ "Exclusions"] -anchor w -font HeaderFont -indicatoron 0 \
	    -activeforeground red 	-activebackground navajowhite
	pack $hd.excf.ckb -side right -expand 1 -fill both -anchor e
    }

    bind $hd.exc.labf <<B3>> ExplainExclusions
    bind $hd.excf.ckb  <<B3>> ExplainExclusions
    bind $hd.exc.bot <<B3>> ExplainExclusions
    bind $hd.exc.rdb1 <<B3>> ExplainExclusions
    bind $hd.exc.bot.rdb2 <<B3>> ExplainExclusions
    bind $hd.exc.bot.b1 <<B3>> ExplainExclusions
    bind $hd.exc.bot.b2 <<B3>> ExplainExclusions
    bind $hd.exc.bot.ent <<B3>> ExplainExclusions
    balloonhelp_for $hd.exc.bot.b1 [_ "The exclusions will be read from the file shown\nin the entry box to the right. You can type the name of the\nfile in the entry box if you like, or you can press this button\nto bring up a file selection dialogue."]
    balloonhelp_for $hd.exc.bot.b2 [_ "Press this button if you want to define exclusions interactively.\nThe result will be written out to a file,\nwhose name will appear in the entry box to the right."]
    balloonhelp_for $hd.exc.bot.ent [_ "Exclusions will be read from the file whose name\n is shown in this entry box. You may select an\nexisting file by entering its name directly or by using\nthe file selection dialogue. You may also\ndefine exclusions interactively, after which\nthey will be written to a file, whose name\nwill appear in the entry box."]

     #Substitutions
    frame       $hd.sub -relief $mrelief -border $mborder
    frame       $hd.sub.labf -relief $mrelief -border $mborder
    label       $hd.sub.labf.lab  -text [_ "Substitutions"] -anchor w -font HeaderFont
    checkbutton $hd.sub.labf.ckb -variable KeyInfo(SubstitutionsDisplayedP,$hd) \
	-onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
    pack $hd.sub.labf.lab -side left  -expand 1 -fill both -anchor w
    pack $hd.sub.labf.ckb -side right -expand 0 -fill both -anchor e
    radiobutton $hd.sub.rdb1 -variable KeyInfo(SubstitutionsP,$hd) -value 0 \
	-text [_ "no"] -anchor w
    frame       $hd.sub.bot 
    radiobutton $hd.sub.bot.rdb2 -variable KeyInfo(SubstitutionsP,$hd) \
	-value 1 -text [_ "yes"] -anchor w
    $hd.sub.rdb1 select
    button      $hd.sub.bot.b1  -text [_ "Specify File"] \
	-state disabled -command [list SpecifySubstitutionFile $hd]
    button      $hd.sub.bot.b2  -text [_ "Define"]  -state disabled \
	-command "EnterSubstitutions $hd"
    entry  $hd.sub.bot.ent -width $FileNameEntryWidth -state disabled -font UserPlainTextFont
    bind $hd.sub.bot.ent <FocusIn> "SetInsertionTargets $hd.sub.bot.ent"

    frame $hd.sub.bsep -height 5
    frame $hd.sub.tsep -height 5
    pack $hd.sub.bot.rdb2 -side left -expand 0 -fill x  -anchor w -padx {0 4}
    pack $hd.sub.bot.b2 -side left -expand 0  -anchor w -padx {2 4}
    pack $hd.sub.bot.b1 -side left -expand 0  -anchor w
    pack $hd.sub.bot.ent  -expand 0 -fill none  -anchor w -side left  -padx 3

    pack $hd.sub.tsep -side top -expand 1 -fill both
    pack $hd.sub.labf -side top -anchor w -expand 1 -fill both -padx 5
    pack $hd.sub.bsep -side bottom -expand 1 -fill both
    pack $hd.sub.bot  -side bottom -expand 1 -fill x  -anchor w -padx {15 4}
    pack $hd.sub.rdb1 -side top -expand 1 -fill y  -anchor w -padx {15 4}

    if {$::UseCheckbuttonP} {
	frame       $hd.subf -relief $mrelief -border $mborder
	label  $hd.subf.lab  -text [_ "Substitutions"]   -anchor w -font HeaderFont
	checkbutton $hd.subf.ckb -variable KeyInfo(SubstitutionsDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" 
	pack $hd.subf.lab -side left -expand 0 -fill y -anchor w
	pack $hd.subf.ckb -side right -expand 1 -fill y -anchor e
    } else {
	frame       $hd.subf -relief $mrelief -border $mborder
	checkbutton $hd.subf.ckb -variable KeyInfo(SubstitutionsDisplayedP,$hd) \
	    -onvalue 1 -offvalue 0 -command "PackKeyPage $hd;" \
	    -text [_ "Substitutions"] -anchor w -font HeaderFont -indicatoron 0 \
	    -activeforeground red 	-activebackground navajowhite
	pack $hd.subf.ckb -side right -expand 1 -fill both -anchor e
    }

    bind $hd.sub.labf <<B3>> ExplainSubstitutions
    bind $hd.sub.bot <<B3>> ExplainSubstitutions
    bind $hd.sub.rdb1 <<B3>> ExplainSubstitutions
    bind $hd.sub.bot.rdb2 <<B3>> ExplainSubstitutions
    bind $hd.sub.bot.b1 <<B3>> ExplainSubstitutions
    bind $hd.sub.bot.b2 <<B3>> ExplainSubstitutions
    bind $hd.sub.bot.ent <<B3>> ExplainSubstitutions
    balloonhelp_for $hd.sub.bot.b1 [_ "The substitutions will be read from the file shown\nin the entry box to the right. You can type the name of the\nfile in the entry box if you like, or you can press this button\nto bring up a file selection dialogue."]
    balloonhelp_for $hd.sub.bot.b2 [_ "Press this button if you want to define substitutions interactively.\nThe result will be written out to a file,\nwhose name will appear in the entry box to the right."]
    balloonhelp_for $hd.sub.bot.ent [_ "Substitutions will be read from the file whose name\n is shown in this entry box. You may select an\nexisting file by entering its name directly or by using\nthe file selection dialogue. You may also\ndefine substitutions interactively, after which\nthey will be written to a file, whose name\nwill appear in the entry box."]

    # Pack it all together
    PackKeyPage $hd;

    balloonhelp_for $hd.exc    [_ "Should some characters in some positions\nbe ignored when comparing records on this key?"]
    balloonhelp_for $hd.sub    [_ "Should some sequences be replaced with others?"]
    balloonhelp_for $hd.key.ki $::MainKeyFieldIdentBH;
    balloonhelp_for $hd.misco.bd.li     [_ "Invert the sense of comparisons on this key?"]
    balloonhelp_for $hd.misco.bd.rk     [_ "Reverse the order of the characters in the key?"]
    balloonhelp_for $hd.key.opt    [_ "Is this key obligatory or may it be omitted?"]
    balloonhelp_for $hd.key.opt.oc [_ "How should a record with a missing optional key\ncompare with a record in which the same key is present?"]
    balloonhelp_for $hd.so     [_ "What sort order do you wish to use for this key?"]
    balloonhelp_for $hd.st.cnt     [_ "What sort of comparison should be done on this field?"]

    trace add variable ::KeyInfo(KeySelection,$hd) write ChooseActiveKeySelection
    trace add variable ::KeyInfo(ExclusionsP,$hd) write ToggleExclusionEntryActive
    trace add variable ::KeyInfo(SubstitutionsP,$hd) write ToggleSubstitutionEntryActive
    trace add variable ::KeyInfo(SortOrderSpecType,$hd) write ToggleCustomSortOrderEntryActive
    trace add variable ::KeyInfo(CustomSortOrderSeparatorsP,$hd) write ToggleCustomSortOrderSeparatorsEntryActive
    trace add variable ::KeyInfo(SortType,$hd) write ToggleOnSortType
    trace add variable ::KeyInfo(OptionalP,$hd) write ToggleOptComparisonActive
    BindKeys $hd;
}

proc ChangePage {n e o} {
    foreach i [array names ::KeyMap] {
	set w $::KeyMap($i);
	if {[winfo exists $w]} {
	    $::SelectionButtonMap($i) configure -fg $::ColorSpecs(PageChoice,UnselectedPageForeground)
	    pack forget $w;
	} else {
	    dmsg "ChangePage: attempt to unpack non-existant window $w";
	}
    }
    set w $::KeyMap($::WhichKey)
    if {[winfo exists $w] == 0} {
	dmsg "ChangePage: attempt to pack non-existant window $w";
	return ;
    }
    pack $w -expand 1 -fill both -pady 0
    $::SelectionButtonMap($::WhichKey) configure -fg $::ColorSpecs(PageChoice,ActiveForeground);
    if {$::WhichKey > 0} {
	if {$::WholeRecordIsKeyP} {
	    DisableKeyFieldID $::KeyMap($::WhichKey);
	} else {
	    EnableKeyFieldID $::KeyMap($::WhichKey);
	}
    }
}

proc ChooseActiveKeySelection {n e o} {
    set hd [lindex [split $e ","] 1]
    switch $::KeyInfo($e) {
	0 {
	    $hd.key.ki.pos.posent.fldent1   configure -state normal
	    $hd.key.ki.pos.posent.fldent2   configure -state normal
	    $hd.key.ki.pos.posent.chrent1   configure -state normal
	    $hd.key.ki.pos.posent.chrent2   configure -state normal
	    $hd.key.ki.pos.tagent   configure -state disabled
	    $hd.key.ki.pos.firstent configure -state disabled
	    $hd.key.ki.pos.lastent  configure -state disabled
	    focus $hd.key.ki.pos.posent.fldent1;
	}
	1 {
	    $hd.key.ki.pos.posent.fldent1   configure -state disabled
	    $hd.key.ki.pos.posent.fldent2   configure -state disabled
	    $hd.key.ki.pos.posent.chrent1   configure -state disabled
	    $hd.key.ki.pos.posent.chrent2   configure -state disabled
	    $hd.key.ki.pos.tagent   configure -state normal
	    $hd.key.ki.pos.firstent configure -state disabled
	    $hd.key.ki.pos.lastent  configure -state disabled
	    focus $hd.key.ki.pos.tagent;
	}
	2 {
	    $hd.key.ki.pos.posent.fldent1   configure -state disabled
	    $hd.key.ki.pos.posent.fldent2   configure -state disabled
	    $hd.key.ki.pos.posent.chrent1   configure -state disabled
	    $hd.key.ki.pos.posent.chrent2   configure -state disabled
	    $hd.key.ki.pos.tagent   configure -state disabled
	    $hd.key.ki.pos.firstent configure -state normal
	    $hd.key.ki.pos.lastent  configure -state normal
	    focus $hd.key.ki.pos.firstent;
	}
    }
}

proc ToggleExclusionEntryActive {n e o} {
    set hd [lindex [split $e ","] 1]
    if {$::KeyInfo($e)} {
	$hd.exc.bot.b1 configure -state normal
	$hd.exc.bot.b2 configure -state normal
	$hd.exc.bot.ent configure -state normal
    } else {
	$hd.exc.bot.b1 configure -state disabled
	$hd.exc.bot.b2 configure -state disabled
	$hd.exc.bot.ent configure -state disabled
    }
}

proc ToggleSubstitutionEntryActive {n e o} {
    set hd [lindex [split $e ","] 1]
    if {$::KeyInfo($e)} {
	$hd.sub.bot.b1 configure -state normal
	$hd.sub.bot.b2 configure -state normal
	$hd.sub.bot.ent configure -state normal
    } else {
	$hd.sub.bot.b1 configure -state disabled
	$hd.sub.bot.b2 configure -state disabled
	$hd.sub.bot.ent configure -state disabled
    }
}

proc ToggleCustomSortOrderEntryActive {n e o} {
    set hd [lindex [split $e ","] 1]
    switch $::KeyInfo($e) {
	2 {
	    $hd.so.bot.b1 configure -state normal
	    $hd.so.bot.b2 configure -state normal
	    $hd.so.bot.ent configure -state normal
	    $hd.so.loc.ent configure -state disabled
	}
	1 {
	    $hd.so.bot.b1 configure -state disabled
	    $hd.so.bot.b2 configure -state disabled
	    $hd.so.bot.ent configure -state disabled
	    $hd.so.loc.ent configure -state normal
	}
	0 {
	    $hd.so.bot.b1 configure -state disabled
	    $hd.so.bot.b2 configure -state disabled
	    $hd.so.bot.ent configure -state disabled
	    $hd.so.loc.ent configure -state disabled
	}
    }
    PackKeyPage $hd;
}

proc ToggleCustomSortOrderSeparatorsEntryActive {n e o} {
    set hd [lindex [split $e ","] 1]
    if {$::KeyInfo($e)} {
	$hd.sos.bot.b1 configure -state normal
	$hd.sos.bot.b2 configure -state normal
	$hd.sos.bot.ent configure -state normal
    } else {
	$hd.sos.bot.b1 configure -state disabled
	$hd.sos.bot.b2 configure -state disabled
	$hd.sos.bot.ent configure -state disabled
    }
}

proc ToggleOnSortType {n e o} {
    set hd [lindex [split $e ","] 1]
    set SortType $::KeyInfo($e)

    if {$SortType == 5} {
	$hd.st.cnt.dfb configure -state normal
    } else {
	$hd.st.cnt.dfb configure -state disabled
    }
    PackKeyPage $hd
}

proc ToggleFieldSeparatorEntryActive {n e o} {
     if {$::WholeRecordIsKeyP} {
	.gen.ctr.wr.fsent configure -state disabled
    } else {
	.gen.ctr.wr.fsent configure -state normal
	focus .gen.ctr.wr.fsent
    }
}

proc ToggleRecordSeparatorEntryActive {n e o} {
    .gen.ctr.wr.fsent delete 0 end
    switch $::RecordParse {
	0 {
	    .gen.ctr.rp.cent configure -state disabled
	    .gen.ctr.rp.fl.len configure -state disabled
	    .gen.ctr.wr.fsent insert insert {\t\040}
	}
	1 {
	    if {$::EOLCarriageReturnP} {
		.gen.ctr.wr.fsent insert insert {\015}
	    } else {
		.gen.ctr.wr.fsent insert insert {\n}
	    }
	    .gen.ctr.rp.fl.len configure -state disabled
	    .gen.ctr.rp.cent configure -state disabled
	}
	2 {
	    .gen.ctr.rp.fl.len configure -state disabled
	    .gen.ctr.rp.cent configure -state normal
	    focus .gen.ctr.rp.cent;
	}
	3 {
	    .gen.ctr.rp.fl.len configure -state normal
	    focus .gen.ctr.rp.fl.len
	    .gen.ctr.rp.cent configure -state disabled
	}
    }
}

proc UpdateDefaultFieldSeparator {n e o} {
    if {$::RecordParse == 1} {
	.gen.ctr.wr.fsent delete 0 end;
	if {$::EOLCarriageReturnP} {
	    .gen.ctr.wr.fsent insert insert {\015}
	}  else {
	    .gen.ctr.wr.fsent insert insert {\n}
	}
    }
}

proc ToggleOptComparisonActive {n e o} {
    set hd [lindex [split $e ","] 1]
    if {$::KeyInfo($e)} {
	$hd.key.opt.oc.rdb3 configure -state normal
	$hd.key.opt.oc.rdb4 configure -state normal
	$hd.key.opt.oc.rdb5 configure -state normal
    } else {
	$hd.key.opt.oc.rdb3 configure -state disabled
	$hd.key.opt.oc.rdb4 configure -state disabled
	$hd.key.opt.oc.rdb5 configure -state disabled
    }
}

#Character entry code begins here
#Set this to the path to character entry menu
set CEM .menubar.charentry
#Set the default insertion target here.
set InsertionTarget .gen.ctr.wr.fsent;

set CharentryMenuItems 6;
set IPAAIsDisplayedP 0;
set IPACIsDisplayedP 0;
set IPAVIsDisplayedP 0;
set IPADIsDisplayedP 0;
set CharEntryByCodeIsDisplayedP 0;

set DisplayConsonantChartColumnLabelsP 1
set DisplayConsonantChartRowLabelsP 1
set DisplayVowelChartColumnLabelsP 1
set DisplayVowelChartRowLabelsP 1

option add *ipaec.Button.font CharacterEntryFont 100
option add *ipaec.Label.background  $ColorSpecs(IPAEntry,Background)
option add *ipaec.Button.background $ColorSpecs(IPAEntry,Background)
option add *ipaec.Button.foreground $ColorSpecs(IPAEntry,Foreground)
option add *ipaec.Label.relief raised

option add *ipaev.Button.font CharacterEntryFont 100
option add *ipaev.Label.background  $ColorSpecs(IPAEntry,Background)
option add *ipaev.Button.background $ColorSpecs(IPAEntry,Background)
option add *ipaev.Button.foreground $ColorSpecs(IPAEntry,Foreground)
option add *ipaev.Label.relief raised

option add *ipaea.Button.font AccentedLetterFont 100
option add *ipaea.Label.background  "\#E0E0E0"
option add *ipaea.Button.background "\#FFFFFF"
option add *ipaea.Button.foreground "\#000000"
option add *ipaea.Label.relief raised

option add *ipaed.Button.font DiacriticFont 100
#option add *ipaed.Label.background  $ColorSpecs(IPAEntry,Background)
option add *ipaed.Button.background "\#E0E0FF"
option add *ipaed.Label.background "\#E0E0E0"
option add *ipaed.Button.foreground "\#000000"
option add *ipaed.Label.relief raised

proc SetInsertionTargets {tgt} {
    set ::InsertionTarget $tgt;
    set ::CharByCodeInsertionTarget $tgt;
}
#Insert character entry stuff here
set CustomCEMEntries 0;
option add *ipae*.*Label.font CharacterEntryLabelFont 100
option add *ipaec.*Button.font CharacterEntryFont 100
namespace eval ipaentry {
    variable IPAECColumnLabelList [list];
    variable IPAECRowLabelList [list];
    variable IPAEVColumnLabelList [list];
    variable IPAEVRowLabelList [list];

    set row 0;
    variable poa $row;
    incr row;
    variable vlstop $row;
    incr row;
    variable vdstop $row;
    incr row;
    variable vdimplosive $row;
    incr row;
    variable click $row;
    incr row;
    variable nasal  $row;
    incr row;
    variable trill $row;
    incr row;
    variable tap $row;
    incr row;
    variable vlfric $row;
    incr row;
    variable vdfric $row;
    incr row;
    variable vlaffric $row;
    incr row;
    variable vdaffric $row;
    incr row;
    variable approx $row;
    incr row; 
    variable latapprox $row;
    variable lastrow $row;

    set col 0;
    variable manner $col; 
    incr col;
    variable labial $col; 
    incr col;
    variable labiodental $col; 
    incr col;
    variable dental $col; 
    incr col;
    variable alveolar $col;
    incr col;
    variable postalveolar $col; 
    incr col;
    variable retroflex $col; 
    incr col;
    variable palatal $col; 
    incr col;
    variable velar $col; 
    incr col;
    variable uvular $col; 
    incr col;
    variable pharyngeal $col; 
    incr col;
    variable epiglottal $col; 
    incr col;
    variable glottal $col;
    variable lastcolumn $col;

    proc cfesh {} {
	variable vlfric
	variable postalveolar
	eval [list [format ".ipaec.r%dc%d" $vlfric $postalveolar] configure -text "\u0161"];
    }
    proc cresh {} {
	variable vlfric
	variable postalveolar
	eval [list [format ".ipaec.r%dc%d" $vlfric $postalveolar] configure -text "\u0283"];
    }

    proc cfeshr {} {
	variable vlfric
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vlfric $retroflex] configure -text "\u1E63"];
    }
    proc creshr {} {
	variable vlfric
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vlfric $retroflex] configure -text "\u0282"];
    }

    proc cryod {} {
	variable approx
	variable palatal
	eval [list [format ".ipaec.r%dc%d" $approx $palatal] configure -text "j"];
    }

    proc cfyod {} {
	variable approx
	variable palatal
	eval [list [format ".ipaec.r%dc%d" $approx $palatal] configure -text "y"];
    }

    proc crny {} {
	variable nasal
	variable palatal
	eval [list [format ".ipaec.r%dc%d" $nasal $palatal] configure -text "\u0272"];
    }
    proc cfny {} {
	variable nasal
	variable palatal
	eval [list [format ".ipaec.r%dc%d" $nasal $palatal] configure -text "\u00F1"];
    }

    proc crch {} {
	variable vlaffric
	variable postalveolar
	eval [list [format ".ipaec.r%dc%d" $vlaffric $postalveolar] configure -text "t\u0283"];
    }
    proc cfch {} {
	variable vlaffric
	variable postalveolar
	eval [list [format ".ipaec.r%dc%d" $vlaffric $postalveolar] configure -text "\u010D"];
    }

    proc crchr {} {
	variable vlaffric
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vlaffric $retroflex] configure -text "\u0288\u0282"];
    }

    proc cfchr {} {
	variable vlaffric
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vlaffric $retroflex] configure -text "\u1E6D\u1E63"];
    }


    proc crdj {} {
	eval [list [format ".ipaec.r%dc%d" $ipaentry::vdaffric $ipaentry::postalveolar] configure -text "d\u0292"];
    }

    proc cfdj {} {
	eval [list [format ".ipaec.r%dc%d" $ipaentry::vdaffric $ipaentry::postalveolar] configure -text "\u01C6"];
    }

    proc crdjr {} {
	eval [list [format ".ipaec.r%dc%d" $ipaentry::vdaffric $ipaentry::retroflex] configure -text "\u0256\u0290"];
    }

    proc cfdjr {} {
	eval [list [format ".ipaec.r%dc%d" $ipaentry::vdaffric $ipaentry::retroflex] configure -text "\u1E0D\u1E93"];
    }

    proc crzh {} {
	variable vdfric
	variable postalveolar
	eval [list [format ".ipaec.r%dc%d" $vdfric $postalveolar] configure -text "\u0292"];
    }
    proc cfzh {} {
	variable vdfric
	variable postalveolar
	eval [list [format ".ipaec.r%dc%d" $vdfric $postalveolar] configure -text "\u017E"];
    }

    proc crtr {} {
	variable vlstop
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vlstop $retroflex] configure -text "\u0288"];
    }
    proc cftr {} {
	variable vlstop
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vlstop $retroflex] configure -text "\u1E6D"];
    }

    proc crdr {} {
	variable vdstop
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vdstop $retroflex] configure -text "\u0256"];
    }
    proc cfdr {} {
	variable vdstop
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vdstop $retroflex] configure -text "\u1E0D"];
    }


    proc crnr {} {
	variable nasal
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $nasal $retroflex] configure -text "\u0273"];
    }
    proc cfnr {} {
	variable nasal
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $nasal $retroflex] configure -text "\u1E47"];
    }


    proc crzhr {} {
	variable vdfric
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vdfric $retroflex] configure -text "\u0290"];
    }
    proc cfzhr {} {
	variable vdfric
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $vdfric $retroflex] configure -text "\u1E93"];
    }

    proc crflapr {} {
	variable tap;
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $tap $retroflex] configure -text "\u027D"];
    }
    proc cfflapr {} {
	variable tap;
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $tap $retroflex] configure -text "\u1E5B"];
    }

    proc crlr {} {
	variable latapprox;
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $latapprox $retroflex] configure -text "\u026D"];
    }
    proc cflr {} {
	variable latapprox;
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $latapprox $retroflex] configure -text "\u1E37"];
    }

    proc crrr {} {
	variable approx;
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $approx $retroflex] configure -text "\u027B"];
    }
    proc cfrr {} {
	variable approx;
	variable retroflex
	eval [list [format ".ipaec.r%dc%d" $approx $retroflex] configure -text "\u1E5B"];
    }

    proc bhcham {w} {
	balloonhelp_for $w "The standard IPA character is shown by default. When the mouse\npointer is over the button, the \"North American\" variant is shown.\nRight click to insert the variant."
    }

    proc bhsp {w} {
	balloonhelp_for $w "The characters below and to the right are characters\nthat do not fit neatly into the chart."
    }

    proc UnpackConsonantColumnLabels {} {
	variable IPAECColumnLabelList;
	foreach l $IPAECColumnLabelList {
	    grid forget $l;
	}
    }

    proc PackConsonantColumnLabels {} {
	destroy .ipaec
	PopupIPAEntryC;
    }

    proc UnpackConsonantRowLabels {} {
	variable IPAECRowLabelList;
	foreach l $IPAECRowLabelList {
	    grid forget $l;
	}
    }

    proc PackConsonantRowLabels {} {
	destroy .ipaec;
	PopupIPAEntryC;
    }

    proc UnpackVowelColumnLabels {} {
	variable IPAEVColumnLabelList;
	foreach l $IPAEVColumnLabelList {
	    grid forget $l;
	}
    }

    proc PackVowelColumnLabels {} {
	destroy .ipaev;
	PopupIPAEntryV;
    }

    proc UnpackVowelRowLabels {} {
	variable IPAEVRowLabelList;
	foreach l $IPAEVRowLabelList {
	    grid forget $l;
	}
    }

    proc PackVowelRowLabels {} {
	destroy .ipaev;
	PopupIPAEntryV;
    }

    proc PopupIPAEntryC {} {
	variable poa
	variable vlstop 
	variable vdstop
	variable vdimplosive 
	variable click 
	variable nasal  
	variable trill 
	variable tap 
	variable vlfric 
	variable vdfric 
	variable vlaffric 
	variable vdaffric 
	variable approx 
	variable latapprox
	variable lastrow

	variable manner 
	variable labial 
	variable labiodental 
	variable dental 
	variable alveolar 
	variable postalveolar 
	variable retroflex 
	variable palatal 
	variable velar 
	variable uvular 
	variable pharyngeal 
	variable epiglottal 
	variable glottal
	variable lastcolumn

	variable IPAECColumnLabelList;
	variable IPAECRowLabelList;
	global DisplayConsonantChartColumnLabelsP
	global DisplayConsonantChartRowLabelsP

	set IPAECColumnLabelList [list];
	set IPAECRowLabelList [list];
	
	set xp 2;
	set yp 3;
	set spcolor 	\#c46276;
	toplevel .ipaec -borderwidth 3 -class CharEntry
	wm title .ipaec [_ "Consonant Symbols"]
	BindKeys .ipaec;
	set DownMsg [_ "Display IPA Consonant Chart"];
	set UpMsg   [_ "Remove IPA Consonant Chart"];
	bind .ipaec <Destroy> \
	    "set ::IPACIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPACIndex -label \"$DownMsg\""
	bind .ipaec <Unmap> \
	    "set ::IPACIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPACIndex -label \"$DownMsg\""
	bind .ipaec <Map> \
	    "set ::IPACIsDisplayedP 1;$::CEM entryconfigure $::ToggleIPACIndex -label \"$UpMsg\""
	set tmp [label [format ".ipaec.r%dc%d" $manner $poa] \
	  -text [_ "IPA\nConsonants"]  -padx 5 -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList $tmp;
	lappend IPAECRowLabelList $tmp;
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $labial] \
	    -text [_ "labial"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $labiodental] \
	    -text [_ "labio\ndental"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $dental] \
	    -text [_ "dental"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $alveolar] \
	    -text [_ "alveolar"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $postalveolar] \
	    -text [_ "post\nalveolar"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $retroflex] \
	    -text [_ "retro\nflex"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $palatal] \
	    -text [_ "palatal"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $velar] \
	    -text [_ "velar"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $uvular] \
	    -text [_ "uvular"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $pharyngeal] \
	    -text [_ "pharyn\ngeal"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $epiglottal] \
	    -text [_ "epi\nglottal"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAECColumnLabelList [label [format ".ipaec.r%dc%d" $manner $glottal] \
	    -text [_ "glottal"]   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]

	#Voiceless oral stops
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vlstop $poa] \
	       -text [_ "voiceless stops"] -anchor w  -padx $xp -pady $yp \
	       -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list p]}
	set bn [format ".ipaec.r%dc%d"  $vlstop $labial]
	button $bn  -text "p" -padx $xp -pady $yp -command  $cmd \
	    -fg $::ColorSpecs(UserTextEntry,Foreground); #debug
	balloonhelpd_for $bn [_ "voiceless labial stop"]

	label [format ".ipaec.r%dc%d" $vlstop $labiodental] \
	    -text " " -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list t\u032A]}
	set bn [format ".ipaec.r%dc%d"  $vlstop $dental]
	button $bn  -text "t\u032A"   -padx $xp -pady $yp -command  $cmd
	balloonhelpd_for $bn [_ "voiceless dental stop"]

	set cmd {$::InsertionTarget insert insert [list t]}
	set bn [format ".ipaec.r%dc%d"  $vlstop $alveolar]
	button $bn -text "t"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless alveolar stop"]

	label [format ".ipaec.r%dc%d"  $vlstop $postalveolar] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u0288]}
	set bn [format ".ipaec.r%dc%d"  $vlstop $retroflex]
	button $bn  -text \u0288   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E6D]}
	bind $bn <Enter> {ipaentry::cftr}
	bind $bn <Leave> {ipaentry::crtr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiceless retroflex stop"]

	set cmd {$::InsertionTarget insert insert [list c]}
	set bn [format ".ipaec.r%dc%d"  $vlstop $palatal]
	button $bn -text "c"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless palatal stop"]

	set cmd {$::InsertionTarget insert insert [list k]}
	set bn [format ".ipaec.r%dc%d" $vlstop $velar]
	button $bn -text "k" -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless velar stop"]

	set cmd {$::InsertionTarget insert insert [list q]}
	set bn [format ".ipaec.r%dc%d" $vlstop $uvular]
	button $bn  -text "q"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless uvular stop"]

	label [format ".ipaec.r%dc%d" $vlstop $pharyngeal] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $vlstop $epiglottal] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u0294]}
	set bn [format ".ipaec.r%dc%d" $vlstop $glottal]
	button $bn -text \u0294   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless glottal stop"]

	#Voiced oral stops
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vdstop $poa] \
	       -text [_ "voiced stops"] -anchor w   -padx $xp -pady $yp \
	       -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list b]}
	set bn [format ".ipaec.r%dc%d" $vdstop $labial]
	button $bn -text "b"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced bilabial stop"]

	label [format ".ipaec.r%dc%d" $vdstop $labiodental] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list d\u032A]}
	set bn [format ".ipaec.r%dc%d" $vdstop $dental]
	button $bn -text "d\u032A"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced dental stop"]

	set cmd {$::InsertionTarget insert insert [list d]}
	set bn [format ".ipaec.r%dc%d" $vdstop $alveolar]
	button  $bn -text "d"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced alveolar stop"]

	label [format ".ipaec.r%dc%d"  $vdstop $postalveolar] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u0256]}
	set bn [format ".ipaec.r%dc%d"  $vdstop $retroflex]
	button $bn  -text \u0256   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E0D]}
	bind $bn <Enter> {ipaentry::cfdr}
	bind $bn <Leave> {ipaentry::crdr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiced retroflex stop"]

	set cmd {$::InsertionTarget insert insert [list \u025F]}
	set bn [format ".ipaec.r%dc%d" $vdstop $palatal]
	button $bn -text \u025F   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced palatal stop"]


	set cmd {$::InsertionTarget insert insert [list g]}
	set bn [format ".ipaec.r%dc%d" $vdstop $velar]
	button $bn -text "g"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced velar stop"]

	set cmd {$::InsertionTarget insert insert [list \u0262]}
	set bn [format ".ipaec.r%dc%d" $vdstop $uvular]
	button $bn -text \u0262  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced uvular stop"]

	label [format ".ipaec.r%dc%d" $vdstop $pharyngeal] -text " "  -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u02A1]}
	set bn [format ".ipaec.r%dc%d" $vdstop $epiglottal]
	button $bn -text \u02A1  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced epiglottal stop"]

	label [format ".ipaec.r%dc%d" $vdstop $glottal] -text " "  -padx $xp -pady $yp

	#Nasals
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $nasal $poa] \
	       -text [_ "nasals"] -anchor w -padx $xp -pady $yp \
	       -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list m]}
	set bn [format ".ipaec.r%dc%d" $nasal $labial]
	button $bn -text "m" -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "bilabial nasal"]

	set cmd {$::InsertionTarget insert insert [list \u0271]}
	set bn [format ".ipaec.r%dc%d" $nasal $labiodental]
	button $bn -text \u0271   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "labiodental nasal"]

	set cmd {$::InsertionTarget insert insert [list n\u032A]}
	set bn [format ".ipaec.r%dc%d" $nasal $dental]
	button $bn -text n\u032A   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "dental nasal"]

	set cmd {$::InsertionTarget insert insert [list n]}
	set bn [format ".ipaec.r%dc%d" $nasal $alveolar]
	button $bn -text "n"      -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "alveolar nasal"]

	label [format ".ipaec.r%dc%d"  $nasal $postalveolar] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u0273]}
	set bn [format ".ipaec.r%dc%d"  $nasal $retroflex]
	button $bn  -text \u0273   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E47]}
	bind $bn <Enter> {ipaentry::cfnr}
	bind $bn <Leave> {ipaentry::crnr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "retroflex nasal"]

	set cmd {$::InsertionTarget insert insert [list \u0272]}
	set bn [format ".ipaec.r%dc%d" $nasal $palatal]
	button $bn  -text \u0272    -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u00F1]}
	bind $bn <Enter> {ipaentry::cfny}
	bind $bn <Leave> {ipaentry::crny}
	bhcham $bn;
	balloonhelpd_for $bn [_ "palatal nasal"]

	set cmd {$::InsertionTarget insert insert [list \u014B]}
	set bn [format ".ipaec.r%dc%d" $nasal $velar]
	button $bn -text \u014B    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "velar nasal"]

	set cmd {$::InsertionTarget insert insert [list \u0274]}
	set bn [format ".ipaec.r%dc%d" $nasal $uvular]
	button $bn -text \u0274    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "uvular nasal"]

	label [format ".ipaec.r%dc%d" $nasal $pharyngeal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $nasal $epiglottal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $nasal $glottal] -text " "    -padx $xp -pady $yp

	#Voiceless fricatives
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vlfric $poa] \
	       -text [_ "voiceless fricatives"] -anchor w  -padx $xp -pady $yp \
	       -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list \u03C6]}
	set bn [format ".ipaec.r%dc%d" $vlfric $labial]
	button $bn -text \u03C6  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless bilabial fricative"]

	set cmd {$::InsertionTarget insert insert [list f]}
	set bn [format ".ipaec.r%dc%d" $vlfric $labiodental]
	button $bn  -text "f"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless labiodental fricative"]

	set cmd {$::InsertionTarget insert insert [list \u03B8]}
	set bn [format ".ipaec.r%dc%d" $vlfric $dental]
	button $bn -text \u03B8   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless dental fricative"]

	set cmd {$::InsertionTarget insert insert [list "s"]}
	set bn [format ".ipaec.r%dc%d" $vlfric $alveolar]
	button $bn -text "s"  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless alveolar fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0283]}
	set bn [format ".ipaec.r%dc%d" $vlfric $postalveolar];
	button $bn -text \u0283   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u0161]}
	bind $bn <Enter> {ipaentry::cfesh}
	bind $bn <Leave> {ipaentry::cresh}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiceless postalveolar fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0282]}
	set bn [format ".ipaec.r%dc%d" $vlfric $retroflex];
	button $bn -text \u0282   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E63]}
	bind $bn <Enter> {ipaentry::cfeshr}
	bind $bn <Leave> {ipaentry::creshr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiceless retroflex fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0255]}
	set bn [format ".ipaec.r%dc%d" $vlfric $palatal]
	button $bn -text \u0255  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless palatal fricative"]

	set cmd {$::InsertionTarget insert insert [list "x"]}
	set bn [format ".ipaec.r%dc%d" $vlfric $velar]
	button $bn -text "x" -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless velar fricative"]

	set cmd {$::InsertionTarget insert insert [list "\u03C7"]}
	set bn [format ".ipaec.r%dc%d" $vlfric $uvular]
	button $bn -text "\u03C7"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless uvular fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0127]}
	set bn [format ".ipaec.r%dc%d" $vlfric $pharyngeal]
	button $bn -text \u0127 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless pharyngeal fricative"]

	set cmd {$::InsertionTarget insert insert [list \u029C]}
	set bn [format ".ipaec.r%dc%d" $vlfric $epiglottal]
	button $bn -text \u029C   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless epiglottal fricative"]

	set cmd {$::InsertionTarget insert insert [list "h"]}
	set bn [format ".ipaec.r%dc%d" $vlfric $glottal]
	button $bn -text "h" -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless glottal fricative"]

	#Voiced fricatives
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vdfric $poa] \
	       -text [_ "voiced fricatives"] -anchor w \
	       -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list \u03B2]}
	set bn [format ".ipaec.r%dc%d" $vdfric $labial]
	button $bn -text \u03B2  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced bilabial fricative"]

	set cmd {$::InsertionTarget insert insert [list v]}
	set bn [format ".ipaec.r%dc%d" $vdfric $labiodental]
	button $bn -text "v" -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced labiodental fricative"]

	set cmd {$::InsertionTarget insert insert [list "\u00F0"]}
	set bn [format ".ipaec.r%dc%d" $vdfric $dental]
	button $bn -text "\u00F0"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced dental fricative"]

	set cmd {$::InsertionTarget insert insert [list "z"]}
	set bn [format ".ipaec.r%dc%d" $vdfric $alveolar]
	button $bn -text "z"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced alveolar fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0292]}
	set bn [format ".ipaec.r%dc%d" $vdfric $postalveolar]
	button $bn -text \u0292  -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u017E]}
	bind $bn <Enter> {ipaentry::cfzh}
	bind $bn <Leave> {ipaentry::crzh}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiced postalveolar fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0290]}
	set bn [format ".ipaec.r%dc%d" $vdfric $retroflex]
	button $bn -text \u0290   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E93]}
	bind $bn <Enter> {ipaentry::cfzhr}
	bind $bn <Leave> {ipaentry::crzhr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiced retroflex fricative"]

	set cmd {$::InsertionTarget insert insert [list \u029D]}
	set bn [format ".ipaec.r%dc%d" $vdfric $palatal]
	button $bn -text  \u029D  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced palatal fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0263]}
	set bn [format ".ipaec.r%dc%d" $vdfric $velar]
	button $bn -text \u0263  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced velar fricative"]

	set cmd {$::InsertionTarget insert insert [list "\u0281"]}
	set bn [format ".ipaec.r%dc%d" $vdfric $uvular]
	button $bn -text "\u0281" -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced uvular fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0295]}
	set bn [format ".ipaec.r%dc%d" $vdfric $pharyngeal]
	button $bn -text \u0295 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced pharyngeal fricative"]

	set cmd {$::InsertionTarget insert insert [list \u02A2]}
	set bn [format ".ipaec.r%dc%d" $vdfric $epiglottal] 
	button $bn -text \u02A2    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced epiglottal fricative"]

	set cmd {$::InsertionTarget insert insert [list \u0266]}
	set bn [format ".ipaec.r%dc%d" $vdfric $glottal]
	button $bn -text \u0266    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced glottal fricative"]

	#trills
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $trill $poa] \
	       -text [_ "trills"] -anchor w   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list \u0299]}
	set bn [format ".ipaec.r%dc%d" $trill $labial]
	button  $bn -text \u0299      -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "bilabial trill"]

	label [format ".ipaec.r%dc%d" $trill $labiodental] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $trill $dental] -text " "   -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list r]}
	set bn [format ".ipaec.r%dc%d" $trill $alveolar]
	button $bn -text "r"      -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "alveolar trill"]

	label [format ".ipaec.r%dc%d"  $trill $postalveolar] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d"  $trill $retroflex] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $trill $palatal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $trill $velar] -text " "    -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u0280]}
	set bn [format ".ipaec.r%dc%d" $trill $uvular]
	button $bn -text \u0280    -padx $xp -pady $yp
	balloonhelpd_for $bn [_ "uvular trill"] -command $cmd

	label [format ".ipaec.r%dc%d" $trill $pharyngeal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $trill $epiglottal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $trill $glottal] -text " "    -padx $xp -pady $yp

	#taps and flaps
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $tap $poa] \
	       -text [_ "taps/flaps"] -anchor w   -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	label [format ".ipaec.r%dc%d" $tap $labial] -text " "      -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $labiodental] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $dental] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u027E]}
	set bn [format ".ipaec.r%dc%d" $tap $alveolar]
	button $bn -text \u027E -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "uvular trill"]

	label [format ".ipaec.r%dc%d"  $tap $postalveolar] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u027D]}
	set bn [format ".ipaec.r%dc%d"  $tap $retroflex];
	button $bn -text \u027D   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E5B]}
	bind $bn <Enter> {ipaentry::cfflapr}
	bind $bn <Leave> {ipaentry::crflapr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "retroflex tap"]

	label [format ".ipaec.r%dc%d" $tap $palatal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $velar] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $uvular] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $pharyngeal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $epiglottal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $tap $glottal] -text " "    -padx $xp -pady $yp

	#approximants
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $approx $poa] \
	       -text [_ "approximants"] -anchor w \
	       -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	label [format ".ipaec.r%dc%d" $approx $labial] -text " "      -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list \u028B]}
	set bn [format ".ipaec.r%dc%d" $approx $labiodental]
	button $bn -text \u028B   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "labiodental approximant"]

	label [format ".ipaec.r%dc%d" $approx $dental] -text " "   -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list \u0279]}
	set bn [format ".ipaec.r%dc%d" $approx $alveolar]
	button $bn -text  \u0279 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "alveolar approximant"]

	label [format ".ipaec.r%dc%d"  $approx $postalveolar] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u027B]}
	set bn [format ".ipaec.r%dc%d"  $approx $retroflex]
	button $bn -text \u027B   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E5B]}
	bind $bn <Enter> {ipaentry::cfrr}
	bind $bn <Leave> {ipaentry::crrr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "retroflex approximant"]

	set cmd {$::InsertionTarget insert insert [list "j"]}
	set bn [format ".ipaec.r%dc%d" $approx $palatal]
	button $bn -text "j"    -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list "y"]}
	bind $bn <Enter> {ipaentry::cfyod}
	bind $bn <Leave> {ipaentry::cryod}
	bhcham $bn;
	balloonhelpd_for $bn [_ "palatal approximant"]

	set cmd {$::InsertionTarget insert insert [list \u0270]}
	set bn [format ".ipaec.r%dc%d" $approx $velar]
	button $bn  -text \u0270    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "velar approximant"]

	label [format ".ipaec.r%dc%d" $approx $uvular] -text " "  \
	    -padx $xp -pady $yp -bg $spcolor
	label [format ".ipaec.r%dc%d" $approx $pharyngeal] -text " "  \
	    -padx $xp -pady $yp -bg $spcolor

	#label [format ".ipaec.r%dc%d" $approx $epiglottal] -text " "  \
	    -padx $xp -pady $yp -bg $spcolor
	#label [format ".ipaec.r%dc%d" $approx $glottal] -text " "   \
	    -padx $xp -pady $yp -bg $spcolor

	#lateral approximants
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $latapprox $poa] \
	       -text [_ "lateral approximants"] -anchor w \
	       -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	label [format ".ipaec.r%dc%d" $latapprox $labial] -text " "      -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $latapprox $labiodental] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $latapprox $dental] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list "l"]}
	set bn [format ".ipaec.r%dc%d" $latapprox $alveolar]
	button $bn -text "l"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "alveolar lateral approximant"]

	label [format ".ipaec.r%dc%d"  $latapprox $postalveolar] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u026D]}
	set bn [format ".ipaec.r%dc%d"  $latapprox $retroflex]
	button $bn  -text \u026D   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E37]}
	bind $bn <Enter> {ipaentry::cflr}
	bind $bn <Leave> {ipaentry::crlr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "retroflex lateral approximant"]

	set cmd {$::InsertionTarget insert insert [list \u028E]}
	set bn [format ".ipaec.r%dc%d" $latapprox $palatal]
	button $bn -text \u028E    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "palatal lateral approximant"]

	set cmd {$::InsertionTarget insert insert [list \u029F]}
	set bn [format ".ipaec.r%dc%d" $latapprox $velar]
	button $bn -text \u029F    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "velar lateral approximant"]

	label [format ".ipaec.r%dc%d" $latapprox $uvular] -text " "  \
	    -padx $xp -pady $yp -bg $spcolor
	#label [format ".ipaec.r%dc%d" $latapprox $pharyngeal] -text " "  \
	    -padx $xp -pady $yp -bg $spcolor
	#label [format ".ipaec.r%dc%d" $latapprox $epiglottal] -text " "  \
	    -padx $xp -pady $yp
	#label [format ".ipaec.r%dc%d" $latapprox $glottal] -text " "  \
	    -padx $xp -pady $yp

	#voiced implosives
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vdimplosive $poa] \
	       -text [_ "voiced implosives"] -anchor w \
	       -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list \u0253]}
	set bn [format ".ipaec.r%dc%d" $vdimplosive $labial]
	button $bn -text \u0253 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced bilabial implosive"]

	label [format ".ipaec.r%dc%d" $vdimplosive $labiodental] -text " " \
	    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $vdimplosive $dental] -text " "  \
	    -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u0257]}
	set bn [format ".ipaec.r%dc%d" $vdimplosive $alveolar]
	button $bn -text \u0257   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced alveolar implosive"]

	label [format ".ipaec.r%dc%d"  $vdimplosive $postalveolar] -text " " \
	    -padx $xp -pady $yp

	#Not officially recognized by IPA yet.
	set cmd {$::InsertionTarget insert insert [list \u1D91]}
	set bn [format ".ipaec.r%dc%d"  $vdimplosive $retroflex]
	button $bn -text \u1D91  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced retroflex implosive"]

	set cmd {$::InsertionTarget insert insert [list \u0284]}
	set bn [format ".ipaec.r%dc%d" $vdimplosive $palatal]
	button $bn -text \u0284  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced palatal implosive"]

	set cmd {$::InsertionTarget insert insert [list \u0260]}
	set bn [format ".ipaec.r%dc%d" $vdimplosive $velar]
	button $bn -text \u0260 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced velar implosive"]

	set cmd {$::InsertionTarget insert insert [list \u029B]}
	set bn [format ".ipaec.r%dc%d" $vdimplosive $uvular]
	button $bn -text \u029B    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced uvular implosive"]

	label [format ".ipaec.r%dc%d" $vdimplosive $pharyngeal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $vdimplosive $epiglottal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $vdimplosive $glottal] -text " "    -padx $xp -pady $yp

	#clicks
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $click $poa] \
	       -text [_ "clicks"] -anchor w  -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list \u0298]}
	set bn [format ".ipaec.r%dc%d" $click $labial]
	button $bn -text \u0298      -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "labial click"]

	label [format ".ipaec.r%dc%d" $click $labiodental] -text " "   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list \u01C0]}
	set bn [format ".ipaec.r%dc%d" $click $dental]
	button $bn -text \u01C0   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "dental click"]

	set cmd {$::InsertionTarget insert insert [list \u01C2]}
	set bn [format ".ipaec.r%dc%d" $click $alveolar]
	button $bn -text \u01C2   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "alveolar click"]

	set cmd {$::InsertionTarget insert insert [list \u01C3]}
	set bn [format ".ipaec.r%dc%d"  $click $postalveolar]
	button $bn -text \u01C3 \
	    -padx $xp -pady $yp  -command $cmd
	balloonhelpd_for $bn [_ "postalveolar click"]

	label [format ".ipaec.r%dc%d"  $click $retroflex] -text " "   -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $click $palatal] -text " "  -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $click $velar] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $click $uvular] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $click $pharyngeal] -text " "    -padx $xp -pady $yp
	label [format ".ipaec.r%dc%d" $click $epiglottal] -text " "    -padx $xp -pady $yp
	set cmd {BackDelete $::InsertionTarget}
	set bn [format ".ipaec.r%dc%d" $click $glottal];
	button $bn -text [_ "Delete"]  -padx $xp -pady $yp -command $cmd -background "\#E0D0FF";
	set DeleteButton $bn;


	#Voiceless affricates
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vlaffric $poa] \
	       -text [_ "voiceless affricates"] -anchor w -padx $xp -pady $yp \
	       -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list p\u03C6]}
	set bn [format ".ipaec.r%dc%d" $vlaffric $labial]
	button $bn -text "p\u03C6"      -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless bilabial affricate"]

	label [format ".ipaec.r%dc%d" $vlaffric $labiodental] -text "" -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list t\u03B8]}
	set bn [format ".ipaec.r%dc%d" $vlaffric $dental]
	button $bn -text "t\u03B8"   -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless dental affricate"]

	set cmd {$::InsertionTarget insert insert [list "ts"]}
	set bn [format ".ipaec.r%dc%d" $vlaffric $alveolar]
	button $bn -text "ts"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless alveolar affricate"]

	set cmd {$::InsertionTarget insert insert [list t\u0283]}
	set bn [format ".ipaec.r%dc%d"  $vlaffric $postalveolar]
	button $bn -text "t\u0283"   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u010D]}
	bind $bn <Enter> {ipaentry::cfch}
	bind $bn <Leave> {ipaentry::crch}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiceless postalveolar affricate"]

	set cmd {$::InsertionTarget insert insert [list \u0288\u0282]}
	set bn [format ".ipaec.r%dc%d"  $vlaffric $retroflex]
	button $bn -text \u0288\u0282   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E6D\u1E63]}
	bind $bn <Enter> {ipaentry::cfchr}
	bind $bn <Leave> {ipaentry::crchr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiceless retroflex affricate"]

	set cmd {$::InsertionTarget insert insert [list t\u0255]}
	set bn [format ".ipaec.r%dc%d" $vlaffric $palatal]
	button $bn -text t\u0255 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless palatal affricate"]

	set cmd {$::InsertionTarget insert insert [list "kx"]}
	set bn [format ".ipaec.r%dc%d" $vlaffric $velar]
	button $bn -text "kx"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless velar affricate"]

	set cmd {$::InsertionTarget insert insert [list "q\u03C7"]}
	set bn [format ".ipaec.r%dc%d" $vlaffric $uvular]
	button $bn -text "q\u03C7"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiceless uvular affricate"]

	label [format ".ipaec.r%dc%d" $vlaffric $pharyngeal] -text "" \
	    -padx $xp -pady $yp -bg $spcolor
	label [format ".ipaec.r%dc%d" $vlaffric $epiglottal] -text "" \
	    -padx $xp -pady $yp -bg $spcolor
	label [format ".ipaec.r%dc%d" $vlaffric $glottal] -text "" \
	    -padx $xp -pady $yp -bg $spcolor

	#Voiced affricate
	lappend IPAECRowLabelList [label [format ".ipaec.r%dc%d" $vdaffric $poa] \
	       -text [_ "voiced affricates"] -anchor w -padx $xp -pady $yp \
	       -bg $::ColorSpecs(IPAHeadings,Background)]

	set cmd {$::InsertionTarget insert insert [list b\u03B2]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $labial]
	button $bn -text b\u03B2      -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced bilabial affricate"]

	label [format ".ipaec.r%dc%d" $vdaffric $labiodental] -text ""   -padx $xp -pady $yp

	set cmd {$::InsertionTarget insert insert [list "d\u00F0"]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $dental]
	button $bn -text "d\u00F0"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced dental affricate"]

	set cmd {$::InsertionTarget insert insert [list "dz"]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $alveolar]
	button  $bn -text "dz"  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced alveolar affricate"]

	set cmd {$::InsertionTarget insert insert [list d\u0292]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $postalveolar]
	button $bn -text d\u0292  -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u01C6]}
	bind $bn <Enter> {ipaentry::cfdj}
	bind $bn <Leave> {ipaentry::crdj}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiced postalveolar affricate"]

	set cmd {$::InsertionTarget insert insert [list \u0256\u0290]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $retroflex]
	button $bn -text \u0256\u0290   -padx $xp -pady $yp -command $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u1E0D\u1E93]}
	bind $bn <Enter> {ipaentry::cfdjr}
	bind $bn <Leave> {ipaentry::crdjr}
	bhcham $bn;
	balloonhelpd_for $bn [_ "voiced retroflex affricate"]

	set cmd {$::InsertionTarget insert insert [list d\u029D]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $palatal]
	button $bn -text  d\u029D  -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced palatal affricate"]

	set cmd {$::InsertionTarget insert insert [list g\u0263]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $velar]
	button $bn -text g\u0263    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced velar affricate"]

	set cmd {$::InsertionTarget insert insert [list "\u0262\u0281"]}
	set bn [format ".ipaec.r%dc%d" $vdaffric $uvular]
	button $bn -text "\u0262\u0281"    -padx $xp -pady $yp -command $cmd
	balloonhelpd_for $bn [_ "voiced uvular affricate"]

	label [format ".ipaec.r%dc%d" $vdaffric $pharyngeal] -text "" \
	    -padx $xp -pady $yp -bg $spcolor
	#label [format ".ipaec.r%dc%d" $vdaffric $epiglottal] -text "" \
	    -padx $xp -pady $yp -bg $spcolor
	#label [format ".ipaec.r%dc%d" $vdaffric $glottal] -text ""   \
	    -padx $xp -pady $yp -bg $spcolor

	set cmd {$::InsertionTarget insert insert [list "\u028D"]}
	button .ipaec.r13c10 -text \u028D -padx $xp -pady $yp -command $cmd
	balloonhelpd_for .ipaec.r13c10  [_ "Voiceless labio-velar fricative"]
	set cmd {$::InsertionTarget insert insert [list "\u0265"]}
	button .ipaec.r11c11 -text \u0265 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for .ipaec.r11c11  [_ "Voiced labio-palatal approximant"]
	set cmd {$::InsertionTarget insert insert [list "\u0077"]}
	button .ipaec.r11c12 -text \u0077 -padx $xp -pady $yp -command $cmd
	balloonhelpd_for .ipaec.r11c12  [_ "Voiced labio-velar approximant"]
	set cmd {$::InsertionTarget insert insert [list "\u01C1"]}
	button .ipaec.r12c11 -text \u01C1 -padx $xp -pady $yp -command $cmd
	set cmd {$::InsertionTarget insert insert [list "\u02A0"]}
	button .ipaec.r12c12 -text \u02A0 -padx $xp -pady $yp -command $cmd
	set cmd {$::InsertionTarget insert insert [list "\u026C"]}
	button .ipaec.r13c11 -text \u026C -padx $xp -pady $yp -command $cmd
	set cmd {$::InsertionTarget insert insert [list "\u026E"]}
	button .ipaec.r13c12 -text \u026E -padx $xp -pady $yp -command $cmd

	for {set row 0} {$row <= $lastrow} {incr row} {
	    set line [list];
	    for {set col 0} {$col <= $lastcolumn} {incr col} {
		set cell [format ".ipaec.r%dc%d" $row $col]
		lappend line $cell
		if {[string equal [winfo class $cell] "Label"]} {
		    balloonhelp_for $cell  [_ "Left click to insert standard characters.\nRight click to insert alternative characters."]
		}
	    }
	    eval grid $line -sticky news;
	}
	balloonhelpd_for .ipaec.r12c11  [_ "Lateral click"]
	balloonhelpd_for .ipaec.r12c12  [_ "Voiceless uvular implosive"]
	balloonhelpd_for .ipaec.r13c11  [_ "Voiceless lateral fricative"]
	balloonhelpd_for .ipaec.r13c12  [_ "Voiced lateral fricative"]
	bhsp [format ".ipaec.r%dc%d" $latapprox $uvular]
	bhsp [format ".ipaec.r%dc%d" $approx $uvular]
	bhsp [format ".ipaec.r%dc%d" $approx $pharyngeal]
	bhsp [format ".ipaec.r%dc%d" $vlaffric $pharyngeal]
	bhsp [format ".ipaec.r%dc%d" $vlaffric $epiglottal]
	bhsp [format ".ipaec.r%dc%d" $vlaffric $glottal]
	bhsp [format ".ipaec.r%dc%d" $vdaffric $pharyngeal]
	balloonhelpd_for $DeleteButton [_ "No, there isn't a glottal click  called \"Delete\".\nThis is so that you can correct mistakes while using\nthe mouse, without having to go back to the keyboard."]
	after idle {
	    update idletasks
	    raise .ipaec;
	}
	if {!$DisplayConsonantChartColumnLabelsP} {UnpackConsonantColumnLabels}
	if {!$DisplayConsonantChartRowLabelsP} {UnpackConsonantRowLabels}
    }

    set row 0;
    variable frontness $row;
    incr row;
    variable close $row;
    incr row;
    variable closeclosemid $row;
    incr row;
    variable closemid $row;
    incr row;
    variable midmid $row;
    incr row;
    variable openmid $row;
    incr row;
    variable openopenmid $row;
    incr row;
    variable open $row;
    variable lastvrow $row;

    set col 0;
    variable height $col; 
    incr col;
    variable fronturd $col;
    incr col;
    variable frontrnd $col;
    incr col;
    variable centurd $col;
    incr col;
    variable centrnd $col;
    incr col;
    variable backurd $col;
    incr col;
    variable backrnd $col;
    variable lastvcolumn $col;

    proc vry {} {
	variable close
	variable frontrnd
	eval [list [format ".ipaev.r%dc%d" $close $frontrnd] configure -text "y"];
	}
    
    proc vfy {} {
	variable close
	variable frontrnd
	eval [list [format ".ipaev.r%dc%d" $close $frontrnd] configure -text "\u00FC"];
	}

    proc vrphi {} {
	variable frontrnd;
	variable closemid
	eval [list [format ".ipaev.r%dc%d" $closemid $frontrnd] configure -text "\u00F8"];
    }
    
    proc vfphi {} {
	variable frontrnd;
	variable closemid
	eval [list [format ".ipaev.r%dc%d" $closemid $frontrnd] configure -text "o\u0308"];
    }

    proc vroe {} {
	variable frontrnd;
	variable openmid
	eval [list [format ".ipaev.r%dc%d" $openmid $frontrnd] configure -text "\u0153"];
    }

    proc vfoe {} {
	variable frontrnd;
	variable openmid
	eval [list [format ".ipaev.r%dc%d" $openmid $frontrnd] configure -text "\u0254\0308"];
    }


    proc PopupIPAEntryV {} {
	variable frontness
	variable close
	variable closeclosemid
	variable closemid
	variable openmid
	variable midmid
	variable openopenmid
	variable open
	variable lastvrow

	variable height
	variable fronturd
	variable frontrnd
	variable centurd
	variable centrnd
	variable backurd
	variable backrnd
	variable lastvcolumn

	variable IPAEVColumnLabelList
	variable IPAEVRowLabelList
	global DisplayVowelChartColumnLabelsP
	global DisplayVowelChartRowLabelsP

	set xp 2;
	set yp 3;
	toplevel .ipaev -borderwidth 3
	wm title .ipaev [_ "Vowel Symbols"]
	BindKeys .ipaev;
	set DownMsg [_ "Display IPA Vowel Chart"];
	set UpMsg   [_ "Remove IPA Vowel Chart"];
	bind .ipaev <Destroy> \
	    "set ::IPAVIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPAVIndex -label \"$DownMsg\""
	bind .ipaev <Unmap> \
	    "set ::IPAVIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPAVIndex -label \"$DownMsg\""
	bind .ipaev <Map> \
	    "set ::IPAVIsDisplayedP 1;$::CEM entryconfigure $::ToggleIPAVIndex -label \"$UpMsg\""

	#Column headers
	set tmp [label [format ".ipaev.r%dc%d" $height $frontness] \
	     -text [_ "IPA\nVowels"]  -padx $xp -pady $yp -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAEVColumnLabelList $tmp;
	lappend IPAEVRowLabelList $tmp;
	lappend IPAEVColumnLabelList [label [format ".ipaev.r%dc%d" $height $fronturd] \
	     -text [_ "Front\nUnrounded"]  -padx $xp -pady $yp \
     	     -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAEVColumnLabelList [label [format ".ipaev.r%dc%d" $height $frontrnd] \
	     -text [_ "Front\nRounded"]  -padx $xp -pady $yp \
  	     -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAEVColumnLabelList [label [format ".ipaev.r%dc%d" $height $centurd] \
	     -text [_ "Central\nUnrounded"]  -padx $xp -pady $yp \
	     -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAEVColumnLabelList [label [format ".ipaev.r%dc%d" $height $centrnd] \
	     -text [_ "Central\nRounded"]  -padx $xp -pady $yp \
   	     -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAEVColumnLabelList [label [format ".ipaev.r%dc%d" $height $backurd] \
	     -text [_ "Back\nUnrounded"]  -padx $xp -pady $yp \
  	     -bg $::ColorSpecs(IPAHeadings,Background)]
	lappend IPAEVColumnLabelList [label [format ".ipaev.r%dc%d" $height $backrnd] \
	     -text [_ "Back\nRounded"]  -padx $xp -pady $yp \
  	     -bg $::ColorSpecs(IPAHeadings,Background)]


	#Close vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $close $frontness] \
	   -text [_ "Close"] -anchor w  -padx $xp -pady $yp \
  	   -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list i]}
	button [format ".ipaev.r%dc%d"  $close $fronturd] \
	     -text "i"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list y]}
	set bn  [format ".ipaev.r%dc%d"  $close $frontrnd]
	button $bn -text "y"   -padx $xp -pady $yp -command  $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u00FC]}
	bind $bn <Enter> {ipaentry::vfy}
	bind $bn <Leave> {ipaentry::vry}
	bhcham $bn;

	set cmd {$::InsertionTarget insert insert [list \u0268]}
	button [format ".ipaev.r%dc%d"  $close $centurd] \
	     -text "\u0268"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0289]}
	button [format ".ipaev.r%dc%d"  $close $centrnd] \
	     -text "\u0289"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0268]}
	button [format ".ipaev.r%dc%d"  $close $backurd] \
	     -text "\u0268"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list u]}
	button [format ".ipaev.r%dc%d"  $close $backrnd] \
	     -text "u"   -padx $xp -pady $yp -command  $cmd

	# Close close mid vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $closeclosemid $frontness] \
	     -text [_ "Close Close Mid"] -anchor w  -padx $xp -pady $yp \
	  -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list I]}
	button [format ".ipaev.r%dc%d"  $closeclosemid $fronturd] \
	     -text "I"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list Y]}
	button [format ".ipaev.r%dc%d"  $closeclosemid $frontrnd] \
	     -text "Y"   -padx $xp -pady $yp -command  $cmd
	label [format ".ipaev.r%dc%d"  $closeclosemid $centurd] \
	     -text ""   -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list \u028A]}
	button [format ".ipaev.r%dc%d"  $closeclosemid $centrnd] \
	     -text "\u028A"   -padx $xp -pady $yp -command  $cmd
	label [format ".ipaev.r%dc%d"  $closeclosemid $backurd] \
	     -text ""   -padx $xp -pady $yp
	label [format ".ipaev.r%dc%d"  $closeclosemid $backrnd] \
	     -text ""   -padx $xp -pady $yp

	# Close mid vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $closemid $frontness] \
	     -text [_ "Close Mid"] -anchor w  -padx $xp -pady $yp \
	  -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list e]}
	button [format ".ipaev.r%dc%d"  $closemid $fronturd] \
	     -text "e"   -padx $xp -pady $yp -command  $cmd

	set cmd {$::InsertionTarget insert insert [list \u00F8]}
	set bn [format ".ipaev.r%dc%d"  $closemid $frontrnd]
	button $bn -text "\u00F8"   -padx $xp -pady $yp -command  $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list o\u0308]}
	bind $bn <Enter> {ipaentry::vfphi}
	bind $bn <Leave> {ipaentry::vrphi}
	bhcham $bn;

	set cmd {$::InsertionTarget insert insert [list \u0258]}
	button [format ".ipaev.r%dc%d"  $closemid $centurd] \
	     -text "\u0258"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0275]}
	button [format ".ipaev.r%dc%d"  $closemid $centrnd] \
	     -text "\u0275"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0264]}
	button [format ".ipaev.r%dc%d"  $closemid $backurd] \
	     -text "\u0264"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list o]}
	button [format ".ipaev.r%dc%d"  $closemid $backrnd] \
	     -text "o"   -padx $xp -pady $yp -command  $cmd

	# Mid mid vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $midmid $frontness] \
	     -text [_ "Mid Mid"] -anchor w  -padx $xp -pady $yp \
	  -bg $::ColorSpecs(IPAHeadings,Background)]
	label [format ".ipaev.r%dc%d"  $midmid $fronturd] \
	     -text ""   -padx $xp -pady $yp
	label [format ".ipaev.r%dc%d"  $midmid $frontrnd] \
	     -text ""   -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list \u0259]}
	button [format ".ipaev.r%dc%d"  $midmid $centurd] \
	     -text "\u0259"   -padx $xp -pady $yp -command  $cmd
	set cmd {BackDelete $::InsertionTarget}
	set bn [format ".ipaev.r%dc%d" $midmid $centrnd];
	button $bn -text [_ "Delete"]  \
	     -padx $xp -pady $yp -command $cmd -background "\#E0D0FF";
	set DeleteButton $bn;
	label [format ".ipaev.r%dc%d"  $midmid $backurd] -text ""   -padx $xp -pady $yp
	label [format ".ipaev.r%dc%d"  $midmid $backrnd] -text ""   -padx $xp -pady $yp

	# Open mid vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $openmid $frontness] \
	     -text [_ "Open Mid"] -anchor w  -padx $xp -pady $yp \
  	     -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list \u025B]}
	button [format ".ipaev.r%dc%d"  $openmid $fronturd] \
	     -text \u025B   -padx $xp -pady $yp -command  $cmd

	set cmd {$::InsertionTarget insert insert [list \u0153]}
	set bn [format ".ipaev.r%dc%d"  $openmid $frontrnd]
	button $bn -text "\u0153"   -padx $xp -pady $yp -command  $cmd
	bind $bn <<B3>> {$::InsertionTarget insert insert [list \u0254\u0308]}
	bind $bn <Enter> {ipaentry::vfoe}
	bind $bn <Leave> {ipaentry::vroe}
	bhcham $bn;

	set cmd {$::InsertionTarget insert insert [list \u025C]}
	button [format ".ipaev.r%dc%d"  $openmid $centurd] \
	     -text "\u025C"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u025E]}
	button [format ".ipaev.r%dc%d"  $openmid $centrnd] \
	     -text "\u025E"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u028C]}
	button [format ".ipaev.r%dc%d"  $openmid $backurd] \
	     -text "\u028C"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0254]}
	button [format ".ipaev.r%dc%d"  $openmid $backrnd] \
	     -text "\u0254"   -padx $xp -pady $yp -command  $cmd

	# Open open mid vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $openopenmid $frontness] \
	     -text [_ "Open Open Mid"] -anchor w  -padx $xp -pady $yp \
  	     -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list \u00E6]}
	button [format ".ipaev.r%dc%d"  $openopenmid $fronturd] \
	     -text \u00E6   -padx $xp -pady $yp -command  $cmd
	label [format ".ipaev.r%dc%d"  $openopenmid $frontrnd] -text ""   -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list \u0250]}
	button [format ".ipaev.r%dc%d"  $openopenmid $centurd] \
	     -text "\u0250"   -padx $xp -pady $yp -command  $cmd
	label [format ".ipaev.r%dc%d"  $openopenmid $centrnd] \
		     -text ""   -padx $xp -pady $yp
	label [format ".ipaev.r%dc%d"  $openopenmid $backurd] \
	     -text ""   -padx $xp -pady $yp
	label [format ".ipaev.r%dc%d"  $openopenmid $backrnd] \
	     -text ""   -padx $xp -pady $yp

	# Open  vowels
	lappend IPAEVRowLabelList [label [format ".ipaev.r%dc%d" $open $frontness] \
	   -text [_ "Open"] -anchor w  -padx $xp -pady $yp \
	   -bg $::ColorSpecs(IPAHeadings,Background)]
	set cmd {$::InsertionTarget insert insert [list a]}
	button [format ".ipaev.r%dc%d"  $open $fronturd] \
	     -text "a"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0276]}
	button [format ".ipaev.r%dc%d"  $open $frontrnd] \
	     -text "\u0276"   -padx $xp -pady $yp -command  $cmd
	label [format ".ipaev.r%dc%d"  $open $centurd] \
	     -text ""   -padx $xp -pady $yp
	label [format ".ipaev.r%dc%d"  $open $centrnd] \
	     -text ""   -padx $xp -pady $yp
	set cmd {$::InsertionTarget insert insert [list \u0251]}
	button [format ".ipaev.r%dc%d"  $open $backurd] \
	     -text "\u0251"   -padx $xp -pady $yp -command  $cmd
	set cmd {$::InsertionTarget insert insert [list \u0252]}
	button [format ".ipaev.r%dc%d"  $open  $backrnd] \
	     -text "\u0252"   -padx $xp -pady $yp -command  $cmd

	for {set row 0} {$row <= $lastvrow} {incr row} {
	    set line [list];
	    for {set col 0} {$col <= $lastvcolumn} {incr col} {
		set cell [format ".ipaev.r%dc%d" $row $col]
		lappend line $cell
		balloonhelp_for $cell  \
		    [_ "Left click to insert standard characters.\n\
Right click to insert alternative characters."]
	    }
	    eval grid $line -sticky news;
	}
	balloonhelpd_for $DeleteButton [_ "No, there isn't a mid-mid central rounded vowel called \"Delete\".\nThis is so that you can correct mistakes while using\nthe mouse, without having to go back to the keyboard."]
	if {!$DisplayVowelChartColumnLabelsP} {UnpackVowelColumnLabels}
	if {!$DisplayVowelChartRowLabelsP} {UnpackVowelRowLabels}
    }

#Rearrange into functional groups, e.g. tone marks, vowel quality marks,
#etc. Separate IPA and non-IPA.

    proc PopupIPAEntryD {} {
	set PerRow 10;
	set BaseChar "o";
	set xp 2;
	set yp 3;
	toplevel .ipaed -borderwidth 3
	wm title .ipaed [_ "Phonetic Diacritics"]
	BindKeys .ipaed
	set DownMsg [_ "Display IPA Diacritic Chart"];
	set UpMsg   [_ "Remove IPA Diacritic Chart"];
	bind .ipaed <Destroy> \
	    "set ::IPADIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPADIndex -label \"$DownMsg\""
	bind .ipaed <Unmap> \
	    "set ::IPADIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPADIndex -label \"$DownMsg\""
	bind .ipaed <Map> \
	    "set ::IPADIsDisplayedP 1;$::CEM entryconfigure $::ToggleIPADIndex -label \"$UpMsg\""
	array set Diacritics [list\
				  0  {"\u02D0" "long"}\
				  1  {"\u02D1" "half long"}\
				  2  {"\u0303" "nasalized"}\
				  3  {"\u0324" "breathy voice"}\
				  4  {"\u0325" "voiceless"}\
				  5  {"\u032C" "voiced"}\
				  6  {"\u031A" "unreleased"}\
				  7  {"\u031C" "lower/open variety of vowel"}\
				  8  {"\u031D" "raised/closed variety of vowel"}\
				  9  {"\u031E" "lower/open variety of vowel"}\
				  10  {"\u031F" "advanced/fronted"}\
				  11  {"\u0320" "retracted/backed"}\
				  12  {"\u0321" "palatalized"}\
				  13  {"\u0322" "retroflex"}\
				  14  {"\u0323" "closer variety of vowel/retroflex"}\
				  15  {"\u0329" "syllabic"}\
				  16  {"\u032A" "dental"}\
				  17  {"\u032B" "labialized"}\
				  18  {"\u0330" "creaky voice"}\
				  19  {"\u0334" "velarized/pharyngealized"}\
				  20  {"\u0346" "dentolabial"}\
				  21  {"\u0347" "alveolar"}\
				  22  {"\u0348" "strong articulation"}\
				  23  {"\u0349" "weak articulation"}\
				  24  {"\u02CA" "high tone"}\
				  25  {"\u02CB" "low tone"}\
				  26  {"\u0302" "falling tone"}\
				  27  {"\u0304" "long"}\
				  28  {"\u0306" "short"}\
				  29  {"\u0308" "non-canonical backness"}\
				  30  {"\u02CC" "secondary stress"}\
				  31  {"\u02C8" "primary stress/downstep"}\
				  32  {"\u02B8" "palatalized"}\
				  33  {"\u02DE" "rhotacized"}\
				  34  {"\u0328" "nasalized"}\
				  35  {"\u034A" "denasal"}\
				  36  {"\u034B" "nasal escape"}\
				  37  {"\u034C" "velopharyngeal friction"}\
				  38  {"\u034D" "labial spreading"}\
				  39  {"\u034E" "whistled"}
				 ];

	set DiacriticCnt [llength [array names Diacritics]];
	set Rows [expr ceil(double($DiacriticCnt)/double($PerRow))]
	set Total [expr $PerRow * $Rows]
	#Generate buttons with blank labels for empty padding slots
	for {set k 0} {$k < $Total} {incr k} {
	    set row [expr $k/$PerRow]
	    set col [expr $k%$PerRow]
	    if {[info exist Diacritics($k)]} {
		button .ipaed.r${row}c${col} -text "$BaseChar[lindex $Diacritics($k) 0]"\
		    -padx $xp -pady $yp\
		    -command "\$::InsertionTarget insert insert  [lindex $Diacritics($k) 0]"
		balloonhelpd_for .ipaed.r${row}c${col} [_ [lindex $Diacritics($k) 1]]  
	    } else {
		label .ipaed.r${row}c${col} -text "";
	    }
	    set LastRow $row;
	    set LastCol $col;
	}
	#Lay the buttons and labels out in a grid.
	for {set row 0} {$row <= $LastRow} {incr row} {
	    set line [list];
	    for {set col 0} {$col <= $LastCol} {incr col} {
		set cell [format ".ipaed.r%dc%d" $row $col]
		lappend line $cell
	    }
	    eval grid $line -sticky news;
	}
    }

#Accented letters
    proc PopupIPAEntryA {} {
	set PerRow 12;
	set xp 3;
	set yp 3;
	toplevel .ipaea -borderwidth 4
	wm title .ipaea [_ "Accented Letters"]
	BindKeys .ipaea
	set DownMsg [_ "Display Accented Letter Chart"]
	set UpMsg   [_ "Display Accented Letter Chart"]
	bind .ipaea <Destroy> \
	    "set ::IPAAIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPAAIndex -label \"$DownMsg\""
	bind .ipaea <Unmap> \
	    "set ::IPAAIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPAAIndex -label \"$DownMsg\""
	bind .ipaea <Map> \
	    "set ::IPAAIsDisplayedP 0;$::CEM entryconfigure $::ToggleIPAAIndex -label \"$UpMsg\""
	set AccentedLetters [list \
				 {"\u00E0" "a with grave"}\
				 {"\u00E1" "a with acute"}\
				 {"\u00E2" "a with circumflex"}\
				 {"\u00E3" "a with tilde"}\
				 {"\u00E4" "a with diaresis"}\
				 {"\u00E5" "a with ring above"}\
				 {"\u0101" "a with macron"}\
				 {"\u0103" "a with breve"}\
				 {"\u0105" "a with ogonek"}\
				 {"\u01CE" "a with caron"}\
				 {"\u0201" "a with double grave"}\
				 {"\u0203" "a with inverted breve"}\
				 {"\u01DF" "a with diaresis and macron"}\
				 {"\u01E1" "a with dot above and macron"}\
				 {"\u01FB" "a with ring above"}\
				 {"\u0227" "a with dot above"}\
				 {"\u1EA1" "a with dot below"}\
				 {"\u01E3" "ae with macron"}\
				 {"\u01FD" "ash with acute"}\
				 {"\u0180" "b with stroke"}\
				 {"\u0183" "b with topbar"}\
				 {"\u1E03" "b with dot above"}\
				 {"\u1E05" "b with dot below"}\
				 {"\u1E07" "b with line below"}\
				 {"\u0188" "c with hook"}\
				 {"\u0107" "c with acute"}\
				 {"\u0109" "c with circumflex"}\
				 {"\u010B" "c with dot above"}\
				 {"\u00E7" "c with cedilla"}\
				 {"\u1E09" "c with cedilla and acute"}\
				 {"\u010D" "c with caron"}\
				 {"\u018c" "d with topbar"}\
				 {"\u010F" "d with caron"}\
				 {"\u0111" "d with stroke"}\
				 {"\u0221" "d with curl"}\
				 {"\u1E0B" "d with dot above"}\
				 {"\u1E0D" "d with dot below"}\
				 {"\u1E0F" "d with line below"}\
				 {"\u021F" "h with caron"}\
				 {"\u00E8" "e with grave"}\
				 {"\u00E9" "e with acute"}\
				 {"\u00EA" "e with circumflex"}\
				 {"\u00EB" "e with diaresis"}\
				 {"\u0113" "e with macron"}\
				 {"\u0115" "e with breve"}\
				 {"\u0117" "e with dot above"}\
				 {"\u1EB9" "e with dot below"}\
				 {"\u0119" "e with ogonek"}\
				 {"\u011B" "e with caron"}\
				 {"\u0229" "e with cedilla"}\
				 {"\u0205" "e with double grave"}\
				 {"\u0207" "e with inverted breve"}\
				 {"\u01F5" "g with acute"}\
				 {"\u011D" "g with circumflex"}\
				 {"\u011F" "g with breve"}\
				 {"\u0121" "g with dot above"}\
				 {"\u0123" "g with cedilla"}\
				 {"\u01E5" "g with stroke"}\
				 {"\u01E7" "g with caron"}\
				 {"\u0125" "h with circumflex"}\
				 {"\u0127" "h with stroke"}\
				 {"\u1E25" "h with dot below"}\
				 {"\u00EC" "i with grave"}\
				 {"\u00ED" "i with acute"}\
				 {"\u00EE" "i with circumflex"}\
				 {"\u00EF" "i with diaresis"}\
				 {"\u0129" "i with tilde"}\
				 {"\u012B" "i with macron"}\
				 {"\u012D" "i with breve"}\
				 {"\u012F" "i with ogonek"}\
				 {"\u01D0" "i with caron"}\
				 {"\u0209" "i with double grave"}\
				 {"\u020B" "i with inverted breve"}\
				 {"\u1ECB" "i with dot below"}\
				 {"\u0131" "dotless i"}\
				 {"\u0135" "j with circumflex"}\
				 {"\u01F0" "j with caron"}\
				 {"\u01E9" "k with caron"}\
				 {"\u0199" "k with hook"}\
				 {"\u0137" "k with cedilla "}\
				 {"\u1E33" "k with dot below"}\
				 {"\u013A" "l with acute"}\
				 {"\u013C" "l with cedilla"}\
				 {"\u013E" "l with caron"}\
				 {"\u0140" "l with middle dot"}\
				 {"\u0142" "l with stroke"}\
				 {"\u019A" "l with bar"}\
				 {"\u0234" "l with curl"}\
				 {"\u1E37" "l with dot below"}\
				 {"\u1E39" "l with dot below and macron"}\
				 {"\u1E3B" "l with line below"}\
				 {"\u019B" "lambda with stroke"}\
				 {"\u0235" "n with a curl"}\
				 {"\u0146" "n with cedilla"}\
				 {"\u0148" "n with caron"}\
				 {"\u01F9" "n with grave"}\
				 {"\u00F1" "n with tilde"}\
				 {"\u019E" "n with long right leg"}\
				 {"\u0144" "n with acute"}\
				 {"\u1E43" "m with dot below"}\
				 {"\u1E45" "n with dot above"}\
				 {"\u1E47" "n with dot below"}\
				 {"\u1E49" "n with line below"}\
				 {"\u00F2" "o with grave"}\
				 {"\u00F3" "o with acute"}\
				 {"\u00F4" "o with circumflex"}\
				 {"\u00F5" "o with tilde"}\
				 {"\u00F6" "o with diaresis"}\
				 {"\u00F8" "o with stroke"}\
				 {"\u014D" "o with macron"}\
				 {"\u014F" "o with breve"}\
				 {"\u0151" "o with double acute"}\
				 {"\u01A1" "o with horn"}\
				 {"\u01EB" "o with ogonek"}\
				 {"\u01ED" "o with ogonek and macron"}\
				 {"\u01D2" "o with caron"}\
				 {"\u022B" "o with diaresis and macron"}\
				 {"\u022D" "o with tilde and macron"}\
				 {"\u022F" "o with dot above"}\
				 {"\u01FF" "o with stroke and acute"}\
				 {"\u020D" "o with double grave"}\
				 {"\u020F" "o with inverted breve"}\
				 {"\u0231" "o with dot above and macron"}\
				 {"\u1ECD" "o with dot below"}\
				 {"\u0223" "ou"}\
				 {"\u01A5" "p with hook"}\
				 {"\u0155" "r with acute"}\
				 {"\u0157" "r with cedilla"}\
				 {"\u0159" "r with caron"}\
				 {"\u0211" "r with double grave"}\
				 {"\u0213" "r with inverted breve"}\
				 {"\u1E5B" "r with dot below"}\
				 {"\u027F" "reversed r with fishhook"}\
				 {"\u015B" "s with acute"}\
				 {"\u015D" "s with circumflex"}\
				 {"\u015F" "s with cedilla"}\
				 {"\u0161" "s with caron"}\
				 {"\u0219" "s with comma below"}\
				 {"\u1E63" "s with dot below"}\
				 {"\u021B" "t with comma below"}\
				 {"\u01AB" "t with palatal hook"}\
				 {"\u01AD" "t with hook"}\
				 {"\u0163" "t wwith cedilla"}\
				 {"\u0165" "t with caron"}\
				 {"\u0167" "t with stroke"}\
				 {"\u0236" "t with curl"}\
				 {"\u00F9" "u with grave"}\
				 {"\u00FA" "u with acute"}\
				 {"\u00FB" "u with circumflex"}\
				 {"\u00FC" "u with diaresis"}\
				 {"\u01D4" "u with caron"}\
				 {"\u01D6" "u with diaresis and macron"}\
				 {"\u01D8" "u with diaresis and acute"}\
				 {"\u01DA" "u with diaresis and caron"}\
				 {"\u01DC" "u with diaresis and grave"}\
				 {"\u01B0" "u with horn"}\
				 {"\u0169" "u with tilde"}\
				 {"\u016B" "u with macron"}\
				 {"\u016D" "u with breve"}\
				 {"\u016F" "u with ring above"}\
				 {"\u0171" "u with double acute"}\
				 {"\u0173" "u with ogonek"}\
				 {"\u0215" "u with double grave"}\
				 {"\u0217" "u with inverted breve"}\
				 {"\u1EE5" "u with dot below"}\
				 {"\u01B4" "y with hook"}\
				 {"\u00FD" "y with acute"}\
				 {"\u00FF" "y with diaresis"}\
				 {"\u0233" "y with macron"}\
				 {"\u0175" "w with circumflex"}\
				 {"\u0177" "y with circumflex"}\
				 {"\u017A" "z with acute"}\
				 {"\u017C" "z with dot above"}\
				 {"\u017E" "z with caron"}\
				 {"\u01B6" "z with stroke"}\
				 {"\u0225" "z with hook"}\
				 {"\u01EF" "ezh with caron"}\
							 ];

#Done through 0236 			 {"\u" ""}\

	set AccentedLetterCnt [llength $AccentedLetters];
	set Rows [expr ceil(double($AccentedLetterCnt)/double($PerRow))]
	set Total [expr $PerRow * $Rows]
	#Generate buttons with blank labels for empty padding slots
	for {set k 0} {$k < $Total} {incr k} {
	    set row [expr $k/$PerRow]
	    set col [expr $k%$PerRow]
	    if {$k < $AccentedLetterCnt} {
		button .ipaea.r${row}c${col} -text "[lindex [lindex $AccentedLetters $k] 0]"\
		    -padx $xp -pady $yp\
		    -command "\$::InsertionTarget insert insert  [lindex [lindex $AccentedLetters $k] 0]"
		balloonhelpd_for .ipaea.r${row}c${col} [_ [lindex [lindex $AccentedLetters $k] 1] ]  
	    } else {
		label .ipaea.r${row}c${col} -text "";
	    }
	}
	set LastRow $row;
	set LastCol $col;
	set LastCell .ipaea.r${row}c${col}
	destroy $LastCell
	button $LastCell -image $::LeftArrowImage -command {BackDelete $::InsertionTarget}
	balloonhelpd_for $LastCell  [_ "Delete, so that you can correct mistakes while using\nthe mouse, without having to go back to the keyboard."]
	
	#Lay the buttons and labels out in a grid.
	for {set row 0} {$row <= $LastRow} {incr row} {
	    set line [list];
	    for {set col 0} {$col <= $LastCol} {incr col} {
		set cell [format ".ipaea.r%dc%d" $row $col]
		lappend line $cell
	    }
	    eval grid $line -sticky news;
	}
    }

}



set LeftArrowImage [image create photo -data {
R0lGODlhFAAUAIAAAAAAAL+/vyH5BAEAAAEALAAAAAAUABQAAAIqjI+py43gGIAxTVrNxXVz
54WTIopWGZ7oRq7X4o4wm2lvbX+ZjGv9/ysAADs=
}]

proc BackDelete {w} {
    if {[string equal [winfo class $w] "Text"]} {
	set Insert [$w index insert]
	set FirstHalf  [$w get 1.0 $Insert-1chars]
	set SecondHalf [string trimright [$w get insert end]]
	$w delete 1.0 end;
	$w insert 1.0 $FirstHalf$SecondHalf;
    } else {
	set delind [expr [$w index insert] -1];
	$w delete $delind;
    }
}

proc PopupCharEntryByCode {} {
    toplevel .charent -borderwidth 4 -relief raised
    BindKeys .charent;
    wm title .charent "Entry By Codepoint";
    after idle {
	update idletasks
	set xmax [winfo screenwidth .charent]
	set ymax [winfo screenheight .charent]
	set x0 [expr 1 * ($xmax -[winfo reqwidth .charent])/3];
	set y0 [expr 1 * ($ymax -[winfo reqheight .charent])/3];
	wm geometry .charent "+$x0+$y0";
    }
    label .charent.title -text [_ "Insert Character by Numerical Code"]
    frame .charent.ef;
    entry .charent.ef.ent -width 8 -font MainFont -relief flat -bg \#E0EEFF;
    label .charent.ef.prefix -text "0x" -font MainFont
    pack .charent.ef.prefix -side left  -expand 1 -fill x -anchor e
    pack .charent.ef.ent    -side right -expand 1 -fill x -anchor w
    button .charent.d  -text [_ "Dismiss"] -command {destroy .charent}
    button .charent.x  -text [_ "Delete"] -command [list BackDelete $::CharByCodeInsertionTarget]
    button .charent.i  -text [_ "Insert"] -command {InsertUnicode}
    pack .charent.title -side top
    pack .charent.ef -side top
    pack .charent.d -side left -expand 1 -fill both
    pack .charent.x -side left -expand 1 -fill both
    pack .charent.i -side right -expand 1 -fill both
    focus .charent.ef.ent;
    set DownMsg [_ "Display Widget for Entering Characters by Unicode Code"];
    set UpMsg   [_ "Remove Widget for Entering Characters by Unicode Code"];
    bind .charent <Destroy> \
	"set ::CharEntryByCodeIsDisplayedP 0;$::CEM entryconfigure $::ToggleCharEntryByCodeIndex -label \"$DownMsg\""
    bind .charent <Unmap> \
	"set ::CharEntryByCodeIsDisplayedP 0;$::CEM entryconfigure $::ToggleCharEntryByCodeIndex -label \"$DownMsg\""
    bind .charent <Map> \
	"set ::CharEntryByCodeIsDisplayedP 1;$::CEM entryconfigure $::ToggleCharEntryByCodeIndex -label \"$UpMsg\""
    bind .charent.ef.ent <Destroy> {set ::InsertionTarget  [focus -lastfor .gen]}
    bind .charent.ef.ent <Return> {InsertUnicode}
    bind .charent.ef.ent <Control-k> {.charent.ef.ent delete 0 end;break}


    set charentbh  [_ "Enter a character by its Unicode code, as a sequence of four hexadecimal digits.\nFor example, you may specify \'\u0298\', the International Phonetic Alphabet\nsymbol for the bilabial click, as 0298, and the Chinese character \'\u4E39\' \"egg\" as 4E39.\nDo not type the prefix 0x shown to the left of the entry box.\nIt is entered for you automatically. Return inserts the current character.\nControl-k erases the current character code."]
    balloonhelp_for .charent $charentbh
    balloonhelp_for .charent.title $charentbh
    balloonhelp_for .charent.ef.ent $charentbh
    balloonhelp_for .charent.i $charentbh
    balloonhelp_for .charent.d $charentbh
    balloonhelp_for .charent.x $charentbh
    set ::CharEntryByCodeIsDisplayedP 1;
}

#Get entry from popup, validate it, and insert it into the insertion target.
proc InsertUnicode {} {
    ClearMessageWindow;
    set str [.charent.ef.ent get];
    if {[string length $str] == 0} {
	ShowMessage [_ "The empty string is not a valid Unicode codepoint"]
	return ;
    }
    #Validate
    set BadValueP 0;
    #Make sure all digits are hex and that prefix is appropriate.
    if {[regexp {[[:xdigit:]]{4,4}} $str] == 0} {
	ShowMessage [_ "$str is not a well-formed hexadecimal Unicode value"]
	return;
    }
    set str [format "0x%s" $str]
    if {[scan $str "%x" num] < 1} {
	ShowMessage [format [_ "Ill-formed code %s"] $str]
	return;
    }

    #Reject illegal codepoints.
    if {$num  >  65535} {
	ShowMessage [_ "Tcl/Tk may not support codepoints outside the BMP (Plane 0)."]
    }
    set BadValueP 0;
    set BadRanges [list\
  	       0x0750 0x077F\
	       0x07C0 0x08FF\
	       0x1380 0x139F\
	       0x18B0 0x18FF\
	       0x1980 0x19DF\
	       0x1A00 0x1CFF\
	       0x1D80 0x1DFF\
	       0x2C00 0x2E7F\
	       0x2FE0 0x2FEF\
	       0x31C0 0x31EF\
	       0x9FB0 0x9FFF\
	       0xA4D0 0xABFF\
	       0xD7B0 0xD7FF\
	       0xD800 0xDBFF\
	       0xDC00 0xDFFF\
	       0xFE10 0xFE1F\
	       0x10140 0x102FF\
	       0x104B0 0x107FF\
	       0x10840 0x1CFFF\
	       0x1D200 0x1D2FF\
	       0x1D360 0x1D3FF\
	       0x1D800 0x1FFFF\
	       0x2A6E0 0x2F7FF\
	       0x2FAB0 0x2FFFF\
	       0xE0080 0xE00FF\
	       0xE01F0 0xEFFFF\
	       0xFFFFE 0xFFFFF];

    for {set k 0} {$k < [llength $BadRanges]} {incr k} {
	if { ($num >= [lindex $BadRanges [expr 2 * $k]]) &&\
		 ($num <= [lindex $BadRanges [expr (2 * $k) + 1]])} { 
	    set BadValueP 1;
	    break; 
	}
    }
    if {$BadValueP} {
	ShowMessage [format [_ "%s is not a valid Unicode codepoint"] $str]
	return;
    }
    if {$num >= 1114110} {
	ShowMessage [_ "Warning: codepoints above 0x10FFFD have not been assigned as of version 4.0."]
    }

    #Insert
    $::CharByCodeInsertionTarget insert insert [format "%c" $num]
}

proc IndicateIPACUp {} {
	$::CEM entryconfigure $::ToggleIPACIndex \
	    -label [_ "Remove IPA Consonant Chart"];
	set ::IPACIsDisplayedP 1;
}

proc IndicateIPACDown {} {
	$::CEM entryconfigure $::ToggleIPACIndex \
	    -label [_ "Display IPA Consonant Chart"];
	set ::IPACIsDisplayedP 0;
}

proc ToggleIPAC {} {
    global IPACIsDisplayedP;
    global m;

    if { $IPACIsDisplayedP == 0} {
	if {[winfo exists .ipaec]} {
	    wm deiconify .ipaec;
	} else { 
	    ipaentry::PopupIPAEntryC;
	}
	IndicateIPACUp;
    } else {
	wm iconify .ipaec;
	IndicateIPACDown;
    }
}

proc IndicateIPAVUp {} {
	$::CEM entryconfigure $::ToggleIPAVIndex \
	    -label [_ "Remove IPA Vowel Chart"];
	set ::IPAVIsDisplayedP 1;
}

proc IndicateIPAVDown {} {
	$::CEM entryconfigure $::ToggleIPAVIndex \
	    -label [_ "Display IPA Vowel Chart"];
	set ::IPAVIsDisplayedP 0;
}

proc ToggleIPAV {} {
    global IPAVIsDisplayedP;
    global m;

    if { $IPAVIsDisplayedP == 0} {
	if {[winfo exists .ipaev]} {
	    wm deiconify .ipaev;
	} else { 
	    ipaentry::PopupIPAEntryV;
	}
	$::CEM entryconfigure $::ToggleIPAVIndex -label [_ "Remove IPA Vowel Chart"];
	set IPAVIsDisplayedP 1;
    } else {
	wm iconify .ipaev;
	$::CEM entryconfigure $::ToggleIPAVIndex -label [_ "Display IPA Vowel Chart"];
	set IPAVIsDisplayedP 0;
    }
}

proc IndicateIPADUp {} {
	$::CEM entryconfigure $::ToggleIPADIndex \
	    -label [_ "Remove IPA Diacritic Chart"];
	set ::IPADIsDisplayedP 1;
}

proc IndicateIPADDown {} {
	$::CEM entryconfigure $::ToggleIPADIndex \
	    -label [_ "Display IPA Diacritic Chart"];
	set ::IPADIsDisplayedP 0;
}

proc ToggleIPAD {} {
    global IPADIsDisplayedP;
    global m;

    if { $IPADIsDisplayedP == 0} {
	if {[winfo exists .ipaed]} {
	    wm deiconify .ipaed;
	} else { 
	    ipaentry::PopupIPAEntryD;
	}
	IndicateIPADUp
    } else {
	wm iconify .ipaed;
	IndicateIPADDown;
    }
}

proc IndicateIPAAUp {} {
	$::CEM entryconfigure $::ToggleIPAAIndex \
	    -label [_ "Remove Accented Letter Chart"];
	set ::IPAAIsDisplayedP 1;
}

proc IndicateIPAADown {} {
	$::CEM entryconfigure $::ToggleIPAAIndex \
	    -label [_ "Display Accented Letter Chart"];
	set ::IPAAIsDisplayedP 0;
}

proc ToggleIPAA {} {
    global m;

    if { $::IPAAIsDisplayedP == 0} {
	if {[winfo exists .ipaea]} {
	    wm deiconify .ipaea;
	} else { 
	    ipaentry::PopupIPAEntryA;
	}
	IndicateIPAAUp;
    } else {
	wm iconify .ipaea;
	IndicateIPAADown;
    }
}


proc IndicateCharEntryByCodeUp {} {
	$::CEM entryconfigure $::ToggleCharEntryByCodeIndex \
	    -label [_ "Remove Widget for Entering Characters by Unicode Code"];
	set ::CharEntryByCodeIsDisplayedP 1;
}

proc IndicateCharEntryByCodeDown {} {
	$::CEM entryconfigure $::ToggleCharEntryByCodeIndex \
	    -label [_ "Display Widget for Entering Characters by Unicode Code"];
	set ::CharEntryByCodeIsDisplayedP 0;
}

proc ToggleCharEntryByCode {} {
    global CharEntryByCodeIsDisplayedP;
    global m;

    if { $CharEntryByCodeIsDisplayedP == 0} {
	if {[winfo exists .charent]} {
	    wm deiconify .charent;
	    raise .charent;
	} else { 
	    PopupCharEntryByCode;
	}
	IndicateCharEntryByCodeUp;
    } else {
	wm iconify .charent;
	IndicateCharEntryByCodeDown;
    }
}

proc SetUseCheckbuttonP {b} {
    set ::UseCheckbuttonP [Boolean $b];
}
proc SetExclusionsDisplayedP {b} {
    set ::ExclusionsDisplayedP [Boolean $b]
}
proc SetKeyFieldIdentificationDisplayedP {b} {
    set ::KeyFieldIdentificationDisplayedP [Boolean $b]
}
proc SetMiscellaneousOptionsDisplayedP {b} {
    set ::MiscellaneousOptionsDisplayedP [Boolean $b]
}
proc SetSortOrderDisplayedP {b} {
    set ::SortOrderDisplayedP [Boolean $b]
} 
proc SetSortTypeDisplayedP {b} {
    set ::SortTypeDisplayedP [Boolean $b]
}
proc SetSubstitutionsDisplayedP {b} {
    set ::SubstitutionsDisplayedP [Boolean $b]
}

proc SetRecordParseDisplayedP {b} {
    set ::RecordParseDisplayedP [Boolean $b];
}

proc SetMiscOptionsDisplayedP {b} {
    set ::MiscOptionsDisplayedP [Boolean $b];
}

proc SetUnicodeDisplayedP {b} {
    set ::UnicodeDisplayedP [Boolean $b];
}

proc SetOutputFileDisplayedP {b} {
    set ::OutputFileDisplayedP [Boolean $b];
}

proc SetAlgorithmDisplayedP {b} {
    set ::AlgorithmDisplayedP [Boolean $b];
}

proc SetInputFileDisplayedP {b} {
    set ::InputFileDisplayedP [Boolean $b];
}
proc SetSortAlgorithm {s} {
    set ::SortAlgorithm $s;
}

proc SetDisplayConsonantChartColumnLabelsP {b} {
    global DisplayConsonantChartColumnLabelsP
    set DisplayConsonantChartColumnLabelsP [Boolean $b];
}

proc SetDisplayConsonantChartRowLabelsP {b} {
    global DisplayConsonantChartRowLabelsP
    set DisplayConsonantChartRowLabelsP [Boolean $b];
}

proc SetDisplayVowelChartColumnLabelsP {b} {
    global DisplayVowelChartColumnLabelsP
    set DisplayVowelChartColumnLabelsP [Boolean $b]
}

proc SetDisplayVowelChartRowLabelsP {b} {
    global DisplayVowelChartRowLabelsP
    set DisplayVowelChartRowLabelsP [Boolean $b]
}

proc ControlDisplayConsonantChartColumnLabels {} {
    if {$::DisplayConsonantChartColumnLabelsP} {
	ipaentry::PackConsonantColumnLabels;
    } else {
	ipaentry::UnpackConsonantColumnLabels;
    }
}

proc ControlDisplayConsonantChartRowLabels {} {
    if {$::DisplayConsonantChartRowLabelsP} {
	ipaentry::PackConsonantRowLabels;
    } else {
	ipaentry::UnpackConsonantRowLabels;
    }
}

proc ControlDisplayVowelChartColumnLabels {} {
    if {$::DisplayVowelChartColumnLabelsP} {
	ipaentry::PackVowelColumnLabels;
    } else {
	ipaentry::UnpackVowelColumnLabels;
    }
}

proc ControlDisplayVowelChartRowLabels {} {
    if {$::DisplayVowelChartRowLabelsP} {
	ipaentry::PackVowelRowLabels;
    } else {
	ipaentry::UnpackVowelRowLabels;
    }
}

####################################################

#File     -> LineList	LoadCustomCharacterChart
#LineList -> CDEF	DefineCustomCharacterChart
#CDEF     -> Popup      PopupSpecialPalette

proc ReadCustomCharacterChartPopup {args} {
    if {[llength $args]} {
	set dl [LoadCustomCharacterChart [lindex $args 0]]
    } else {
	set dl [LoadCustomCharacterChart];
    }
    PopupSpecialPalette [DefineCustomCharacterChart $dl]
}

proc DefineCustomCharacterChartPopup {ll} {
    PopupSpecialPalette [DefineCustomCharacterChart $ll];
}

#Returns a linelist.
proc LoadCustomCharacterChart {args} {
    if {[llength $args]} {
	set fn [lindex $args 0]
    } else {
	set fn [tk_getOpenFile -title [_ "Load Custom Character Chart"]];
	if {[string equal $fn ""]} {
	    ShowMessage [_ "File selection cancelled."];
	    return ;
	}
    }
    if { [catch {open $fn "r"} fhd ] != 0} {
	ShowMessage [format [_ "Unable to open character chart definition file %s."] $fn];
	return ;
    }
    set LineCnt 0
    while { [gets $fhd line] > 0} {
	lappend Lines $line;
	incr LineCnt
    }
    close $fhd;
    if {$LineCnt < 1} {
	ShowMessage [_ "File %s is empty" $fn]
	return "";
    }
    ShowMessage [format [_ "Loaded custom character chart definition from %s."] $fn]
    return $Lines;
}


#This procedure takes a list of lines defining a custom
#character entry chart, which may have been read from a
#standalone file or may be an instant list in an
#init file, and generates an internal character chart
#definition, which it stores. It does not actualy
#create a display
proc DefineCustomCharacterChart {LineList {fn NONE}} {
    #The first line is special. It contains meta-information:
    #the title, the desired number of buttons per row,
    #and the proposed font family and size. Only the
    #title is obligatory.
    set line [lindex $LineList 0]
    set flds [split $line "|"]
    set FieldCnt [llength $flds];
    set Title [lindex $flds 0];
    #The remaining lines contain pairs of code sequences and glosses.
    set cd [list];
    set LineList [lrange $LineList 1 end]
    foreach line $LineList {
	set f [split $line "|"];
	set gloss [lindex $f 1];
	set c [string trim [lindex $f 0] \"];
	set cf [split $c];
	set str "\{\"";
	foreach n $cf {
	    append str [format "%s" $n]
	}
	append str "\""
	append str [format " \"%s\"\}" $gloss]
	lappend cd $str;
    }
    set info [list [join $cd]]
    if {$FieldCnt > 1} {
	lappend info [lindex $flds 1];	# Columns
    }
    if {$FieldCnt > 2} {
	lappend info [lindex $flds 2];	# Font family
    }
    if {$FieldCnt > 3} {
	lappend info [lindex $flds 3];	# Font size
    }
    if {![string equal $fn NONE]} {	# File name
	lappend info $fn;
    }
    set ::SpecialCharacterPalette($Title) $info;
    $::CEM add command -label $Title -command "PopupSpecialPalette $Title";
    incr ::CustomCEMEntries;
    return $Title;
}

#Creates a popup chart from a stored definition.
set SccCnt 0;
proc PopupSpecialPalette {Title} {
    set name [format ".scc%d" $::SccCnt] 
    if {[info exists ::SpecialCharacterPalette($Title,WidgetName)]} {
	set w $::SpecialCharacterPalette($Title,WidgetName);
	if {[winfo exists $w]} {
	    wm deiconify $w;
	    raise $w;
	    return ;
	}
    }
    set cdefs [lindex $::SpecialCharacterPalette($Title) 0];
    set Items [llength $::SpecialCharacterPalette($Title)];
    if {$Items > 1} {
	set PerRow [lindex $::SpecialCharacterPalette($Title) 1];
    } else {
	set PerRow 5;
    }
    if {$Items > 2} {
	set FontFamily [lindex $::SpecialCharacterPalette($Title) 2];
    } else {
	set FontFamily $::FontInfo(CharacterEntryFont,family)
    }
    if {$Items > 3} {
	set FontSize [lindex $::SpecialCharacterPalette($Title) 3];
    } else {
	set FontSize $::FontInfo(CharacterEntryFont,size);
    }
    incr ::SccCnt;
    set xp 3;
    set yp 3;
    toplevel $name -borderwidth 4
    set ::SpecialCharacterPalette($Title,WidgetName) $name; 
    wm title $name $Title
    iwidgets::scrolledframe $name.sf -vscrollmode dynamic -hscrollmode dynamic \
	-height 180 -width 250
    set tf [$name.sf childsite]
    pack $name.sf -expand 1 -fill both
    BindKeys $name
    bind [$name.sf component vertsb] <<B3>> \
	"ScrollbarMoveBigIncrement [$name.sf component vertsb] 0.20 %x %y"
    set CdefCnt [llength $cdefs];
    set Rows [expr int(ceil(double($CdefCnt)/double($PerRow)))]
    set Total [expr $PerRow * $Rows]
    if {$CdefCnt == $Total} {
	incr Rows;
	set Total [expr $PerRow * $Rows]
    }
    set fontname [string trimleft $name .]Font
    font create $fontname -family $FontFamily -size $FontSize
    for {set k 0} {$k < $Total} {incr k} {
	set row [expr $k/$PerRow]
	set col [expr $k%$PerRow]
	if {$k < $CdefCnt} {
	    set csalt "";
	    set entry [lindex $cdefs $k];
	    set chstr [lindex $entry 0];
	    set gloss [lindex $entry 1];
	    set chstrParts [split $chstr ":"];
	    if {[llength $chstrParts] > 1} {
		set cs    [lindex $chstrParts 0];
		set csalt [lindex $chstrParts 1];
	    } else {
		set cs [lindex $chstr 0];
	    }
	    button $tf.r${row}c${col} -text $cs\
		-padx $xp -pady $yp -font $fontname\
		-command "\$::InsertionTarget insert insert $cs"
	    if {![string equal $csalt ""]} {
		bind $tf.r${row}c${col} <<B3>> "\$::InsertionTarget insert insert $csalt"
	    }
	    balloonhelpd_for $tf.r${row}c${col} $gloss;
	} else {
	    set ln $tf.r${row}c${col};
	    #Generate buttons with blank labels for empty padding slots
	    label $ln  -text "";
	    balloonhelp_for $ln [_ "This space for rent."]
	}
    }
    set LastRow $row;
    set LastCol $col;
    set LastCell $tf.r${row}c${col}
    destroy $LastCell
    button $LastCell -image $::LeftArrowImage -command {BackDelete $::InsertionTarget}
    balloonhelpd_for $LastCell  [_ "Delete, so that you can correct mistakes while using\nthe mouse, without having to go back to the keyboard."]
    
    #Lay the buttons and labels out in a grid.
    for {set row 0} {$row <= $LastRow} {incr row} {
	set line [list];
	for {set col 0} {$col <= $LastCol} {incr col} {
	    set cell [format "%s.r%dc%d" $tf $row $col]
	    lappend line $cell
	}
	eval grid $line -sticky news;
    }
    return $name;
}

proc ExplainStringLength {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(StringLength)]} {
	if {[winfo exists $PopupList(StringLength)]} {
	    destroy $PopupList(StringLength);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "String Length Comparison"] $HPWidth 12]
    BindKeys $po;
    set PopupList(StringLength) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, records are compared on this key on the basis of the length of the key string in characters. Length is measured after preprocessing, including multigraph compression and deletion of excluded characters. Keys compared for size are treated exactly like lexicographic keys except that when comparisons are finally done on them, the comparison is a numeric comparison of the length of the key rather than a lexicographic comparison of the key as a string."];
    AppendToTextDisplay $po [_ "\n\n"]
    AppendToTextDisplay $po [_ "If you use string length as the primary key with a data set of any size, you will probably find that the sort takes a long time. This is because the number of different word-lengths in most data sets is very small in comparison to the number of distinct words. This means that on the primary key most words are ranked the same as many other words. The result is that the sort routine does many comparisons in a largely futile effort to break ties."]
    AppendToTextDisplay $po [_ "\n\n"]
    AppendToTextDisplay $po [_ "In this situation, you can speed up the sort considerably by adding a subsidiary key that can be used to break ties. For example, a straight lexicographic sort of a list of 254,817 English words required 6,852,774 comparisons and took 10 seconds on my desktop machine.Sorting the same list purely on string length required 3,173,191,623 comparisons, more than 12,000 times as many, and took almost 25 minutes, 1500 times as long. This is because this file contains only 25 different word lengths, of which nine comprise over 92% of the list. Sorting the list with string length as the primary key and a lexicographic secondary key required 6,671,259 comparisons, slightly fewer than the pure lexicographic sort, and took about 10 seconds. Much of the time you probably want to do this anyway as you won't want the results to be ordered randomly so long as they are of the same length, but even if you really don't care about anything other than string length, it will probably be to your advantage to add a tie-breaking key."]
}
 
proc ExplainNumericComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Numeric)]} {
	if {[winfo exists $PopupList(Numeric)]} {
	    destroy $PopupList(Numeric);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Numeric Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Numeric) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, records are compared on this key on the basis of the numerical value of the keys. Each string is converted to a floating point number.
Consider, for example, the strings \`9\' and \`10\' . If these are compared lexicographically, \`10\' will be ordered before \`9\' because the character \`1\' precedes \`9\'. If they are compared numerically, \`9\' is ordered before \`10\' because 9 is less than 10."];}

proc ExplainTimeComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Time)]} {
	if {[winfo exists $PopupList(Time)]} {
	    destroy $PopupList(Time);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Time Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Time) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, records are compared on this key on the basis of the key interpreted as a time. Each string is parsed as a time and converted to a purely numerical form. The strings are then compared numerically."];
    AppendToTextDisplay $po [_ "A time is expected to be of the form \`HH:MM(.SS)\'or \'HH:MM:SS(.sss)\'. In the first case,  an integer specifying the hour followed by a colon followed by an integer specifying the minute is expected, with an integer specifying the second, separated from the minute by a period, optional. In the second case, an integer specifying the hour followed by a colon specifying the mintute followed by an integer specifying the second is expected, with an integer specying fractional seconds separated from the second by a period optional."];
    AppendToTextDisplay $po [_ "Msort detects and reports inappropriate values such as minutes greater than 59."];
    AppendToTextDisplay $po [_ "Examples of well-formed times are: 23:11:08, 23:11:08.5, 11:10.55, and 14:11"];
}

proc ExplainAngleComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Angle)]} {
	if {[winfo exists $PopupList(Angle)]} {
	    destroy $PopupList(Angle);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Angle Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Angle) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, the key is treated as an angle in sexagesimal degree format. The angle is expected to be in the format DD:MM:SS(.sss), that is to say, degrees, minutes, and seconds separated by colons, with an optional fractional part for seconds. The key is converted to a value with dimensions of seconds and compared numerically. Msort detects and reports invalid values such as seconds greater than or equal to 60."];}

proc ExplainDateComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Date)]} {
	if {[winfo exists $PopupList(Date)]} {
	    destroy $PopupList(Date);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Date Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Date) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, records are compared on this key on the basis of the key interpreted as a date. Each string is parsed as a date and converted to a purely numerical form. The strings are then compared numerically."];
    AppendToTextDisplay $po [_ "For example, a date in the"];
    AppendLinkToTextDisplay $po [_ "International Date Format"] {ShowWebPage http://www.saqqara.demon.co.uk/datefmt.htm};
AppendToTextDisplay $po [_ "consists of the year followed by the month followed by the day separated by dashes, e.g. 2005-04-24 for April 4, 2005. The format for such a date is: y-m-d."];
    AppendToTextDisplay $po [_ "The same date in the format usual in Europe has the format d/m/y, i.e. 24/04/2005."];
    AppendToTextDisplay $po [_ "The same date in the format usual in the United States has the format m/d/y, i.e. 4/24/2005."];
    AppendToTextDisplay $po [_ "The punctuation used to separate the components of dates also varies. In many countries periods are used instead of slashes, e.g. 24.04.2005, for which the format specification is: d.m.y"];
}

proc ExplainISO8601Comparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(ISO8601)]} {
	if {[winfo exists $PopupList(ISO8601)]} {
	    destroy $PopupList(ISO8601);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "ISO8601 Date/Time Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(ISO8601) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, records are compared on this key on the basis of the assumption that the key consists of a combined date and time in the format standardized by International Standards Organization standard 8601. Such combined dates and times consist of a date in ISO8601 format, the upper case letter <T>, and a time in international format. An ISO8601 date consists of a year number followed by month number followed by day number, separated by hyphens. A time in international format is a 24-hour time with hours preceding minutes separated by a colon. Seconds optionally follow, separated from minutes by a period."];
    AppendToTextDisplay $po [_ "For example, 2005-04-25T14:11.33 represents 33 seconds past 2:11 p.m. on April 25th, 2005."];
}

proc ExplainLexicographicComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Lexicographic)]} {
	if {[winfo exists $PopupList(Lexicographic)]} {
	    destroy $PopupList(Lexicographic);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Lexicographic Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Lexicographic) $po;
    AppendToTextDisplay $po [_ "If this option is chosen, records are compared on this key by \'dictionary ordering\'. If the first letters of the two words have different sort ranks, that determines the ordering of the two words. If not, we consider the next letter of each word. Again, if they have different sort ranks, we are done, but if they do not, we move on to the next letter. If one string comes to an end before the other, the shorter string ranks first."];
    AppendToTextDisplay $po "\n\n"
    AppendToTextDisplay $po [_ "It is important to be careful about the use of terms like \"lexicographic\" and \"dictonary\" order because they are used inconsistently. Here I am using the term \"lexicographic\" only to mean that the ordering is based on the ranking of the individual letters and not on interpretation of the string in some fashion, as it is when we sort numerically. At least three other meanings of this term exist. First, some people consider \"dictionary\" order to be case-insensitive. With Msort, if you want case-insensitive behavior you can get it, but it is controlled by a separate option. Secondly, some people use this term for sorting in which sequences of digits are treated as integers and compared numerically. If you want this behavior in Msort, you want \"hybrid\" sorting. Third, the term may imply that non-alphanumeric characters are ignored. Again, this behavior is available in Msort if you want it, but it is not implicit in either \"lexicographic\" or \"hybrid\" comparison. If you want to ignore certain characters, use the exclusion facilities."]
   AppendToTextDisplay $po [_ "As examples of the other usages, the --dictionary option in GNU sort has the effect of making the comparison case-insensitive and of making it ignore non-alphanumeric characters. The -dict option to Tcl's lsort command has the effect of treating sequences of digits as integers and comparing them numerically."]
}

proc ExplainHybridComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Hybrid)]} {
	if {[winfo exists $PopupList(Hybrid)]} {
	    destroy $PopupList(Hybrid);
	    return;
	}
    }

    set po [CreateTextDisplay [_ "Hybrid Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Hybrid) $po;
    AppendToTextDisplay $po [_ "This type of comparison is the same as lexicographic comparison except for the treatment of numerals."];
    AppendToTextDisplay $po [_ "In lexicographic comparison, numerals are treated like any other characters."]
    AppendToTextDisplay $po [_ "For example, A12 sorts before A3 because 1 precedes 3."]
    AppendToTextDisplay $po [_ "In hybrid comparison, each uninterrupted string of numerals is converted to the number that it represents. "];
    AppendToTextDisplay $po [_ "These numbers are then compared numerically."]
    AppendToTextDisplay $po [_ "In a hybrid sort, A3 sorts before A12 because the number 3 is less than the number 12."];
}

proc ExplainRandomComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Random)]} {
	if {[winfo exists $PopupList(Random)]} {
	    destroy $PopupList(Random);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Random Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Random) $po;
    AppendToTextDisplay $po [_ "This option is used for randomizing files, not actually for sorting them. Its effect is that, each time two records are compared on the specified key, the result of the comparison is determined by a random number generator rather than by the contents of the key fields. The records will be reported as comparing less-than, equal, or greater-than with equal probability."];
}

proc ExplainPositionID {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(PositionID)]} {
	if {[winfo exists $PopupList(PositionID)]} {
	    destroy $PopupList(PositionID);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Identifying Keys by Position"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(PositionID) $po;
    AppendToTextDisplay $po [_ "A key is identified by position by specifying a contiguous range of positions, where a position consists of a field number and a character offset into the field. Fields may be counted from the end of the record by using negative numbers. The first field is numbered 1, the last field -1. Character offsets must be positive and begin at 1. The beginning and end of a range may be the same. For example, the range 1.2,1.2 specifies the key as consisting of the second character of the first field. The range 1.1,-1.2 specifies the key as the first field through the last field, with only the first character of the last field in the key."];
}

proc ExplainTagID {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(TagID)]} {
	if {[winfo exists $PopupList(TagID)]} {
	    destroy $PopupList(TagID);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Identifying Key Fields By Tag"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(TagID) $po;
    AppendToTextDisplay $po [_ "When key fields are identified by tags, each field must begin with a string that indicates what field it is. For example, in an address list one tag might be \`Postal Code\', which would indicate that the remainder of the field contains a postal code. Tags do not have to be fixed strings; Msort allows tags to be specified by regular expressions. The regular expression notation used is that of the "];
    AppendLinkToTextDisplay $po [_ "TRE library."] {ShowWebPage http://laurikari.net/tre/};
    AppendToTextDisplay $po [_ " Details may be found in the "];
    AppendLinkToTextDisplay $po [_ "TRE Syntax Manual."] {ShowWebPage http://laurikari.net/tre/syntax.html};
}

proc ExplainRangeID {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(RangeID)]} {
	if {[winfo exists $PopupList(RangeID)]} {
	    destroy $PopupList(RangeID);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Identifying Key Fields By Character Range"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(RangeID) $po;
    AppendToTextDisplay $po [_ "A range of characters may be specified as a key. A range\
consists of the characters M through N. The first character in the record has index 1.\
Negative indices indicate position with respect to the end of the record.  For\
example, the range 1 through 3 contains the first three characters of the record.
The range 3 through -2 contains the third character through the next-to-last character."];
}

proc ExplainReverseKey {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(ReverseKey)]} {
	if {[winfo exists $PopupList(ReverseKey)]} {
	    destroy $PopupList(ReverseKey);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Key Reversal"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(ReverseKey) $po;
    AppendToTextDisplay $po [_ "Key reversal means reversing the order of characters in the key. For example, reversing \'spelling\' yields \'gnilleps\'. Reversing the order of characters in a key gives priority to the end of a string rather than to its beginning. Among other things, this option allows the generation of reverse dictionaries. Note that reversal is done following multigraph compression. For example, if \`lh\' is declared in the sort order specification as a multigraph, \`bilh\' will sort as if it were \`lhib\', not as if it were \`hlib\'. Key reversal is only applicable to lexicographic keys."]
}

proc ExplainInvertComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(InvertComparison)]} {
	if {[winfo exists $PopupList(InvertComparison)]} {
	    destroy $PopupList(InvertComparison);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Inversion of Comparison"] $HPWidth 20]
    BindKeys $po;
    set PopupList(InvertComparison) $po;
    AppendToTextDisplay $po [_ "Sorting ultimately comes down to pairwise comparisons of records, where it is decided whether one record is greater than the other, less than it, or equal to it. Inverting comparison means making the opposite decision for comparisons on this key. For instance, if we are comparing numerical values and record A's key is 100 and record B's key is 200, the usual comparison will order A before B. If comparisons on this key are inverted, A will be ordered after B."]
    AppendToTextDisplay $po [_ "If there is only a single key, inverting comparisons has the same effect as a global inversion of the sort order. If there is more than one key, however, the effect of inverting comparisons on invidual keys is not the same as that of a global inversion of the sort order. Suppose, for example, that you have a list of names and dates, such as this:\n\n"];
    AppendToTextDisplay $po [_ "    2004/01/17 Smith\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Khaled\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Gomez\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Yamamoto\n"]
    AppendToTextDisplay $po [_ "    2003/02/03 Maurois\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Burger\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "Here is the result of sorting first by date, then by name, in both cases in the usual ascending order:\n\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Burger\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Yamamoto\n"]
    AppendToTextDisplay $po [_ "    2003/02/03 Maurois\n"]
    AppendToTextDisplay $po [_ "    2004/01/17 Smith\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Gomez\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Khaled\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "Now suppose that you decide that you would like the most recent dates first. If you invert the order globally, the result is this:\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Khaled\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Gomez\n"]
    AppendToTextDisplay $po [_ "    2004/01/17 Smith\n"]
    AppendToTextDisplay $po [_ "    2003/02/03 Maurois\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Yamamoto\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Burger\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "This is probably not what you want. The dates are now descending, but the names are in reverse alphabetical order. What you want is more likely this:\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Gomez\n"]
    AppendToTextDisplay $po [_ "    2004/03/15 Khaled\n"]
    AppendToTextDisplay $po [_ "    2004/01/17 Smith\n"]
    AppendToTextDisplay $po [_ "    2003/02/03 Maurois\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Burger\n"]
    AppendToTextDisplay $po [_ "    1999/01/28 Yamamoto\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "This is the result of inverting comparisons on dates but not on names.\n"]
}

proc ExplainFoldCase {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(FoldCase)]} {
	if {[winfo exists $PopupList(FoldCase)]} {
	    destroy $PopupList(FoldCase);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Case Folding"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(FoldCase) $po;
    AppendToTextDisplay $po [_ "It is often the case that upper- and lower-case letters should be treated the same for sorting purposes. Setting case folding on makes comparisons insensitive to case.This can be done by the use of a suitable sort order definition, but it is common enough that msort, like other sorting programs, provides a separate option for it."]
}

proc ExplainConvertStylistic {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown ConvertStylistic] ==1} {return}
    set po [CreateTextDisplay [_ "Convert Stylistic Variants"] $HPWidth $HPLines]
    set PopupList(ConvertStylistic) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "This option specifies that characters in special styles are to be replaced with the corresponding plain character. The special styles include: small capitals (e.g. \u1D04 U+1D04), script forms (e.g. \u2131 U+2131), black letter forms (e.g. \u2128 U+2128), Arabic presentation forms (e.g. \uFE81 U+FE81), Hebrew presentation forms (e.g. \uFB1D U+FB1D), fullwidth forms (e.g. \uFF21 U+FF21), halfwidth forms (e.g. \uFF7B U+FF7B), and the mathematical alphanumeric symbols (e.g. U+1D400)."]
}

proc ExplainConvertEnclosures {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown ConvertEnclosures] ==1} {return}
    set po [CreateTextDisplay [_ "Convert Enclosures"] $HPWidth $HPLines]
    set PopupList(ConvertEnclosures) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "This option causes characters enclosed in circles or parentheses, e.g. \u249C U+249C and \u24B6 U+24B6, to be converted to their plain equivalents.\n"];
}

proc ExplainStripDiacritics {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown StripDiacritics] ==1} {return}
    set po [CreateTextDisplay [_ "Strip Diacritics"] $HPWidth $HPLines]
    set PopupList(StripDiacritics) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "If this option is chosen, combining diacritics such as the acute accent U+0301 are deleted and characters with intrinsic diacritics such as \u00E9 U+00E9 are replaced with their plain equivalents."];
}

proc ExplainSortOrder {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(SortOrder)]} {
	if {[winfo exists $PopupList(SortOrder)]} {
	    destroy $PopupList(SortOrder);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Sort Order"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(SortOrder) $po;
    AppendToTextDisplay $po [_ "Msort allows arbitrary sort orders to be defined for each key. By default Unicode order is used. That is, there are no collating sequences and the rank of a character is determined by its Unicode codepoint. For the ASCII characters, Unicode order is the same as ASCII order. You may specify other orders in two ways: by explicitly describing the order or by indicating that the collation rules of a specified locale should be used.\n\n"];
    AppendToTextDisplay $po [_ "A sort order definition consists of a file in which each line represents a sort rank. The first line of the file has the highest rank, the last line the lowest rank. Items on the same line are assigned the same rank. "]
    AppendToTextDisplay $po [_ "Not infrequently sequences of characters must be treated as if they were single characters and given their own place in the sort order. For example, in Spanish a sequence of two <l>s represents a single sound, a palatal lateral, and is conventionally ordered after plain <l>. Thus, <llama> follows words such <lobo>; if <ll> were not given special treatment
<llama> would precede <lobo> since <l> precedes <o>. Such sequences of characters that must be treated as if they were single characters are known as multigraphs.\n"]
    AppendToTextDisplay $po [_ "Msort provides for multigraphs by allowing sequences of characters to be listed in the sort order definition along with single characters. For example, to obtain the conventional Spanish alphabetical order, the relevant portion of the sort order file would look like this:\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "l\n"]
    AppendToTextDisplay $po [_ "ll\n"]
    AppendToTextDisplay $po [_ "m\n"]
    AppendToTextDisplay $po [_ "\n"]
    AppendToTextDisplay $po [_ "Msort permits virtually unlimited numbers of multigraphs of virtually unlimited length.\n\n"]
    AppendToTextDisplay $po [_ "A locale is a specification of various pieces of information that vary with language and culture, including language, character set, character encoding, number format, and sort order. On systems conforming to the POSIX standard, the locale known as \"C\" or \"POSIX\" is always available. In this default locale, everything is as it was on traditional American English Unix systems. The character set and encoding is ASCII, the sort order is the order of the ASCII characters, and so forth. Most other locale names consist of a language abbreviation followed by a country abbreviation, connected by an underscore.\nExamples are:\n"]
    AppendToTextDisplay $po [_ "   en_US   American English\n"]
    AppendToTextDisplay $po [_ "   en_CA   Canadian English\n"]
    AppendToTextDisplay $po [_ "   fr_CA   Canadian French\n"]
    AppendToTextDisplay $po [_ "   ca_ES   Catalan as used in Spain\n"]
    AppendToTextDisplay $po [_ "   de_DE   German as used in Germany\n"]
    AppendToTextDisplay $po [_ "   ja_JP   Japanese as used in Japan\n"]
    AppendToTextDisplay $po [_ "   pt_BR   Brazilian Portuguese\n"]
    AppendToTextDisplay $po [_ "   zh_CN   Chinese as used in China\n"]
    AppendToTextDisplay $po [_ "   zh_HK   Chinese as used in Hong Kong\n"]
    AppendToTextDisplay $po [_ "\nIf your system uses glibc or an equivalent localization system, msort can be told to use the collating rules for a specified locale. This saves you the work of creating a sort order definition and defining any necessary exclusions and substitutions. The locale whose collating rules you wish to use must be installed on your system. On GNU systems, the command \"locale -a\" will give you a list of the available locales. You may also enter \"current\" or \"locale\", in which case the collating rules of the current locale will be used."]

}

proc ExplainSortOrderSeparators {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(SortSeparators)]} {
	if {[winfo exists $PopupList(SortSeparators)]} {
	    destroy $PopupList(SortSeparators);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Sort Order File Separators"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(SortSeparators) $po;
    AppendToTextDisplay $po [_ "In a sort order definition file entries on the same line are assigned the same sort rank."]
    AppendToTextDisplay $po [_ "A sequence of two or more characters not containing any separator characters is treated as a multigraph."]
    AppendToTextDisplay $po [_ "Sequences of characters separated by separators are treated as distinct characters or multigraphs assigned the same sort rank."]
    AppendToTextDisplay $po "\n\n"
    AppendToTextDisplay $po [_ "By default, all Unicode characters that fall into the broad class of \'whitespace\', including tab (0x0009), space (0x0020), and ideographic space (0x3000), are treated as separators."]
    AppendToTextDisplay $po [_ "In some circumstances, however, it is desirable to include whitespace characters in multigraphs."];
    AppendToTextDisplay $po [_ "For example, if a unique ordering consisting of names is defined, it may be desirable to treat names with components separated by spaces as multigraphs."];
    AppendToTextDisplay $po [_ "In order to allow for this, you may change the definition of separator character used in parsing sort order definitions."];
    AppendToTextDisplay $po [_ "For instance, if your list of names looks like this:"];
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po "Smith, Mary\n";
    AppendToTextDisplay $po "Yamamoto, Sachiko\n";
    AppendToTextDisplay $po "Klein, Joel\n";
    AppendToTextDisplay $po "Sanchez, Marta\n";
    AppendToTextDisplay $po "\n";
    AppendToTextDisplay $po [_ "you would want to omit space from the list of separators."];
    AppendToTextDisplay $po [_ "The characters that you enter here will replace the default list of separator characters for this key."];
}

proc ExplainSortType {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(SortType)]} {
	if {[winfo exists $PopupList(SortType)]} {
	    destroy $PopupList(SortType);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Sort Type"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(SortType) $po;
    AppendToTextDisplay $po [_ "Text is usually sorted lexicographically, but some kinds of text require special treatment. In most cases numbers should be sorted by numerical value rather than lexicographically. Msort provides several specialized types of comparison in addition to the two conventional ones. Msort understands dates and times in certain formats. Msort also allows the length of the character string to be used as the basis for sorting.\n"];}

proc ExplainHowToIdentifyKey {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(HowToIdentifyKey)]} {
	if {[winfo exists $PopupList(HowToIdentifyKey)]} {
	    destroy $PopupList(HowToIdentifyKey);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "How To Identify the Key"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(HowToIdentifyKey) $po;
    AppendToTextDisplay $po [_ "Msort provides three ways to identify the portion of the record to be used as a key. The conventional approach is based on the order of fields within the record. This is sufficient for simple databases, but has the disadvantage that every record must contain all fields always in the same order. More sophisticated databases therefore often make use of tags, strings of characters that identify the remainder of the field. Msort allows key fields to be identified by matching regular expressions to tags. The third way of identifying the key is by specifying a certain range of characters, for example, the fourth through eighth characters in the record. This method is useful in dealing with packed binary data and old-fashioned databases using records with fixed-length fields."]}

proc ExplainExclusions {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Exclusions)]} {
	if {[winfo exists $PopupList(Exclusions)]} {
	    destroy $PopupList(Exclusions);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Exclusions"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Exclusions) $po;
    AppendToTextDisplay $po [_ "It is sometimes desirable to ignore certain characters in certain positions when sorting. For example, when sorting credit card numbers it may be desirable to ignore spaces or hyphens separating groups of numbers. To take another example, some dictionaries list bound forms (words that require a prefix or suffix) with leading or trailing hyphens to show where something must attach. In alphabetizing such a dictionary, these hyphens should be ignored except when necessary to break a tie. This is accomplished by using the same key twice, the first time ignoring the hyphens, the second time taking them into account. Msort allows specified characters to be ignored. Since in some cases a character should only be ignored in certain positions, msort allows the user to specify whether a character should ignored in string-initial, -medial, or -final position, or any combination thereof."]}

proc ExplainSubstitutions {} {
    global HPWidth;
    global PopupList;
    if {[info exists PopupList(Substitutions)]} {
	if {[winfo exists $PopupList(Substitutions)]} {
	    destroy $PopupList(Substitutions);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Substitutions"] $HPWidth 10]
    BindKeys $po;
    set PopupList(Substitutions) $po;
    AppendToTextDisplay $po [_ "The substitution mechanism allows you to define more-or-less arbitrary transformations of keys. Each set of substitutions is specific to a particular key. Each individual substitution consists of two parts: a regular expression and a fixed string. Each portion of the original key that matches the regular expression is replaced by the fixed string. In the simplest case a regular expression is itself a fixed string, so if you are not familiar with regular expressions you can still use this mechanism to replace one fixed string with another. The regular expression notation is the same as that used for matching tags. A link to a description of the notation will be found on the Help menu."];
    AppendToTextDisplay $po "\n\n";
    AppendToTextDisplay $po [_ "This device has several uses. It provides a way to handle cases in which certain components of names are alphabetized as if they were written differently. For example, in English the prefix Mc is supposed to sort as if it were spelled Mac (as it sometimes is). \"McArthur\" should precede \"MacCawley\", as if it were spelled \"MacArthur\". Similarly, it can be used to disregard components of names that are supposed to be ignored in alphabetization, such as \"de\" and \"van\"."];
    AppendToTextDisplay $po "\n\n";
    AppendToTextDisplay $po [_ "This device can also be used to handle the rare cases in which a character is supposed to sort as if it were a sequence of characters. An example is the German letter \u00DF, (0x00DF)\"eszet \", which is treated as if it were \"ss\" for purposes of sorting."];
    AppendToTextDisplay $po "\n";
}

proc ExplainOptional {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Optional)]} {
	if {[winfo exists $PopupList(Optional)]} {
	    destroy $PopupList(Optional);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Optional Keys"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Optional) $po;
    AppendToTextDisplay $po [_ "It is sometimes desirable to sort on the basis of a field that is not always present. By default all keys are obligatory, but msort allows keys to be declared optional. If a field is declared optional, it is not an error for it to be absent from a record."]
}

proc ExplainComparison {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Comparison)]} {
	if {[winfo exists $PopupList(Comparison)]} {
	    destroy $PopupList(Comparison);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Comparison"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Comparison) $po;
    AppendToTextDisplay $po [_ "If a key field is missing from a record, the question arises as to how it should compare with records in which it is present. Msort requires this to be specified when a key is declared optional. Absent keys may be ranked before, after, or equal to present keys."]
}

proc ExplainHowToParseIntoRecords {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(HowToParseIntoRecords)]} {
	if {[winfo exists $PopupList(HowToParseIntoRecords)]} {
	    destroy $PopupList(HowToParseIntoRecords);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "How To Parse Into Records"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(HowToParseIntoRecords) $po;
    AppendToTextDisplay $po [_ "Material to be sorted must be divided into records since these are the units that are ordered by sorting. In most cases, an initial parse into records is also necessary so as to locate key fields. Msort provides several ways of delimiting records."]
}

proc ExplainBlock {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Block)]} {
	if {[winfo exists $PopupList(Block)]} {
	    destroy $PopupList(Block);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Blocks"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Block) $po;
    AppendToTextDisplay $po [_ "Many flat-file databases keep records separate by one or more blank lines. Although very common and important, this format cannot be handled by parsers that look for a single delimiter character, so msort makes a separate provision for it."]
}

proc ExplainLine {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(Line)]} {
	if {[winfo exists $PopupList(Line)]} {
	    destroy $PopupList(Line);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Line"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(Line) $po;
    AppendToTextDisplay $po [_ "In this case each line constitutes a record."]
}

proc ExplainFieldSeparatorCharacter {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(FieldSeparatorCharacter)]} {
	if {[winfo exists $PopupList(FieldSeparatorCharacter)]} {
	    destroy $PopupList(FieldSeparatorCharacter);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Field Separator Characters"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(FieldSeparatorCharacter) $po;
    AppendToTextDisplay $po [_ "In many cases different parts of a record play different roles in sorting. Some must be ignored, some must given priority over others, and some must be ranked according to different criteria. The parts into which a record is divided are called fields. Msort divides records into fields on the basis of seperator characters. In the case of text, whitespace often serves as the field delimiter, but a variety of other field delimiters are found. Examples include the the Unix password file, which uses a colon, the common spreadsheet export format, which uses a comma, and formats for telephone numbers, which in the United States usually use a hyphen to separate components. You may specify the field delimiters by entering them in the entry box provided."]}

proc ExplainRecordSeparatorCharacter {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(RecordSeparatorCharacter)]} {
	if {[winfo exists $PopupList(RecordSeparatorCharacter)]} {
	    destroy $PopupList(RecordSeparatorCharacter);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Record Separator Character"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(RecordSeparatorCharacter) $po;
    AppendToTextDisplay $po [_ "One way in which records may be delimited is by a separator character, that is, a character that appears between records. If this option is selected, you must enter the separator character in the entry box provided."]
}

proc ExplainEntireRecordAsKey {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(EntireRecordAsKey)]} {
	if {[winfo exists $PopupList(EntireRecordAsKey)]} {
	    destroy $PopupList(EntireRecordAsKey);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Entire Record As Key"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(EntireRecordAsKey) $po;
    AppendToTextDisplay $po [_ "Much of the time sorting is based on only a subset of the information in a record, so it is necessary to parse the record into fields and extract specified keys. In some cases, however, the entire record is relevant and no parse is necessary. If selected, this option uses the entire content of each record as the sole sort key."]
}

proc ExplainBMPOnly {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(BMPOnly)]} {
	if {[winfo exists $PopupList(BMPOnly)]} {
	    destroy $PopupList(BMPOnly);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "BMP Only"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(BMPOnly) $po;
    AppendToTextDisplay $po [_ "If the input is known to be restricted to the Basic Multilingual Plane (codepoints not exceeding 0xFFFF), the program may be informed of this fact. This permits a signficant reduction in memory usage. The characters of virtually all writing systems in current use are located within the BMP. Only ancient and exotic writing systems and some special symbols require the use of characters beyond the BMP."]
}

proc ExplainReservePrivateUseAreas {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(ReservePrivateUseAreas)]} {
	if {[winfo exists $PopupList(ReservePrivateUseAreas)]} {
	    destroy $PopupList(ReservePrivateUseAreas);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Reserve Private Use Areas"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(ReservePrivateUseAreas) $po;
    AppendToTextDisplay $po [_ "Msort internally assigns multigraphs to codepoints in regions that are not normally used. By default it uses the Supplementary Private Use Areas, or, if informed that input is restricted to the Basic Multilingual Plane, the Private Use Area. If the input text contains character codes in the Private Use areas, a conflict will result. It is therefore possible to tell Msort not to use the Private Use areas. In this case, Msort will assign multigraphs to codepoints in the Low and High Surrogate ranges, which are guaranteed to be unused."]
    AppendToTextDisplay $po "\n\n";
    AppendToTextDisplay $po [_ "The disadvantage of using the Surrogate areas for multigraphs is that this limits the number of multigraphs to 2,048. This is of course more than enough for most purposes."];
    AppendToTextDisplay $po "\n\n";
    AppendToTextDisplay $po [_ "Note that Msort will not use the Private Use Area unless told that input is restricted to the BMP. It is therefore only necessary to use this option if your input makes use of codepoints in the Private Use Area and you wish to obtain the reduction in memory usage permitted by informing Msort that input is restricted to the BMP."];
}

proc ExplainInvertOrderGlobally {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[info exists PopupList(InvertOrderGlobally)]} {
	if {[winfo exists $PopupList(InvertOrderGlobally)]} {
	    destroy $PopupList(InvertOrderGlobally);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "Invert Order Globally"] $HPWidth $HPLines]
    BindKeys $po;
    set PopupList(InvertOrderGlobally) $po;
    AppendToTextDisplay $po [_ "If set, records are written out in the order opposite to what would otherwise be the case. If the basic order is alphabetical, the result will be reverse alphabetical. If the basic order is increasing numerical order, the result will be decreasing numerical order."]
    AppendToTextDisplay $po [_ "For the difference between global inversion of order and key-specific inversion of comparison, see the help text for Invert Comparison."]
}

proc ExplainEndOfLineMarker {} {
    global HPWidth;
    global PopupList;
    if {[info exists PopupList(EndOfLineMarker)]} {
	if {[winfo exists $PopupList(EndOfLineMarker)]} {
	    destroy $PopupList(EndOfLineMarker);
	    return;
	}
    }
    set po [CreateTextDisplay [_ "End-of-line Marker"] $HPWidth 10]
    BindKeys $po;
    set PopupList(EndOfLineMarker) $po;
    AppendToTextDisplay $po [_ "Once upon a time, to move to the beginning of the next \
line on teletype machines, it was necessary to send two control characters: a Carriage \
Return, to move the printer carriage back to the beginning of the line, and a Line Feed, \
to move to the next line. DOS and Microsoft Windows continue this practice of using a \
CRLF sequence to mark end-of-line.\n\n"];
    AppendToTextDisplay $po [_ "On more modern operating systems end-of-line is marked \
by a single character. On Unix systems end-of-line is normally marked by a Line Feed \
character (0x0A). On Macintoshen, end-of-line is normally marked by a Carriage Return \
character (0x0D).\n\n"];
    AppendToTextDisplay $po [_ "By default, Msort assumes that end-of-line is marked \
by a single Line Feed character. However, in order to facilitate the use of files in \
Macintosh format, it is possible to specify that end-of-line is marked by Carriage \
Return. This setting affects both the reading of records consisting of a single
line and the reading of records separated by two or more end-of-line characters.\n"];
}

set RightArrowImage [image create photo -data {
R0lGODlhGAAYAPMPAAAAAAQEBAgICAwMDCAgICMjIysrK0BAQKurq7+/v8PDw8fHx8/Pz+/v
7////wAAACH5BAEAAA8ALAAAAAAYABgAAAT+8L333nvvvefeg++999577733Hnzvveecc+69
9+B777333nvvvQefc80599577z343nvvvffec85A5Zx777333nsPvvfee885MpRz0L333nvv
vffeg+895wgYyjn3Hnzvvfecc84556AjAAzlnHvvvQefc84555xzBAAIhnLOvffec47Bc845
5wgAABgKOufee8+5BAAAEAAAAAAADOWcg+895xIAAAAAAAQAAACGcs6956BLAAAAAAAAAAAB
AGM5955zCQAIAAAAAAAAAGDA5dx7ziUAAAAAAgAAAAAM5Zx78DmXAAAAAAAAgACAoZxz7z3n
FiSEEEJIAACAoaCFc+6995xzzjkHnXMEABCUc+49+N57zjnnnHOOQAACcs699957D7733nvP
OQICctC5995777333oPvPedIQM659x5877333nvvveegI8g59957770H33vvvfeec8w56N57
77333nvvwffee8455957D7733nvvvffee/A959x777333oPvRQA7
}]


proc EnterExclusions {KeyName} {

    if {[winfo exists .xdf]} {
	raise .xdf;
	return
    }
    toplevel .xdf -borderwidth 4 -relief raised
    wm title .xdf "";
    after idle {
	update idletasks
	set xmax [winfo screenwidth .xdf]
	set ymax [winfo screenheight .xdf]
	set x0 [expr 1 * ($xmax -[winfo reqwidth .xdf])/3];
	set y0 [expr 1 * ($ymax -[winfo reqheight .xdf])/3];
	wm geometry .xdf "+$x0+$y0";
    }
    label .xdf.title -text [_ "Define Exclusions"]
    frame .xdf.boxes;   #Listbox and entry box
    frame .xdf.opbs;    #Buttons for operations
    frame .xdf.finalbs; #Buttons for finalizing - cancel and proceed

    listbox .xdf.boxes.lb -width 12 -height 6 -font UserTechnicalTextFont
    InsertStoredExclusionsIntoListbox $KeyName;

    frame .xdf.boxes.ce -relief sunken -borderwidth 3
    entry .xdf.boxes.ce.ent -width 1 -state disabled  -font UserTechnicalTextFont
    checkbutton .xdf.boxes.ce.cbi -variable ExcInitialP \
	-onvalue 1 -offvalue 0 -text [_ "Initial"] -state disabled
    checkbutton .xdf.boxes.ce.cbm -variable ExcMedialP  \
	-onvalue 1 -offvalue 0 -text [_ "Medial"]  -state disabled
    checkbutton .xdf.boxes.ce.cbf -variable ExcFinalP   \
	-onvalue 1 -offvalue 0 -text [_ "Final"]   -state disabled

    button .xdf.boxes.ins  -image $::RightArrowImage -command InsertExclusionEntry
    bind .xdf.boxes.ce.ent <FocusIn> {SetInsertionTargets .xdf.boxes.ce.ent}
    pack .xdf.boxes.ce.cbi -side top -expand 1 -anchor w
    pack .xdf.boxes.ce.cbm -side top -expand 1 -anchor w
    pack .xdf.boxes.ce.cbf -side top -expand 1 -anchor w
    pack .xdf.boxes.ce.ent -side top -expand 1 -anchor w
    pack .xdf.boxes.ce -side left  -anchor w -padx 6 -expand 1
    pack .xdf.boxes.ins -side left -padx 6 -expand 1 -fill x 
    pack .xdf.boxes.lb -side right -padx 6 -expand 1 -fill x

    frame .xdf.arbut
    button .xdf.arbut.add  -text [_ "Add"]    -command AddExclusionEntry
    button .xdf.arbut.rem  -text [_ "Remove"] -command RemoveExclusionEntry
    label .xdf.arbut.ins  -text "" -width 12;
    pack .xdf.arbut.ins -side left -expand 1 -anchor w -padx 4 -pady 4 -fill x
    pack .xdf.arbut.add -side left -expand 1 -anchor e -padx 4 -pady 4
    pack .xdf.arbut.rem -side left -expand 1 -anchor e -padx 1 -pady 4

    button .xdf.finalbs.p  -text [_ "Save"]  -command [ list SaveExclusionDefinitions $KeyName]
    button .xdf.finalbs.c  -text [_ "Cancel"] -command {destroy .xdf} -anchor w
    pack .xdf.finalbs.c -side left  -expand 1 -anchor w 
    pack .xdf.finalbs.p -side right -expand 1 -anchor e 

    #Put it all together
    pack .xdf.title   -side top -anchor w -expand 1
    pack .xdf.finalbs -side bottom -expand 1 -fill x
    frame .xdf.fbsep -height 6 -borderwidth 0 -relief flat
    pack .xdf.fbsep   -side bottom -expand 1 -fill x
    pack .xdf.boxes   -side top -expand 1
    pack .xdf.arbut    -side top -expand 1
    pack .xdf.opbs    -side top -expand 1
    bind .xdf.boxes.ce.ent <Control-v> {InsertClipboardContents %W}
    set bhmsg [_ "Change the exclusions for this key.\nTo delete an entry, elect it, then\npress Remove. To add an entry, press\nAdd, then enter the character in the entry\nbox and select the position or positions\nin which it is to be excluded."]
    balloonhelp_for .xdf $bhmsg;
    balloonhelp_for .xdf.fbsep $bhmsg;
    balloonhelp_for .xdf.arbut $bhmsg;
    balloonhelp_for .xdf.arbut.ins $bhmsg; 
    balloonhelp_for .xdf.finalbs $bhmsg;
    balloonhelp_for .xdf.opbs $bhmsg;
    balloonhelp_for .xdf.boxes [_ "Define a new exclusion by entering the character\nto be excluded and selecting the position or\npositions in which it is to be excluded."]
    balloonhelp_for .xdf.finalbs.p [_ "Store the exclusions you have defined and exit\n this dialogue. You will be asked to specify the name\nof the file into which they will be written."]
    balloonhelp_for .xdf.finalbs.c [_ "Exit this dialogue without changing anything."]
    balloonhelp_for .xdf.arbut.rem [_ "Delete the selected entry from the exclusion list."]
    balloonhelp_for .xdf.arbut.add [_ "Prepare to add a new exclusion definition to the list.\nThe character entry box and position\nselection buttons will be activated."]
    balloonhelp_for .xdf.boxes.ins [_ "Add the new definition to the exclusion list."]
    balloonhelp_for .xdf.boxes.lb [_ "This is the working list of exclusions.\nIt initially contains whatever exclusions\nwere previously defined for this key.\nYou may not edit it directly. Instead, use\nthe Remove button to delete an entry and the\nAdd button to add a new one."]
    balloonhelp_for .xdf.boxes.ce.ent [_ "Enter the character you wish to exclude."]
    balloonhelp_for .xdf.boxes.ce.cbi [_ "Press this button to exclude the character you\nhave entered in field-initial position."]
    balloonhelp_for .xdf.boxes.ce.cbm [_ "Press this button to exclude the character you\nhave entered in field-medial position."]
    balloonhelp_for .xdf.boxes.ce.cbf [_ "Press this button to exclude the character you\nhave entered in field-final position."]
    balloonhelp_for .xdf.boxes.ce [_ "Define a new exclusion by entering the character\nto be excluded and selecting the position or\npositions in which it is to be excluded."]
}

proc AddExclusionEntry {} {
    .xdf.boxes.ce.ent configure -state normal
    .xdf.boxes.ce.cbi configure -state normal
    .xdf.boxes.ce.cbm configure -state normal
    .xdf.boxes.ce.cbf configure -state normal
    focus .xdf.boxes.ce.ent
}

proc RemoveExclusionEntry {} {
    .xdf.boxes.lb delete [.xdf.boxes.lb index active]
}

# Put the entries in the listbox into storage
# and write them to a file. Insert the name of the file into the exclusion file
# entry field.
proc SaveExclusionDefinitions {KeyName} {
    set cnt [.xdf.boxes.lb size];
    set xf [tk_getSaveFile -initialfile [format [_ "ExclusionDefinitionsForKey%d"] $::InverseKeyMap($KeyName)]];
    if {[string equal $xf ""]} {
	ShowMessage [_ "File selection aborted."]
	return
    }
    if {[catch {open $xf "w"} xfh ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $xf];
	return ;
    }
    for {set i 0} {$i < $cnt} {incr i} {
	set entry [string trimright [.xdf.boxes.lb get $i]]
	if {[scan $entry "%s%s" chr pos] == 2} {
	    lappend stlist [list $chr $pos]
	    puts $xfh $entry;
	}
    }
    close $xfh;
    set ::KeyInfo(ExclusionList,$KeyName) $stlist;
    $KeyName.exc.bot.ent delete 0 end;
    $KeyName.exc.bot.ent insert 0 $xf;
    destroy .xdf;
}

#Transfer the new entry from the entry widgets to the listbox.
proc InsertExclusionEntry {} {
    set pl "";
    if {[expr $::ExcInitialP + $::ExcMedialP + $::ExcFinalP] == 0} {
	ShowMessage [_ "No position is specified."]
	return
    }
    if {$::ExcInitialP} {lappend pl "i"}
    if {$::ExcMedialP}  {lappend pl "m"}
    if {$::ExcFinalP}   {lappend pl "f"}
    .xdf.boxes.lb insert end [format "%-*s%s" $::xcwidth  [.xdf.boxes.ce.ent get] [join $pl ""]]
    .xdf.boxes.ce.ent delete 0 end;
    .xdf.boxes.ce.cbi deselect
    .xdf.boxes.ce.cbm deselect
    .xdf.boxes.ce.cbf deselect
    .xdf.boxes.ce.ent configure -state disabled
    .xdf.boxes.ce.cbi configure -state disabled
    .xdf.boxes.ce.cbm configure -state disabled
    .xdf.boxes.ce.cbf configure -state disabled
}

proc InsertStoredExclusionsIntoListbox {KeyName} {
    if {[info exists ::KeyInfo(ExclusionList,$KeyName)]} {
	foreach i $::KeyInfo(ExclusionList,$KeyName) {
	    set chr [lindex $i 0];
	    set pos [lindex $i 1];
	    .xdf.boxes.lb insert end [format "%-*s%s" $::xcwidth $chr $pos]
	}
    }
}

proc InsertStoredSortOrderIntoTextWidget {KeyName} {
    if {[info exists ::KeyInfo(SortOrder,$KeyName)]} {
	.sod.tf.txt insert 1.0 $::KeyInfo(SortOrder,$KeyName);
    }
}

proc InsertStoredSubstitutionsIntoTextWidget {KeyName} {
    if {[info exists ::KeyInfo(Substitutions,$KeyName)]} {
	.subd.tf.txt insert  1.0 [lindex $::KeyInfo(Substitutions,$KeyName) 0];
	.subd.tf.txt2 insert 1.0 [lindex $::KeyInfo(Substitutions,$KeyName) 1];
    }
}

proc InsertStoredSortOrderSeparatorsIntoTextWidget {KeyName} {
    if {[info exists ::KeyInfo(SortOrderSeparators,$KeyName)]} {
	.sos.tf.txt insert 1.0 $::KeyInfo(SortOrderSeparators,$KeyName);
    }
}

proc PopupSetDateFormat {KeyName} {
    if {[winfo exists .ymd]} {
	raise .ymd;
	return
    }
    toplevel .ymd -borderwidth 4 -relief raised
    wm title .ymd "";
    after idle {
	update idletasks
	set xmax [winfo screenwidth .ymd]
	set ymax [winfo screenheight .ymd]
	set x0 [expr 1 * ($xmax -[winfo reqwidth .ymd])/3];
	set y0 [expr 1 * ($ymax -[winfo reqheight .ymd])/3];
	wm geometry .ymd "+$x0+$y0";
    }
    frame .ymd.top
    label .ymd.title -text [format [_ "Enter Date Format for Key %d"] $::InverseKeyMap($KeyName)];
    frame .ymd.bot -borderwidth 2
    radiobutton .ymd.top.c1rb1 -text [_ "y"] -variable ::ymdc1 -value 1 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c1rb2 -text [_ "m"] -variable ::ymdc1 -value 2 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c1rb3 -text [_ "d"] -variable ::ymdc1 -value 3 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c2rb1 -text [_ "y"] -variable ::ymdc2 -value 1 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c2rb2 -text [_ "m"] -variable ::ymdc2 -value 2 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c2rb3 -text [_ "d"] -variable ::ymdc2 -value 3 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c3rb1 -text [_ "y"] -variable ::ymdc3 -value 1 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c3rb2 -text [_ "m"] -variable ::ymdc3 -value 2 -bg $::ColorSpecs(PageChoice,Background)
    radiobutton .ymd.top.c3rb3 -text [_ "d"] -variable ::ymdc3 -value 3 -bg $::ColorSpecs(PageChoice,Background)
    entry .ymd.top.de1 -width 1
    entry .ymd.top.de2 -width 1
    bind .ymd.top.de1 <Control-v> {InsertClipboardContents %W}
    bind .ymd.top.de2 <Control-v> {InsertClipboardContents %W}
    .ymd.top.de1 insert 0 [string index $::KeyInfo(DateFormat,$KeyName) 1]
    .ymd.top.de2 insert 0 [string index $::KeyInfo(DateFormat,$KeyName) 3]
    
    for {set c 1} {$c <= 3} {incr c} {
	for {set r 1} {$r <= 3} {incr r} {
	    grid .ymd.top.c${c}rb$r -row [expr $r -1] -column [expr 2*($c-1)]
	    .ymd.top.c${c}rb$r deselect
	}
    }
    grid .ymd.top.de1 -row 1 -column 1
    grid .ymd.top.de2 -row 1 -column 3

    grid columnconfigure .ymd.top 1 -pad 20
    grid columnconfigure .ymd.top 3 -pad 20
    grid columnconfigure .ymd.top 5 -pad 20

    button .ymd.bot.b1 -text [_ "Cancel"] -command {destroy .ymd}
    button .ymd.bot.b2 -text [_ "Reset"] -command YMDReset
    button .ymd.bot.b3 -text [_ "Save"]  -command [list YMDSave $KeyName]
    pack .ymd.title -side top  -expand 1
    pack .ymd.bot.b1 .ymd.bot.b2 .ymd.bot.b3 -side left -padx 6
    pack .ymd.top -side top -pady 3
    pack .ymd.bot -side bottom -pady 3
    trace variable ::ymdc1 w ymdc1Action
    trace variable ::ymdc2 w ymdc2Action
    trace variable ::ymdc3 w ymdc3Action
    set bhmsg [_ "Set the format used for dates."]
    balloonhelp_for .ymd       $bhmsg
    balloonhelp_for .ymd.title $bhmsg
    balloonhelp_for .ymd.top   $bhmsg
    balloonhelp_for .ymd.bot   $bhmsg
    balloonhelp_for .ymd.top.de1 [_ "Enter the first delimiter character here."]
    balloonhelp_for .ymd.top.de2 [_ "Enter the second delimiter character here."]
    balloonhelp_for .ymd.top.c1rb1 [_ "Click to make this the first field of a date."]
    balloonhelp_for .ymd.top.c1rb2 [_ "Click to make this the first field of a date."]
    balloonhelp_for .ymd.top.c1rb3 [_ "Click to make this the first field of a date."]
    balloonhelp_for .ymd.top.c2rb1 [_ "Click to make this the second field of a date."]
    balloonhelp_for .ymd.top.c2rb2 [_ "Click to make this the second field of a date."]
    balloonhelp_for .ymd.top.c2rb3 [_ "Click to make this the second field of a date."]
    balloonhelp_for .ymd.top.c3rb1 [_ "Click to make this the third field of a date."]
    balloonhelp_for .ymd.top.c3rb2 [_ "Click to make this the third field of a date."]
    balloonhelp_for .ymd.top.c3rb3 [_ "Click to make this the third field of a date."]
    balloonhelp_for .ymd.bot.b1    [_ "Cancel resetting the date format.\nNothing will be changed."]
    balloonhelp_for .ymd.bot.b2    [_ "Deselect all selections and clear the delimiters."]
    balloonhelp_for .ymd.bot.b3    [_ "Store the current settings and close the window."]
}

proc YMDReset {} {
    trace vdelete ::ymdc1 w ymdc1Action
    trace vdelete ::ymdc2 w ymdc2Action
    trace vdelete ::ymdc3 w ymdc3Action
    for {set c 1} {$c <= 3} {incr c} {
	for {set r 1} {$r <= 3} {incr r} {
	    .ymd.top.c${c}rb$r configure -state normal
	    .ymd.top.c${c}rb$r deselect
	}
    }
    .ymd.top.de1 delete 0 end
    .ymd.top.de2 delete 0 end
    trace variable ::ymdc1 w ymdc1Action
    trace variable ::ymdc2 w ymdc2Action
    trace variable ::ymdc3 w ymdc3Action
}

proc ymdc1Action {n e o} {
    .ymd.top.c2rb$::ymdc1 configure -state disabled
    .ymd.top.c3rb$::ymdc1 configure -state disabled
}

proc ymdc2Action {n e o} {
    .ymd.top.c1rb$::ymdc2 configure -state disabled
    .ymd.top.c3rb$::ymdc2 configure -state disabled
}

proc ymdc3Action {n e o} {
    .ymd.top.c1rb$::ymdc3 configure -state disabled
    .ymd.top.c2rb$::ymdc3 configure -state disabled
}

proc YMDSave {KeyName} {
    set ic [list y m d];
    set df [format "%s%s%s%s%s"\
		[lindex $ic [expr $::ymdc1 -1]]\
		[.ymd.top.de1 get]\
		[lindex $ic [expr $::ymdc2 -1]]\
		[.ymd.top.de2 get]\
		[lindex $ic [expr $::ymdc3 -1]]]
    set ::KeyInfo(DateFormat,$KeyName) $df;
    trace vdelete ::ymdc1 w ymdc1Action
    trace vdelete ::ymdc2 w ymdc2Action
    trace vdelete ::ymdc3 w ymdc3Action
    destroy .ymd;
}

# Put the entries in the listbox into storage
# and write them to a file. Insert the name of the file into the sort order file
# entry field.
proc SaveSortOrder {KeyName} {
    set xf [tk_getSaveFile -initialfile [format [_ "SortOrderDefinitionsForKey%d"] $::InverseKeyMap($KeyName)]];
    if {[string equal $xf ""]} {
	ShowMessage [_ "File selection aborted."]
	return
    }
    if {[catch {open $xf "w"} xfh ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $xf];
	return ;
    }
    set contents [string trimright [.sod.tf.txt get 1.0 end]]
    puts $xfh $contents;
    close $xfh;
    set ::KeyInfo(SortOrderDef,$KeyName) $contents;
    $KeyName.so.bot.ent delete 0 end;
    $KeyName.so.bot.ent insert 0 $xf;
    destroy .sod;
}

proc EnterSortOrder {KeyName} {
    if {[winfo exists .sod]} {
	raise .sod;
	return
    }
    set kn $::InverseKeyMap($KeyName);
    toplevel .sod -borderwidth 4 -relief raised
    wm title .sod "";
    after idle {
	update idletasks
	set xmax [winfo screenwidth .sod]
	set ymax [winfo screenheight .sod]
	set x0 [expr 1 * ($xmax -[winfo reqwidth .sod])/3];
	set y0 [expr 1 * ($ymax -[winfo reqheight .sod])/3];
	wm geometry .sod "+$x0+$y0";
    }
    label .sod.title -text [format [_ "Define Sort Order for Key %d"] $kn];
    frame .sod.tf
    text  .sod.tf.txt -width 20 -height 15 -yscrollcommand {.sod.tf.sbar set}\
	-bg $::ColorSpecs(SortOrderDefinition,Background) -font UserTechnicalTextFont
    scrollbar .sod.tf.sbar -command {.sod.tf.txt yview} -troughcolor $::ColorSpecs(SortOrderDefinition,Background)\
	-bg $::ColorSpecs(SortOrderDefinition,Background) -activebackground $::ColorSpecs(SortOrderDefinition,Background);
    InsertStoredSortOrderIntoTextWidget $kn;
    pack .sod.tf.txt  -side left  -expand 1 -fill both
    pack .sod.tf.sbar -side right -expand 1 -fill y
    frame .sod.finalbs; #Buttons for finalizing - cancel and proceed
    button .sod.finalbs.p  -text [_ "Save"]  -command "SaveSortOrder $KeyName"
    button .sod.finalbs.c  -text [_ "Cancel"] -command {destroy .sod} -anchor w
    pack .sod.finalbs.c -side left  -expand 1 -anchor w 
    pack .sod.finalbs.p -side right -expand 1 -anchor e 

    #Put it all together
    pack .sod.title   -side top -anchor w -expand 1
    pack .sod.finalbs -side bottom -expand 1 -fill x
    frame .sod.fbsep -height 6 -borderwidth 0 -relief flat
    pack .sod.fbsep   -side bottom -expand 1 -fill x
    pack .sod.tf  -side top -expand 1 -fill both

    focus .sod.tf.txt
    BindKeys .sod;
    bind .sod <FocusIn> {SetInsertionTargets .sod.tf.txt}
    bind .sod.tf.txt <Control-v> {InsertClipboardContents %W}
    balloonhelp_for .sod.tf.txt "Everything on the same line has the same sort rank.\nThe higher the line, the higher the sort rank.\nStrings of characters not containing separators define multigraphs.";
}

# Put the entries in the listbox into storage
# and write them to a file. Insert the name of the file into the sort order file
# entry field.
proc SaveSubstitutions {KeyName} {
    set xf [tk_getSaveFile -initialfile [format [_ "SubstitutionDefinitionsForKey%d"] $::InverseKeyMap($KeyName)]];
    if {[string equal $xf ""]} {
	ShowMessage [_ "File selection aborted."]
	return
    }
    if {[catch {open $xf "w"} xfh ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $xf];
	return ;
    }
    set Input  [string trimright [.subd.tf.txt get 1.0 end]]
    set Output [string trimright [.subd.tf.txt2 get 1.0 end]]
    set ::KeyInfo(Substitutions,$KeyName) [list $Input $Output];
    set InputLines  [split $Input "\n"];
    set OutputLines [split $Output "\n"];
    foreach a $InputLines b $OutputLines { 
	puts $xfh [format "%s\t%s" [string trimright $a "\n"] [string trimright $b "\n"]];
    }
    close $xfh;
    $KeyName.sub.bot.ent delete 0 end;
    $KeyName.sub.bot.ent insert 0 $xf;
    destroy .subd;
}

set OldSubEntryLimit 0;
proc UpdateSubEntryLineNumbers {KeyNumber} {
    set LimitL [expr [lindex [.subd.tf.txt  index end] 0] - 1];
    set LimitR [expr [lindex [.subd.tf.txt2 index end] 0] - 1];
    if {$LimitL > $LimitR} {
	set Limit $LimitL;
    } else {
	set Limit $LimitR;
    }
    if {$Limit == $::OldSubEntryLimit} {return}
    .subd.tf.num configure -state normal;
    .subd.tf.num delete 1.0 end
    for {set i 1} {$i <= $Limit} {incr i} {
	.subd.tf.num insert insert \
	    [format "%02d" $i]
    }
    .subd.tf.num configure -state disabled
    set ::OldSubEntryLimit $Limit;
}

proc YViewBoth {args} {
    set ac [llength $args];
    set a  [lindex $args 0]
    set b  [lindex $args 1]
    if {$ac == 3} {
	set c  [lindex $args 2]
	.subd.tf.txt  yview $a $b $c;
	.subd.tf.num  yview $a $b $c;
	.subd.tf.txt2 yview $a $b $c;
    } else {
	.subd.tf.txt  yview $a $b;
	.subd.tf.num  yview $a $b;
	.subd.tf.txt2 yview $a $b;
    }
}

proc SubdScrollSet {a b} {
    .subd.tf.sbar set $a $b;
    .subd.tf.sbar2 set $a $b;
}

proc EnterSubstitutions {KeyName} {
    if {[winfo exists .subd]} {
	raise .subd;
	return
    }
    set KeyNumber $::InverseKeyMap($KeyName);
    toplevel .subd -borderwidth 4 -relief raised
    wm title .subd "";
    after idle {
	update idletasks
	set xmax [winfo screenwidth .subd]
	set ymax [winfo screenheight .subd]
	set x0 [expr 1 * ($xmax -[winfo reqwidth .subd])/3];
	set y0 [expr 1 * ($ymax -[winfo reqheight .subd])/3];
	wm geometry .subd "+$x0+$y0";
    }
    label .subd.title -text [format [_ "Define Substitutions for Key %d"] $KeyNumber];
    frame .subd.tf
    text  .subd.tf.txt -width 12 -height $::SubdHeight -yscrollcommand SubdScrollSet \
	-bg $::ColorSpecs(Substitutions,Background) -font UserTechnicalTextFont
    text .subd.tf.num -width 2 -height $::SubdHeight -yscrollcommand SubdScrollSet \
	-bg $::ColorSpecs(Substitutions,Background) -exportselection 0 -font MainFont
    text  .subd.tf.txt2 -width 12 -height $::SubdHeight -yscrollcommand SubdScrollSet \
	-bg $::ColorSpecs(Substitutions,Background) -font UserTechnicalTextFont
    scrollbar .subd.tf.sbar -command {YViewBoth} \
	-troughcolor $::ColorSpecs(Substitutions,Background)\
	-bg $::ColorSpecs(Substitutions,Background) -activebackground $::ColorSpecs(Substitutions,Background);
    scrollbar .subd.tf.sbar2 -command {YViewBoth} \
	-troughcolor $::ColorSpecs(Substitutions,Background)\
	-bg $::ColorSpecs(Substitutions,Background) -activebackground $::ColorSpecs(Substitutions,Background);
    InsertStoredSubstitutionsIntoTextWidget $KeyName;
    UpdateSubEntryLineNumbers $KeyNumber;
    pack .subd.tf.sbar2 -side left -expand 1 -fill y
    pack .subd.tf.txt  -side left  -expand 1 -fill both
    pack .subd.tf.num  -side left  -expand 1 -fill both
    pack .subd.tf.txt2 -side left  -expand 1 -fill both
    pack .subd.tf.sbar -side right -expand 1 -fill y
    frame .subd.finalbs; #Buttons
    button .subd.finalbs.p  -text [_ "Save"]  -command "SaveSubstitutions $KeyName"
    button .subd.finalbs.c  -text [_ "Cancel"] -command {destroy .subd} -anchor w
    pack .subd.finalbs.c -side left  -expand 1 -fill both
    pack .subd.finalbs.p -side right -expand 1 -fill both

    #Put it all together
    pack .subd.title   -side top -anchor w -expand 1
    pack .subd.finalbs -side bottom -expand 1 -fill x
    frame .subd.fbsep -height 6 -borderwidth 0 -relief flat
    pack .subd.fbsep   -side bottom -expand 1 -fill x
    pack .subd.tf  -side top -expand 1 -fill both

    focus .subd.tf.txt
    BindKeys .subd;
    bind .subd.tf.txt <KeyPress> "UpdateSubEntryLineNumbers $KeyNumber"
    bind .subd.tf.txt2 <KeyPress> "UpdateSubEntryLineNumbers $KeyNumber"
    bind .subd.tf.txt   <Tab> {focus -force .subd.tf.txt2;break}
    bind .subd.tf.txt2  <Tab> {focus -force .subd.tf.txt;break}
    bind .subd.tf.txt  <Motion>  {
	.subd.tf.txt2 yview moveto [lindex [.subd.tf.txt  yview] 0]
	.subd.tf.num  yview moveto [lindex [.subd.tf.txt  yview] 0]
    }
    bind .subd.tf.txt2 <Motion>  {
	.subd.tf.txt  yview moveto [lindex [.subd.tf.txt2 yview] 0]
	.subd.tf.num  yview moveto [lindex [.subd.tf.txt2 yview] 0]
    }
    bind .subd <FocusIn> {SetInsertionTargets .subd.tf.txt}
    bind .subd.tf.txt <Control-v> {InsertClipboardContents %W}
    balloonhelp_for .subd.tf.txt [_ "On this side enter the regular expressions that\nmatch the text you want to replace."];
    balloonhelp_for .subd.tf.txt2 [_ "On this side enter the text that you want to substitute\nfor the strings that match the regular expressions on the left"];
}

# Put the entries in the listbox into storage
# and write them to a file. Insert the name of the file into the sort order file field.
proc SaveSortOrderSeparators {KeyName} {
    set xf [tk_getSaveFile -initialfile \
		[format [_ "SortOrderSeparatorsForKey%d"] $::InverseKeyMap($KeyName)]];
    if {[string equal $xf ""]} {
	ShowMessage [_ "File selection aborted."]
	return
    }
    if {[catch {open $xf "w"} xfh ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $xf];
	return ;
    }
    set contents [string trimright [.sos.tf.txt get 1.0 end]]
    puts $xfh $contents;
    close $xfh;
    set ::KeyInfo(SortOrderSeparators,$KeyName) $contents;
    $KeyName.sos.bot.ent delete 0 end;
    $KeyName.sos.bot.ent insert 0 $xf;
    destroy .sos;
}


proc InsertSepChar {c} {
    .sos.tf.txt insert insert $c;
}


set SepChars [list \
	      {\u0009 "tab"} \
	      {\u0020 "space"} \
	      {\u1361 "ethiopic wordspace"} \
	      {\u1680 "ogham space mark"} \
	      {\u3000 "ideographic space"}
];

# Other space characters we could insert if desired.
#	      {\u2000 "en quad"} \
#	      {\u2001 "em quad"} \
#	      {\u2002 "en space"} \
#	      {\u2003 "em space"} \
#	      {\u2004 "three-per-em space"} \
#	      {\u2005 "four-per-em space"} \
#	      {\u2006 "six-per-em space"} \
#	      {\u2007 "figure space"} \
#	      {\u2008 "punctuation space"} \
#	      {\u2009 "thin space"} \
#	      {\u200A "hair space"} \
#	      {\u200B "zero width space"} \
#	      {\u205F "medium mathematical space"} \

proc EnterSortOrderSeparators {KeyName} {
    if {[winfo exists .sos]} {
	raise .sos;
	return
    }
    set kn $::InverseKeyMap($KeyName);
    toplevel .sos -borderwidth 4 -relief raised
    after idle {
	update idletasks
	set xmax [winfo screenwidth .sos]
	set ymax [winfo screenheight .sos]
	set x0 [expr 1 * ($xmax -[winfo reqwidth .sos])/3];
	set y0 [expr 1 * ($ymax -[winfo reqheight .sos])/3];
	wm geometry .sos "+$x0+$y0";
    }
    label .sos.title -text [format [_ "Specify Separators for Key %d"] $kn];
    frame .sos.tf
    text  .soso.tf.txt -width 20 -height 15 -yscrollcommand {.sos.tf.sbar set}\
	-bg $::ColorSpecs(SortOrderSeparator,Background) -font UserTechnicalTextFont
    scrollbar .sos.tf.sbar -command {.sos.tf.txt yview} -troughcolor $::ColorSpecs(SortOrderSeparator,Background)\
	-bg $::ColorSpecs(SortOrderSeparator,Background) -activebackground $::ColorSpecs(SortOrderSeparator,Background);
    InsertStoredSortOrderSeparatorsIntoTextWidget $kn;
    frame .sos.tf.clist
    set cnt 0;
    set code 0;
    set msg [_ "Press this button to enter the specified character into\nthe text box and make it one of the separator\ncharacters for this key's sort order definition file."]
    foreach k $::SepChars {
	incr cnt;
	scan [lindex $k 0] "%c" code
	set bn .sos.tf.clist.b$cnt;
	button $bn \
	    -text [format "0x%04x   %s" $code  [lindex $k 1]] -anchor w\
	    -command [list InsertSepChar [lindex $k 0]]
	pack $bn -side top -expand 1 -fill both
	balloonhelp_for $bn $msg;
    }
    pack .sos.tf.txt  -side left  -expand 1 -fill both
    pack .sos.tf.sbar -side left -expand 1 -fill y
    pack .sos.tf.clist -side right -expand 1 -fill y
    frame .sos.finalbs; #Buttons for finalizing - cancel and proceed
    button .sos.finalbs.p  -text [_ "Save"]  -command "SaveSortOrderSeparators $KeyName"
    button .sos.finalbs.c  -text [_ "Cancel"] -command {destroy .sos} -anchor w
    pack .sos.finalbs.c -side left  -expand 1 -anchor w 
    pack .sos.finalbs.p -side right -expand 1 -anchor e 

    #Put it all together
    pack .sos.title   -side top -anchor w -expand 1
    pack .sos.finalbs -side bottom -expand 1 -fill x
    frame .sos.fbsep -height 6 -borderwidth 0 -relief flat
    pack .sos.fbsep   -side bottom -expand 1 -fill x
    pack .sos.tf  -side top -expand 1 -fill both

    focus .sos.tf.txt
    BindKeys .sos;
    bind .sos <FocusIn> {SetInsertionTargets .sos.tf.txt}
    bind .sos.tf.txt <Control-v> {InsertClipboardContents %W}
    balloonhelp_for .sos.tf.txt \
	[_ "Enter here the characters that should be treated\nas separators in this key's sort order file."];
    set msg	[_ "Enter the characters that you want to be treated as\nseparators into the text box. You may enter them\ndirectly or press the buttons on the right. "];
    balloonhelp_for .sos $msg;
    balloonhelp_for .sos.title $msg;
    balloonhelp_for .sos.tf $msg;
    balloonhelp_for .sos.finalbs $msg;
    balloonhelp_for .sos.fbsep $msg;
    set msg [_ "Press this button to store the separators that you have entered."];
    balloonhelp_for .sos.finalbs.p $msg;
    set msg [_ "Press this button to dismiss this control panel without making\nany change in the sort order file separators for this key."];
    balloonhelp_for .sos.finalbs.c $msg;
}

proc ConfigureGenOne {} {
    set rpwid [lindex [grid bbox .gen.ctr 0 0] 2]
    grid columnconfigure .gen.ctr 1 -minsize $rpwid
}

proc ConfigureGenTwo {} {
    set wrwid [lindex [grid bbox .gen.ctr 1 0] 2]
    grid columnconfigure .gen.ctr 0 -minsize $wrwid
}

proc SetBalloonHelpShowP {b} {
    set v [Boolean $b]
    set ::BalloonHelpP $v;
    balloonhelp_control $v;
}

proc SetBMPOnlyP {b} {
    set ::BMPOnlyP [Boolean $b];
}

proc SetEOLCarriageReturnP {b} {
    set ::EOLCarriageReturnP [Boolean $b];
}

proc SetInvertOrderGloballyP {b} {
    set ::GlobalInversionP [Boolean $b];
}

proc SetReservePrivateUseAreasP {b} {
    set ::ReservePrivateUseAreasP [Boolean $b];
}

proc SetWholeRecordIsKeyP {b} {
    set ::WholeRecordIsKeyP [Boolean $b];
}

proc SelectBrowser {s} {
    set ::DefaultBrowser $s;
}

proc SetFieldSeparators {s} {
    set ::FieldSeparator $s;
}

proc SetRecordSeparator {s} {
    set ::RecordSeparator $s;
}

proc SetRecordParse {s} {
    set ::RecordParse $s;
}

set CustomCharacterChartFileNameList "";
proc StoreCustomCharacterChartFileName {fn} {
    lappend ::CustomCharacterChartFileNameList $fn;
}

set CustomCharacterChartDataList "";
proc StoreCustomCharacterChartInPlace {dl} {
    lappend ::CustomCharacterChartDataList $dl;
}

#Initialization file stuff

#This creates a procedure for setting the color of each combination of
#object and feature. The procedure validates the color specification, so
#invalid color specifications are trapped in the slave interpreter.
proc DefineColorSettingProcs {} {
    global ColorProcList;
    foreach cs [array names ::ColorSpecs] {
	set csl [split $cs ","]
	set obj   [lindex $csl 0]
	set which [lindex $csl 1]
	set procname [format "Set%s%sColor" $obj $which]
	lappend ColorProcList $procname
	eval [list "proc" $procname \
		   \
		  "\{cs\}" \
		  "if \{!\[::validcolor::IsColorSpecQ \$cs\]\} \{
		  puts \[format \[_ \"\\\"%s\\\" is not a valid color specification.\"\] \$cs\]
		  return;
		 \}
		  set ::ColorSpecs($obj,$which) \$cs"]
    }
}

#Returns a list of init file commands representing the current color settings.
proc SaveColorSettings {} {
    set cl [list]
    foreach cs [array names ::ColorSpecs] {
	set csl [split $cs ","]
	set obj   [lindex $csl 0]
	set which [lindex $csl 1]
	set ps $::ColorSpecs($cs)
	if {[string index $ps 0] == "\#"} {
	    set ps \\$::ColorSpecs($cs)
	} else {
	    set ps [list $ps];
	}
	lappend cl [format "%s%sColor %s" $obj $which $ps]
    }
    return [lsort $cl];
}

#Check whether the argument represents a Boolean value
#and interpret it if it is.
proc Boolean {s} {
     switch -regexp $s {
	 1	{return 1}
	 T.*	{return 1}
	 t.*	{return 1}
	 Y.*	{return 1}
	 y.*	{return 1}
	 ok	{return 1}
	 on	{return 1}
	 0 	{return 0}
	 F.*	{return 0}
	 f.*	{return 0}
	 N.*	{return 0}
	 n.*	{return 0}
	 off	{return 0}
	 default {error}
     }
}

proc GetInitCommands {} {
    # We want a fresh interpreter so that we don't
    # list any commands defined in a user's init file.
    if { [interp exists init]} {
	interp delete init
    }
    InitFileSetup;
    set n 0
    #We rely on the fact that all builtin commands have lower-case
    #case names to separate the commands we provide from the builtins.
    return [interp eval init {lsort [info commands \[A-Z\]*]}]
}

proc SaveInitCommands {} {
    set SaveFile [tk_getSaveFile -initialfile [_ "InitFileCommands"]];
    if {[string equal $SaveFile ""]} {
	return ;
    } else {
        if { [catch {open $SaveFile "w+"} fh] != 0} {
	    ShowMessage [format \
		     [_ "Unable to open file %s in which to save class definitions."] \
		     $SaveFile];
	    return ;
        }
    }
    set n 0;
    foreach l [GetInitCommands] {
	incr n;
	puts $fh [format "\[%03d\] %s" $n $l]
    }
    close $fh;
}

proc PopupInitCommandList {} {
    if {[PopupDown InitCommandList] ==1} {return}
    set po  [CreateTextDisplay [_ "Initialization File Commands"] 65 10];
    BindKeys $po;
    set ::PopupList(InitCommandList) $po;
    set n 0;
    foreach l [GetInitCommands] {
	incr n;
	AppendToTextDisplay $po [format "\[%03d\] %s\n" $n $l];
    }
}

proc SaveMiscellaneousSettings {} {
    set MiscellaneousInitCommandList \
	[list \
	     [SlaveName SetUseCheckbuttonP] $::UseCheckbuttonP\
	     [SlaveName SetRecordParseDisplayedP] $::RecordParseDisplayedP\
	     [SlaveName SetMiscOptionsDisplayedP] $::MiscOptionsDisplayedP\
	     [SlaveName SetUnicodeDisplayedP] $::UnicodeDisplayedP\
	     [SlaveName SetAlgorithmDisplayedP] $::AlgorithmDisplayedP\
	     [SlaveName SetInputFileDisplayedP] $::InputFileDisplayedP\
	     [SlaveName SetOutputFileDisplayedP] $::OutputFileDisplayedP\
	     [SlaveName SetExclusionsDisplayedP] $::ExclusionsDisplayedP\
	     [SlaveName SetKeyFieldIdentificationDisplayedP] $::KeyFieldIdentificationDisplayedP\
	     [SlaveName SetMiscellaneousOptionsDisplayedP] $::MiscellaneousOptionsDisplayedP\
	     [SlaveName SetSortOrderDisplayedP] $::SortOrderDisplayedP\
	     [SlaveName SetSortType DisplayedP] $::SortType DisplayedP\
	     [SlaveName SetSubstitutionsDisplayedP] $::SubstitutionsDisplayedP\
	     [SlaveName SetSortAlgorithm] $::SortAlgorithm\
	     [SlaveName SelectBrowser] $::DefaultBrowser\
	     [SlaveName SetBalloonHelpShowP]			$::BalloonHelpP \
	     [SlaveName SetBMPOnlyP]			$::BMPOnlyP \
	     [SlaveName SetDisplayConsonantChartColumnLabelsP]	$::DisplayConsonantChartColumnLabelsP \
	     [SlaveName SetDisplayConsonantChartRowLabelsP]	$::DisplayConsonantChartRowLabelsP \
	     [SlaveName SetDisplayVowelChartColumnLabelsP]	$::DisplayVowelChartColumnLabelsP \
	     [SlaveName SetDisplayVowelChartRowLabelsP]		$::DisplayVowelChartRowLabelsP \
	     [SlaveName SetEOLCarriageReturnP]			$::EOLCarriageReturnP \
	     [SlaveName SetFieldSeparators]			[list $::FieldSeparator] \
	     [SlaveName SetInvertOrderGloballyP]		$::GlobalInversionP \
	     [SlaveName SetRecordSeparator]			[list $::RecordSeparator] \
	     [SlaveName SetRecordParse]				$::RecordParse \
	     [SlaveName SetReservePrivateUseAreasP]		$::ReservePrivateUseAreasP\
	     [SlaveName SetWholeRecordIsKeyP]			$::WholeRecordIsKeyP]
    set cl [list]
    foreach {command value} $MiscellaneousInitCommandList {
	lappend cl [format "%s %s" $command $value]
    }
    foreach title [array names ::SpecialCharacterPalette] {
	if {[string match "*WidgetName" $title]} {continue}
	set info $::SpecialCharacterPalette($title);
	if {[llength $info] > 2} {
	    lappend cl [list ReadCharacterChart [lindex $info 2]];
	} else {
	    lappend cl [ReconstructCharacterChartDefinition $title];
	}
    }
    return $cl;
}

proc ReconstructCharacterChartDefinition {Title} {
    set info $::SpecialCharacterPalette($Title);
    set txt DefineCharacterChart;
    append txt " \{\n"
    append txt [format "%s" [Quote $Title]]
    set flds [llength $info];
    if {$flds > 1} {
	append txt [format "|%d" [lindex $info 1]];# Number of columns
    }
    if {$flds > 2} {
	append txt [format "|%s" [lindex $info 2]];# Font family
    }
    if {$flds > 3} {
	append txt [format "|%s" [lindex $info 3]];# Font size
    }
    append txt "\n"
    #Each entry consists of a unicode string and a gloss
    foreach x [lindex $info 0] {
	set string [lindex $x 0]
	set gloss  [lindex $x 1]
	append txt [format "%s|%s\n" [UnicodeStringToHex $string] [Quote $gloss]];
    }
    append txt "\}"
    return  $txt;
}

proc PopupDown {n} {
    global PopupList;
    if {[info exists PopupList($n)]} {
	if {[winfo exists $PopupList($n)]} {
	    destroy $PopupList($n);
	    return 1;
	}
    }
    return 0;
}

proc ExplainInsertionSort {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown InsertionSort] ==1} {return}
    set po [CreateTextDisplay [_ "Insertion Sort"] $HPWidth $HPLines]
    set PopupList(InsertionSort) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "Insertion sort is a stable sorting algorithm.\n"];
    AppendToTextDisplay $po [_ "That is, if two items compare as equal, their order in the output will be the same as in the input.\n"];
    AppendToTextDisplay $po [_ "It tends to be slower than than the other algorithms for large numbers of items, but tends to be fast when the input is already in order or nearly so."];
    AppendToTextDisplay $po [_ "For more information, consult the Wikipedia entry:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Insertion_sort" {ShowWebPage http://en.wikipedia.org/wiki/Insertion_sort};
    AppendToTextDisplay $po [_ "\nFor comparison of sorting algorithms see:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Sorting_algorithm" {ShowWebPage http://en.wikipedia.org/wiki/Sorting_algorithm};
}

proc ExplainMergeSort {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown MergeSort] ==1} {return}
    set po [CreateTextDisplay [_ "Merge Sort"] $HPWidth $HPLines]
    set PopupList(MergeSort) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "Merge sort is a stable sorting algorithm.\n"];
    AppendToTextDisplay $po [_ "That is, if two items compare as equal, their order in the output will be the same as in the input.\n"];
    AppendToTextDisplay $po [_ "It scales well for large numbers of items."];
    AppendToTextDisplay $po [_ "For more information, consult the Wikipedia entry:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Merge_sort" {ShowWebPage http://en.wikipedia.org/wiki/Merge_sort};
    AppendToTextDisplay $po [_ "\nFor comparison of sorting algorithms see:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Sorting_algorithm" {ShowWebPage http://en.wikipedia.org/wiki/Sorting_algorithm};
}

proc ExplainQuickSort {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown QuickSort] ==1} {return}
    set po [CreateTextDisplay [_ "Quick Sort"] $HPWidth $HPLines]
    set PopupList(QuickSort) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "Quick Sort is an unstable sorting algorithm.\n"];
    AppendToTextDisplay $po [_ "That is, if two items compare as equal, their order in the output may not be the same as in the input.\n"];
    AppendToTextDisplay $po [_ "It performs well with large numbers of items."];
    AppendToTextDisplay $po [_ "For more information, consult the Wikipedia entry:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Quick_sort" {ShowWebPage http://en.wikipedia.org/wiki/Quick_sort};
    AppendToTextDisplay $po [_ "\nFor comparison of sorting algorithms see:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Sorting_algorithm" {ShowWebPage http://en.wikipedia.org/wiki/Sorting_algorithm};
}

proc ExplainShellSort {} {
    global HPWidth;
    global HPLines;
    global PopupList;
    if {[PopupDown ShellSort] ==1} {return}
    set po [CreateTextDisplay [_ "Shell Sort"] $HPWidth $HPLines]
    set PopupList(ShellSort) $po;
    BindKeys $po;
    AppendToTextDisplay $po [_ "Shell Sort is an unstable sorting algorithm.\n"];
    AppendToTextDisplay $po [_ "That is, if two items compare as equal, their order in the output may not be the same as in the input.\n"];
    AppendToTextDisplay $po [_ "For small numbers of items (say less than 1,000) it may be faster than the other algorithms. For large numbers of items it will probably be slower."];
    AppendToTextDisplay $po [_ "For more information, consult the Wikipedia entry:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Shell_sort" {ShowWebPage http://en.wikipedia.org/wiki/Shell_sort};
    AppendToTextDisplay $po [_ "\nFor comparison of sorting algorithms see:"];
    AppendLinkToTextDisplay $po "http://en.wikipedia.org/wiki/Sorting_algorithm" {ShowWebPage http://en.wikipedia.org/wiki/Sorting_algorithm};
}

#This is a kludge but I encountered quoting hell trying to use (\\s) -> \\ \\1
#e.g.   regsub -all "\\s" $s {\ } r

#Escape spaces and tabs
proc Quote {s} {
    regsub -all "\u0020" $s {\ } r
    regsub -all "\t" $r {\	} ret
    return $ret
}

proc UnicodeStringToHex {s} {
    set rv "";
    foreach c [split $s ""] {
	scan $c "%c" t
	if {$t <= 0xFF} {
	    append rv $c;
	} else {
	    append rv [format "\\u%04X" $t];
	}
    }
    return $rv;
 }

#This package defines a routine IsColorSpecQ that validates X11 color specifications.
#It returns 1 if its argument is either a recognized X11 color name or
#a well formed hexadecimal RGB specification. If so desired, the two routines on
#which it is based, IsColorNameQ and IsColorNumericQ can also be exported.
namespace eval validcolor {
    namespace export IsColorSpecQ;
    variable CN;

    set CN([list ghost white]) 1
    set CN([list white smoke]) 1
    set CN([list floral white]) 1
    set CN([list antique white]) 1
    set CN([list old lace]) 1
    set CN([list papaya whip]) 1
    set CN([list blanched almond]) 1
    set CN([list peach puff]) 1
    set CN([list navajo white]) 1
    set CN([list lemon chiffon]) 1
    set CN([list mint cream]) 1
    set CN([list alice blue]) 1
    set CN([list lavender blush]) 1
    set CN([list misty rose]) 1
    set CN([list dark slate gray]) 1
    set CN([list dark slate grey]) 1
    set CN([list dim gray]) 1
    set CN([list dim grey]) 1
    set CN([list slate gray]) 1
    set CN([list slate grey]) 1
    set CN([list light slate gray]) 1
    set CN([list light slate grey]) 1
    set CN([list light grey]) 1
    set CN([list light gray]) 1
    set CN([list midnight blue]) 1
    set CN([list navy blue]) 1
    set CN([list cornflower blue]) 1
    set CN([list dark slate blue]) 1
    set CN([list slate blue]) 1
    set CN([list medium slate blue]) 1
    set CN([list light slate blue]) 1
    set CN([list medium blue]) 1
    set CN([list royal blue]) 1
    set CN([list dodger blue]) 1
    set CN([list deep sky blue]) 1
    set CN([list sky blue]) 1
    set CN([list light sky blue]) 1
    set CN([list steel blue]) 1
    set CN([list light steel blue]) 1
    set CN([list light blue]) 1
    set CN([list powder blue]) 1
    set CN([list pale turquoise]) 1
    set CN([list dark turquoise]) 1
    set CN([list medium turquoise]) 1
    set CN([list light cyan]) 1
    set CN([list cadet blue]) 1
    set CN([list medium aquamarine]) 1
    set CN([list dark green]) 1
    set CN([list dark olive green]) 1
    set CN([list dark sea green]) 1
    set CN([list sea green]) 1
    set CN([list medium sea green]) 1
    set CN([list light sea green]) 1
    set CN([list pale green]) 1
    set CN([list spring green]) 1
    set CN([list lawn green]) 1
    set CN([list medium spring green]) 1
    set CN([list green yellow]) 1
    set CN([list lime green]) 1
    set CN([list yellow green]) 1
    set CN([list forest green]) 1
    set CN([list olive drab]) 1
    set CN([list dark khaki]) 1
    set CN([list pale goldenrod]) 1
    set CN([list light goldenrod yellow]) 1
    set CN([list light yellow]) 1
    set CN([list light goldenrod]) 1
    set CN([list dark goldenrod]) 1
    set CN([list rosy brown]) 1
    set CN([list indian red]) 1
    set CN([list saddle brown]) 1
    set CN([list sandy brown]) 1
    set CN([list dark salmon]) 1
    set CN([list light salmon]) 1
    set CN([list dark orange]) 1
    set CN([list light coral]) 1
    set CN([list orange red]) 1
    set CN([list hot pink]) 1
    set CN([list deep pink]) 1
    set CN([list light pink]) 1
    set CN([list pale violet red]) 1
    set CN([list medium violet red]) 1
    set CN([list violet red]) 1
    set CN([list medium orchid]) 1
    set CN([list dark orchid]) 1
    set CN([list dark violet]) 1
    set CN([list blue violet]) 1
    set CN([list medium purple]) 1
    set CN(snow) 1
    set CN(GhostWhite) 1
    set CN(WhiteSmoke) 1
    set CN(gainsboro) 1
    set CN(FloralWhite) 1
    set CN(OldLace) 1
    set CN(linen) 1
    set CN(AntiqueWhite) 1
    set CN(PapayaWhip) 1
    set CN(BlanchedAlmond) 1
    set CN(bisque) 1
    set CN(PeachPuff) 1
    set CN(NavajoWhite) 1
    set CN(moccasin) 1
    set CN(cornsilk) 1
    set CN(ivory) 1
    set CN(LemonChiffon) 1
    set CN(seashell) 1
    set CN(honeydew) 1
    set CN(MintCream) 1
    set CN(azure) 1
    set CN(AliceBlue) 1
    set CN(lavender) 1
    set CN(LavenderBlush) 1
    set CN(MistyRose) 1
    set CN(white) 1
    set CN(black) 1
    set CN(DarkSlateGray) 1
    set CN(DarkSlateGrey) 1
    set CN(DimGray) 1
    set CN(DimGrey) 1
    set CN(SlateGray) 1
    set CN(SlateGrey) 1
    set CN(LightSlateGray) 1
    set CN(LightSlateGrey) 1
    set CN(gray) 1
    set CN(grey) 1
    set CN(LightGrey) 1
    set CN(LightGray) 1
    set CN(MidnightBlue) 1
    set CN(navy) 1
    set CN(NavyBlue) 1
    set CN(CornflowerBlue) 1
    set CN(DarkSlateBlue) 1
    set CN(SlateBlue) 1
    set CN(MediumSlateBlue) 1
    set CN(LightSlateBlue) 1
    set CN(MediumBlue) 1
    set CN(RoyalBlue) 1
    set CN(blue) 1
    set CN(DodgerBlue) 1
    set CN(DeepSkyBlue) 1
    set CN(SkyBlue) 1
    set CN(LightSkyBlue) 1
    set CN(SteelBlue) 1
    set CN(LightSteelBlue) 1
    set CN(LightBlue) 1
    set CN(PowderBlue) 1
    set CN(PaleTurquoise) 1
    set CN(DarkTurquoise) 1
    set CN(MediumTurquoise) 1
    set CN(turquoise) 1
    set CN(cyan) 1
    set CN(LightCyan) 1
    set CN(CadetBlue) 1
    set CN(MediumAquamarine) 1
    set CN(aquamarine) 1
    set CN(DarkGreen) 1
    set CN(DarkOliveGreen) 1
    set CN(DarkSeaGreen) 1
    set CN(SeaGreen) 1
    set CN(MediumSeaGreen) 1
    set CN(LightSeaGreen) 1
    set CN(PaleGreen) 1
    set CN(SpringGreen) 1
    set CN(LawnGreen) 1
    set CN(green) 1
    set CN(chartreuse) 1
    set CN(MediumSpringGreen) 1
    set CN(GreenYellow) 1
    set CN(LimeGreen) 1
    set CN(YellowGreen) 1
    set CN(ForestGreen) 1
    set CN(OliveDrab) 1
    set CN(DarkKhaki) 1
    set CN(khaki) 1
    set CN(PaleGoldenrod) 1
    set CN(LightGoldenrodYellow) 1
    set CN(LightYellow) 1
    set CN(yellow) 1
    set CN(gold) 1
    set CN(LightGoldenrod) 1
    set CN(goldenrod) 1
    set CN(DarkGoldenrod) 1
    set CN(RosyBrown) 1
    set CN(IndianRed) 1
    set CN(SaddleBrown) 1
    set CN(sienna) 1
    set CN(peru) 1
    set CN(burlywood) 1
    set CN(beige) 1
    set CN(wheat) 1
    set CN(SandyBrown) 1
    set CN(tan) 1
    set CN(chocolate) 1
    set CN(firebrick) 1
    set CN(brown) 1
    set CN(DarkSalmon) 1
    set CN(salmon) 1
    set CN(LightSalmon) 1
    set CN(orange) 1
    set CN(DarkOrange) 1
    set CN(coral) 1
    set CN(LightCoral) 1
    set CN(tomato) 1
    set CN(OrangeRed) 1
    set CN(red) 1
    set CN(HotPink) 1
    set CN(DeepPink) 1
    set CN(pink) 1
    set CN(LightPink) 1
    set CN(PaleVioletRed) 1
    set CN(maroon) 1
    set CN(MediumVioletRed) 1
    set CN(VioletRed) 1
    set CN(magenta) 1
    set CN(violet) 1
    set CN(plum) 1
    set CN(orchid) 1
    set CN(MediumOrchid) 1
    set CN(DarkOrchid) 1
    set CN(DarkViolet) 1
    set CN(BlueViolet) 1
    set CN(purple) 1
    set CN(MediumPurple) 1
    set CN(thistle) 1
    set CN(snow1) 1
    set CN(snow2) 1
    set CN(snow3) 1
    set CN(snow4) 1
    set CN(seashell1) 1
    set CN(seashell2) 1
    set CN(seashell3) 1
    set CN(seashell4) 1
    set CN(AntiqueWhite1) 1
    set CN(AntiqueWhite2) 1
    set CN(AntiqueWhite3) 1
    set CN(AntiqueWhite4) 1
    set CN(bisque1) 1
    set CN(bisque2) 1
    set CN(bisque3) 1
    set CN(bisque4) 1
    set CN(PeachPuff1) 1
    set CN(PeachPuff2) 1
    set CN(PeachPuff3) 1
    set CN(PeachPuff4) 1
    set CN(NavajoWhite1) 1
    set CN(NavajoWhite2) 1
    set CN(NavajoWhite3) 1
    set CN(NavajoWhite4) 1
    set CN(LemonChiffon1) 1
    set CN(LemonChiffon2) 1
    set CN(LemonChiffon3) 1
    set CN(LemonChiffon4) 1
    set CN(cornsilk1) 1
    set CN(cornsilk2) 1
    set CN(cornsilk3) 1
    set CN(cornsilk4) 1
    set CN(ivory1) 1
    set CN(ivory2) 1
    set CN(ivory3) 1
    set CN(ivory4) 1
    set CN(honeydew1) 1
    set CN(honeydew2) 1
    set CN(honeydew3) 1
    set CN(honeydew4) 1
    set CN(LavenderBlush1) 1
    set CN(LavenderBlush2) 1
    set CN(LavenderBlush3) 1
    set CN(LavenderBlush4) 1
    set CN(MistyRose1) 1
    set CN(MistyRose2) 1
    set CN(MistyRose3) 1
    set CN(MistyRose4) 1
    set CN(azure1) 1
    set CN(azure2) 1
    set CN(azure3) 1
    set CN(azure4) 1
    set CN(SlateBlue1) 1
    set CN(SlateBlue2) 1
    set CN(SlateBlue3) 1
    set CN(SlateBlue4) 1
    set CN(RoyalBlue1) 1
    set CN(RoyalBlue2) 1
    set CN(RoyalBlue3) 1
    set CN(RoyalBlue4) 1
    set CN(blue1) 1
    set CN(blue2) 1
    set CN(blue3) 1
    set CN(blue4) 1
    set CN(DodgerBlue1) 1
    set CN(DodgerBlue2) 1
    set CN(DodgerBlue3) 1
    set CN(DodgerBlue4) 1
    set CN(SteelBlue1) 1
    set CN(SteelBlue2) 1
    set CN(SteelBlue3) 1
    set CN(SteelBlue4) 1
    set CN(DeepSkyBlue1) 1
    set CN(DeepSkyBlue2) 1
    set CN(DeepSkyBlue3) 1
    set CN(DeepSkyBlue4) 1
    set CN(SkyBlue1) 1
    set CN(SkyBlue2) 1
    set CN(SkyBlue3) 1
    set CN(SkyBlue4) 1
    set CN(LightSkyBlue1) 1
    set CN(LightSkyBlue2) 1
    set CN(LightSkyBlue3) 1
    set CN(LightSkyBlue4) 1
    set CN(SlateGray1) 1
    set CN(SlateGray2) 1
    set CN(SlateGray3) 1
    set CN(SlateGray4) 1
    set CN(LightSteelBlue1) 1
    set CN(LightSteelBlue2) 1
    set CN(LightSteelBlue3) 1
    set CN(LightSteelBlue4) 1
    set CN(LightBlue1) 1
    set CN(LightBlue2) 1
    set CN(LightBlue3) 1
    set CN(LightBlue4) 1
    set CN(LightCyan1) 1
    set CN(LightCyan2) 1
    set CN(LightCyan3) 1
    set CN(LightCyan4) 1
    set CN(PaleTurquoise1) 1
    set CN(PaleTurquoise2) 1
    set CN(PaleTurquoise3) 1
    set CN(PaleTurquoise4) 1
    set CN(CadetBlue1) 1
    set CN(CadetBlue2) 1
    set CN(CadetBlue3) 1
    set CN(CadetBlue4) 1
    set CN(turquoise1) 1
    set CN(turquoise2) 1
    set CN(turquoise3) 1
    set CN(turquoise4) 1
    set CN(cyan1) 1
    set CN(cyan2) 1
    set CN(cyan3) 1
    set CN(cyan4) 1
    set CN(DarkSlateGray1) 1
    set CN(DarkSlateGray2) 1
    set CN(DarkSlateGray3) 1
    set CN(DarkSlateGray4) 1
    set CN(aquamarine1) 1
    set CN(aquamarine2) 1
    set CN(aquamarine3) 1
    set CN(aquamarine4) 1
    set CN(DarkSeaGreen1) 1
    set CN(DarkSeaGreen2) 1
    set CN(DarkSeaGreen3) 1
    set CN(DarkSeaGreen4) 1
    set CN(SeaGreen1) 1
    set CN(SeaGreen2) 1
    set CN(SeaGreen3) 1
    set CN(SeaGreen4) 1
    set CN(PaleGreen1) 1
    set CN(PaleGreen2) 1
    set CN(PaleGreen3) 1
    set CN(PaleGreen4) 1
    set CN(SpringGreen1) 1
    set CN(SpringGreen2) 1
    set CN(SpringGreen3) 1
    set CN(SpringGreen4) 1
    set CN(green1) 1
    set CN(green2) 1
    set CN(green3) 1
    set CN(green4) 1
    set CN(chartreuse1) 1
    set CN(chartreuse2) 1
    set CN(chartreuse3) 1
    set CN(chartreuse4) 1
    set CN(OliveDrab1) 1
    set CN(OliveDrab2) 1
    set CN(OliveDrab3) 1
    set CN(OliveDrab4) 1
    set CN(DarkOliveGreen1) 1
    set CN(DarkOliveGreen2) 1
    set CN(DarkOliveGreen3) 1
    set CN(DarkOliveGreen4) 1
    set CN(khaki1) 1
    set CN(khaki2) 1
    set CN(khaki3) 1
    set CN(khaki4) 1
    set CN(LightGoldenrod1) 1
    set CN(LightGoldenrod2) 1
    set CN(LightGoldenrod3) 1
    set CN(LightGoldenrod4) 1
    set CN(LightYellow1) 1
    set CN(LightYellow2) 1
    set CN(LightYellow3) 1
    set CN(LightYellow4) 1
    set CN(yellow1) 1
    set CN(yellow2) 1
    set CN(yellow3) 1
    set CN(yellow4) 1
    set CN(gold1) 1
    set CN(gold2) 1
    set CN(gold3) 1
    set CN(gold4) 1
    set CN(goldenrod1) 1
    set CN(goldenrod2) 1
    set CN(goldenrod3) 1
    set CN(goldenrod4) 1
    set CN(DarkGoldenrod1) 1
    set CN(DarkGoldenrod2) 1
    set CN(DarkGoldenrod3) 1
    set CN(DarkGoldenrod4) 1
    set CN(RosyBrown1) 1
    set CN(RosyBrown2) 1
    set CN(RosyBrown3) 1
    set CN(RosyBrown4) 1
    set CN(IndianRed1) 1
    set CN(IndianRed2) 1
    set CN(IndianRed3) 1
    set CN(IndianRed4) 1
    set CN(sienna1) 1
    set CN(sienna2) 1
    set CN(sienna3) 1
    set CN(sienna4) 1
    set CN(burlywood1) 1
    set CN(burlywood2) 1
    set CN(burlywood3) 1
    set CN(burlywood4) 1
    set CN(wheat1) 1
    set CN(wheat2) 1
    set CN(wheat3) 1
    set CN(wheat4) 1
    set CN(tan1) 1
    set CN(tan2) 1
    set CN(tan3) 1
    set CN(tan4) 1
    set CN(chocolate1) 1
    set CN(chocolate2) 1
    set CN(chocolate3) 1
    set CN(chocolate4) 1
    set CN(firebrick1) 1
    set CN(firebrick2) 1
    set CN(firebrick3) 1
    set CN(firebrick4) 1
    set CN(brown1) 1
    set CN(brown2) 1
    set CN(brown3) 1
    set CN(brown4) 1
    set CN(salmon1) 1
    set CN(salmon2) 1
    set CN(salmon3) 1
    set CN(salmon4) 1
    set CN(LightSalmon1) 1
    set CN(LightSalmon2) 1
    set CN(LightSalmon3) 1
    set CN(LightSalmon4) 1
    set CN(orange1) 1
    set CN(orange2) 1
    set CN(orange3) 1
    set CN(orange4) 1
    set CN(DarkOrange1) 1
    set CN(DarkOrange2) 1
    set CN(DarkOrange3) 1
    set CN(DarkOrange4) 1
    set CN(coral1) 1
    set CN(coral2) 1
    set CN(coral3) 1
    set CN(coral4) 1
    set CN(tomato1) 1
    set CN(tomato2) 1
    set CN(tomato3) 1
    set CN(tomato4) 1
    set CN(OrangeRed1) 1
    set CN(OrangeRed2) 1
    set CN(OrangeRed3) 1
    set CN(OrangeRed4) 1
    set CN(red1) 1
    set CN(red2) 1
    set CN(red3) 1
    set CN(red4) 1
    set CN(DeepPink1) 1
    set CN(DeepPink2) 1
    set CN(DeepPink3) 1
    set CN(DeepPink4) 1
    set CN(HotPink1) 1
    set CN(HotPink2) 1
    set CN(HotPink3) 1
    set CN(HotPink4) 1
    set CN(pink1) 1
    set CN(pink2) 1
    set CN(pink3) 1
    set CN(pink4) 1
    set CN(LightPink1) 1
    set CN(LightPink2) 1
    set CN(LightPink3) 1
    set CN(LightPink4) 1
    set CN(PaleVioletRed1) 1
    set CN(PaleVioletRed2) 1
    set CN(PaleVioletRed3) 1
    set CN(PaleVioletRed4) 1
    set CN(maroon1) 1
    set CN(maroon2) 1
    set CN(maroon3) 1
    set CN(maroon4) 1
    set CN(VioletRed1) 1
    set CN(VioletRed2) 1
    set CN(VioletRed3) 1
    set CN(VioletRed4) 1
    set CN(magenta1) 1
    set CN(magenta2) 1
    set CN(magenta3) 1
    set CN(magenta4) 1
    set CN(orchid1) 1
    set CN(orchid2) 1
    set CN(orchid3) 1
    set CN(orchid4) 1
    set CN(plum1) 1
    set CN(plum2) 1
    set CN(plum3) 1
    set CN(plum4) 1
    set CN(MediumOrchid1) 1
    set CN(MediumOrchid2) 1
    set CN(MediumOrchid3) 1
    set CN(MediumOrchid4) 1
    set CN(DarkOrchid1) 1
    set CN(DarkOrchid2) 1
    set CN(DarkOrchid3) 1
    set CN(DarkOrchid4) 1
    set CN(purple1) 1
    set CN(purple2) 1
    set CN(purple3) 1
    set CN(purple4) 1
    set CN(MediumPurple1) 1
    set CN(MediumPurple2) 1
    set CN(MediumPurple3) 1
    set CN(MediumPurple4) 1
    set CN(thistle1) 1
    set CN(thistle2) 1
    set CN(thistle3) 1
    set CN(thistle4) 1
    set CN(gray0) 1
    set CN(grey0) 1
    set CN(gray1) 1
    set CN(grey1) 1
    set CN(gray2) 1
    set CN(grey2) 1
    set CN(gray3) 1
    set CN(grey3) 1
    set CN(gray4) 1
    set CN(grey4) 1
    set CN(gray5) 1
    set CN(grey5) 1
    set CN(gray6) 1
    set CN(grey6) 1
    set CN(gray7) 1
    set CN(grey7) 1
    set CN(gray8) 1
    set CN(grey8) 1
    set CN(gray9) 1
    set CN(grey9) 1
    set CN(gray10) 1
    set CN(grey10) 1
    set CN(gray11) 1
    set CN(grey11) 1
    set CN(gray12) 1
    set CN(grey12) 1
    set CN(gray13) 1
    set CN(grey13) 1
    set CN(gray14) 1
    set CN(grey14) 1
    set CN(gray15) 1
    set CN(grey15) 1
    set CN(gray16) 1
    set CN(grey16) 1
    set CN(gray17) 1
    set CN(grey17) 1
    set CN(gray18) 1
    set CN(grey18) 1
    set CN(gray19) 1
    set CN(grey19) 1
    set CN(gray20) 1
    set CN(grey20) 1
    set CN(gray21) 1
    set CN(grey21) 1
    set CN(gray22) 1
    set CN(grey22) 1
    set CN(gray23) 1
    set CN(grey23) 1
    set CN(gray24) 1
    set CN(grey24) 1
    set CN(gray25) 1
    set CN(grey25) 1
    set CN(gray26) 1
    set CN(grey26) 1
    set CN(gray27) 1
    set CN(grey27) 1
    set CN(gray28) 1
    set CN(grey28) 1
    set CN(gray29) 1
    set CN(grey29) 1
    set CN(gray30) 1
    set CN(grey30) 1
    set CN(gray31) 1
    set CN(grey31) 1
    set CN(gray32) 1
    set CN(grey32) 1
    set CN(gray33) 1
    set CN(grey33) 1
    set CN(gray34) 1
    set CN(grey34) 1
    set CN(gray35) 1
    set CN(grey35) 1
    set CN(gray36) 1
    set CN(grey36) 1
    set CN(gray37) 1
    set CN(grey37) 1
    set CN(gray38) 1
    set CN(grey38) 1
    set CN(gray39) 1
    set CN(grey39) 1
    set CN(gray40) 1
    set CN(grey40) 1
    set CN(gray41) 1
    set CN(grey41) 1
    set CN(gray42) 1
    set CN(grey42) 1
    set CN(gray43) 1
    set CN(grey43) 1
    set CN(gray44) 1
    set CN(grey44) 1
    set CN(gray45) 1
    set CN(grey45) 1
    set CN(gray46) 1
    set CN(grey46) 1
    set CN(gray47) 1
    set CN(grey47) 1
    set CN(gray48) 1
    set CN(grey48) 1
    set CN(gray49) 1
    set CN(grey49) 1
    set CN(gray50) 1
    set CN(grey50) 1
    set CN(gray51) 1
    set CN(grey51) 1
    set CN(gray52) 1
    set CN(grey52) 1
    set CN(gray53) 1
    set CN(grey53) 1
    set CN(gray54) 1
    set CN(grey54) 1
    set CN(gray55) 1
    set CN(grey55) 1
    set CN(gray56) 1
    set CN(grey56) 1
    set CN(gray57) 1
    set CN(grey57) 1
    set CN(gray58) 1
    set CN(grey58) 1
    set CN(gray59) 1
    set CN(grey59) 1
    set CN(gray60) 1
    set CN(grey60) 1
    set CN(gray61) 1
    set CN(grey61) 1
    set CN(gray62) 1
    set CN(grey62) 1
    set CN(gray63) 1
    set CN(grey63) 1
    set CN(gray64) 1
    set CN(grey64) 1
    set CN(gray65) 1
    set CN(grey65) 1
    set CN(gray66) 1
    set CN(grey66) 1
    set CN(gray67) 1
    set CN(grey67) 1
    set CN(gray68) 1
    set CN(grey68) 1
    set CN(gray69) 1
    set CN(grey69) 1
    set CN(gray70) 1
    set CN(grey70) 1
    set CN(gray71) 1
    set CN(grey71) 1
    set CN(gray72) 1
    set CN(grey72) 1
    set CN(gray73) 1
    set CN(grey73) 1
    set CN(gray74) 1
    set CN(grey74) 1
    set CN(gray75) 1
    set CN(grey75) 1
    set CN(gray76) 1
    set CN(grey76) 1
    set CN(gray77) 1
    set CN(grey77) 1
    set CN(gray78) 1
    set CN(grey78) 1
    set CN(gray79) 1
    set CN(grey79) 1
    set CN(gray80) 1
    set CN(grey80) 1
    set CN(gray81) 1
    set CN(grey81) 1
    set CN(gray82) 1
    set CN(grey82) 1
    set CN(gray83) 1
    set CN(grey83) 1
    set CN(gray84) 1
    set CN(grey84) 1
    set CN(gray85) 1
    set CN(grey85) 1
    set CN(gray86) 1
    set CN(grey86) 1
    set CN(gray87) 1
    set CN(grey87) 1
    set CN(gray88) 1
    set CN(grey88) 1
    set CN(gray89) 1
    set CN(grey89) 1
    set CN(gray90) 1
    set CN(grey90) 1
    set CN(gray91) 1
    set CN(grey91) 1
    set CN(gray92) 1
    set CN(grey92) 1
    set CN(gray93) 1
    set CN(grey93) 1
    set CN(gray94) 1
    set CN(grey94) 1
    set CN(gray95) 1
    set CN(grey95) 1
    set CN(gray96) 1
    set CN(grey96) 1
    set CN(gray97) 1
    set CN(grey97) 1
    set CN(gray98) 1
    set CN(grey98) 1
    set CN(gray99) 1
    set CN(grey99) 1
    set CN(gray100) 1
    set CN(grey100) 1

    proc IsColorNameQ {s} {
	variable CN
	if {[info exists CN($s)]} {
	    return 1;
	} else {
	    return 0;
	}
    }

    proc IsColorNumericQ {s} {
	return [regexp \
		    {^\#([[:xdigit:]]{6}|[[:xdigit:]]{3}|[[:xdigit:]]{9}|[[:xdigit:]]{12})$} $s]
    }
		
    proc IsColorSpecQ {s} {
	if {[IsColorNameQ $s]} {
	    return 1;
	} elseif {[IsColorNumericQ $s]} {
	    return 1;
	} else {
	    return 0;
	}
    }
}
#End of namespace validcolor

#Ths procedure takes as its sole argument the name
#of an existing procedure in the same interpreter
#and creates a new master level procedure that
#wraps the other procedure with a test for the
#correct argument count.

proc WrapMaster {pn} {
    regsub "^::fontsel::" $pn "" Stem
    set WrappedName Wrapped${Stem};
    set sp [format "proc $WrappedName \{args\} \{\n"];
    set ArgsNeeded [llength [info args $pn]];
    append sp [format "\tif \{\[llength \$args\] != %d\} \{\n" $ArgsNeeded]
    if {$ArgsNeeded == 1} {
	set emsg [format [_ "The command %s in the file %%s expects one argument"] \
	[SlaveName $WrappedName]]
	set cmd [format "\t\tputs \"$emsg\"\n" \$::InitFile]
	append sp $cmd
    } else {
	set emsg [format [_ "The command %s in the file %%s expects %d arguments\n"] \
	      [SlaveName $WrappedName] $ArgsNeeded]
	set cmd [format "\t\tputs \"$emsg\"\n" \$::InitFile]; 
	append sp $cmd;
    }
    append sp "\t\treturn;\n\t\}\n\t"
    append sp  $pn
    for {set i 0} {$i < $ArgsNeeded} {incr i} {
	append sp " \[lindex \$args $i\]"
    }
    append sp "\n\}"
    eval $sp;
    return $WrappedName;
}

proc ReadInitFile {name} {
    if { [catch {interp invoke init source $name} msg] } {
	puts [format [_  "Error on reading init file %s: %s"] $name $msg]
    } else {
	ShowMessage [format [_ "Executed init file %s"] $name]
    }
    interp delete init
}

proc SlavePuts {msg} {
    puts $msg;
}

proc SlaveMsgcat {} {
    package require msgcat
    ::msgcat::mcload [file join [file dirname [info script]] msgs];
}

#This is the list of the master interpreter commands
#other than those generated from data structures (currently
#color- and font-setting) that we wish to expose for use in
#the init file slave interpreter.
set MiscellaneousSlaveCommandList {
    SelectBrowser
    SetUseCheckbuttonP
    SetRecordParseDisplayedP
    SetMiscOptionsDisplayedP
    SetUnicodeDisplayedP
    SetAlgorithmDisplayedP
    SetInputFileDisplayedP
    SetOutputFileDisplayedP
    SetExclusionsDisplayedP
    SetKeyFieldIdentificationDisplayedP
    SetMiscellaneousOptionsDisplayedP
    SetSortOrderDisplayedP
    SetSortTypeDisplayedP
    SetSubstitutionsDisplayedP
    SetBMPOnlyP
    SetBalloonHelpShowP
    SetDisplayConsonantChartColumnLabelsP
    SetDisplayConsonantChartRowLabelsP
    SetDisplayVowelChartColumnLabelsP
    SetDisplayVowelChartRowLabelsP
    SetEOLCarriageReturnP
    SetFieldSeparators
    SetInvertOrderGloballyP
    SetRecordParse
    SetRecordSeparator
    SetReservePrivateUseAreasP
    SetSortAlgorithm
    SetWholeRecordIsKeyP
    SlavePuts
    StoreCustomCharacterChartFileName
    StoreCustomCharacterChartInPlace
}

#This procedure creates the master procedures that will be exposed
#in the slave interpreter. It takes care of creating the base procedures
#where, as in the case of color and font setting, these are created
#programmatically, then wraps these as well as the manually
#written procedures that are to be exposed.
proc CreateInitProcs {} {
    set ::FontProcList "";
    fontsel::DefineFontSettingProcs
    DefineColorSettingProcs;
    #This is the complete list of commands that we want to expose 
    #in the slave init file interpreter. These are the raw, unwrapped, master
    #level commands.
    set ::MasterCommandsToBeExposedInInitList [concat \
	   [lsort $::ColorProcList] \
	   [lsort $::FontProcList]  \
	   [lsort $::MiscellaneousSlaveCommandList]];
    #Now we wrap them all so as to trap valence errors
    set ::WrappedMasterCommandList [list]
    foreach c $::MasterCommandsToBeExposedInInitList {
	lappend ::WrappedMasterCommandList [WrapMaster $c]
    }
}

#This procedure takes the list of wrapped master
#procedures and exposes them in a specified, already
#created, slave interpreter.
proc ExposeInitProcs {Interpreter} {
    foreach c $::WrappedMasterCommandList {
	set sl [SlaveName $c];
	$Interpreter alias $sl $c;
    }
}

set MasterToSlaveName(SetDisplayConsonantChartColumnLabelsP) ShowConsonantChartColumnLabelsP
set MasterToSlaveName(SetDisplayConsonantChartRowLabelsP) ShowConsonantChartRowLabelsP
set MasterToSlaveName(SetDisplayVowelChartColumnLabelsP) ShowVowelChartColumnLabelsP
set MasterToSlaveName(SetDisplayVowelChartRowLabelsP) ShowVowelChartRowLabelsP
set MasterToSlaveName(SelectBrowser) Browser;
set MasterToSlaveName(SlavePuts) Puts;
set MasterToSlaveName(StoreCustomCharacterChartInPlace) DefineCharacterChart 
set MasterToSlaveName(StoreCustomCharacterChartFileName) ReadCharacterChart 

#Given the name of a wrapped command in the master interpreter,
#returns the name that the command is to have when
#exposed in the slave interpreter. In most cases the alias
#is computed from the master name, but in some cases
#the relationship is irregular. 
proc SlaveName {command} {
    regsub "^Wrapped" $command "" Unwrapped
    if {[info exists ::MasterToSlaveName($Unwrapped)]} {
	set EnglishSlaveName $::MasterToSlaveName($Unwrapped);
    } else {
	regsub "^Set" $Unwrapped "" EnglishSlaveName
    }
    return [_ $EnglishSlaveName]
}

proc InitFileSetup {} {
    interp create -safe -- init
    #We use slaveMsgcat rather than SlaveMsgcat so that the name does not
    #appear in the list that the user sees.
    init alias slaveMsgcat SlaveMsgcat
    init eval slaveMsgcat
    init alias mc ::msgcat::mc
    ExposeInitProcs init
    #This takes care of errors in the name of commands while still in the
    #slave interpreter, without aborting, and using the user-name in the error-message.
    init eval {
	proc unknown {args} {
	    set cmd [lindex $args 0]
	    Puts [format  [mc "Unknown command %s in script file %s."]  $cmd [info script]];
#	    Puts [format  [mc "Unknown command %1\$s in script file %2\$s."]  $cmd [info script]];
	    return ;
	}
    }
}

proc SlaveBoolean {s} {
    switch -regexp $s {
	1	{return 1}
	T.*	{return 1}
	t.*	{return 1}
	Y.*	{return 1}
	y.*	{return 1}
	ok	{return 1}
	on	{return 1}
	0 	{return 0}
	F.*	{return 0}
	f.*	{return 0}
	N.*	{return 0}
	n.*	{return 0}
	off	{return 0}
	default {
	    puts [format [msgcat::mc "%s is not a Boolean argument."] $s]
	    return -1; 
	}
    }
}

#Write out a list of init file commands that will reconstruct
#the current configuration 
proc SaveState {{SaveFile Ask}} {
    if {[string equal $SaveFile Ask]} { 
	set SaveFile [tk_getSaveFile -initialfile [_ "SavedState"]];
	if {$SaveFile == ""} {
	    ShowMessage [_ "File selection cancelled."]
	    return ;
	}
    }
    if {[catch {open $SaveFile "w"} fh ] != 0} {
	ShowMessage [format [_ "Unable to open file %s."] $SaveFile];
	return ;
    }
    set cl [concat \
	[list [format "\#%s" [clock format [clock seconds]]]] \
	[list [format "\#%s" [_ Fonts]]] \
	[fontsel::SaveFontSettings] \
	[list [format "\#%s" [_ Colors]]] \
	[SaveColorSettings] \
	[list [format "\#%s" [_ Miscellaneous]]] \
	[SaveMiscellaneousSettings]];
    foreach x $cl {
	puts $fh $x;
    }
    close $fh;
}

proc Usage {} {
    puts {Usage: msg [options]};
    puts [_ "    -d            set debug flag"];
    puts [_ "    -h            help"];
    puts [_ "    -i            do not read init file"];
    puts [_ "    -v            version"];
}

proc argshift {k} {
    set ::argv [lrange $::argv $k end];
}

set Locale [::msgcat::mclocale];
LoadMessageCatalog;

#Handle command line arguments
while {[string match -* [lindex $argv 0]]} {
    switch -glob -- [lindex $argv 0] {
	-d* {
	    set DebugP 1;
	    argshift 1;
	}
	-h* {
	    Usage;
	    exit 0;
	}
	-i* {
	    set ReadInitFileP 0;
	    argshift 1;
	}
	-v* {
	    puts "msg $Version";
 	    puts "msort package version $PackageVersion";
	    exit 0;
	}
	default {
	    puts [format [::msgcat::mc "Command line option %s not recognized."] [lindex $argv 0]];
	    Usage;
	    exit 0;
	}
    }
};

proc PopupSelectionHandler {w} {
    clipboard append [$w get sel.first sel.last];
}

puts "msg $Version";
puts "Copyright (C) 2005, 2006 William J. Poser.";
puts [_ "This program is free software; you can redistribute it
and/or modify it under the terms of version 2 of the GNU
General Public License as published by the Free Software Foundation.\n"];
wm withdraw [fontsel::CreateFontControlPanel]
CreateInitProcs;
if {$ReadInitFileP} {
    InitFileSetup;
    #Look first in current directory, then in home directory.
    if { [file exists $InitFile] } {
	ReadInitFile $InitFile;
    } else {
	set cwd [pwd];
	cd;
	if {[file exists $InitFile] } {
	    ReadInitFile $InitFile;
	}
	cd $cwd;
    }
}
fontsel::SetFontSelectionDefaults
fontsel::RecordDefaults

option add *Background $ColorSpecs(Default,Background) 60;

option add *ipaec.Button.font CharacterEntryFont 100
option add *ipaec.Label.background  $ColorSpecs(IPAEntry,Background)
option add *ipaec.Button.background $ColorSpecs(IPAEntry,Background)
option add *ipaec.Button.foreground $ColorSpecs(IPAEntry,Foreground)
option add *ipaec.Label.relief raised

option add *ipaev.Button.font CharacterEntryFont 100
option add *ipaev.Label.background  $ColorSpecs(IPAEntry,Background)
option add *ipaev.Button.background $ColorSpecs(IPAEntry,Background)
option add *ipaev.Button.foreground $ColorSpecs(IPAEntry,Foreground)
option add *ipaev.Label.relief raised

option add *ipaea.Button.font AccentedLetterFont 100
option add *ipaea.Label.background  "\#E0E0E0"
option add *ipaea.Button.background "\#FFFFFF"
option add *ipaea.Button.foreground "\#000000"
option add *ipaea.Label.relief raised

option add *ipaed.Button.font DiacriticFont 100
option add *ipaed.Button.background "\#E0E0FF"
option add *ipaed.Label.background "\#E0E0E0"
option add *ipaed.Button.foreground "\#000000"
option add *ipaed.Label.relief raised

option add *Entry.Background  $::ColorSpecs(Entry,Background) 100; 
option add *DisabledBackground "\#949CD2" 60;
option add *Button.activeBackground "\#FF0044" 60;
option add *Radiobutton.activeBackground $::ColorSpecs(RadioButton,ActiveBackground) 100;
option add *Radiobutton.selectColor  $::ColorSpecs(RadioButton,SelectedBackground) 100;

option add *Menu.Background $ColorSpecs(Menu,Background) 100
option add *Menu.Foreground $ColorSpecs(Menu,Foreground) 100
option add *Menu*selectColor $ColorSpecs(Menu,Select) 100
option add *Menu.activeBackground $ColorSpecs(Menu,ActiveBackground) 100
option add *Menu.activeForeground $ColorSpecs(Menu,ActiveForeground) 100

option add *Balloonhelp*background $ColorSpecs(BalloonHelp,Background) 100
option add *Balloonhelp*foreground $ColorSpecs(BalloonHelp,Foreground) 100
option add *Balloonhelpinfo.wrapLength 3i  100
option add *Balloonhelp.info.justify left 100

#Set up balloon help
toplevel .balloonhelp -class Balloonhelp -borderwidth 1 -relief flat
label .balloonhelp.info -font BalloonHelpFont;
pack .balloonhelp.info -side left -fill y
wm overrideredirect .balloonhelp 1
wm withdraw .balloonhelp

#Construct the user interface
set m [menu .menubar -tearoff 0 \
	   -bg $ColorSpecs(Menubar,Background) \
	   -fg $ColorSpecs(Menubar,Foreground) \
	   -activebackground $ColorSpecs(Menubar,ActiveBackground) \
	   -activeforeground $ColorSpecs(Menubar,Background)]

$m add cascade -label [_ "File"]   -menu [menu $m.file]
$m.file add command -label [_ "Save Command Line"] -command SaveCommandLine
$m.file add command -label [_ "Show Command Line"] -command ShowCommandLine
$m.file add command -label [_ "Quit"]  -command ShutDown
if {!$AquaP} {
    $m add command -label [_ "Sort"]  -command ExecuteMsort
    if {[string equal $tcl_platform(platform) "unix"]} {
	$m add command -label [_ "Abort Sort"]  -command AbortSort
    }
    $m add command -label [_ "Add Key"]  -command AddKey 
    $m add command -label [_ "Remove Key"]  -command RemoveKey
}
$m add cascade -label [_ "Character Entry"]   -menu [menu $m.charentry];
set ind 0;
incr ind;
set ::CustomCharacterChartIndex $ind;
$m.charentry add command -label \
    [_ "Load Custom Character Chart Definition"] -command ReadCustomCharacterChartPopup;

incr ind;
set ::ToggleIPAAIndex $ind;
$m.charentry add command -label \
    [_ "Display Accented Letter Chart"] -command ToggleIPAA;

incr ind;
set ::ToggleIPACIndex $ind;
$m.charentry add command -label \
    [_ "Display IPA Consonants"] -command ToggleIPAC;

incr ind;
set ::ToggleIPAVIndex $ind;
$m.charentry add command -label \
    [_ "Display IPA Vowel Chart"] -command ToggleIPAV;

incr ind;
set ::ToggleIPADIndex $ind;
$m.charentry add command -label \
    [_ "Display IPA Diacritics"] -command ToggleIPAD;

incr ind;
set ::ToggleCharEntryByCodeIndex $ind;
$m.charentry add command -label \
    [_ "Display Widget for Entering Characters by Unicode Code"] -command ToggleCharEntryByCode;

$m add cascade -label [_ "Configure"]   -menu [menu $m.configure];
set ind 1;
destroy $m.configure.charentry
$m.configure add cascade -label [_ "Character Entry"] -menu [menu $m.configure.charentry]
incr ind;
$m.configure.charentry delete 1 4
$m.configure.charentry add checkbutton -label [_ "Display Consonant Chart Column Labels"] \
    -variable DisplayConsonantChartColumnLabelsP -onvalue 1 -offvalue 0 \
    -command ControlDisplayConsonantChartColumnLabels
$m.configure.charentry add checkbutton -label [_ "Display Consonant Chart Row Labels"] \
    -variable DisplayConsonantChartRowLabelsP -onvalue 1 -offvalue 0 \
    -command ControlDisplayConsonantChartRowLabels
$m.configure.charentry add checkbutton -label [_ "Display Vowel Chart Column Labels"] \
    -variable DisplayVowelChartColumnLabelsP -onvalue 1 -offvalue 0 \
    -command ControlDisplayVowelChartColumnLabels
$m.configure.charentry add checkbutton -label [_ "Display Vowel Chart Row Labels"] \
    -variable DisplayVowelChartRowLabelsP -onvalue 1 -offvalue 0 \
    -command ControlDisplayVowelChartRowLabels
incr ind;
$m.configure add command -label [_ "Save Configuration"] -command {SaveState};
incr ind;
$m.configure add command -label [_ "Select Font"] -command fontsel::CreateFontControlPanel;

incr ind;
set BalloonHelpIndex $ind;
if {$::BalloonHelpP} {
    set lab [_ "Disable Irritating Balloon Help"] 
} else {
    set lab [_ "Enable Balloon Help"]
}
$m.configure add command -label $lab -command ToggleBalloonHelp;


$m add cascade -label [_ "Help"]   -menu [menu $m.help];
set ind 0;
$m.help add command -label [_ "About"] -command About;
incr ind;
$m.help add command -label [_ "Bug Reports"] -command BugReports;
incr ind;
$m.help add command -label [_ "How To Use This Program"] -command HowTo;
incr ind;
$m.help add command -label [_ "Key Bindings"] -command DescribeKeyBindings;
incr ind;
$m.help add cascade -label [_ "Initialization File Commands"]  -menu [menu $m.help.ifc];
$m.help.ifc add command -label [_ "Popup List"] -command PopupInitCommandList;
$m.help.ifc add command -label [_ "Save to File"] -command SaveInitCommands;
incr ind;
$m.help add command -label [_ "License"] -command ShowGPL;
incr ind;
$m.help add command -label [_ "Regular Expression Sytax"] -command {ShowWebPage http://laurikari.net/tre/syntax.html}
incr ind;
. configure -menu .menubar

if {$AquaP} {
    frame .cmnd
    button .cmnd.sort        -text [_ "Sort"]       -anchor w -command ExecuteMsort \
	-bg $::ColorSpecs(Menubar,Background) -fg $::ColorSpecs(Menubar,Foreground);
    button .cmnd.abort       -text [_ "Abort Sort"] -anchor w -command AbortSort \
	-bg $::ColorSpecs(Menubar,Background) -fg $::ColorSpecs(Menubar,Foreground);
    button .cmnd.addkey      -text [_ "Add Key"]    -anchor w -command AddKey \
	-bg $::ColorSpecs(Menubar,Background) -fg $::ColorSpecs(Menubar,Foreground);
    button .cmnd.removekey   -text [_ "Remove Key"] -anchor w -command RemoveKey \
	-bg $::ColorSpecs(Menubar,Background) -fg $::ColorSpecs(Menubar,Foreground);
    pack .cmnd.sort      -side left -expand 1 -fill both
    pack .cmnd.abort     -side left -expand 1 -fill both
    pack .cmnd.addkey    -side left -expand 1 -fill both
    pack .cmnd.removekey -side left -expand 1 -fill both
    pack .cmnd -side top -expand y -fill x
    frame .cmndsep6  -height 4 -borderwidth 1 -relief raised
    pack  .cmndsep6 -side top -expand 1 -fill x
}

frame .top -bg $::ColorSpecs(Backdrop,Background) -border 2 -relief ridge
text .top.msg -bg $ColorSpecs(Messages,Background) \
    -fg $ColorSpecs(Messages,Foreground)  -height 1 -relief flat \
    -font MainFont -exportselection 1
balloonhelp_for .top.msg [_ "Messages from the program appear here."]
bind .top.msg <ButtonRelease-1> {PopupSelectionHandler %W}

frame .top.sel -border 1 -relief solid
radiobutton .top.sel.gen -text [_ "General"] -variable WhichKey -value 0\
    -bg $ColorSpecs(PageChoice,Background) \
    -activebackground $::ColorSpecs(PageChoice,ActiveBackground)\
    -selectcolor $::ColorSpecs(PageChoice,Selected) -indicatoron 0 \
    -height $KeySelHeight -width [expr $KeySelWidth + 2]
pack .top.sel.gen -side left
pack .top.msg -side top -expand 1 -fill x -padx 6;
pack .top.sel -side top -expand 1 -fill x -padx 6
frame .tsep  -height 4 -borderwidth 1 -relief raised
pack .top  -side top -expand 1 -fill x -pady 0
pack .tsep -side top -expand 0 -fill x -pady 0
balloonhelp_for .top.sel [_ "Use these buttons to switch between general\n\
and key-specific settings and among settings\nfor different keys."]

balloonhelp_for .top.sel.gen [_ "Click on this button to switch to\n\
the non-key-specific settings."];

if {[MsortAvailableP]} {
    set ::MsortVersion [FindoutMsortVersion];
    ShowMessage [format [_  "Msort version %s"] $::MsortVersion];
} else {
    ShowMessage [_ "Msort is not available on this machine or is not in your path."]
    if {!$DebugP} {
	after 3500 {
	    ShowMessage [_ "Exiting."];
	    after 3500 {
		exit 1
	    }
	}
    }
}
wm title . [format "msg %s    msort %s" $Version $::MsortVersion];
set genrelief solid
set genborder 1
frame .gen -bg $::ColorSpecs(Backdrop,Background);
frame .gen.ctr -bg $::ColorSpecs(Default,Background);

if {$UseCheckbuttonP} {
    frame  .gen.ctr.parf
    label  .gen.ctr.parf.lab  -text [_ "Record Parse Options"]   -anchor w -font HeaderFont
    checkbutton .gen.ctr.parf.ckb -variable RecordParseDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen
    pack .gen.ctr.parf.lab -side left -expand 0 -fill y -anchor w
    pack .gen.ctr.parf.ckb -side right -expand 1 -fill y -anchor e
} else {
    frame .gen.ctr.parf  -relief $grelief -border $gborder
    checkbutton .gen.ctr.parf.ckb -variable RecordParseDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen \
	-text [_ "Record Parse Options"] -anchor w -font HeaderFont -indicatoron 0 \
	-activeforeground red 	-activebackground navajowhite
    pack .gen.ctr.parf.ckb -side right -expand 1 -fill both -anchor e
}

checkbutton .gen.ctr.parckb -variable RecordParseDisplayedP \
    -onvalue 1 -offvalue 0 -command PackGen -text [_ "roll up"] \
    -activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
    -activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)

#How to parse input into records
frame       .gen.ctr.rp -relief $genrelief -border $genborder
label       .gen.ctr.rp.lab  -text [_ "How to Parse Into Records"]  -font HeaderFont
radiobutton .gen.ctr.rp.rdb1 -variable RecordParse -value 0 -text [_ "line"]
radiobutton .gen.ctr.rp.rdb2 -variable RecordParse -value 1 -text [_ "block"]
radiobutton .gen.ctr.rp.rdb3 -variable RecordParse -value 2 -text [_ "separator"]
frame       .gen.ctr.rp.fl		
radiobutton .gen.ctr.rp.fl.rdb4 -variable RecordParse -value 3 -text [_ "fixed length (bytes)"]
iwidgets::spinint .gen.ctr.rp.fl.len\
    -labelpos e \
    -width 3 \
    -range {0 200} \
    -wrap 0 \
    -labeltext ""\
    -textbackground $::ColorSpecs(Entry,Background)\
    -increment {.gen.ctr.rp.fl.len up} \
    -decrement {.gen.ctr.rp.fl.len down} \
    -invalid {};# Prevents irritating flashing or ringing of bell
.gen.ctr.rp.fl.len clear
.gen.ctr.rp.fl.len insert 0 $::FixedRecordLength; 
.gen.ctr.rp.fl.len configure -state disabled
pack        .gen.ctr.rp.fl.rdb4 -side left -expand 1 -fill both -anchor w
pack        .gen.ctr.rp.fl.len -side left -expand 1 -fill both -anchor w -padx 3
.gen.ctr.rp.rdb2 select
label       .gen.ctr.rp.sclab  -text [_ "character:"] 
entry       .gen.ctr.rp.cent -width 2 -state disabled -relief flat \
    -font UserTechnicalTextFont \
    -foreground $::ColorSpecs(UserText,Foreground) \
    -disabledbackground $::ColorSpecs(Entry,DisabledBackground)
bind 	    .gen.ctr.rp.cent <FocusIn> {SetInsertionTargets .gen.ctr.rp.cent}
bind 	    .gen.ctr.rp.cent  <Control-v> {InsertClipboardContents %W}
pack .gen.ctr.rp.lab -side top  -anchor w -padx 4
pack .gen.ctr.rp.rdb2 -side top   -anchor w  -padx 15 -pady {5 3}
pack .gen.ctr.rp.rdb1 -side top    -anchor w -padx 15 -pady {3 3}
pack .gen.ctr.rp.fl   -side bottom    -anchor w -padx 15 -pady {3 5}
pack .gen.ctr.rp.rdb3 -side left   -anchor w -padx {15 2} -pady {3 7}
pack .gen.ctr.rp.sclab -side left -anchor w -pady {3 7}
pack .gen.ctr.rp.cent -side left -padx 3 -anchor w -pady {3 7}
bind .gen.ctr.rp <<B3>> ExplainHowToParseIntoRecords;
bind .gen.ctr.rp.lab   <<B3>> ExplainHowToParseIntoRecords;
bind .gen.ctr.rp.rdb1  <<B3>> ExplainLine;
bind .gen.ctr.rp.rdb2  <<B3>> ExplainBlock
bind .gen.ctr.rp.rdb3  <<B3>> ExplainRecordSeparatorCharacter
bind .gen.ctr.rp.sclab <<B3>> ExplainRecordSeparatorCharacter
bind .gen.ctr.rp.cent  <<B3>> ExplainRecordSeparatorCharacter
balloonhelp_for .gen.ctr.rp.fl.len [_ "Enter the length of a record in bytes."];
balloonhelp_for .gen.ctr.rp.cent [_ "Enter the character that separates one record from another here."];
#Parsing into keys
frame       .gen.ctr.wr -relief $genrelief -border $genborder
label       .gen.ctr.wr.lab  -text [_ "Use Entire Record as Key?"] -anchor nw -font HeaderFont
radiobutton .gen.ctr.wr.rdb1 -variable WholeRecordIsKeyP -value 0 -text [_ "no"]
radiobutton .gen.ctr.wr.rdb2 -variable WholeRecordIsKeyP -value 1 -text [_ "yes"]
.gen.ctr.wr.rdb1 select
label       .gen.ctr.wr.fslab  -text [_ "    field separators:"]
entry       .gen.ctr.wr.fsent -width 6 -relief flat -font UserTechnicalTextFont\
    -disabledbackground $::ColorSpecs(Entry,DisabledBackground)
.gen.ctr.wr.fsent insert insert {\n}
bind 	    .gen.ctr.wr.fsent <Control-v> {InsertClipboardContents %W}
bind 	    .gen.ctr.wr.fsent <FocusIn> {SetInsertionTargets .gen.ctr.wr.fsent}
pack .gen.ctr.wr.lab -side top -expand y -fill y  -anchor w -padx 4 -pady {5 3}
pack .gen.ctr.wr.rdb2 -side top  -anchor w -padx 15  -pady 3
pack .gen.ctr.wr.rdb1 -side left  -anchor w  -padx 15   -pady 3
pack .gen.ctr.wr.fslab -side left -padx {15 2} -pady {3 7}
pack .gen.ctr.wr.fsent -side left -padx {4 15} -anchor w -pady {3 7}

bind .gen.ctr.wr      <<B3>> ExplainEntireRecordAsKey;
bind .gen.ctr.wr.lab  <<B3>> ExplainEntireRecordAsKey;
bind .gen.ctr.wr.rdb1 <<B3>> ExplainEntireRecordAsKey;
bind .gen.ctr.wr.rdb2 <<B3>> ExplainEntireRecordAsKey;
bind .gen.ctr.wr.fslab <<B3>> ExplainFieldSeparatorCharacter
bind .gen.ctr.wr.fsent <<B3>> ExplainFieldSeparatorCharacter
balloonhelp_for .gen.ctr.wr.fsent [_ "Enter the characters that separate one field from another other here."];

if {$UseCheckbuttonP} {
    frame  .gen.ctr.miscf
    label  .gen.ctr.miscf.lab  -text [_ "Miscellaneous Options"]   -anchor w -font HeaderFont
    checkbutton .gen.ctr.miscf.ckb -variable MiscOptionsDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen
    pack .gen.ctr.miscf.lab -side left -expand 0 -fill y -anchor w
    pack .gen.ctr.miscf.ckb -side right -expand 1 -fill y -anchor e
} else {
    frame .gen.ctr.miscf  -relief $grelief -border $gborder
    checkbutton .gen.ctr.miscf.ckb -variable MiscOptionsDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen \
	-text [_ "Miscellaneous Options"] -anchor w -font HeaderFont -indicatoron 0 \
	-activeforeground red 	-activebackground navajowhite
    pack .gen.ctr.miscf.ckb -side right -expand 1 -fill both -anchor e
}

checkbutton .gen.ctr.miscckb -variable MiscOptionsDisplayedP \
    -onvalue 1 -offvalue 0 -command PackGen -text [_ "roll up"] \
    -activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
    -activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)

#Invert order globally?
frame       .gen.ctr.gi -relief $genrelief -border $genborder
label       .gen.ctr.gi.lab  -text [_ "Invert Order Globally?"]   -anchor w -font HeaderFont
radiobutton .gen.ctr.gi.rdb1 -variable GlobalInversionP -value 0 -text [_ "no"]     -anchor w
radiobutton .gen.ctr.gi.rdb2 -variable GlobalInversionP -value 1 -text [_ "yes"]    -anchor w
.gen.ctr.gi.rdb1 select
pack .gen.ctr.gi.lab -side top -expand y -fill x -anchor w -padx 4
pack .gen.ctr.gi.rdb1 -expand 1 -fill y  -anchor w -padx 15 -pady {5 3}
pack .gen.ctr.gi.rdb2 -expand 1 -fill y  -anchor w -padx 15 -pady {3 7}
bind .gen.ctr.gi       <<B3>> ExplainInvertOrderGlobally;
bind .gen.ctr.gi.lab   <<B3>> ExplainInvertOrderGlobally;
bind .gen.ctr.gi.rdb1  <<B3>> ExplainInvertOrderGlobally;
bind .gen.ctr.gi.rdb2  <<B3>> ExplainInvertOrderGlobally;

#End of line marker in input
frame       .gen.ctr.eol -relief $genrelief -border $genborder
label       .gen.ctr.eol.lab  -text [_ "End of line marker in input"]  -anchor w -font HeaderFont
radiobutton .gen.ctr.eol.rdb1 -variable EOLCarriageReturnP -value 0 -anchor w \
    -text [_ "Line Feed (0x0A = \\012)"]
radiobutton .gen.ctr.eol.rdb2 -variable EOLCarriageReturnP -value 1 -anchor w \
    -text [_ "Carriage Return (0x0D = \\015)"]
.gen.ctr.eol.rdb1 select
pack .gen.ctr.eol.lab -side top -expand y -fill x -anchor w  -padx 4
pack .gen.ctr.eol.rdb1 -expand 1 -fill y  -anchor w -padx 15 -pady {5 3}
pack .gen.ctr.eol.rdb2 -expand 1 -fill y  -anchor w -padx 15 -pady {3 7}
bind .gen.ctr.eol       <<B3>> ExplainEndOfLineMarker
bind .gen.ctr.eol.lab   <<B3>> ExplainEndOfLineMarker
bind .gen.ctr.eol.rdb1  <<B3>> ExplainEndOfLineMarker
bind .gen.ctr.eol.rdb2  <<B3>> ExplainEndOfLineMarker

checkbutton .gen.ctr.unickb -variable UnicodeDisplayedP \
    -onvalue 1 -offvalue 0 -command PackGen -text [_ "roll up"] \
    -activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
    -activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)

if {$UseCheckbuttonP} {
    frame  .gen.ctr.unif
    label  .gen.ctr.unif.lab  -text [_ "Unicode Options"]   -anchor w -font HeaderFont
    checkbutton .gen.ctr.unif.ckb -variable UnicodeDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen
    pack .gen.ctr.unif.lab -side left -expand 0 -fill y -anchor w
    pack .gen.ctr.unif.ckb -side right -expand 1 -fill y -anchor e
} else {
    frame .gen.ctr.unif  -relief $grelief -border $gborder
    checkbutton .gen.ctr.unif.ckb -variable UnicodeDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen \
	-text [_ "Unicode Options"] -anchor w -font HeaderFont -indicatoron 0 \
	-activeforeground red 	-activebackground navajowhite
    pack .gen.ctr.unif.ckb -side right -expand 1 -fill both -anchor e
}

#Input restricted to BMP?
frame       .gen.ctr.bmp -relief $genrelief -border $genborder
label       .gen.ctr.bmp.lab  -text [_ "Input Limited to BMP?"]   -anchor w -font HeaderFont
radiobutton .gen.ctr.bmp.rdb1 -variable BMPOnlyP -value 0 -text [_ "no"]     -anchor w
radiobutton .gen.ctr.bmp.rdb2 -variable BMPOnlyP -value 1 -text [_ "yes"]    -anchor w

pack .gen.ctr.bmp.lab -side top -expand y -fill x -anchor w -padx 4
pack .gen.ctr.bmp.rdb1 -expand 1 -fill y  -anchor w -padx 15 -pady {5 3}
pack .gen.ctr.bmp.rdb2 -expand 1 -fill y  -anchor w -padx 15 -pady {3 7}
bind .gen.ctr.bmp       <<B3>> ExplainBMPOnly
bind .gen.ctr.bmp.lab   <<B3>> ExplainBMPOnly
bind .gen.ctr.bmp.rdb1  <<B3>> ExplainBMPOnly
bind .gen.ctr.bmp.rdb2  <<B3>> ExplainBMPOnly
set msg \
    [_ "Do all the character codes in the material\nthat you are sorting fall within the BMP?"];
balloonhelp_for .gen.ctr.bmp       $msg
balloonhelp_for .gen.ctr.bmp.lab   $msg
balloonhelp_for .gen.ctr.bmp.rdb1  $msg
balloonhelp_for .gen.ctr.bmp.rdb2  $msg

#Reserve Private Use Areas?
frame       .gen.ctr.pua -relief $genrelief -border $genborder
label       .gen.ctr.pua.lab  -text [_ "Reserve Private Use Areas?"]   -anchor w \
     -font HeaderFont
radiobutton .gen.ctr.pua.rdb1 -variable ReservePrivateUseAreasP -value 0 -text [_ "no"]  -anchor w
radiobutton .gen.ctr.pua.rdb2 -variable ReservePrivateUseAreasP -value 1 -text [_ "yes"] -anchor w
pack .gen.ctr.pua.lab -side top -expand y -fill x -anchor w -padx 4
pack .gen.ctr.pua.rdb1 -expand 1 -fill y  -anchor w -padx 15 -pady {5 3}
pack .gen.ctr.pua.rdb2 -expand 1 -fill y  -anchor w -padx 15 -pady {3 7}
bind .gen.ctr.pua       <<B3>> ExplainReservePrivateUseAreas
bind .gen.ctr.pua.lab   <<B3>> ExplainReservePrivateUseAreas
bind .gen.ctr.pua.rdb1  <<B3>> ExplainReservePrivateUseAreas
bind .gen.ctr.pua.rdb2  <<B3>> ExplainReservePrivateUseAreas
set msg [_ "Does the material you are sorting make use\nof codepoints in the Private Use Areas?"];
balloonhelp_for .gen.ctr.pua       $msg
balloonhelp_for .gen.ctr.pua.lab   $msg
balloonhelp_for .gen.ctr.pua.rdb1  $msg
balloonhelp_for .gen.ctr.pua.rdb2  $msg

#Choice of algorithm
frame       .gen.ctr.al -relief $genrelief -border $genborder
label       .gen.ctr.al.lab  -text [_ "Which Sort Algorithm to Use"] -anchor nw -font HeaderFont
frame       .gen.ctr.al.opt
radiobutton .gen.ctr.al.opt.rdb0 -variable SortAlgorithm -value i -text [_ "Insertionsort"]
radiobutton .gen.ctr.al.opt.rdb1 -variable SortAlgorithm -value m -text [_ "Mergesort"]
radiobutton .gen.ctr.al.opt.rdb2 -variable SortAlgorithm -value q -text [_ "Quicksort"]
radiobutton .gen.ctr.al.opt.rdb3 -variable SortAlgorithm -value s -text [_ "Shellsort"]
grid .gen.ctr.al.opt.rdb0 -row 0 -column 0  -sticky w -padx 0 -pady {5 3}
grid .gen.ctr.al.opt.rdb1 -row 0 -column 1  -sticky w -padx {25 5} -pady {5 3}
grid .gen.ctr.al.opt.rdb2 -row 1 -column 0  -sticky w -padx 0 -pady  {3 3}
grid .gen.ctr.al.opt.rdb3 -row 1 -column 1  -sticky w -padx {25 5} -pady {3 3}
pack .gen.ctr.al.lab -side top -expand 1 -fill both -anchor w -padx 4  -pady {3 2}
pack .gen.ctr.al.opt -side top -expand 1 -fill none -anchor w -padx 15 -pady {3 5}
bind .gen.ctr.al.opt.rdb0 <<B3>> ExplainInsertionSort
bind .gen.ctr.al.opt.rdb1 <<B3>> ExplainMergeSort
bind .gen.ctr.al.opt.rdb2 <<B3>> ExplainQuickSort
bind .gen.ctr.al.opt.rdb3 <<B3>> ExplainShellSort

if {$UseCheckbuttonP} {
    frame  .gen.ctr.alf
    label  .gen.ctr.alf.lab  -text [_ "Algorithm Choices"]   -anchor w -font HeaderFont
    checkbutton .gen.ctr.alf.ckb -variable AlgorithmDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen
    pack .gen.ctr.alf.lab -side left -expand 0 -fill y -anchor w
    pack .gen.ctr.alf.ckb -side right -expand 1 -fill y -anchor e
} else {
    frame .gen.ctr.alf  -relief $grelief -border $gborder
    checkbutton .gen.ctr.alf.ckb -variable AlgorithmDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen \
	-text [_ "Algorithm Choices"] -anchor w -font HeaderFont -indicatoron 0 \
	-activeforeground red 	-activebackground navajowhite
    pack .gen.ctr.alf.ckb -side right -expand 1 -fill both -anchor e
}

checkbutton .gen.ctr.alckb -variable AlgorithmDisplayedP \
    -onvalue 1 -offvalue 0 -command PackGen -text [_ "roll up"] \
    -activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
    -activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)


#Name of input file
frame  .gen.ctr.if
frame  .gen.ctr.if.labf
label  .gen.ctr.if.labf.lab  -text [_ "Specify Input File"]   -anchor w -font HeaderFont
checkbutton .gen.ctr.if.labf.ckb -variable InputFileDisplayedP \
    -onvalue 1 -offvalue 0 -command PackGen -text [_ "roll up"] \
    -activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
    -activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
pack .gen.ctr.if.labf.lab -side left  -expand 1 -fill both -anchor w
pack .gen.ctr.if.labf.ckb -side right -expand 0 -fill both -anchor e

set btxt [_ "Browse"]
set blen [string length $btxt];
button .gen.ctr.if.brb  -text $btxt -width $blen -anchor w -command SelectInputFile
entry  .gen.ctr.if.ent -font UserPlainTextFont
bind 	    .gen.ctr.if.ent <FocusIn> {SetInsertionTargets .gen.ctr.if.ent}
.gen.ctr.if.ent insert 0 [_ "TestInput"]
pack .gen.ctr.if.labf -side top -expand y -fill x -anchor w -pady {9 5} -padx {5 3}
pack .gen.ctr.if.brb  -expand 0 -anchor w -side left  -padx {8 3} -pady {5 9}
pack .gen.ctr.if.ent  -expand 1 -fill x  -anchor w -side left -padx {3 8} -pady {5 9}

if {$UseCheckbuttonP} {
    frame  .gen.ctr.iff
    label  .gen.ctr.iff.lab  -text [_ "Specify Input File"]   -anchor w -font HeaderFont
    checkbutton .gen.ctr.iff.ckb -variable InputFileDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen
    pack .gen.ctr.iff.lab -side left -expand 0 -fill y -anchor w
    pack .gen.ctr.iff.ckb -side right -expand 1 -fill y -anchor e
} else {
    frame .gen.ctr.iff  -relief $grelief -border $gborder
    checkbutton .gen.ctr.iff.ckb -variable InputFileDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen \
	-text [_ "Specify Input File"] -anchor w -font HeaderFont -indicatoron 0 \
	-activeforeground red 	-activebackground navajowhite
    pack .gen.ctr.iff.ckb -side right -expand 1 -fill both -anchor e
}


set bhmsg [_ "Specify the name of the input file."];
balloonhelp_for .gen.ctr.if $bhmsg;
balloonhelp_for .gen.ctr.if.labf $bhmsg;
balloonhelp_for .gen.ctr.if.ent [_ "Input will be read from the file whose\nname is shown in this entry box. You may\nenter its name directly or by using the\nfile selection dialogue."];
balloonhelp_for .gen.ctr.if.brb [_ "Press this button to select the name of the input file."]

#Name of output file
frame  .gen.ctr.of
frame  .gen.ctr.of.labf
label  .gen.ctr.of.labf.lab  -text [_ "Specify Output File"]   -anchor w -font HeaderFont
checkbutton .gen.ctr.of.labf.ckb -variable OutputFileDisplayedP \
    -onvalue 1 -offvalue 0 -command PackGen -text [_ "roll up"] \
	-activebackground $::ColorSpecs(RollUpButton,ActiveBackground) \
	-activeforeground $::ColorSpecs(RollUpButton,ActiveForeground)
pack .gen.ctr.of.labf.lab -side left  -expand 1 -fill both -anchor w
pack .gen.ctr.of.labf.ckb -side right -expand 0 -fill both -anchor e

button .gen.ctr.of.brb  -text $btxt -width $blen -anchor w -command SelectOutputFile
entry  .gen.ctr.of.ent -font UserPlainTextFont
bind   .gen.ctr.of.ent <FocusIn> {SetInsertionTargets .gen.ctr.of.ent}
.gen.ctr.of.ent insert 0 [_ "TestOutput"]
pack .gen.ctr.of.labf -side top -expand y -fill x -anchor w -pady {9 5} -padx {5 3}
pack .gen.ctr.of.brb  -expand 0 -anchor w -side left -padx {8 3} -pady {5 9}
pack .gen.ctr.of.ent  -expand 1 -fill x  -anchor w -side left -padx {3 8} -pady {5 9}

if {$UseCheckbuttonP} {
    frame  .gen.ctr.off
    label  .gen.ctr.off.lab  -text [_ "Specify Output File"]   -anchor w -font HeaderFont
    checkbutton .gen.ctr.off.ckb -variable OutputFileDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen
    pack .gen.ctr.off.lab -side left -expand 0 -fill y -anchor w
    pack .gen.ctr.off.ckb  -side right -expand 1 -fill y -anchor e
} else {
    frame .gen.ctr.off  -relief $grelief -border $gborder
    checkbutton .gen.ctr.off.ckb -variable OutputFileDisplayedP \
	-onvalue 1 -offvalue 0 -command PackGen \
	-text [_ "Specify Output File"] -anchor w -font HeaderFont -indicatoron 0 \
	-activeforeground red 	-activebackground navajowhite
    pack .gen.ctr.off.ckb -side right -expand 1 -fill both -anchor e
}

set bhmsg [_ "Specify the name of the output file."];
balloonhelp_for .gen.ctr.of $bhmsg;
balloonhelp_for .gen.ctr.of.labf $bhmsg;
balloonhelp_for .gen.ctr.of.ent [_ "Output will be written to the file whose\nname is shown in this entry box. You may\nenter its name directly or by using the file\nselection dialogue."];
balloonhelp_for .gen.ctr.of.brb [_ "Press this button to select the name of the output file."]


balloonhelp_for .gen.ctr.rp    [_ "How is the input to be divided into records?\nOne option is to treat each line as a record.\nA second option is to treat a sequence of\none or more blank lines as the record separator.\nThe third option is to consider a record to end\nwhen the character you specify is encountered.\nThe fourth option is for each record to consist of\nthe same fixed number of bytes. "]
balloonhelp_for .gen.ctr.wr    [_ "Use the entire record as the sole sort\nkey? If not, you must specify how the\nrecord is divided into fields"]
balloonhelp_for .gen.ctr.gi  [_ "Invert the ordering of the entire\nresult of the sort?"]
balloonhelp_for .gen.ctr.eol [_ "Is end-of-line marked by Line Feed\nor Carriage Return?"]

trace variable RecordParse w  ToggleRecordSeparatorEntryActive
trace variable WhichKey w ChangePage
trace variable WholeRecordIsKeyP w ToggleFieldSeparatorEntryActive
trace variable EOLCarriageReturnP w UpdateDefaultFieldSeparator

#If the default browser is on the list, remove it.
set di [lsearch -exact $BrowserList $DefaultBrowser]
if {$di >= 0} {
    set BrowserList [lreplace $BrowserList $di $di]
}
#Add the default browser to the beginning of the list.
set BrowserList [linsert $BrowserList 0 $DefaultBrowser];

foreach fn $CustomCharacterChartFileNameList {
    wm withdraw [ReadCustomCharacterChartPopup $fn]
}
foreach dl $CustomCharacterChartDataList {
    wm withdraw [DefineCustomCharacterChartPopup $dl]
}

set KeyMap(0) .gen;
set SelectionButtonMap(0) ".top.sel.gen";
AddKey;					# Create key 1
UpdateInverseKeyMap;
UpdateInverseSelectionButtonMap;
BindKeys .
SetCommandGlosses;
.top.sel.gen invoke
set SavedRecordParseDisplayedP $RecordParseDisplayedP
set RecordParseDisplayedP 1;
PackGen
after idle {
    update idletasks
    ConfigureGenOne;
    ConfigureGenTwo;
}
tk_focusFollowsMouse;
SetInsertionTargets .gen.ctr.if.ent
after 500 {
    set RecordParseDisplayedP $SavedRecordParseDisplayedP
    PackGen
}

