/*
 * convcard - A tool for converting vcards
 * Copyright (C) 2004-2005  Armin Bauer <armin.bauer@opensync.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 * 
 */
 
#include <opensync/opensync.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <glib.h>

static void usage (char *name, int ecode)
{
	fprintf (stderr, "Usage: %s [options] <source>\n\n", name);
	fprintf (stderr, "Converts between OpenSync object formats.\n");
	fprintf (stderr, "By default the source file format is autodetected.  You must specify\n");
	fprintf (stderr, "the target format via the '--to' argument.  You may view a list of\n");
	fprintf (stderr, "valid formats by using the 'msynctool --showformats <objtype> command.\n\n");
	fprintf (stderr, "One of the tools from OpenSync.\n\n");
	fprintf (stderr, "Options:\n");
	fprintf (stderr, "  <source>\t\tThe file to convert.\n");
	fprintf (stderr, "  --from <objformat>\t\tAssume <file> is of format <objformat>\n");
	fprintf (stderr, "  --to <objformat>\t\tConvert <file> to format <objformat>\n");
	fprintf (stderr, "  --out <file>\t\tSave the output in <file>.  Default is stdout.\n");
	exit (ecode);
}

int main (int argc, char *argv[])
{
	if (argc < 2) {
		usage (argv[0], 1);
	}

	// Parse the argument list
	char *filename = argv[1];
	char *sourceformatname = NULL;
	char *targetformatname = NULL;
	char *output = NULL;
	
	int i;
	for (i = 1; i < argc; i++) {
		char *arg = argv[i];
		if (!strcmp (arg, "--from")) {
			sourceformatname = argv[i + 1];
			i++;
			if (!sourceformatname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--to")) {
			targetformatname = argv[i + 1];
			i++;
			if (!targetformatname)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--out")) {
			output = argv[i + 1];
			i++;
			if (!output)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--help")) {
			usage (argv[0], 0);
		} else if (!strcmp (arg, "--")) {
			break;
		} else if (arg[0] == '-') {
			usage (argv[0], 1);
		/*
		} else {
			usage (argv[0], 1);
		*/
		}
	}
	
	if (!targetformatname) {
		usage (argv[0], 1);
	}

	// Initialize opensync environments
	OSyncEnv *env = osync_env_new();
	OSyncError *error = NULL;
	if (!osync_env_initialize(env, &error)) {
		fprintf(stderr, "Unable to load environment: %s\n", osync_error_print(&error));
		exit(1);
	}
	
	OSyncFormatEnv *conv_env = osync_conv_env_new(env);
	if (!conv_env) {
		fprintf(stderr, "Unable to create format environment");
		exit(1);
	}
	
	// Read in the source file
	char *buffer;
	int size;
	if (!osync_file_read(filename, &buffer, &size, &error)) {
		fprintf(stderr, "%s\n", osync_error_print(&error));
		osync_error_free(&error);
		return 1;
	}
	
	// Construct a change object with the source file as a base
	OSyncChange *change = osync_change_new();
	osync_change_set_uid(change, filename);		
	osync_change_set_data(change, buffer, size, TRUE);
	osync_change_set_conv_env(change, conv_env);
	osync_change_set_objformat_string(change, "plain");

	// Figure out what the source object format is (detect if user did not provide it)
	OSyncObjFormat *sourceformat = NULL;
	if (sourceformatname) {
		sourceformat = osync_conv_find_objformat(conv_env, sourceformatname);
		if (!sourceformat) {
			fprintf(stderr, "'%s' is not a valid object format", sourceformatname);
			exit(1);
		}
	} else {
		sourceformat = osync_change_detect_objformat(conv_env, change, &error);
		if (!sourceformat) {
			fprintf(stderr, "Unable to detect format of the file\n");
			exit(1);
		}
	}
	osync_change_set_objformat(change, sourceformat);
	osync_change_set_objtype(change, osync_objformat_get_objtype(sourceformat));

	// Figure out the target object format
	OSyncObjFormat *targetformat = osync_conv_find_objformat(conv_env, targetformatname);
	if (!targetformat) {
		fprintf(stderr, "'%s' is not a valid object format", targetformatname);
		exit(1);
	}
	
	// Do the conversion
	if (!osync_change_convert(conv_env, change, targetformat, &error)) {
		fprintf(stderr, "Unable to convert data: %s\n", osync_error_print(&error));
		exit(1);
	}
	
	buffer = osync_change_get_data(change);
	size = osync_change_get_datasize(change);
	
	// Write out the result
	if (output) {
		if (!osync_file_write(output, buffer, size, 0644, &error)) {
			fprintf(stderr, "Unable to write file %s: %s", output, osync_error_print(&error));
			osync_error_free(&error);
			return 1;
		}
	} else {
		puts(buffer);
		printf("\n");
	}
	
	// Free up whatever we allocated
	osync_conv_env_free(conv_env);
	osync_env_finalize(env, NULL);
	osync_env_free(env);
	
	return 0;
}

