//muroard.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010
 *
 *  This file is part of RoarD,
 *  a sound server daemon for using the RoarAudio protocol.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "muroard.h"

int g_alive = 0;

#ifdef MUROARD_FEATURE_HELP_TEXT
void usage (void) {
 printf("Usage: muroard [OPTIONS]...\n\n");

 printf("Misc Options:\n\n");
 printf(""
#ifdef MUROAR_FEATURE_DAEMONIZE
        " --daemon              - Bring the server into background after init\n"
#endif
       );

 printf("\nAudio Options:\n\n");
 printf(
        " -R  --rate   RATE     - Set server rate\n"
        " -C  --chans  CHANNELS - Set server channels\n"
       );

 printf("\nDriver Options:\n\n");
 printf(
        " -O  --odevice DEV     - Set output device\n"
       );

 printf("\nNetwork Options:\n\n");
 printf(
        "     --no-listen       - Disable listen socket\n"
#ifdef MUROAR_FEATURE_CLIENT_FH
        "     --client-fh FH    - Add a client via FH\n"
#endif
#ifdef MUROAR_FEATURE_SOCKET_LISTEN
        "     --bind ADDR       - Set path/hostname of listen socket\n"
        "     --port PORT       - Set port of listen socket\n"
#endif
#ifdef MUROAR_FEATURE_SOCKET_UNIX
        " -u  --unix            - Use UNIX Domain listen socket\n"
#endif
#ifdef MUROAR_FEATURE_SOCKET_INET
        " -t  --tcp             - Use TCP listen socket\n"
#endif
       );
}
#else
#define usage()
#endif

// signal handler:
#ifdef MUROAR_FEATURE_SIGNALS
void on_sig(int sig) {
 (void)sig; // we do not need this parameter

 g_alive = 0; // terminate daemon
}
#endif

int main (int argc, char * argv[]) {
#ifdef MUROAR_FEATURE_SOCKET_LISTEN
 int    n_type = MUROAR_DEFAULT_LISTEN_TYPE;
 char * n_addr = NULL; // use defualts from network module
 int    n_port = MUROAR_DEFAULT_LISTEN_PORT;
 int    n_no_listen = 0;
#endif
#ifdef MUROAR_FEATURE_ARGV
 int i;
 char * k;
 char * o_dev = NULL;
#ifdef MUROAR_FEATURE_DAEMONIZE
 int daemon = 0;
#endif
#ifdef MUROAR_FEATURE_CLIENT_FH
 int    c_sock[MUROAR_MAX_CLIENTS];
 int    c_sock_count = 0;
#endif
#endif

#ifdef MUROAR_FEATURE_ARGV
 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( !strcmp(k, "--help") || !strcmp(k, "-h") ) {
   usage();
   return 0;
#ifdef MUROAR_FEATURE_DAEMONIZE
  } else if ( !strcmp(k, "--daemon") ) {
   daemon = 1;
#endif
  } else if ( !strcmp(k, "--rate") || !strcmp(k, "-R") ) {
   g_sa_rate = atoi(argv[++i]);
  } else if ( !strcmp(k, "--channels") || !strcmp(k, "-C") ) {
   g_sa_channels = atoi(argv[++i]);
  } else if ( !strcmp(k, "--odevice") || !strcmp(k, "-O") ) {
   o_dev  = argv[++i];
  } else if ( !strcmp(k, "--no-listen") ) {
   n_no_listen = 1;
#ifdef MUROAR_FEATURE_CLIENT_FH
  } else if ( !strcmp(k, "--client-fh") ) {
   c_sock[c_sock_count++] = atoi(argv[++i]);
#endif
#ifdef MUROAR_FEATURE_SOCKET_LISTEN
  } else if ( !strcmp(k, "--bind") ) {
   n_addr = argv[++i];
  } else if ( !strcmp(k, "--port") ) {
   n_port = atoi(argv[++i]);
#ifdef MUROAR_FEATURE_SOCKET_UNIX
  } else if ( !strcmp(k, "--unix") || !strcmp(k, "-u") ) {
    n_type = NETWORK_TYPE_UNIX;
#endif
#ifdef MUROAR_FEATURE_SOCKET_INET
  } else if ( !strcmp(k, "--tcp") || !strcmp(k, "-t") ) {
    n_type = NETWORK_TYPE_INET;
#endif
#endif
  } else {
   usage();
   return 1;
  }
 }
#endif

 if ( network_init() == -1 ) {
  return 1;
 }

 if ( waveform_init() == -1 ) {
  return 1;
 }

 if ( client_init() == -1 ) {
  return 1;
 }

 if ( stream_init() == -1 ) {
  return 1;
 }

#ifdef MUROAR_FEATURE_ARGV
 if ( driver_init(o_dev) == -1 ) {
#else
 if ( driver_init(NULL) == -1 ) {
#endif
  return 3;
 }

#ifdef MUROAR_FEATURE_CLIENT_FH
 for (i = 0; i < c_sock_count; i++) {
  if ( client_new(c_sock[i]) == -1 ) {
   return 1;
  }
 }
#endif

#ifdef MUROAR_FEATURE_SOCKET_LISTEN
 if ( !n_no_listen ) {
  if ( network_listen(n_type, n_addr, n_port) == -1 ) {
   return 2;
  }
 }
#endif

 // everything is up, get it running!:
 g_alive = 1;

#ifdef MUROAR_FEATURE_SIGNALS
 signal(SIGINT,  on_sig);
 signal(SIGTERM, on_sig);
 signal(SIGPIPE, SIG_IGN);
#endif

#ifdef MUROAR_FEATURE_DAEMONIZE
 if ( daemon ) {
  close(STDIN_FILENO);
  close(STDOUT_FILENO);
  close(STDERR_FILENO);

  if ( fork() )
   exit(0);

  setsid();
 }
#endif

 while (g_alive) {
  if ( network_check() == -1 ) {
   g_alive = 0;
   break;
  }

  if ( waveform_update() == -1 ) {
   g_alive = 0;
   break;
  }
 }

 network_prefree(); // this will free the listen socket early

 driver_free();
 stream_free();
 client_free();
 waveform_free();
 network_free();

 return 0;
}

//ll
