 /******************************************************************

        MUSCLE SmartCard Development ( http://www.linuxnet.com )
            Title  : muscleTool.c
            Package: MuscleTools
            Author : David Corcoran
                     Karsten Ohme
            Date   : 11/11/05
            License: Copyright (C) 2002 David Corcoran
                     <corcoran@linuxnet.com>
            Purpose: A shell for the MuscleCard framework

********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "muscleTool.h"

#ifndef WIN32
#include <unistd.h>
#endif

#ifdef WIN32
#include <io.h>
#define snprintf _snprintf
#endif

#include <sys/stat.h>
#include <pthread.h>
#ifndef __APPLE__
#include <musclecard.h>
#else
#include <PCSC/musclecard.h>
#endif


#define MUSCLETOOL_CHV_UNBLOCK_TRIES	3
#define MUSCLETOOL_PIN_MAXTRIES		8


static MSCLPTokenConnection pConnection = 0;
static MSCLPTokenInfo tokenList = 0;
static MSCULong32 tokenSize = 0;
MSCString connectToken = 0;		/* used for the prompt in main() */
static int quitStat = 0;

static int getChoice(const char *, const char *, const char **, int);
static int getStringValue(const char *header, const char *footer, char *stringValue, int stringValueLength);
static int getNumberValue(const char *header, const char *footer, unsigned long *numberValue);
static void doPrintAUT(MSCUShort16);
#ifdef VERSION2
static void doPrintOwner(MSCUShort16);
void doPrintKeyFlags(MSCUShort16);
#endif
static MSCUShort16 getUShort16(MSCPUChar8 srcValue);
static MSCULong32 getULong32(MSCPUChar8 srcValue);

#define CHECK_ERR(cond, msg) { if (cond) { \
  printf("ERR: %s (0x%lX %s)\n", msg, rv, msc_error(rv)); goto end; } }

#define PRINT_SUPPORT(cond) { if (cond) { \
  printf("X\n"); } else { printf("\n"); } }


#ifdef VERSION2
static void setUShort16(MSCPUChar8 destValue, MSCUShort16 srcValue) {
  destValue[0] = (srcValue & 0xFF00) >> 8;
  destValue[1] = (srcValue & 0x00FF);
}

static void _8ByteTo4Short(MSCUChar8 byteArray[8], MSCUShort16 shortArray[4])
{
	shortArray[0] = getUShort16(byteArray);
	shortArray[1] = getUShort16(byteArray+2);
	shortArray[2] = getUShort16(byteArray+4);
	shortArray[3] = getUShort16(byteArray+6);
}
#endif

static MSCUShort16 getUShort16(MSCPUChar8 srcValue) 
{
	return (MSCUShort16)( ((srcValue[0]) << 8) | srcValue[1] );
}

static MSCULong32 getULong32(MSCPUChar8 srcValue) 
{
	MSCULong32 temp;
	temp  = srcValue[0] * 0x1000000;
	temp += srcValue[1] * 0x10000;
	temp += srcValue[2] * 0x100;
	temp += srcValue[3];
	return temp;
}

static void *printStatStars(void *arg)
{
	while (1)
	{
#ifndef WIN32
		usleep(500000);
#else
		Sleep(500);
#endif
		putc('*', stdout);
		fflush(stdout);
		if (quitStat == 1)
		{
			break;
		}
	}

	printf("] : ");
	return NULL;
}


#if 0
static MSCULong32 tokenCallback(MSCLPTokenInfo tokInfo, MSCULong32 tokSize,
	MSCPVoid32 data)
{
	doRelease();

	printf("Token Removed - Press Enter to Continue\n");

	return 0;
}
#endif

static MSCUChar8 muscleDefaultKey[8] =
	{ 'M', 'u', 's', 'c', 'l', 'e', '0', '0' };

static int textToBytes(MSCString inStr, MSCPUChar8 Buffer, MSCPULong32 Length)
{
	int i;
	int j;
	int inLen;

	j = 0;
	inLen = 0;

	inLen = strlen(inStr);

	if (inLen > MSC_MAXSIZE_AID)
		return -1;

	for (i = 0; i < inLen; i += 2)
	{
		if (inStr[i] <= '9' && inStr[i] >= '0')
		{
			Buffer[j] = (inStr[i] - '0') * 16;
		}
		else if (inStr[i] <= 'F' && inStr[i] >= 'A')
		{
			Buffer[j] = (inStr[i] - 'A' + 10) * 16;
		}

		if (inStr[i + 1] <= '9' && inStr[i + 1] >= '0')
		{
			Buffer[j] += inStr[i + 1] - '0';
		}
		else if (inStr[i + 1] <= 'F' && inStr[i + 1] >= 'A')
		{
			Buffer[j] += inStr[i + 1] - 'A' + 10;
		}

		j += 1;
	}

	*Length = j;

	return 0;
}


static MSCULong32 hexToBin(MSCUChar8 * data, MSCUChar8 * out)
{
	MSCULong32 i;
	MSCULong32 count = 0;
	unsigned long c;
	char shortData[3];
	MSCULong32 start = 0;
	int ret;

	MSCULong32 len = strlen((char *)data);
	/* First character must be padded with additional 0 */
	if ((len % 2) != 0) {
		start = 1;
		shortData[0] = 0;
		shortData[1] = data[0];
		shortData[2] = '\0';
		ret = sscanf(shortData, "%lx", &c);
		if (ret <= 0) {
			c = 0;
		}
		out[count] = (MSCUChar8)c;
		count++;
	}

	for (i = start; i < len; i += 2)
	{
		shortData[0] = data[i];
		shortData[1] = data[i + 1];
		shortData[2] = '\0';
		ret = sscanf(shortData, "%lx", &c);
		if (ret <= 0) {
			c = 0;
		}
		out[count] = (MSCUChar8)c;
		count++;
	}

	return count;
}

/**
* Pads data at the end with 0x00.
*/
static void padData(MSCUChar8 * data, MSCULong32 inSize, MSCULong32 outSize)
{
	MSCULong32 i;

	for (i = inSize; i < outSize; i++)
		data[i] = 0;
}

/**
* Pads data at the beginning with 0x01 and continues 0x00s so. 
* If inSize == outSize the first byte is now allowed to be larger than
* 0x80.
* So the RSA modulus is smaller and the exponentiation exceeds modulus.
* \return 1 for success, 0 if failure,
*/
static MSC_RV padRSAData(MSCUChar8 * data, MSCULong32 inSize, MSCULong32 outSize)
{
	MSCULong32 i;
	MSC_RV rv = 0;
	MSCPUChar8 tmp = NULL;

	tmp = malloc(outSize);
	/* modulus must be larger than data */
	if ((inSize == outSize) && (data[0] > 0x80)) {
		goto end;
	}
	memcpy(tmp, data, outSize);
	data[0] = 0x01;

	for (i = 1; i < outSize-inSize; i++)
		data[i] = 0;

	memcpy(data+i, tmp, inSize);
	rv = 1;
end:
	if (tmp)
		free(tmp);
	return rv;
}

void doExit(void) {
	if (pConnection != 0) {
		MSCReleaseConnection(pConnection, MSC_RESET_TOKEN);
		free(pConnection);
		pConnection = 0;
	}
	if (connectToken != 0) {
		free(connectToken);
		connectToken = 0;
	}
	if (tokenList)
	{
		free(tokenList);
		tokenList = 0;
		tokenSize = 0;
	}
}

#ifdef VERSION2
void doDeleteKey(int key_nb) {
	MSCLong32 rv = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCDeleteKey(pConnection, (MSCUChar8)key_nb);
	CHECK_ERR(rv != MSC_SUCCESS, "DeleteKey Failed !");
	printf("DeleteKey Successful\n");
end:
	return;
}
#else
void doDeleteKey(int key_nb) {
  printf("Not supported. \n");
  return;
}
#endif

#ifdef VERSION2
void doChangeACL(char *objectID) {
	MSCLong32 rv = 0;
	MSCObjectACL objectACL;
	MSCULong32 readACL, writeACL, deleteACL;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = getNumberValue("Enter the new ACL for reading the object", "Enter ACL mask", &readACL);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the new ACL for writing the object", "Enter ACL mask", &writeACL);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the new ACL for deleting the object", "Enter ACL mask", &deleteACL);
	if (rv == 0)
		goto abort;

	objectACL.readPermission = (MSCUShort16)readACL;
	objectACL.writePermission = (MSCUShort16)writeACL;
	objectACL.deletePermission = (MSCUShort16)deleteACL;

	rv = MSCChangeObjectACL(pConnection, (MSCString)objectID, &objectACL);
	CHECK_ERR(rv != MSC_SUCCESS, "ChangeACL Failed !");
	printf("ChangeACL Successful\n");
abort:
end:
	return;
}
#else
void doChangeACL(char *objectID) {
  printf("Not supported. \n");
}
#endif

#ifdef VERSION2
void doChangeKeyACL(int key_nb) {
	MSCLong32 rv = 0;
	MSCKeyACL keyACL;
	MSCULong32 readACL, writeACL, useACL;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = getNumberValue("Enter the new ACL for reading the key", "Enter ACL mask", &readACL);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the new ACL for writing the key", "Enter ACL mask", &writeACL);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the new ACL for deleting the key", "Enter ACL mask", &useACL);
	if (rv == 0)
		goto abort;

	keyACL.readPermission = (MSCUShort16)readACL;
	keyACL.writePermission = (MSCUShort16)writeACL;
	keyACL.usePermission = (MSCUShort16)useACL;

	rv = MSCChangeKeyACL(pConnection, (MSCUChar8)key_nb, &keyACL);
	CHECK_ERR(rv != MSC_SUCCESS, "ChangeACL Failed !");
	printf("ChangeACL Successful\n");
abort:
end:
	return;
}
#else
void doChangeKeyACL(int key_nb) {
  printf("Not supported. \n");
}
#endif

#ifdef VERSION2
void doSetStatus(void) {
	MSCLong32 rv = 0;
	MSCULong32 choice;
	MSCULong32 dummy = 256;
	MSCUChar8 temp[256];
	const char *choices[5] = {"Serial Number", "Card Label", "ID Creation ACL",
	"Object Creation ACL", "Key Creation ACL"};

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	choice = getChoice("What do you want to change?: ", "Choose", choices, 5);

	switch (choice) {
		case 0:
			goto abort;
		case 1:
			rv = getStringValue(
				"Please enter the serial number (exactly 16 chars, eventually padded with ' ')",
				"Enter serial number",
				temp,
				MSC_SERIALNUMBER_SIZE+1
			);
			if (rv == 0)
				goto abort;
			rv = MSCSetStatus(pConnection, MSC_TAG_STATE_SERIAL_NUMBER, temp, 16);
			if (rv == MSC_SERIAL_ERROR) {
				printf("ERR: Serial Number already set.\n");
				goto end;
			}
			else if (rv == MSC_UNSUPPORTED_FEATURE) {
				printf("ERR: Serial Number not supported.\n");
				goto end;
			}
			else {
				CHECK_ERR(rv != MSC_SUCCESS, "SetState Failed !");
			}
			break;
		case 2:
			rv = getStringValue(
				"Please enter the card label (exactly 32 chars, eventually padded with ' ')",
				"Enter card label",
				temp,
				MSC_TOKENLABEL_SIZE+1
			);
			if (rv == 0)
				goto abort;
			rv = MSCSetStatus(pConnection, MSC_TAG_STATE_TOKEN_LABEL, temp, 32);
			if (rv == MSC_LABEL_ERROR) {
				printf("ERR: Card label already set.\n");
				goto end;
			}
			else if (rv == MSC_UNSUPPORTED_FEATURE) {
				printf("ERR: Card label not supported.\n");
				goto end;
			}
			else {
				CHECK_ERR(rv != MSC_SUCCESS, "SetState Failed !");
			}
			break;
		case 3:
			rv = getNumberValue("Enter the ACL for creating IDs\n", "Enter ACL mask", &dummy);
			if (rv == 0)
				goto abort;
			setUShort16(temp, (MSCUShort16)dummy);
			rv = MSCSetStatus(pConnection, MSC_TAG_STATE_ID_CREATION_ACL, temp, 2);
			CHECK_ERR(rv != MSC_SUCCESS, "SetState Failed !");
			break;
		case 4:
			rv = getNumberValue("Enter the ACL for creating objects\n", "Enter ACL mask", &dummy);
			if (rv == 0)
				goto abort;
			setUShort16(temp, (MSCUShort16)dummy);
			rv = MSCSetStatus(pConnection, MSC_TAG_STATE_OBJECT_CREATION_ACL, temp, 2);
			CHECK_ERR(rv != MSC_SUCCESS, "SetState Failed !");
			break;
		case 5:
			rv = getNumberValue("Enter the ACL for creating keys\n", "Enter ACL mask", &dummy);
			if (rv == 0)
				goto abort;
			setUShort16(temp, (MSCUShort16)dummy);
			rv = MSCSetStatus(pConnection, MSC_TAG_STATE_KEY_CREATION_ACL, temp, 2);
			CHECK_ERR(rv != MSC_SUCCESS, "SetState Failed !");
			break;
		default:
			break;
	}
	printf("SetStatus Successful\n");
end:
	return;
abort:
	printf("SetStatus Cancelled.\n");
	return;
}
#else
void doSetStatus(void) {
  printf("Not supported. \n");
}
#endif

#ifdef VERSION2
void doDeleteID(int pin_nb) {
	MSCLong32 rv = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCDeleteID(pConnection, (MSCUChar8)pin_nb);
	CHECK_ERR(rv != MSC_SUCCESS, "DeleteID Failed !");
	printf("DeleteID Successful\n");
end:
	return;
}
#else
void doDeleteID(int pin_nb) {
  printf("Not supported. \n");
}
#endif

#ifdef VERSION2
void doMove(char *oldObjectID, char *newObjectID) {
	MSCLong32 rv = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCMoveObject(pConnection, (MSCString)oldObjectID, (MSCString)newObjectID);
	CHECK_ERR(rv != MSC_SUCCESS, "MoveObject Failed !");
	printf("MoveObject Successful\n");
end:
	return;
}
#else
void doMove(char *oldObjectID, char *newObjectID) {
  printf("Not supported. \n");
}
#endif

#ifdef VERSION2
void doMoveKey(int old_key_nb, int new_key_nb) {
	MSCLong32 rv = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCMoveKey(pConnection, (MSCUChar8)old_key_nb, (MSCUChar8)new_key_nb);
	CHECK_ERR(rv != MSC_SUCCESS, "MoveKey Failed !");
	printf("MoveKey Successful\n");
end:
	return;
}
#else
void doMoveKey(int old_key_nb, int new_key_nb) {
  printf("Not supported. \n");
}
#endif

void doListTokens(void)
{
	MSCLong32 rv;
	MSCULong32 i;

	if (tokenList)
	{
		free(tokenList);
		tokenList = NULL;
		tokenSize = 0;
	}

	rv = MSCListTokens(MSC_LIST_KNOWN, tokenList, &tokenSize);
	CHECK_ERR(rv != MSC_SUCCESS, "ListTokens Failed !");

	if (tokenSize == 0)
	{
		printf("No Valid Tokens Found\n");
		return;
	}

	tokenList = malloc(sizeof(MSCTokenInfo) * tokenSize);
	CHECK_ERR(tokenList == 0, "Malloc Failed");

	rv = MSCListTokens(MSC_LIST_KNOWN, tokenList, &tokenSize);
	CHECK_ERR(rv != MSC_SUCCESS, "ListTokens Failed !");

	for (i = 0; i < tokenSize; i++)
	{
		printf("   %ld.    %s\n", i + 1, tokenList[i].tokenName);
	}
	printf("\n");
	printf("ListTokens Success.\n");
end:
	return;
}

void doConnect(int tokenNumber)
{
	MSCLong32 rv = 0;

	CHECK_ERR(tokenList == 0, "List Tokens First !");

	if (tokenNumber < 1 || (MSCULong32)tokenNumber > tokenSize)
	{
		printf("ERR: Invalid choice made !\n");
		goto end;
	}

	if (pConnection == 0)
	{
		pConnection = malloc(sizeof(MSCTokenConnection));
	}
	else
	{
		printf("ERR: Must Release Other Connection !\n");
		goto end;
	}

	rv = MSCEstablishConnection(&tokenList[tokenNumber - 1],
		MSC_SHARE_SHARED, NULL, 0, pConnection);
	if (rv != MSC_SUCCESS)
	{
		free(pConnection);
		pConnection = 0;
		printf("ERR: EstablishConnection Failed !\n");
		printf("Is your card already formated?\n");
		goto end;
	}

	//  rv = MSCCallbackForTokenEvent(&tokenList[tokenNumber-1], 1,
	//                tokenCallback, 0 );

	connectToken = strdup(tokenList[tokenNumber - 1].tokenName);
	printf("Connect Success.\n");
end:
	return;
}

void doRelease(void)
{
	MSCLong32 rv = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCReleaseConnection(pConnection, MSC_RESET_TOKEN);

	if (pConnection) {
		free(pConnection);
		pConnection = NULL;
	}
	if (connectToken) {
		free(connectToken);
		connectToken = NULL;
	}

	CHECK_ERR(rv != MSC_SUCCESS, "ReleaseConnection Failed !");
	printf("ReleaseConnection Success.\n");
end:
	return;
}

void doList(void)
{
	MSCLong32 rv = 0;
	MSCObjectInfo objInfo;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCListObjects(pConnection, MSC_SEQUENCE_RESET, &objInfo);
	CHECK_ERR((rv != MSC_SUCCESS && rv != MSC_SEQUENCE_END), "ListObjects Failed");

	printf("%16s %12s\n", "Object ID", "Object Size");
	printf("%16s  %12s\n", "---------------", "-----------\n");

	if (rv == MSC_SUCCESS)
	{
		printf("%16s   %10ld\n", objInfo.objectID, objInfo.objectSize);
		printf("\n");
		printf("            READ\n");
		printf("       ---------\n");
		doPrintAUT(objInfo.objectACL.readPermission),
		printf("\n");
		printf("           WRITE\n");
		printf("       ---------\n");
		doPrintAUT(objInfo.objectACL.writePermission),
		printf("\n");
		printf("          DELETE\n");
		printf("       ---------\n");
		doPrintAUT(objInfo.objectACL.deletePermission);
		printf("\n");
	}
	if (rv == MSC_SEQUENCE_END) {
		printf("ListObjects Success.\n");
		goto end;
	}

	do
	{
		rv = MSCListObjects(pConnection, MSC_SEQUENCE_NEXT, &objInfo);
		CHECK_ERR((rv != MSC_SUCCESS && rv != MSC_SEQUENCE_END), "ListObjects Failed");
		if (rv == MSC_SUCCESS)
		{
			printf("%16s   %10ld\n", objInfo.objectID, objInfo.objectSize);
			printf("\n");
			printf("            READ\n");
			printf("       ---------\n");
			doPrintAUT(objInfo.objectACL.readPermission),
			printf("\n");
			printf("           WRITE\n");
			printf("       ---------\n");
			doPrintAUT(objInfo.objectACL.writePermission),
			printf("\n");
			printf("          DELETE\n");
			printf("       ---------\n");
			doPrintAUT(objInfo.objectACL.deletePermission);
			printf("\n");
		}
	}
	while (rv != MSC_SEQUENCE_END);
	printf("ListObjects Success.\n");
end:
	return;
}

#ifdef VERSION2
void doPrintKeyPolicyCiphDir(MSCKeyInfo keyInfo) {
	if (keyInfo.keyPolicy.cipherDirection == 0xFFFF) {
		printf("%16s\n", "ANY");
		return;
	}
	/* What for ??? */
	if (keyInfo.keyPolicy.cipherDirection == 0x0000) {
		printf("%16s\n", "NONE");
		return;
	}
	if (keyInfo.keyPolicy.cipherDirection & MSC_KEYPOLICY_DIR_DECRYPT)
		printf("%16s\n", "DECRYPT");
	if (keyInfo.keyPolicy.cipherDirection & MSC_KEYPOLICY_DIR_ENCRYPT)
		printf("%16s\n", "ENCRYPT");
	if (keyInfo.keyPolicy.cipherDirection & MSC_KEYPOLICY_DIR_SIGN)
		printf("%16s\n", "SIGN");
	if (keyInfo.keyPolicy.cipherDirection & MSC_KEYPOLICY_DIR_VERIFY)
		printf("%16s\n", "VERIFY");
}

void doPrintKeyPolicyCiphMode(MSCKeyInfo keyInfo) {
	MSCUShort16 cipherSignatureMode = keyInfo.keyPolicy.cipherSignatureMode;
	MSCUShort16 cipherCipherMode = keyInfo.keyPolicy.cipherCipherMode;
	MSCUShort16 ciphDir = keyInfo.keyPolicy.cipherDirection;
	if (cipherSignatureMode == 0xFFFF && cipherCipherMode == 0xFFFF) {
		printf("%16s\n", "ANY");
		return;
	}
	/* What for ??? */
	if (cipherSignatureMode == 0x0000 && cipherCipherMode == 0x0000) {
		printf("%16s\n", "NONE");
		return;
	}
	switch (keyInfo.keyType)
	{
		case MSC_KEY_DSA_PUBLIC:
		case MSC_KEY_DSA_PRIVATE:
			if ((ciphDir & MSC_KEYPOLICY_DIR_SIGN) || (ciphDir & MSC_KEYPOLICY_DIR_VERIFY)) {
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DSA_SHA)
					printf("%16s\n", "SHA");
				}
			break;
		case MSC_KEY_RSA_PUBLIC:
		case MSC_KEY_RSA_PRIVATE:
		case MSC_KEY_RSA_PRIVATE_CRT:
			if ((ciphDir & MSC_KEYPOLICY_DIR_SIGN) || (ciphDir & MSC_KEYPOLICY_DIR_VERIFY)) {
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_MD5_PKCS1)
					printf("%16s\n", "MD5 PKCS1");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_MD5_RFC2409)
					printf("%16s\n", "MD5 RFC2409");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_RIPEMD160_ISO9796)
					printf("%16s\n", "RIPEMD 160 ISO9796");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_RIPEMD160_PKCS1)
					printf("%16s\n", "RIPEMD 160 PKCS1");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_SHA_ISO9796)
					printf("%16s\n", "SHA ISO9796");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_SHA_PKCS1)
					printf("%16s\n", "SHA PKCS1");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_RSA_SHA_RFC2409)
					printf("%16s\n", "SHA RFC2409");
			}
			if ((ciphDir & MSC_KEYPOLICY_DIR_DECRYPT) || (ciphDir & MSC_KEYPOLICY_DIR_ENCRYPT)) {
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_RSA_ISO14888)
					printf("%16s\n", "ISO 14888");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_RSA_ISO9796)
					printf("%16s\n", "ISO 9796");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_RSA_NOPAD)
					printf("%16s\n", "NOPAD");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_RSA_PKCS1)
					printf("%16s\n", "PKCS1");
			}
			break;
		case MSC_KEY_DES:
		case MSC_KEY_3DES:
			if ((ciphDir & MSC_KEYPOLICY_DIR_DECRYPT) || (ciphDir & MSC_KEYPOLICY_DIR_ENCRYPT)) {
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_CBC_ISO9797_M1)
					printf("%16s\n", "CBC ISO9797 M1");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_CBC_ISO9797_M2)
					printf("%16s\n", "CBC ISO9797 M2");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_ECB_ISO9797_M1)
					printf("%16s\n", "ECB ISO9797 M1");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_ECB_ISO9797_M2)
					printf("%16s\n", "ECB ISO9797 M1");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_ECB_NOPAD)
					printf("%16s\n", "ECB NOPAD");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_CBC_NOPAD)
					printf("%16s\n", "ECB NOPAD");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_CBC_PKCS5)
					printf("%16s\n", "CBC PKCS5");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_DES_ECB_PKCS5)
					printf("%16s\n", "ECB PKCS5");
			}
			if ((ciphDir & MSC_KEYPOLICY_DIR_SIGN) || (ciphDir & MSC_KEYPOLICY_DIR_VERIFY)) {
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC4_ISO9797_M1)
					printf("%16s\n", "MAC4 ISO9797 M1");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC4_ISO9797_M2)
					printf("%16s\n", "MAC4 ISO9797 M2");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC4_NOPAD)
					printf("%16s\n", "MAC4 NOPAD");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC4_PKCS5)
					printf("%16s\n", "MAC4 PKCS5");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC8_ISO9797_M1)
					printf("%16s\n", "MAC8 ISO9797 M1");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC8_ISO9797_M2)
					printf("%16s\n", "MAC8 ISO9797 M2");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC8_NOPAD)
					printf("%16s\n", "MAC8 NOPAD");
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_DES_MAC8_PKCS5)
					printf("%16s\n", "MAC8 PKCS5");
			}
			break;
		case MSC_KEY_AES:
			if ((ciphDir & MSC_KEYPOLICY_DIR_DECRYPT) || (ciphDir & MSC_KEYPOLICY_DIR_ENCRYPT)) {
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_AES_BLOCK_128_CBC_NOPAD)
					printf("%16s\n", "128 CBC NOPAD");
				if (cipherCipherMode & MSC_KEYPOLICY_MODE_CIPHER_AES_BLOCK_128_ECB_NOPAD)
					printf("%16s\n", "128 ECB NOPAD");
			}
			if ((ciphDir & MSC_KEYPOLICY_DIR_SIGN) || (ciphDir & MSC_KEYPOLICY_DIR_VERIFY)) {
				if (cipherSignatureMode & MSC_KEYPOLICY_MODE_SIGNATURE_AES_MAC_128_NOPAD)
					printf("%16s\n", "MAC128 NOPAD");
			}
			break;
	}
}

/**
 * Prints the key flags.
 * \param flags The key flags.
 */
void doPrintKeyFlags(MSCUShort16 flags) {
	switch (flags) {
		case MSC_KEY_TOKEN_GENERATED:
			printf("%s\n", "Key is generated on card.");
			break;
		case MSC_KEY_NEVER_READABLE:
			printf("%s\n", "Key is never readable (exportable).");
			break;
		case MSC_KEY_NEVER_WRITABLE:
			printf("%s\n", "Key is never writable (but the admin can delete it).");
			break;
		default:
			printf("%s%02X\n", "Unknown flag(s): ", flags);
			break;
	}
}
#endif

/**
 * Prints the needed authentications
 * \param aut The authentication mask.
 */
static void doPrintAUT(MSCUShort16 aut)
{
	if (aut == MSC_AUT_NONE)
	{
		printf("%16s\n", "NEVER");
		return;
	}
	else if (aut == MSC_AUT_ALL)
	{
		printf("%16s\n", "ALWAYS");
		return;
	}

	if (aut & MSC_AUT_PIN_0)
		printf("%16s\n", "PIN #0");

	if (aut & MSC_AUT_PIN_1)
		printf("%16s\n", "PIN #1");

	if (aut & MSC_AUT_PIN_2)
		printf("%16s\n", "PIN #2");

	if (aut & MSC_AUT_PIN_3)
		printf("%16s\n", "PIN #3");

	if (aut & MSC_AUT_PIN_4)
		printf("%16s\n", "PIN #4");

#ifdef VERSION2
	if (aut & MSC_AUT_PIN_5)
		printf("%16s\n", "PIN #5");

	if (aut & MSC_AUT_PIN_6)
		printf("%16s\n", "PIN #6");

	if (aut & MSC_AUT_PIN_7)
		printf("%16s\n", "PIN #7");
#endif

	if (aut & MSC_AUT_KEY_0)
		printf("%16s\n", "KEY #0");

	if (aut & MSC_AUT_KEY_1)
		printf("%16s\n", "KEY #1");

	if (aut & MSC_AUT_KEY_2)
		printf("%16s\n", "KEY #2");

	if (aut & MSC_AUT_KEY_3)
		printf("%16s\n", "KEY #3");

	if (aut & MSC_AUT_KEY_4)
		printf("%16s\n", "KEY #4");

	if (aut & MSC_AUT_KEY_5)
		printf("%16s\n", "KEY #5");
}


#ifdef VERSION2
void doPrintOwner(MSCUShort16 owner) {
	if (owner == MSC_AUT_NONE)
	{
		printf("NONE");
		return;
	}

	if (owner & MSC_AUT_PIN_0)
		printf("PIN #0 ");

	if (owner & MSC_AUT_PIN_1)
		printf("PIN #1 ");

	if (owner & MSC_AUT_PIN_2)
		printf("PIN #2 ");

	if (owner & MSC_AUT_PIN_3)
		printf("PIN #3 ");

	if (owner & MSC_AUT_PIN_4)
		printf("PIN #4 ");

	if (owner & MSC_AUT_PIN_5)
		printf("PIN #5");

	if (owner & MSC_AUT_PIN_6)
		printf("PIN #6");

	if (owner & MSC_AUT_PIN_7)
		printf("PIN #7");

	if (owner & MSC_AUT_KEY_0)
		printf("KEY #0 ");

	if (owner & MSC_AUT_KEY_1)
		printf("KEY #1 ");

	if (owner & MSC_AUT_KEY_2)
		printf("KEY #2 ");

	if (owner & MSC_AUT_KEY_3)
		printf("KEY #3 ");

	if (owner & MSC_AUT_KEY_4)
		printf("KEY #4 ");

	if (owner & MSC_AUT_KEY_5)
		printf("KEY #5 ");
}
#endif

static void doLoggedID(MSCUShort16 aut)
{
	if (aut == 0)
	{
		printf("NONE");
		return;
	}

	if (aut & MSC_AUT_PIN_0)
		printf("PIN #0 ");

	if (aut & MSC_AUT_PIN_1)
		printf("PIN #1 ");

	if (aut & MSC_AUT_PIN_2)
		printf("PIN #2 ");

	if (aut & MSC_AUT_PIN_3)
		printf("PIN #3 ");

	if (aut & MSC_AUT_PIN_4)
		printf("PIN #4 ");

#ifdef VERSION2
	if (aut & MSC_AUT_PIN_5)
		printf("PIN #5 ");

	if (aut & MSC_AUT_PIN_6)
		printf("PIN #6 ");

	if (aut & MSC_AUT_PIN_7)
		printf("PIN #7 ");
#endif

	if (aut & MSC_AUT_KEY_0)
		printf("KEY #0 ");

	if (aut & MSC_AUT_KEY_1)
		printf("KEY #1 ");

	if (aut & MSC_AUT_KEY_2)
		printf("KEY #2 ");

	if (aut & MSC_AUT_KEY_3)
		printf("KEY #3 ");

	if (aut & MSC_AUT_KEY_4)
		printf("KEY #4 ");

	if (aut & MSC_AUT_KEY_5)
		printf("KEY #5 ");
}

void doStatus(void)
{
	MSCLong32 rv = 0;
#ifdef VERSION2
	MSCULong32 dummy = 256;
	MSCUChar8 temp[256];
#else
    MSCStatusInfo pStatusStruct;
#endif

	CHECK_ERR(pConnection == 0, "Must Connect First !");

#ifdef VERSION2
	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_PROTO_VERSION, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("      Protocol Version: %d.%d\n", temp[0], temp[1]);

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_APPLET_VERSION, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("      Software Version: %d.%d\n", temp[0], temp[1]);

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_FREE_MEM, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");

	dummy = (temp[0] << 24) | (temp[1] << 16) | (temp[2] << 8) | temp[3];
	printf("           Free Memory: %ld\n", dummy);

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_TOTAL_MEM, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	dummy = (temp[0] << 24) | (temp[1] << 16) | (temp[2] << 8) | temp[3];
	printf("          Total Memory: %ld\n", dummy);

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_USED_PINS, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("             PINs Used: %d\n", temp[0]);

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_USED_KEYS, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("             Keys Used: %d\n", temp[0]);

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_LOGGED_IDS, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("            Logged IDs: ");
	doLoggedID(getUShort16(temp));
	printf("\n");

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_ID_CREATION_ACL, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("    How can create IDs: \n");
	doPrintAUT(getUShort16(temp));

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_OBJECT_CREATION_ACL, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("How can create objects: \n");
	doPrintAUT(getUShort16(temp));

	dummy = 8;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_KEY_CREATION_ACL, temp, &dummy);
	CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
	printf("   How can create keys: \n");
	doPrintAUT(getUShort16(temp));

	dummy = 32;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_TOKEN_LABEL, temp, &dummy);
	if (rv == MSC_LABEL_ERROR) {
		printf("            Card Label: Not set.\n");
	}
	else if (rv == MSC_UNSUPPORTED_FEATURE) {
		printf("            Card Label: Not supported.\n");
	}
	else {
		CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
		temp[32] = '\0';
		printf("            Card Label: %s\n", temp);
	}

	dummy = 16;
	rv = MSCGetStatus(pConnection, MSC_TAG_STATE_SERIAL_NUMBER, temp, &dummy);
	if (rv == MSC_SERIAL_ERROR) {
		printf("         Serial Number: Not set.\n");
	}
	else if (rv == MSC_UNSUPPORTED_FEATURE) {
		printf("         Serial Number: Not supported.\n");
	}
	else {
		CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");
		temp[16] = '\0';
		printf("         Serial Number: %s\n", temp);
	}

#else
    rv = MSCGetStatus(pConnection, &pStatusStruct);
    CHECK_ERR(rv != MSC_SUCCESS, "GetState Failed !");

    printf(" Protocol Version: %d.%d\n", (pStatusStruct.appVersion>>8) & 0xFF, 
            (pStatusStruct.appVersion & 0x00FF));
    printf(" Software Version: %d.%d\n", (pStatusStruct.swVersion>>8) & 0xFF, 
            (pStatusStruct.swVersion & 0x00FF));
    printf("      Free Memory: %ld\n", pStatusStruct.freeMemory);
    printf("     Total Memory: %ld\n", pStatusStruct.totalMemory);
    printf("        PINs Used: %d\n", pStatusStruct.usedPINs);
    printf("        Keys Used: %d\n", pStatusStruct.usedKeys);
    printf("       Logged IDs: ");
    doLoggedID(pStatusStruct.loggedID);
	printf("\n");
#endif
	printf("GetStatus Successful\n");
end:
	return;
}

void doResume(const char *type)
{
	MSCLong32 rv = 0;
	MSCUChar8 temp[8];
	MSCULong32 retLength;
#ifdef VERSION2
	MSCUShort16 capability[4];
#endif
	MSCULong32 ulValue;
	MSCUChar8 ucValue;
	MSCUShort16 usValue;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	ulValue = 0;
#ifdef VERSION2

	if (type != NULL)
	{
		if (strcmp(type, "crypt") == 0)
		{
			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_SUPPORT_CRYPTOALG,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_SUPPORT_CRYPTOALG");
			usValue = getUShort16(temp);
			printf("\n");
			printf("Crypto Algorithm    Supported\n");
			printf("-------------------------------\n");
			printf("RSA                    ");
			PRINT_SUPPORT(usValue & MSC_SUPPORT_RSA);
			printf("DSA                    ");
			PRINT_SUPPORT(usValue & MSC_SUPPORT_DSA);
			printf("DES                    ");
			PRINT_SUPPORT(usValue & MSC_SUPPORT_DES);
			printf("3DES                   ");
			PRINT_SUPPORT(usValue & MSC_SUPPORT_3DES);
			printf("AES                    ");
			PRINT_SUPPORT(usValue & MSC_SUPPORT_AES);
			printf("EC                     ");
			PRINT_SUPPORT(usValue & MSC_SUPPORT_ECURVE);
			goto success;
		}
		else if (strcmp(type, "rsa") == 0)
		{
			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_RSA,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_RSA");
			_8ByteTo4Short(temp, capability);
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_LENGTHS];
			printf("%04lX\n",usValue);
			printf("\n");
			printf("RSA Capabilities                                      Supported\n");
			printf("---------------------------------------------------------------\n");
			printf("\n");
			printf("Possible Key Lengths:\n\n");
			{
				printf("512 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_512);
				printf("768 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_768);
				printf("1024 Bit                                                  ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_1024);
				printf("2048 Bit                                                  ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_2048);
				printf("512 Bit CRT                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_CRT_512);
				printf("768 Bit CRT                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_CRT_768);
				printf("1024 Bit CRT                                              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_CRT_1024);
				printf("2048 Bit CRT                                              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_RSA_CRT_2048);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_GENS];
			printf("\n");
			printf("Possible Key Generations:\n\n");
			{
				printf("512 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_512);
				printf("768 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_768);
				printf("1024 Bit                                                  ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_1024);
				printf("2048 Bit                                                  ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_2048);
				printf("512 Bit CRT                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_CRT_512);
				printf("768 Bit CRT                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_CRT_768);
				printf("1024 Bit CRT                                              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_CRT_1024);
				printf("2048 Bit CRT                                              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_RSA_CRT_2048);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_SIGNATURE_ALGS];
			printf("\n");
			printf("Possible Signature Algorithms:\n\n");
			{
				printf("Signature with MD5 Hash and PKCS#1 Padding                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_MD5_PKCS1);
				printf("Signature with MD5 Hash and RFC 2409 Padding              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_MD5_RFC2409);
				printf("Signature with RIPE MD-160 Hash and ISO 9796 Padding      ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_RIPEMD160_ISO9796);
				printf("Signature with RIPE MD-160 Hash and PKCS#1 Padding        ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_RIPEMD160_PKCS1);
				printf("Signature with SHA-1 Hash and ISO 9796 Padding            ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_SHA_ISO9796);
				printf("Signature with SHA-1 Hash and PKCS#1 Padding              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_SHA_PKCS1);
				printf("Signature with SHA-1 Hash and RFC 2409 Padding            ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_SHA_RFC2409);
				printf("Signature with SHA-1 Hash and PKCS#1-PSS Padding           ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_SHA_PKCS1_PSS);
				printf("Signature with MD5 Hash and PKCS#1-PSS Padding             ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_MD5_PKCS1_PSS);
				printf("Signature with RIPE MD-160 Hash and PKCS#1-PSS Padding     ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_RSA_RIPEMD160_PKCS1_PSS);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_CIPHER_ALGS];
			printf("\n");
			printf("Possible Cipher Algorithms:\n\n");
			{
				printf("Cipher with ISO 14888 Padding                             ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_CIPHER_RSA_ISO_14888)
				printf("Cipher with ISO 9796 Padding                              ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_CIPHER_RSA_ISO_9796);
				printf("Cipher with No Padding                                    ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_CIPHER_RSA_NOPAD);
				printf("Cipher with PKCS#1 Padding                                ");
				PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_RSA_PKCS1);
				printf("Cipher with PKCS#1-OAEP Padding                           ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_CIPHER_RSA_PKCS1_OAEP);
			}
			goto success;
		}
		else if (strcmp(type, "dsa") == 0)
		{
			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_DSA,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_DSA");
			_8ByteTo4Short(temp, capability);
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_LENGTHS];
			printf("\n");
			printf("DSA Capabilities                                      Supported\n");
			printf("---------------------------------------------------------------\n");
			printf("\n");
			printf("Possible Key Lengths:\n\n");
			{
				printf("512 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_DSA_512);
				printf("768 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_DSA_768);
				printf("1024 Bit                                                  ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_DSA_1024);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_GENS];
			printf("\n");
			printf("Possible Key Generations:\n\n");
			{
				printf("512 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_DSA_512);
				printf("768 Bit                                                   ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_DSA_768);
				printf("1024 Bit                                                  ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_DSA_1024);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_SIGNATURE_ALGS];
			printf("\n");
			printf("Possible Signature Algorithms:\n\n");
			{
				printf("Signature with SHA-1 Hash                                 ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_DSA_SHA);
				printf("Signature with Elliptic Curves Key and SHA-1 Hash         ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_ECDSA_SHA);
			}
			goto success;
		}
		else if (strcmp(type, "ec") == 0)
		{
			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_ECURVE,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_ECURVE");
			_8ByteTo4Short(temp, capability);
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_LENGTHS];
			printf("\n");
			printf("EC Capabilities                                       Supported\n");
			printf("---------------------------------------------------------------\n");
			printf("\n");
			printf("Possible Key Lengths:\n\n");
			{
				printf("112 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_FP_112);
				printf("113 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_F2M_113);
				printf("128 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_FP_128);
				printf("131 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_F2M_131);
				printf("160 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_FP_160);
				printf("163 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_F2M_163);
				printf("192 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_FP_192);
				printf("193 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_EC_F2M_193);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_GENS];
			printf("\n");
			printf("Possible Key Generations:\n\n");
			{
				printf("112 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_FP_112);
				printf("113 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_F2M_113);
				printf("128 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_FP_128);
				printf("131 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_F2M_131);
				printf("160 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_FP_160);
				printf("163 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_F2M_163);
				printf("192 Bit FP                                                ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_FP_192);
				printf("193 Bit F2M                                               ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_KEYGEN_EC_F2M_193);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_SIGNATURE_ALGS];
			printf("\n");
			printf("Possible Signature Algorithms:\n\n");
			{
				printf("Signature with SHA-1 Hash                                 ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_DSA_SHA);
				printf("Signature with Elliptic Curves Key and SHA-1 Hash         ");
				PRINT_SUPPORT(usValue &
					MSC_CAPABLE_SIGNATURE_ECDSA_SHA);
			}
			goto success;
		}
		else if (strcmp(type, "aes") == 0)
		{
			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_AES,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_AES");
			_8ByteTo4Short(temp, capability);
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_LENGTHS];
			printf("\n");
			printf("AES Capabilities                                      Supported\n");
			printf("---------------------------------------------------------------\n");
			printf("\n");
			printf("Possible Key Lengths:\n\n");
			{
			printf("128 Bit                                                   ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_AES_128);
			printf("192 Bit                                                   ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_AES_192);
			printf("256 Bit                                                   ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_AES_256);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_SIGNATURE_ALGS];
			printf("\n");
			printf("Possible Signature Algorithms:\n\n");
			{
			printf("Signature AES with blocksize 128 in CBC mode              ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_AES_MAC_128_NOPAD);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_CIPHER_ALGS];
			printf("\n");
			printf("Possible Cipher Algorithms:\n\n");
			{
			printf("Cipher with block size 128 in CBC mode with no padding    ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_AES_BLOCK_128_CBC_NOPAD);
			printf("Cipher with block size 128 in ECB mode with no padding    ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_AES_BLOCK_128_ECB_NOPAD);
			}
			goto success;
		}
		else if ((strcmp(type, "des") == 0) ||(strcmp(type, "3des") == 0))
		{
			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_DES,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_RSA");
			_8ByteTo4Short(temp, capability);
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_KEY_LENGTHS];
			printf("\n");
			printf("DES Capabilities                                      Supported\n");
			printf("---------------------------------------------------------------\n");
			printf("\n");
			printf("Possible Key Lengths:\n\n");
			{
			printf("64  Bit                                                   ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_DES);
			printf("128 Bit                                                   ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_3DES_2KEY);
			printf("192 Bit                                                   ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_3DES_3KEY);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_SIGNATURE_ALGS];
			printf("\n");
			printf("Possible Signature Algorithms:\n\n");
			{
			printf("Signature 4 Byte MAC with ISO 9797 Method 1 Padding       ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_DES_MAC4_ISO9797_M1);
			printf("Signature 4 Byte MAC with ISO 9797 Method 2 Padding       ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_DES_MAC4_ISO9797_M2);
			printf("Signature 4 Byte MAC with No Padding                      ");
			PRINT_SUPPORT(usValue &
			MSC_CAPABLE_SIGNATURE_DES_MAC4_NOPAD);
			printf("Signature 4 Byte MAC with PKCS#5 Padding                  ");
			PRINT_SUPPORT(usValue &
			MSC_CAPABLE_SIGNATURE_DES_MAC4_PKCS5);
			printf("Signature 8 Byte MAC with ISO 9797 Method 1 Padding       ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_DES_MAC8_ISO9797_M1);
			printf("Signature 8 Byte MAC with ISO 9797 Method 2 Padding       ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_DES_MAC8_ISO9797_M2);
			printf("Signature 8 Byte MAC with No Padding                      ");
			PRINT_SUPPORT(usValue &
			MSC_CAPABLE_SIGNATURE_DES_MAC8_NOPAD);
			printf("Signature 8 Byte MAC with PKCS#5 Padding                  ");
			PRINT_SUPPORT(usValue &
			MSC_CAPABLE_SIGNATURE_DES_MAC8_PKCS5);
			printf("Signature 4 Byte MAC with ISO 9797-1 Alg. 3 Meth. 2       ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_DES_MAC4_ISO9797_1_M2_ALG3);
			printf("Signature 8 Byte MAC with ISO 9797-1 Alg. 3 Meth. 2       ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_SIGNATURE_DES_MAC8_ISO9797_1_M2_ALG3);
			}
			usValue = capability[MSC_ALGORITHM_CAPABILITIES_OFFSET_POSSIBLE_CIPHER_ALGS];
			printf("\n");
			printf("Possible Cipher Algorithms:\n\n");
			{
			printf("Cipher with CBC and ISO 9797 Method 1 Padding             ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_CBC_ISO9797_M1)
			printf("Cipher with CBC and ISO 9797 Method 2 Padding             ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_CBC_ISO9797_M2)
			printf("Cipher with CBC and No Padding                            ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_CBC_NOPAD)
			printf("Cipher with CBC and PKCS#5 Padding                        ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_CBC_PKCS5)
			printf("Cipher with ECB and ISO 9797 Method 1 Padding             ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_ECB_ISO9797_M1)
			printf("Cipher with ECB and ISO 9797 Method 2 Padding             ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_ECB_ISO9797_M2)
			printf("Cipher with ECB and No Padding                            ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_ECB_NOPAD)
			printf("Cipher with ECB and PKCS#5 Padding                        ");
			PRINT_SUPPORT(usValue &
				MSC_CAPABLE_CIPHER_DES_ECB_PKCS5)
			}
			goto success;
		}
		else if (strcmp(type, "keys") == 0)
		{
			retLength = 8;
			printf("\n");
			printf("AUT needed to import/generate keys:\n    ");

			usValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_KEY_AUTH,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_KEY_AUTH");
			usValue = getUShort16(temp);
			doPrintAUT(usValue);
			printf("\n");

			goto success;
		}
		else if (strcmp(type, "pins") == 0)
		{
			retLength = 8;
			printf("\n");
			printf("PIN Policies\n");
			printf("-------------------------------\n");
			printf("\n");
			ucValue = 0;
			printf("Maximum PIN size         ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXSIZE,
				(MSCPUChar8) & ucValue, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_MAXSIZE");

			printf("%03d\n", ucValue);

			retLength = 8;
			ucValue = 0;
			printf("Minimum PIN size         ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MINSIZE,
				(MSCPUChar8) & ucValue, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_MINSIZE");

			printf("%03d\n", ucValue);

			retLength = 8;
			ucValue = 0;
			printf("Maximum number of PINs   ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXNUM,
				(MSCPUChar8) & ucValue, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_MAXNUM");

			printf("%03d\n", ucValue);

			retLength = 8;
			ucValue = 0;
			printf("Unblock PIN resets PIN   ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_ATTR,
				(MSCPUChar8) & ucValue, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_ATTR");

			if (ucValue & MSC_CAPABLE_PIN_RESET)
				printf("YES\n");
			else
				printf(" NO\n");

			printf("Unblock PIN leaves PIN   ");
			if (ucValue & MSC_CAPABLE_PIN_LEAVE)
				printf("YES\n");
			else
				printf(" NO\n");

			retLength = 8;
			printf("AUT needed to create pins:\n    ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_AUTH,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_AUTH");
			usValue = getUShort16(temp);
			doPrintAUT(usValue);
			printf("\n");

			goto success;
		}
		else if (strcmp(type, "objects") == 0)
		{
			retLength = 8;
			printf("\n");
			printf("Object Capabilities\n");
			printf("-------------------------------\n");

			ucValue = 0;
			printf("\n");
			printf("Maximum Object ID size   ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_IDSIZE,
				(MSCPUChar8) & ucValue, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_OBJ_IDSIZE");

			printf("%03d\n", ucValue);

			retLength = 8;
			printf("Maximum number objects   ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_MAXNUM,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_OBJ_AUTH");
			usValue = getUShort16(temp);

			printf("%03ld\n", usValue);

			retLength = 8;
			printf("AUT needed to create objects:\n    ");
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_AUTH,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_OBJ_AUTH");
			usValue = getUShort16(temp);
			doPrintAUT(usValue);
			printf("\n");

			goto success;
		}
	}
#else
	if (type != NULL)
	{
		if (strcmp(type, "crypt") == 0)
		{
			retLength = 8;
			ulValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_SUPPORT_CRYPTOALG,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_SUPPORT_CRYPTOALG");
			ulValue = getULong32(temp);
			printf("Crypto Algorithm    Supported\n");
			printf("-------------------------------\n");
			printf("RSA                    ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_RSA);
			printf("DSA                    ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_DSA);
			printf("ELGAMAL                ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_ELGAMAL);
			printf("Elliptic Curves        ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_ECURVE);
			printf("DES                    ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_DES);
			printf("3DES                   ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_3DES);
			printf("IDEA                   ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_IDEA);
			printf("AES                    ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_AES);
			printf("BLOWFISH               ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_BLOWFISH);
			printf("TWOFISH                ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_TWOFISH);
			printf("SHA1                   ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_SHA1);
			printf("MD5                    ");
			PRINT_SUPPORT(ulValue & MSC_SUPPORT_MD5);
			goto success;
		}
		else if (strcmp(type, "rsa") == 0)
		{
			retLength = 8;
			ulValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_RSA,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_RSA");
			ulValue = getULong32(temp);
			printf("RSA Capabilities     Supported\n");
			printf("-------------------------------\n");
			printf("512 Bit                ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_512);
			printf("768 Bit                ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_768);
			printf("1024 Bit               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_1024);
			printf("2048 Bit               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_2048);
			printf("4096 Bit               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_4096);
			printf("Key Generation         ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_KEYGEN);
			printf("No Padding             ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_NOPAD);
			printf("PKCS#1 Padding         ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_PKCS1);
			goto success;
		}
		else if (strcmp(type, "aes") == 0) {
			printf("Not supported in this MuscleTool version.\n");
			goto success;
		}
		else if (strcmp(type, "dsa") == 0)
		{
			retLength = 8;
			ulValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_DSA,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_DSA");
			ulValue = getULong32(temp);
			printf("DSA Capabilities     Supported\n");
			printf("-------------------------------\n");
			printf("512 Bit                ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_512);
			printf("768 Bit                ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_768);
			printf("1024 Bit               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_1024);
			printf("2048 Bit               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_2048);
			printf("4096 Bit               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_4096);
			printf("Key Generation         ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_KEYGEN);
			goto success;
		}
		else if (strcmp(type, "des") == 0)
		{
			retLength = 8;
			ulValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_DES,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_DES");
			ulValue = getULong32(temp);
			printf("DES Capabilities     Supported\n");
			printf("-------------------------------\n");
			printf("CBC Mode               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_CBC);
			printf("ECB Mode               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_ECB);
			printf("Key Generation         ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_KEYGEN);
			goto success;
		}
		else if (strcmp(type, "3des") == 0)
		{
			retLength = 8;
			ulValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_3DES,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_3DES");
			ulValue = getULong32(temp);
			printf("3DES Capabilities     Supported\n");
			printf("-------------------------------\n");
			printf("CBC Mode               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_CBC);
			printf("ECB Mode               ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_ECB);
			printf("3 Key Mode             ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_3KEY);
			printf("Key Generation         ");
			PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_KEYGEN);
			goto success;
		}
		else if (strcmp(type, "keys") == 0)
		{
			retLength = 8;
			printf("AUT needed to import/generate keys:\n    ");

			usValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_KEY_AUTH,
				temp, &retLength);
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_KEY_AUTH");
			usValue = getUShort16(temp);
			doPrintAUT(usValue);
			printf("\n");

			goto success;
		}
		else if (strcmp(type, "pins") == 0)
		{
			retLength = 8;
			ucValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXSIZE,
				(MSCPUChar8) & ucValue, &retLength);

			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_MAXSIZE");

			printf("PIN Policies\n");
			printf("-------------------------------\n");
			printf("Maximum PIN size         %03d\n", ucValue);

			retLength = 8;
			ucValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MINSIZE,
				(MSCPUChar8) & ucValue, &retLength);

			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_MINSIZE");

			printf("Minimum PIN size         %03d\n", ucValue);

			retLength = 8;
			ucValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXNUM,
				(MSCPUChar8) & ucValue, &retLength);

			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_MAXNUM");

			printf("Maximum number of PINs   %03d\n", ucValue);

			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_ATTR,
				temp, &retLength);

			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_ATTR");
			ulValue = getULong32(temp);
			printf("Unblock PIN resets PIN   ");
			if (ulValue & MSC_CAPABLE_PIN_RESET)
				printf("YES\n");
			else
				printf(" NO\n");

			printf("Unblock PIN leaves PIN   ");
			if (ulValue & MSC_CAPABLE_PIN_LEAVE)
				printf("YES\n");
			else
				printf(" NO\n");

			printf("AUT needed to create pins:\n    ");

			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_AUTH,
				temp, &retLength);
			
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_PIN_AUTH");
			usValue = getUShort16(temp);
			doPrintAUT(usValue);
			printf("\n");

			goto success;
		}
		else if (strcmp(type, "objects") == 0)
		{
			retLength = 8;
			ucValue = 0;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_IDSIZE,
				(MSCPUChar8) & ucValue, &retLength);

			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_OBJ_IDSIZE");

			printf("Object Capabilities\n");
			printf("-------------------------------\n");
			printf("Maximum Object ID size   %03d\n", ucValue);

			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_MAXNUM,
				temp, &retLength);

			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_OBJ_MAXNUM");
			ulValue = getULong32(temp);

			printf("Maximum number objects   %03ld\n", ulValue);

			printf("AUT needed to create objects:\n    ");

			retLength = 8;
			rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_AUTH,
				temp, &retLength);
			
			CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_CAPABLE_OBJ_AUTH");
			usValue = getUShort16(temp);
			doPrintAUT(usValue);
			printf("\n");

			goto success;
		}
	}

#endif
	else {
#ifdef VERSION2
		retLength = 8;
		rv = MSCGetCapabilities(pConnection, MSC_TAG_SUPPORT_FUNCTIONS,
			temp, &retLength);
		CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_SUPPORT_FUNCTIONS");
		_8ByteTo4Short(temp, capability);
		printf("\n");
		printf("Functions             Supported\n");
		printf("-------------------------------\n");
		printf("MSCGenerateKeys          ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_GENKEYS);
		printf("MSCImportKey             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_IMPORTKEY);
		printf("MSCMoveKey               ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_MOVEKEY);
		printf("MSCChangeKeyACL          ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_CHANGE_KEY_ACL);
		printf("MSCExportKey             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_EXPORTKEY);
		printf("MSCComputeCrypt          ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_COMPUTECRYPT);
		printf("MSCListKeys              ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_LISTKEYS);
		printf("MSCDeleteKey             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_KEY_USE_AND_MANAGEMENT] & MSC_SUPPORT_DELETEKEY);
		printf("MSCExternalAuthenticate  ");

		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_EXTAUTH);
		printf("MSCCreatePIN             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_CREATEPIN);
		printf("MSCVerifyPIN             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_VERIFYPIN);
		printf("MSCChangePIN             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_CHANGEPIN);
		printf("MSCUnblockPIN            ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_UNBLOCKPIN);
		printf("MSCDeleteID              ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_DELETEID)
		printf("MSCLogoutAll             ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_LOGOUTALL);
		printf("MSCGetChallenge          ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_GETCHALLENGE);
		printf("MSCListPINs              ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_EXT_AUTH] & MSC_SUPPORT_LISTPINS);

		printf("MSCListObjects           ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_LISTOBJECTS);
		printf("MSCMoveObject            ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_MOVEOBJECT);
		printf("MSCChangeObjectACL       ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_CHANGE_OBJECT_ACL);
		printf("MSCCreateObject          ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_CREATEOBJECT);
		printf("MSCDeleteObject          ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_DELETEOBJECT);
		printf("MSCWriteObject           ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_WRITEOBJECT);
		printf("MSCReadObject            ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_OBJECT_USE_AND_MANGEMENT] & MSC_SUPPORT_READOBJECT);
		printf("MSCGetCapabilities       ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_STATUS] & MSC_SUPPORT_GETCAPABILITIES);
		printf("MSCGetState              ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_STATUS] & MSC_SUPPORT_GETSTATUS);
		printf("MSCSetState              ");
		PRINT_SUPPORT(capability[MSC_SUPPORTED_FUNCTIONS_OFFSET_STATUS] & MSC_SUPPORT_SETSTATUS);
#else
	retLength = 8;
	rv = MSCGetCapabilities(pConnection, MSC_TAG_SUPPORT_FUNCTIONS,
		temp, &retLength);
	CHECK_ERR(rv != MSC_SUCCESS, "Getting attribute TAG_SUPPORT_FUNCTIONS");
	ulValue = getULong32(temp);
	printf("Functions             Supported\n");
	printf("-------------------------------\n");
        printf("MSCGenerateKeys          ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_GENKEYS);
        printf("MSCImportKey             ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_IMPORTKEY);
        printf("MSCExportKey             ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_EXPORTKEY);
        printf("MSCComputeCrypt          ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_COMPUTECRYPT);
        printf("MSCExternalAuthenticate  ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_EXTAUTH);
        printf("MSCListKeys              ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_LISTKEYS);
        printf("MSCCreatePIN             ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_CREATEPIN);
        printf("MSCVerifyPIN             ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_VERIFYPIN);
        printf("MSCChangePIN             ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_CHANGEPIN);
        printf("MSCUnblockPIN            ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_UNBLOCKPIN);
        printf("MSCListPINs              ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_LISTPINS);
        printf("MSCCreateObject          ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_CREATEOBJECT);
        printf("MSCDeleteObject          ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_DELETEOBJECT);
        printf("MSCWriteObject           ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_WRITEOBJECT);
        printf("MSCReadObject            ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_READOBJECT);
        printf("MSCListObjects           ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_LISTOBJECTS);
        printf("MSCLogoutAll             ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_LOGOUTALL);
        printf("MSCGetChallenge          ");
        PRINT_SUPPORT(ulValue & MSC_SUPPORT_GETCHALLENGE);
#endif
        }

success:
	printf("GetCapabilities Successful\n");
end:
	return;
}

void doLogout(void)
{
	MSCLong32 rv;

	rv = MSCLogoutAll(pConnection);
	CHECK_ERR(rv != MSC_SUCCESS, "Logout Failed !");

	if (rv == MSC_UNSUPPORTED_FEATURE)
	{
		printf("ERR: Feature Unsupported\n");
	}
	else if (rv == MSC_SUCCESS)
	{
		printf("Logged out identities\n");
	}
	else
	{
		printf("ERR: Logout Failed !\n");
	}
	printf("Logout Successful\n");
end:
	return;
}

void doChallenge(int len)
{
	MSCLong32 rv = 0;
	MSCUShort16 challengeLength;
	MSCUChar8 randomData[1024];

	int i;

	challengeLength = 8;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	if (len > 0)
		challengeLength = len;

	if (challengeLength > sizeof(randomData))
	{
		printf("ERR: Invalid choice made !\n");
		return;
	}

	rv = MSCGetChallenge(pConnection, 0, 0, randomData, challengeLength);

	CHECK_ERR(rv != MSC_SUCCESS, "Challenge Failed !");

	for (i = 0; i < challengeLength; i++)
		printf("%02X ", randomData[i]);
	printf("\n");
	printf("Challenge Successful\n");
end:
	return;
}

void doVerify(int pinNumber)
{
	MSCLong32 rv = 0;
	MSCULong32 pinSize;
	int usePinpad = 0;
	MSCUChar8 pinData[17];
#ifdef VERSION2
	MSCULong32 dummy, units;
#endif

	CHECK_ERR(pConnection == 0, "Must Connect First !");

#ifdef VERSION2
	rv = MSCGetCapabilities(pConnection, MSC_TAG_FUNCTIONAL_UNITS, (MSCPUChar8)&units, &dummy);
	if (rv == MSC_SUCCESS) {
    if ((units & MSC_SUPPORT_PINPAD) != 0) {
			usePinpad = 1;
		}
	}
#endif
	if (usePinpad) {
		printf("Enter PIN at card terminal.\n");
		rv = MSCVerifyPIN(pConnection, (MSCUChar8) pinNumber,
			NULL, 0);
	}
	else {
        rv = getStringValue("Enter PIN", "Enter PIN", (char *)pinData, 17);
		if (rv == 0)
			goto abort;

                pinSize = strlen((char *)pinData);
		rv = MSCVerifyPIN(pConnection, (MSCUChar8) pinNumber,
			(MSCUChar8 *) pinData, pinSize);
	}
	CHECK_ERR(rv != MSC_SUCCESS, "VerifyPIN Failed !");
	printf("VerifyPIN Successful\n");
	goto end;
abort:
	printf("VerifyPIN Cancelled.\n");
end:
	return;
}

void doUnblockPIN(int pinNumber)
{
	MSCLong32 rv = 0;
	MSCULong32 pinSize;
	int usePinpad = 0;
	MSCUChar8 pinData[17];
#ifdef VERSION2
	MSCULong32 newPinSize;
	MSCULong32 dummy, units;
	MSCUChar8 newPinData[17];
#endif

	CHECK_ERR(pConnection == 0, "Must Connect First !");
#ifdef VERSION2
	rv = MSCGetCapabilities(pConnection, MSC_TAG_FUNCTIONAL_UNITS, (MSCPUChar8)&units, &dummy);
	if (rv == MSC_SUCCESS) {
    if ((units & MSC_SUPPORT_PINPAD) != 0) {
			usePinpad = 1;
		}
	}
#endif
	if (usePinpad) {
		printf("Unblock PIN at card terminal.\n");
#ifdef VERSION2
                rv = MSCUnblockPIN(pConnection, (MSCUChar8) pinNumber,
			NULL, 0, NULL, 0);
#else
                rv = MSCUnblockPIN(pConnection, (MSCUChar8) pinNumber,
                                   NULL, 0);
#endif
	}
	else {
        rv = getStringValue("Enter unblock PIN", "Enter PIN", (char *)pinData, 17);
		if (rv == 0)
			goto abort;
        pinSize = strlen((char *)pinData);
#ifdef VERSION2
        rv = getStringValue("Enter new PIN", "Enter PIN", (char *)newPinData, 17);
		if (rv == 0)
			goto abort;
        newPinSize = strlen((char *)newPinData);
		rv = MSCUnblockPIN(pConnection, (MSCUChar8) pinNumber,
			(MSCUChar8 *) pinData, pinSize, newPinData, newPinSize);
#else
		rv = MSCUnblockPIN(pConnection, (MSCUChar8) pinNumber,
			(MSCUChar8 *) pinData, pinSize);
#endif
	}

	CHECK_ERR(rv != MSC_SUCCESS, "UnblockPIN Failed !");
	printf("Unblock PIN Unblock Successful\n");
        goto end;
abort:
	printf("User aborted UnblockPIN\n");
end:
	return;
}

void doMakePIN(int pinNumber, char *pinData, char *pinUnblockData)
{
	MSCLong32 rv = 0;
	MSCULong32 pinSize;
	MSCULong32 pinUnblockSize;
#ifdef VERSION2
	MSCULong32 dummy;
	MSCUChar8 pinMaxSize = 16;
#endif
	MSCUChar8 pinTries = MUSCLETOOL_PIN_MAXTRIES;

	CHECK_ERR(pConnection == 0, "Must Connect First !");
	pinSize = strlen(pinData);
	pinUnblockSize = strlen(pinUnblockData);

#ifdef VERSION2
	rv = MSCGetCapabilities(pConnection,  MSC_TAG_CAPABLE_PIN_TRIES,
				(MSCPUChar8) &pinTries, &dummy);

	if (rv != MSC_SUCCESS)
	{
		printf("Can't get PIN tries from card - using default of %d\n", MUSCLETOOL_PIN_MAXTRIES);
		pinTries = MUSCLETOOL_PIN_MAXTRIES;
	}

	rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXSIZE,
			(MSCPUChar8) &pinMaxSize, &dummy);
	if (rv != MSC_SUCCESS)
	{
		printf("Can't get maximal PIN size from card - using default of 16\n");
		pinMaxSize = 16;
	}
#endif
	rv = MSCCreatePIN(pConnection,
		(MSCUChar8) pinNumber, pinTries,
		(MSCUChar8 *) pinData, (MSCUChar8)pinSize,
		(MSCUChar8 *) pinUnblockData, (MSCUChar8)pinUnblockSize);

	CHECK_ERR(rv != MSC_SUCCESS, "CreatePIN Failed !");
	printf("PIN Create Successful\n");
end:
	return;
}

void doChangePIN(int pinNumber)
{
	MSCLong32 rv = 0;
	MSCULong32 pinSize;
	MSCULong32 newPinSize;
	MSCUChar8 pinData[17], newPinData[17];
#ifdef VERSION2
	MSCULong32 dummy, units;
#endif
	MSCUChar8 pinMaxSize = 16;
	int usePinpad = 0;
	char dummyString[32];

	CHECK_ERR(pConnection == 0, "Must Connect First !");
#ifdef VERSION2
	rv = MSCGetCapabilities(pConnection, MSC_TAG_FUNCTIONAL_UNITS, (MSCPUChar8)&units, &dummy);
	if (rv == MSC_SUCCESS) {
    if (((units & MSC_SUPPORT_PINPAD) != 0) && (units & MSC_SUPPORT_DISPLAY) != 0) {
			usePinpad = 1;
		}
	}
#endif
	if (usePinpad) {
		printf("Change PIN at card terminal.\n");
		rv = MSCChangePIN(pConnection, (MSCUChar8) pinNumber,
			NULL, 0, NULL, 0);
	}
	else {
#ifdef VERSION2
		rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXSIZE,
				(MSCPUChar8) &pinMaxSize, &dummy);
		if (rv != MSC_SUCCESS)
		{
			printf("Can't get maximal PIN size from card - using default of 16\n");
			pinMaxSize = 16;
		}
#endif
		rv = getStringValue("Enter old PIN", "Enter PIN", (char *)pinData, 17);
		if (rv == 0)
			goto abort;

                pinSize = strlen((char *)pinData);

		snprintf(dummyString, 32, "Enter new PIN (max %d)", pinMaxSize);
                rv = getStringValue(dummyString, "Enter PIN", (char *)newPinData, 17);
		if (rv == 0)
			goto abort;
		
                newPinSize = strlen((char *)newPinData);
		
		rv = MSCChangePIN(pConnection, (MSCUChar8) pinNumber,
			(MSCUChar8 *) pinData, (MSCUChar8)pinSize, (MSCUChar8 *) newPinData, (MSCUChar8)newPinSize);
	}

	CHECK_ERR(rv != MSC_SUCCESS, "ChangePIN Failed !");
	printf("ChangePIN Successful\n");
	goto end;
abort:
	printf("ChangePIN Cancelled\n");
end:
	return;
}

void doListPINs(void)
{
	MSCLong32 rv = 0;
	MSCUShort16 pinStack;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCListPINs(pConnection, &pinStack);
	CHECK_ERR(rv != MSC_SUCCESS, "ListPINs Failed !");
	printf("   ");
	doLoggedID(pinStack);
	printf("\n");
	printf("ListPINs Successful\n");
end:
	return;
}

void doCreate(char *objectID, int objectSize)
{
	MSCLong32 rv = 0;
	MSCObjectACL objectACL;

	unsigned long readPerm = 0, writePerm = 0, deletePerm = 0;

	printf("doCreate(%s, %d)\n", objectID, objectSize);
	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = getNumberValue("Enter the ACL for reading the object\nExample: 0xFFFF for all users", "Enter ACL mask", &readPerm);
	objectACL.readPermission = (MSCUShort16)readPerm;
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the ACL for writing the object\nExample: 0x0002 for user PIN #1", "Enter ACL mask", &writePerm);
	objectACL.writePermission = (MSCUShort16)writePerm;
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the ACL for deleting the object\nExample: 0x0002 for user PIN #1", "Enter ACL mask", &deletePerm);
	objectACL.deletePermission = (MSCUShort16)deletePerm;
	if (rv == 0)
		goto abort;

	rv = MSCCreateObject(pConnection, objectID, objectSize, &objectACL);

	CHECK_ERR(rv != MSC_SUCCESS, "CreateObject Failed !");
	printf("CreateObject Successful\n");
	goto end;
abort:
	printf("CreateObject Cancelled.\n");
end:
	return;
}

void doDelete(char *objectID)
{
	MSCLong32 rv = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

#ifdef VERSION2
	rv = MSCDeleteObject(pConnection, objectID);
#else
	rv = MSCDeleteObject(pConnection, objectID, MSC_ZF_WRITE_ZERO);
#endif
	CHECK_ERR(rv != MSC_SUCCESS, "DeleteObject Failed !");
	printf("DeleteObject Successful\n");
end:
	return;
}

static void myCallback(void *randValue, int pctDone)
{
	int intPct;
	double dbPct;

	dbPct = (double) pctDone / (double) 1000;

	intPct = (int) (dbPct * 100);

	putc('\b', stdout);
	putc('\b', stdout);
	putc('\b', stdout);
	putc('\b', stdout);
	printf("%03d%%", intPct);

	fflush(stdout);
}

void doRead(char *objectID, char *fileName)
{
	MSCLong32 rv = 0;
	MSCPUChar8 objectData = NULL;
	MSCULong32 dataSize;
	FILE *fp = NULL;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	printf("Percentage read:     ");

	rv = MSCReadAllocateObject(pConnection, objectID,
		&objectData, &dataSize, (LPRWEventCallback) myCallback, NULL);

	printf("\n");
	CHECK_ERR(rv != MSC_SUCCESS, "ReadObject Failed !");

	fp = fopen(fileName, "w+");

	if (fp == NULL)
	{
		printf("ERR: Open file %s failed\n", fileName);
		return;
	}

	fwrite(objectData, 1, dataSize, fp);

	printf("ReadObject Successful\n");
end:
	if (fp)
		fclose(fp);
	if (objectData)
		free(objectData);
	return;
}

void doWrite(char *fileName, char *objectID)
{
	MSCLong32 rv = 0;
	MSCPUChar8 objectData = NULL;
	MSCULong32 dataSize;
	struct stat fileStat;
	FILE *fp = NULL;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	printf("\n");

	fp = fopen(fileName, "r");

	if (fp == NULL)
	{
		printf("ERR: Open file %s failed\n", fileName);
		goto end;
	}

	if (stat(fileName, &fileStat) != 0)
	{
		printf("ERR: Failed to stat file\n");
		goto end;
	}

	dataSize = fileStat.st_size;
	objectData = malloc(dataSize);
	if (!objectData)
	{
		printf("ERR: Malloc failed\n");
		goto end;
	}

	fread(objectData, 1, dataSize, fp);

	printf("Percentage written:     ");

	rv = MSCWriteObject(pConnection, objectID,
		0, objectData, dataSize, (LPRWEventCallback) myCallback, NULL);

	printf("\n");
	CHECK_ERR(rv != MSC_SUCCESS, "WriteObject Failed !");

	printf("WriteObject Successful\n");
end:
	if (fp)
		fclose(fp);
	if (objectData)
		free(objectData);
	return;
}

/**
* Retrieves a string value from the user. The functions loops until a valid
* input is entered. The header is prefixing the query.
* The footer is displaying the footer message and querying the user to enter a number.
* A message indicates that 'a' aborts the query.
* The entered string can have a length of 4095 characters.
* \param header The message displayed to explain the number value.
* \param footer The message displayed to query the user.
* \param stringValue The returned string.
* \param stringValueLength The length of the stringValue buffer.
* \return 1 for success. 0 means that the user has canceled.
*/
static int getStringValue(const char *header, const char *footer, char *stringValue, int stringValueLength) {
	char stringString[4096];
	stringValue[0] = '\0';
	printf("\n");
	printf("%s\n", header);
	printf("'a' aborts this query.\n");
	while (1) {
		printf("%s : ", footer);
		fflush(stdin);
		scanf("%4095s", stringString);
		if (strlen(stringString) != 0)
			break;
		printf("Invalid string entered!\n");
	}
	if (strlen(stringString) == 1 && stringString[0] == 'a') {
		return 0;
	}
	strncpy(stringValue, stringString, stringValueLength-1);
	stringValue[stringValueLength-1] = '\0';
	return 1;
}

/**
* Retrieves a unsigned number value from the user. The functions loops until a valid
* input is entered. The header is prefixing the query.
* The input can be in decimal, octal or hexadecimal format within the range of an unsigned long.
* The footer is displaying the footer message and querying the user to enter a number.
* A message indicates that 'a' aborts the query.
* \param header The message displayed to explain the number value.
* \param footer The message displayed to query the user.
* \param numberValue The returned unsigned number.
* \return 1 for success. 0 means that the user has canceled.
*/
static int getNumberValue(const char *header, const char *footer, unsigned long *numberValue) {
	unsigned long num;
	char numberString[256];
	char *endChar;
	*numberValue = 0;
	printf("\n");
	printf("%s\n", header);
	printf("'a' aborts this query.\n");
	while (1) {
		printf("%s : ", footer);
		fflush(stdin);
		scanf("%255s", numberString);
		if (strlen(numberString) == 0) {
			goto error;
		}
		if (strlen(numberString) == 1 && numberString[0] == 'a')
			break;
		endChar = numberString;
		num = strtoul(numberString, &endChar, 0);
		if (endChar[0] == '\0') {
			break;
		}
error:
		printf("Invalid number entered!\n");
	}
	if (strlen(numberString) == 1 && numberString[0] == 'a')
		return 0;
	*numberValue = num;
	return 1;
}

/**
* Displays a list of choices. The header is prefixing the choices.
* Each choice is a string and is prefixed with the number of its position
* in the choice list. The footer is displaying the footer message
* and querying the user to choose between 1 and the number of choices.
* A message indicates that 0 aborts the selection. The number of choices is limited to 999.
* \param header The message displayed to explain the following choices.
* \param footer The message displayed to query the user.
* \param choices List for the choices.
* \param numChoices Number of choices.
* \return The number in the list of the chosen choice. 0 means that the user has canceled. -1 if an error occured.
*/
static int getChoice(const char *header, const char *footer, const char **choices, int numChoices) {
	char chosen[4];
	int num, i;
	printf("\n");
	if (numChoices > 999) {
		printf("Too many choices.\n");
		return -1;
	}
	printf("%s\n", header);
	printf("0. Abort this selection.\n");
	for (i=0; i < numChoices; i++) {
		printf("%d. %s\n", i+1, choices[i]);
	}
	printf("\n");
	while (1) {
		printf("%s (0-%d): ", footer, numChoices);
		fflush(stdin);
		scanf("%3s", chosen);
		num = atoi(chosen);
		if (!(num < 0 || num > numChoices))
			break;
		else
			printf("Invalid choice made!\n");
	}
	return num;
}

void doCrypt(int keyNum, char *inFileName, char *outFileName)
{
	MSCLong32 rv = 0;
	MSCChar8 hexString[4096];
	MSCUChar8 *inCryptData = NULL;
	MSCUChar8 *outCryptData = NULL;
	MSCULong32 inDataSize=0;
	MSCULong32 outDataSize=0;
	MSCCryptInit cryptInit;
	MSCKeyInfo keyInfo;
	MSCULong32 keySize;
	MSCULong32 dummySize = 0;
	MSCUChar8 keyType;
	MSCULong32 i=0;
	MSCULong32 inputBlockSize = 0;
	FILE *inFile = NULL;
	FILE *outFile = NULL;
	int choice=0;

#ifdef VERSION2
	const char *sved[4] = {"Sign", "Verify", "Encrypt", "Decrypt"};
#endif
	const char *s [1] = {"Sign"};
	const char *v [1] = {"Verify"};
	const char *sd [2] = {"Sign", "Decrypt"};
	const char *ve [2] = {"Verify", "Encrypt"};
#ifndef VERSION2
	const char *ed [2] = {"Encrypt", "Decrypt"};
#endif
#ifdef VERSION2
	const char *desCiph [8] = {"DES in CBC mode with no padding",
	"DES in ECB mode with no padding",
	"DES in CBC mode with ISO 9797 method 1 padding",
	"DES in CBC mode with ISO 9797 method 2 padding",
	"DES in ECB mode with ISO 9797 method 1 padding",
	"DES in ECB mode with ISO 9797 method 2 padding",
	"DES in CBC mode with PKCS#5 padding",
	"DES in ECB mode with PKCS#5 padding"
	};
#else
	const char *desCiph [2] = {"DES in CBC mode with no padding",
	"DES in ECB mode with no padding",
	};
#endif
#ifdef VERSION2
	const char *rsaCiph [5] = {"RSA with no padding",
	"RSA with PKCS#1 padding",
	"RSA with ISO 9796 padding",
	"RSA with ISO 14888 padding",
	"RSA with PKCS#1-OAEP padding"};
#else
	const char *rsaCiph [2] = {"RSA with no padding",
  "RSA with PKCS#1 padding"};
#endif

	const char *dsaSign [1] = {"DSA with SHA-1 hash"};
#ifdef VERSION2
	const char *aesSign [1] = {"AES with blocksize 128 in CBC mode"};
	const char *aesCiph [2] = {"AES with block size 128 in CBC mode with no padding",
	"AES with block size 128 in ECB mode with no padding"};
	const char *ecSign [1] = {"ECDSA with SHA-1 hash"};
#endif
#ifdef VERSION2
	const char *rsaSign [10] = {"RSA with MD5 hash and PKCS#1 padding",
		"RSA with SHA-1 hash and PKCS#1 padding",
		"RSA with SHA-1 hash and ISO 9796 padding",
		"RSA with RIPE MD-160 hash and ISO 9796 padding",
		"RSA with RIPE MD-160 hash and PKCS#1 padding",
		"RSA with SHA-1 hash and RFC2409 padding",
		"RSA with MD5 hash and RFC 2409 padding",
		"RSA with MD5 hash and PKCS#1-PSS padding",
		"RSA with RIPE MD-160 hash PKCS#1-PSS padding",
		"RSA with SHA-1 hash and PKCS#1-PSS padding"
	};
#else
	const char *rsaSign [1] = {"RSA with MD5 hash and PKCS#1 padding",
	};
#endif
#ifdef VERSION2
	const char *desSign [10] = {"DES 4 byte MAC with no padding",
		"DES 4 short MAC with ISO 9797 method 1 padding",
		"DES 4 short MAC with ISO 9797 method 2 padding",
		"DES 4 short MAC mode with PKCS#5 padding",
		"DES 8 short MAC with no padding",
		"DES 8 short MAC with ISO 9797 method 1 padding",
		"DES 8 short MAC with ISO 9797 method 2 padding",
		"DES 8 short MAC mode with PKCS#5 padding",
		"4-byte MAC using a 2-key DES3 according to ISO9797-1 MAC algorithm 3 with method2",
		"8-byte MAC using a 2-key DES3 key according to ISO9797-1 MAC algorithm 3 with method2"
	};
#endif

	int ciphDir = 0;
	int ciphMode = 0;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	rv = MSCGetKeyAttributes(pConnection, (MSCUChar8)keyNum, &keyInfo);
	CHECK_ERR(rv != MSC_SUCCESS, "GetKeyAttributes Failed !");

	keySize = keyInfo.keySize;
	keyType = keyInfo.keyType;

	switch (keyType) {
		case MSC_KEY_3DES:
		case MSC_KEY_3DES3:
		case MSC_KEY_DES:
#ifdef VERSION2	
        case MSC_KEY_AES:
#endif
#ifdef VERSION2
			choice = getChoice("Would you like to: ", "Choose", sved, 4);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_SIGN;
				break;
				case 2:
				ciphDir = MSC_DIR_VERIFY;
				break;
				case 3:
				ciphDir = MSC_DIR_ENCRYPT;
				break;
				case 4:
				ciphDir = MSC_DIR_DECRYPT;
				break;
				default: goto abort;
			}
#else
			choice = getChoice("Would you like to: ", "Choose", ed, 2);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_ENCRYPT;
				break;
				case 2:
				ciphDir = MSC_DIR_DECRYPT;
				break;
				default: goto abort;
			}
#endif
			break;
#ifdef VERSION2	
		case MSC_KEY_EC_F2M_PRIVATE:
		case MSC_KEY_EC_FP_PRIVATE:
#endif
		case MSC_KEY_DSA_PRIVATE:
			choice = getChoice("Would you like to: ", "Choose", s, 1);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_SIGN;
				break;
				default: goto abort;
			}
			break;
#ifdef VERSION2	
		case MSC_KEY_EC_F2M_PUBLIC:
		case MSC_KEY_EC_FP_PUBLIC:
#endif
		case MSC_KEY_DSA_PUBLIC:
			choice = getChoice("Would you like to: ", "Choose", v, 1);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_VERIFY;
				break;
				default: goto abort;
			}
			break;
		case MSC_KEY_RSA_PRIVATE:
			choice = getChoice("Would you like to: ", "Choose", sd, 2);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_SIGN;
				break;
				case 2:
				ciphDir = MSC_DIR_DECRYPT;
				break;
				default: goto abort;
			}
			break;
		case MSC_KEY_RSA_PUBLIC:
			choice = getChoice("Would you like to: ", "Choose", ve, 2);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_VERIFY;
				break;
				case 2:
				ciphDir = MSC_DIR_ENCRYPT;
				break;
				default: goto abort;
			}
			break;
		case MSC_KEY_RSA_PRIVATE_CRT:
			choice = getChoice("Would you like to: ", "Choose", sd, 2);
			switch (choice) {
				case 1:
				ciphDir = MSC_DIR_SIGN;
				break;
				case 2:
				ciphDir = MSC_DIR_DECRYPT;
				break;
				default: goto abort;
			}
			break;
	}

	if (inFileName == NULL) {
		if (ciphDir == MSC_DIR_VERIFY || ciphDir == MSC_DIR_SIGN) {
			rv = getStringValue("Please enter text to sign in hexadecimal ASCII (at most 1023 characters)\nExample: 30313233 for 0123\n", "Enter text", hexString, sizeof(hexString));
		}
		else if (ciphDir == MSC_DIR_ENCRYPT) {
			rv = getStringValue("Please enter text to encrypt in hexadecimal ASCII (at most 1023 characters)\nExample: 30313233 for 0123\n", "Enter text", hexString, sizeof(hexString));
		}
		else if (ciphDir == MSC_DIR_DECRYPT) {
			rv = getStringValue("Please enter text to decrypt in hexadecimal ASCII (at most 1023 characters)\nExample: 30313233 for 0123\n", "Enter text", hexString, sizeof(hexString));
		}
		if (rv == 0)
			goto abort;

		/* Calculating multiple of block size */

		dummySize = keySize;
		if (keySize % 8 != 0) {
			dummySize += (8-(keySize % 8));
		}
		inputBlockSize = (((strlen(hexString) / 2) * 8) + dummySize - (((strlen(hexString) / 2) * 8) % dummySize))/8;
		if ((((strlen(hexString) / 2) * 8) % dummySize) == 0)
			inputBlockSize-=(dummySize/8);

		inCryptData = malloc(sizeof(MSCUChar8)*inputBlockSize);
		/* I assume input block length = output block length */
		outCryptData = malloc(sizeof(MSCUChar8)*inputBlockSize);

		inDataSize = hexToBin((MSCUChar8 *)hexString, inCryptData);
		outDataSize = inputBlockSize;


		if (ciphDir == MSC_DIR_VERIFY) {
			rv = getStringValue("Please enter signature to verify in hexadecimal ASCII (at most 1023 characters)\nExample: 30313233 for 0123\n", "Enter text", hexString, sizeof(hexString));
			if (rv == 0)
				goto abort;
			if ((strlen(hexString)/2 + 1) > outDataSize) {
				outCryptData = realloc(outCryptData, sizeof(MSCUChar8)*(strlen(hexString)/2 + 1));
			}
                        outDataSize = hexToBin((MSCUChar8 *)hexString, outCryptData);
		}
	}
	else {
		long fileSize = 0;
		long read = 0;
		inFile = fopen(inFileName, "rb");
		if (inFile == NULL) {
			printf("ERR: Cannot open infile\n");
			goto end;
		}
#ifdef WIN32
		fileSize = _filelength(inFile->_file);
#else
		fileSize = fseek(inFile, 0, SEEK_END);
		if (fileSize == -1) {
			printf("ERR: Cannot seek in infile\n");
			goto end;
		}
		fileSize = ftell(inFile);
#endif
		/* Calculating multiple of block size */
		dummySize = keySize;
		if (keySize % 8 != 0) {
			dummySize += (keySize % 8);
		}
		inputBlockSize = (fileSize*8 + dummySize - ((fileSize*8) % dummySize))/8;
		if (((fileSize * 8) % dummySize) == 0)
			inputBlockSize-=(dummySize/8);

		inCryptData = malloc(sizeof(MSCUChar8)*inputBlockSize);
		read = fread(inCryptData, sizeof(MSCUChar8), fileSize, inFile);
		if (ferror(inFile)) {
			printf("ERR: Error reading file\n");
			goto end;
		}
		inDataSize = read;

		/* I assume input block length = output block length */
		outCryptData = malloc(sizeof(MSCUChar8)*inputBlockSize);
		outDataSize = inputBlockSize;

		if (ciphDir == MSC_DIR_VERIFY) {
			rv = getStringValue("Please enter signature to verify in hexadecimal ASCII (at most 1023 characters)\nExample: 30313233 for 0123\n", "Enter text", hexString, sizeof(hexString));
			if (rv == 0)
				goto abort;
			if ((strlen(hexString)/2 + 1) > outDataSize) {
				outCryptData = realloc(outCryptData, sizeof(MSCUChar8)*(strlen(hexString)/2 + 1));
			}
                        outDataSize = hexToBin((MSCUChar8 *)hexString, outCryptData);
		}
	}

	switch (ciphDir) {
		case MSC_DIR_ENCRYPT:
		case MSC_DIR_DECRYPT:
			switch (keyType) {
				case MSC_KEY_3DES:
				case MSC_KEY_3DES3:
				case MSC_KEY_DES:
#ifdef VERSION2
					choice = getChoice("Select the algorithm: ", "Choose", desCiph, 8);
					switch (choice) {
						case 1: ciphMode = MSC_MODE_CIPHER_DES_CBC_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						case 2: ciphMode = MSC_MODE_CIPHER_DES_ECB_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						case 3: ciphMode = MSC_MODE_CIPHER_DES_CBC_ISO9797_M1;
							break;
						case 4: ciphMode = MSC_MODE_CIPHER_DES_CBC_ISO9797_M2;
							break;
						case 5: ciphMode = MSC_MODE_CIPHER_DES_ECB_ISO9797_M1;
							break;
						case 6: ciphMode = MSC_MODE_CIPHER_DES_ECB_ISO9797_M2;
							break;
						case 7: ciphMode = MSC_MODE_CIPHER_DES_CBC_PKCS5;
							break;
						case 8: ciphMode = MSC_MODE_CIPHER_DES_ECB_PKCS5;
							break;
						default: goto abort;
					}
#else
					choice = getChoice("Select the algorithm: ", "Choose", desCiph, 8);
					switch (choice) {
						case 1: ciphMode = MSC_MODE_DES_CBC_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						case 2: ciphMode = MSC_MODE_DES_ECB_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						default: goto abort;
					}
#endif
					break;
#ifdef VERSION2
				case MSC_KEY_AES:
					choice = getChoice("Select the algorithm: ", "Choose", aesCiph, 2);
					switch (choice) {
						case 1:
							ciphMode = MSC_MODE_CIPHER_AES_BLOCK_128_CBC_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						case 2:
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							ciphMode = MSC_MODE_CIPHER_AES_BLOCK_128_ECB_NOPAD;
							break;
						default: goto abort;
							}
					break;
#endif
				case MSC_KEY_RSA_PRIVATE:
				case MSC_KEY_RSA_PRIVATE_CRT:
				case MSC_KEY_RSA_PUBLIC:
#ifdef VERSION2
					choice = getChoice("Select the algorithm: ", "Choose", rsaCiph, 5);
					switch (choice) {
						case 1: ciphMode = MSC_MODE_CIPHER_RSA_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							rv = padRSAData(inCryptData, inDataSize, inputBlockSize);
							if (rv == 0) {
								printf("ERR: Input data too large to be padded.");
								goto end;
							}
							inDataSize = inputBlockSize;
							break;
						case 2: ciphMode = MSC_MODE_CIPHER_RSA_PKCS1;
							break;
						case 3: ciphMode = MSC_MODE_CIPHER_RSA_ISO9796;
							break;
						case 4: ciphMode = MSC_MODE_CIPHER_RSA_ISO14888;
							break;
						case 5: ciphMode = MSC_MODE_CIPHER_RSA_PKCS1_OAEP;
							break;
						default:
							goto abort;
					}
					break;
#else
					choice = getChoice("Select the algorithm: ", "Choose", rsaCiph, 2);
					switch (choice) {
						case 1: ciphMode = MSC_MODE_RSA_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							rv = padRSAData(inCryptData, inDataSize, inputBlockSize);
							if (rv == 0) {
								printf("ERR: Input data too large to be padded.");
								goto end;
							}
							inDataSize = inputBlockSize;
							break;
						case 2: ciphMode = MSC_MODE_RSA_PAD_PKCS1;
							break;
						default:
							goto abort;
					}
					break;
#endif
			}
			break;
		case MSC_DIR_SIGN:
		case MSC_DIR_VERIFY:
			switch(keyType) {
#ifdef VERSION2
				case MSC_KEY_3DES:
				case MSC_KEY_3DES3:
				case MSC_KEY_DES:
					choice = getChoice("Select the algorithm: ", "Choose", desSign, 10);
					switch(choice) {
						case 1: ciphMode = MSC_MODE_SIGNATURE_DES_MAC4_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						case 2: ciphMode = MSC_MODE_SIGNATURE_DES_MAC4_ISO9797_M1;
							break;
						case 3: ciphMode = MSC_MODE_SIGNATURE_DES_MAC4_ISO9797_M2;
							break;
						case 4: ciphMode = MSC_MODE_SIGNATURE_DES_MAC4_PKCS5;
							break;
						case 5: ciphMode = MSC_MODE_SIGNATURE_DES_MAC8_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
						case 6: ciphMode = MSC_MODE_SIGNATURE_DES_MAC8_ISO9797_M1;
							break;
						case 7: ciphMode = MSC_MODE_SIGNATURE_DES_MAC8_ISO9797_M2;
							break;
						case 8: ciphMode = MSC_MODE_SIGNATURE_DES_MAC8_PKCS5;
							break;
						case 9: ciphMode = MSC_MODE_SIGNATURE_DES_MAC4_ISO9797_1_M2_ALG3;
							break;
						case 10: ciphMode = MSC_MODE_SIGNATURE_DES_MAC8_ISO9797_1_M2_ALG3;
							break;
						default:
							goto abort;
					}
					break;
				case MSC_KEY_AES:
					choice = getChoice("Select the algorithm: ", "Choose", aesSign, 1);
					switch (choice) {
						case 1:
							ciphMode = MSC_MODE_SIGNATURE_AES_MAC_128_NOPAD;
							/* Padding for algoritm with no padding
							inCryptData is a multiple of block size */
							padData(inCryptData, inDataSize, inputBlockSize);
							inDataSize = inputBlockSize;
							break;
							}
					break;
#endif
				case MSC_KEY_RSA_PRIVATE:
				case MSC_KEY_RSA_PRIVATE_CRT:
				case MSC_KEY_RSA_PUBLIC:
#ifdef VERSION2
					choice = getChoice("Select the algorithm: ", "Choose", rsaSign, 10);
					switch (choice) {
						case 1: ciphMode = MSC_MODE_SIGNATURE_RSA_MD5_PKCS1;
							break;
						case 2: ciphMode = MSC_MODE_SIGNATURE_RSA_SHA_PKCS1;
							break;
						case 3: ciphMode = MSC_MODE_SIGNATURE_RSA_SHA_ISO9796;
							break;
						case 4: ciphMode = MSC_MODE_SIGNATURE_RSA_RIPEMD160_ISO9796;
							break;
						case 5: ciphMode = MSC_MODE_SIGNATURE_RSA_RIPEMD160_PKCS1;
							break;
						case 6: ciphMode = MSC_MODE_SIGNATURE_RSA_SHA_RFC2409;
							break;
						case 7: ciphMode = MSC_MODE_SIGNATURE_RSA_MD5_RFC2409;
							break;
						case 8: ciphMode = MSC_MODE_SIGNATURE_RSA_MD5_PKCS1_PSS;
							break;
						case 9: ciphMode = MSC_MODE_SIGNATURE_RSA_RIPEMD160_PKCS1_PSS;
							break;
						case 10: ciphMode = MSC_MODE_SIGNATURE_RSA_SHA_PKCS1_PSS;
							break;
						default:
							goto abort;
					}
					break;
#else
					choice = getChoice("Select the algorithm: ", "Choose", rsaSign, 1);
					switch (choice) {
						case 1: ciphMode = MSC_MODE_RSA_PAD_PKCS1;
							break;
						default:
							goto abort;
					}
					break;
#endif
#ifdef VERSION2	
				case MSC_KEY_EC_F2M_PRIVATE:
				case MSC_KEY_EC_FP_PRIVATE:
				case MSC_KEY_EC_FP_PUBLIC:
				case MSC_KEY_EC_F2M_PUBLIC:
					choice = getChoice("Select the algorithm: ", "Choose", ecSign, 1);
					switch(choice) {
						case 1: ciphMode = MSC_MODE_SIGNATURE_ECDSA_SHA;
							if (6 + (2 * ((keySize / 8) + 1)) > outDataSize) {
								outCryptData = realloc(outCryptData, sizeof(MSCUChar8)* (6 + (2 * ((keySize / 8) + 1))));
							}
							break;
						default:
							goto abort;
					}
					break;
#endif
				case MSC_KEY_DSA_PUBLIC:
				case MSC_KEY_DSA_PRIVATE:
					choice = getChoice("Select the algorithm: ", "Choose", dsaSign, 1);
					switch(choice) {
#ifdef VERSION2
						case 1: ciphMode = MSC_MODE_SIGNATURE_DSA_SHA;
#else
						case 1: ciphMode = MSC_MODE_DSA_SHA;
#endif
							break;
						default:
							goto abort;
					}
					break;
			}
	}

	cryptInit.keyNum = keyNum;
	cryptInit.cipherMode = ciphMode;
	cryptInit.cipherDirection = ciphDir;
	cryptInit.optParams = 0;
	cryptInit.optParamsSize = 0;

	rv = MSCComputeCrypt(pConnection, &cryptInit,
		inCryptData, inDataSize, outCryptData, &outDataSize);

	CHECK_ERR(rv != MSC_SUCCESS, "Crypt Failed !");

	if (outFileName == NULL) {
		printf("Result    : ");

		if (ciphDir == MSC_DIR_VERIFY) {
			printf("Signature valid.");
		}
		else {
			for (i = 0; i < outDataSize; i++)
				printf("%02X", outCryptData[i]);
		}
		printf("\n");
	}
	else {
		long written = 0;
		if (ciphDir == MSC_DIR_VERIFY) {
			printf("Signature valid.");
		}
		else {
			outFile = fopen(outFileName, "w");
			if (outFile == NULL) {
				printf("ERR: Cannot open outfile\n");
				goto end;
			}
			written = fwrite(outCryptData, sizeof(MSCUChar8), outDataSize, outFile);
			if (ferror(outFile)) {
				printf("ERR: Error writing outfile\n");
				goto end;
			}
		}
	}

	printf("Crypt Successful.\n");
	goto end;
abort:
	printf("Crypt Cancelled.\n");
end:
	if (inCryptData)
		free(inCryptData);
	if (outCryptData)
		free(outCryptData);
	if (inFile)
		fclose(inFile);
	if (outFile)
		fclose(outFile);
	return;
}

void doListKeys(void)
{
	MSCLong32 rv = 0;
	MSCKeyInfo keyInfo;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	printf("\n");

	rv = MSCListKeys(pConnection, MSC_SEQUENCE_RESET, &keyInfo);

	CHECK_ERR((rv != MSC_SEQUENCE_END && rv != MSC_SUCCESS), "ListKeys Failed !");

	printf("%16s %8s %5s\n", "Key Type", "Key Num", "SIZE");
	printf("%16s %8s %5s\n", "---------------",	"-------", "----");

	switch (keyInfo.keyType)
	{
		case MSC_KEY_DSA_PUBLIC:
			printf("%16s", "DSA PUBLIC");
			break;
		case MSC_KEY_DSA_PRIVATE:
			printf("%16s", "DSA PRIVATE");
			break;
		case MSC_KEY_RSA_PUBLIC:
			printf("%16s", "RSA PUBLIC");
			break;
		case MSC_KEY_RSA_PRIVATE:
			printf("%16s", "RSA PRIVATE");
			break;
		case MSC_KEY_RSA_PRIVATE_CRT:
			printf("%16s", "RSA PRIVATE CRT");
			break;
		case MSC_KEY_DES:
			printf("%16s", "DES");
			break;
		case MSC_KEY_3DES:
			printf("%16s", "Triple DES");
			break;
#ifdef VERSION2
		case MSC_KEY_AES:
			printf("%16s", "AES");
			break;
		case MSC_KEY_EC_F2M_PRIVATE:
			printf("%16s", "EC F2M PRIVATE");
			break;
		case MSC_KEY_EC_F2M_PUBLIC:
			printf("%16s", "EC F2M PUBLIC");
			break;
		case MSC_KEY_EC_FP_PRIVATE:
			printf("%16s", "EC FP PRIVATE");
			break;
		case MSC_KEY_EC_FP_PUBLIC:
			printf("%16s", "EC FP PUBLIC");
			break;
#endif
	}

	if (rv == MSC_SUCCESS)
	{
		printf("%9d  %04d\n", keyInfo.keyNum, keyInfo.keySize);
#ifdef VERSION2
		printf("\n");
		printf("       Key Flags\n");
		printf(" ---------------\n");
		doPrintKeyFlags(keyInfo.keyFlags);
#endif
		printf("\n");
		printf("            READ\n");
		printf("          ------\n");
		doPrintAUT(keyInfo.keyACL.readPermission);
		printf("\n");
		printf("           WRITE\n");
		printf("          ------\n");
		doPrintAUT(keyInfo.keyACL.writePermission);
		printf("\n");
		printf("             USE\n");
		printf("          ------\n");
		doPrintAUT(keyInfo.keyACL.usePermission);
		printf("\n");
#ifdef VERSION2
		printf("\n");
		printf("        CIPH DIR\n");
		printf("       ---------\n");
		doPrintKeyPolicyCiphDir(keyInfo);
		printf("\n");
		printf("       CIPH MODE\n");
		printf(" ---------------\n");
		doPrintKeyPolicyCiphMode(keyInfo);
		printf("\n");
#endif
	}

	if (rv == MSC_SEQUENCE_END) {
		printf("ListObjects Success.\n");
		goto end;
	}
	do
	{
		rv = MSCListKeys(pConnection, MSC_SEQUENCE_NEXT, &keyInfo);
		CHECK_ERR((rv != MSC_SEQUENCE_END && rv != MSC_SUCCESS), "ListKeys Failed !");
		if (rv == MSC_SUCCESS)
		{
			printf("%16s %8s %5s\n", "Key Type", "Key Num", "SIZE");
			printf("%16s %8s %5s\n", "---------------",	"-------", "----");
			switch (keyInfo.keyType)
			{
				case MSC_KEY_DSA_PUBLIC:
					printf("%16s", "DSA PUBLIC");
					break;
				case MSC_KEY_DSA_PRIVATE:
					printf("%16s", "DSA PRIVATE");
					break;
				case MSC_KEY_RSA_PUBLIC:
					printf("%16s", "RSA PUBLIC");
					break;
				case MSC_KEY_RSA_PRIVATE:
					printf("%16s", "RSA PRIVATE");
					break;
				case MSC_KEY_RSA_PRIVATE_CRT:
					printf("%16s", "RSA PRIVATE CRT");
					break;
				case MSC_KEY_DES:
					printf("%16s", "DES");
					break;
				case MSC_KEY_3DES:
					printf("%16s", "Triple DES");
					break;
#ifdef VERSION2
				case MSC_KEY_AES:
					printf("%16s", "AES");
					break;
				case MSC_KEY_EC_F2M_PRIVATE:
					printf("%16s", "EC F2M PRIVATE");
					break;
				case MSC_KEY_EC_F2M_PUBLIC:
					printf("%16s", "EC F2M PUBLIC");
					break;
				case MSC_KEY_EC_FP_PRIVATE:
					printf("%16s", "EC FP PRIVATE");
					break;
				case MSC_KEY_EC_FP_PUBLIC:
					printf("%16s", "EC FP PUBLIC");
					break;
#endif
			}

			printf("%9d  %04d\n", keyInfo.keyNum, keyInfo.keySize);
#ifdef VERSION2
			printf("\n");
			printf("       Key Flags\n");
			printf("       ---------\n");
			doPrintKeyFlags(keyInfo.keyFlags);
#endif
			printf("\n");
			printf("            READ\n");
			printf("          ------\n");
			doPrintAUT(keyInfo.keyACL.readPermission);
			printf("\n");
			printf("           WRITE\n");
			printf("          ------\n");
			doPrintAUT(keyInfo.keyACL.writePermission);
			printf("\n");
			printf("             USE\n");
			printf("          ------\n");
			doPrintAUT(keyInfo.keyACL.usePermission);
			printf("\n");
#ifdef VERSION2
			printf("\n");
			printf("        CIPH DIR\n");
			printf("       ---------\n");
			doPrintKeyPolicyCiphDir(keyInfo);
			printf("\n");
			printf("       CIPH MODE\n");
			printf(" ---------------\n");
			doPrintKeyPolicyCiphMode(keyInfo);
			printf("\n");
#endif
		}
	}
	while (rv != MSC_SEQUENCE_END);
	printf("ListKeys Success.\n");
end:
	return;
}

void doExportKey(int keyNumber, char *fileName)
{
	MSCLong32 rv = 0;
	MSCUChar8 keyData[2000];
	MSCULong32 keySize;
	FILE *fp = NULL;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	printf("\n");
	
	keySize = sizeof(keyData);

	rv = MSCExportKey(pConnection, (MSCUChar8) keyNumber,
		keyData, &keySize, 0, 0);

	CHECK_ERR(rv != MSC_SUCCESS, "ExportKey Failed !");

	fp = fopen(fileName, "w+");

	if (fp == NULL)
	{
		printf("ERR: Open file %s failed\n", fileName);
		goto end;
	}

	fwrite(keyData, 1, keySize, fp);

	printf("ExportKey Successful\n");
end:
	if (fp)
		fclose(fp);
	return;
}

void doImportKey(int keyNumber, char *fileName)
{
	MSCLong32 rv = 0;
	MSCKeyACL keyACL;
	MSCKeyPolicy keyPolicy;
	MSCPUChar8 objectData = NULL;
#ifdef VERSION2
	unsigned long keyPartner;
#endif
	unsigned long readACL, writeACL, useACL;
	MSCULong32 dataSize;
	struct stat fileStat;
	FILE *fp = NULL;

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	printf("\n");

	fp = fopen(fileName, "rb");

	if (fp == NULL)
	{
		printf("ERR: Open file %s failed\n", fileName);
		goto end;
	}

	if (stat(fileName, &fileStat) != 0)
	{
		printf("ERR: Failed to stat file\n");
		goto end;
	}

	dataSize = fileStat.st_size;
	objectData = malloc(dataSize);
	if (!objectData)
	{
		printf("ERR: Malloc failed\n");
		goto end;
	}

	fread(objectData, 1, dataSize, fp);

#ifdef VERSION2
	keyPolicy.cipherDirection = MSC_KEYPOLICY_DIR_ANY;
	keyPolicy.cipherCipherMode = MSC_KEYPOLICY_MODE_CIPHER_ANY;
	keyPolicy.cipherSignatureMode = MSC_KEYPOLICY_MODE_SIGNATURE_ANY;
#else
	keyPolicy.cipherDirection = 0xFFFF;
	keyPolicy.cipherMode = 0xFFFF;
#endif

	rv = getNumberValue("Enter the ACL for reading the key", "Enter ACL mask", &readACL);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the ACL for writing the key", "Enter ACL mask", &writeACL);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Enter the ACL for using the key", "Enter ACL mask", &useACL);
	if (rv == 0)
		goto abort;

	keyACL.readPermission = (MSCUShort16)readACL;
	keyACL.writePermission = (MSCUShort16)writeACL;
	keyACL.usePermission = (MSCUShort16)useACL;
#ifdef VERSION2
	rv = getNumberValue("Enter the key partner number", "Enter key number", &keyPartner);
	if (rv == 0)
		goto abort;

	rv = MSCImportKey(pConnection, (MSCUChar8) keyNumber,
		&keyACL, objectData, dataSize, &keyPolicy, 0, (MSCUChar8) keyPartner, NULL, 0);
#else
	rv = MSCImportKey(pConnection, (MSCUChar8) keyNumber,
                          &keyACL, objectData, dataSize, &keyPolicy, 0, 0);
#endif
	CHECK_ERR(rv != MSC_SUCCESS, "ImportKey Failed !");

	printf("ImportKey Successful\n");
	goto end;
abort:
end:
	if (fp)
		fclose(fp);
	if (objectData)
		free(objectData);
	return;
}

void doGenKeys(const char *keyType, int keySize )
{
	MSC_RV rv = 0;
	unsigned long prvKeyNum=0, pubKeyNum=0;
	MSCGenKeyParams keyParams;
	pthread_t pthThread;
	unsigned long prvReadACL=0, prvWriteACL=0, prvUseACL=0, pubReadACL=0, pubWriteACL=0, pubUseACL=0;
	int choice;
	char algorithm[8];

	CHECK_ERR(pConnection == 0, "Must Connect First !");

	keyParams.algoType = 0;
#ifdef VERSION2
	if (strcmp(keyType, "ecfp") == 0) {
		keyParams.algoType = MSC_GEN_ALG_EC_FP;
		strcpy(algorithm, "EC FP\0");
	}
	if (strcmp(keyType, "ecf2m") == 0) {
		keyParams.algoType = MSC_GEN_ALG_EC_F2M;
		strcpy(algorithm, "EC F2M\0");
	}
	else if (strcmp(keyType, "aes") == 0) {
		keyParams.algoType = MSC_GEN_ALG_AES;
		strcpy(algorithm, "AES\0");
	}
#endif
	if (strcmp(keyType, "rsacrt") == 0) {
		keyParams.algoType = MSC_GEN_ALG_RSA_CRT;
		strcpy(algorithm, "RSA CRT\0");
	}
	else if (strcmp(keyType, "dsa") == 0) {
		keyParams.algoType = MSC_GEN_ALG_DSA;
		strcpy(algorithm, "DSA\0");
	}
	else if (strcmp(keyType, "rsa") == 0) {
		keyParams.algoType = MSC_GEN_ALG_RSA;
		strcpy(algorithm, "RSA\0");
	}
	else if (strcmp(keyType, "des") == 0) {
		keyParams.algoType = MSC_GEN_ALG_DES;
		strcpy(algorithm, "DES\0");
	}
	else if (strcmp(keyType, "3des") == 0) {
		keyParams.algoType = MSC_GEN_ALG_3DES;
		strcpy(algorithm, "3DES\0");
	}
	else if (strcmp(keyType, "3des3") == 0) {
		keyParams.algoType = MSC_GEN_ALG_3DES3;
		strcpy(algorithm, "3DES3\0");
	}

#ifdef VERSION2
	keyParams.privateKeyPolicy.cipherDirection = MSC_KEYPOLICY_DIR_ANY;
	keyParams.privateKeyPolicy.cipherCipherMode = MSC_KEYPOLICY_MODE_CIPHER_ANY;
	keyParams.privateKeyPolicy.cipherSignatureMode = MSC_KEYPOLICY_MODE_SIGNATURE_ANY;
	keyParams.publicKeyPolicy.cipherDirection = MSC_KEYPOLICY_DIR_ANY;
	keyParams.publicKeyPolicy.cipherCipherMode = MSC_KEYPOLICY_MODE_CIPHER_ANY;
	keyParams.publicKeyPolicy.cipherSignatureMode = MSC_KEYPOLICY_MODE_SIGNATURE_ANY;
	keyParams.privateKeyFlags = 0;
	keyParams.publicKeyFlags = 0;
#else
	keyParams.privateKeyPolicy.cipherDirection = 0xFFFF;
	keyParams.privateKeyPolicy.cipherMode= 0xFFFF;
	keyParams.publicKeyPolicy.cipherDirection = 0xFFFF;
	keyParams.publicKeyPolicy.cipherMode = 0xFFFF;
#endif

	keyParams.keyGenOptions = 0;
	keyParams.pOptParams = 0;
	keyParams.optParamsSize = 0;
	keyParams.keySize = (MSCUShort16)keySize;

	printf("\n");

	printf("***************** Key Generation Routine *****************\n");
	printf("       using the %s algorithm for generation.\n", algorithm);
	printf("**********************************************************\n");

	if (keyParams.algoType == MSC_GEN_ALG_RSA_CRT || keyParams.algoType == MSC_GEN_ALG_RSA || keyParams.algoType == MSC_GEN_ALG_DSA
#ifdef VERSION2
		|| keyParams.algoType == MSC_GEN_ALG_EC_F2M || keyParams.algoType == MSC_GEN_ALG_EC_FP
#endif
		) {
		rv = getNumberValue("Enter the public key number", "Enter key number", &pubKeyNum);
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the private key number", "Enter key number", &prvKeyNum);
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the ACL for reading the public key\nExample: 0x0002 for user PIN #1", "Enter ACL mask", &pubReadACL);
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the ACL for deleting the public key\nExample: 0x0002 for user PIN #1", "Enter ACL mask", &pubWriteACL);
		if (rv == 0)
			goto abort;

#ifdef VERSION2
		rv = getNumberValue("Enter the ACL for using the public key\nExample: 0xFFFF for all users", "Enter ACL mask", &pubUseACL);
#else
		rv = getNumberValue("Enter the ACL for using the public key\nExample: 0x0000 for all users", "Enter ACL mask", &pubUseACL);
#endif
			
		if (rv == 0)
			goto abort;

#ifdef VERSION2
		rv = getNumberValue("Enter the ACL for reading the private key\nExample: 0x0000 for nobody", "Enter ACL mask", &prvReadACL);			
#else
		rv = getNumberValue("Enter the ACL for reading the private key\nExample: 0xFFFF for nobody", "Enter ACL mask", &prvReadACL);
#endif
			
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the ACL for deleting the private key\nExample: 0x0002 for user PIN #1", "Enter ACL mask", &prvWriteACL);
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the ACL for using the private key\nExample: 0x0002 for user PIN #1", "Enter ACL mask", &prvUseACL);
		if (rv == 0)
			goto abort;
	}
	else {
		rv = getNumberValue("Enter the secret key number", "Enter key number", (unsigned long *)&pubKeyNum);
		if (rv == 0)
			goto abort;

#ifdef VERSION2
		rv = getNumberValue("Enter the ACL for reading the secret key\nExample: 0x0000 for nobody", "Enter ACL mask", (unsigned long *)&pubReadACL);			
#else
		rv = getNumberValue("Enter the ACL for reading the secret key\nExample: 0xFFFF for nobody", "Enter ACL mask", (unsigned long *)&pubReadACL);
#endif
			
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the ACL for deleting the secret key\nExample: 0x0002 for user PIN #1", "Enter ACL mask", (unsigned long *)&pubWriteACL);
		if (rv == 0)
			goto abort;

		rv = getNumberValue("Enter the ACL for using the secret key\nExample: 0x0002 for user PIN #1", "Enter ACL mask", (unsigned long *)&pubUseACL);
		if (rv == 0)
			goto abort;

	}

	keyParams.privateKeyACL.readPermission = (MSCUShort16)prvReadACL;
	keyParams.privateKeyACL.writePermission = (MSCUShort16)prvWriteACL;
	keyParams.privateKeyACL.usePermission = (MSCUShort16)prvUseACL;

	keyParams.publicKeyACL.readPermission = (MSCUShort16)pubReadACL;
	keyParams.publicKeyACL.writePermission = (MSCUShort16)pubWriteACL;
	keyParams.publicKeyACL.usePermission = (MSCUShort16)pubUseACL;

	{
		const char *choices[2] = {"YES", "NO"};
		choice = getChoice("Are you sure ?", "Choose", choices, 2);
	}

	if (choice == 0 || choice == 2)
	{
		goto abort;
	}

	printf("Generating keys [");
	quitStat = 0;
	rv = pthread_create(&pthThread, NULL, printStatStars, 0);

	fflush(stdout);
	fflush(stdin);

	rv = MSCGenerateKeys(pConnection, (MSCUChar8)prvKeyNum, (MSCUChar8)pubKeyNum, &keyParams);

	quitStat = 1;
	if (pthread_join(pthThread, NULL) != 0)
		printf("pthread_join error: %s\n", strerror(errno));;

	CHECK_ERR(rv != MSC_SUCCESS, "GenKeys Failed !");
	printf("GenKeys Success.\n");
	goto end;
abort:
	printf("GenKeys Cancelled.\n");
end:
	return;
}

void doVersion(void)
{
	printf("muscleTool version: %s\n", PACKAGE_VERSION);
	printf("Copyright (C) 1998-2002 David Corcoran\n");
	printf("Copyright (C) 2003 Ludovic Rousseau\n");
	printf("Copyright (C) 2004 Toni Andjelkovic\n");
	printf("Copyright (C) 2005 Karsten Ohme\n");
}

void doFormat(int tokenNumber)
{
	MSC_RV rv = 0;
	MSCULong32 transportKeyChoice;
	MSCInitTokenParams initParams;
	MSCUChar8 userTransportKey[9];
	MSCULong32 userTransportKeyLength = 8;
	MSCChar8 userStringKey[17];
	MSCULong32 userStringKeyLength=17;

	MSCUChar8 adminPIN[17];
#ifdef VERSION2
	MSCUChar8 adminUnblockPIN[17];
#endif
	MSCUChar8 userPIN[17];
	MSCUChar8 userUnblockPIN[17];
#ifdef VERSION2
	MSCULong32 adminPINTries;
	MSCULong32 adminUnblockPINTries;
#endif
	MSCULong32 userPINTries;
	MSCULong32 userUnblockPINTries;

	MSCULong32 memoryAlloc;
	MSCUChar8 transportKey[9];
	MSCULong32 transportKeyLength = 9;
	pthread_t pthThread;
#ifdef VERSION2
	MSCChar8 dummyString[256];
	MSCUChar8 pinMaxSize;
	MSCULong32 dummy;
	MSCUChar8 pinMinSize = 4;
#endif


	unsigned long pinACL;
	unsigned long keyACL;
	unsigned long objectACL;
#ifdef VERSION2
	MSCULong32 pinPolicyChoice;
	MSCULong32 pinSizePolicyChoice;
	MSCULong32 pinCharsetPolicyChoice;
	MSCULong32 pinMixTypePolicyChoice;
#endif
	MSCULong32 choice;
#ifdef VERSION2
	MSCUChar8 pinPolicies = 0;
	MSCUChar8 pinMixType = 0;
	MSCUChar8 pinCharset = 0;
	MSCUChar8 setSerialNumber = 0;
	MSCUChar8 serialNumber[MSC_SERIALNUMBER_SIZE+1];
	MSCUChar8 setCardLabel = 0;
	MSCUChar8 cardLabel[MSC_TOKENLABEL_SIZE+1];
#endif

	CHECK_ERR(tokenList == 0, "List Tokens First !");

	if (tokenNumber < 1 || (MSCULong32)tokenNumber > tokenSize)
	{
		printf("ERR: Invalid choice made !\n");
		goto end;
	}

	if (pConnection)
	{
		MSCReleaseConnection(pConnection, MSC_RESET_TOKEN);
		free(pConnection);
		pConnection = NULL;
	}

	pConnection = malloc(sizeof(MSCTokenConnection));

	rv = MSCEstablishConnection(&tokenList[tokenNumber - 1],
		MSC_SHARE_EXCLUSIVE, 0, 0, pConnection);
	CHECK_ERR(rv != MSC_SUCCESS, "EstablishConnection Failed !");

#ifdef VERSION2
	rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXSIZE,
				(MSCPUChar8) &pinMaxSize, &dummy);

	if (rv != MSC_SUCCESS)
	{
		printf("Can't get maximal PIN size from card - using default of 16\n");
		pinMaxSize = 16;
	}
#endif
	{
		const char *choices[2] = {"Use the default key : 4D7573636C653030",
		"Enter your own transport key"};
		transportKeyChoice = getChoice("Would you like to: ", "Choose", choices, 2);
	}
	if (transportKeyChoice == 0)
		goto abort;

	if (transportKeyChoice == 2)
	{
		rv = getStringValue("Enter your transport key below (2 chars per byte, uppercase, exactly 8 bytes)\nExample: 4D7573636C653030\n", "Enter key", userStringKey, userStringKeyLength);
		if (rv == 0)
			goto abort;
		rv = textToBytes(userStringKey, userTransportKey, &userStringKeyLength);
		if (rv != 0)
		{
			printf("Invalid transport key\n");
			goto end;
		}
		userTransportKeyLength = strlen(userStringKey);
	}

#ifdef VERSION2
	{
		const char *choices[2] = {"YES", "NO"};
		pinPolicyChoice = getChoice("Do you want to enforce any PIN policies ?", "Choose", choices, 2);
		if (pinPolicyChoice == 0)
			goto abort;

		if (pinPolicyChoice == 1)
		{
			pinSizePolicyChoice = getChoice("Do you want to enforce PIN size policy ?", "Choose", choices, 2);
			if (pinSizePolicyChoice == 0)
				goto abort;

			if (pinSizePolicyChoice == 1)
			{
				pinPolicies |= MSC_PIN_POLICY_SIZE;
				snprintf(dummyString, sizeof(dummyString), "Enter the maximal PIN size (max. %d)", pinMaxSize);
				rv = getNumberValue(dummyString, "Enter PIN size", &dummy);
				pinMaxSize = (MSCUChar8)dummy;
				if (rv == 0)
					goto abort;
				snprintf(dummyString, sizeof(dummyString), "Enter the minimal PIN size (max. %d)", pinMaxSize);
				rv = getNumberValue(dummyString, "Enter PIN size", &dummy);
				pinMinSize = (MSCUChar8)dummy;
				if (rv == 0)
					goto abort;
			}

			pinCharsetPolicyChoice = getChoice("Do you want to enforce PIN charset policy ?", "Choose", choices, 2);
			if (pinCharsetPolicyChoice == 0)
				goto abort;

			if (pinCharsetPolicyChoice == 1)
			{
				pinPolicies |= MSC_PIN_POLICY_CHARSET;

				choice = getChoice("Do you want to allow upper case letters ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinCharset |= MSC_PIN_CHARSET_UC_LETTERS;

				choice = getChoice("Do you want to allow lower case letters ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinCharset |= MSC_PIN_CHARSET_LC_LETTERS;

				choice = getChoice("Do you want to allow numbers ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinCharset |= MSC_PIN_CHARSET_NUMBERS;

				choice = getChoice("Do you want to allow other chars ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinCharset |= MSC_PIN_CHARSET_OTHERS;
			}

			pinMixTypePolicyChoice = getChoice("Do you want to enforce PIN charset mix type policy ?", "Choose", choices, 2);
			if (pinMixTypePolicyChoice == 0)
				goto abort;

			if (pinMixTypePolicyChoice == 1)
			{
				pinPolicies |= MSC_PIN_POLICY_MIXED;

				choice = getChoice("Must the PIN contain at least a upper case letter ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinMixType |= MSC_PIN_MIXED_UC_LETTERS;

				choice = getChoice("Must the PIN contain at least a lower case letter ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinMixType |= MSC_PIN_MIXED_LC_LETTERS;

				choice = getChoice("Must the PIN contain at least a number ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinMixType |= MSC_PIN_MIXED_NUMBERS;

				choice = getChoice("Must the PIN contain at least a other character ?", "Choose", choices, 2);
				if (choice == 0)
					goto abort;
				if (choice == 1)
					pinMixType |= MSC_PIN_MIXED_OTHER;
			}
		}
	}

#endif
	rv = getStringValue("Please enter your admin PIN (PIN #0)", "Enter admin PIN", (char *)adminPIN, 17);
	if (rv == 0)
		goto abort;
#ifdef VERSION2
	rv = getNumberValue("Please enter your number of PIN tries for admin PIN (PIN #0)", "Enter admin PIN tries", &adminPINTries);
	if (rv == 0)
		goto abort;

        rv = getStringValue("Please enter your admin unblock PIN (PIN #0)", "Enter admin unblock PIN", (char *)adminUnblockPIN, 17);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Please enter your number of PIN tries for admin unblock PIN (PIN #0)", "Enter admin unblock PIN tries", &adminUnblockPINTries);
	if (rv == 0)
		goto abort;
#endif

	rv = getStringValue("Please enter your user PIN (PIN #1)", "Enter user PIN", (char *)userPIN, 17);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Please enter your number of PIN tries for user PIN (PIN #1)", "Enter user PIN tries", &userPINTries);
	if (rv == 0)
		goto abort;

        rv = getStringValue("Please enter your user unblock PIN (PIN #1)", "Enter user unblock PIN", (char *)userUnblockPIN, 17);
	if (rv == 0)
		goto abort;

	rv = getNumberValue("Please enter your number of PIN tries for user unblock PIN (PIN #1)", "Enter user unblock PIN tries", &userUnblockPINTries);
	if (rv == 0)
		goto abort;

	printf("\n");

	rv = getNumberValue("How much bytes for object memory would you like to allocate ?\nExample: 10000\n", "Enter object memory size", &memoryAlloc);
	if (rv == 0)
		goto abort;

	if (transportKeyChoice == 1)
	{
		memcpy(transportKey, muscleDefaultKey, sizeof(muscleDefaultKey));
		transportKeyLength = sizeof(muscleDefaultKey);
	}
	else if (transportKeyChoice == 2)
	{
		memcpy(transportKey, userTransportKey, 8);
		transportKeyLength = userTransportKeyLength;
	}

	transportKey[8] = '\0';

	rv = getNumberValue("Enter the ACL who can create objects\nExample: 0x0002 for user with PIN #1\n", "Enter ACL", &objectACL);
	if (rv == 0)
		goto abort;
	rv = getNumberValue("Enter the ACL who can create keys\nExample: 0x0002 for user with PIN #1\n", "Enter ACL", &keyACL);
	if (rv == 0)
		goto abort;
	rv = getNumberValue("Enter the ACL who can create PINs\nExample: 0x0001 for admin with PIN #0\n", "Enter ACL", &pinACL);
	if (rv == 0)
		goto abort;
#ifdef VERSION2
	/* Matheus: added card label and serial number information */
	{
		const char *choices[2] = { "YES", "NO" };
		rv = getChoice(
					"Do you want to set the card label?\nCard label can be (re)set later by admin.\n",
					"Choose:",
					choices, 2);

		if (rv == 1){
			setCardLabel = 1;
			// enter label
			rv = getStringValue(
				"Please enter the card label (exactly 32 chars, eventually padded with ' ')",
				"Enter card label",
				cardLabel,
				MSC_TOKENLABEL_SIZE+1
			);
		}
		
		if (rv == 0){
			goto abort;
		}

		rv = getChoice(
					"Do you want to set the card serial number?\nSerial number is a write once attribute, and if not set now can only be set later by card admin.\n",
					"Choose",
					choices, 2);

		if (rv == 1){
			setSerialNumber = 1;
			// enter snumber
			rv = getStringValue(
				"Please enter the card serial number (exactly 16 chars, eventually padded with ' ')",
				"Enter serial number",
				serialNumber,
				MSC_SERIALNUMBER_SIZE+1
			);
		}
		
		if (rv == 0){
			goto abort;
		}
		printf("\n");
	}
#endif

	printf("Transport key       : %s\n", transportKey);
#ifdef VERSION2
	printf("Admin PIN           : %s (%d tries)\n", adminPIN, adminPINTries);
#else
  printf("Admin PIN           : %s\n", adminPIN);
#endif
#ifdef VERSION2
	printf("Admin unblock PIN   : %s (%d tries)\n", adminUnblockPIN, adminUnblockPINTries);
#endif
	printf("User PIN            : %s (%ld tries)\n", userPIN, userPINTries);
	printf("User unblock PIN    : %s (%ld tries)\n", userUnblockPIN, userUnblockPINTries);
	printf("Object memory       : %ld bytes\n", memoryAlloc);
	printf("Create object ACL   : 0x%04lx\n", objectACL);
	printf("Create key ACL      : 0x%04lx\n", keyACL);
	printf("Create PIN ACL      : 0x%04lx\n", pinACL);
#ifdef VERSION2
	if (pinPolicyChoice == 1)
	{
		printf("PIN policies :\n");
		if (pinSizePolicyChoice == 1) {
			printf("PIN size policy     :\n");
			printf("    Max PIN size    : %d\n", pinMaxSize);
			printf("    Min PIN size    : %d\n", pinMinSize);
		}
		if (pinCharsetPolicyChoice == 1) {
			printf("PIN charset policy  :\n");
			if ((pinCharset & MSC_PIN_CHARSET_UC_LETTERS) != 0)
				printf("    PIN can contain upper case letter\n");
			if ((pinCharset & MSC_PIN_CHARSET_LC_LETTERS) != 0)
				printf("    PIN can contain lower case letter\n");
			if ((pinCharset & MSC_PIN_CHARSET_NUMBERS) != 0)
				printf("    PIN can contain number\n");
			if ((pinCharset & MSC_PIN_CHARSET_OTHERS) != 0)
				printf("    PIN can contain other character\n");
		}
		if (pinMixTypePolicyChoice == 1) {
			printf("PIN mix type policy :\n");
				if ((pinCharset & MSC_PIN_MIXED_UC_LETTERS) != 0)
				printf("    PIN must contain upper case letter\n");
			if ((pinCharset & MSC_PIN_MIXED_LC_LETTERS) != 0)
				printf("    PIN must contain lower case letter\n");
			if ((pinCharset & MSC_PIN_MIXED_NUMBERS) != 0)
				printf("    PIN must contain number\n");
			if ((pinCharset & MSC_PIN_MIXED_OTHER) != 0)
				printf("    PIN must contain other character\n");
		}
	}
	if (setCardLabel) {
		printf("Card label          : %s\n", cardLabel);
	}
	if (setSerialNumber) {
		printf("Serial number       : %s\n", serialNumber);
	}

#endif
	{
		const char *choices[2] = {"YES", "NO"};
		choice = getChoice("Is the above okay ?", "Choose", choices, 2);
		if (choice == 0 || choice == 2)
			goto abort;
	}

	printf("\n");

	printf("Formating token [");
	quitStat = 0;
	rv = pthread_create(&pthThread, NULL, printStatStars, 0);

	memcpy(initParams.transportKey, transportKey, transportKeyLength);
	initParams.transportKeyLen = transportKeyLength;
#ifdef VERSION2
	memcpy(initParams.adminCHV, adminPIN, strlen((char *)adminPIN));
	initParams.adminCHVLen = strlen((char *)adminPIN);

	memcpy(initParams.adminCHVUnblock, adminUnblockPIN, strlen((char *)adminUnblockPIN));
	initParams.adminCHVUnblockLen = strlen((char *)adminUnblockPIN);
#else
	memcpy(initParams.newTransportKey, adminPIN, strlen((char *)adminPIN));
	initParams.newTransportKeyLen = strlen((char *)adminPIN);
#endif

	memcpy(initParams.defaultCHV, userPIN, strlen((char *)userPIN));
	initParams.defaultCHVLen = strlen((char *)userPIN);

	memcpy(initParams.defaultCHVUnblock, userUnblockPIN,
		strlen((char *)userUnblockPIN));
	initParams.defaultCHVUnblockSize = strlen((char *)userUnblockPIN);

#ifdef VERSION2
	initParams.adminCHVTries = (MSCUChar8) adminPINTries;
	initParams.adminCHVUnblockTries = (MSCUChar8) adminUnblockPINTries;
#endif
	initParams.defaultCHVTries = (MSCUChar8) userPINTries;
	initParams.defaultCHVUnblockTries = (MSCUChar8) userUnblockPINTries;

	initParams.objectMemory = memoryAlloc;
	initParams.createKeysACL = (MSCUShort16)keyACL;
	initParams.createObjectACL = (MSCUShort16)objectACL;
	initParams.createPINsACL = (MSCUShort16)pinACL;
#ifdef VERSION2
	initParams.pinPolicies = pinPolicies;
	initParams.pinMaxSize = (MSCUChar8)pinMaxSize;
	initParams.pinMinSize = (MSCUChar8)pinMinSize;
	initParams.pinCharset = pinCharset;
	initParams.pinMixType = pinMixType;
	initParams.setSerialNumber = setSerialNumber;
	memcpy(initParams.serialNumber, serialNumber, MSC_SERIALNUMBER_SIZE);
	initParams.setTokenLabel = setCardLabel;
	memcpy(initParams.tokenLabel, cardLabel, MSC_TOKENLABEL_SIZE);
#endif

	rv = MSCWriteFramework(pConnection, &initParams);

	quitStat = 1;
	if (pthread_join(pthThread, NULL) != 0)
		printf("pthread_join error: %s\n", strerror(errno));

	CHECK_ERR(rv != MSC_SUCCESS, "Format failed !");

	printf("Format Successful.\n");
	goto end;

abort:
	printf("Format Cancelled.\n");
end:
	if (pConnection) {
		MSCReleaseConnection(pConnection, MSC_RESET_TOKEN);
		free(pConnection);
		pConnection = NULL;
	}
	return;
}

