//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: alsa5midi.cpp,v 1.2 2002/02/21 12:41:44 muse Exp $
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "alsamidi.h"
#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==5)
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <qstring.h>
#include <glob.h>
#include "globals.h"
#include "midirawdev.h"
#include "seq.h"
#include "../midiport.h"

//---------------------------------------------------------
//   MidiAlsaDevice
//---------------------------------------------------------

MidiAlsaDevice::MidiAlsaDevice(const snd_seq_addr_t& a, const QString& n)
   : MidiDevice(n)
      {
      adr = a;
      init();
      }

//---------------------------------------------------------
//   open
//---------------------------------------------------------

QString MidiAlsaDevice::open(int rw)
      {
      snd_seq_port_subscribe_t subs;
      memset(&subs, 0, sizeof(subs));
      subs.sender = musePort;
      subs.dest   = adr;
      int error   = snd_seq_subscribe_port(alsaSeq, &subs);
      if (error < 0) {
            QString s;
            s.sprintf("AlsaMidi: open: subscribe(%d:%d): %s",
               adr.client, adr.port, snd_strerror(error));
            return s;
            }
      rwFlag = rw;
      if (rwFlag & 0x2) {
            memset(&subs, 0, sizeof(subs));
            subs.dest   = musePort;
            subs.sender = adr;
            int error   = snd_seq_subscribe_port(alsaSeq, &subs);
            if (error < 0)
                  return QString("Rec: ") + QString(snd_strerror(error));
            }
      return QString("OK");
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void MidiAlsaDevice::close()
      {
      snd_seq_port_subscribe_t subs;
      memset(&subs, 0, sizeof(subs));
      subs.sender = musePort;
      subs.dest   = adr;
      snd_seq_unsubscribe_port(alsaSeq, &subs);
      if (rwFlag & 2) {
            memset(&subs, 0, sizeof(subs));
            subs.dest   = musePort;
            subs.sender = adr;
            snd_seq_unsubscribe_port(alsaSeq, &subs);
            }
      }

//---------------------------------------------------------
//   noteOn
//---------------------------------------------------------

void MidiAlsaDevice::noteOn(int chn, int p1, int p2)
      {
      if (midiOutputTrace)
            printf("midiAlsa: noteOn  ch:%2d p1:0x%02x p2:0x%02x\n", chn, p1, p2);
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      snd_seq_ev_set_noteon(&event, chn, p1, p2);
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("noteOn: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   noteOff
//---------------------------------------------------------

void MidiAlsaDevice::noteOff(int chn, int p1, int p2)
      {
      if (midiOutputTrace)
            printf("midiAlsa: noteOff ch:%2d p1:0x%02x p2:0x%02x\n", chn, p1, p2);
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
//      snd_seq_ev_set_noteoff(&event, chn, p1, p2);
      snd_seq_ev_set_noteon(&event, chn, p1, 0);
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("noteOff: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   pgmChange
//---------------------------------------------------------

void MidiAlsaDevice::programChange(int chn, int p1)
      {
      if (midiOutputTrace)
            printf("midiAlsa: programChange ch:%2d p1:0x%02x\n", chn, p1);
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue = SND_SEQ_QUEUE_DIRECT;
      event.source = musePort;
      event.dest = adr;
      snd_seq_ev_set_pgmchange(&event, chn, p1);
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("programChange: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   chnEvent
//---------------------------------------------------------

void MidiAlsaDevice::setCtrl(int chn, int p1, int p2)
      {
      if (midiOutputTrace)
            printf("midiAlsa: setCtrl ch:%2d p1:0x%02x p2:0x%02x\n", chn, p1, p2);
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      snd_seq_ev_set_controller(&event, chn, p1, p2);
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("setCtrl: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   bender
//---------------------------------------------------------

void MidiAlsaDevice::bender(int chn, int val)
      {
      if (midiOutputTrace)
            printf("midiAlsa: pitch ch:%2d val:%d\n", chn, val);
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      snd_seq_ev_set_pitchbend(&event, chn, val);
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("bender: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   polyPressure
//---------------------------------------------------------

void MidiAlsaDevice::polyPressure(int chn, int p1, int p2)
      {
      if (midiOutputTrace)
            printf("midiAlsa: not send polyPressure ch:%2d p1:0x%02x p2:0x%02x\n", chn, p1, p2);
#if 0 // TODO
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;

      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("channelEvent2: alsa midi write error: %s\n", snd_strerror(error));
#endif
      }

//---------------------------------------------------------
//   chnPressure
//---------------------------------------------------------

void MidiAlsaDevice::chnPressure(int chn, int p1)
      {
      if (midiOutputTrace)
            printf("midiAlsa: chnPressure ch:%2d p1:0x%02x\n", chn, p1);
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      snd_seq_ev_set_chanpress(&event, chn, p1);
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("channelPressure: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putClock
//---------------------------------------------------------

void MidiAlsaDevice::putClock()
      {
      if (midiOutputTrace)
            printf("midiAlsa: clock\n");
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      event.type    = SND_SEQ_EVENT_CLOCK;
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("putClock: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putStart
//---------------------------------------------------------

void MidiAlsaDevice::putStart()
      {
      if (midiOutputTrace)
            printf("midiAlsa: start\n");
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      event.type    = SND_SEQ_EVENT_START;
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("putStart: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putStop
//---------------------------------------------------------

void MidiAlsaDevice::putStop()
      {
      if (midiOutputTrace)
            printf("midiAlsa: stop\n");
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      event.type    = SND_SEQ_EVENT_STOP;
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("putStop: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putContinue
//---------------------------------------------------------

void MidiAlsaDevice::putContinue()
      {
      if (midiOutputTrace)
            printf("midiAlsa: continue\n");
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      event.type    = SND_SEQ_EVENT_CONTINUE;
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("putContinue: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putSongpos
//---------------------------------------------------------

void MidiAlsaDevice::putSongpos(int beat)
      {
      if (midiOutputTrace)
            printf("midiAlsa: beat\n");
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));
      event.queue   = SND_SEQ_QUEUE_DIRECT;
      event.source  = musePort;
      event.dest    = adr;
      event.data.control.value = beat;
      event.type    = SND_SEQ_EVENT_SONGPOS;
      int error = write(alsaSeqFd, &event, sizeof(event));
      if (error != sizeof(event))
            printf("putContinue: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//    sysex
//---------------------------------------------------------

extern void dump(const unsigned char*, int);

void MidiAlsaDevice::sysex(const unsigned char* p, int n)
      {
      if (midiOutputTrace) {
            printf("midiAlsa: sysex:\n");
            dump(p, n);
            }
      snd_seq_event_t event;
      memset(&event, 0, sizeof(event));

      int len             = n + sizeof(event) + 2;
      char* buf           = new char[len];

      event.type          = SND_SEQ_EVENT_SYSEX;
      event.flags         = SND_SEQ_EVENT_LENGTH_VARIABLE;
      event.data.ext.len  = n + 2;
      event.data.ext.ptr  = (void*)(buf + sizeof(event));
      event.tag           = 0;
      event.queue         = SND_SEQ_QUEUE_DIRECT;
      event.source        = musePort;
      event.dest          = adr;

      memcpy(buf, &event, sizeof(event));
      char* pp = buf + sizeof(event);
      *pp++ = 0xf0;
      memcpy(pp, p, n);
      pp += n;
      *pp = 0xf7;

      int error = write(alsaSeqFd, buf, len);

      delete buf;

      if (error != len)
            printf("sysex: midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   initMidiAlsa
//---------------------------------------------------------

bool initMidiAlsa()
      {
      int error = snd_seq_open(&alsaSeq, SND_SEQ_OPEN);
      if (error < 0) {
            fprintf(stderr, "Could not open ALSA sequencer: %s\n",
               snd_strerror(error));
            return true;
            }

      alsaFound = true;
      snd_seq_system_info_t sysinfo;
      error = snd_seq_system_info(alsaSeq, &sysinfo);
      if (error < 0) {
            fprintf(stderr, "Cound not get ALSA sequencer info: %s\n",
               snd_strerror(error));
            return true;
            }
      const int inCap  = SND_SEQ_PORT_CAP_SUBS_READ;
      const int outCap = SND_SEQ_PORT_CAP_SUBS_WRITE;
      for (int client = 0; client < sysinfo.clients; ++client) {
            snd_seq_client_info_t cinfo;
            if (snd_seq_get_any_client_info(alsaSeq, client, &cinfo) < 0)
                  continue;
            for (int port = 0; port < sysinfo.ports; port++) {
                  snd_seq_port_info_t pinfo;

                  if (snd_seq_get_any_port_info(alsaSeq, client, port, &pinfo) < 0)
                        continue;
                  if ((pinfo.capability & outCap) == 0)
                        continue;
                  snd_seq_addr_t adr = { pinfo.client, pinfo.port };
                  MidiAlsaDevice* dev = new MidiAlsaDevice(adr, pinfo.name);
                  int flags = 0;
                  if (pinfo.capability & outCap)
                        flags |= 1;
                  if (pinfo.capability & inCap)
                        flags |= 2;
                  dev->setrwFlags(flags);
                  midiDevices.add(dev);
                  }
            }
      snd_seq_set_client_name(alsaSeq, "MusE Sequencer");
      alsaSeqFd = snd_seq_file_descriptor(alsaSeq);
      int port  = snd_seq_create_simple_port(alsaSeq, "MusE Port 0",
         inCap | outCap | SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_WRITE,
         SND_SEQ_PORT_TYPE_APPLICATION);
      if (port < 0) {
            perror("create port");
            exit(1);
            }
      musePort.port   = port;
      musePort.client = snd_seq_client_id(alsaSeq);

      //-----------------------------------------
      //    subscribe to "Announce"
      //    this enables callbacks for any
      //    alsa port changes
      //-----------------------------------------

      snd_seq_addr_t aadr;
      aadr.client = SND_SEQ_CLIENT_SYSTEM;
      aadr.port   = SND_SEQ_PORT_SYSTEM_ANNOUNCE;

      snd_seq_port_subscribe_t* subs;

      subs = new snd_seq_port_subscribe_t;
      memset(subs, 0, sizeof(*subs));
      subs->dest   = musePort;
      subs->sender = aadr;
      error   = snd_seq_subscribe_port(alsaSeq, subs);
      if (error < 0) {
            printf("Subscribe System failed: %s", snd_strerror(error));
            return true;
            }
      delete subs;
      return false;
      }

//---------------------------------------------------------
//   alsaSelectRfd
//---------------------------------------------------------

int alsaSelectRfd()
      {
      return alsaSeqFd;
      }

//---------------------------------------------------------
//   alsaSelectWfd
//---------------------------------------------------------

int alsaSelectWfd()
      {
      return alsaSeqFd;
      }

struct AlsaPort {
      snd_seq_addr_t adr;
      char* name;
      int flags;
      AlsaPort(snd_seq_addr_t a, const char* s, int f) {
            adr = a;
            name = strdup(s);
            flags = f;
            }
      };

//---------------------------------------------------------
//   alsaScanMidiPorts
//---------------------------------------------------------

void alsaScanMidiPorts()
      {
      std::list<AlsaPort> portList;

// printf("scan midi port\n");
      const int inCap  = SND_SEQ_PORT_CAP_SUBS_READ;
      const int outCap = SND_SEQ_PORT_CAP_SUBS_WRITE;

      portList.clear();

      snd_seq_system_info_t sysinfo;
      int error = snd_seq_system_info(alsaSeq, &sysinfo);
      if (error < 0) {
            fprintf(stderr, "Cound not get ALSA sequencer info: %s\n",
               snd_strerror(error));
            return;
            }
      for (int client = 0; client < sysinfo.clients; ++client) {
            snd_seq_client_info_t cinfo;
            if (snd_seq_get_any_client_info(alsaSeq, client, &cinfo) < 0)
                  continue;

            for (int port = 0; port < sysinfo.ports; port++) {
                  snd_seq_port_info_t pinfo;

                  if (snd_seq_get_any_port_info(alsaSeq, client, port, &pinfo) < 0)
                        continue;

		      unsigned int cap = pinfo.capability;
                  if (((cap & outCap) == 0) && ((cap & inCap) == 0))
                        continue;
                  const char* name = pinfo.name;
                  if (strcmp(name, "Timer") == 0)
                        continue;
                  if (strcmp(name, "Announce") == 0)
                        continue;
                  snd_seq_addr_t adr;
                  adr.client = pinfo.client;
                  adr.port   = pinfo.port;
                  if (adr.client == musePort.client && adr.port == musePort.port)
                        continue;
                  int flags = 0;
                  if (cap & outCap)
                        flags |= 1;
                  if (cap & inCap)
                        flags |= 2;
                  portList.push_back(AlsaPort(adr, name, flags));
                  }
            }

      //
      //  check for devices to delete
      //
      for (iMidiDevice i = midiDevices.begin(); i != midiDevices.end();) {
            MidiAlsaDevice* d = dynamic_cast<MidiAlsaDevice*>(*i);
            if (d == 0) {
                  ++i;
                  continue;
                  }
            std::list<AlsaPort>::iterator k = portList.begin();
            for (; k != portList.end(); ++k) {
                  if ((k->adr.client == d->adr.client)
                     && (k->adr.port == d->adr.port)) {
                        break;
                        }
                  }
            if (k == portList.end()) {
                  if (d->port() != -1) {
                        printf("PORT %d\n", d->port());
                        midiPorts[d->port()].setMidiDevice(0);
                        }
                  iMidiDevice ii = i;
                  ++i;
                  midiDevices.erase(ii);
                  }
            else {
                  ++i;
                  }
            }
      //
      //  check for devices to add
      //
      for (std::list<AlsaPort>::iterator k = portList.begin();
         k != portList.end(); ++k) {
            iMidiDevice i = midiDevices.begin();
// printf("ALSA port: <%s>\n", k->name);
            for (;i != midiDevices.end(); ++i) {
                  MidiAlsaDevice* d = dynamic_cast<MidiAlsaDevice*>(*i);
                  if (d == 0)
                        continue;
                  if ((k->adr.client == d->adr.client) && (k->adr.port == d->adr.port)) {
                        break;
                        }
                  }
            if (i == midiDevices.end()) {
                  // add device
                  MidiAlsaDevice* dev = new MidiAlsaDevice(k->adr,
                     k->name);
                  dev->setrwFlags(k->flags);
                  midiDevices.add(dev);
                  }
            }
      }

//---------------------------------------------------------
//   selectWfd
//---------------------------------------------------------

int MidiAlsaDevice::selectWfd()
      {
      return alsaSeqFd;
      }
#endif

