/*
 * Entry point and user interface for the applet.
 *
 * Music Applet
 * Copyright (C) 2004-2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include <config.h>

#include "ma-any-proxy.h"
#include "ma-conf.h"
#include "ma-launcher-button.h"
#include "ma-proxy.h"
#include "ma-rating.h"
#include "ma-song-tooltip.h"
#include "ma-stock-icons.h"
#include "ma-util.h"

#include <panel-applet.h>
#include <glib/gi18n.h>
#include <gtk/gtkaboutdialog.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtkvbox.h>
#include <glade/glade-xml.h>

#include <string.h>

/* Avoid compiler warnings in BONOBO_UI_VERB and friends. */
#define MY_BONOBO_UI_UNSAFE_VERB(name,cb) \
	{ (name), ((BonoboUIVerbFn)(cb)), NULL, NULL }
#define MY_BONOBO_UI_VERB_END             \
	{ NULL, NULL, NULL, NULL }

typedef struct {
	GtkWidget *applet;

	MaProxy *proxy;
	MaConf *conf;

	gint min_time_width;

	GtkTooltips *tooltips;
	MaSongTooltip *song_tooltip;

	GtkWidget *preferences_dialog;

	GtkWidget *main_box;
	GtkWidget *button_box;
	GtkWidget *launch;
	GtkWidget *rating;
	GtkWidget *time;
	GtkWidget *time_event;
	GtkWidget *previous;
	GtkWidget *toggle_playback;
	GtkWidget *next;
} AppletData;

static gboolean	ma_applet_fill (PanelApplet *applet,
				const gchar *iid,
				gpointer unused);

static void	set_initial_state (AppletData *data);

static void	recalculate_time_extents (AppletData *data);

static void	preferences_cb (BonoboUIComponent *uic,
				AppletData *data,
				const gchar *verbname);

static void	about_cb (BonoboUIComponent *uic,
			  AppletData *data,
			  const gchar *verbname);

static void	rate_cb (MaRating *rating,
			 double score,
			 AppletData *data);

static void	response_cb (GtkDialog *dialog,
			     gint response,
			     gpointer unused);

static void	launch_cb (GtkButton *button,
			   AppletData *data);

static void	toggle_playback_cb (GtkButton *button,
				    AppletData *data);

static void	previous_cb (GtkButton *button,
			     AppletData *data);

static void	next_cb (GtkButton *button,
			 AppletData *data);

static void	change_orient_cb (PanelApplet *applet,
				  PanelAppletOrient orient,
				  AppletData *data);

static void	change_background_cb (PanelApplet *applet,
				      PanelAppletBackgroundType type,
				      GdkColor *color,
				      GdkPixmap *pixmap,
				      AppletData *data);

static void	destroy_cb (GtkWidget *widget,
			    AppletData *data);

static void	style_set_cb (GtkWidget *widget,
			      GtkStyle *previous,
			      AppletData *data);

static void	time_size_request_cb (GtkWidget *widget,
				      GtkRequisition *requisition,
				      AppletData *data);

static void	time_updated_cb (MaProxy *proxy,
				 GParamSpec *pspec,
				 AppletData *data);

static void	title_changed_cb (MaProxy *proxy,
				  GParamSpec *pspec,
				  AppletData *data);

static void	rating_changed_cb (MaProxy *proxy,
				   GParamSpec *pspec,
				   AppletData *data);

static void	playing_changed_cb (MaProxy *proxy,
				    GParamSpec *pspec,
				    AppletData *data);

static void	state_changed_cb (MaProxy *proxy,
				  GParamSpec *psepc,
				  AppletData *data);

static void	icon_name_changed_cb (MaProxy *proxy,
				      GParamSpec *pspec,
				      AppletData *data);

static void	error_reported_cb (MaProxy *proxy,
				   const gchar *primary,
				   const gchar *secondary,
				   gboolean important,
				   AppletData *data);

static void	show_rating_changed_cb (GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer user_data);

static const BonoboUIVerb ma_applet_menu_verbs[] = {
	MY_BONOBO_UI_UNSAFE_VERB ("Preferences", preferences_cb),
	MY_BONOBO_UI_UNSAFE_VERB ("About", about_cb),
	MY_BONOBO_UI_VERB_END
};

/*********************************************************************
 *
 * The applet itself
 *
 *********************************************************************/

static gboolean
ma_applet_fill (PanelApplet *applet,
		const gchar *iid,
		gpointer unused)
{
	AppletData *data;
	GtkWidget *label;
	gchar *key;

	if (strcmp (iid, "OAFIID:GNOME_Music_Applet") != 0)
		return FALSE;

	data = g_new0 (AppletData, 1);
	data->applet = GTK_WIDGET (applet);
	panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);

	ma_stock_icons_init ();

	panel_applet_add_preferences (applet,
				      "/schemas/apps/music-applet/prefs",
				      NULL);

	data->min_time_width = 0;
	data->conf = ma_conf_new (applet);
	data->proxy = ma_any_proxy_new (data->conf);
	data->tooltips = gtk_tooltips_new ();
	data->song_tooltip = ma_song_tooltip_new (data->proxy);
	data->main_box = NULL;
	data->preferences_dialog = NULL;

	g_object_ref (data->tooltips);
	gtk_object_sink (GTK_OBJECT (data->tooltips));

	/* Applet widgets */

	data->launch = ma_launcher_button_new_from_stock (GTK_STOCK_EXECUTE);
	gtk_tooltips_set_tip (data->tooltips, data->launch,
			      _("Launch music player"), NULL);
	g_signal_connect (data->launch, "clicked",
			  G_CALLBACK (launch_cb), data);

	data->rating = GTK_WIDGET (ma_rating_new ());
	gtk_tooltips_set_tip (data->tooltips, data->rating,
			      _("Rate current song"), NULL);
	g_signal_connect (data->rating, "rated",
			  G_CALLBACK (rate_cb), data);

	data->time_event = gtk_event_box_new ();
	ma_song_tooltip_attach (data->song_tooltip, data->time_event);
	data->time = gtk_label_new ("--:--");
	gtk_container_add (GTK_CONTAINER (data->time_event), data->time);

	data->button_box = gtk_hbox_new (FALSE, 0);

	data->previous = ma_launcher_button_new_from_stock (GTK_STOCK_MEDIA_PREVIOUS);
	gtk_tooltips_set_tip (data->tooltips, data->previous,
			      _("Play previous song"), NULL);
	g_signal_connect (data->previous, "clicked",
			  G_CALLBACK (previous_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->previous);

	data->toggle_playback = ma_launcher_button_new_from_stock (GTK_STOCK_MEDIA_PLAY);
	gtk_tooltips_set_tip (data->tooltips, data->toggle_playback,
			      _("Start playing"), NULL);
	g_signal_connect (data->toggle_playback, "clicked",
			  G_CALLBACK (toggle_playback_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->toggle_playback);

	data->next = ma_launcher_button_new_from_stock (GTK_STOCK_MEDIA_NEXT);
	gtk_tooltips_set_tip (data->tooltips, data->next,
			      _("Play next song"), NULL);
	g_signal_connect (data->next, "clicked",
			  G_CALLBACK (next_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->next);

	panel_applet_setup_menu_from_file (applet,
					   DATA_DIR,
					   "GNOME_Music_Applet.xml",
					   NULL,
					   ma_applet_menu_verbs,
					   data);

	/* GConf */

	ma_conf_add_notify (data->conf, "show_rating", show_rating_changed_cb, data);

	/* Misc. signals */

	g_signal_connect (data->proxy, "notify::title",
			  G_CALLBACK (title_changed_cb), data);

	g_signal_connect (data->proxy, "notify::rating",
			  G_CALLBACK (rating_changed_cb), data);

	g_signal_connect (data->proxy, "notify::playing",
			  G_CALLBACK (playing_changed_cb), data);

	g_signal_connect (data->proxy, "notify::elapsed",
			  G_CALLBACK (time_updated_cb), data);

	g_signal_connect (data->proxy, "notify::state",
			  G_CALLBACK (state_changed_cb), data);

	g_signal_connect (data->proxy, "notify::icon-name",
			  G_CALLBACK (icon_name_changed_cb), data);

	g_signal_connect (data->proxy, "error-reported::important",
			  G_CALLBACK (error_reported_cb), data);

	g_signal_connect (applet, "change_orient",
			  G_CALLBACK (change_orient_cb), data);

	g_signal_connect (applet, "change_background",
			  G_CALLBACK (change_background_cb), data);

	g_signal_connect (applet, "destroy",
			  G_CALLBACK (destroy_cb), data);

	g_signal_connect (data->time, "style-set",
			  G_CALLBACK (style_set_cb), data);

	g_signal_connect (data->time, "size-request",
			  G_CALLBACK (time_size_request_cb), data);

	/* Initialize with current state. */

	set_initial_state (data);
	recalculate_time_extents (data);

	/* Finish up. */

	gtk_widget_show (GTK_WIDGET (applet));

	ma_proxy_enable (data->proxy);

	return TRUE;
}

static void
destroy_cb (GtkWidget *widget, AppletData *data)
{
	g_signal_handlers_disconnect_by_func (data->proxy, title_changed_cb, data);
	g_signal_handlers_disconnect_by_func (data->proxy, playing_changed_cb, data);
	g_signal_handlers_disconnect_by_func (data->proxy, time_updated_cb, data);
	g_signal_handlers_disconnect_by_func (data->applet, change_orient_cb, data);
	g_signal_handlers_disconnect_by_func (data->applet, destroy_cb, data);
	g_signal_handlers_disconnect_by_func (data->time, style_set_cb, data);

	g_object_unref (data->song_tooltip);
	g_object_unref (data->tooltips);

	g_object_unref (data->proxy);
	g_object_unref (data->conf);

	ma_stock_icons_shutdown ();

	g_free (data);
}


static void
set_initial_state (AppletData *data)
{
	/* eew, hackish */

	PanelAppletOrient orient = panel_applet_get_orient (
		PANEL_APPLET (data->applet));

	change_orient_cb (PANEL_APPLET (data->applet), orient, data);
	icon_name_changed_cb (data->proxy, NULL, data);
	state_changed_cb (data->proxy, NULL, data);
	title_changed_cb (data->proxy, NULL, data);
	rating_changed_cb (data->proxy, NULL, data);
	playing_changed_cb (data->proxy, NULL, data);
}

static void 
recalculate_time_extents (AppletData *data)
{
	gchar *zero;
	PangoLayout *layout;
	PangoRectangle extents;

	/* "minimum width" to avoid changing size in most cases */
	zero = ma_util_format_elapsed (0);
	layout = gtk_widget_create_pango_layout (data->time, zero);
	pango_layout_get_pixel_extents (layout, NULL, &extents);
	g_object_unref (layout);
	g_free (zero);

	data->min_time_width = extents.width;
}


/*********************************************************************
 *
 * User interface callbacks
 *
 *********************************************************************/

static void
launch_cb (GtkButton *button, AppletData *data)
{
	ma_proxy_launch (data->proxy);
}

static void
toggle_playback_cb (GtkButton *button, AppletData *data)
{
	ma_proxy_toggle_playback (data->proxy);
}

static void
previous_cb (GtkButton *button, AppletData *data)
{
	ma_proxy_previous (data->proxy);
}

static void
next_cb (GtkButton *button, AppletData *data)
{
	ma_proxy_next (data->proxy);
}

static void
rate_cb (MaRating *rating, double score, AppletData *data)
{
	ma_proxy_set_rating (data->proxy, score);
}


/*********************************************************************
 *
 * Preferences dialog
 *
 *********************************************************************/

static void
preferences_cb (BonoboUIComponent *uic,
		AppletData *data,
		const gchar *verbname)
{
	static GladeXML *xml = NULL;
	GtkWidget *show_rating;
	gboolean rating_visible;

	if (data->preferences_dialog == NULL)
	{
		if (xml != NULL)
			g_object_unref (xml);

		xml = glade_xml_new (PKG_DATA_DIR "music-applet.glade",
				     NULL, NULL);

		ma_proxy_prepare_prefs (data->proxy, xml);

		data->preferences_dialog = glade_xml_get_widget (
			xml, "preferences-dialog");

		show_rating = glade_xml_get_widget (xml, "show-rating");
		ma_conf_bind_bool (data->conf, "show_rating", GTK_CHECK_BUTTON (show_rating));

		g_signal_connect (data->preferences_dialog,
				  "response",
				  G_CALLBACK (response_cb),
				  NULL);

		g_signal_connect (data->preferences_dialog,
				  "destroy",
				  G_CALLBACK (gtk_widget_destroyed),
				  &data->preferences_dialog);

		gtk_widget_show (data->preferences_dialog);
	}
	else
	{
		gtk_window_set_screen (GTK_WINDOW (data->preferences_dialog),
				       gtk_widget_get_screen (
					       GTK_WIDGET (data->applet)));
		gtk_window_present (GTK_WINDOW (data->preferences_dialog));
	}
}

static void
response_cb (GtkDialog *dialog,
	     gint response,
	     gpointer unused)
{
	gtk_widget_hide (GTK_WIDGET (dialog));
}


/*********************************************************************
 *
 * About dialog
 *
 *********************************************************************/

static void
about_cb (BonoboUIComponent *uic,
	  AppletData *data,
	  const gchar *verbname)
{
	static const gchar *authors[] = {
		"Paul Kuliniewicz <paul.kuliniewicz@gmail.com>",
		NULL
	};

	gtk_show_about_dialog (NULL,
		"name", _("Music Applet"),
		"version", PACKAGE_VERSION,
		"copyright", _("(C) 2004-2006 Paul Kuliniewicz"),
		"comments", _("Control your favorite music player from a GNOME panel."),
		"authors", authors,
		"translator-credits", _("translator-credits"),
		"website", "http://web.ics.purdue.edu/~kuliniew/music-applet/",
		NULL);
}


/*********************************************************************
 *
 * Panel callbacks
 *
 *********************************************************************/

static void
change_orient_cb (PanelApplet *applet,
		  PanelAppletOrient orient,
		  AppletData *data)
{
	if (data->main_box != NULL)
	{
		g_object_ref (G_OBJECT (data->launch));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->launch);
				      
		g_object_ref (G_OBJECT (data->rating));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->rating);

		g_object_ref (G_OBJECT (data->time_event));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->time_event);

		g_object_ref (G_OBJECT (data->button_box));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->button_box);

		gtk_container_remove (GTK_CONTAINER (applet),
				      data->main_box);
		gtk_widget_destroy (data->main_box);
	}

	switch (orient)
	{
	case PANEL_APPLET_ORIENT_UP:
	case PANEL_APPLET_ORIENT_DOWN:
		data->main_box = gtk_hbox_new (FALSE, 0);
		break;
	case PANEL_APPLET_ORIENT_LEFT:
	case PANEL_APPLET_ORIENT_RIGHT:
		data->main_box = gtk_vbox_new (FALSE, 0);
		break;
	default:
		g_return_if_reached ();
	}

	ma_launcher_button_set_orientation (MA_LAUNCHER_BUTTON (data->launch), orient);
	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->launch);

	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->rating);

	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->time_event);
	gtk_widget_show_all (data->time_event);

	ma_launcher_button_set_orientation (MA_LAUNCHER_BUTTON (data->previous), orient);
	ma_launcher_button_set_orientation (MA_LAUNCHER_BUTTON (data->toggle_playback), orient);
	ma_launcher_button_set_orientation (MA_LAUNCHER_BUTTON (data->next), orient);
	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->button_box);
	gtk_widget_show_all (data->button_box);

	state_changed_cb (data->proxy, NULL, data);

	gtk_container_add (GTK_CONTAINER (applet), data->main_box);
	gtk_widget_show (data->main_box);
}

static void
change_background_cb (PanelApplet *applet,
		      PanelAppletBackgroundType type,
		      GdkColor *color,
		      GdkPixmap *pixmap,
		      AppletData *data)
{
	GtkRcStyle *rc_style;
	GtkStyle *style;

	gtk_widget_set_style (GTK_WIDGET (applet), NULL);
	gtk_widget_set_style (data->time_event, NULL);
	gtk_widget_set_style (data->rating, NULL);

	rc_style = gtk_rc_style_new ();
	gtk_widget_modify_style (GTK_WIDGET (applet), rc_style);
	gtk_widget_modify_style (data->time_event, rc_style);
	gtk_widget_modify_style (data->rating, rc_style);
	g_object_unref (rc_style);

	switch (type)
	{
	case PANEL_NO_BACKGROUND:

		break;

	case PANEL_COLOR_BACKGROUND:

		gtk_widget_modify_bg (GTK_WIDGET (applet),
				      GTK_STATE_NORMAL,
				      color);
		gtk_widget_modify_bg (data->time_event,
				      GTK_STATE_NORMAL,
				      color);
		gtk_widget_modify_bg (data->rating,
				      GTK_STATE_NORMAL,
				      color);

		break;

	case PANEL_PIXMAP_BACKGROUND:

		style = gtk_style_copy (GTK_WIDGET (applet)->style);
		if (style->bg_pixmap[GTK_STATE_NORMAL] != NULL)
			g_object_unref (style->bg_pixmap[GTK_STATE_NORMAL]);
		style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref (pixmap);

		gtk_widget_set_style (GTK_WIDGET (applet), style);
		gtk_widget_set_style (data->time_event, style);
		gtk_widget_set_style (data->rating, style);

		g_object_unref (style);

		break;

	default:

		g_critical ("Unexpected change_background type %d", type);

		break;
	}
}

static void
style_set_cb (GtkWidget *widget, GtkStyle *previous_style, AppletData *data)
{
	recalculate_time_extents (data);
}

static void
time_size_request_cb (GtkWidget *widget, GtkRequisition *requisition, AppletData *data)
{
	if (requisition->width < data->min_time_width)
		requisition->width = data->min_time_width;
}


/*********************************************************************
 *
 * Proxy callbacks
 *
 *********************************************************************/

static void
time_updated_cb (MaProxy *proxy, GParamSpec *pspec, AppletData *data)
{
	glong elapsed = ma_proxy_get_elapsed (proxy);

	if (elapsed >= 0)
	{
		gchar *text = ma_util_format_elapsed (elapsed);
		gtk_label_set_text (GTK_LABEL (data->time), text);
		g_free (text);
	}
	else
	{
		gtk_label_set_text (GTK_LABEL (data->time), "--:--");
	}
}

static void
title_changed_cb (MaProxy *proxy, GParamSpec *pspec, AppletData *data)
{
	g_object_set (data->rating, "inert", (ma_proxy_get_title (proxy) == NULL), NULL);

	playing_changed_cb (proxy, NULL, data);
	time_updated_cb (proxy, NULL, data);
}

static void
rating_changed_cb (MaProxy *proxy, GParamSpec *pspec, AppletData *data)
{
	g_object_set (data->rating, "score", ma_proxy_get_rating (proxy), NULL);
}

static void
playing_changed_cb (MaProxy *proxy, GParamSpec *pspec, AppletData *data)
{
	const gchar *stock_id;
	const gchar *tip;

	if (!ma_proxy_get_playing (proxy))
	{
		stock_id = GTK_STOCK_MEDIA_PLAY;
		tip = _("Start playing");
	}
	else
	{
		stock_id = GTK_STOCK_MEDIA_PAUSE;
		tip = _("Pause playback");
	}

	ma_launcher_button_set_stock_id (MA_LAUNCHER_BUTTON (data->toggle_playback), stock_id);
	gtk_tooltips_set_tip (data->tooltips, data->toggle_playback, tip, NULL);
}

static void
state_changed_cb (MaProxy *proxy, GParamSpec *pspec, AppletData *data)
{
	if (ma_proxy_get_state (proxy) == MA_PROXY_STATE_CONNECTED)
	{
		gtk_widget_hide_all (data->launch);
		if (ma_conf_get_bool (data->conf, "show_rating"))
			gtk_widget_show_all (data->rating);
		else
			gtk_widget_hide_all (data->rating);
		gtk_widget_show_all (data->time_event);
		gtk_widget_show_all (data->button_box);
	}
	else
	{
		gtk_widget_show_all (data->launch);
		gtk_widget_hide_all (data->rating);
		gtk_widget_hide_all (data->time_event);
		gtk_widget_hide_all (data->button_box);
	}
}

static void
icon_name_changed_cb (MaProxy *proxy, GParamSpec *pspec, AppletData *data)
{
	const gchar *icon_name = ma_proxy_get_icon_name (proxy);

	if (icon_name != NULL)
		ma_launcher_button_set_icon_name (MA_LAUNCHER_BUTTON (data->launch), icon_name);
	else
		ma_launcher_button_set_stock_id (MA_LAUNCHER_BUTTON (data->launch), GTK_STOCK_EXECUTE);
}

static void
error_reported_cb (MaProxy *proxy,
		   const gchar *primary,
		   const gchar *secondary,
		   gboolean important,
		   AppletData *data)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new (NULL,
					 0,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 "%s",
					 primary);

	if (secondary != NULL)
	{
		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  "%s",
							  secondary);
	}

	g_signal_connect_swapped (dialog, "response", G_CALLBACK (gtk_widget_destroy), dialog);
	gtk_widget_show (dialog);
}


/*********************************************************************
 *
 * GConf callbacks
 *
 *********************************************************************/

static void
show_rating_changed_cb (GConfClient *client,
			guint cnxn_id,
			GConfEntry *entry,
			gpointer user_data)
{
	AppletData *data = (AppletData *) user_data;
	GConfValue *value = gconf_entry_get_value (entry);
	MaProxyState state;

	if (value != NULL)
	{
		if (gconf_value_get_bool (value) &&
		    ma_proxy_get_state (data->proxy) == MA_PROXY_STATE_CONNECTED)
		{
			gtk_widget_show_all (data->rating);
		}
		else
			gtk_widget_hide_all (data->rating);
	}
}


/*********************************************************************
 *
 * Entry point
 *
 *********************************************************************/

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_Music_Applet_Factory",
			     PANEL_TYPE_APPLET,
			     "Music Applet Factory",
			     "0",
			     ma_applet_fill,
			     NULL);
