/*
 * Implementation of the GConf helper object.
 *
 * Music Applet
 * Copyright (C) 2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include "ma-conf.h"

#include <panel-applet-gconf.h>
#include <gtk/gtkfilechooserdialog.h>
#include <gtk/gtkstock.h>

#include <string.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_CONF, MaConfPrivate))


typedef struct _MaConfPrivate	MaConfPrivate;
typedef struct _MaConfState	MaConfState;

struct _MaConfPrivate
{
	PanelApplet *applet;
	GConfClient *gconf;
};

struct _MaConfState
{
	GConfClient *gconf;
	GtkWidget *widget;

	gint column;
	gchar *value;

	guint cnxn_id;
	gchar *full_key;
};

static GObjectClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_conf_class_init (MaConfClass *klass);
static void ma_conf_init (MaConf *conf);

static void ma_conf_dispose (GObject *object);
static void ma_conf_finalize (GObject *object);

static void bool_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void button_toggled_cb (GtkToggleButton *button, MaConfState *state);

static void string_entry_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void entry_changed_cb (GtkEntry *entry, MaConfState *state);

static void set_active_combo_box_entry (GtkComboBox *combo_box, gint column, const gchar *value);
static void string_combo_box_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void combo_box_changed_cb (GtkComboBox *combo_box, MaConfState *state);

static void string_radio_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void radio_toggled_cb (GtkToggleButton *button, MaConfState *state);

static void browse_clicked_cb (GtkButton *button, MaConfState *state);
static void browse_response_cb (GtkDialog *dialog, gint response, MaConfState *state);

static MaConfState *ma_conf_state_new (MaConf *conf,
				       const gchar *key,
				       GConfClientNotifyFunc func,
				       GtkWidget *widget,
				       gint column,
				       const gchar *value);
static void object_destroy_cb (GtkObject *object, MaConfState *state);


/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_conf_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaConfClass),				/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_conf_class_init,		/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaConf),				/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_conf_init,		/* instance_init */
			NULL
		};

		type = g_type_register_static (G_TYPE_OBJECT, "MaConf", &info, 0);
	}

	return type;
}

static void
ma_conf_class_init (MaConfClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_conf_dispose;
	object_class->finalize = ma_conf_finalize;

	g_type_class_add_private (klass, sizeof (MaConfPrivate));
}

static void
ma_conf_init (MaConf *conf)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	priv->applet = NULL;
	priv->gconf = gconf_client_get_default ();
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_conf_dispose (GObject *object)
{
	MaConf *conf = MA_CONF (object);
	MaConfPrivate *priv = GET_PRIVATE (conf);

	if (priv->applet != NULL)
	{
		g_object_unref (priv->applet);
		priv->applet = NULL;
	}

	if (priv->gconf != NULL)
	{
		g_object_unref (priv->gconf);
		priv->gconf = NULL;
	}

	parent_class->dispose (object);
}

static void
ma_conf_finalize (GObject *object)
{
	parent_class->finalize (object);
}


/*********************************************************************
 *
 * Generic public interface
 *
 *********************************************************************/

MaConf *
ma_conf_new (PanelApplet *applet)
{
	MaConf *conf;
	MaConfPrivate *priv;

	conf = g_object_new (MA_TYPE_CONF, NULL);

	priv = GET_PRIVATE (conf);

	priv->applet = applet;
	g_object_ref (applet);

	return conf;
}

guint
ma_conf_add_notify (MaConf *conf, const gchar *key, GConfClientNotifyFunc func, gpointer user_data)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	gchar *full_key;
	guint id;

	full_key = panel_applet_gconf_get_full_key (priv->applet, key);
	id = gconf_client_notify_add (priv->gconf, full_key, func, user_data, NULL, NULL);
	g_free (full_key);

	return id;
}


/*********************************************************************
 *
 * Booleans
 *
 *********************************************************************/

gboolean
ma_conf_get_bool (MaConf *conf, const gchar *key)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	return panel_applet_gconf_get_bool (priv->applet, key, NULL);
}

void
ma_conf_bind_bool (MaConf *conf, const gchar *key, GtkCheckButton *button)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	MaConfState *state;

	state = ma_conf_state_new (conf, key,
				   bool_changed_cb,
				   GTK_WIDGET (button), -1, NULL);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				      gconf_client_get_bool (priv->gconf, state->full_key, NULL));

	g_signal_connect (button, "toggled", G_CALLBACK (button_toggled_cb), state);
	g_signal_connect (button, "destroy", G_CALLBACK (object_destroy_cb), state);
}

static void
bool_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	MaConfState *state = (MaConfState *) user_data;
	GConfValue *value = gconf_entry_get_value (entry);

	if (value != NULL)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (state->widget), gconf_value_get_bool (value));
}

static void
button_toggled_cb (GtkToggleButton *button, MaConfState *state)
{
	gboolean value = gtk_toggle_button_get_active (button);
	gconf_client_set_bool (state->gconf, state->full_key, value, NULL);
}


/*********************************************************************
 *
 * Simple strings
 *
 *********************************************************************/

gchar *
ma_conf_get_string (MaConf *conf, const gchar *key)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	return panel_applet_gconf_get_string (priv->applet, key, NULL);
}

void
ma_conf_bind_string_entry (MaConf *conf, const gchar *key, GtkEntry *entry)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	MaConfState *state;
	gchar *value;

	state = ma_conf_state_new (conf, key,
				   string_entry_changed_cb,
				   GTK_WIDGET (entry), -1, NULL);

	value = gconf_client_get_string (priv->gconf, state->full_key, NULL);
	if (value != NULL)
		gtk_entry_set_text (entry, value);
	g_free (value);

	g_signal_connect (entry, "changed", G_CALLBACK (entry_changed_cb), state);
	g_signal_connect (entry, "destroy", G_CALLBACK (object_destroy_cb), state);
}

static void
string_entry_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	MaConfState *state = (MaConfState *) user_data;
	GConfValue *value = gconf_entry_get_value (entry);
	if (value != NULL)
		gtk_entry_set_text (GTK_ENTRY (state->widget), gconf_value_get_string (value));
}

static void
entry_changed_cb (GtkEntry *entry, MaConfState *state)
{
	const gchar *value = gtk_entry_get_text (entry);
	gconf_client_set_string (state->gconf, state->full_key, value, NULL);
}


/*********************************************************************
 *
 * Enumerated strings, in a combo box
 *
 *********************************************************************/

void
ma_conf_bind_string_combo_box (MaConf *conf, const gchar *key, GtkComboBox *combo_box, gint column)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	MaConfState *state;
	gchar *value;

	state = ma_conf_state_new (conf, key,
				   string_combo_box_changed_cb,
				   GTK_WIDGET (combo_box), column, NULL);

	value = gconf_client_get_string (priv->gconf, state->full_key, NULL);
	if (value != NULL)
		set_active_combo_box_entry (combo_box, column, value);
	g_free (value);

	g_signal_connect (combo_box, "changed", G_CALLBACK (combo_box_changed_cb), state);
	g_signal_connect (combo_box, "destroy", G_CALLBACK (object_destroy_cb), state);
}

static void
set_active_combo_box_entry (GtkComboBox *combo_box, gint column, const gchar *value)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *iter_value;
	gboolean match;

	g_return_if_fail (value != NULL);

	model = gtk_combo_box_get_model (combo_box);
	if (gtk_tree_model_get_iter_first (model, &iter))
	{
		do
		{
			gtk_tree_model_get (model, &iter, column, &iter_value, -1);
			match = (strcmp (value, iter_value) == 0);
			g_free (iter_value);

			if (match)
			{
				gtk_combo_box_set_active_iter (combo_box, &iter);
				return;
			}
		} while (gtk_tree_model_iter_next (model, &iter));
	}
}

static void
string_combo_box_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	MaConfState *state = (MaConfState *) user_data;
	GConfValue *value = gconf_entry_get_value (entry);
	if (value != NULL)
		set_active_combo_box_entry (GTK_COMBO_BOX (state->widget), state->column, gconf_value_get_string (value));
}

static void
combo_box_changed_cb (GtkComboBox *combo_box, MaConfState *state)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *value;

	model = gtk_combo_box_get_model (combo_box);
	if (gtk_combo_box_get_active_iter (combo_box, &iter))
	{
		gtk_tree_model_get (model, &iter, state->column, &value, -1);
		gconf_client_set_string (state->gconf, state->full_key, value, NULL);
		g_free (value);
	}
}


/*********************************************************************
 *
 * Enumerated strings, in radio buttons
 *
 *********************************************************************/

void
ma_conf_bind_string_radio (MaConf *conf, const gchar *key, const gchar *value, GtkRadioButton *button)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	MaConfState *state;
	gchar *current_value;

	state = ma_conf_state_new (conf, key,
				   string_radio_changed_cb,
				   GTK_WIDGET (button), -1, value);

	current_value = gconf_client_get_string (priv->gconf, state->full_key, NULL);
	if (current_value != NULL && strcmp (value, current_value) == 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);
	g_free (current_value);

	g_signal_connect (button, "toggled", G_CALLBACK (radio_toggled_cb), state);
	g_signal_connect (button, "destroy", G_CALLBACK (object_destroy_cb), state);
}

static void
string_radio_changed_cb (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	MaConfState *state = (MaConfState *) user_data;
	GConfValue *value = gconf_entry_get_value (entry);
	if (value != NULL && strcmp (state->value, gconf_value_get_string (value)) == 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (state->widget), TRUE);
}

static void
radio_toggled_cb (GtkToggleButton *button, MaConfState *state)
{
	if (gtk_toggle_button_get_active (button))
		gconf_client_set_string (state->gconf, state->full_key, state->value, NULL);
}


/*********************************************************************
 *
 * Strings, selected via a browse button
 *
 *********************************************************************/

void
ma_conf_bind_string_browse (MaConf *conf, const gchar *key, GtkButton *button)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	MaConfState *state;

	state = ma_conf_state_new (conf, key,
				   NULL,
				   NULL, -1, NULL);

	g_signal_connect (button, "clicked", G_CALLBACK (browse_clicked_cb), state);
	g_signal_connect (button, "destroy", G_CALLBACK (object_destroy_cb), state);
}

static void
browse_clicked_cb (GtkButton *button, MaConfState *state)
{
	if (state->widget == NULL)
	{
		state->widget = gtk_file_chooser_dialog_new (_("Choose path to music player"),
							     GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (button))),
							     GTK_FILE_CHOOSER_ACTION_OPEN,
							     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
							     GTK_STOCK_OK, GTK_RESPONSE_OK,
							     NULL);

		g_signal_connect (state->widget, "response", G_CALLBACK (browse_response_cb), state);
		g_signal_connect (state->widget, "destroy", G_CALLBACK (gtk_widget_destroyed), &state->widget);

		gtk_widget_show (state->widget);
	}
	else
	{
		gtk_window_set_screen (GTK_WINDOW (state->widget),
				       gtk_widget_get_screen (GTK_WIDGET (button)));
		gtk_window_present (GTK_WINDOW (state->widget));
	}
}

static void
browse_response_cb (GtkDialog *dialog, gint response, MaConfState *state)
{
	if (response == GTK_RESPONSE_OK)
	{
		GError *error = NULL;
		gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		gchar *utf8 = g_filename_to_utf8 (filename, -1, NULL, NULL, &error);

		if (utf8 != NULL && error == NULL)
			gconf_client_set_string (state->gconf, state->full_key, utf8, NULL);
		else if (error != NULL)
		{
			g_critical ("Failed to convert filename to UTF-8: %s", error->message);
			g_error_free (error);
		}

		g_free (filename);
		g_free (utf8);
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}


/*********************************************************************
 *
 * Miscellaneous
 *
 *********************************************************************/

static MaConfState *
ma_conf_state_new (MaConf *conf,
		   const gchar *key,
		   GConfClientNotifyFunc func,
		   GtkWidget *widget,
		   gint column,
		   const gchar *value)
{
	MaConfPrivate *priv = GET_PRIVATE (conf);

	MaConfState *state = g_new0 (MaConfState, 1);
	state->gconf = priv->gconf;
	g_object_ref (state->gconf);
	state->full_key = panel_applet_gconf_get_full_key (priv->applet, key);
	state->widget = widget;
	state->column = column;
	state->value = g_strdup (value);

	if (func != NULL)
	{
		state->cnxn_id = gconf_client_notify_add (state->gconf, state->full_key,
							  func, state,
							  NULL, NULL);
	}
	else
		state->cnxn_id = 0;

	return state;
}

static void
object_destroy_cb (GtkObject *object, MaConfState *state)
{
	if (state->cnxn_id != 0)
		gconf_client_notify_remove (state->gconf, state->cnxn_id);
	g_object_unref (state->gconf);
	g_free (state->full_key);
	g_free (state->value);
	g_free (state);
}
