/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 asyncfile.cpp  -  Core asynchronous file i/o wrapper

    begin                : Sat Sep 1 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "mthread.h"
#include "xsleep.h"
#include "asyncsocket.h"
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "tstring.h"
#include "asyncfile.h"
#include "rcobject.h"

class MAFThread;

#define HANDLE_UNKNOWN -2

class MAFDescriptor : public MRCObject {
public:
	MAFDescriptor(MAsyncFile* pAsyncFile);
	void Detach() { // null the m_pAsyncFile;
		m_mutex.lock();
		m_pAsyncFile = NULL;
		m_mutex.unlock();
	}
	MAsyncFile* GetFile() { return m_pAsyncFile; }
	void PostRequest(MAFRequest* pReq);
	//
	SAFileState m_actual;
	SAFileState m_requested;
	//
	inline void Lock() { m_mutex.lock(); }
	inline void Unlock() { m_mutex.unlock(); }
protected:
	MAFThread*  m_pThread;
	MAsyncFile* m_pAsyncFile;
	int m_nMode;
	//
	virtual ~MAFDescriptor();
};

bool operator==(const SAFileState& l, const SAFileState& r) {
	return  l.hFile == HANDLE_UNKNOWN || r.hFile == HANDLE_UNKNOWN ||
			(l.hFile == r.hFile && l.nSize == r.nSize && l.nPos == r.nPos);
}

class MAFThread : public MThread {
public:
	MAFThread();
	~MAFThread();
	void OnRequestCompleted(MAFRequest*);
	static bool CheckAndCreatePath(const CString& path);
	void run();
	void SendStopRequest();
	void Enqueue(MAFRequest* pRequest);
	//
	MMutex                       m_mtxRequest;
	MWaitCondition               m_waitRequest;
	std::deque< TSmartPtr<MAFRequest> > m_queueRequest;
private:
	bool _eof(int hFile)
	{
		off_t pos = lseek(hFile,0,SEEK_CUR);
		if (pos == lseek(hFile,0,SEEK_END))
			return true;
		lseek(hFile,pos,SEEK_SET);
		return false;
	}
	long _tell(int hFile)
	{
		return lseek(hFile,0,SEEK_CUR);
	}
};

class MAFDelayed : public MAsyncSocket
{
public:
	MAFDelayed();
	~MAFDelayed();

	void Enqueue(MAFRequest* pRequest);
protected:
	MMutex m_mutex;
	std::deque< TSmartPtr<MAFRequest> > m_queueRequest;
	//
	SOCKET  m_hOtherSide;
	//
	virtual void OnReceive(int nErrorCode);
};

class MAFContext
{
public:
	MMutex mutex;
	MAFThread* pAFThreadRead;
	MAFThread* pAFThreadWrite;
	int nReadClients;
	int nWriteClients;
	//
	MAFDelayed m_Delayed;
	//
	MAFContext(){
		pAFThreadRead = NULL;
		pAFThreadWrite = NULL;
		nReadClients = 0;
		nWriteClients = 0;
	}
	~MAFContext(){
	}
	void WaitForThreadsStop()
	{
		if (nReadClients || nWriteClients)
		{
			int nsresult = -1;
			struct timespec rqtp, rmtp;

			rqtp.tv_sec = 3;
			rqtp.tv_nsec = rmtp.tv_sec = rmtp.tv_nsec = 0;	

			// Give 'em a chance to close.
			// That means waiting three actual seconds.
			while (nsresult!=0)
			{
				nsresult = safe_nanosleep(&rqtp, &rmtp);
				if (nsresult!=0)
				{
					if (errno==EINTR) 
					{
						// Reset our timespecs and rerun
						// the remaining time.
						rqtp.tv_sec = rmtp.tv_sec;
						rqtp.tv_nsec = rmtp.tv_nsec;
						rmtp.tv_sec = rmtp.tv_nsec = 0;
					}
					else nsresult=0; //got something else.
				} 
			}
		}	
		ASSERT(0==nReadClients);
		ASSERT(0==nWriteClients);
		if (pAFThreadRead && 0==nReadClients)
		{
			TRACE("waiting for async read thread...");
			pAFThreadRead->SendStopRequest();
			//pAFThreadRead->wait();
			delete pAFThreadRead;
		}
		if (pAFThreadWrite && 0==nWriteClients)
		{
			TRACE("waiting for async write thread...");
			pAFThreadWrite->SendStopRequest();
			//pAFThreadWrite->wait();
			delete pAFThreadWrite;
		}
	}
};

static MAFContext* s_pContext = NULL;

MAFThread::MAFThread()
{
}

MAFThread::~MAFThread()
{
}

void MAFThread::OnRequestCompleted(MAFRequest* pRequest)
{
	if (pRequest->pFileDesc == NULL)
		return;
	// update the file descriptor
	pRequest->pFileDesc->Lock();
	pRequest->pFileDesc->m_actual = pRequest->file_state;
	#warning This requires no requests to be sent before OPEN or ATTACH competes
	if (pRequest->type == AFR_OPEN || pRequest->type == AFR_ATTACH)
		pRequest->pFileDesc->m_requested = pRequest->file_state;
	// call sync notifyers
	MAsyncFile* pAF = pRequest->pFileDesc->GetFile();
	if (pAF)
	{
		pAF->m_dwCurrCompReq = pRequest->seqID;
		//
		if (pRequest->file_state.nErrNo)
			pAF->OnError(pRequest->seqID, pRequest->type, pRequest->file_state);
		else
			pAF->OnSuccess(pRequest->seqID, pRequest->type, pRequest->file_state);
		//
		pRequest->pFileDesc->Unlock();
		// queue the request for async notifiers
		ASSERT(s_pContext);
		s_pContext->m_Delayed.Enqueue(pRequest);
	}
	else
		pRequest->pFileDesc->Unlock();
}

bool MAFThread::CheckAndCreatePath(const CString& path)
{
	// remove the last element from the path
	CString part_path = path.substr(0,path.rfind("/"));
	while (part_path.length() && part_path[part_path.length()-1]=='/')
		part_path = part_path.substr(0,part_path.length()-1);
	if (!part_path.length())
		return false;
	// stat the path to check if the directory exists
	struct stat st;
	if (0==stat(part_path.c_str(), &st) && (S_ISDIR(st.st_mode)))
		return true;
	// it doesnt exist -- try to create
	TRACE3("MAFThread::CheckAndCreatePath: creating the directory \'", part_path, "\'");
	if (0==mkdir(part_path.c_str(), S_IRWXU))
		return true;
	// if cannot create -- try to do it recursively up the tree
	if (!CheckAndCreatePath(part_path))
		return false;
	// tree up to the current location has been created somehow -- now create the current dir
	TRACE3("MAFThread::CheckAndCreatePath: creating the directory \'", part_path, "\'");
	return 0==mkdir(part_path.c_str(), S_IRWXU);
}

void MAFThread::run()
{
	TSmartPtr<MAFRequest> pRequest;
	m_mtxRequest.lock();
	int type_cache;
	while (1)
	{
		// request mutex is locked here
		if (m_queueRequest.size()==0)
			if (!m_waitRequest.wait(&m_mtxRequest))
				continue;
		// request mutex is still locked here
		pRequest = m_queueRequest.front();
		m_queueRequest.pop_front();
		m_mtxRequest.unlock();
		// check whether the predicted state is correct. otherwise the request is incorrect
		// we could lock pRequest->pFileDesc here but only this thread ever modifies the
		// pRequest->pFileDesc->m_actual fields
		if (pRequest->type != AFR_STOP && pRequest->type != AFR_CLOSE && !(pRequest->file_state == pRequest->pFileDesc->m_actual))
		{
			// the state is wrong -- some of the previous requests failed!
			ASSERT(0);
			// call on error
			#warning call error notifiers here!!!
			// release the request
			pRequest = NULL;
			// continue looping
			m_mtxRequest.lock();
			continue;
		}
		//
		SAFileState& state = pRequest->file_state; // only to save typing and -> calls
		
		if (pRequest->type & AFR_SEEK)
		{
			if (state.hFile>=0)
			{
				ASSERT(_tell(state.hFile) == state.nPos);
				state.nReturn = lseek(state.hFile, pRequest->offset, pRequest->whence);
				if ( state.nReturn < 0 )
					state.nErrNo = errno;
				else
					state.nErrNo = 0;
				state.nPos = _tell(state.hFile);
				state.bEoF = _eof(state.hFile);
			}
		}
		switch ( pRequest->type )
		{
			case AFR_OPEN:
				if (state.hFile>=0)
				{
					close(state.hFile);
				}
				if (pRequest->mode & AFM_WRITE)
				{
					state.hFile = open(state.sPath.c_str(), O_CREAT|O_RDWR, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
					if (state.hFile < 0 && (pRequest->mode & AFM_CREATEPATH))
					{
						// my be we still can fix this little problem
						if (CheckAndCreatePath(state.sPath))
						{
							// now try again
							state.hFile = open(state.sPath.c_str(), O_CREAT|O_RDWR, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
						}
					}
				}
				else
					state.hFile = open(state.sPath.c_str(), O_RDONLY );
				//
				state.nReturn = state.hFile;
				if (state.nReturn<0)
				{
					state.nErrNo = errno;
					if (state.nErrNo == 0) // this is possible if CheckAndCreatePath fails or whatever
						state.nErrNo = -1; // unknown error :-)
				}
				else
					state.nErrNo = 0;
				// continue here
			case AFR_ATTACH:
				if ( pRequest->type == AFR_ATTACH )
				{
					if (state.hFile>=0)
						close(state.hFile);
					state.hFile = pRequest->hAttachFile;
				}
				if (state.hFile>=0)
				{
					state.nSize = lseek(state.hFile,0,SEEK_END);
					lseek(state.hFile,0,SEEK_SET);
					state.bEoF = _eof(state.hFile);
				}
				else
				{
					state.nSize = 0;
					state.bEoF = true;
				}
				state.nPos = 0;
				if (pRequest->type == AFR_ATTACH)
				{
					state.nReturn = 0;
					state.nErrNo = 0;
				}
				break;
			case AFR_PREREAD|AFR_SEEK:
			case AFR_READ|AFR_SEEK:
			case AFR_PREREAD:
			case AFR_READ:
				if (state.hFile>=0)
				{
					//TODO: properly react here -- say close file
					ASSERT(_tell(state.hFile) == state.nPos);
					state.nReturn = read(state.hFile, pRequest->pBuffer->buffer(), min(pRequest->pBuffer->volume(), pRequest->size));
					if (state.nReturn <= 0)
						state.nErrNo = errno;
					else
					{
						state.nPos += state.nReturn;
						state.nErrNo = 0;
					}
					state.bEoF = _eof(state.hFile);
				}
				break;
			case AFR_WRITE|AFR_SEEK:
			case AFR_WRITE:
				if (state.hFile>=0)
				{
					//TODO: properly react here -- say close file
					ASSERT(_tell(state.hFile) == state.nPos);
					int nSize = min(pRequest->pBuffer->volume(), pRequest->size);
					state.nReturn = write(state.hFile, pRequest->pBuffer->buffer() , nSize);
					if ( nSize != state.nReturn )
					{
						state.nErrNo = errno;
					}
					if (state.nReturn > 0)
					{
						state.nPos += state.nReturn;
						state.nErrNo = 0;
						if (state.nPos>state.nSize)
							state.nSize = state.nPos;
					}
					state.bEoF = _eof(state.hFile);
				}
				break;
			case AFR_CLOSE:
			case AFR_STOP:
				//if (!(pRequest->mode & AFM_WRITE))
				//{
				//	TRACE4("Closing file '", state.sPath, "'  fd = ", state.hFile);
				//}
				if (state.hFile>=0)
					state.nReturn = close(state.hFile);
				else
					state.nReturn = 0;
				state.hFile = -1;
				if (state.nReturn<0)
					state.nErrNo = errno;
				else
					state.nErrNo = 0;
				break;
			case AFR_DETACH:
				state.nReturn = 0;
				state.nErrNo = 0;
				state.hFile = -1;
				break;
			case AFR_NONE:
				if (pRequest->type != AFR_SEEK)
				{
					TRACE("AFR_NONE: should not happen");
				}
			case AFR_SEEK:
				break;
			case AFR_CUSTOM:
				state.nReturn = 0;
				state.nErrNo = 0;
				ASSERT(pRequest->pCustomRequest != NULL);
				if (pRequest->pCustomRequest==NULL || !pRequest->pCustomRequest->Do(pRequest))
				{
					if (state.nErrNo == 0)
						state.nErrNo = EIO;
					if (state.nReturn >= 0)
						state.nReturn = -1;
				}
				break;
			default:
				TRACE2("unknown async-file request ", pRequest->type);
		}
		// notify the world about how we are doing
		OnRequestCompleted(pRequest);
		//
		if (pRequest->type == AFR_STOP)
		{
			// queue mutex is unlocked here
			return;
		}
		// release the request
		pRequest = NULL;
		// go for the next one!
		m_mtxRequest.lock();
	}
}

void MAFThread::Enqueue(MAFRequest* pRequest)
{
	m_mtxRequest.lock();
	m_queueRequest.push_back(pRequest);
	m_waitRequest.wakeAll();
	m_mtxRequest.unlock();
}

void MAFThread::SendStopRequest()
{
	MAFRequest* pRequest = new MAFRequest;
	pRequest->file_state.hFile = -1;
	pRequest->pFileDesc = NULL;
	pRequest->type = AFR_STOP;
	m_mtxRequest.lock();
	m_queueRequest.push_back(pRequest);
	pRequest->Release();
	m_waitRequest.wakeAll();
	wait(&m_mtxRequest);
	m_mtxRequest.unlock();
}

////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////

MAFDescriptor::MAFDescriptor(MAsyncFile* pAsyncFile) : MRCObject(true), m_pAsyncFile(pAsyncFile), m_nMode(pAsyncFile->m_nMode)
{
	m_actual.hFile   = -1;
	m_actual.nSize   = 0;
	m_actual.nPos    = 0;
	m_actual.nErrNo  = 0;
	m_actual.bEoF    = true;
	m_actual.nReturn = 0;
	m_requested = m_actual;
	//
	ASSERT(s_pContext);
	s_pContext->mutex.lock();
	if (m_nMode == AFM_READ)
	{
		if (s_pContext->pAFThreadRead==NULL)
		{
			s_pContext->pAFThreadRead = new MAFThread();
			ASSERT(s_pContext->pAFThreadRead);
			s_pContext->pAFThreadRead->start();
		}
	    m_pThread=s_pContext->pAFThreadRead;
	    ++s_pContext->nReadClients;
	}
	else
	{
		if (s_pContext->pAFThreadWrite==NULL)
		{
			s_pContext->pAFThreadWrite = new MAFThread();
			ASSERT(s_pContext->pAFThreadWrite);
			s_pContext->pAFThreadWrite->start();
		}
	    m_pThread=s_pContext->pAFThreadWrite;
	    ++s_pContext->nWriteClients;
	}
	s_pContext->mutex.unlock();
}

MAFDescriptor::~MAFDescriptor()
{
	ASSERT(s_pContext);
	ASSERT(m_actual.hFile < 0);
	s_pContext->mutex.lock();
	// here we have to notify the MAsyncFile object about completion
	if (m_nMode == AFM_READ)
		--s_pContext->nReadClients;
	else
		--s_pContext->nWriteClients;
	s_pContext->mutex.unlock();
}

void MAFDescriptor::PostRequest(MAFRequest* pReq)
{
	ASSERT(m_pThread);
	m_pThread->Enqueue(pReq);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////

MAFDelayed::MAFDelayed()
{
	int sock_pair[2];
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, sock_pair))
	{
		cout << "Failed to create pair of sockets\n";
		exit(1);
	}
	// make sockets non-blocking
	::fcntl(sock_pair[0],F_SETFL,O_NONBLOCK);
	::fcntl(sock_pair[1],F_SETFL,O_NONBLOCK);
	// disable Nagle algorithm
	//int val = 1;
	//setsockopt(sock_pair[0],SOL_SOCKET,SO_NODELAY,(char*)&val,sizeof(val));
	//setsockopt(sock_pair[1],SOL_SOCKET,SO_NODELAY,(char*)&val,sizeof(val));
	//
	m_hOtherSide = sock_pair[0];
	Attach(sock_pair[1],FD_READ);
}

MAFDelayed::~MAFDelayed()
{
	::close(m_hOtherSide);
}

void MAFDelayed::Enqueue(MAFRequest* pRequest)
{
	m_mutex.lock();
	m_queueRequest.push_back(pRequest);
	//
	::send(m_hOtherSide, "f", 1, 0); // 'f' means "file" :-))
	//
	m_mutex.unlock();
}

void MAFDelayed::OnReceive(int nErrorCode)
{
	static char tmp[128];
	TSmartPtr<MAFRequest> pRequest;
	m_mutex.lock();
	while (m_queueRequest.size())
	{
		pRequest = m_queueRequest.front();
		m_queueRequest.pop_front();
		m_mutex.unlock();
		// call the async notifiers
		pRequest->pFileDesc->Lock();
		MAsyncFile* pAF = pRequest->pFileDesc->GetFile();
		if (pAF)
		{
			pAF->m_dwCurrDelayedComplReq = pRequest->seqID;
			//
			if (pRequest->file_state.nErrNo)
				pAF->OnErrorDelayed(pRequest->seqID, pRequest->type, pRequest->file_state);
			else
				pAF->OnSuccessDelayed(pRequest->seqID, pRequest->type, pRequest->file_state);
		}
		pRequest->pFileDesc->Unlock();
		//
		m_mutex.lock();
	}
	// 
	while (Receive(tmp, 128)>0);
	//
	m_mutex.unlock();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////

#define MAF_REQUEST_BEGIN(_type, _req, _bad_return) \
	m_pDesc->Lock();                                  \
	_req->pFileDesc = m_pDesc;                        \
	_req->file_state = m_pDesc->m_requested;          \
	_req->type = _type;

#define MAF_REQUEST_END(_req)    \
	_req->seqID = ++m_dwCurrReq; \
	m_pDesc->Unlock();           \
	m_pDesc->PostRequest(_req);  \
	_req->Release();

MAsyncFile::MAsyncFile(int mode /*=AFM_READWRITE*/)
{
	m_nMode     = mode;
	m_dwCurrReq = 0;
	m_dwCurrCompReq = 0;
	m_dwCurrDelayedComplReq = 0;
	//
	m_pDesc = new MAFDescriptor(this);
	m_pDesc->Release(); // smartPtr takes responsibility...
}


MAsyncFile::~MAsyncFile()
{
	Close();
	m_pDesc->Detach();
}

void MAsyncFile::InitStaticStructures() // MUST be called before the first use!!!
{
	s_pContext = new MAFContext;
}

void MAsyncFile::WaitForThreadsStop()
{
	ASSERT(s_pContext);
	s_pContext->WaitForThreadsStop();
}

void MAsyncFile::OnSuccess(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	//cout << "success\n";
}

void MAsyncFile::OnError(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	//cout << "error\n";
}

void MAsyncFile::OnSuccessDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	//cout << "success\n";
}

void MAsyncFile::OnErrorDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	//cout << "error\n";
}

/*bool MAsyncFile::IsInProgress()
{
	MLock lock(m_pThread->m_mtxRequest);
	
	return m_pThread->HaveRequests(this);
}

bool MAsyncFile::IsReady()
{
	MLock lock(m_pThread->m_mtxRequest);
	
	return !m_pThread->HaveRequests(this);
}

bool MAsyncFile::WaitTillReady()
{
	MAF_REQUEST_CHECK(AFR_WAIT, false)
        m_pThread->m_mtxRequest.unlock();
        return true;
}*/

/*void MAsyncFile::Destroy()
{
	ASSERT(m_pThread==s_pContext->pAFThreadRead || m_pThread==s_pContext->pAFThreadWrite);
	//MAF_REQUEST_CHECK(AFR_FREE, false)
	m_pThread->m_mtxRequest.lock();
	m_pRequest->type = AFR_FREE;
	MAF_REQUEST_END
}*/

bool MAsyncFile::IsOpen()
{
	return m_pDesc->m_actual.hFile>=0;
}

bool MAsyncFile::Open(LPCSTR szName)
{
	ASSERT(szName);
	ASSERT(strlen(szName)<65536);
	//
	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_OPEN, pReq, false)
	pReq->file_state.sPath = szName;
	pReq->mode  = m_nMode;
	// update the predicted state
	m_pDesc->m_requested.hFile = HANDLE_UNKNOWN;
	//
	MAF_REQUEST_END(pReq)
	//
	return true;
}

bool MAsyncFile::IsClosed()
{
	return m_pDesc->m_actual.hFile<0;
}

bool MAsyncFile::Close()
{
	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_CLOSE, pReq, false)
	// update the predicted state
	m_pDesc->m_requested.hFile = HANDLE_UNKNOWN; // this is better than -1 because we might want to reopen file again
	MAF_REQUEST_END(pReq)
	//
	return true;
}

int MAsyncFile::GetLastError()
{
	return m_pDesc->m_actual.nErrNo;
}

bool MAsyncFile::Attach(int hFile)
{
	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_ATTACH, pReq, false)
	pReq->hAttachFile = hFile;
	// update the predicted state
	m_pDesc->m_requested.hFile = HANDLE_UNKNOWN; // hFile would force us to know the size of the file and so on...
	MAF_REQUEST_END(pReq)
	//
	return true;
}

int MAsyncFile::Detach()
{
	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_DETACH, pReq,  -1)
	int hFile = pReq->file_state.hFile;
	// update the predicted state
	m_pDesc->m_requested.hFile = HANDLE_UNKNOWN; // this is better than -1 because we might want to reopen file again
	MAF_REQUEST_END(pReq)
	return hFile;
}

bool MAsyncFile::Seek(int nOffset, int nWhence)
{
	if ( nWhence == SEEK_CUR && nOffset == 0 )
		return true;
		
	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_SEEK, pReq, false)
	pReq->offset = nOffset;
	pReq->whence = nWhence;
	// update the predicted state
	switch (nWhence)
	{
		case SEEK_SET: m_pDesc->m_requested.nPos = nOffset; break;
		case SEEK_CUR: m_pDesc->m_requested.nPos += nOffset; break;
		case SEEK_END: m_pDesc->m_requested.nPos = m_pDesc->m_requested.nSize + nOffset; break;
	}
	MAF_REQUEST_END(pReq)
	return true;
}

int MAsyncFile::GetPos()
{
	if (!IsOpen() || GetLastError())
		return -1;
	return m_pDesc->m_actual.nPos;
}

int MAsyncFile::GetAsyncPos()
{
	if (!IsOpen() || GetLastError())
		return -1;
	return m_pDesc->m_requested.nPos;
}

int MAsyncFile::GetSize()
{
	if (!IsOpen() || GetLastError())
		return -1;
	return m_pDesc->m_actual.nSize;
}

int MAsyncFile::GetAsyncSize()
{
	if (!IsOpen() || GetLastError())
		return -1;
	return m_pDesc->m_requested.nSize;
}

bool MAsyncFile::EoF()
{
	if (!IsOpen() || GetLastError())
		return true;
	return m_pDesc->m_actual.bEoF;
}

int MAsyncFile::Read(MAFBuffer* pBuff, int nBytes) // return number of bytes read or -1
{
	if (0==(m_nMode&AFM_READ))
		return -1;

	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_READ, pReq, -1)
	pReq->pBuffer = pBuff;
	pReq->size = nBytes;
	// update the predicted state
	m_pDesc->m_requested.nPos += nBytes;
	if (m_pDesc->m_requested.nSize < m_pDesc->m_requested.nPos)
		m_pDesc->m_requested.nPos = m_pDesc->m_requested.nSize;
	MAF_REQUEST_END(pReq)
	return 0;
}

int MAsyncFile::ReadSeek(int nOffset, int nWhence, MAFBuffer* pBuff, int nBytes) // return number of bytes read or -1
{
	if (0==(m_nMode&AFM_READ))
		return -1;

	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_READ|AFR_SEEK, pReq, -1)
	pReq->pBuffer = pBuff;
	pReq->size = nBytes;
	pReq->offset = nOffset;
	pReq->whence = nWhence;
	// update the predicted state
	switch (nWhence)
	{
		case SEEK_SET: m_pDesc->m_requested.nPos = nOffset; break;
		case SEEK_CUR: m_pDesc->m_requested.nPos += nOffset; break;
		case SEEK_END: m_pDesc->m_requested.nPos = m_pDesc->m_requested.nSize + nOffset; break;
	}
	m_pDesc->m_requested.nPos += nBytes;
	if (m_pDesc->m_requested.nSize < m_pDesc->m_requested.nPos)
		m_pDesc->m_requested.nPos = m_pDesc->m_requested.nSize;
	MAF_REQUEST_END(pReq)
	return 0;
}

int MAsyncFile::Write(MAFBuffer* pBuff, int nBytes)
{
	if (0==(m_nMode&AFM_WRITE))
		return -1;
	if (nBytes<=0)
		return 0;

	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_WRITE, pReq, -1)
	pReq->pBuffer = pBuff;
	pReq->size = nBytes;
	// update the predicted state
	m_pDesc->m_requested.nPos += nBytes;
	if (m_pDesc->m_requested.nSize < m_pDesc->m_requested.nPos)
		m_pDesc->m_requested.nSize = m_pDesc->m_requested.nPos;
	MAF_REQUEST_END(pReq)
	return 0;
}

int MAsyncFile::WriteSeek(int nOffset, int nWhence, MAFBuffer* pBuff, int nBytes)
{
	if (0==(m_nMode&AFM_WRITE))
		return -1;
	if (nBytes<=0)
		return 0;

	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_WRITE|AFR_SEEK, pReq, -1)
	pReq->pBuffer = pBuff;
	pReq->size = nBytes;
	pReq->offset = nOffset;
	pReq->whence = nWhence;
	// update the predicted state
	switch (nWhence)
	{
		case SEEK_SET: m_pDesc->m_requested.nPos = nOffset; break;
		case SEEK_CUR: m_pDesc->m_requested.nPos += nOffset; break;
		case SEEK_END: m_pDesc->m_requested.nPos = m_pDesc->m_requested.nSize + nOffset; break;
	}
	m_pDesc->m_requested.nPos += nBytes;
	if (m_pDesc->m_requested.nSize < m_pDesc->m_requested.nPos)
		m_pDesc->m_requested.nSize = m_pDesc->m_requested.nPos;
	MAF_REQUEST_END(pReq)
	return 0;
}

bool MAsyncFile::CustomRequest(MCustomAFRequest* pRequestObject)
{
	MAFRequest* pReq = new MAFRequest;
	//
	MAF_REQUEST_BEGIN(AFR_CUSTOM, pReq, false)
	pReq->type = AFR_CUSTOM;
	pReq->pCustomRequest = pRequestObject;
	MAF_REQUEST_END(pReq)
	return true;
} 

////////////////////////////////////////////////////////////////////////////////////
// just the async delete and move requests -- I could not find a better place for it

MRequestDelete::MRequestDelete(const CString& path) : m_path(path)
{
}

bool MRequestDelete::Do(MAFRequest* pRequest)
{
	pRequest->file_state.nReturn = unlink(m_path.c_str());
	return pRequest->file_state.nReturn == 0;
}

void MRequestDelete::Predict(SAFileState& state)
{
	// this is really brutal request, dont know what to wrire here, really
}

MRequestMove::MRequestMove(const CString& pathOld, const CString& pathNew) :
	m_pathOld(pathOld),
	m_pathNew(pathNew)
{}

bool MRequestMove::Do(MAFRequest* pRequest)
{
	pRequest->file_state.nReturn = rename(m_pathOld.c_str(), m_pathNew.c_str());
	return pRequest->file_state.nReturn == 0;
}

void MRequestMove::Predict(SAFileState& state)
{
	// this does not really change the file
	state.sPath = m_pathNew;
}

