unit AdminServiceControl;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  gnugettext, Windows, Messages, SysUtils, Variants, Classes,
  TntClasses, Graphics, TntControls, TntForms,
  Dialogs, TntExtCtrls, TntStdCtrls, TntButtons, Contnrs,
  Sections, InstanceSections, PNGImage,
  AuxFuncs, ComCtrls, myx_public_interface,
  myx_admin_public_interface, AdminService,
  Registry, WinSvc, Menus, Options, MyxError, ExtCtrls, TntMenus,
  TntComCtrls, StdCtrls, Forms, Buttons, Controls, TntRegistry,
  TntDialogs;

type
  TAdminServiceControlForm = class;

  TStartStopServiceThread = class(TThread)
    constructor Create(CreateSuspended: Boolean;
      AdminServiceControlForm: TAdminServiceControlForm;
      StartService: Boolean; ServiceName: WideString); reintroduce;
    destructor Destroy; override;
  protected
    procedure Execute; override;
  public
    AdminServiceControlForm: TAdminServiceControlForm;
    StartService: Boolean;
    ServiceName: WideString;
  end;

  TAdminServiceControlForm = class(TInstanceSectionForm)
    ServiceControlPnl: TTntPanel;
    PageControl: TTntPageControl;
    StartStopServiceSheet: TTabSheet;
    ServiceConfigSheet: TTabSheet;
    SettingsScrollBox: TTntScrollBox;
    ServiceControlCBox: TTntGroupBox;
    LogMessageLbl: TTntLabel;
    StartupLogMemo: TTntMemo;
    SubTreePnl: TTntPanel;
    Panel2: TTntPanel;
    Label15: TTntLabel;
    ServicesTreeView: TTntTreeView;
    Panel1: TTntPanel;
    UserInfoBevel: TTntBevel;
    ServiceNameHeaderLbl: TTntLabel;
    PageHeaderImg: TTntImage;
    UserInfoLbl: TTntLabel;
    PopupMenu: TTntPopupMenu;
    InstallnewServiceMI: TTntMenuItem;
    UninstallselectedServiceMI: TTntMenuItem;
    Panel4: TTntPanel;
    Bevel1: TTntBevel;
    ServiceNameHeader2Lbl: TTntLabel;
    PageHeader2Img: TTntImage;
    Label14: TTntLabel;
    ServiceStatusGBox: TTntGroupBox;
    ServerStatusLbl: TTntLabel;
    Label1: TTntLabel;
    ServiceStatusImg: TTntImage;
    RefreshServiceStatusMI: TTntMenuItem;
    N1: TTntMenuItem;
    StartServiceBtn: TTntBitBtn;
    StopServiceBtn: TTntBitBtn;
    StartServiceLbl: TTntLabel;
    StopServiceLbl: TTntLabel;
    ConnectToInstanceAni: TAnimate;
    ServiceSettingsGBox: TTntGroupBox;
    Label44: TTntLabel;
    Label3: TTntLabel;
    Label4: TTntLabel;
    Label5: TTntLabel;
    Label6: TTntLabel;
    ServiceAutoStartCBox: TTntCheckBox;
    ServiceDisplayNameEd: TTntEdit;
    ServiceDescriptionEd: TTntEdit;
    ServiceConfigFileGBox: TTntGroupBox;
    Label7: TTntLabel;
    Label8: TTntLabel;
    Label9: TTntLabel;
    Label10: TTntLabel;
    ConfigFilenameEd: TTntEdit;
    ServiceSectionNameEd: TTntEdit;
    ServiceFeaturesGBox: TTntGroupBox;
    Label57: TTntLabel;
    Label58: TTntLabel;
    Label61: TTntLabel;
    Label62: TTntLabel;
    Label11: TTntLabel;
    Label12: TTntLabel;
    PathToBinaryEd: TTntEdit;
    ServiceFeatureDebugCBox: TTntCheckBox;
    ServiceFeatureNamedPipedCBox: TTntCheckBox;
    ServiceFeatureSupportForInnoDBCBox: TTntCheckBox;
    ServiceFeatureBDBCBox: TTntCheckBox;
    PathToBinaryBrowseBtn: TTntBitBtn;
    Panel3: TTntPanel;
    InstallServiceBtn: TTntButton;
    UninstallServiceBtn: TTntButton;
    ApplyChangesBtn: TTntButton;
    DiscardChangesBtn: TTntButton;
    LocalhostOnlyPnl: TTntPanel;
    LocalhostWarningLbl: TTntLabel;
    ServiceCheckerTmr: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure StopServiceBtnClick(Sender: TObject);
    procedure StartServiceBtnClick(Sender: TObject);

    procedure UpdateLogMemo(ServerStarted: Boolean);
    procedure DoThreadTerminatedStartServiceAction;
    procedure DoThreadTerminatedStopServiceAction;
    procedure ThreadTerminated(Sender: TObject);

    procedure EnableDisableServiceControls(Enable: Boolean);

    procedure DoPageContentChanged(Sender: TObject);

    procedure SetCurrentService(MySQLService: TMySQLService);

    procedure ValidateServiceFeatureCBoxes(Sender: TObject);
    procedure ValidateFileName(Sender: TObject);

    procedure ApplyChanges;
    procedure DiscardChanges;
    procedure ApplyChangesBtnClick(Sender: TObject);
    procedure DiscardChangesBtnClick(Sender: TObject);

    procedure CheckUniqueServiceName(ServiceName,
      DisplayName: WideString);
    procedure RefreshServiceList;
    procedure ServicesTreeViewChange(Sender: TObject; Node: TTreeNode);

    procedure InstallnewServiceMIClick(Sender: TObject);
    procedure UninstallselectedServiceMIClick(Sender: TObject);

    procedure RefreshServiceStatusMIClick(Sender: TObject);
    procedure RefreshServiceStatus;
    procedure PathToBinaryEdChange(Sender: TObject);
    procedure PathToBinaryBrowseBtnClick(Sender: TObject);

    procedure SetServerType(PathToBinary: WideString; ServerType: WideString);
    procedure PathToBinaryEdExit(Sender: TObject);
    procedure ConfigFilenameEdChange(Sender: TObject);
    procedure ServiceCheckerTmrTimer(Sender: TObject);
    procedure PageControlChange(Sender: TObject);
  private
    { Private declarations }
    StartStopServiceThread: TStartStopServiceThread;
    StartStopServiceThreadHandle: THandle;
    StartStopServiceRunning: Boolean;

    ServiceStatusRunningPNGImg,
      ServiceStatusStoppedPNGImg,
      ServiceConfigPNGImg: TPNGObject;
    ServiceStatusImgPicture: TPNGObject;

    CurrentService: TMySQLService;

    PageContentChanged: Boolean;

    LastServiceName: WideString;
  public
    { Public declarations }
  end;

const
  ServicesImageIndex = 25;

var
  AdminServiceControlForm: TAdminServiceControlForm;

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  ApplicationDataModule, ConnectToInstance, MySQLConnection, PNGTools;

{$R *.dfm}

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.FormCreate(Sender: TObject);

begin
  InitForm(self);

  DockedPanel := ServiceControlPnl;
  SubTreePanel := SubTreePnl;

  CurrentService := nil;
  //ServiceList:=TObjectList.Create;

  PageControl.ActivePageIndex := 0;

  StartStopServiceRunning := False;

  ConnectToInstanceAni.ResName := 'progress_indicator';

  if (not (MySQLConn.ConnectedToLocalhost)) and
    (not (ApplicationDM.Options.ShowOnlyServiceSections)) then
  begin
    ServiceConfigPNGImg := LoadPNGImageFromResource('service_config_gray', PageHeaderImg);
    DisableEnableControls(StartStopServiceSheet, False);
    DisableEnableControls(ServiceConfigSheet, False);
    ServiceStatusRunningPNGImg := LoadPNGImageFromResource('service_status_running_gray');
  end
  else
  begin
    ServiceConfigPNGImg := LoadPNGImageFromResource('service_config', PageHeaderImg);
    ServiceStatusRunningPNGImg := LoadPNGImageFromResource('service_status_running');
  end;

  ServiceStatusStoppedPNGImg := LoadPNGImageFromResource('service_status_stopped');
  PageHeader2Img.Picture.Assign(PageHeaderImg.Picture);

  ServiceStatusImg.Picture.Graphic := ServiceStatusRunningPNGImg;
  ServiceStatusImgPicture := ServiceStatusRunningPNGImg;

  if (MYXCommonOptions.XPStyleEnabled) then
    SettingsScrollBox.Color := clWhite;

  InstallServiceBtn.Visible := ApplicationDM.Options.ShowOnlyServiceSections;
  UninstallServiceBtn.Visible := ApplicationDM.Options.ShowOnlyServiceSections;

  LastServiceName := ApplicationDM.Options.MySQLInstance;

  //Get all MySQL Services
  if (MySQLConn.ConnectedToLocalhost) or
    (ApplicationDM.Options.ShowOnlyServiceSections) then
  begin
    RefreshServiceList;
  end
  else
    LocalhostOnlyPnl.Show;

  ServiceCheckerTmr.Enabled := True;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.FormDestroy(Sender: TObject);

begin
  if (StartStopServiceRunning) then
  begin
    TerminateThread(StartStopServiceThreadHandle, 0);
    StartStopServiceThread.Free;
    StartStopServiceThread := nil;
  end;

  ServiceConfigPNGImg.Free;
  ServiceStatusRunningPNGImg.Free;
  ServiceStatusStoppedPNGImg.Free;
  //ServiceList.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.FormClose(Sender: TObject; var Action: TCloseAction);

begin
  //
end;

//----------------------------------------------------------------------------------------------------------------------

// Handle Server start request

procedure TAdminServiceControlForm.StartServiceBtnClick(Sender: TObject);

begin
  if (CurrentService <> nil) then
  begin
    ServiceCheckerTmr.Enabled := False;

    if (CurrentService.ErrorLog <> '') then
      CurrentService.logfile_size := GetFileSize(CurrentService.ErrorLog);

    StartupLogMemo.Lines.Add('Trying to start the server ...'#13#10);
    StartServiceBtn.Enabled := False;
    Application.ProcessMessages;

    StartStopServiceThread := TStartStopServiceThread.Create(True, self, True,
      CurrentService.ServiceName);
    try
      StartStopServiceThreadHandle := StartStopServiceThread.Handle;
      StartStopServiceThread.Priority := tpNormal;

      StartStopServiceThread.OnTerminate := ThreadTerminated;
      StartStopServiceThread.FreeOnTerminate := True;

      StartStopServiceThread.Resume;
    except
      StartStopServiceThread.Free;

      raise;
    end;

    //Show animation
    ConnectToInstanceAni.Visible := True;
    ConnectToInstanceAni.Active := True;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.DoThreadTerminatedStartServiceAction;

begin
  Sleep(500);

  if (not (ApplicationDM.Options.ShowOnlyServiceSections)) then
    if (not (MySQLConn.Reconnect)) then
    begin
      StartupLogMemo.Lines.Add(
        _('Could not re-connect to the MySQL Server.'));
    end;

  UpdateLogMemo(True);

  ConnectToInstanceAni.Visible := False;
  ConnectToInstanceAni.Active := False;

  StartServiceBtn.Enabled := True;

  ServiceCheckerTmr.Enabled := True;
end;

//----------------------------------------------------------------------------------------------------------------------

// Handle Server stop request

procedure TAdminServiceControlForm.StopServiceBtnClick(Sender: TObject);

begin
  if (CurrentService <> nil) then
  begin
    ServiceCheckerTmr.Enabled := False;

    if (CurrentService.ErrorLog <> '') then
      CurrentService.logfile_size := GetFileSize(CurrentService.ErrorLog);

    StartupLogMemo.Lines.Add('Trying to stop the server ...'#13#10);
    StopServiceBtn.Enabled := False;
    Application.ProcessMessages;

    MySQLConn.Disconnect;

    Sleep(1000);

    StartStopServiceThread := TStartStopServiceThread.Create(True, self, False,
      CurrentService.ServiceName);
    try
      StartStopServiceThreadHandle := StartStopServiceThread.Handle;
      StartStopServiceThread.Priority := tpNormal;

      StartStopServiceThread.OnTerminate := ThreadTerminated;
      StartStopServiceThread.FreeOnTerminate := True;

      StartStopServiceThread.Resume;
    except
      StartStopServiceThread.Free;

      raise;
    end;

    //Show animation
    ConnectToInstanceAni.Visible := True;
    ConnectToInstanceAni.Active := True;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.DoThreadTerminatedStopServiceAction;

begin
  Sleep(1000);

  UpdateLogMemo(False);

  ConnectToInstanceAni.Visible := False;
  ConnectToInstanceAni.Active := False;

  StopServiceBtn.Enabled := True;

  ServiceCheckerTmr.Enabled := True;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.UpdateLogMemo(ServerStarted: Boolean);

var
  new_logfile_size: longword;
  s: TTntStringList;
  fs: TTntFileStream;
  ActionResult: Integer;

begin
  Sleep(500);

  if (CurrentService = nil) then
    Exit;

  new_logfile_size := CurrentService.logfile_size;
  if (CurrentService.ErrorLog <> '') then
    new_logfile_size := GetFileSize(CurrentService.ErrorLog);

  if (FileExists(CurrentService.ErrorLog)) then
  begin
    if (new_logfile_size <> CurrentService.logfile_size) then
    begin
      try
        fs := TTntFileStream.Create(CurrentService.ErrorLog, fmOpenRead or fmShareDenyNone);
        s := TTntStringList.Create;
        try
          fs.Seek(CurrentService.logfile_size, soFromBeginning);

          s.LoadFromStream(fs);

          StartupLogMemo.Lines.Add(s.Text);
        finally
          s.Free;
          fs.Free;
        end;
      except
      end;
    end;
  end;

  if (ServerStarted) then
  begin
    //Check if service really running
    ActionResult := ServiceStatus('', CurrentService.ServiceName);

    if (ActionResult = SERVICE_RUNNING) then
      StartupLogMemo.Lines.Add('Server was started.'#13#10)
    else
      if (ActionResult = -1) then
        StartupLogMemo.Lines.Add(Format('Server state could not be fetched (%s [%d]).'#13#10,
          [SysErrorMessage(GetLastError), GetLastError]))
      else
        StartupLogMemo.Lines.Add('Server could not be started.'#13#10);
  end
  else
  begin
    //Check if service really stopped
    ActionResult := ServiceStatus('', CurrentService.ServiceName);

    if (ActionResult = SERVICE_STOPPED) then
      StartupLogMemo.Lines.Add('Server was stopped.'#13#10)
    else
      if (ActionResult = -1) then
        StartupLogMemo.Lines.Add(Format('Server state could not be fetched (%s [%d]).'#13#10,
          [SysErrorMessage(GetLastError), GetLastError]))
      else
        StartupLogMemo.Lines.Add('Server could not be stopped.'#13#10);
  end;

  RefreshServiceStatus;
end;

//----------------------------------------------------------------------------------------------------------------------

constructor TStartStopServiceThread.Create(CreateSuspended: Boolean; AdminServiceControlForm: TAdminServiceControlForm;
  StartService: Boolean; ServiceName: WideString);

begin
  inherited Create(CreateSuspended);

  self.StartService := StartService;
  self.ServiceName := ServiceName;
  self.AdminServiceControlForm := AdminServiceControlForm;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TStartStopServiceThread.Destroy;

begin
  inherited Destroy;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TStartStopServiceThread.Execute;

begin
  if (StartService) then
    ServiceStart('', ServiceName)
  else
    ServiceStop('', ServiceName);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ThreadTerminated(Sender: TObject);

begin
  if (TStartStopServiceThread(Sender).StartService) then
    TStartStopServiceThread(Sender).Synchronize(DoThreadTerminatedStartServiceAction)
  else
    TStartStopServiceThread(Sender).Synchronize(DoThreadTerminatedStopServiceAction);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.RefreshServiceStatus;

var
  i: integer;
  OldConnectionStatus: Boolean;

begin
  if (not ((MySQLConn.ConnectedToLocalhost) or
    (ApplicationDM.Options.ShowOnlyServiceSections))) then
    Exit;

  OldConnectionStatus := MySQLConn.Connected;

  //If there are no Items in the ServicesTreeView, refresh it
  if (ServicesTreeView.Items.Count = 0) then
    RefreshServiceList;

  for i := 0 to ServicesTreeView.Items.Count - 1 do
  begin
    if (ServicesTreeView.Items[i].Data <> nil) then
    begin
      TMySQLService(ServicesTreeView.Items[i].Data).Status :=
        ServiceStatus('',
        TMySQLService(ServicesTreeView.Items[i].Data).ServiceName);

      ServicesTreeView.Items[i].ImageIndex := ServicesImageIndex +
        Ord(TMySQLService(ServicesTreeView.Items[i].Data).Status <>
        SERVICE_RUNNING);
      ServicesTreeView.Items[i].SelectedIndex :=
        ServicesTreeView.Items[i].ImageIndex;
    end;
  end;

  SetCurrentService(CurrentService);

  //If the service was stopped from outside, disconnect
  if (CurrentService <> nil) then
    if (MySQLConn.Connected) and (CurrentService.Status <> SERVICE_RUNNING) then
      MySQLConn.Disconnect
    else
      if (OldConnectionStatus = False) and (CurrentService.Status = SERVICE_RUNNING) then
        MySQLConn.Reconnect;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.EnableDisableServiceControls(Enable: Boolean);

begin
  StopServiceLbl.Enabled := Enable;
  StopServiceBtn.Enabled := Enable;
  StartServiceLbl.Enabled := Enable;
  StartServiceBtn.Enabled := Enable;
  LogMessageLbl.Enabled := Enable;
  StartupLogMemo.Enabled := Enable;

  DisableEnableControls(ServiceStatusGBox, Enable);
  DisableEnableControls(ServiceControlCBox, Enable);

  DisableEnableControls(ServiceSettingsGBox, Enable);
  DisableEnableControls(ServiceConfigFileGBox, Enable);
  DisableEnableControls(ServiceFeaturesGBox, Enable);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.DoPageContentChanged(Sender: TObject);

begin
  if (not (InitControls)) then
  begin
    PageContentChanged := True;

    ApplyChangesBtn.Enabled := True;
    DiscardChangesBtn.Enabled := True;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ValidateServiceFeatureCBoxes(Sender: TObject);

var
  VersionAtLeast41: Boolean;

begin
  if (ServiceFeatureDebugCBox.Checked) then
  begin
    ServiceFeatureSupportForInnoDBCBox.Checked := True;
    ServiceFeatureSupportForInnoDBCBox.Enabled := False;
    ServiceFeatureBDBCBox.Checked := True;
    ServiceFeatureBDBCBox.Enabled := False;
    ServiceFeatureNamedPipedCBox.Checked := False;
    ServiceFeatureNamedPipedCBox.Enabled := False;
  end
  else
  begin
    ServiceFeatureSupportForInnoDBCBox.Enabled := True;
    ServiceFeatureBDBCBox.Enabled := True;
    ServiceFeatureNamedPipedCBox.Enabled := True;
  end;

  if (ServiceFeatureNamedPipedCBox.Checked) then
  begin
    ServiceFeatureDebugCBox.Checked := False;
    ServiceFeatureDebugCBox.Enabled := False;
  end
  else
  begin
    ServiceFeatureDebugCBox.Enabled := True;
  end;

  if (CurrentService <> nil) then
  begin
    if (FileExists(CurrentService.PathToBinary + 'mysqld-debug.exe')) then
      VersionAtLeast41 := True
    else
      VersionAtLeast41 := False;

    if (ServiceFeatureDebugCBox.Checked) then
    begin
      //The service name of the optimized binary has been changed from
      //mysqld to mysqld-debug
      if (VersionAtLeast41) then
        CurrentService.ServerType := 'mysqld-debug'
      else
        CurrentService.ServerType := 'mysqld';
    end
    else
      if (ServiceFeatureNamedPipedCBox.Checked) then
      begin
        if (ServiceFeatureSupportForInnoDBCBox.Checked) or
          (ServiceFeatureBDBCBox.Checked) then
          CurrentService.ServerType := 'mysqld-max-nt'
        else
          CurrentService.ServerType := 'mysqld-nt';
      end
      else
        if (ServiceFeatureBDBCBox.Checked) then
          CurrentService.ServerType := 'mysqld-max'
        else
        begin
      //The service name of the optimized binary has been changed from
      //mysql-opt to mysqld
          if (VersionAtLeast41) then
            CurrentService.ServerType := 'mysqld'
          else
            CurrentService.ServerType := 'mysqld-opt';
        end;

    PathToBinaryEd.Text := CurrentService.PathToBinary +
      CurrentService.ServerType;

    ValidateFileName(PathToBinaryEd);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ValidateFileName(Sender: TObject);

begin
  if (Sender is TTntEdit) then
  begin
    if (not (FileExists(Trim(TTntEdit(Sender).Text)))) then
    begin
      if (not (FileExists(Trim(TTntEdit(Sender).Text) + '.exe'))) then
        TTntEdit(Sender).Font.Color := clRed
      else
        TTntEdit(Sender).Font.Color := clWindowText;
    end
    else
      TTntEdit(Sender).Font.Color := clWindowText;

    DoPageContentChanged(self);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ApplyChanges;

var
  Key: HKEY;
  s: WideString;
  SCMHandle,
  ServiceHandle: SC_Handle;

begin
  if (PageContentChanged) and (CurrentService <> nil) then
  begin
    //Apply changes to CurrentService
    CurrentService.DisplayName := ServiceDisplayNameEd.Text;
    CurrentService.Description := ServiceDescriptionEd.Text;
    CurrentService.StartType := Ord(not (ServiceAutoStartCBox.Checked)) + 2;
    CurrentService.ConfigFile := ConfigFilenameEd.Text;
    CurrentService.ConfigFileSection := ServiceSectionNameEd.Text;

    //For existing services update the registry
    if (CurrentService.ExistingService) then
    begin
      if (RegCreateKeyW(HKEY_LOCAL_MACHINE, PWideChar('SYSTEM\' +
        'CurrentControlSet\Services\' + CurrentService.ServiceName), Key) =
        ERROR_SUCCESS) then
      try
        RegSetValueEx(Key, 'Start', 0, REG_DWORD,
          @CurrentService.StartType,
          4);
        RegSetValueExW(Key, 'DisplayName', 0, REG_SZ,
          PWideChar(CurrentService.DisplayName),
          Length(CurrentService.DisplayName) * 2 + 2);
        RegSetValueExW(Key, 'Description', 0, REG_SZ,
          PWideChar(CurrentService.Description),
          Length(CurrentService.Description) * 2 + 2);

        CurrentService.PathToBinary :=
          IncludeTrailingPathDelimiter(CurrentService.PathToBinary);

        if (Pos(' ', CurrentService.PathToBinary) > 0) then
          s := '"' + CurrentService.PathToBinary +
            CurrentService.ServerType + '" ' +
            '--defaults-file="' + CurrentService.ConfigFile + '" ' +
            CurrentService.ServiceName
        else
          s := CurrentService.PathToBinary +
            CurrentService.ServerType + ' ' +
            '--defaults-file="' + CurrentService.ConfigFile + '" ' +
            CurrentService.ServiceName;
        RegSetValueExW(Key, 'ImagePath', 0, REG_EXPAND_SZ,
          PWideChar(s),
          Length(s) * 2 + 2);
      finally
        RegCloseKey(Key);
      end;
    end
    else
    //For new services use Service Manager
    begin
      CheckUniqueServiceName(CurrentService.ServiceName,
        CurrentService.DisplayName);

      SCMHandle := OpenSCManager(nil,
        nil, //SERVICES_ACTIVE_DATABASE
        SC_MANAGER_CREATE_SERVICE); //SC_MANAGER_ALL_ACCESS

      if (SCMHandle > 0) then
      begin
        try
          //Create new service
          ServiceHandle := CreateServiceW(SCMHandle,
            PWideChar(CurrentService.ServiceName), //lpServiceName,
            PWideChar(CurrentService.DisplayName), //lpDisplayName,
            SERVICE_ALL_ACCESS, //dwDesiredAccess,
            SERVICE_WIN32_OWN_PROCESS, //dwServiceType,
            Ord(not (ServiceAutoStartCBox.Checked)) + 2, //dwStartType,
            SERVICE_ERROR_NORMAL, //dwErrorControl,
            PWideChar('"' + CurrentService.PathToBinary + //lpBinaryPathName,
            CurrentService.ServerType + '" ' +
            '--defaults-file="' + CurrentService.ConfigFile + '" ' +
            CurrentService.ServiceName),
            nil, //lpLoadOrderGroup,
            nil, //lpdwTagId,
            nil, //lpDependencies,
            nil, //lpServiceStartName,
            nil); //lpPassword

          if (ServiceHandle = 0) then
            raise EMyxSystemError.Create('Cannot create new service.',
              GetLastError);

          CloseServiceHandle(ServiceHandle);

          {//Modify service
          LockHandle:=LockServiceDatabase(SCMHandle);

          if(LockHandle<>nil)then
          begin

            // open a handle to the specified service
            ServiceHandle:=OpenService(SCMHandle,
              PChar(CurrentService.ServiceName), SERVICE_ALL_ACCESS);

            if(ServiceHandle>0)then
            begin
              if(ChangeServiceConfig(ServiceHandle,
                SERVICE_WIN32_OWN_PROCESS, //dwServiceType,
                Ord(Not(ServiceAutoStartCBox.Checked))+2, //2 (auto) or 3 (manual)
                SERVICE_ERROR_NORMAL, //dwErrorControl,
                PChar(CurrentService.PathToBinary+ //lpBinaryPathName,
                  CurrentService.ServerType+' '+
                  '--defaults-file="'+CurrentService.ConfigFile+'" '+
                  CurrentService.ConfigFileSection),
                nil, //lpLoadOrderGroup,
                nil, //lpdwTagId,
                nil, //lpDependencies,
                nil, //lpServiceStartName,
                nil, //lpPassword,
                PChar(CurrentService.DisplayName)//lpDisplayName
                ))then
                ShowModalDialog('Error', 'Cannot update service information.'#13#10+
                  'Error Nr.:'+IntToStr(GetLastError), myx_mtError);

              CloseServiceHandle(ServiceHandle);
            end;
          end
          else
            ShowModalDialog('Error', 'Cannot lock service manager.'#13#10+
              'Error Nr.:'+IntToStr(GetLastError), myx_mtError);

          UnlockServiceDatabase(LockHandle);}

          //Set Description
          if (RegCreateKeyW(HKEY_LOCAL_MACHINE, PWideChar('SYSTEM\' +
            'CurrentControlSet\Services\' + CurrentService.ServiceName), Key) =
            ERROR_SUCCESS) then
          begin
            try
              RegSetValueExW(Key, 'Description', 0, REG_SZ,
                PWideChar(CurrentService.Description),
                Length(CurrentService.Description) + 2);
            finally
              RegCloseKey(Key);
            end;
          end;

          CurrentService.ExistingService := True;
        finally
          CloseServiceHandle(SCMHandle);
        end;
      end
      else
        raise EMyxSystemError.Create(
          'Could not connect to the Service Control Manager.',
          GetLastError);
    end;
  end;

  PageContentChanged := False;
  ApplyChangesBtn.Enabled := False;
  DiscardChangesBtn.Enabled := False;

  InstallServiceBtn.Enabled := True;
  UninstallServiceBtn.Enabled := (CurrentService <> nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.DiscardChanges;

begin
  if (PageContentChanged) and (CurrentService <> nil) then
  begin
    if (CurrentService.ExistingService) then
    begin
      SetCurrentService(CurrentService);
    end
    else
    begin
      RefreshServiceList;
    end;
  end;

  PageContentChanged := False;
  ApplyChangesBtn.Enabled := False;
  DiscardChangesBtn.Enabled := False;

  InstallServiceBtn.Enabled := True;
  UninstallServiceBtn.Enabled := (CurrentService <> nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.SetCurrentService(MySQLService: TMySQLService);

begin
  CurrentService := MySQLService;

  if (CurrentService <> nil) then
  begin
    LastServiceName := CurrentService.ServiceName;

    EnableDisableServiceControls(True);

    ServiceNameHeaderLbl.Caption := CurrentService.ServiceName;
    ServiceNameHeader2Lbl.Caption := CurrentService.ServiceName;

    if (CurrentService.Status = SERVICE_RUNNING) then
    begin
      ServerStatusLbl.Caption := CurrentService.ServiceName + ' ' +
        'Service is running.';
      if (ServiceStatusImgPicture <> ServiceStatusRunningPNGImg) then
      begin
        ServiceStatusImg.Picture.Graphic := ServiceStatusRunningPNGImg;
        ServiceStatusImgPicture := ServiceStatusRunningPNGImg;
      end;

      StartServiceLbl.Visible := False;
      StartServiceBtn.Visible := False;
      StopServiceLbl.Visible := True;
      StopServiceBtn.Visible := True;
    end
    else
    begin
      ServerStatusLbl.Caption := CurrentService.ServiceName + ' ' +
        'Service is stopped.';
      if (ServiceStatusImgPicture <> ServiceStatusStoppedPNGImg) then
      begin
        ServiceStatusImg.Picture.Graphic := ServiceStatusStoppedPNGImg;
        ServiceStatusImgPicture := ServiceStatusStoppedPNGImg;
      end;

      StartServiceLbl.Visible := True;
      StartServiceBtn.Visible := True;
      StopServiceLbl.Visible := False;
      StopServiceBtn.Visible := False;
    end;

    InitControls := True;
    try
      ServiceAutoStartCBox.Checked := (CurrentService.StartType = 2);

      ServiceDisplayNameEd.Text := CurrentService.DisplayName;
      ServiceDescriptionEd.Text := CurrentService.Description;

      ConfigFilenameEd.Text := CurrentService.ConfigFile;
      ServiceSectionNameEd.Text := CurrentService.ConfigFileSection;

      SetServerType(CurrentService.PathToBinary,
        CurrentService.ServerType);

      PathToBinaryEd.Text := CurrentService.PathToBinary +
        CurrentService.ServerType;

      UninstallServiceBtn.Enabled := True;
    finally
      InitControls := False;
    end;
  end
  else
  begin
    EnableDisableServiceControls(False);

    ServiceDisplayNameEd.Text := '';
    ServiceDescriptionEd.Text := '';
    ConfigFilenameEd.Text := '';
    ServiceSectionNameEd.Text := '';
    PathToBinaryEd.Text := '';

    ServerStatusLbl.Caption := 'No Service selected.';
    ServiceStatusImg.Picture.Graphic := nil;
    ServiceStatusImg.Invalidate;

    UninstallServiceBtn.Enabled := False;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.SetServerType(PathToBinary: WideString; ServerType: WideString);

var
  VersionAtLeast41: Boolean;

begin
  if (CurrentService = nil) then
    Exit;

  if (FileExists(CurrentService.PathToBinary + 'mysqld-debug.exe')) then
    VersionAtLeast41 := False
  else
    VersionAtLeast41 := False;

  InitControls := True;
  try
    ServiceFeatureSupportForInnoDBCBox.Checked := False;
    ServiceFeatureBDBCBox.Checked := False;
    ServiceFeatureNamedPipedCBox.Checked := False;
    ServiceFeatureDebugCBox.Checked := False;

    CurrentService.PathToBinary := PathToBinary;
    CurrentService.ServerType := ServerType;

    if ((CompareText(CurrentService.ServerType, 'mysqld') = 0) and
      (not (VersionAtLeast41))) or
      (CompareText(CurrentService.ServerType, 'mysqld-debug') = 0) then
    begin
      ServiceFeatureDebugCBox.Checked := True;
    end
    else
      if (CompareText(CurrentService.ServerType, 'mysqld-opt') = 0) or
      ((CompareText(CurrentService.ServerType, 'mysqld') = 0) and
        VersionAtLeast41) then
      begin
        ServiceFeatureSupportForInnoDBCBox.Checked := True;
      end
      else
        if (CompareText(CurrentService.ServerType, 'mysqld-nt') = 0) then
        begin
          ServiceFeatureNamedPipedCBox.Checked := True;
        end
        else
          if (CompareText(CurrentService.ServerType, 'mysqld-max') = 0) then
          begin
            ServiceFeatureSupportForInnoDBCBox.Checked := True;
            ServiceFeatureBDBCBox.Checked := True;
          end
          else
            if (CompareText(CurrentService.ServerType, 'mysqld-max-nt') = 0) then
            begin
              ServiceFeatureSupportForInnoDBCBox.Checked := True;
              ServiceFeatureBDBCBox.Checked := True;
              ServiceFeatureNamedPipedCBox.Checked := True;
            end;

    //ValidateServiceFeatureCBoxes(self);
  finally
    InitControls := False;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.CheckUniqueServiceName(ServiceName, DisplayName: WideString);

var
  ServiceReg: TTntRegistry;
  KeyInfo: TRegKeyInfo;
  SubKeyNames: TTntStringList;
  ServiceKeyNames: TTntStringList;
  i: integer;
  DisplayName2: WideString;

begin
  ServiceReg := TTntRegistry.Create;
  SubKeyNames := TTntStringList.Create;
  ServiceKeyNames := TTntStringList.Create;
  try
    ServiceReg.RootKey := HKEY_LOCAL_MACHINE;

    ServiceReg.OpenKey('SYSTEM\' +
      'CurrentControlSet\Services', False);

    ServiceReg.GetKeyInfo(KeyInfo);
    ServiceReg.GetKeyNames(SubKeyNames);

    ServiceReg.CloseKey;

    for i := 0 to SubKeyNames.Count - 1 do
    begin
      if (ServiceReg.OpenKey('SYSTEM\' +
        'CurrentControlSet\Services\' + SubKeyNames[i], False)) then
      begin
        ServiceReg.GetValueNames(ServiceKeyNames);

        if (ServiceKeyNames.IndexOf('DisplayName') <> -1) then
        begin
          try
            try
              DisplayName2 := ServiceReg.ReadString('DisplayName');
            except
              DisplayName2 := '';
            end;

            if (AnsiCompareText(SubKeyNames[i],
              ServiceName) = 0) or
              (AnsiCompareText(DisplayName2,
              ServiceName) = 0) then
              raise EMyxError.Create('The name ' + ServiceName + ' is already used. ' +
                'You have to specify a unique ServiceName.');

            if (AnsiCompareText(SubKeyNames[i],
              DisplayName) = 0) or
              (AnsiCompareText(DisplayName2,
              DisplayName) = 0) then
              raise EMyxError.Create('The name ' + DisplayName + ' is already used. ' +
                'You have to specify a unique DisplayName.');
          finally
            ServiceReg.CloseKey;
          end;
        end;
      end;
    end;

  finally
    SubKeyNames.Free;
    ServiceKeyNames.Free;
    ServiceReg.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.RefreshServiceList;

var
  i: integer;

begin
  if (not ((MySQLConn.ConnectedToLocalhost) or
    (ApplicationDM.Options.ShowOnlyServiceSections))) then
    Exit;

  //Clear tree
  for i := 0 to ServicesTreeView.Items.Count - 1 do
    if (ServicesTreeView.Items[i].Data <> nil) then
      TObject(ServicesTreeView.Items[i].Data).Free;
  ServicesTreeView.Items.Clear;

  //Get all MySQL Services
  ScanForServices(ServicesTreeView, ServicesImageIndex);

  //If there is at least one service
  if (ServicesTreeView.Items.Count > 0) then
  begin
    for i := 0 to ServicesTreeView.Items.Count - 1 do
      if (ServicesTreeView.Items[i].Data <> nil) then
        if (CompareText(LastServiceName,
          TMySQLService(ServicesTreeView.Items[i].Data).ServiceName) = 0) then
        begin
          ServicesTreeView.Selected := ServicesTreeView.Items[i];
          Exit;
        end;

    //ServicesTreeView.Selected:=ServicesTreeView.Items[0]
  end
  else
    SetCurrentService(nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ApplyChangesBtnClick(Sender: TObject);

begin
  ApplyChanges;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.DiscardChangesBtnClick(Sender: TObject);

begin
  DiscardChanges;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ServicesTreeViewChange(Sender: TObject; Node: TTreeNode);

begin
  if (not (InitControls)) then
  begin
    if (PageContentChanged) then
      DiscardChanges;

    CurrentService := nil;

    if (Node <> nil) then
      if (Node.Data <> nil) then
        CurrentService := TMySQLService(Node.Data);

    SetCurrentService(CurrentService);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.InstallnewServiceMIClick(Sender: TObject);

var
  servicename: WideString;
  MySQLService: TMySQLService;

begin
  if (PageContentChanged) then
    DiscardChanges;

  PageControl.ActivePage := ServiceConfigSheet;

  servicename := 'MySQL';

  if (ShowModalEditDialog('New Service',
    'Please enter a name for the new service. This service name will ' +
    'be used to start and stop the service. You can only use letters ' +
    'and numbers, not even spaces.'#13#10#13#10 +
    'Please note that you can specify a different display name for ' +
    'the service later, which can be more descriptive.',
    myx_mtEdit, 'OK'#13#10'Abort',
    True, 'Service name:', servicename) = 1) then
  begin
    //This procedure will raise an exception if the servicename is not unique
    CheckUniqueServiceName(servicename, servicename);

    MySQLService := TMySQLService.Create;
    MySQLService.ExistingService := False;
    //ServiceList.Add(MySQLService);
    //ServicesCBox.Items.AddObject(servicename, MySQLService);

    MySQLService.ServiceName := servicename;
    MySQLService.DisplayName := servicename;
    MySQLService.Description := 'MySQL Server';
    MySQLService.ConfigFileSection := 'mysqld';
    MySQLService.StartType := SERVICE_AUTO_START; //Automatic
    MySQLService.ServerType := 'mysqld-opt';
    MySQLService.Status := SERVICE_STOPPED;

    MySQLService.ConfigFile := ApplicationDM.Options.MySQLInstallPath +
      'my.cnf';
      //GetWindowsDir+'my.ini';
    MySQLService.PathToBinary := ApplicationDM.Options.MySQLInstallPath +
      'bin\';

    //Add new service to tree
    ServicesTreeView.Selected :=
      AddTreeViewChildNode(ServicesTreeView, nil,
      servicename, ServicesImageIndex, MySQLService);

    //SetCurrentService(MySQLService);

    InstallServiceBtn.Enabled := False;
    UninstallServiceBtn.Enabled := False;

    DoPageContentChanged(self);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.UninstallselectedServiceMIClick(Sender: TObject);

var
  SCMHandle,
  ServiceHandle: SC_Handle;

begin
  if (CurrentService <> nil) then
  begin
    //If the service was just created, simply discard it
    if (not (CurrentService.ExistingService)) then
    begin
      DiscardChanges;
    end
    else
    begin
      if (CurrentService.Status <> SERVICE_STOPPED) then
      begin
        if (ShowModalDialog('Stop Service',
          'Please stop the service ' +
          CurrentService.ServiceName + ' before uninstalling.'#13#10#13#10 +
          'Do you want to stop the service now?',
          myx_mtConfirmation, 'Yes'#13#10'Abort') = 1) then
        begin
          PageControl.ActivePage := StartStopServiceSheet;
          StopServiceBtnClick(self);
        end;

        Exit;
      end;

      if (ShowModalDialog('Delete Service',
        'Are you sure you want to remove the selected service ' +
        CurrentService.ServiceName + '?',
        myx_mtConfirmation, 'Yes'#13#10'No') = 1) then
      begin
        {SetStatusText('Stopping Service ...');

        if(Not(ServiceStop('', CurrentService.ServiceName)))then
          raise EMyxError.Create('The Service couldn''t be stopped. ');}

        SCMHandle := OpenSCManager(nil,
          nil, //SERVICES_ACTIVE_DATABASE
          SC_MANAGER_CREATE_SERVICE); //SC_MANAGER_ALL_ACCESS

        if (SCMHandle > 0) then
        begin
          SetStatusText('Deleting Service ...');

          try
            //Open service
            ServiceHandle := OpenServiceW(SCMHandle,
              PWideChar(CurrentService.ServiceName),
              SERVICE_ALL_ACCESS);
            try
              if (ServiceHandle = 0) then
                raise EMyxSystemError.Create('Cannot open service.',
                  GetLastError);

              if (not (DeleteService(ServiceHandle))) then
                raise EMyxSystemError.Create('Cannot delete service.', GetLastError);

            finally
              CloseServiceHandle(ServiceHandle);
            end;
          finally
            CloseServiceHandle(SCMHandle);
          end;

          CurrentService := nil;

          RefreshServiceList;

          PageContentChanged := False;
          ApplyChangesBtn.Enabled := False;
          DiscardChangesBtn.Enabled := False;

          SetStatusText('');
        end
        else
          raise EMyxSystemError.Create(
            'Could not connect to the Service Control Manager.', GetLastError);
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.RefreshServiceStatusMIClick(Sender: TObject);

begin
  RefreshServiceStatus;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.PathToBinaryEdChange(Sender: TObject);

begin
  ValidateFileName(Sender);

  if (CurrentService <> nil) then
    CurrentService.PathToBinary := ExtractFilePath(PathToBinaryEd.Text);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.PathToBinaryBrowseBtnClick(Sender: TObject);

var
  OpenDlg: TTntOpenDialog;

begin
  OpenDlg := TTntOpenDialog.Create(self);
  try
    OpenDlg.Filter := 'Executable files (*.exe)|*.exe';
    OpenDlg.InitialDir := ExtractFilePath(PathToBinaryEd.Text);

    if (OpenDlg.Execute) then
    begin
      if (Pos('mysqld', ExtractFileName(OpenDlg.FileName)) = 0) then
        if (ShowModalDialog('Correct file?',
          'The file you selected does not seem to be a MySQL server binary.'#13#10 +
          'Are you sure you want to use this file?',
          myx_mtConfirmation, 'Yes'#13#10'No') = 2) then
          Exit;

      //PathToBinaryEd.Text:=ChangeFileExt(OpenDlg.FileName, '');

      SetServerType(ExtractFilePath(OpenDlg.FileName),
        ChangeFileExt(ExtractFileName(OpenDlg.FileName), ''));

      ValidateFileName(PathToBinaryEd);
    end;
  finally
    OpenDlg.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.PathToBinaryEdExit(Sender: TObject);

begin
  SetServerType(ExtractFilePath(PathToBinaryEd.Text),
    ChangeFileExt(ExtractFileName(PathToBinaryEd.Text), ''));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ConfigFilenameEdChange(Sender: TObject);

begin
  self.ValidateFileName(sender);
  DoPageContentChanged(sender);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.ServiceCheckerTmrTimer(Sender: TObject);

begin
  RefreshServiceStatus;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServiceControlForm.PageControlChange(Sender: TObject);

begin
  // Only refresh Service status when the StartStopServiceSheet is active
  ServiceCheckerTmr.Enabled := (PageControl.ActivePage = StartStopServiceSheet);
end;

//----------------------------------------------------------------------------------------------------------------------

end.

