/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layout.cpp 
 * @brief Implementation of the layout computation classes.
 * 
 */

#include "myx_gc.h"
#include "myx_gc_layout.h"
#include "myx_gc_figure.h"

//----------------- CLayouter ------------------------------------------------------------------------------------------

CLayouter::CLayouter(CFigureElement* Element)
{
  FElement = Element;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether there is still a next value available.
 *
 * @return True, if there is a next value, otherwise false.
 */
bool CLayouter::HasNext(void)
{
  if (FElement == NULL)
    return false;
  else
    return FIterator != FElement->Children()->end();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the current child element and moves on to the next in the list.
 */
void CLayouter::RenderNext(void)
{
  if (FIterator != FElement->Children()->end())
  {
    CFigureElement* Child = *FIterator;

    TBoundingBox LocalBox;
    NextBoundingBox(&LocalBox);

    glPushMatrix();                                         
    glTranslatef(LocalBox.UpperLeftX, LocalBox.UpperLeftY, LocalBox.UpperLeftZ);
    
#if 0 // Render bounding box (for debugging)
    TBoundingBox* Temp = Child->BoundingBox();
    glPushAttrib(GL_CURRENT_BIT | GL_POLYGON_BIT);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    glColor3f(0, 0.5, 0);
    glRectf(Temp->UpperLeftX, Temp->UpperLeftY, Temp->LowerRightX, Temp->LowerRightY);
    glPopAttrib();
#endif 

    Child->Render();
    glPopMatrix();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Resets layout computation to start over from origin.
 */
void CLayouter::Reset(void)
{
  FIterator = FElement->Children()->begin();
  FX = 0;
  FY = 0;
};

//----------------- CColumnLayouter ------------------------------------------------------------------------------------

/**
  * Returns the transformed bounding box of the next element.
  *
  * @param BoundingBox The bounding box to fill with the new values.
  */
void CColumnLayouter::NextBoundingBox(TBoundingBox* BoundingBox)
{
  TBoundingBox* LocalBox = (*FIterator)->BoundingBox();

  BoundingBox->UpperLeftX = LocalBox->UpperLeftX;
  BoundingBox->UpperLeftY = FY + LocalBox->UpperLeftY;
  BoundingBox->UpperLeftZ = 0;

  FY += LocalBox->LowerRightY;

  BoundingBox->LowerRightX = LocalBox->LowerRightX;
  BoundingBox->LowerRightY = FY;
  BoundingBox->LowerRightZ = 0;

  if (FIterator != FElement->Children()->end())
    ++FIterator;
}

//----------------- CRowLayouter ---------------------------------------------------------------------------------------

/**
  * Returns the transformed bounding box of the next element.
  *
  * @param BoundingBox The bounding box to fill with the new values.
  */
void CRowLayouter::NextBoundingBox(TBoundingBox* BoundingBox)
{
  TBoundingBox* LocalBox = (*FIterator)->BoundingBox();

  BoundingBox->UpperLeftX = FX + LocalBox->UpperLeftX;
  BoundingBox->UpperLeftY = LocalBox->UpperLeftY;
  BoundingBox->UpperLeftZ = 0;

  FX += LocalBox->LowerRightX;

  BoundingBox->LowerRightX = FX;
  BoundingBox->LowerRightY = LocalBox->LowerRightY;
  BoundingBox->LowerRightZ = 0;

  if (FIterator != FElement->Children()->end())
    ++FIterator;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Static helper method to create a concrete layouter class for a figure element.
 *
 * @param Element The element for which an instance of a layouter is returned.
 * @return An instance of a new layouter class (or NULL if not supported).
 * @note The caller is responsible to free the returned instance.
 */
CLayouter* LayoutMapper::LayouterForElement(CFigureElement* Element)
{
  CLayouter* Result = NULL;

  switch (Element->Layout())
  {
    case GC_LAYOUT_ROW:
      {
        Result = new CRowLayouter(Element);
        break;
      };
    case GC_LAYOUT_COLUMN:
      {
        Result = new CColumnLayouter(Element);
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

