/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file  myx_grt_lua_shell.c
 * @brief Lua Shell Frontend
 * 
 * See also: <a href="../grt.html#LuaShell">GRT Lua Shell</a>
 */


#ifdef ENABLE_LUA_MODULES

#include "myx_grt_private.h"
//#include "myx_grt_lua_shell.h"
#include "myx_grt_lua_private.h"

#include <math.h>

#ifdef _WINDOWS
#include <Windows.h>
#endif


typedef struct {
  MYX_GRT *grt;
  char *cwd;
} LUA_CONTEXT;


typedef struct MYX_GRT_LUA_SHELL {
  lua_State *lua;
  char *current_line;
} MYX_GRT_LUA_SHELL;


static int myx_lua_add_grt_module_to_table(lua_State *L, MYX_GRT_MODULE *mod, int tbl);
static int myx_lua_call_grt_method(lua_State *L);
static int myx_lua_push_wrap_grt_value(lua_State *L, MYX_GRT_VALUE *value);
static int myx_lua_refresh(MYX_GRT *grt, lua_State *L);
static MYX_GRT_VALUE *pop_grt_value(lua_State *l);

static MYX_GRT *myx_lua_get_grt(lua_State *L);
static LUA_CONTEXT *myx_lua_get_ctx(lua_State *L);



static const char *mlua_popstring(lua_State *L)
{
  const char *s= luaL_checkstring(L, -1);
  lua_pop(L, 1);
  return s;
}


static void mlua_checkempty(lua_State *L)
{
  if (lua_gettop(L)>0)
    luaL_error(L, "too many arguments to function");
}


static char *get_abspath(LUA_CONTEXT *ctx, const char *dir)
{
  if (strcmp(dir, "..")==0)
  {
    char *tmp;
    
    if (strcmp(ctx->cwd, "/")==0 || strrchr(ctx->cwd, '/')==ctx->cwd)
      return g_strdup("/");
    else
    {
      tmp= g_strdup(ctx->cwd);
      *strrchr(tmp, '/')= 0;
      return tmp;
    }
  }
  //XXX TODO extend this

  if (*dir=='/')
    return g_strdup(dir);
  else if (strcmp(ctx->cwd, "/")==0)
    return g_strdup_printf("/%s", dir);
  else
    return g_strdup_printf("%s/%s", ctx->cwd, dir);
}


static int l_list_modules(lua_State *L)
{
  MYX_GRT_MODULE **modules;
  MYX_GRT *grt= myx_lua_get_grt(L);
  int count, i;

  count= myx_grt_modules_get_that_extend(grt, NULL, &modules);
  for (i= 0; i < count; i++)
  {
    myx_grt_printf(grt, "%s"NL, modules[i]->name);
  }
  g_free(modules);

  return 0;
}


static int l_show_module(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *name= mlua_popstring(L);
  MYX_GRT_MODULE *module= myx_grt_module_get(grt, name);
  unsigned int i;

  if (!module)
  {
    myx_grt_printf(grt, "Module '%s' has not been initialized."NL, name);
    return 0;
  }

  if (module->extends)
    myx_grt_printf(grt, "Module '%s' (extends '%s')"NL, name, module->extends);
  else
    myx_grt_printf(grt, "Module '%s'"NL, name);
  for (i= 0; i < module->functions_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;
    char *params= myx_grt_module_function_get_params(func);
    char *return_type= myx_grt_module_function_get_return_type(func);

    myx_grt_printf(grt, " %s:%s:%s"NL, module->functions[i].name, params, return_type);
  }

  return 0;
}


static int l_refresh(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);

  return myx_lua_refresh(grt, L);
}


static int l_save_value(lua_State *l)
{
  const char *fn= mlua_popstring(l);
  MYX_GRT_VALUE *value;
  MYX_GRT_ERROR err;

  if (lua_gettop(l) == 0)
    luaL_error(l, "Missing parameter");

  value= pop_grt_value(l);
  if (!value)
    luaL_error(l, "Invalid parameter");

  err= myx_grt_store_to_file(value, fn);
  
  myx_grt_value_release(value);
  
  if (err != MYX_GRT_NO_ERROR)
    luaL_error(l, "Could not save data to file %s", fn);

  return 0;
}


static int l_load_value(lua_State *l)
{
  const char *fn= mlua_popstring(l);
  MYX_GRT_VALUE *value;

  mlua_checkempty(l);
  
  value= myx_grt_retrieve_from_file(fn);
  if (!value)
    luaL_error(l, "Could not load data from file %s", fn);

  myx_lua_push_wrap_grt_value(l, value);

  return 1;
}

static int l_show_value(lua_State *l)
{
  MYX_GRT_VALUE *value;

  value= pop_grt_value(l);
  if (!value)
    luaL_error(l, "Missing argument");

  if (lua_gettop(l)==0)
    myx_grt_value_print(myx_lua_get_grt(l), value);
  else
    luaL_error(l, "Too many arguments");

  myx_grt_value_release(value);

  return 0;
}


static int l_cd(lua_State *l)
{
  LUA_CONTEXT *ctx= myx_lua_get_ctx(l);
  MYX_GRT_VALUE *value;
  const char *path= mlua_popstring(l);
  char *new_path;
  
  mlua_checkempty(l);
  
  if (!path)
    luaL_error(l, "Invalid path");

  if (!*path)
    return 0;

  new_path= get_abspath(ctx, path);

  value= myx_grt_dict_item_get_by_path(ctx->grt->root, new_path);

  if (!value)
  {
    g_free(new_path);
    luaL_error(l, "Invalid path");
  }
  g_free(ctx->cwd);
  ctx->cwd= new_path;

  return 0;
}


static int l_ls(lua_State *l)
{
  LUA_CONTEXT *ctx= myx_lua_get_ctx(l);
  const char *path;
  char *fpath;
  MYX_GRT_VALUE *value;
  unsigned int i;

  if (lua_gettop(l) > 0)
    path= mlua_popstring(l);
  else
    path= ctx->cwd;

  if (!ctx->grt->root)
    return 0;
  
  fpath= get_abspath(ctx, path);
  value= myx_grt_dict_item_get_by_path(ctx->grt->root, fpath);
  g_free(fpath);
  if (!value)
    luaL_error(l, "Invalid path");

  if (value->type == MYX_DICT_VALUE)
  {
    for (i= 0; i < value->value.d->items_num; i++)
    {
      MYX_GRT_DICT_ITEM *item= value->value.d->items+i;

      myx_grt_printf(ctx->grt, "%s"NL, item->key);
    }
  }
  else if (value->type == MYX_LIST_VALUE && value->value.l->content_type == MYX_DICT_VALUE)
  {
    unsigned int unnamed= 0;
    
    for (i= 0; i < value->value.l->items_num; i++)
    {
      MYX_GRT_VALUE *item= myx_grt_list_item_get(value, i);
      const char *name;
      
      name= myx_grt_dict_item_get_as_string(item, "name");
      if (name)
        myx_grt_printf(ctx->grt, "%s"NL, name);
      else
        unnamed++;
    }
    if (unnamed > 0)
      myx_grt_printf(ctx->grt, "Plus %i unnamed objects in the list."NL, unnamed);
  }
  else
    luaL_error(l, "Not in a listable object");

  return 0;
}


static int l_pwd(lua_State *l)
{
  LUA_CONTEXT *ctx= myx_lua_get_ctx(l);

  mlua_checkempty(l);
  
  lua_pushstring(l, ctx->cwd);

  return 1;
}


static int l_get_child(lua_State *l)
{
  LUA_CONTEXT *ctx= myx_lua_get_ctx(l);
  MYX_GRT_VALUE *value= NULL;
  MYX_GRT_VALUE *dict= NULL;
  const char *path;

  path= mlua_popstring(l);

  if (lua_gettop(l) > 0)
  {
    dict= pop_grt_value(l);
    if (!dict)
      luaL_error(l, "bad GRT value in argument");
  }

  if (dict)
  {
    if (*path != '/')
      luaL_error(l, "bad path for child object in dict. Must be an absolute path");
    value= myx_grt_dict_item_get_by_path(dict, path);
    myx_grt_value_release(dict);
  }
  if (!value)
    luaL_error(l, "object '%s' not found", path);

  myx_grt_value_retain(value);
  myx_lua_push_wrap_grt_value(l, value);

  return 1;
}

static int l_get_global(lua_State *l)
{
  LUA_CONTEXT *ctx= myx_lua_get_ctx(l);
  MYX_GRT_VALUE *value;
  MYX_GRT_VALUE *dict= NULL;
  const char *path;

  path= mlua_popstring(l);

  if (lua_gettop(l) > 0)
  {
    dict= pop_grt_value(l);
    if (!dict)
      luaL_error(l, "bad GRT value in argument");
  }

  if (dict)
  {
    if (*path != '/')
      luaL_error(l, "bad path for getobj in dict. Must be an absolute path");
    value= myx_grt_dict_item_get_by_path(dict, path);
    myx_grt_value_release(dict);
  }
  else
  {
    char *tmp= get_abspath(ctx, path);
    value= myx_grt_dict_item_get_by_path(ctx->grt->root, tmp);
    g_free(tmp);
  }
  if (!value)
    luaL_error(l, "object '%s' not found", path);

  myx_grt_value_retain(value);
  myx_lua_push_wrap_grt_value(l, value);

  return 1;
}


static int l_set_global(lua_State *l)
{
  LUA_CONTEXT *ctx= myx_lua_get_ctx(l);
  MYX_GRT_VALUE *value;
  MYX_GRT_VALUE *indict= NULL;
  const char *path;

  value= pop_grt_value(l);
  if (!value)
    luaL_error(l, "missing value parameter");

  path= mlua_popstring(l);

  if (lua_gettop(l) > 0)
    indict= pop_grt_value(l);

  mlua_checkempty(l);

  if (indict)
  {
    if (indict->type != MYX_DICT_VALUE)
    {
      myx_grt_value_release(indict);
      myx_grt_value_release(value);
      luaL_error(l, "invalid argument, expected a dict but got something else");
    }
    if (*path != '/')
    {
      myx_grt_value_release(indict);
      myx_grt_value_release(value);
      luaL_error(l, "bad path for setobj in dict. Must be an absolute path");
    }
    if (myx_grt_dict_item_set_by_path(indict, path, value) < 0)
    {
      myx_grt_value_release(indict);
      myx_grt_value_release(value);
      luaL_error(l, "invalid path '%s'", path);
    }
    myx_grt_value_release(indict);
  }
  else
  {
    if (strcmp2(path, "/") == 0)
    {
      MYX_GRT_ERROR error= myx_grt_set_root(ctx->grt, value);
      myx_grt_value_release(value);
    }
    else
    {
      char *fpath;

      fpath= get_abspath(ctx, path);
      if (myx_grt_dict_item_set_by_path(ctx->grt->root, fpath, value) < 0)
      {
        g_free(fpath);
        myx_grt_value_release(value);
        luaL_error(l, "invalid path '%s'", path);
      }
      g_free(fpath);
    }
  }
  myx_grt_value_release(value);

  return 1;
}


static int l_grt_value_to_lua(lua_State *l)
{
  if (luaL_checkudata(l, -1, "MYX_GRT_VALUE"))
  {
    MYX_GRT_VALUE *value= *(MYX_GRT_VALUE**)lua_touserdata(l, -1);
    lua_pop(l, 1);
    myx_lua_push_grt_value(l, value);
  }
  else
    luaL_error(l, "unwrap expects a MYX_GRT_VALUE as argument");

  return 1;
}

static int l_grt_value_new(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *gstruct_name;
  const char *name;
  const char *_id;
  const char *owner;
  MYX_GRT_VALUE *value;

  if (!lua_isstring(L, -1) || !lua_isstring(L, -2) || 
    !lua_isstring(L, -3) || !lua_isstring(L, -4))
  {
    luaL_error(L, "invalid parameters, use {struct_name, name, _id, owner}");
  }


  owner= mlua_popstring(L);
  _id= mlua_popstring(L);
  name= mlua_popstring(L);
  gstruct_name= mlua_popstring(L);


  value= myx_grt_dict_new_obj(grt, gstruct_name, name, _id, owner);

  //push the Grt value as Lua userdata
  myx_lua_push_wrap_grt_value(L, value);

  return 1;
}

static int l_grt_value_new_list(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *content_type_name= NULL;
  const char *content_struct_name= NULL;
  MYX_GRT_VALUE *value;
  MYX_GRT_VALUE_TYPE content_type;

  if (lua_isstring(L, -1) && lua_isstring(L, -2))
  {
    content_struct_name= mlua_popstring(L);
    content_type_name= mlua_popstring(L);
  }
  else if (lua_isstring(L, -1))
    content_type_name= mlua_popstring(L);

  if (content_type_name)
  {
    MYX_GRT_ERROR error;

    content_type= myx_get_value_type_from_string(content_type_name, &error);

    if (error != MYX_GRT_NO_ERROR)
      luaL_error(L, "invalid content_type. Use int, real, string, list or dict");
  }
  else
    content_type= MYX_ANY_VALUE;

  value= myx_grt_list_new(content_type, content_struct_name);

  //push the Grt value as Lua userdata
  myx_lua_push_wrap_grt_value(L, value);

  return 1;
}

static int l_grt_value_new_dict(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *content_type_name= NULL;
  const char *content_struct_name= NULL;
  MYX_GRT_VALUE *value;
  MYX_GRT_VALUE_TYPE content_type;

  if (lua_isstring(L, -1) && lua_isstring(L, -2))
  {
    content_struct_name= mlua_popstring(L);
    content_type_name= mlua_popstring(L);
  }
  else if (lua_isstring(L, -1))
    content_type_name= mlua_popstring(L);

  if (content_type_name)
  {
    MYX_GRT_ERROR error;

    content_type= myx_get_value_type_from_string(content_type_name, &error);

    if (error != MYX_GRT_NO_ERROR)
      luaL_error(L, "invalid content_type. Use int, real, string, list or dict");
  }
  else
    content_type= MYX_ANY_VALUE;

  value= myx_grt_dict_new_typed(content_type, content_struct_name);

  //push the Grt value as Lua userdata
  myx_lua_push_wrap_grt_value(L, value);

  return 1;
}


static int l_print(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  int n = lua_gettop(L);  /* number of arguments */
  int i;

  lua_getglobal(L, "tostring");
  for (i=1; i<=n; i++)
  {
    const char *s;
    lua_pushvalue(L, -1);  /* function to be called */
    lua_pushvalue(L, i);   /* value to print */
    lua_call(L, 1, 1);
    s = lua_tostring(L, -1);  /* get result */
    if (s == NULL)
      return luaL_error(L, "`tostring' must return a string to `print'");
    if (i>1) myx_grt_printf(grt, " ");
    myx_grt_printf(grt, "%s", s);

    lua_pop(L, 1);  /* pop result */
  }
  myx_grt_printf(grt,NL);

  return 0;
}


static int l_loadstructs(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *path;
  MYX_GRT_ERROR err;

  path= mlua_popstring(L);

  err= myx_grt_struct_load_and_register(grt, path);
  if (err == MYX_GRT_CANT_OPEN_FILE)
    luaL_error(L, "can't open the structs file");
  else if (err != MYX_GRT_NO_ERROR)
    luaL_error(L, "can't register structs");

  return 0;
}


static int l_validate(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *sname;
  MYX_GRT_VALUE *value;
  MYX_GRT_STRUCT *gstruct;
  int strict;
  int argc= lua_gettop(L);

  strict= (int)luaL_checknumber(L, -1);
  lua_pop(L, 1);

  if (argc > 2)
    sname= mlua_popstring(L);
  else
    sname= NULL;

  value= pop_grt_value(L);
  if (!value)
    luaL_error(L, "missing value to validate");

  if (argc == 2)
  {
    if (value->type == MYX_DICT_VALUE)
      sname= myx_grt_dict_struct_get_name(value);
  }

  if (sname)
  {
    gstruct= myx_grt_struct_get(grt, sname);
    if (!gstruct)
    {
      myx_grt_value_release(value);
      luaL_error(L, "bad struct %s", sname);
    }
  }
  else
  {
    if (argc == 2)
    {
      // was asked to validate against the assigned struct. if there's none, then its ok
      myx_grt_value_release(value);
      lua_pushnumber(L, 1);
      return 1;
    }
  }

  if (value->type != MYX_DICT_VALUE)
  {
    myx_grt_value_release(value);
    lua_pushnumber(L, 0);
    return 1;
  }

  if (myx_grt_dict_struct_validate(grt, value, sname, strict)>0)
    lua_pushnumber(L, 1);
  else
    lua_pushnumber(L, 0);

  myx_grt_value_release(value);

  return 1;
}


static int l_getstruct(lua_State *L)
{
  const char *sname;
  MYX_GRT_VALUE *value;

  value= pop_grt_value(L);
  if (!value)
    luaL_error(L, "missing value");
  
  if (value->type != MYX_DICT_VALUE && value->type != MYX_LIST_VALUE)
  {
    myx_grt_value_release(value);
    luaL_error(L, "value must be of type list or dict");
  }

  sname= myx_grt_dict_struct_get_name(value);

  myx_grt_value_release(value);
  
  if (!sname)
    lua_pushnil(L);
  else
    lua_pushstring(L, sname);
  
  return 1;
}



static int l_setstruct(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *sname;
  MYX_GRT_VALUE *value;

  sname= mlua_popstring(L);
  if (!myx_grt_struct_get(grt, sname))
    luaL_error(L, "unknown struct name '%s'", sname);

  value= pop_grt_value(L);
  if (!value)
    luaL_error(L, "missing value");

  if (value->type == MYX_DICT_VALUE)
    myx_grt_dict_struct_set_name(value, sname);
  else if (value->type == MYX_LIST_VALUE)
    myx_grt_list_content_set_struct_name(value, sname);
  else
  {
    myx_grt_value_release(value);
    luaL_error(L, "value must be of type list or dict");
  }
  
  myx_grt_value_release(value);
  
  return 0;
}

static int l_struct_exists(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);

  if (!lua_isstring(L, -1))
    luaL_error(L, "missing struct name");

  if (myx_grt_struct_get(grt, mlua_popstring(L)))
    lua_pushboolean(L, 1);
  else
    lua_pushboolean(L, 0);

  return 1;
}


static int l_setcontenttype(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *content_type_name;
  MYX_GRT_VALUE *value;
  MYX_GRT_VALUE_TYPE content_type;
  MYX_GRT_ERROR error;

  content_type_name= mlua_popstring(L);
  content_type= myx_get_value_type_from_string(content_type_name, &error);
  if (error != MYX_GRT_NO_ERROR)
    luaL_error(L, "the content type has to be \"\", \"string\", \"int\", \"real\", \"list\" or \"dict\".");

  value= pop_grt_value(L);
  if (!value)
    luaL_error(L, "missing value");

  if (value->type == MYX_DICT_VALUE)
    myx_grt_dict_content_set_type(value, content_type);
  else if (value->type == MYX_LIST_VALUE)
    myx_grt_list_content_set_type(value, content_type);
  else
  {
    myx_grt_value_release(value);
    luaL_error(L, "value must be of type list or dict");
  }
  
  myx_grt_value_release(value);
  
  return 0;
}

static int l_run(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  const char *path;
  MYX_GRT_ERROR err;

  path= mlua_popstring(L);

  err= myx_grt_lua_shell_run_file(grt, path, 1);
  if (err == MYX_GRT_CANT_OPEN_FILE)
    luaL_error(L, "can't open the script file");
  else if (err != MYX_GRT_NO_ERROR)
    luaL_error(L, "error while executing script");

  return 0;
}

static int l_sleep(lua_State *L)
{
  int ms= luaL_checkint(L, -1);
  lua_pop(L, 1);

#ifdef _WINDOWS
  Sleep(ms);
#else
  usleep(ms*1000);
#endif

  return 0;
}


/** 
 ****************************************************************************
 * @brief Initialize the Lua Shell frontend for the GRT.
 *
 *   Will initialize a Lua environment in the GRT, which may be used as
 * a Lua frontend/shell interface for the GRT.
 * 
 * @param grt  The GRT environment the shell belongs to.
 * 
 * @return GRT error code
 ****************************************************************************
 */
MYX_GRT_ERROR myx_grt_init_lua_shell(MYX_GRT *grt)
{
  lua_State *lua;
  LUA_CONTEXT *ctx;
  MYX_GRT_LUA_SHELL *lshell;
  MYX_GRT_MODULE_LOADER *loader;

  // The Lua module loader must be initialized before the shell
  loader= myx_grt_get_loader_of_type(grt, MYX_LUA_MODULE_TYPE);
  if (!loader)
    return MYX_GRT_INTERNAL_ERROR;

  lua= loader->priv->lua;

  lshell= g_new0(MYX_GRT_LUA_SHELL, 1);
  lshell->lua= lua;
  lshell->current_line= NULL;

  grt->lshell= lshell;

  // register a global __GRT variable 
  lua_pushstring(lua, "__GRT");
  ctx= lua_newuserdata(lua, sizeof(LUA_CONTEXT));
  ctx->grt= grt;
  ctx->cwd= g_strdup("/");
  luaL_newmetatable(lua, "MYX_GRT");
  lua_setmetatable(lua, -2);
  lua_settable(lua, LUA_GLOBALSINDEX);

  // override the built-in print with our own
  lua_register(lua, "print", l_print);

  // register our exported functions
  lua_register(lua, "refresh", l_refresh);

  // struct stuff
  lua_register(lua, "structLoad", l_loadstructs);
  lua_register(lua, "structVal", l_validate);
  lua_register(lua, "structSet", l_setstruct);
  lua_register(lua, "structGet", l_getstruct);
  lua_register(lua, "structExists", l_struct_exists);

  // contenttype
  lua_register(lua, "contentTypeSet", l_setcontenttype);

  // module stuff
  lua_register(lua, "lsm", l_list_modules);
  lua_register(lua, "showm", l_show_module);
  // value stuff
  lua_register(lua, "save", l_save_value);
  lua_register(lua, "load", l_load_value);
  lua_register(lua, "show", l_show_value);
  
  // grt
  lua_register(lua, "grt2Lua", l_grt_value_to_lua);
  lua_register(lua, "grtNewObj", l_grt_value_new);
  lua_register(lua, "grtNewList", l_grt_value_new_list);
  lua_register(lua, "grtNewDict", l_grt_value_new_dict);
  
  lua_register(lua, "cd", l_cd);
  lua_register(lua, "ls", l_ls);
  lua_register(lua, "pwd", l_pwd);
  lua_register(lua, "getGlobal", l_get_global);
  lua_register(lua, "setGlobal", l_set_global);
  lua_register(lua, "getChild", l_get_child);

  lua_register(lua, "run", l_run);

  lua_register(lua, "sleep", l_sleep);
 
  myx_lua_refresh(grt, lua);
  
  return MYX_GRT_NO_ERROR;
}


static MYX_GRT *myx_lua_get_grt(lua_State *L)
{
  LUA_CONTEXT *ctx;
  lua_getglobal(L, "__GRT");

  ctx= luaL_checkudata(L, -1, "MYX_GRT");
  if (ctx)
  {
    lua_pop(L, 1);
    return ctx->grt;
  }
  return NULL;
}


static int myx_lua_refresh(MYX_GRT *grt, lua_State *L)
{
  int i;
  int count;
  MYX_GRT_MODULE **modules;
//  int module_name_list_tbl;

//  lua_newtable(L);
//  module_name_list_tbl= lua_gettop(L);

  // Will create tables representing the available modules containing
  // the functions it implements. You can then call the functions with
  // module->function()

  count= myx_grt_modules_get_that_extend(grt, NULL, &modules);
  for (i= 0; i < count; i++)
  {
    lua_pushstring(L, modules[i]->name);
    lua_newtable(L);
    myx_lua_add_grt_module_to_table(L, modules[i], lua_gettop(L));
    lua_settable(L, LUA_GLOBALSINDEX);

//    lua_getglobal(L, modules[i]->name);
//    lua_rawseti(L, module_name_list_tbl, i+1);
  }

  return 0;
}


static LUA_CONTEXT *myx_lua_get_ctx(lua_State *L)
{
  LUA_CONTEXT *ctx;
  lua_getglobal(L, "__GRT");

  ctx= luaL_checkudata(L, -1, "MYX_GRT");
  if (ctx)
  {
    lua_pop(L, 1);
    return ctx;
  }
  return NULL;
}


static int gc_function(lua_State *L)
{
  MYX_GRT_VALUE *value= *(MYX_GRT_VALUE**)lua_touserdata(L, 1);

  //g_message("collecting a grt value of type %i", value->type);
  if (value)
    myx_grt_value_release(value);

  return 0;
}


static int myx_lua_push_wrap_grt_value(lua_State *L, MYX_GRT_VALUE *value)
{
  MYX_GRT_VALUE **ud_ptr= lua_newuserdata(L, sizeof(MYX_GRT_VALUE*));
  *ud_ptr= value;

  // metatable with GC function
  luaL_newmetatable(L, "MYX_GRT_VALUE");
  lua_pushstring(L, "__gc");
  lua_pushcfunction(L, gc_function);
  lua_settable(L, -3);

  lua_setmetatable(L, -2);

  return 1;
}


static int myx_lua_add_grt_module_to_table(lua_State *L, MYX_GRT_MODULE *mod, int tbl)
{
  unsigned int i;

  lua_pushstring(L, "_name_");
  lua_pushstring(L, mod->name);
  lua_settable(L, tbl);

  lua_pushstring(L, "_extends_");
  lua_pushstring(L, mod->extends);
  lua_settable(L, tbl);

  // add all functions to the table, so that they can be called as table.func()
  for (i= 0; i < mod->functions_num; i++)
  {
    lua_pushstring(L, mod->functions[i].name);
    lua_pushcfunction(L, myx_lua_call_grt_method);
    lua_settable(L, tbl);
  }

  return 1;
}


/** 
 ****************************************************************************
 * @brief push a GRT value into the Lua stack as a Lua value
 *
 *   Pushes a GRT value into stack of the given Lua environment, converting
 * the data to Lua types.
 * 
 * @param L  Lua state
 * @param value  GRT value
 *
 * @return number of objects in the stack
 ****************************************************************************
 */
int myx_lua_push_grt_value(lua_State *L, MYX_GRT_VALUE *value)
{
  unsigned int i;
  
  switch (value->type)
  {
  case MYX_INT_VALUE:
    lua_checkstack(L, lua_gettop(L)+1);
    lua_pushnumber(L, value->value.i);
    break;
  case MYX_STRING_VALUE:
    lua_checkstack(L, lua_gettop(L)+1);
    if (value->value.s)
      lua_pushstring(L, value->value.s);
    else
      lua_pushnil(L);
    break;
  case MYX_REAL_VALUE:
    lua_checkstack(L, lua_gettop(L)+1);
    lua_pushnumber(L, value->value.r);
    break;
  case MYX_LIST_VALUE:
    lua_checkstack(L, lua_gettop(L)+value->value.l->items_num+1);
    lua_newtable(L);
    for (i= 0; i < value->value.l->items_num; i++)
    {
      myx_lua_push_grt_value(L, value->value.l->items[i]);
      lua_rawseti(L, -2, i+1);
    }
    break;
  case MYX_DICT_VALUE:
    lua_checkstack(L, lua_gettop(L)+value->value.d->items_num*2+1);
    lua_newtable(L);
    for (i= 0; i < value->value.d->items_num; i++)
    {
      lua_pushstring(L, value->value.d->items[i].key);
      myx_lua_push_grt_value(L, value->value.d->items[i].value);
      lua_settable(L, -3);
    }
    break;
  }
  return 1;
}


static MYX_GRT_VALUE *pop_grt_value(lua_State *L)
{
 // MYX_GRT *grt= myx_lua_get_grt(L);
  MYX_GRT_VALUE *res= NULL;

  // if this is a GRT value
  if (lua_isuserdata(L, -1))
  {
    luaL_checkudata(L, -1, "MYX_GRT_VALUE");
    res= *(MYX_GRT_VALUE**)lua_touserdata(L, -1);
    lua_pop(L, 1);
    myx_grt_value_retain(res);
  }

  if (!res)
  {
    res= myx_lua_pop_grt_value(L);
  }
  return res;
}


MYX_GRT_VALUE *myx_lua_pop_grt_value(lua_State *L)
{
  MYX_GRT_VALUE *value= NULL;
  
  switch (lua_type(L, -1))
  {
  case LUA_TBOOLEAN:
    value= myx_grt_value_from_int(lua_toboolean(L, -1));
    lua_pop(L, 1);
    break;
  case LUA_TNUMBER:
    {
      lua_Number n= lua_tonumber(L, -1);
      if (n - floor(n) == 0)
        value= myx_grt_value_from_int((int)n);
      else
        value= myx_grt_value_from_real(n);
      lua_pop(L, 1);
    }
    break;
  case LUA_TSTRING:
    value= myx_grt_value_from_string(lua_tostring(L, -1));
    lua_pop(L, 1);
    break;
  case LUA_TTABLE:
    {
      MYX_GRT_VALUE_TYPE list_type;
      int tbl= lua_gettop(L);
      int can_be_list= 1;
      unsigned int i;
      unsigned int nexti;

      value= myx_grt_dict_new(NULL);

      nexti= 1;
      // first we create a dict from the table, checking if the indices
      // are numeric and in sequence
      lua_pushnil(L);
      while (lua_next(L, tbl) != 0)
      {
        MYX_GRT_VALUE *item_value;

        // handle value
        //item_value= myx_lua_pop_grt_value(L);
        item_value= pop_grt_value(L);

        // handle key
        lua_pushvalue(L, -1);
        myx_grt_dict_item_set_value(value, lua_tostring(L, -1), item_value);
        lua_pop(L, 1);

        if (lua_type(L, -1) != LUA_TNUMBER || lua_tonumber(L, -1) != nexti)
          can_be_list= 0;
        
        nexti++;
        
        // don't pop key, as it should be kept for lua_next() to fetch the
        // next item
      }
      lua_pop(L, 1);

      if (can_be_list && (value->value.d->items_num > 0))
      {
        // now we check if the values are all the same 
        // and remember the type of the list
        for (i= 0; i < value->value.d->items_num; i++)
        {
          if (i > 0 && list_type != value->value.d->items[i].value->type)
          {
            list_type= MYX_ANY_VALUE;
            break;
          }
          list_type= value->value.d->items[i].value->type;
        }
      }

      if (can_be_list && value->value.d->items_num > 0)
      {
        MYX_GRT_DICT *dict= value->value.d;

        value->type= MYX_LIST_VALUE;
        value->value.l= g_new0(MYX_GRT_LIST, 1);
        value->value.l->content_type= list_type;
        value->value.l->items_num= dict->items_num;
        value->value.l->items= g_new0(MYX_GRT_VALUE*, dict->items_num);

        for (i= 0; i < dict->items_num; i++)
        {
          int idx= atoi(dict->items[i].key)-1;

          value->value.l->items[idx]= dict->items[i].value;

          g_free(dict->items[i].key);
        }
        g_free(dict->items);
        g_free(dict);
      }
    }
    break;

  case LUA_TNIL:
    lua_pop(L, 1);
    break;
  case LUA_TFUNCTION:
  case LUA_TLIGHTUSERDATA:
  case LUA_TUSERDATA:
  case LUA_TTHREAD:
    g_warning("Invalid data (type=%s) in a Lua result value",
              lua_typename(L, lua_type(L, -1)));
    lua_pop(L, 1);
    break;
  }

  return value;
}


static int myx_lua_call_grt_method(lua_State *L)
{
  MYX_GRT *grt= myx_lua_get_grt(L);
  MYX_GRT_FUNCTION *func;
  lua_Debug dbg;
  const char *name;
  MYX_GRT_ERROR error;
  MYX_GRT_VALUE *argument;
  MYX_GRT_VALUE *retval;

  // pop argument from stack, if its there
  if (lua_gettop(L)==1)
    argument= NULL;
  else if (lua_gettop(L)==2)
    argument= pop_grt_value(L);
  else
    luaL_error(L, "too many arguments");

  // get info about the function that's being called
  lua_getstack(L, 0, &dbg);
  lua_getinfo(L, "n", &dbg);

  // get name of the module from the 1st table in stack (which is "self")
  lua_pushstring(L, "_name_");
  lua_gettable(L, -2);

  name= lua_tostring(L, -1);
  lua_pop(L, 1);

  func= myx_grt_function_get(grt, name, dbg.name, 1);
  g_return_val_if_fail(func != NULL, 0);

  retval= NULL;

  // if there was no argument given or the argument was a single value (and not a list as required)
  // create a new list and add the given argument to that list
  if ((argument == NULL) || (myx_grt_value_get_type(argument) != MYX_LIST_VALUE))
  {
    MYX_GRT_VALUE *argument_list= myx_grt_list_new(MYX_ANY_VALUE, NULL);

    if (argument)
      myx_grt_list_item_add(argument_list, argument);

    retval= myx_grt_function_call(grt, func, argument_list, &error);

    myx_grt_value_release(argument_list);
  }
  else
    retval= myx_grt_function_call(grt, func, argument, &error);

  if (retval)
    myx_lua_push_wrap_grt_value(L, retval);
  else
    lua_pushnil(L);

  if (argument)
    myx_grt_value_release(argument);

  if (error != MYX_GRT_NO_ERROR)
    return luaL_error(L, "error calling GRT function %s.%s (%i)",
                      name, dbg.name, error);
  else
    return 1;
}


static void myx_lua_report_error(MYX_GRT *grt, int status)
{
  if (status != 0)
  {
    const char *msg= lua_tostring(grt->lshell->lua, -1);
    myx_grt_printf(grt, "luart: error: %s"NL, msg);
    lua_pop(grt->lshell->lua, 1);
  }
}


static int myx_lua_execute(MYX_GRT *grt, const char *linebuf)
{
  int rc= 0;
  int status= 0;
  MYX_GRT_LUA_SHELL *lshell= grt->lshell;
  
  if (lshell->current_line)
    lshell->current_line= str_g_append(lshell->current_line, linebuf);
  else
    lshell->current_line= g_strdup(linebuf);

  status= luaL_loadbuffer(lshell->lua, lshell->current_line, strlen(lshell->current_line), "=stdin");
  if (status == LUA_ERRSYNTAX && strstr(lua_tostring(lshell->lua, -1), "near `<eof>'"))
  {
    // line is continued
    lua_pop(lshell->lua, 1);
    return 1;
  }

  if (status == 0)
    status= lua_pcall(lshell->lua, lua_gettop(lshell->lua)-1, 0, 0);
  else
    rc= -1;

  g_free(lshell->current_line);
  lshell->current_line= NULL;

  if (status!=0)
  {
    rc= -1;
    myx_lua_report_error(grt, status);
  }
  if (lua_gettop(lshell->lua) > 0)
  { // print stack contents
    lua_getglobal(lshell->lua, "print");
    lua_insert(lshell->lua, 1);
    if (lua_pcall(lshell->lua, lua_gettop(lshell->lua)-2, 0, 0) != 0)
    {
      myx_grt_printf(grt, "luart: error calling print (%s)"NL,
                     lua_tostring(lshell->lua, -1));
    }
  }

  return rc;
}


//============================================================================

/** 
 ****************************************************************************
 * @brief Prints welcome text for the GRT shell.
 *
 * Shows a welcome text in the GRT shell.
 * 
 * @param grt  The GRT environment the shell belongs to.
 ****************************************************************************
 */
void myx_grt_shell_print_welcome(MYX_GRT *grt)
{
  char *version= "2.0.4 alpha";

  myx_grt_printf(grt, "MySQL Generic Runtime Environment %s"NL, version);

  myx_grt_printf(grt, NL"Type 'help' or '\\h' for help. Type 'quit' to exit the shell."NL);
}


static void myx_grt_shell_show_command_help(MYX_GRT *grt, MYX_GRT_SHELL_COMMAND command)
{
  if((command == MYX_GRT_SHELL_COMMAND_LS) || (command == MYX_GRT_SHELL_COMMAND_ALL))
  {
    myx_grt_printf(grt, "ls [-d##]"NL
      "List all available objects in a tree view."NL
      "Options:      -d              Specifies the depth of the tree view"NL);
  }

  if((command == MYX_GRT_SHELL_COMMAND_CD) || (command == MYX_GRT_SHELL_COMMAND_ALL))
  {
    myx_grt_printf(grt, "cd (name | objid)"NL
      "Changes the current object."NL
      "Arguments:    name            Name of the object to change to."NL
      "              objid           ID of the object to change to."NLNL);
  }

  if((command == MYX_GRT_SHELL_COMMAND_RUN) || (command == MYX_GRT_SHELL_COMMAND_ALL))
  {
    myx_grt_printf(grt, "run filename"NL
      "Load and execute a lua script file."NL
      "Arguments:    filename        File that should be loaded and executed."NLNL);
  }
}

//use the following regex to detect all global paths in a function
//((.*?)[\(\,\=]\s*)((\/(\/(.*?))\/)|(\/(.*?)))(\s*[\)\,\n\r](.*))
// 1: cmd prefix
// 2: -
// 3: -
// 4: -
// 5: global path (if escaped //path/)
// 6: -
// 7: global path (if not an escaped /path)
// 8: -
// 9: cmd postfix
// 10: -
// tests:
//   mod:func( /test/path_entry )
//   mod:func("/test", /test/path_entry)
//   mod:func("/test", //test/path_entry/)
//   test= /test
//   test= //test path/
#define REGEX_GLOBAL_PATH_IN_FUNC "((.*?)[\\(\\,\\=]\\s*)((\\/(\\/(.*?))\\/)|(\\/(.*?)))(\\s*[\\)\\,\\n\\r](.*))"
//#define REGEX_GLOBAL_PATH_IN_FUNC "((.*?))((\\/(\\/(.*?))\\/)|(\\/(.*?)))"

static char *substitude_global_path(char *cmd)
{
  pcre *pcre_exp;
  const char *error_str;
  int erroffset;
  int o_vector[64];
  int rc;
  const char *prefix;
  const char *path;
  const char *postfix;
  char *value= NULL;

  if(cmd && cmd[0])
  {
    char *txt= g_strdup(cmd);

    pcre_exp= pcre_compile(REGEX_GLOBAL_PATH_IN_FUNC, PCRE_MULTILINE, &error_str, &erroffset, NULL);
    if (pcre_exp)
    {
      if ( (rc= pcre_exec(pcre_exp, NULL, txt, (unsigned int)strlen(txt), 0, 
                          0, o_vector, sizeof(o_vector)/sizeof(int)) ) > 0)
      {
        // get prefix, e.g. `mod:func(`
        pcre_get_substring(txt, o_vector, rc, 1, &prefix);
        value= g_strdup(prefix);
        pcre_free_substring((char*)prefix);

        // get escaped path, e.g. //test/path, replace it with getGlobal("/test/path")
        // if there is no escaped path, it must be a normal path,
        // e.g. /test/path, replace it with getGlobal("/test/path")      
        if (pcre_get_substring(txt, o_vector, rc, 5, &path) == 0)
          pcre_get_substring(txt, o_vector, rc, 7, &path);
        value= str_g_append_and_free(value, g_strdup_printf("getGlobal(\"%s\")", path));
        pcre_free_substring((char*)path);

        /*if (o_vector[10] > -1)
        {
          pcre_get_substring(txt, o_vector, rc, 5, &path);
          value= str_g_append_and_free(value, g_strdup_printf("getGlobal(\"%s\")", path));
          pcre_free_substring((char*)path);
        }
        // get path, e.g. /test/path, replace it with getGlobal("/test/path")        
        else
        {
          pcre_get_substring(txt, o_vector, rc, 7, &path);
          value= str_g_append_and_free(value, g_strdup_printf("getGlobal(\"%s\")", path));
          pcre_free_substring((char*)path);
        }*/

        // get postfix, e.g. `)`
        pcre_get_substring(txt, o_vector, rc, 9, &postfix);
        value= str_g_append(value, postfix);
        pcre_free_substring((char*)postfix);

        g_free(cmd);

        //call recursive to replace all global paths
        cmd= substitude_global_path(value);
      }

      pcre_free(pcre_exp);
    }

    g_free(txt);
  }

  return cmd;
}

/** 
 ****************************************************************************
 * @brief Shows help text pertaining to the Lua Shell.
 *
 * Displays help text for the given command or a general help text, if
 * NULL is passed.
 * 
 * @param grt  The GRT environment the shell belongs to.
 * @param command  The command to give help about or NULL, for general help.
 ****************************************************************************
 */
void myx_grt_shell_show_help(MYX_GRT *grt, const char *command)
{
#define BLOCK_SEPERATOR NL"--------------------------------------------------------------------------------"NL

#define BLOCK_HEADER(varname, text, linechar, width) {\
    const char *txt= text;\
    unsigned int i;\
    unsigned int len= (unsigned int)strlen(txt);\
    unsigned int p= div(width, 2).quot - div(len, 2).quot;\
    varname= g_malloc(sizeof(char)*81);\
    for (i= 0; i < width; i++)\
      varname[i]= linechar;\
    varname[width]= 0;\
    for (i= 0; i < len; i++)\
      varname[p + i]= txt[i];\
  }

  if(!command)
  {
    char *file_funcs;
    char *globals_vars;
    char *grt_st;
    //char *grt_obj;

    BLOCK_HEADER(file_funcs, "FILE FUNCTIONS", '-', 80)
    BLOCK_HEADER(globals_vars, "GLOBALS / VARIABLES", '-', 80)
    BLOCK_HEADER(grt_st, "GRT STRUCTS", '-', 80)
    

    myx_grt_printf(grt, "List of shell commands:"NL
      "Type 'help command' to get detailed information about the command. "NLNL
      "help    (\\h)    Display this help."NL
      "?       (\\?)    Synonym for 'help'."NL
      "quit    (\\q)    Exit the shell."NL
      "exit    (\\e)    Synonym for 'quit'."NL
      NL
      "%s"NL // FILE FUNCTIONS
      NL
      "run filename    (\\r)           Load and execute a lua script file."NL
      "obj= load(\"filename\")          Loads an object"NL
      "save(obj, \"filename\")          Saves an object"NL
      NL
      "%s"NL // GLOBALS / VARIABLES
      NL
      "ls [-m [module]]               List all child objects of the current global."NL
      "cd path                        Changes the current global."NL
      "show [/path | obj]             Display the global or variable"NLNL
      "/path = obj                    Assigns the object to the global"NL
      "//path with space/ = obj"NL
      "setGlobal(\"/path\", obj)"NLNL
      "obj = /path                    Assigns the global to the object"NL
      "obj = //path with space/"NL
      "getGlobal(\"/path\")             Assigns the global to the object"NLNL
      "getChild(obj, \"/path\")         Returns the object's child defined by the path"NL
      NL 
      "lua_value= grt2Lua(grt_value)  Converts the GRT object to a lua value."NL
      "grt_value= grtNewObj(\"struct\", \"name\", \"_id\", \"owner\")"NL
      "                               Create a new Grt object with the given values"NL
      "grt_value= grtNewList(\"content-type\" [, \"content-struct\"])"NL
      "                               Create a new Grt list with the given content type"NL
      "grt_value= grtNewDict(\"content-type\" [, \"content-struct\"])"NL
      "                               Create a new Grt dict with the given content type"NL
      NL
      "%s"NL // GRT STRUCTS
      NL
      "structLoad(\"filename\")         Register structs from file"NL
      "structSet(obj, \"struct\")       Assigns the struct to the object"NL
      "structGet(obj)                 Returns the struct of an object"NL
      "structVal(obj, \"struct\", 0)    Returns ~= 0 if the obj corresponds to the struct"NL
      "structVal(obj, 0)              Returns ~= 0 if the objects corresponds to its "NL
      "                               assigned struct"NL
      "structExists(\"struct\")         Returns 1 if the struct exists"NL
      "contentTypeSet(obj, \"type\")    Sets the content type of the given list or dict"NL,
      file_funcs,
      globals_vars,
      grt_st);

     g_free(file_funcs);
     g_free(globals_vars);
     g_free(grt_st);
  }
  else
  {
    if(strcmp2(command, "all")==0)
      myx_grt_shell_show_command_help(grt, MYX_GRT_SHELL_COMMAND_ALL);
    else if(strcmp2(command, "ls")==0)
      myx_grt_shell_show_command_help(grt, MYX_GRT_SHELL_COMMAND_LS);
    else if(strcmp2(command, "cd")==0)
      myx_grt_shell_show_command_help(grt, MYX_GRT_SHELL_COMMAND_CD);
    else if(strcmp2(command, "run")==0)
      myx_grt_shell_show_command_help(grt, MYX_GRT_SHELL_COMMAND_RUN);
  }

#undef BLOCK_SEPERATOR
}

/** 
 ****************************************************************************
 * @brief Executes a Lua command in the Lua shell
 *
 * This will execute the given Lua command in the Lua Shell from the 
 * GRT environment. Some pre-processing will be performed in the command
 * to handle special GRT shell commands.
 * 
 * @param grt The GRT environment the shell belongs to.
 * @param linebuf Line containing the command to be executed.
 *
 * @return 
 ****************************************************************************
 */
MYX_GRT_SHELL_COMMAND myx_grt_lua_shell_execute(MYX_GRT *grt, const char *linebuf)
{
  char *cmd;
  unsigned int cmd_len;
  char *cmd_param;
  MYX_GRT_SHELL_COMMAND res= MYX_GRT_SHELL_COMMAND_UNKNOWN;
  char *preprocessed_cmd= NULL;
  char *line= g_strdup(linebuf);

  //substitude global path with getGlobal(globale path)
  line= substitude_global_path(line);

  cmd= g_strdup(line);
  cmd= str_trim(cmd);
  cmd_len= (unsigned int)strlen(cmd);
  cmd_param= NULL;

  //Help command
  if((cmd_param= get_value_from_text_ex(cmd, cmd_len, "^(help|help\\;|\\\\h|\\?|\\-\\?)\\s*(\\w*)", 2)))
  {
    if(cmd_param[0])
      myx_grt_shell_show_help(grt, cmd_param);
    else
      myx_grt_shell_show_help(grt, NULL);

    res= MYX_GRT_SHELL_COMMAND_HELP;
  } 
  //Quit command
  else if( (strcmp2(cmd, "quit")==0) || (strcmp2(cmd, "quit;")==0) || (strcmp2(cmd, "exit")==0) ||
    (strcmp2(cmd, "exit;")==0) || (strcmp2(cmd, "\\q")==0) || (strcmp2(cmd, "q")==0) || (strcmp2(cmd, "\\e")==0))
  {
    myx_grt_printf(grt, "Exiting...");

    res= MYX_GRT_SHELL_COMMAND_EXIT;
  }
  else if( (strcmp2(cmd, "run")==0) || (str_beginswith(cmd, "\\r")) || (str_beginswith(cmd, "run ")))
  {
    char *file_name= get_value_from_text_ex(cmd, (int)strlen(cmd), "(run|\\\\r)\\s+([\\w\\.]+)", 2);

    if((file_name) && (file_name[0]))
    {
      preprocessed_cmd= g_strdup_printf("run(\"%s\")"NL, file_name);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }
    else
    {
      myx_grt_shell_show_command_help(grt, MYX_GRT_SHELL_COMMAND_RUN);
      res= MYX_GRT_SHELL_COMMAND_HELP;
    }

    if(file_name)
      g_free(file_name);
  }
  // Automatically convert cd.. to cd(".." and
  //   cd objectname to cd("objectname")
  else if ((strcmp2(cmd, "cd")==0) || (strcmp2(cmd, "cd..")==0) || (str_beginswith(cmd, "cd ")))
  {
    char *path= get_value_from_text_ex(cmd, (int)strlen(cmd), "cd\\s*(.+)", 1);

    if((path) && (path[0]))
    {
      preprocessed_cmd= g_strdup_printf("cd(\"%s\")"NL, path);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }
    else
    {
      preprocessed_cmd= g_strdup_printf("print(pwd())"NL);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }

    if(path)
      g_free(path);
  }
  // Automatically convert ls to ls()
  else if (strcmp2(cmd, "ls")==0)
  {
    preprocessed_cmd= g_strdup("ls()"NL);
    res= MYX_GRT_SHELL_COMMAND_LUA;
  }
  // Automatically convert ls -m to lsm()
  else if (strcmp2(cmd, "ls -m")==0)
  {
    preprocessed_cmd= g_strdup("lsm()"NL);
    res= MYX_GRT_SHELL_COMMAND_LUA;
  }
  // Automatically convert ls -m module to showm("module")
  else if (str_beginswith(cmd, "ls -m "))
  {
    char *path= get_value_from_text_ex(cmd, (int)strlen(cmd), "ls\\s+\\-m\\s+(.+)", 1);

    if((path) && (path[0]))
    {
      preprocessed_cmd= g_strdup_printf("showm(\"%s\")"NL, path);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }
  }
  // Automatically convert show to show(getobj(pwd()))
  else if (strcmp2(cmd, "show") == 0)
  {
    preprocessed_cmd= g_strdup_printf("show(getobj(pwd()))"NL);
    res= MYX_GRT_SHELL_COMMAND_LUA;
  }
  // Automatically convert show objectname to show(getObj("objectname"))
  else if (str_beginswith(cmd, "show "))
  {
    char *path= get_value_from_text_ex(cmd, (int)strlen(cmd), "show\\s+(.+)", 1);

    if ((path) && (path[0]))
    {
      preprocessed_cmd= g_strdup_printf("show(getGlobal(\"%s\"))"NL, path);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }

    g_free(path);
  }
  // Convert `/test/path_entry = res` to `setGlobal("/test/path_entry", res)`
  else if (str_beginswith(cmd, "/"))
  {
    char *path= get_value_from_text_ex(cmd, cmd_len, "(/(.*?))\\s*=\\s*(.*)", 1);
    char *val= get_value_from_text_ex(cmd, cmd_len, "(/(.*?))\\s*=\\s*(.*)", 3);

    if (path && path[0] && val && val[0])
    {
      preprocessed_cmd= g_strdup_printf("setGlobal(\"%s\", %s)"NL, path, val);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }

    g_free(path);
    g_free(val);
  }
  // Convert `//test/path entry/ = res` to `setGlobal("/test/path entry", res)`
  else if (str_beginswith(cmd, "//"))
  {
    char *path= get_value_from_text_ex(cmd, cmd_len, "(\\/(.*?)\\/)\\s*=\\s*(.*)", 2);
    char *val= get_value_from_text_ex(cmd, cmd_len, "(\\/(.*?)\\/)\\s*=\\s*(.*)", 3);

    if (path && path[0] && val && val[0])
    {
      preprocessed_cmd= g_strdup_printf("setGlobal(\"%s\", %s)"NL, path, val);
      res= MYX_GRT_SHELL_COMMAND_LUA;
    }

    g_free(path);
    g_free(val);
  }
  g_free(cmd);
  if(cmd_param)
    g_free(cmd_param);

  //If the command is still unknown, it needs to be a Lua command
  if((res == MYX_GRT_SHELL_COMMAND_UNKNOWN) || (res == MYX_GRT_SHELL_COMMAND_LUA))
  {
    if(!preprocessed_cmd)
      res= myx_lua_execute(grt, line);
    else
    {
      res= myx_lua_execute(grt, preprocessed_cmd);
      g_free(preprocessed_cmd);
    }

    g_free(line);
    
    if (res > 0)
      return MYX_GRT_SHELL_COMMAND_UNKNOWN;
    else if (res < 0)
      return MYX_GRT_SHELL_COMMAND_ERROR;
    else 
      return MYX_GRT_SHELL_COMMAND_LUA;
  }

  g_free(line);

  return res;
}

/** 
 ****************************************************************************
 * @brief Returns a prompt suitable for displaying in the shell.
 *
 * This will return a string suitable for displaying in the Lua shell,
 * displaying special information such as current object path, line 
 * edit status etc.
 * 
 * @param grt The GRT environment the shell belongs to.
 *
 * @return A string containing the prompt text. Should be freed with g_free()
 ****************************************************************************
 */
char *myx_grt_lua_shell_get_prompt(MYX_GRT *grt)
{
  LUA_CONTEXT *ctx;
  
  ctx= myx_lua_get_ctx(grt->lshell->lua);

  if (grt->lshell->current_line)
    return g_strdup_printf("%s>> ", ctx ? ctx->cwd : "");
  else
    return g_strdup_printf("%s > ", ctx ? ctx->cwd : "");
}


/** 
 ****************************************************************************
 * @brief Returns the Lua object used in the Lua Shell of the GRT.
 *
 * @param grt  The GRT environment the shell belongs to.
 *
 * @return The lua_State object.
 ****************************************************************************
 */
lua_State *myx_grt_lua_shell_get_lua(MYX_GRT *grt)
{
  g_return_val_if_fail(grt!=NULL, NULL);
  g_return_val_if_fail(grt->lshell!=NULL, NULL);
  
  return grt->lshell->lua;
}


/** 
 ****************************************************************************
 * @brief Loads and executes a Lua script file.
 *
 *   This will load the named file and attempt to execute it in the given
 * GRT environment. The script will have access to all modules that are
 * registered inside that environment.
 * 
 * @param grt The GRT environment the shell belongs to.
 * @param file_name name of the script file.
 *
 * @return 0 if the file is executed successfully
 * @return < 0 if there's an error executing the file
 ****************************************************************************
 */
int myx_grt_lua_shell_run_file(MYX_GRT *grt, const char *file_name, int interactive)
{
  int status= luaL_loadfile(grt->lshell->lua, file_name);
  int rc;

  if (interactive)
    myx_grt_printf(grt, "Opening script file %s ..."NL, file_name);

  if (status)
  {
    myx_grt_printf(grt, "Error in file: %s"NL, lua_tostring(grt->lshell->lua, -1));
    return -1;
  }

  if (interactive)
    myx_grt_printf(grt, "Executing script file %s ..."NLNL, file_name);

  status= lua_pcall(grt->lshell->lua, 0, LUA_MULTRET, 0);
  if (status)
  {
    myx_grt_printf(grt, "error executing script: %s"NL, lua_tostring(grt->lshell->lua, -1));
    rc= -2;
  }
  else
    rc= 0;

  if ((rc == 0) && (interactive))
    myx_grt_printf(grt, "\nExecution finished."NL);

  return rc;
}

/** 
 ****************************************************************************
 * @brief Gets the value of a global variable in the lua shell 
 *
 *   This will return the value of a global lua variable as a MYX_GRT_VALUE
 * 
 * @param grt The GRT environment the shell belongs to.
 * @param var_name name of the lua variable
 *
 * @return if the var is found, the MYX_GRT_VALUE is returned otherwise NULL
 ****************************************************************************
 */
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_lua_shell_get_global_var(MYX_GRT *grt, const char *var_name)
{
  MYX_GRT_VALUE *value= NULL;

  lua_getglobal(grt->lshell->lua, var_name);
  if (!lua_isnil(grt->lshell->lua, -1))
     value= pop_grt_value(grt->lshell->lua);

  return value;
}

/** 
 ****************************************************************************
 * @brief Sets the value of a global variable in the lua shell 
 *
 *   This will set the value of a global lua variable to a MYX_GRT_VALUE
 * 
 * @param grt The GRT environment the shell belongs to.
 * @param var_name name of the lua variable
 * @param value the MYX_GRT_VALUE the variable should be set to
 *
 ****************************************************************************
 */
MYX_PUBLIC_FUNC void myx_grt_lua_shell_set_global_var(MYX_GRT *grt, const char *var_name, MYX_GRT_VALUE *value)
{
  myx_lua_push_wrap_grt_value(grt->lshell->lua, value);
  lua_setglobal(grt->lshell->lua, var_name);
}


MYX_PUBLIC_FUNC int myx_grt_lua_shell_global_var_is_grt_value(MYX_GRT *grt, const char *var_name)
{
  int rc;

  lua_getglobal(grt->lshell->lua, var_name);

  if (luaL_checkudata(grt->lshell->lua, -1, "MYX_GRT_VALUE"))
    rc= 1;
  else
    rc= 0;
  
  lua_pop(grt->lshell->lua, 1);
	
  return rc;
}

#endif
