/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include "myx_grt_gc_bridge.h"

//----------------- General initialization and finalization functions --------------------------------------------------

/**
 * Registers a new GRT-to-GC bridge (as new built-in module) with the given GRT.
 *
 * @param grt The runtime structure with which the bridge should be registered.
 * @return Either MYX_GRT_NO_ERROR if all went fine, otherwise an error code.
 */
MYX_GRT_ERROR myx_register_grt_gc_module(MYX_GRT* Runtime)
{
  CGrtToGcBridge* Bridge = new CGrtToGcBridge(Runtime);

  // Register the new module and keep a reference to it.
  MYX_GRT_MODULE* Module = myx_grt_module_register_builtin(Runtime, &grt_module_grt_gc, Bridge);
  Bridge->Module(Module);

  if (Module != NULL)
    return MYX_GRT_NO_ERROR;
  else                                 
    return MYX_GRT_MODULE_INIT_ERROR;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the GRT-to-GC bridge from the given grt.
 *
 * @param Runtime The runtime structure from which the bridge should be unregistered.
 */
void myx_unregister_grt_gc_module(MYX_GRT* Runtime)
{
  // Look if there is a module with that name.
  MYX_GRT_MODULE* Module = myx_grt_find_module(Runtime, grt_module_grt_gc.name);

  if (Module != NULL)
  {
    CGrtToGcBridge* Bridge = (CGrtToGcBridge*) Module->priv;
    myx_grt_module_unregister_builtin(Runtime, Module);
    delete Bridge;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Initializes a new mapping between the given canvas and path.
 *
 * @param Runtime The runtime in which the resulting proxy is to be insterted.
 * @param Canvas The canvas that is used for the mapping.
 * @param Path The (absolute) target path for the resulting proxy object.
 */
void myx_grt_gc_bridge_initialize(MYX_GRT *Runtime, CGenericCanvas *Canvas, const char *Path)
{
  // Look if there is a module with that name.
  MYX_GRT_MODULE* Module = myx_grt_find_module(Runtime, grt_module_grt_gc.name);

  if (Module != NULL)
  {
    CGrtToGcBridge* Bridge = (CGrtToGcBridge*) Module->priv;
    Bridge->AddCanvas(Path, Canvas);
    myx_grt_dict_new_proxy(Runtime, Bridge->Module(), Path, "canvas.Canvas");
  };
}

//----------------------------------------------------------------------------------------------------------------------

void myx_grt_gc_bridge_free(MYX_GRT *Runtime, CGenericCanvas *Canvas, const char *Path)
{
  // Look if there is a module with that name.
  MYX_GRT_MODULE* Module = myx_grt_find_module(Runtime, grt_module_grt_gc.name);

  if (Module != NULL)
  {
    CGrtToGcBridge* Bridge = (CGrtToGcBridge*) Module->priv;
    Bridge->RemoveCanvas(Path);
  };
}


//----------------- Module internal functions --------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_value_get(MYX_GRT_VALUE *param, void *data)
{
  const char *grt_global_path;
  MYX_GRT_VALUE *value = NULL;
  
  if (myx_grt_value_get_type(param) != MYX_LIST_VALUE)
  {
    //error
  }

  if (myx_grt_list_item_count(param) < 1)
  {
    //error
  }
  
  grt_global_path= myx_grt_list_item_get_as_string(param, 0);

  // find out real value, e.g. /workbench/model/layers/0/description

  // to which gc belongs /workbench/model, check with mapping

  // parse /layers/0/description

  // retrieve value

  //value= myx_grt_value_from_string(grt_gc_subpath("/layers/0/description"));

  return value;
}


//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_value_set(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_list_size(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_list_item_get(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_list_item_add(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_list_item_remove(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_dict_item_get(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_dict_item_set(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_dict_item_count(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Grt module function for Bridge implementation
 *
 *   Function to deal with a Grt request for the bridge
 *
 * @param param the list of parameters
 * @param data pointer to Grt
 * 
 * @return  The return value if any, otherwise NULL
 */
MYX_GRT_VALUE *grt_gc_dict_item_key(MYX_GRT_VALUE *param, void *data)
{
  return NULL;
}

//----------------- CGrtToGcBridge -------------------------------------------------------------------------------------

CGrtToGcBridge::CGrtToGcBridge(MYX_GRT* Runtime)
{
  FRuntime = Runtime;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a path to canvas mapping to the bridge.
 *
 * @param Path The path under which the canvas can be accessed.
 * @param Canvas The canvas for which a mapping is to be created.
 */
void CGrtToGcBridge::AddCanvas(const char* Path, CGenericCanvas* Canvas)
{
  FCanvasMap[Path] = Canvas;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the module that holds the reference to this bridge.
 *
 * @param AModule The module to keep a reference of.
 */
void CGrtToGcBridge::Module(MYX_GRT_MODULE* AModule)
{
  FModule = AModule;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes a previously established mapping for a certain path from the bridge.
 *
 * @param Path The path under which a mapping was registered.
 */
void CGrtToGcBridge::RemoveCanvas(const char* Path)
{
  FCanvasMap.erase(Path);
}

//----------------------------------------------------------------------------------------------------------------------

