#
# C header parser
#
#

import re


class ParseError: pass


verbose=0

def strip_comment(s):
    g=re.match("^(.*?)(//.*)?$", s)
    if not g:
        return s
    return g.groups()[0]


def parse_struct(lines):
    elements=[]
    tname=None
    sname=None

    # parse declaration line
    line= lines[0].strip()
    toks= line.split()
    if line.endswith(";"):
        # typedef struct bla ble;
        m= re.match("^(typedef)?\ *(struct)?\ *(\w+)\ *(\w+);", line)
        if not m:
            print "Unrecognized typedef struct declaration", line
            raise ParseError
        sname=toks[2]
        tname=toks[3][:-1]
    else:
        m= re.match("^(typedef)?\ *(struct)?\ *(\w+)\ *{?", line)
        if not m:
            print "Expected struct declaration, got ", line
            raise ParseError

        typedef, struct, sname= m.groups()
        if sname == "struct":
            sname=None
        
        if "{" not in line:
            if lines[1].strip()=="{":
                del lines[1]

        assert struct!=None

    del lines[0]

    # parse body
    while lines:
        line= lines[0].strip()

        if line.startswith("//"):
            del lines[0]
            continue

        if line.startswith("}"): # end of struct
            m= re.match("}(.*);", line)
            if m:
                tname= m.groups()[0].strip()
            del lines[0]
            break

        m= re.match("(.*[ \*])([\w_]+);(.*)", line)
        if m:
            typ, name, rest= m.groups()

            # get argument
            extra= re.match(".*///\ *\[([^]]+)\]", rest)

            element= {
            "type": typ.strip(),
            "name": name.strip(),
            "options": []
            }

            if extra:
                element["options"]= extra.groups()[0].split(",")

            elements.append(element)

        del lines[0]

    struct= {
    "typedef":tname,
    "struct":sname,
    "elements":elements
    }
    return struct


def parse_enum(lines):
    values=[]
    tname=None
    sname=None

    # parse declaration line
    line= lines[0].strip()
    toks= line.split()
    if line.endswith(";"):
        # typedef enum bla ble;
        m= re.match("^(typedef)?\ *(enum)?\ *(\w+)\ *(\w+);", line)
        if not m:
            print "Unrecognized typedef enum declaration", line
            raise ParseError
        sname=toks[2]
        tname=toks[3][:-1]
    else:
        m= re.match("^(typedef)?\ *(enum)?\ *(\w+)\ *{?", line)
        if not m:
            print "Expected enum declaration, got ", line
            raise ParseError

        typedef, enum, sname= m.groups()
        if sname == "enum":
            sname=None

        if "{" not in line:
            if lines[1].strip()=="{":
                del lines[1]

        assert enum!=None

    del lines[0]

    values=[]
    # parse body
    while lines:
        line= strip_comment(lines[0]).strip()

        if not line:
            del lines[0]
            continue

        if line.startswith("}"): # end of enum
            m= re.match("}(.*);", line)
            if m:
                tname= m.groups()[0].strip()
            del lines[0]
            break

        values+= [s.strip() for s in line.strip().split(",")]
        del lines[0]

    while "" in values: values.remove("")

    elems=[]
    i=0
    for val in values:
        name,value=re.match("(\w*)(\ *=\ *\w*)?", val).groups()
        if value:
            v=int(re.match("=\ *(\d*)", value.strip()).groups()[0])
            i=v+1
        else:
            v=i
            i+=1
        elems.append((name, v))

    return {
    "typedef":tname,
    "enum":sname,
    "values":elems
    }


def parse_lines(lines, module):
    groups= []
    current_group= None
    current_entity= None
    global_objects={
    "enums":[]
    }

    structs= []
    while lines:
        line=lines[0].strip()
        if line.startswith("///"):
            m= re.match("///\ *(\w+)\ *(\w+)\ *(\w+)[^[]*(\[[^]]*\])?", line)
            if m:
                mod, cmd, name, options= m.groups()
                if mod != module:
                    print "Ignoring command for module", mod
                    del lines[0]
                    continue
                
                if options:
                    options= options[1:-1].split(",")
                else:
                    options= []

                if cmd == "begin":
                    assert current_group==None
                    current_group= {
                    "name": name,
                    "options": options,
                    "structs": []
                    }
                elif cmd == "end":
                    assert current_group!=None
                    assert current_group["name"]==name

                    # merge typedefs with struct declarations
                    typemap= {}
                    for s in current_group["structs"]:
                        if s["typedef"]==None:
                            typemap[s["name"]]= s["typedef"]
                            current_group["name"].remove(s)

                    for s in structs:
                        if s["typedef"]==None:
                            s["typedef"]= typemap.get(s["name"])
                            
                    groups.append(current_group)
                    current_group= None
                elif cmd == "entity":
                    assert current_entity==None
                    current_entity= (name, options)
            del lines[0]
        elif line.startswith("typedef struct") or line.startswith("struct"):
            if current_group!=None:
                struct= parse_struct(lines)

                if verbose:
                    print "Parsed struct", struct["struct"] or struct["typedef"]

                if current_entity:
                    struct["entity"], struct["options"]= current_entity
                else:
                    struct["entity"]= None
                    struct["options"]= []

                current_group["structs"].append(struct)
                current_entity= None
            else:
                # parse and ignore
                parse_struct(lines)
        elif line.startswith("typedef enum") or line.startswith("enum"):
            enum= parse_enum(lines)

            if verbose:
                print "Parsed enum", enum["enum"] or enum["typedef"]

            global_objects["enums"].append(enum)
        else:
            del lines[0]
    return groups, global_objects


def parse(data, module):
    # remove all /* comments */
    data= re.sub("/\*.*?\*/", "", data)
    
    # replace DOS or Mac style newlines with unix
    data= data.replace("\r\n","\n").replace("\r","\n")

    groups= None
    lines= data.split("\n")
    try:
        groups, global_objects= parse_lines(lines, module)
    except AssertionError, exc:
        print "In line:", lines[0]
        import traceback
        traceback.print_exc()
    except ParseError, exc:
        print "Parse error in line:", lines[0]
        import traceback
        traceback.print_exc()

    return groups, global_objects



def is_int(s, no_ptr=0):
    if not no_ptr:
        s=s.replace("*","").strip()
    typ= s.split()
    if typ[0] in ["unsigned", "signed"]:
        return 1
    if (len(typ)==2 and typ[0] in ["unsigned", "signed"]) or len(typ)==1:
        if typ[-1] in ["int", "long", "short", "char"]:
            return 1
    return 0

def is_fp(s, no_ptr=0):
    if not no_ptr:
        s=s.replace("*","").strip()
    if s in ["float","double"]:
        return 1
    return 0

def is_ptr(s):
    if s[-1]=="*":
        return 1
    return 0

def is_str(s, extra=[]):
    if re.match("(const\ *)?(signed\ *)?(unsigned\ *)?char\ *\*", s):
        return 1
    elif s in extra:
        return 1
    return 0


if __name__ == "__main__":
    import sys
    import pprint
    for g in parse(sys.stdin.read(), "xml"):
        pprint.pprint(g)
