--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Exceptions;
with Ada.Characters.Handling;

with Utils;
with Types;

package body Narval.Parameters is

   use Ada.Strings.Unbounded;

   ---------
   -- "=" --
   ---------

   --   function "=" (Left, Right : Parametre_Access) return Boolean is
   --     begin
   --        if Left = null and Right = null then
   --           return True;
   --        elsif Left = null or Right = null then
   --           return False;
   --        end if;
   --        if Left.Sorte_Contenant /= Right.Sorte_Contenant then
   --           return False;
   --        end if;
   --        return Left.Nom = Right.Nom;
   --     end "=";

   ------------
   -- To_Xml --
   ------------

   procedure To_Xml (Parameter : Parameter_Type;
                     Xml_Buffer : in out String_Buffering.String_Buffer)
   is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
   begin
      if Parameter.Mode = Write_Only then
         Ada.Exceptions.Raise_Exception (Write_Only_Parameter'Identity,
                                         To_String (Parameter.Name) &
                                         " is a write only parameter");
      end if;
      Current_Format := McKae.XML.EZ_Out.Continuous_Stream;
      case Parameter.Container_Kind is
         when Boolean_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Boolean_Value'Img);
         when Log_Level_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Level_Value'Img);
         when Unsigned_8_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Unsigned_8_Value'Img);
         when Unsigned_16_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Unsigned_16_Value'Img);
         when Unsigned_32_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Unsigned_32_Value'Img);
         when Unsigned_64_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Unsigned_64_Value'Img);
         when Natural_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Natural_Value'Img);
         when Positive_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Positive_Value'Img);
         when Integer_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Integer_Value'Img);
         when Float_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Float_Value'Img);
         when Long_Float_Type =>
            Output_Element (Xml_Buffer, "value",
                            Parameter.Long_Float_Value'Img);
         when String_Type =>
            Output_Element (Xml_Buffer, "value",
                            To_String (Parameter.String_Value));
      end case;
   end To_Xml;

   -----------
   -- Image --
   -----------

   function Image (Parameter : Parameter_Type) return String is
   begin
      if Parameter.Mode = Write_Only then
         Ada.Exceptions.Raise_Exception (Write_Only_Parameter'Identity,
                                         To_String (Parameter.Name) &
                                         " is a write only parameter");
      end if;
      case Parameter.Container_Kind is
         when Boolean_Type =>
            return Parameter.Boolean_Value'Img;
         when Log_Level_Type =>
            return Parameter.Level_Value'Img;
         when Unsigned_8_Type =>
            return Parameter.Unsigned_8_Value'Img;
         when Unsigned_16_Type =>
            return Parameter.Unsigned_16_Value'Img;
         when Unsigned_32_Type =>
            return Parameter.Unsigned_32_Value'Img;
         when Unsigned_64_Type =>
            return Parameter.Unsigned_64_Value'Img;
         when Natural_Type =>
            return Parameter.Natural_Value'Img;
         when Positive_Type =>
            return Parameter.Positive_Value'Img;
         when Integer_Type =>
            return Parameter.Integer_Value'Img;
         when Float_Type =>
            return Parameter.Float_Value'Img;
         when Long_Float_Type =>
            return Parameter.Long_Float_Value'Img;
         when String_Type =>
            return To_String (Parameter.String_Value);
      end case;
   end Image;

   ---------
   -- Set --
   ---------

   procedure Set
     (Parameter : in out Parameter_Type;
      Value : String)
   is
      use Interfaces;
   begin
      if Parameter.Mode = Read_Only then
         Ada.Exceptions.Raise_Exception (Read_Only_Parameter'Identity,
                                         To_String (Parameter.Name) &
                                         " is a read only parameter");
      end if;
      case Parameter.Container_Kind is
         when Boolean_Type =>
            Parameter.Boolean_Value := Boolean'Value (Value);
         when Log_Level_Type =>
            Parameter.Level_Value := Log4ada.Level_Type'Value (Value);
         when Unsigned_8_Type =>
            Parameter.Unsigned_8_Value := Unsigned_8'Value (Value);
         when Unsigned_16_Type =>
            Parameter.Unsigned_16_Value := Unsigned_16'Value (Value);
         when Unsigned_32_Type =>
            Parameter.Unsigned_32_Value := Unsigned_32'Value (Value);
         when Unsigned_64_Type =>
            Parameter.Unsigned_64_Value := Unsigned_64'Value (Value);
         when Natural_Type =>
            Parameter.Natural_Value := Natural'Value (Value);
         when Positive_Type =>
            Parameter.Positive_Value := Positive'Value (Value);
         when Integer_Type =>
            Parameter.Integer_Value := Integer'Value (Value);
         when Float_Type =>
            Parameter.Float_Value := Float'Value (Value);
         when Long_Float_Type =>
            Parameter.Long_Float_Value := Long_Float'Value (Value);
         when String_Type =>
            Parameter.String_Value := To_Unbounded_String (Value);
      end case;
   exception
      when Constraint_Error =>
         declare
            Postfix : Unbounded_String := Null_Unbounded_String;
         begin
            case Parameter.Container_Kind is
               when Boolean_Type =>
                  Postfix := Postfix & "boolean";
               when Log_Level_Type =>
                  Postfix := Postfix & "log_level";
               when Unsigned_8_Type =>
                  Postfix := Postfix & "unsigned_8";
               when Unsigned_16_Type =>
                  Postfix := Postfix & "unsigned_16";
               when Unsigned_32_Type =>
                  Postfix := Postfix & "unsigned_32";
               when Unsigned_64_Type =>
                  Postfix := Postfix & "unsigned_64";
               when Natural_Type =>
                  Postfix := Postfix & "natural";
               when Positive_Type =>
                  Postfix := Postfix & "positive";
               when Integer_Type =>
                  Postfix := Postfix & "integer";
               when Float_Type =>
                  Postfix := Postfix & "float";
               when Long_Float_Type =>
                  Postfix := Postfix & "long_float";
               when String_Type =>
                  Postfix := Postfix & "string";
            end case;
            Ada.Exceptions.Raise_Exception (Bad_Format'Identity,
                                            Value &
                                            " : not conformant with type " &
                                            To_String (Postfix));
         end;
   end Set;

   function Get_Attributes (Parameter : Parameter_Type)
                           return Attributes_List is
      Prefix : Unbounded_String := Null_Unbounded_String;
   begin
      if Parameter.Mode = Read_Only then
         Prefix := To_Unbounded_String ("constant ");
      end if;
      case Parameter.Container_Kind is
         when Boolean_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "boolean"),
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Log_Level_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "log_level"),
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Unsigned_8_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "unsigned_integer"),
                    "size" = "8",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Unsigned_16_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "unsigned_integer"),
                    "size" = "16",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Unsigned_32_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "unsigned_integer"),
                    "size" = "32",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Unsigned_64_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "unsigned_integer"),
                    "size" = "64",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Natural_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "natural"),
                    "size" = "31",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Positive_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "positive"),
                    "size" = "31",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Integer_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "integer"),
                    "size" = "32",
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Float_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "float"),
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when Long_Float_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "long float"),
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
         when String_Type =>
            return ("mode" = Parameter.Mode'Img,
                    "monitor" = Parameter.Monitor'Img,
                    "type" = To_String (Prefix & "string"),
                    "run_parameter" = Parameter.Run_Parameter'Img,
                    "editor" = Parameter.Editor'Img);
      end case;
   end Get_Attributes;

   function "+"(Left : Attributes_List;
                Right : Attribute_Value_Pairs) return Attributes_List is
      Array_To_Return : Attributes_List (1 .. Left'Length + 1);
   begin
      Array_To_Return (1 .. Left'Length) := Left;
      Array_To_Return (Left'Length + 1) := Right;
      return Array_To_Return;
   end "+";

   function "+"(Left : Attribute_Value_Pairs;
                Right : Attribute_Value_Pairs) return Attributes_List is
   begin
      return (1 => Left, 2 => Right);
   end "+";

   function New_Parameter (Item : String) return Parameter_Access is
      Infos : constant Types.String_Array := Utils.Split (Item, ';');
      Container : Container_Type;
      Parameter : Parameter_Access;
      Mode : Mode_Type;
   begin
      if Infos'Length /= 4 then
         raise Parameter_Bad_Format;
      else
         Mode := Mode_Type'Value (To_String (Infos (3)));
         Container := Container_Type'Value (To_String (Infos (2)));
         case Container is
            when Boolean_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Boolean_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Boolean_Value => Boolean'Value
                    (To_String (Infos (4))));
            when Log_Level_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Log_Level_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Level_Value => Log4ada.Level_Type'Value
                    (To_String (Infos (4))));
            when Unsigned_8_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Unsigned_8_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Unsigned_8_Value =>
                    Interfaces.Unsigned_8'Value
                    (To_String (Infos (4))));
            when Unsigned_16_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Unsigned_16_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                   Editor => None,
                  Unsigned_16_Value =>
                    Interfaces.Unsigned_16'Value
                    (To_String (Infos (4))));
            when Unsigned_32_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Unsigned_32_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Unsigned_32_Value =>
                    Interfaces.Unsigned_32'Value
                    (To_String (Infos (4))));
            when Unsigned_64_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Unsigned_64_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Unsigned_64_Value =>
                    Interfaces.Unsigned_64'Value
                    (To_String (Infos (4))));
            when Natural_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Natural_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Natural_Value => Natural'Value
                    (To_String (Infos (4))));
            when Positive_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Positive_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Positive_Value => Positive'Value
                    (To_String (Infos (4))));
            when Integer_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Integer_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Integer_Value => Integer'Value
                    (To_String (Infos (4))));
            when Float_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Float_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Float_Value => Float'Value
                    (To_String (Infos (4))));
            when Long_Float_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => Long_Float_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  Long_Float_Value => Long_Float'Value
                    (To_String (Infos (4))));
            when String_Type =>
               Parameter := new Parameter_Type'
                 (Container_Kind => String_Type,
                  Mode => Mode,
                  Monitor => Never,
                  Name => Infos (1),
                  Run_Parameter => False,
                  Editor => None,
                  String_Value => Infos (4));
         end case;
      end if;
      return Parameter;
   end New_Parameter;

   function Find_Parameter (List : Parameter_Vector_Package.Vector;
                            Name : String;
                            Case_Sensitivity : Boolean := False)
                           return Parameter_Access is
   begin
      return Find_Parameter (List,
                             To_Unbounded_String (Name),
                             Case_Sensitivity);
   end Find_Parameter;

   function Find_Parameter (List : Parameter_Vector_Package.Vector;
                            Name : Unbounded_String;
                            Case_Sensitivity : Boolean := False)
                           return Parameter_Access is
      use Parameter_Vector_Package;
      Index : Natural;
      Parameter : Parameters.Parameter_Access;
      List_Length : Natural;
   begin
      Index := First_Index (List);
      List_Length := Natural (Length (List));
      for I in Index .. Index + List_Length - 1 loop
         Parameter := Element (List, I);
         if Case_Sensitivity then
            if Name = Parameter.Name then
               return Parameter;
            end if;
         else
            if Ada.Characters.Handling.To_Lower (To_String (Parameter.Name)) =
              To_String (Name) then
               return Parameter;
            end if;
         end if;
      end loop;
      raise Parameter_Not_Found;
   end Find_Parameter;

end Narval.Parameters;
