--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Strings.Unbounded;

with Shell_Commands;
with Exception_Message;
with Error_Message;

with Get_Command;
with Launch_Command;
with Set_Command;
with Send_Order_Command;
with Load_Command;
with Sleep_Command;
with Wait_Command;
with Echo_Command;
with Group_Command;

with McKae.XML.EZ_Out.String_Stream;

function Help (Shell_Line : String) return String is
   use Ada.Strings.Unbounded;
   function Local_Syntax (Cmd : String; Syntax : String)
                         return Unbounded_String;
   function Syntax_String
     (Status : Shell_Commands.Status_Type := Shell_Commands.Syntax_Error)
     return String;
   function Local_Syntax (Cmd : String; Syntax : String)
                         return Unbounded_String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      Start_Element (Xml_Buffer, "result", ("cmd" = Cmd, "status" = "OK"));
      Start_Element (Xml_Buffer, "message", "type" = "help");
      Output_Element (Xml_Buffer, "syntax", Syntax);
      End_Element (Xml_Buffer, "message");
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return To_Unbounded_String (String_To_Return);
      end;
   end Local_Syntax;
   Help_Array : constant array
     (Shell_Commands.Command_Type) of Unbounded_String :=
     (Shell_Commands.Get =>
        To_Unbounded_String (Get_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Help => Local_Syntax ("help",
                                           "Help [command_name|commands]"),
      Shell_Commands.Launch =>
        To_Unbounded_String (Launch_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Send_Order =>
        To_Unbounded_String (Send_Order_Command.Syntax_String
                             (Shell_Commands.Ok)),
      Shell_Commands.Set =>
        To_Unbounded_String (Set_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Load =>
        To_Unbounded_String (Load_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Sleep =>
        To_Unbounded_String (Sleep_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Wait =>
        To_Unbounded_String (Wait_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Echo =>
        To_Unbounded_String (Echo_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Dump => Local_Syntax ("dump", "dump"),
      Shell_Commands.Group =>
        To_Unbounded_String (Group_Command.Syntax_String (Shell_Commands.Ok)),
      Shell_Commands.Special_Domi =>
        Local_Syntax ("special_domi", "special_domi sub_system_name"));
   use type Shell_Commands.Command_Type;
   function Syntax_String
     (Status : Shell_Commands.Status_Type := Shell_Commands.Syntax_Error)
     return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "help", "status" = Status'Img));
      Start_Element (Xml_Buffer, "message", "type" = "help");
      Output_Element (Xml_Buffer, "syntax",
                      "help command_name");
      Output_Element (Xml_Buffer, "syntax",
                      "help commands");
      End_Element (Xml_Buffer, "message");
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Syntax_String;
begin
   declare
      Arguments : constant Shell_Commands.Arguments_Array :=
        Shell_Commands.Arguments (Shell_Line);
      Help_Command : Shell_Commands.Command_Type;
   begin
      if Arguments'Length /= 1 then
         return Syntax_String;
      else
         if Arguments (1) = "commands" then
            declare
               Return_String : Unbounded_String := Null_Unbounded_String;
            begin
               for I in Shell_Commands.Command_Type'Range loop
                  Return_String := Return_String & I'Img;
                  if I /= Shell_Commands.Command_Type'Last then
                     Return_String := Return_String & ",";
                  end if;
               end loop;
               return To_String (Return_String);
            end;
         end if;
         Help_Command := Shell_Commands.Find (Arguments (1));
         return To_String (Help_Array (Help_Command));
      end if;
   exception
      when Shell_Commands.Bad_Command =>
         return Error_Message (Cmd => "help",
                               Severity => "WARNING",
                               Message => To_String
                                 (Arguments (1) & " command_name not valid"));
   end;
exception
   when Shell_Commands.No_Argument =>
      return Syntax_String;
   when E : others =>
      return Exception_Message ("help", E);
end Help;
