--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Strings.Unbounded;
with Ada.Text_IO;
with Ada.Exceptions;

with System_Calls;
with Exception_Message;
with Error_Message;
with Ok_Message;

with Narval.Configurator.Abstract_Actors_Coordination;
with Narval.Narval_Naming_Registry;

with McKae.XML.EZ_Out.String_Stream;

package body Launch_Command is

   use Narval.Configurator.Abstract_Actors_Coordination;

   ------------
   -- Launch --
   ------------

   function Launch (Shell_Line : String) return String is
   begin
      declare
         Arguments : constant Shell_Commands.Arguments_Array :=
           Shell_Commands.Arguments (Shell_Line);
         Process_Return : Integer;
         use Ada.Strings.Unbounded;
         Config : Configuration_Access;
         Nombre_Tests : Natural := 0;
         Launch_Seem_To_Failed : exception;
      begin
         if Arguments'Length = 2 or Arguments'Length = 3 then
            declare
               Base_Command : constant String :=
                 "ssh -X -Y " & To_String (Arguments (2)) &
                 " sub_system_coordinator --config_name " &
                 To_String (Arguments (1)) & " &";
               Command_String : constant String := "xterm -sl 500 -T """ &
                 "narval subsystem " & To_String (Arguments (1)) &
                 """ -e " & Base_Command;
            begin
               begin
                  Config := Narval.Narval_Naming_Registry.Get_Configuration
                    (To_String (Arguments (1)));
                  return Error_Message ("launch",
                                        "WARNING",
                                        To_String (Arguments (1)) &
                                        " already exist");
               exception
                  when Narval.Unknown_Sub_System =>
                     Ada.Text_IO.Put_Line
                       (To_String (Arguments (1)) &
                        " don't exist in registry, can continue");
               end;
               if Arguments'Length = 2 then
                  Process_Return := System_Calls.Exec_Command
                    (Command_String);
               elsif Arguments'Length = 3 and Arguments (3) = "local" then
                  Process_Return := System_Calls.Exec_Command
                    (Base_Command);
               else
                  Process_Return := 10101;
               end if;
               if Process_Return /= 0 then
                  return Error_Message ("launch",
                                        "ERROR",
                                        "command failed with result" &
                                        Process_Return'Img);
               end if;
               loop
                  begin
                     Config := Narval.Narval_Naming_Registry.Get_Configuration
                       (To_String (Arguments (1)));
                     if Config /= null then
                        return Ok_Message ("launch",
                                           To_String (Arguments (1)));
                     end if;
                  exception
                     when Narval.Unknown_Sub_System =>
                        Ada.Text_IO.Put_Line
                          ("waiting for sub system to launch");
                        delay 1.0;
                        Nombre_Tests := Nombre_Tests + 1;
                        if Nombre_Tests = 10 then
                           Ada.Exceptions.Raise_Exception
                             (Launch_Seem_To_Failed'Identity,
                              "waited 10 seconds without success");
                        end if;
                  end;
               end loop;
            end;
         else
            return Syntax_String;
         end if;
      end;
   exception
      when Shell_Commands.No_Argument =>
         return Syntax_String;
      when E : others =>
         return Exception_Message ("launch", E);
   end Launch;

   function Syntax_String
     (Status : Shell_Commands.Status_Type := Shell_Commands.Syntax_Error)
     return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "launch", "status" = Status'Img));
      Start_Element (Xml_Buffer, "message", "type" = "help");
      Output_Element (Xml_Buffer, "syntax",
                      "launch sub_system_name host_name");
      Output_Element (Xml_Buffer, "syntax",
                      "launch sub_system_name host_name local");
      End_Element (Xml_Buffer, "message");
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Syntax_String;

end Launch_Command;
