/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  bootp.c
 * Purpose: Get information for client with BOOTP protocol
 * Entries: bootp
 *
 **************************************************************************
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <general.h>
#include <net.h>
#include <arpa.h>
#include <romlib.h>
#include "./bootp.h"



/*
 **************************************************************************
 * 
 * To make it easier to specify the original BOOTP record size use the
 * following define:
 */
#define BOOTP_REC_SIZE	((int)((char *)boot_rec.bp_ext - (char *)(&boot_rec)))


/*
 **************************************************************************
 * 
 * Global variables:
 */
static unsigned long boot_xid;			/* bootp transaction ID	*/
static unsigned long boot_start_time;		/* when started to boot	*/
struct bootp boot_rec;				/* bootp record		*/


/*
 **************************************************************************
 * 
 * Return information from the vendor area of the BOOTP record. Note
 * that this will work only with RFC1048-compliant messages.
 *
 */
unsigned char *get_vend(id)
int id;
{
  register unsigned char *cp, *endp;

  /*
   * If the bootp record has been modified, there is no ending pointer. In
   * that case we assume that the vendor information field is correctly
   * terminated with an ending tag.
   */
  endp = (unsigned char *)(&boot_rec) + sizeof(boot_rec);
  cp = boot_rec.bp_vend;

  /* Check if vendor area is RFC1048-compliant */
  if (memcmp(cp, VM_RFC1048, VM_SIZE))
	return(NULL);

  /* Now scan the vendor information field */
  cp += VM_SIZE;
  while (*cp != VEND_END && (endp == NULL || cp < endp)) {
	if (*cp != VEND_NOP) {
		if (*cp == id)
			break;
		cp += *(cp + 1) + 1;
	}
	cp++;
  }
  if (*cp == id)
	return(cp + 1);
  return(NULL);
}



/*
 **************************************************************************
 * 
 * Send a BOOTP request via UDP.
 *
 */
static void bootp_send()
{
  /* Assemble the BOOTP request */
  memset(&boot_rec, 0, BOOTP_REC_SIZE);
  memcpy(boot_rec.bp_vend, VM_RFC1048, VM_SIZE);
  get_hw_addr(boot_rec.bp_chaddr);
  boot_rec.bp_op = BOOTP_REQUEST;
  boot_rec.bp_hwtype = BOOTP_ETHER;
  boot_rec.bp_hlen = ETH_ALEN;
  boot_rec.bp_xid = boot_xid;
  boot_rec.bp_secs = htons(((unsigned short)(get_ticks() - boot_start_time)) / 18);
  boot_rec.bp_ciaddr = htonl(IP_ANY);
  boot_rec.bp_siaddr = htonl(IP_BROADCAST);

  /* Send the bootp record as a broadcast message to all servers */
  udp_write((char *)(&boot_rec), BOOTP_REC_SIZE);
}



/*
 **************************************************************************
 * 
 * Get a BOOTP record from the server.
 *
 */
static int bootp_get()
{
  char errch;
  int timeout, i;

  /*
   * Setup the IP interface information. Our own address has to be
   * IP_ANY, so that the IP layer will not discard the incoming packets.
   * Then open a UDP socket.
   */
  if_config(IP_ANY, IP_CLASS_A);
  (void)udp_open(IP_BROADCAST, BOOTP_C_PORT, BOOTP_S_PORT);

  /*
   * Now loop until receiving a reply from a server. The retry time is
   * computed as suggested in RFC951.
   */
  timeout = MIN_TIMEOUT;
  printf("BOOTP: Sending request (press ESC to abort): ");
  while (TRUE) {
	boot_xid = get_ticks() + random();
	bootp_send();
	i = udp_read((char *)(&boot_rec), BOOTP_REC_SIZE, timeout*18, CHR_ESC);
	if (i < 0) {				/* user pressed ESC */
		printf("\nAborted\n");
		return(FALSE);
	} else if (i == 0)			/* timeout */
		errch = '.';
	else if (boot_rec.bp_xid != boot_xid)	/* rule out duplicate packet */
		errch = '?';
	else
		break;
	timeout = (timeout << 1) + (random() & 0x1f) - 32;
	if (timeout < MIN_TIMEOUT)
		timeout = MIN_TIMEOUT;
	else if (timeout > 60)		/* dont allow more than 60 seconds */
		timeout = 60;
	printf("%c", errch);
  }

  if (i < BOOTP_REC_SIZE ||
      (boot_rec.bp_vend[0] != '\0' && get_vend(VEND_END) == NULL)) {
	printf("\nBOOTP: Invalid record\n");
	return(FALSE);
  }

  printf("ok\n");
  return(TRUE);
}



#ifndef NOBPEXT
/*
 **************************************************************************
 * 
 * Get vendor extensions file with TFTP.
 */
static int get_ext(ftag)
unsigned char *ftag;
{
  unsigned char *cp;
  char *inbuf;
  int first_flag = TRUE;
  int size;

  /* Open the TFTP connection */
  printf("BOOTP: Loading %ls: ", ftag + 1, *ftag);
  if ((inbuf = tftp_open(ntohl(boot_rec.bp_siaddr), ftag + 1, *ftag)) == NULL)
	return(FALSE);

  /*
   * Read all blocks from vendor extension file. The file has the RFC
   * vendor ID at the beginning, so we have to take special care with
   * the first block.
   */
  cp = get_vend(VEND_END) - 1;
  while (cp < ((unsigned char *)(&boot_rec) + sizeof(boot_rec) - SEGSIZE)) {
	if ((size = tftp_get()) < 0)
		return(FALSE);
	if (first_flag) {
		if (size < (VM_SIZE + 1) ||
		    memcmp(inbuf, VM_RFC1048, VM_SIZE)) {
			printf("Invalid file\n");
			return(FALSE);
		}
		first_flag = FALSE;
		memcpy(cp, (unsigned char *)inbuf + VM_SIZE, size - VM_SIZE);
		cp += size - VM_SIZE;
	} else if (size > 0) {
		memcpy(cp, (unsigned char *)inbuf, size);
		cp += size;
	}
	if (size < SEGSIZE)
		break;
  }

  /* If the file is too large, we are probably missing the end identifier */
  if (cp >= ((unsigned char *)(&boot_rec) + sizeof(boot_rec))) {
	printf("File too large\n");
	return(FALSE);
  }

  /* Finally check if we have an end identifier at all */
  if (get_vend(VEND_END) == NULL) {
	printf("No end tag\n");
	return(FALSE);
  }
  printf("ok\n");
  return(TRUE);
}
#endif



#ifdef BPDEBUG
/*
 **************************************************************************
 * 
 * Print the complete BOOTP record
 */
static void print_rec()
{
  register unsigned char *cp, *end;
  int i, c, adr;

  adr = 0;
  cp = (unsigned char *)boot_rec;
  if ((end = get_vend(VEND_END)) == NULL)
	end = cp + sizeof(struct bootp);
  printf("\n\n");
  while(cp < end) {
	if (adr < 16) printf("00%x  ", adr);
	else if (adr < 256) printf("0%x  ", adr);
	else printf("%x  ", adr);

	for (i = 0; i < 16 && cp <= end; i++, cp++) {
		c = *cp & 0xff;
		if (c < 16) printf("0%x ", c);
		else printf("%x ", c);
	}
	printf("\n");
	adr += 16;
  }
  if (getkey() == 0)
	(void)getkey();
  printf("\n\n");
}
#endif



/*
 **************************************************************************
 * 
 * Handle BOOTP protocol. If we have a BOOTP record already, we just set
 * the network parameters anew.
 */
int bootp(redo)
int redo;
{
  register unsigned char *cp;
  t_ipaddr netmask;

  /* Set boot start time - required for gateway booting */
  boot_start_time = get_ticks();

  /* Get BOOTP record from server */
  if (!redo && !bootp_get())
	return(FALSE);

  /* Setup the network interface with the values received from BOOTP server */
  netmask = IP_ANY;
  if ((cp = get_vend(VEND_SUBNET)) != NULL)
	netmask = ntohl(*((t_ipaddr *)(cp + 1)));
  if_config(ntohl(boot_rec.bp_yiaddr), netmask);

  /* Also setup the default gateway so that we can reach the server. */
  if ((cp = get_vend(VEND_ROUTER)) != NULL)
	set_gateway(ntohl(*((t_ipaddr *)(cp + 1))));

#ifndef NOBPEXT
  /* Next try to get the extensions file if there is any specified */
  if (!redo && (cp = get_vend(VEND_EXTFILE)) != NULL && !get_ext(cp))
	return(NULL);
#endif

#ifdef BPDEBUG
  print_rec();
#endif

  return(TRUE);
}

