
/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_misc_h
#define __nd_misc_h

#include <stdlib.h>
#include <net/if.h>
#include <gtk/gtk.h>
#include <netdude/nd_protocol.h>

/**
 * nd_misc_writen - writes n bytes to a descriptor
 * @fd: file descriptor
 * @data: data buffer
 * @size: size of buffer
 *
 * The function tries to write n buffers to the given
 * file descriptor.
 *
 * Returns: number of bytes written, or value < 0 on error.
 */
guint     nd_misc_writen(int fd, const guchar *data, guint size);


/**
 * nd_misc_readn - reads ny bytes from a descriptor
 * @fd: file descriptor
 * @data: data buffer
 * @size: size of buffer
 *
 * The function tries to read n buffers from the given
 * file descriptor.
 *
 * Returns: number of bytes read, or value < 0 on error.
 */
guint     nd_misc_readn(int fd, guchar *data, guint size);


/**
 * nd_misc_ones_complement_checksum - computes ones-complement checksum
 * @data: pointer to data to checksum.
 * @data_len: length of @data.
 * @sum: pre-added input.
 *
 * The function computes the ones-complement checksum of the data
 * pointed to by @data. If you want to extend the checksum over data
 * areas not included in @data, you can pre-add the sum of that data
 * in units of short ints (16bit) and pass it as @sum.
 *
 * Returns: checksum.
 */
int       nd_misc_ones_complement_checksum(const void *data, int data_len, u_int32_t sum);

/**
 * nd_misc_in_cksum - computes Internet checksum
 * @addr: data to checksum.
 * @addr_len: length of @addr data.
 * @preadd: pre-added input.
 *
 * The function computes the Internet checksum of @addr. Other data
 * can be included in the checksum by pre-added the sum if that data
 * in units of short ints (16bit) and passing it as @preadd.
 *
 * Returns: checksum.
 */
u_short   nd_misc_in_cksum(register u_short *addr, register int addr_len, u_int preadd);

/**
 * nd_misc_exists - checks file existance.
 * @filename: name of file.
 *
 * The function checks whether @filename exists and returns
 * the result.
 *
 * Returns: %TRUE when @filename exists, %FALSE otherwise.
 */
gboolean  nd_misc_exists(const char *filename);

/**
 * nd_misc_can_read - checks file read access.
 * @filename: name of file.
 *
 * The function checks whether @filename can be read and
 * returns the result.
 * 
 * Returns: %TRUE if file is readable, %FALSE if it is not or
 * if there is another error (such as @filename doesn't exist).
 */
gboolean  nd_misc_can_read(const char *filename);

/**
 * nd_misc_is_tcpdump_file - checks whether file is tcpdump savefile.
 * @filename: name of file.
 *
 * The function checks whether @filename is a tcpdump savefile
 * by looking for tcpdump's magic %0xa1b2c3d4 and its variations.
 *
 * Returns: %TRUE if test was positive, %FALSE otherwise or if
 * there was an error.
 */
gboolean  nd_misc_is_tcpdump_file(const char *filename);

/**
 * nd_misc_can_exec - checks file execution access.
 * @filename: name of file.
 *
 * The function checks whether @filename can be executed and
 * returns the result.
 * 
 * Returns: %TRUE if file is executable, %FALSE if it is not or
 * if there is another error (such as @filename doesn't exist).
 */
gboolean  nd_misc_can_exec(const char *filename);

/**
 * nd_misc_is_dir - checks if file is a directory.
 * @filename: name of file.
 *
 * The function checks whether @filename is a directory and
 * returns the result.
 * 
 * Returns: %TRUE if file is a directory, %FALSE if it is not or
 * if there is another error (such as @filename doesn't exist).
 */
gboolean  nd_misc_is_dir(const char *filename);


/**
 * nd_misc_int_cmp - glib compare function.
 * @int1: pointer using integer value.
 * @int2: pointer using integer value.
 *
 * This is a simple compare function for pointers that can be casted
 * (not dereferenced!) to ints, returning -1/0/1 of row1 is smaller/
 * equal/larger than row2. The signature allows usage with various
 * glib functions.
 *
 * Returns: comparison result.
 */
gint      nd_misc_int_cmp(gconstpointer int1, gconstpointer int2);


/**
 * nd_misc_add_slash - adds slasg to end of filename if necessary.
 * @filename: filename to test.
 * 
 * The function checks if @filename ends with a slash and if not,
 * reallocs the memory @filename uses and appends a slash.
 *
 * Returns: resulting string.
 */
char     *nd_misc_add_slash(char *filename);


/**
 * nd_misc_get_if_names - returns network interfaces satisfying given conditions.
 * @flags: flags the interface must have set (IFF_xxx values, see net/if.h).
 * 
 * The function returns an alphabetically ordered glib list of strings containing
 * the names of those network interfaces that satisfy the given flag combination.
 *
 * Returns: list of strings. You need to free this list and its entries if you
 * don't need the result any more.
 */
GList    *nd_misc_get_if_names(int flags);

/**
 * nd_misc_get_if_mtu - returns interface MTU.
 * @ifname: name of interface to query.
 *
 * The function returns the MTU for a given interface.
 * I have no idea whether this compiles on anything non-Linux ...
 */
int       nd_misc_get_if_mtu(const char *ifname);

/**
 * nd_misc_get_hardware_string - assembles string displaying hardware address.
 * @str: pointer to space filled with string.
 * @str_len: available string length.
 * @address: hardware address data.
 * @addr_len: length of hardware address in bytes.
 *
 * The function places into the space pointed to by @str
 * a string representing the hardware @address as "xx : xx : ... : xx".
 */
void      nd_misc_get_hardware_string(char *str, guint str_len,
				      guchar *address, guint addr_len);

/**
 * nd_misc_get_unnamed_string - returns numbered dummy file name.
 *
 * The function returns a pointer to statically allocated data
 * that gets filled with a numbered dummy file name. You need
 * to strdup() the pointer if you want to keep it around.
 *
 * Returns: new filename.
 */
char     *nd_misc_get_unnamed_string(void);

#endif
     
